function [theta_bias training_theta] = ...
                      construct_theta_bias(obs_training, training_truth, ...
                                           obs_testing, varargin)
% CONSTRUCT_THETA_BIAS - constructs the biasing theta that can be passed to
%                        the STAPLER algorithm based upon previously observed
%                        training data.
%
% [theta_bias training_theta] = construct_theta_bias(obs_training, ...
%                                                    training_truth, ...
%                                                    obs_testing);
%
% Input: obs_training - the training observations
%        training_truth - the truth associated with each voxel in the training
%                         set
%        obs_testing - the testing observations
%        cons_flag - the flag indicating whether or not to ignore consensus
%
% Output: theta_bias - the biasing theta that is passed to STAPLER
%         training_theta - the training theta

if (length(varargin) == 0)
    cons_flag = 0;
else
    cons_flag = varargin{1};
end

% first make sure that the observations are of type 'slice'
if strcmp(obs_testing.type, 'volume')
    obs_testing = convert_volume_to_slice_obs(obs_testing);
end
if strcmp(obs_training.type, 'volume')
    obs_training = convert_volume_to_slice_obs(obs_training);
end

if (cons_flag == 1)
    [MV_est MV_W] = majority_vote(obs_training);
    consensus = max(MV_W, [], 4) == 1;
else
    consensus = zeros(obs_training.dims);
end

% get the unique labels for the training and testing data
unique_training_labels = [];
for o = 1:obs_training.num_obs
    unique_training_labels = union(unique_training_labels, ...
                                   unique(obs_training.data{o}));
end
unique_testing_labels = [];
for o = 1:obs_testing.num_obs
    unique_testing_labels = union(unique_testing_labels, ...
                                  unique(obs_testing.data{o}));
end
num_labels = max([unique_testing_labels; unique_training_labels])+1;

% get the unique raters for the training and testing data
unique_training_raters = unique(obs_training.raters);
unique_testing_raters = unique(obs_testing.raters);
num_training_raters = length(unique_training_raters);
num_testing_raters = length(unique_testing_raters);

% construct training theta
training_theta = zeros(num_labels, num_labels, num_training_raters);

% iterate over all of the observations
for i = 1:obs_training.num_obs

    % set the rater number
    rt = obs_training.raters(i);
    rinds = find(rt == unique_training_raters);
    r = rinds(1);

    % set the slice number
    sl = obs_training.slices(i);

    % apply it to training theta
    for x = 1:obs_training.dims(1)
        for y = 1:obs_training.dims(2)
            if ~consensus(x, y, sl)
                l = training_truth(x, y, sl);
                s = obs_training.data{i}(x, y);
                training_theta(s+1, l+1, r) = training_theta(s+1, l+1, r) + 1;
            end
        end
    end
end

% normalize the training theta
for r = 1:size(training_theta, 3)
    for l = 1:size(training_theta, 2)
        stt = sum(training_theta(:, l, r));
        if (stt > 0)
            training_theta(:, l, r) = training_theta(:, l, r) / stt;
        else
            training_theta(:, l, r) = 0;
            training_theta(l, l, r) = 1;
        end
    end
end

% find the number of observed voxels for each label and for each rater
Njs = zeros(num_training_raters, num_labels);
for i = 1:obs_training.num_obs

    % get the rater number
    ri = obs_training.raters(i);
    rind = find(unique_training_raters == ri);
    r = rind(1);
    sl = obs_training.slices(i);

    data = obs_training.data{i};

    for l = 1:num_labels
        Njs(r, l) = Njs(r, l) + length(find(data == l-1 & ...
                                            consensus(:, :, sl) == 0));
    end
end

% construct the theta bias
theta_bias = zeros(num_labels, num_labels, num_testing_raters);
for r = 1:num_testing_raters
    rind = find(unique_training_raters == unique_testing_raters(r));

    % make sure the rater observed training data
    if length(rind) > 0
        rt = rind(1);

        theta_bias(:, :, r) = training_theta(:, :, rt);

        for l = 1:num_labels
            theta_bias(:, l, r) = Njs(rt, l) * theta_bias(:, l, r);
        end
    end
end

