function [target ims] = norm_int(target, ims, minl, maxl, obs, est, ...
                                 num_coef, varargin)
% NORM_INT - normalizes target and atlas intensity images
%
% Two possible forms:
% 1) [target ims] = norm_int(target, ims, minl, maxl, obs, est, num_coef);
% 2) [target ims] = norm_int(target, ims, minl, maxl, obs, est, num_coef, rho);
%
% Input: target - the target intensity image (X by Y by Z)
%        ims - a collection of atlas intensities (X by Y by Z by R)
%        minl - the minimum percentile (lower bound), e.g. .05
%        maxl - the maximum percentile (upper bound), e.g. .95
%        obs - the observation struct associated with ims
%        est - an estimate of the target label values
%        num_coef - the number of polynomial coefficients to estimate
%        rho - (optional) decay coefficient for intensity decay
%
% Output: target - normalized target intensity image
%         ims - the normalized atlas intensity images
%

    % a little bit of error checking
    if minl > 1 || maxl > 1 || minl < 0 || maxl < 0
        error('percentiles should be between 0 and 1');
    end
    if minl >= maxl
        error('Minimum percentile should be less than maximum percentile');
    end
    if ~min(size(target) == size(ims(:, :, :, 1)))
        error('target and ims should be the same size');
    end

    % first make sure we're dealing with doubles
    target = double(target);
    ims = double(ims);

    % first do a quantile normalization
    [target ims] = quantile_normalize(target, ims, est, obs, minl, maxl);

    % second, do a polynomial normalization
    if (num_coef > 0)
        [target ims] = polynomial_normalize(target, ims, est, obs, num_coef);

        % third, do another quantile normalization
        [target ims] = quantile_normalize(target, ims, est, obs, minl, maxl);
    end

    if length(varargin) == 1
        rho = varargin{1};
        [target ims] = decay_intensity(target, ims, est, obs, rho);
    end

end

function [target ims] = decay_intensity(target, ims, est, obs, rho)

    % determine the overall min
    mina = min([min(target(:)), min(ims(:))]);

    % first do the target
    target = (target - mina) .* exp(-rho * bwdist(est > 0)) + mina;

    % second do all of the images
    for a = 1:size(ims, 4)
        iml = ims(:, :, :, a);
        iml = (iml - mina) .* exp(-rho * bwdist(obs.data{a} > 0)) + mina;
        ims(:, :, :, a) = iml;
    end
end


function [target ims] = quantile_normalize(target, ims, est, obs, minl, maxl)

    min_t = quantile(target(est > 0), minl);
    max_t = quantile(target(est > 0), maxl);

    % normalize the target
    target = (target - min_t) / (max_t - min_t);

    % normalize the atlases
    for l = 1:size(ims, 4)
        iml = ims(:, :, :, l);
        min_l = quantile(iml(obs.data{l} > 0), minl);
        max_l = quantile(iml(obs.data{l} > 0), maxl);
        ims(:, :, :, l) = (iml - min_l) / (max_l - min_l);
    end

end

function [target ims] = polynomial_normalize(target, ims, est, obs, num_coef)

    % some settings
    num_atlases = size(ims, 4);

    % first choose the valid labels
    uniquelabels = unique(est(:));
    for a = 1:num_atlases
        uniquelabels = union(uniquelabels, unique(double(obs.data{a})));
    end
    num_appear = zeros(size(uniquelabels));
    num_labels = length(uniquelabels);
    for l = 1:num_labels
        num_appear(l) = length(find(est == uniquelabels(l)));
        for a = 1:num_atlases
            num_appear(l) = min([num_appear(l), ...
                                 length(find(obs.data{a} == uniquelabels(l)))]);
        end
    end
    ul = uniquelabels(num_appear > 50)
    num_labels = length(ul)

    % find the atlas that has the lowest standard deviation among the labels
    stds = zeros([num_atlases 1]);
    for a = 1:num_atlases
        ima = ims(:, :, :, a);
        for l = 1:num_labels
            stds(a) = stds(a) + std(ima(obs.data{a} == ul(l)));
        end
    end

    inds = find(stds == min(stds));
    minind = inds(1);

    Y = zeros([num_labels 1]);
    ima = ims(:, :, :, minind);
    for l = 1:num_labels
        Y(l) = median(ima(obs.data{minind} == ul(l)));
    end

    % apply to each atlas
    for a = 1:num_atlases
        ims(:, :, :, a) = apply_coefficients(Y, ims(:, :, :, a), ...
                                             obs.data{a}, num_labels, ...
                                             num_coef, ul);
    end

    % apply to the target
    target = apply_coefficients(Y, target, est, num_labels, num_coef, ul);

end

function im = apply_coefficients(Y, im, labels, num_labels, num_coef, ul)

    Z = zeros([num_labels 1]);
    for l = 1:num_labels
        Z(l) = median(im(labels == ul(l)));
    end

    X = zeros(num_labels, num_coef);
    for b = 1:num_coef
        X(:, b) = Z.^(b-1);
    end

    alpha = inv(X'*X)*X'*Y;

    X2 = zeros(numel(im), num_coef);
    for b = 1:num_coef
        X2(:, b) = im(:).^(b-1);
    end
    Z2 = X2 * alpha;

    im = reshape(Z2, size(im));

end

