/** \file   regpath.c
 *  \brief  Solves a family of l1-regularized logistic regression problem
 *          with different values of lambda.
 *
 *  Reads and standardizes data, solves a family of problems for
 *  lambda values, and writes the result.
 *
 */

#if HAVE_CONFIG_H
#   include "config.h"
#endif
#include "pkgdef.h"

#include <time.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef Rpackage
#include <R.h>
#include <Rinternals.h>
#endif

#include "def.h"
#include "l1_logreg.h"
#include "util.h"

#define BUFFER_SIZE     1024
#define MAX_TRIAL       1


static const char comment1[] = {
"%% This is a coefficients file of %s\n"
"%%    generated by %s_regpath ver %s\n"
"%% \n"
"%% Note that it is not a model file!\n"
"%% \n"
"%% Each column is coefficient vector of corresponding lambda\n"
"%% The corresponding lambda vector is stored at %s_lambda\n"
};

static const char comment2[] = {
"%% This is a lambda file of %s\n"
"%%    generated by %s_regpath ver %s\n"
"%% \n"
"%% The corresponding coefficient vector is stored at %s\n"
"%% \n"
};


static void summary_prob(const dmatrix *M, const int sflag,
                         const double lambda_max, const double lambda)
{
    fprintf(stderr,"Problem summary:\n");
    if (M->nz >= 0)
        fprintf(stderr,"    [feature matrix]        sparse matrix of "
            "(%d examples x %d features), %d non-zeros\n", M->m, M->n, M->nz);
    else
        fprintf(stderr,"    [feature matrix]        dense matrix of "
            "(%d examples x %d features)\n", M->m, M->n);

    if (sflag == TRUE)
        fprintf(stderr,"    [standardization]       yes\n");
    else
        fprintf(stderr,"    [standardization]       no\n");

    fprintf(stderr,"    [lambda_max]            %g\n", lambda_max);
    fprintf(stderr,"    [lambda]                %g", lambda);
    fprintf(stderr,"        (ratio = %.4f)\n\n", lambda/lambda_max);
}


static void summary_time(clock_t t1, clock_t t2, clock_t t3)
{
    fprintf(stderr,"Timing:\n");
    fprintf(stderr,"    [read data]         %10.3g (sec)\n",
        (double)t1/CLOCKS_PER_SEC);
    fprintf(stderr,"    [solve problem]     %10.3g (sec)\n",
        (double)t2/CLOCKS_PER_SEC);
    fprintf(stderr,"    [write solution]    %10.3g (sec)\n",
        (double)t3/CLOCKS_PER_SEC);
    fprintf(stderr,"    [total time]        %10.3g (sec)\n\n",
        (double)(t1+t2+t3)/CLOCKS_PER_SEC);
}


static void summary_all(const int m, const int n, const int lambda_count,
                        const double lambda_min, const double lambda_max,
                        const clock_t tt, const int ntiter, const int pcgiter,
                        const double *coeffs)
{
    int i;
    int card;
    card = 0;
    for (i = 0; i < n; i++)
        if (coeffs[i] != 0.0) card++;

    /* lambda : real lambda value */
    fprintf(stderr,"(%d examples x %d features) lambda_min =%9.4e lambda_max=%9.4e (%d),  card(last)= %d,  NTiter= %d,  PCGiter= %d : %10.3g (sec)\n", m, n, lambda_min, lambda_max, lambda_count, card, ntiter, pcgiter, (double)tt/CLOCKS_PER_SEC);
}


#ifndef Rpackage
static void usage(char *package, char *version)
{
    static const char *help[] = 
    {
"\n%s version %s\n",
"\nUsage: %s_regpath [options] feature_file class_file lambda_min num_lambda model_file\n",
"Solve a family of l1-regularized logistic regression problems\n",
"   for lambda values uniformly spaced from lambda_max to lambda_min\n"
"   in log scale.\n\n"
"   Arguments:\n",
"       feature_file        - feature matrix\n",
"       class_file          - output vector\n",
"       lambda_min          - minimum value of regularization parameter\n",
"       num_lambda          - number of lambdas (sample)\n",
"       model_file          - store results to file model_file\n",
"   Options:\n",
"       -c                  - only coefficients (to know real coefficients).\n",
"                             The model_file generated with -c option cannot\n",
"                             be used for classification.\n",
"                             Use this option only to plot regularization path.\n",
"       -q                  - quiet mode (same as -v0)\n",
"       -v [0..3]           - set verbosity level\n",
"                               0 : quiet mode\n",
"                               1 : show one line summary\n",
"                               2 : show simple log\n",
"                               3 : show detailed log\n",
"       -r                  - use relative lambda\n",
"                               if used,     lambda := lambda*lambda_max\n",
"                               if not used, lambda := lambda\n",
"       -s                  - standardize data\n",
"   Advanced options:\n"
"       -k <double>         - set tolerance for zero coefficients from KKT\n",
"                             condition\n",
"       -t <double>         - set tolerance for duality gap\n",
"\n",
NULL,
    };

    const char **p;
    for (p = help; *p != NULL; p++)
        fprintf(stderr,*p, package, version);
    exit (EXIT_SUCCESS);
}


static void parse_command_line_args(int argc, char *argv[], double *lambda, 
                                 int *lambda_count, train_opts *to, int *rflag,
                                 char **ifile_x, char **ifile_y, char **ofile)
{
    int c;
    int quiet = FALSE;
        
    /* default values */
    to->sflag         = FALSE;
    to->cflag         = FALSE;
    to->verbose_level = 1;
    to->tolerance     = 1.0e-8;
    to->ktolerance    = 0.999;
    *rflag            = FALSE;
            

    while ((c = getopt(argc, argv, "ck:rst:v:q")) != -1)
    {
        switch (c)
        {
        case 'r': *rflag            = TRUE;         break;
        case 'c': to->cflag         = TRUE;         break;
        case 's': to->sflag         = TRUE;         break;
        case 'k': to->ktolerance    = atof(optarg); break;
        case 't': to->tolerance     = atof(optarg); break;
        case 'v': to->verbose_level = atoi(optarg); break;
        case 'q': quiet = TRUE;            break;
        case '?':
        default : abort();
        }
    }
    if (quiet == TRUE) to->verbose_level = 0;
    argc -= optind;
    argv += optind;

    switch (argc)
    {
    case 5:
        *ofile        = argv[4];
        *lambda_count = atoi(argv[3]);
        *lambda       = atof(argv[2]);
        *ifile_y      = argv[1];
        *ifile_x      = argv[0];
        break;
    default:
        usage(PACKAGE_NAME, VERSION);
        exit (EXIT_SUCCESS);
    }
}
#endif

#ifdef Rpackage
static void convert_Rdata(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx,
             SEXP prdx, SEXP pb, SEXP plambda, SEXP prflag,
             SEXP psflag, SEXP pcflag, SEXP pqflag, SEXP pvval,
             SEXP ptol, SEXP pktol,
             SEXP plambda_count,
             dmatrix **matX, double **b, double *lambda, int *lambda_count,
             train_opts *to, int *rflag )
{
    dmatrix *X;

    *b                =  REAL(pb);
    *lambda           = *REAL(plambda);
    *lambda_count     = *INTEGER(plambda_count);
    *rflag            = *INTEGER(prflag);

    to->cflag         = *INTEGER(pcflag);
    to->sflag         = *INTEGER(psflag);
    to->verbose_level = *INTEGER(pvval);
    to->tolerance     = *REAL(ptol);
    to->ktolerance    = *REAL(pktol);

    if (*INTEGER(pqflag)) to->verbose_level = 0;

    X = malloc(sizeof(dmatrix));
    X->m   = *INTEGER(pm);
    X->n   = *INTEGER(pn);
    X->nz  = *INTEGER(pnz);
    X->val = REAL(pval);

    if (X->nz >= 0)
    {
        int i;
        X->idx = malloc(sizeof(int)*X->nz);
        X->jdx = malloc(sizeof(int)*X->nz);
        X->rdx = malloc(sizeof(int)*(X->m+1));

        /* C uses zero-based indexing while R uses one-based indexing */
        for (i = 0; i < X->nz;  i++) X->jdx[i] = INTEGER(pjdx)[i]-1;
        for (i = 0; i < X->m+1; i++) X->rdx[i] = INTEGER(prdx)[i]-1;
        /* build idx from rdx and jdx */
        dmat_build_idx(X);
    }
    else
    {
        X->idx = NULL;
        X->jdx = NULL;
        X->rdx = NULL;
    }
    *matX = X;
}

static SEXP create_Rdata_to_return(int count, int lambda_count, int n,
                                    double *lambda_vec, buffer_t *buf_val,
                                    buffer_t *buf_jdx, buffer_t *buf_rdx)
{
    SEXP res, Rval, Rjdx, Rrdx, Rm, Rn, Rnz, Rlambdas, names;

    PROTECT(res  = allocVector(VECSXP,7));
    PROTECT(Rm   = allocVector(INTSXP,1));
    PROTECT(Rn   = allocVector(INTSXP,1));
    PROTECT(Rnz  = allocVector(INTSXP,1));
    PROTECT(Rval = allocVector(REALSXP,count));
    PROTECT(Rjdx = allocVector(INTSXP,count));
    PROTECT(Rrdx = allocVector(INTSXP,lambda_count+1));
    PROTECT(Rlambdas = allocVector(REALSXP,lambda_count));
    INTEGER(Rm  )[0]= lambda_count;
    INTEGER(Rn  )[0]= n+1; /* solution length (# feat + # intercept) */
    INTEGER(Rnz )[0]= count;

    memcpy(REAL(Rval), buf_val->ptr, count*buf_val->elemsize);
    memcpy(INTEGER(Rjdx), buf_jdx->ptr, count*buf_jdx->elemsize);
    memcpy(INTEGER(Rrdx), buf_rdx->ptr, (lambda_count+1)*buf_rdx->elemsize);
    memcpy(REAL(Rlambdas), lambda_vec, lambda_count*sizeof(double));

    SET_VECTOR_ELT(res, 0, Rm  );
    SET_VECTOR_ELT(res, 1, Rn  );
    SET_VECTOR_ELT(res, 2, Rnz );
    SET_VECTOR_ELT(res, 3, Rval);
    SET_VECTOR_ELT(res, 4, Rjdx);
    SET_VECTOR_ELT(res, 5, Rrdx);
    SET_VECTOR_ELT(res, 6, Rlambdas);

    PROTECT(names = allocVector(STRSXP,7));
    SET_STRING_ELT(names, 0, mkChar("num.lambdas"));
    SET_STRING_ELT(names, 1, mkChar("num.intcoeffs"));
    SET_STRING_ELT(names, 2, mkChar("num.nonzeros"));
    SET_STRING_ELT(names, 3, mkChar("val"));
    SET_STRING_ELT(names, 4, mkChar("jdx"));
    SET_STRING_ELT(names, 5, mkChar("rdx"));
    SET_STRING_ELT(names, 6, mkChar("lambdas"));
    setAttrib(res, R_NamesSymbol, names);

    UNPROTECT(9);

    return res;
}

#endif

/*
 *  Standardize data and solve l_1-regularized logistic regression problems
 *  by calling l1_logreg_train() repeatedly.
 *
 */
#ifndef Rpackage
int main(int argc, char *argv[])
#else
SEXP regpathR(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx,
             SEXP prdx, SEXP pb, SEXP plambda, SEXP pqflag, SEXP prflag,
             SEXP psflag, SEXP pcflag, SEXP pvval, SEXP ptol, SEXP pktol,
             SEXP plambda_count)
#endif
{
    /* problem data */
    dmatrix *matX;      /* feature matrix */
    double  *b;         /* class vector */
    double  lambda;     /* regularization parameter */

    /* solution buffer */
    double  *solution;  /* entry   1    : intercept,    */
                        /* entries 2~n+1: coefficients  */
    int *index;

    train_opts to;
    int     rflag;
    char    *ifile_x, *ifile_y, *ofile;
    double  lambda_max, lambda_min, *lambda_vec;
    clock_t t_start, t_end, t_read, t_write, 
            t_solve, t_solve_start, t_solve_end;

    int count_nt, count_pcg, total_nt, total_pcg;
    int i, m, n, ret;
    int pos_count;
    int lambda_count;
    int count;
    double alpha;
    double t0;
    double *x0;

    /* output file */
    FILE *fp;
    char line[BUFFER_SIZE];
    char name[BUFFER_SIZE];
    int  tmp_int1, tmp_int2, tmp_int3;

#ifdef Rpackage
    buffer_t *buf_val;  /* val */
    buffer_t *buf_jdx;  /* jdx */
    buffer_t *buf_rdx;  /* rdx */

    buffer_new(&buf_val, sizeof(double), BUFFER_INC_SIZE);
    buffer_new(&buf_jdx, sizeof(int), BUFFER_INC_SIZE);
    buffer_new(&buf_rdx, sizeof(int), BUFFER_INC_SIZE);

#endif

    t_start = clock();

    /* read data */
#ifndef Rpackage
    parse_command_line_args(argc, argv, &lambda, &lambda_count, &to,
                            &rflag, &ifile_x, &ifile_y, &ofile);

    if (to.verbose_level>=2) fprintf(stderr,"\nReading data...\n\n");
    read_mm_new_matrix(ifile_x, &matX);
    read_mm_new_vector(ifile_y, &b);
#else
    convert_Rdata(pm,pn,pnz,pval,pjdx,prdx,pb,plambda,prflag,
                  psflag,pcflag,pqflag,pvval,ptol,pktol,plambda_count,
                  &matX,&b,&lambda,&lambda_count,&to,&rflag );
#endif
    m = matX->m;
    n = matX->n;

    /* compute lambda_max, lambda_min, and lambda_vec */
    lambda_max = find_lambdamax(matX, b, to.sflag);
    if (to.verbose_level>=2)
        summary_prob(matX, to.sflag, lambda_max, lambda);

    if (!rflag && lambda_max <= lambda)
    {
        fprintf(stderr,"ERROR: lambda should be less than lambda_max.\n");
        fprintf(stderr,"       check '-r' option.\n");
        exit(EXIT_FAILURE);
    }
    lambda_vec = malloc(sizeof(double)*lambda_count);
    for (i = 0; i < lambda_count; i++)
    {
        lambda_vec[i] = (rflag) ?
                lambda_max*pow(lambda, (float)i/(lambda_count-1)) :
                lambda_max*pow(lambda/lambda_max, (float)i/(lambda_count-1));
    }
    if (rflag)
        lambda_min = lambda*lambda_max;
    else
        lambda_min = lambda;

    t_end   = clock();
    t_read  = t_end - t_start;
    t_start = t_end;

    solution = malloc(sizeof(double)*(n+1));
#ifdef Rpackage
    index    = malloc(sizeof(int)*(n+1));
#endif

    /* prepare initial data */
    pos_count = 0;
    for (i = 0; i < m; i++) 
        if (b[i] > 0) pos_count++;

    t0 = 2.0*n/to.tolerance;
    x0 = malloc(sizeof(double)*(2*n+1));
    x0[0] = log(((double)(m-pos_count))/pos_count); /* initialize v */
    dmat_vset(n, 0.0, x0+1);                        /* initialize w */
    alpha = 1.0/(t0*lambda_max);
    dmat_vset(n, 2*alpha, x0+1+n);                  /* initialize u */

#ifndef Rpackage
    /* run solver and write solution */
    if ((fp = fopen(ofile, "w+")) == NULL)
    {
        fprintf(stderr,"ERROR: Could not open file: %s\n",ofile);
        exit(1);
    }
    /* note: number of nonzero elements is not know at this point.
             thus, use a huge number and replace it at the end */
    sprintf(line, comment1, ifile_x, PACKAGE_NAME, VERSION, ofile);

    if (to.cflag)
        write_mm_matrix_crd_header(fp, n, lambda_count, 999999999, line);
    else
        write_mm_matrix_crd_header(fp, n+1, lambda_count, 999999999, line);
#endif

    count       = 0;
    t_solve     = 0;
    total_nt    = 0;
    total_pcg   = 0;

    for (i = 0; i < lambda_count; i++)
    {
        int trial;

        if (to.verbose_level>=2) fprintf(stderr,"    lambda = %e\n",lambda_vec[i]);
        t_solve_start = clock();

        /* solve problem for each lambda */
        /* if t0 is too high, make it smaller and adjust u */
        if (t0 > 2.0*n/to.tolerance)
        {
            int j;
            double *w, *u;
            double alpha, alpha2;

            w  = x0+1;
            u  = x0+1+n;
            t0 = 2.0*n/to.tolerance;
            alpha  = 1.0/(t0*lambda_vec[i]);
            alpha2 = alpha*alpha;
            for (j = 0; j < n; j++)
                u[j] = alpha + sqrt(alpha2+w[j]*w[j]);
        }

        /* try again if BLS error occurs */
        for (trial = 0; trial < MAX_TRIAL; trial++)
        {
            double tin = t0;
            ret = l1_logreg_train(matX, b, lambda_vec[i], to , x0, &t0,
                                  solution, &count_nt, &count_pcg);
            if (ret == STATUS_SOLUTION_FOUND) break;

            t0 = tin/2;
        }

        t_solve_end = clock();
        t_solve += t_solve_end - t_solve_start;

        /*
         *  write coefficients
         */
        if (to.cflag)
        {
            /* only coefficients */
            #ifndef Rpackage
            count += write_mm_matrix_crd_column(fp, n, i, solution+1);
            #else
            int cur_len;
            int rowidx = count+1;
            cur_len = condense_solution(n, solution, index);
            buffer_write(buf_rdx, &rowidx , 1      );
            buffer_write(buf_jdx, index   , cur_len);
            buffer_write(buf_val, solution, cur_len);
            count += cur_len;
            #endif
        }
        else
        {
            /* intercept + coefficients(/std.dev.) */
            #ifndef Rpackage
            count += write_mm_matrix_crd_column(fp, n+1, i, solution);
            #else
            int cur_len;
            int rowidx = count+1;
            cur_len = condense_solution(n+1, solution, index);
            buffer_write(buf_rdx, &rowidx , 1      );
            buffer_write(buf_jdx, index   , cur_len);
            buffer_write(buf_val, solution, cur_len);
            count += cur_len;
            #endif
        }


        if (to.verbose_level>=2)
        {
            fprintf(stderr,"    [NT  iterations]    %10d\n",count_nt);
            if (matX->nz >= 0)
                fprintf(stderr,"    [PCG iterations]    %10d\n",count_pcg);
            fprintf(stderr,"    [execution time]    %10.3g (sec)\n",
                    (double)(t_solve_end-t_solve_start)/CLOCKS_PER_SEC);
            fprintf(stderr,"\n");
        }
        total_nt  += count_nt;
        total_pcg += count_pcg;
    }
#ifndef Rpackage
    fflush(fp);
    rewind(fp);
    while(fgetc(fp)=='%')
    {
        fgets(line,BUFFER_SIZE,fp);
    }
    fseek(fp, -1, SEEK_CUR);
    fscanf(fp, "%d %d %d", &tmp_int1, &tmp_int2, &tmp_int3);
    fseek(fp, -9, SEEK_CUR);
    fprintf(fp, "%9d", count);
    fclose(fp);

    /* write lambda vector */
    if (strcmp(ofile,"/dev/null")) // if ofile is not /dev/null,
    {
        sprintf(name, "%s_lambda", ofile);
        sprintf(line, comment2, ifile_x, PACKAGE_NAME, VERSION, ofile);
        write_mm_vector(name, lambda_count, lambda_vec, line, TYPE_E);
    }

    t_end   = clock();
    t_write = t_end - t_start - t_solve;

    /* print info */
    if (to.verbose_level>=2)
        summary_time(t_read, t_solve, t_write);
#else
    /* write final row index */
    {
        int rowidx = count+1;
        buffer_write(buf_rdx, &rowidx, 1);
    }
#endif

    if (to.verbose_level==1)
        summary_all(m, n, lambda_count, lambda_min, lambda_max,
                    t_solve, total_nt, total_pcg, solution+1);

#ifdef Rpackage
    if (i < lambda_count)
        return R_NilValue; /* error */
    else
    {
        SEXP res;
        res = create_Rdata_to_return(count, lambda_count, n, lambda_vec,
                                     buf_val, buf_jdx, buf_rdx);
        /* do not free matX->val */
        if (matX->nz >= 0) {
            free(matX->idx);
            free(matX->jdx);
            free(matX->rdx);
        }
        free(matX);
        free(x0);
        free(index);
        free(solution);
        free(lambda_vec);
        buffer_free(buf_val);
        buffer_free(buf_jdx);
        buffer_free(buf_rdx);
        return res;
    }
#else
    dmat_free(matX);
    free(b);
    free(x0);
    free(solution);
    free(lambda_vec);

    return EXIT_SUCCESS;
#endif
}
