package edu.vanderbilt.masi.LabelFusion;
import java.util.Vector;

// The ObservationSlice Class allows for raters to observe slices individually, and not necessarily observe
// all slices in the final output volume. In other words, this allows for incomplete, and over-complete
// observations.
public class ObservationSlice extends ObservationBase {
	
	// fields
	private Vector<int [][]> [][] data; // raters by slices by x by y by num_obs
	private byte [][] is_observed; 
	
	// DataVolume Constructor
	public ObservationSlice (int [][][] data_in, 
					  		 int [] slice_arr_in,
					  		 int [] rater_arr_in,
					  		 int [] dims_in) {
		
		
		super.dims = dims_in;
		set_num_raters(rater_arr_in, slice_arr_in);
		construct_data(slice_arr_in, rater_arr_in, data_in);
		this.set_num_labels();
	}
	
	// construct the data array
	@SuppressWarnings("unchecked")
	private void construct_data(int [] slice_arr, 
								int [] rater_arr,
								int [][][] data_in) {
		
		// first allocate space for the data
		data = new Vector [num_raters][dims[2]];
		for (int r = 0; r < num_raters; r++)
			for (int z = 0; z < dims[2]; z++)
				data[r][z] = new Vector<int [][]>();
		is_observed = new byte [num_raters][dims[2]];
		
		// populate the matrix
		for (int i = 0; i < rater_arr.length; i++) {
			
			// get the observation
			int [][] obs = new int [dims[0]][dims[1]];
			for (int x = 0; x < dims[0]; x++)
				for (int y = 0; y < dims[1]; y++)
					obs[x][y] = data_in[x][y][i];
				
			data[rater_arr[i]][slice_arr[i]].add(obs);
			is_observed[rater_arr[i]][slice_arr[i]] += 1;
		}
	}
	
	// set the number of raters
	private void set_num_raters(int [] rater_arr, int [] slice_arr) {
		
		// find the maximum rater number
		int max_rater_num = -1;
		for (int i = 0; i < rater_arr.length; i++)
			if (rater_arr[i] > max_rater_num)
				max_rater_num = rater_arr[i];
		
		// the number of raters is then the maximum 
		// rater number plus 1 (starts from zero)
		super.num_raters = max_rater_num + 1;
	}
	
	private void set_num_labels() {
		int max_label_num = -1;
		for (int r = 0; r < super.num_raters; r++)
			for (int z = 0; z < super.dims[2]; z++) {
				for (int i = 0; i < (int) is_observed[r][z]; i++)
					for (int x = 0; x < super.dims[0]; x++)
						for (int y = 0; y < super.dims[1]; y++)
		                    if (data[r][z].get(i)[x][y] > max_label_num)
		                    	max_label_num = data[r][z].get(i)[x][y];
			}
		super.num_labels = max_label_num + 1;
	}
	
	// get_vote takes in coordinates and a rater and returns the associated label
	public int [] get_vote(int x, int y, int z, int r) {
		
		int size = (int) is_observed[r][z];
		// check to verify that rater r observed slice z
		if (size > 0) {
			
			// if they did return all of the votes
			int [] votes = new int [size];
			for (int i = 0; i < size; i++)
				votes[i] = data[r][z].get(i)[x][y];
			return(votes);
		
		} else {
			// if not, return null
			return(null);
		}
	}

	public void iterate_votes(LabelFusionAction lfa) {
		for (int r = 0; r < super.num_raters; r++)
			for (int z = 0; z < super.dims[2]; z++) {
				for (int i = 0; i < (int) is_observed[r][z]; i++)
					for (int x = 0; x < super.dims[0]; x++)
						for (int y = 0; y < super.dims[1]; y++)
							lfa.run(x, y, z, r, data[r][z].get(i)[x][y]);
			}
	}
	
	public void iterate_voxel(LabelFusionAction lfa,
							  int x,
							  int y,
							  int z) {
		
		for (int r = 0; r < super.num_raters; r++)
			for (int i = 0; i < (int) is_observed[r][z]; i++) {
				lfa.run(x, y, z, r, data[r][z].get(i)[x][y]);
			}
	}
	
	public void iterate_region(LabelFusionAction lfa,
							   int xs,
							   int ys,
							   int zs,
							   int xe,
							   int ye,
							   int ze) {
		
		for (int r = 0; r < super.num_raters; r++)
			for (int z = zs; z <= ze; z++) {
				for (int i = 0; i < (int) is_observed[r][z]; i++) {
					for (int x = xs; x <= xe; x++)
						for (int y = ys; y <= ye; y++)
							lfa.run(x, y, z, r, data[r][z].get(i)[x][y]);
				}
			}
	}

}