function [cmds chk_files] = ...
                        run_aladin_registration(target, atlas, labels, ...
                                                out_prefix, out_type, ...
                                                out_warp_dir, out_seg_dir, ...
                                                out_atlas_dir, out_temp_dir, ...
                                                regloc, opts);
% RUN_ALADIN_REGISTRATION - runs the aladin registration algorithm between
%                           a given target and an atlas with labels
%
% [cmds chk_files] = run_aladin_registration(target, atlas, labels, ...
%                                            out_prefix, out_type, ...
%                                            out_warp_dir, out_seg_dir, ...
%                                            out_atlas_dir, out_temp_dir, ...
%                                            regloc, opts);
%
% Input: target - the target filename
%        atlas - the atlas filename
%        labels - the atlas labels filename
%        out_prefix - the output prefix for the filenames
%        out_type - the output filetype (1 - .nii, 2 - .nii.gz)
%        out_warp_dir - the output directory for the warp files
%        out_seg_dir - the output directory for the segmentation files
%        out_atlas_dir - the output directory for the registred atlas files
%        out_temp_dir - the output directory to store the temporary files
%        regloc - the location of the registration application
%        opts - the registration-specific options
%
% Output: cmds - a cell array of commands to run
%         chk_files - a cell array of files that should be there when done

% set some filenames
wrp_name = sprintf('%s/%s_affinemat.mtx', out_warp_dir, out_prefix);
if (out_type == 1)
    seg_name = sprintf('%s/%s_seg.nii', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii', out_atlas_dir, out_prefix);
else
    seg_name = sprintf('%s/%s_seg.nii.gz', out_seg_dir, out_prefix);
    reg_name = sprintf('%s/%s.nii.gz', out_atlas_dir, out_prefix);
end

% set the chk_files
chk_files{1} = wrp_name;
chk_files{2} = seg_name;
chk_files{3} = reg_name;

% set the default options
if ~isfield(opts, 'addopts')
    opts.addopts = '';
end
if isfield(opts, 'rmaskdir')
    bname = get_basename(target);
    fname = get_fnames_dir(opts.rmaskdir, ['*', bname, '*.nii.gz']);
    if length(fname) == 0
        warning('no mask found for %s', bname);
    else
        opts.addopts = [opts.addopts, ' -rmask ', fname{end}];
    end
end
if isfield(opts, 'fmaskdir')
    bname = get_basename(atlas);
    fname = get_fnames_dir(opts.fmaskdir, ['*', bname, '*.nii.gz']);
    if length(fname) == 0
        warning('no mask found for %s', bname);
    else
        opts.addopts = [opts.addopts, ' -fmask ', fname{end}];
    end
end


cmds = {};

% run the aladin command
cmds{1} = sprintf('%s/reg_aladin -ref %s -flo %s -aff %s -res %s %s\n', ...
                  regloc, target, atlas, wrp_name, reg_name, opts.addopts);

cmds{2} = sprintf('%s/reg_resample -ref %s -flo %s -aff %s -res %s -NN', ...
                  regloc, target, labels, wrp_name, seg_name);

