function [cmds chk_files] = run_art_registration(target, atlas, labels, ...
                                                 out_prefix, out_type, ...
                                                 out_warp_dir, out_seg_dir, ...
                                                 out_atlas_dir, out_temp_dir,...
                                                 regloc, opts);
% RUN_ART_REGISTRATION - runs the ART registration algorithm between
%                         a given target and an atlas with labels
%
% [cmds chk_files] = run_art_registration(target, atlas, labels, ...
%                                         out_prefix, out_type, ...
%                                         out_warp_dir, out_seg_dir, ...
%                                         out_atlas_dir, out_temp_dir, ...
%                                         regloc, opts);
%
% Input: target - the target filename
%        atlas - the atlas filename
%        labels - the atlas labels filename
%        out_prefix - the output prefix for the filenames
%        out_type - the output filetype (1 - .nii, 2 - .nii.gz)
%        out_warp_dir - the output directory for the warp files
%        out_seg_dir - the output directory for the segmentation files
%        out_atlas_dir - the output directory for the registred atlas files
%        out_temp_dir - the output directory to store the temporary files
%        regloc - the location of the registration application
%        opts - the registration-specific options
%
% Output: cmds - a cell array of commands to run
%         chk_files - a cell array of files that should be there when done
%

% set some filenames
wrp_name = sprintf('%s/%s_warp.nii', out_warp_dir, out_prefix);
seg_name = sprintf('%s/%s_seg.nii', out_seg_dir, out_prefix);
reg_name = sprintf('%s/%s.nii', out_atlas_dir, out_prefix);

if out_type == 1
    chk_files{1} = wrp_name;
    chk_files{2} = seg_name;
    chk_files{3} = reg_name;
else
    chk_files{1} = [wrp_name, '.gz'];
    chk_files{2} = [seg_name, '.gz'];
    chk_files{3} = [reg_name, '.gz'];
end

if files_exist(chk_files)
    cmds = {};
    return;
end

% set the default options
if ~isfield(opts, 'addopts')
    opts.addopts = '-sd 8.0 -A';
end

% set some filenames
sub = [out_temp_dir, 'source.nii'];
trg = [out_temp_dir, 'target.nii'];
lab = [out_temp_dir, 'labels.nii'];

% first copy the files over
if strcmp(atlas(end-2:end), '.gz')
    system(sprintf('gunzip -c %s > %s', atlas, sub));
else
    system(sprintf('cp %s %s', atlas, sub));
end
if strcmp(labels(end-2:end), '.gz')
    system(sprintf('gunzip -c %s > %s', labels, lab));
else
    system(sprintf('cp %s %s', labels, lab));
end
if strcmp(target(end-2:end), '.gz')
    system(sprintf('gunzip -c %s > %s', target, trg));
else
    system(sprintf('cp %s %s', target, trg));
end

% convert to the art datatype
convert_datatype_art(sub, sub);
convert_datatype_art(trg, trg);
convert_datatype_art(lab, lab, 1);

% set the locations of the ART executables
art_reg = [regloc, '/bin/3dwarper'];
art_apply = [regloc, '/bin/applywarp3d'];

cmds = {};

% run the 3dwarper command
cmds{1} = sprintf(['export ARTHOME=%s && %s -v -sub %s -trg %s -u %s ', ...
                  '-o %s %s\n'], regloc, art_reg, sub, trg, wrp_name, ...
                  reg_name, opts.addopts);

% run the applywarp3d command
cmds{2} = sprintf('export ARTHOME=%s && %s -nn -o %s -w %s %s\n', ...
                 regloc, art_apply, seg_name(1:end-4), wrp_name, lab);

% delete the large temporary files
cmds{3} = sprintf('rm %s %s %s %s*_ACPC*\n', sub, trg, lab, out_temp_dir);

if out_type == 2
    cmds{4} = sprintf('gzip %s\n', reg_name);
    cmds{5} = sprintf('gzip %s\n', wrp_name);
    cmds{6} = sprintf('gzip %s\n', seg_name);
end


