function [strip_dir] = run_bet(targets, out_dir, betloc, runtype, varargin)
% RUN_BET - runs BET (brain extraction) on a cell array of filenames
%
% strip_dir = run_bet(targets, out_dir, betloc, runtype)
%
% Input: targets - a cell array of target filenames
%        out_dir - the output directory
%        betloc - the location of the bet application
%        runtype - either 'cluster' or 'single'
%        opts - (optional) a struct of options
%
% Output: strip_dir - the output strip directory

% make sure the output directory exists
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

if length(varargin) == 0
    opts = struct;
else
    opts = varargin{1};
end

% set the default options
if ~isfield(opts, 'memval')
    opts.memval = '2G';
end
if ~isfield(opts, 'pbsfile')
    opts.pbsfile = sprintf('bet-pbsfile.pbs');
end
if ~isfield(opts, 'txtout')
    opts.txtout = sprintf('bet-txtout.txt');
end
if ~isfield(opts, 'addopts')
    opts.addopts = '-R';
end

% set the output directories
strip_dir = sprintf('%s/bet-targets/', out_dir);
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(strip_dir, 'dir')
    mkdir(strip_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end

done_list = zeros([length(targets) 1]);
chk_files = cell([length(targets) 1]);

for i = 1:length(targets)

    if ~exist(targets{i}, 'file')
        error(sprintf('Target file %s does not exist', targets{i}));
    end

    [t_dir t_name t_ext] = fileparts(targets{i});

    % set the output file
    out_file = sprintf('%s/%s%s', strip_dir, t_name, t_ext);

    % set the output type
    out_type = 1;

    % account for the .gz suffix
    if strcmp(t_name(end-3:end), '.nii')
        t_name = t_name(1:end-4);
        out_type = 2;
    end

    % set the output prefix
    out_prefix = sprintf('%s', t_name);

    % set the temporary output directory
    out_temp_dir = sprintf('%s/%s-%s/', temp_dir, 'bet', out_prefix);
    if ~exist(out_temp_dir, 'dir')
        mkdir(out_temp_dir);
    end

    if out_type == 1
        outstr = 'export FSLOUTPUTTYPE=NIFTI';
    else
        outstr = 'export FSLOUTPUTTYPE=NIFTI_GZ';
    end

    % run the flirt command
    cmds{1} = sprintf('%s && %s %s %s -v %s\n', outstr, betloc, targets{i}, ...
                      out_file, opts.addopts);

    % make sure it needs to be run
    chk_files{i}{1} = out_file;
    if files_exist(chk_files{i})
        disp(sprintf('Skipping %s: output files exist', out_prefix));
        pause(0.01);
        done_list(i) = 1;
        continue;
    end

    % run the commands
    if strcmp(runtype, 'cluster')
        pbsout = [out_temp_dir, opts.pbsfile];
        txtout = [out_temp_dir, opts.txtout];
        run_cmd_cluster(cmds, opts.memval, pbsout, txtout);
        done_list(i) = 0;
    else
        run_cmd_single(cmds);
        done_list(i) = 1;
    end
end

% iterate until done
while (min(done_list) == 0)
    for i = 1:length(done_list)
        if (done_list(i) == 0)
            if files_exist(chk_files{i})
                done_list(i) = 1;
            end
        end
    end
    pause(1);
end

