function run_cmd_matlab_cluster(cmds, memval, out_tmp_dir, outprefix)
% RUN_CMD_MATLAB_CLUSTER - runs a collection of MATLAB commands on the local
%                          cluster
%
% run_cmd_matlab_cluster(cmds, memval, out_tmp_dir, outprefix)
%
% Input: cmds - the cell array of commands
%        memval - the amount of memory to allocate for the job
%               - Note: there are three values that can be set with this
%                       variable (mem_free, mem_token, and h_vmem).
%                       1) mem_free - the amount of memory needed to start job
%                       2) mem_token - amount to (virtually) allocate at start
%                       3) h_vmem - the absolute limit this job can use
%                                   (in terms of virtual memory) -
%                                   see the "VIRT" column in "top"
%               - As a result, can take on 4 different forms:
%                 1) a string - which results in:
%                             mem_free = memval;
%                             mem_token = memval;
%                             h_vmem = memval;
%                 2) a one element cell array
%                             mem_free = memval{1};
%                             mem_token = memval{1};
%                             h_vmem = memval{1};
%                 3) a two element cell array
%                             mem_free = memval{1};
%                             mem_token = memval{1};
%                             h_vmem = memval{2};
%                 4) a three element cell array
%                             mem_free = memval{1};
%                             mem_token = memval{3};
%                             h_vmem = memval{2};
%        out_tmp_dir - the temporary output directory
%        outprefix - the prefix used for the output files (pbs, txt, and mfile)
%
% Output: (NONE)

if length(cmds) == 0
    return;
end

% set the memory values
memval_memfree = '';
memval_memtoken = '';
memval_hvmem = '';
if isstr(memval)
    memval_memfree = memval;
    memval_memtoken = memval;
    memval_hvmem = memval;
elseif iscell(memval)
    if length(memval) == 1
        memval_memfree = memval{1};
        memval_memtoken = memval{1};
        memval_hvmem = memval{1};
    elseif length(memval) == 2
        memval_memfree = memval{1};
        memval_memtoken = memval{1};
        memval_hvmem = memval{2};
    elseif length(memval) == 3
        memval_memfree = memval{1};
        memval_memtoken = memval{3};
        memval_hvmem = memval{2};
    else
        error('Invalid "memval" value');
    end
else
    error('Invalid "memval" value');
end

% set the output filenames
txtout = sprintf('%s/%s.txt', out_tmp_dir, outprefix);
pbsout = sprintf('%s/%s.pbs', out_tmp_dir, outprefix);
mfileout = sprintf('%s/%s.m', out_tmp_dir, outprefix);
xvfb_fname = sprintf('/tmp/xvfb-run_%s.auth', outprefix);
null_fname = sprintf('/tmp/null_%s.err', outprefix);

% create the cluster prefix
cluster_prefix = sprintf(['xvfb-run -f %s -e %s -a ', ...
                          '--server-args="-screen 0 1600x1280x24 -ac ', ...
                          '-extension GLX"'], xvfb_fname, null_fname);

% make the m-file
fid = fopen(mfileout, 'w');
for i = 1:length(cmds)
    if strcmp(cmds{i}(end), sprintf('\n'))
        fprintf(fid, '%s', cmds{i});
    else
        fprintf(fid, '%s\n', cmds{i});
    end
end
fclose(fid);

% make the pbs file
fid = fopen(pbsout,'w');
fprintf(fid, '#!/bin/bash\n');

fprintf(fid, '#$ -q clusterjob\n');
fprintf(fid, '#$ -j y\n');
fprintf(fid, '#$ -cwd\n');
fprintf(fid, '#$ -o %s\n', txtout);
fprintf(fid, '#$ -l mem_free=%s\n', memval_memfree);
fprintf(fid, '#$ -l mem_token=%s\n', memval_memtoken);
fprintf(fid, '#$ -l h_vmem=%s\n\n', memval_hvmem);

fprintf(fid, 'uname -a\n');
fprintf(fid, '%s matlab -nodisplay < %s\n', cluster_prefix, mfileout);

fclose(fid);

% submit the job
system(sprintf('qsub %s', pbsout));
