function [fusion_dir fusion_fnames] = run_fusions(atlases, labels, targets, ...
                                                  out_dir, seg_dir, ...
                                                  atlas_dir, fuseloc, ...
                                                  fusetype, runtype, varargin)
% RUN_FUSIONS - runs label fusion algorithm
%
% fusion_dir = run_fusions(atlases, labels, targets, ...
%                          out_dir, seg_dir, atlas_dir, ...
%                          fuseloc, fusetype, runtype, varargin)
%
% Input: atlases - a cell array of atlas filenames
%        labels - a cell array of atlas label filenames
%        targets - a cell array of target filenames
%        out_dir - the output directory
%        seg_dir - the registered labels directory
%        atlas_dir - the regisered images directory
%        fuseloc - the location of the fusion algorithm
%        runtype - either 'cluster' or 'single'
%        opts - (optional) a struct of options
%
% Output: fusion_dir
%         fusion_fnames (optional)

% make sure the output directory exists
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

if length(varargin) == 0
    opts = struct;
else
    opts = varargin{1};
end

% set the default options
if ~isfield(opts, 'use_xvfb')
    if strcmp(runtype, 'cluster')
        opts.use_xvfb = true;
    else
        opts.use_xvfb = false;
    end
end
if ~isfield(opts, 'memval'), opts.memval = '2G'; end
if ~isfield(opts, 'fusename'), opts.fusename = fusetype; end
if ~isfield(opts, 'pbsfile')
    opts.pbsfile = sprintf('%s-pbsfile.pbs', fusetype);
end
if ~isfield(opts, 'txtout')
    opts.txtout = sprintf('%s-txtout.txt', fusetype);
end
if ~isfield(opts, 'waitdone')
    opts.waitdone = true;
end

% set the output directories
fusion_dir = sprintf('%s/fusion-%s/', out_dir, opts.fusename);
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(fusion_dir, 'dir')
    mkdir(fusion_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end

done_list = zeros([length(targets) 1]);
chk_files = cell([length(targets) 1]);

% do all of the pairwise registrations
for j = 1:length(targets)

    % do some error checking
    if ~exist(targets{j}, 'file')
        error(sprintf('Target file %s does not exist', targets{j}));
    end

    % set the output file information
    sfix = '.nii';
    [t_dir t_name t_ext] = fileparts(targets{j});
    if strcmp(t_name(end-3:end), '.nii')
        t_name = t_name(1:end-4);
        sfix = '.nii.gz';
    end

    % set the atlas names
    cc = 0;
    reg_labels = {};
    reg_ims = {};
    for a = 1:length(atlases)
        [s_dir s_name s_ext] = fileparts(atlases{a});
        if strcmp(s_name(end-3:end), '.nii')
            s_name = s_name(1:end-4);
        end
        if ~strcmp(s_name, t_name)
            cc = cc+1;
            pp = sprintf('%s_via_%s', t_name, s_name);
            reg_labels{cc} = sprintf('%s/%s_seg%s', seg_dir, pp, sfix);
            if ~exist(reg_labels{cc}, 'file')
                error(sprintf('Registered labels file %s does not exist', ...
                              reg_labels{cc}));
            end
            reg_ims{cc} = sprintf('%s/%s%s', atlas_dir, pp, sfix);
            if ~exist(reg_ims{cc}, 'file')
                error(sprintf('Registered atlas file %s does not exist', ...
                              reg_ims{cc}));
            end
        end
    end

    % create the output directory
    out_temp_dir = sprintf('%s/fusion-%s-%s/', temp_dir, opts.fusename, t_name);
    if ~exist(out_temp_dir, 'dir')
        mkdir(out_temp_dir);
    end

    % set the output fused file
    fuse_file = sprintf('%s%s_est%s', fusion_dir, t_name, sfix);
    fusion_fnames{j} = fuse_file;

    % let the user know what is going on
    tprintf('-> Running Fusion\n');
    tprintf('Fusion Type: %s\n', fusetype);
    tprintf('Mipav Location: %s\n', fuseloc);
    tprintf('Target: %s\n', targets{j});
    tprintf('Number of Atlases: %d\n', length(reg_ims));
    tprintf('Output Fused Segmentation: %s\n', fuse_file);

    % make sure it needs to be run
    chk_files{j}{1} = fuse_file;
    if files_exist(chk_files{j})
        tprintf('Skipping fusion: %s (output files exist)\n', t_name);
        tprintf('\n');
        pause(0.01);
        done_list(j) = 1;
        continue;
    end

    cmds = {};
    opts.mipav = fuseloc;
    if strcmp(fusetype, 'MajorityVote')
        cmds = run_voting_fusion_jist(fuse_file, reg_labels, opts);
    elseif strcmp(fusetype, 'WeightedVote')
        cmds = run_voting_fusion_jist(fuse_file, reg_labels, targets{j}, ...
                                      reg_ims, opts);
    elseif strcmp(fusetype, 'STAPLE')
        opts.sv_bias = 1;
        cmds = run_statistical_fusion_jist(fuse_file, reg_labels, opts);
    elseif strcmp(fusetype, 'SpatialSTAPLE')
        cmds = run_statistical_fusion_jist(fuse_file, reg_labels, opts);
    elseif strcmp(fusetype, 'NonLocalSTAPLE')
        opts.sv_bias = 1;
        cmds = run_statistical_fusion_jist(fuse_file, reg_labels, ...
                                           targets{j}, reg_ims, opts);
    elseif strcmp(fusetype, 'NonLocalSpatialSTAPLE')
        cmds = run_statistical_fusion_jist(fuse_file, reg_labels, ...
                                           targets{j}, reg_ims, opts);
    elseif strcmp(fusetype, 'brainmask')
        cmds = run_brainmask_fusion(reg_labels, fuse_file, opts);
    else
        error(sprintf('Unkown fusion type: %s', fusetype));
    end

    % enable running jist on the cluster
    if opts.use_xvfb && length(cmds) == 1
        bname = [fusetype, '_', t_name];
        cmds = prepend_jist_cluster_prefix(cmds, 0.5*j, '/tmp', bname);
    end

    % run the commands
    if (length(cmds) > 0)
        txtout = [out_temp_dir, opts.txtout];
        if strcmp(runtype, 'cluster')
            pbsout = [out_temp_dir, opts.pbsfile];
            run_cmd_cluster(cmds, opts.memval, pbsout, txtout);
            done_list(j) = 0;
        else
            run_cmd_single(cmds, txtout);
            tprintf('Fusion finished (see %s)\n', txtout);
            done_list(j) = 1;
        end
    else
        done_list(j) = 1;
    end

    tprintf('\n');

end

% iterate until done
if (opts.waitdone)
    while (min(done_list) == 0)
        for i = 1:length(done_list)
            if (done_list(i) == 0)
                if files_exist(chk_files{i})
                    done_list(i) = 1;
                end
            end
        end
        pause(1);
    end
end

