function [out_atlas_dir out_seg_dir out_warp_dir chk_files] = ...
                        run_registrations(atlases, labels, targets, out_dir, ...
                                          regtype, runtype, regloc, varargin)
% RUN_REGISTRATIONS - performs a pairwise registration between a collection
%                     of atlases and a collection of targets for
%                     multi-atlas segmentation
%
% Two Forms:
% 1) [out_atlas_dir out_seg_dir out_warp_name] = ...
%                    run_registrations(atlases, labels, targets, out_dir, ...
%                                      regtype, runtype, regloc);
% 2) [out_atlas_dir out_seg_dir out_warp_name] = ...
%                    run_registrations(atlases, labels, targets, out_dir, ...
%                                      regtype, runtype, regloc, opts);
%
% Input: atlases - a cell array of (full-path) atlas intensity images
%        labels - a cell array of (full-path) atlas label images
%        targets - a cell array of (full-path) target intensity images
%        out_dir - the directory to save all output
%        regtype - the registration algorithm to use
%        runtype - either 'cluster' or 'single'
%        regloc - the location of the registration application
%        opts - (optional) a struct of options
%
% Output: out_atlas_dir - the registered atlas directory
%         out_seg_dir - the registered labels directory
%         out_warp_dir - the registration warps directory

% make sure the output directory exists
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

if length(varargin) == 0
    opts = struct;
else
    opts = varargin{1};
end

regtype = lower(regtype);

% set the default options
if ~isfield(opts, 'memval')
    opts.memval = {'2G', '5G'};
end
if ~isfield(opts, 'pbsfile')
    opts.pbsfile = sprintf('%s-pbsfile.pbs', regtype);
end
if ~isfield(opts, 'txtout')
    opts.txtout = sprintf('%s-txtout.txt', regtype);
end
if ~isfield(opts, 'waitdone')
    opts.waitdone = true;
end
if ~isfield(opts, 'skipsame')
    opts.skipsame = false;
end

% set the output directories
out_atlas_dir = sprintf('%s/%s-reg-images/', out_dir, regtype);
out_seg_dir = sprintf('%s/%s-reg-labels/', out_dir, regtype);
out_warp_dir = sprintf('%s/%s-reg-warps/', out_dir, regtype);
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(out_atlas_dir, 'dir')
    mkdir(out_atlas_dir);
end
if ~exist(out_seg_dir, 'dir')
    mkdir(out_seg_dir);
end
if ~exist(out_warp_dir, 'dir')
    mkdir(out_warp_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end

done_list = zeros([length(atlases)*length(targets), 1]);
chk_files = cell([length(atlases)*length(targets), 1]);
count = 1;

% do all of the pairwise registrations
for j = 1:length(targets)

    if ~exist(targets{j}, 'file')
        error(sprintf('Target file %s does not exist', targets{j}));
    end

    for i = 1:length(atlases)

        % do some error checking
        if ~exist(atlases{i}, 'file')
            error(sprintf('Atlas file %s does not exist', atlases{i}));
        end
        if ~exist(labels{i}, 'file')
            error(sprintf('Labels file %s does not exist', labels{i}));
        end

        % skip this file if it is registering to itself
        if opts.skipsame
            if strcmp(targets{j}, atlases{i})
                continue;
            end
        end

        % set the output prefix
        [s_dir s_name s_ext] = fileparts(atlases{i});
        [t_dir t_name t_ext] = fileparts(targets{j});

        % set the output type
        out_type = 1;

        % account for the .gz suffix
        if strcmp(s_name(end-3:end), '.nii')
            s_name = s_name(1:end-4);
        end
        if strcmp(t_name(end-3:end), '.nii')
            t_name = t_name(1:end-4);
            out_type = 2;
        end

        % set the output prefix
        out_prefix = sprintf('%s_via_%s', t_name, s_name);

        % set the temporary output directory
        out_temp_dir = sprintf('%s/%s-%s/', temp_dir, regtype, out_prefix);
        if ~exist(out_temp_dir, 'dir')
            mkdir(out_temp_dir);
        end

        % let the user know what is going on
        tprintf('-> Running Registration\n');
        tprintf('Registration Type: %s\n', regtype);
        tprintf('Registration Location: %s\n', regloc);
        tprintf('Target Image: %s\n', targets{j});
        tprintf('Atlas Image: %s\n', atlases{i});
        tprintf('Atlas Labels: %s\n', labels{i});
        tprintf('Output Prefix: %s\n', out_prefix);

        % run the registration algorithm
        if strcmp(regtype, 'art')
            [cmds chk_files{count}] = ...
                           run_art_registration(targets{j}, atlases{i}, ...
                                                labels{i}, out_prefix, ...
                                                out_type, out_warp_dir, ...
                                                out_seg_dir, out_atlas_dir, ...
                                                out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'flirt')
            [cmds chk_files{count}] = ...
                         run_flirt_registration(targets{j}, atlases{i}, ...
                                                labels{i}, out_prefix, ...
                                                out_type, out_warp_dir, ...
                                                out_seg_dir, out_atlas_dir, ...
                                                out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'flirt-strip')
            [cmds chk_files{count}] = ...
                   run_flirt_strip_registration(targets{j}, atlases{i}, ...
                                                labels{i}, out_prefix, ...
                                                out_type, out_warp_dir, ...
                                                out_seg_dir, out_atlas_dir, ...
                                                out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'aladin')
            [cmds chk_files{count}] = ...
                        run_aladin_registration(targets{j}, atlases{i}, ...
                                                labels{i}, out_prefix, ...
                                                out_type, out_warp_dir, ...
                                                out_seg_dir, out_atlas_dir, ...
                                                out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'niftyreg')
            [cmds chk_files{count}] = ...
                      run_niftyreg_registration(targets{j}, atlases{i}, ...
                                                labels{i}, out_prefix, ...
                                                out_type, out_warp_dir, ...
                                                out_seg_dir, out_atlas_dir, ...
                                                out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'ants')
            [cmds chk_files{count}] = ...
                         run_ants_registration(targets{j}, atlases{i}, ...
                                               labels{i}, out_prefix, ...
                                               out_type, out_warp_dir, ...
                                               out_seg_dir, out_atlas_dir, ...
                                               out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'ants-strip')
            [cmds chk_files{count}] = ...
                   run_ants_strip_registration(targets{j}, atlases{i}, ...
                                               labels{i}, out_prefix, ...
                                               out_type, out_warp_dir, ...
                                               out_seg_dir, out_atlas_dir, ...
                                               out_temp_dir, regloc, opts);
        elseif strcmp(regtype, 'aladin-ants')
            [cmds chk_files{count}] = ...
                  run_aladin_ants_registration(targets{j}, atlases{i}, ...
                                               labels{i}, out_prefix, ...
                                               out_type, out_warp_dir, ...
                                               out_seg_dir, out_atlas_dir, ...
                                               out_temp_dir, regloc, opts);
        else
            error(sprintf('Unknown registration type: %s', regtype));
        end


        % make sure it needs to be run
        if files_exist(chk_files{count})
            tprintf('Skipping Registration: %s (output files exist)\n', ...
                    out_prefix);
            tprintf('\n');
            pause(0.01);
            done_list(count) = 1;
            count = count + 1;
            continue;
        end

        % run the commands
        txtout = [out_temp_dir, opts.txtout];
        if strcmp(runtype, 'cluster')
            pbsout = [out_temp_dir, opts.pbsfile];
            run_cmd_cluster(cmds, opts.memval, pbsout, txtout);
            done_list(count) = 0;
        else
            run_cmd_single(cmds, txtout);
            done_list(count) = 1;
        end

        % increment the count
        count = count + 1;


        tprintf('\n');

    end
end

if (opts.waitdone)
    % iterate until done
    while (min(done_list) == 0)
        for i = 1:length(done_list)
            if (done_list(i) == 0)
                if files_exist(chk_files{i})
                    done_list(i) = 1;
                end
            end
        end
        pause(1);
    end
end

