function outseg = transfer_labels(targets, segs, new_targets, out_dir, flirtloc)
% TRANSFER_LABELS - transfer labels from an existing target to a new target.
%
% transfer_labels(targets, segs, new_targets, out_dir, flirtloc)
%
% Input: targets - a cell array of (full-path) target images
%        segs - a cell array of (full-path) segmentation images
%        new_targets - a cell array of corresponding (full-path) target images
%        out_dir - the directory to save all output
%        flirtloc - the location of the flirt executable
%
% Output: outseg - a cell array of (full-path) output segmentation images

% make sure the output directory exists
if ~exist(out_dir, 'dir')
    error(sprintf('Output directory %s does not exist', out_dir));
end

% settings
sr = 30;
ca = 15;
fa = 6;
cc = 'normmi';
dof = 6;

% do some error checking
if length(targets) ~= length(new_targets)
    error('Number of targets has to equal the number of new targets');
end
if length(segs) ~= length(targets)
    error('Number of segmentations does not equal number of targets');
end

% set the output directories
out_seg_dir = [out_dir, '/transferred-labels/'];
temp_dir = [out_dir, '/temp-out/'];

% first make sure the output directories exist
if ~exist(out_seg_dir, 'dir')
    mkdir(out_seg_dir);
end
if ~exist(temp_dir, 'dir')
    mkdir(temp_dir);
end
outseg = cell([length(targets) 1]);

% do all of the pairwise registrations
for j = 1:length(targets)

    % do some more error checking
    if ~exist(targets{j}, 'file')
        error(sprintf('Target file %s does not exist', targets{j}));
    end
    if ~exist(segs{j}, 'file')
        error(sprintf('Labels file %s does not exist', segs{j}));
    end
    if ~exist(new_targets{j}, 'file')
        error(sprintf('Reference file %s does not exist', new_targets{j}));
    end

    % set the output filenames
    [t_dir t_name t_ext] = fileparts(new_targets{j});
    if strcmp(t_name(end-3:end), '.nii')
        t_name = t_name(1:end-4);
    end

    % create the output information
    out_temp_dir = sprintf('%s%s_transfer/', temp_dir, t_name);
    if ~exist(out_temp_dir, 'dir')
        mkdir(out_temp_dir);
    end
    out_mat_name = sprintf('%saffine-mat.mtx', out_temp_dir);
    out_seg_name = sprintf('%s%s_est.nii.gz', out_seg_dir, t_name);
    outseg{j} = out_seg_name;

    % skip if already run
    if exist(out_seg_name, 'file')
        disp(sprintf('Skipping %s: output file exists', t_name));
        pause(0.01);
        continue;
    end

    disp(sprintf('Transferring labels to: %s', t_name));

    % create the system commands
    cmd1 = sprintf(['%s -searchrx -%d %d -searchry -%d %d -searchrz -%d %d ',...
                    '-coarsesearch %d -finesearch %d -dof %d -cost %s ', ...
                    '-searchcost %s -in %s -ref %s -omat %s'], flirtloc, ...
                    sr, sr, sr, sr, sr, sr, ca, fa, dof, cc, cc, ...
                    targets{j}, new_targets{j}, out_mat_name);
    cmd2 = sprintf(['%s -interp nearestneighbour -in %s -ref %s -applyxfm ', ...
                    '-init %s -out %s'], flirtloc, segs{j}, new_targets{j}, ...
                    out_mat_name, out_seg_name);

    % run the system commands
    system(['. ~/.bashrc && ', cmd1]);
    system(['. ~/.bashrc && ', cmd2]);

end

