function [WV_est WV_W] = log_odds_locally_weighted_vote(obs, weights, rho)
% LOG_ODDS_LOCALLY_WEIGHTED_VOTE - performs a log odds locally weighted vote
%
% [WV_est WV_W] = log_odds_locally_weighted_vote(obs, weights, rho)
%
% Input: obs - the observation struct
%        weights - the voxelwise weights for each rater (X by Y by Z by R)
%        rho - the decay coefficient for the log-odds approach
%
% Output: WV_est - the log-odds locally weighted vote result
%         WV_W - the log-odds locally weighted vote label probabilities
%
% NOTE: this is an implementation of the label fusion paper by Sabuncu, et al
%       in TMI, 2010

% first calculate the number of labels
unique_labels = [];
for i = 1:obs.num_obs
    unique_labels = union(unique_labels, unique(obs.data{i}));
end
num_labels = length(unique_labels);

% initialize the W estimates
WV_W = zeros([obs.dims num_labels]);

% perform the signed distance transform
for i = 1:obs.num_obs

    % initialize the new observation
    obsnew = ones([obs.dims num_labels]);

    % iterate over every label
    for l = 1:num_labels
        l0 = l-1;
        n1 = length(find(obs.data{i} == l0));
        n2 = length(find(obs.data{i} ~= l0));
        if n1 > 0 && n2 > 0
            distl = bwdist(obs.data{i} == l0);
            distn = bwdist(obs.data{i} ~= l0) - 1;
            distn(distn < 0) = 0;
            obsnew(:, :, :, l) = exp(rho * (distn - distl));
        elseif n1 > 0 && n2 <= 0
            obsnew(:, :, :, l) = 1;
        else
            obsnew(:, :, :, l) = 0;
        end
    end

    % normalize the probabilities
    obsnew = obsnew ./ repmat(sum(obsnew, 4), [1 1 1 num_labels]);

    % add the contribution to the Weighted Vote estimate
    WV_W = WV_W + obsnew .* repmat(weights(:, :, :, i), [1 1 1 num_labels]);

end

% normalize the Locally Weighted vote probabilities
WV_W = WV_W ./ repmat(sum(WV_W, 4), [1 1 1 num_labels]);
% calculate the Locally Weighted Vote label estimate
WV_est = get_estimate_from_label_probabilities(WV_W);

