function [MV_est MV_W] = log_odds_majority_vote(obs, rho)
% LOG_ODDS_MAJORITY_VOTE - performs a log odds majority vote
%
% [MV_est MV_W] = log_odds_majority_vote(obs, rho)
%
% Input: obs - the observation struct
%        rho - the decay coefficient for the log-odds approach
%
% Output: MV_est - the log-odds majority vote result
%         MV_W - the log-odds majority vote label probabilities
%
% NOTE: this is an implementation of the label fusion paper by Sabuncu, et al
%       in TMI, 2010

% first calculate the number of labels
unique_labels = [];
for i = 1:obs.num_obs
    unique_labels = union(unique_labels, unique(obs.data{i}));
end
num_labels = length(unique_labels);

% initialize the W estimates
MV_W = zeros([obs.dims num_labels]);

% perform the signed distance transform
for i = 1:obs.num_obs

    % initialize the new observation
    obsnew = ones([obs.dims num_labels]);

    % iterate over every label
    for l = 1:num_labels
        l0 = l-1;
        n1 = length(find(obs.data{i} == l0));
        n2 = length(find(obs.data{i} ~= l0));
        if n1 > 0 && n2 > 0
            distl = bwdist(obs.data{i} == l0);
            distn = bwdist(obs.data{i} ~= l0) - 1;
            distn(distn < 0) = 0;
            obsnew(:, :, :, l) = exp(rho * (distn - distl));
        elseif n1 > 0 && n2 <= 0
            obsnew(:, :, :, l) = 1;
        else
            obsnew(:, :, :, l) = 0;
        end
    end

    % normalize the probabilities
    obsnew(obsnew == Inf) = 10000;
    obsnew = obsnew ./ repmat(sum(obsnew, 4), [1 1 1 num_labels]);

    % add the contribution to the Majority Vote estimate
    MV_W = MV_W + obsnew;

end

% normalize the majority vote probabilities
MV_W = MV_W ./ repmat(sum(MV_W, 4), [1 1 1 num_labels]);
% calculate the Majority Vote label estimate
MV_est = get_estimate_from_label_probabilities(MV_W);

