/** \file   classify.c
 *  \brief  Classify test data.
 */

#if HAVE_CONFIG_H
#   include "config.h"
#endif
#include "pkgdef.h"

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef Rpackage
#include <R.h>
#include <Rinternals.h>
#endif

#include "def.h"
#include "l1_logreg.h"
#include "util.h"

#define BUFFER_SIZE     1024

static const char comment1[] = {
    "%% \n"
        "%% This is a result file of %s,\n"
        "%%    generated by %s_classify ver %s\n"
        "%% Each row contains a predicted class of corresponding example.\n"
        "%% \n"
};
static const char comment2[] = {
    "%% \n"
        "%% This is a result file of %s,\n"
        "%%    generated by %s_classify ver %s\n"
        "%% Each row contains the number of test error of corresponding lambda.\n"
        "%% \n"
};


static void summary_prob(const dmatrix *M, const int tflag)
{
    fprintf(stderr,"Problem summary:\n");
    if (M->nz >= 0)
        fprintf(stderr,"    [feature matrix]        sparse matrix of "
                "(%d examples x %d features), %d non-zeros\n", M->m, M->n, M->nz);
    else
        fprintf(stderr,"    [feature matrix]        dense matrix of "
                "(%d examples x %d features)\n", M->m, M->n);

    if (tflag == FALSE)
        fprintf(stderr,"    [mode]                  classify only\n\n");
    else
        fprintf(stderr,"    [mode]                  classify and test\n\n");

}


static void summary_result(const dmatrix *M, const double *res, 
        const int error_count)
{
    int i;
    int positive_count = 0;

    fprintf(stderr,"\n");
    fprintf(stderr,"Classification result:\n");

    for (i = 0; i < M->m; i++)
    {
        /* use 0.5 since res[i] can be either binary outcome [+1,-1]
           or probability in (0,1) */
        if (res[i] > 0.5) positive_count++;
    }

    if (error_count >= 0)
    {
        fprintf(stderr,"    [right predic. count]   %d\n", positive_count);
        fprintf(stderr,"    [wrong predic. count]   %d\n", M->m-positive_count);
        fprintf(stderr,"    [test error]            %d / %d = %g\n\n",
                error_count, M->m, (double)error_count/M->m);
    }
    else
    {
        fprintf(stderr,"    [positive class count]  %d\n", positive_count);
        fprintf(stderr,"    [negative class count]  %d\n", M->m-positive_count);
        fprintf(stderr,"\n");
    }
}


static void summary_time(clock_t t0, clock_t t1, clock_t t2, clock_t t3)
{
    fprintf(stderr,"Timing:\n");
    fprintf(stderr,"    [read data]         %10.3g (sec)\n",
            (double)(t1-t0)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [solve problem]     %10.3g (sec)\n",
            (double)(t2-t1)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [write solution]    %10.3g (sec)\n",
            (double)(t3-t2)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [total time]        %10.3g (sec)\n\n",
            (double)(t3-t0)/CLOCKS_PER_SEC);
}


#ifndef Rpackage
static void usage(char *package, char *version) 
{
    static const char *help[] = 
    {
        "\n%s version %s\n",
        "\nUsage: %s_classify [options] model_file feature_file result_file\n",
        "Classify data using a given model\n\n",
        "   Arguments:\n",
        "       model_file          - model data(coefficients and intercept)\n",
        "                             found by either %s_train or\n",
        "                             %s_regpath\n",
        "       feature_file        - feature matrix\n",
        "       result_file         - store classification results to result_file\n",
        "                             if model_file is generated from\n",
        "                              1) %s_train, then predicted outcomes\n",
        "                              2) %s_regpath, then the number of errors\n"
            "                             will be stored.\n",
        "   Options:\n",
        "       -q                  - quiet mode\n",
        "       -p                  - store probability instead of predicted outcome\n",
        "       -t <class_file>     - test classification result\n",
        "                             against real class vector in <class_file>\n",
        "\n",
        NULL,
    };

    const char **p;
    for (p = help; *p != NULL; p++)
        fprintf(stderr,*p, package, version);
    exit (EXIT_SUCCESS);
}


static void parse_command_line_args(int argc, char *argv[], int *tflag,
        int *pflag, int *qflag, char **ifile_x,
        char **ifile_y, char **ifile_model,
        char **ofile)
{
    int c;

    /* default values */
    *ifile_y = NULL;
    *ofile   = NULL;
    *pflag   = FALSE;
    *qflag   = FALSE;
    *tflag   = FALSE;

    while ((c = getopt(argc, argv, "t:pq")) != -1)
    {
        switch (c)
        {
            case 't':
                *ifile_y = optarg;
                *tflag = TRUE;
                break;
            case 'p':
                *pflag = TRUE;
                break;
            case 'q':
                *qflag = TRUE;
                break;
            case '?':
            default:
                abort();
        }
    }
    argc -= optind;
    argv += optind;

    switch (argc)
    {
        case 3:
            *ofile       = argv[2];
            /* fall through */
        case 2:
            *ifile_x     = argv[1];
            *ifile_model = argv[0];
            break;
        default:
            usage(PACKAGE_NAME, VERSION);
            exit (EXIT_SUCCESS);
    }
}
#endif

#ifdef Rpackage
static void convert_Rdata(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx,
                        SEXP prdx, SEXP pb, SEXP psolution, SEXP ptflag,
                        SEXP ppflag, SEXP pqflag, int *pflag, int *qflag,
                        int *tflag, dmatrix **matX, double **b)
{
    int i;
    dmatrix *X;

    *pflag = *INTEGER(ppflag);
    *qflag = *INTEGER(pqflag);
    *tflag = *INTEGER(ptflag);

    X      = malloc(sizeof(dmatrix));
    X->m   = *INTEGER(pm);
    X->n   = *INTEGER(pn);
    X->nz  = *INTEGER(pnz);
    X->val = REAL(pval);

    if (X->nz >= 0)
    {
        X->idx = malloc(sizeof(int)*X->nz);
        X->jdx = malloc(sizeof(int)*X->nz);
        X->rdx = malloc(sizeof(int)*(X->m+1));

        /* C uses zero-based indexing while R uses one-based indexing */
        for (i = 0; i < X->nz; i++) {
            X->jdx[i] = INTEGER(pjdx)[i]-1;
        }
        for (i = 0; i < X->m+1; i++) {
            X->rdx[i] = INTEGER(prdx)[i]-1;
        }
        /* build idx from rdx and jdx */
        dmat_build_idx(X);
    }
    else
    {
        X->idx = NULL;
        X->jdx = NULL;
        X->rdx = NULL;
    }
    *matX = X;
    if (*tflag == TRUE)
        *b = REAL(pb);
    else
        *b = NULL;
}
#endif

/** \brief  Classify data with given model.
 *
 *  @param argc number of argument.
 *  @param argv array of input arguments.
 */
#ifndef Rpackage
int main(int argc, char *argv[])
#else
SEXP classifyR(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx,
               SEXP prdx, SEXP pb, SEXP psolution, SEXP ptflag, SEXP ppflag,
               SEXP pqflag)
#endif
{
    /* test data */
    dmatrix *matX;      /**< feature matrix */
    double  *b;         /**< class vector */

    /* solution buffer */
    double  *solution;  /* entry 1      : intercept,    */
                        /* entries 2~n+1: coefficients  */
    /* commandline argument */
    char *ifile_model, *ifile_x, *ifile_y, *ofile;
    int  pflag;         /* probability flag */
    int  tflag;         /* test flag */
    int  qflag;         /* quiet flag */

    clock_t clock_pre, clock_sol, clock_wri, clock_end;
    int  error_count;
    double *res;                /**< result [+1,-1] */
    int  model_m, model_n, model_nz, ret;
    char line[BUFFER_SIZE];
    char name[BUFFER_SIZE];

    clock_pre = clock();

#ifndef Rpackage
    parse_command_line_args(argc, argv, &tflag, &pflag, &qflag,
                            &ifile_x, &ifile_y, &ifile_model, &ofile);

    /* read data */
    if (!qflag) fprintf(stderr,"\nReading data...\n\n");
    read_mm_new_matrix(ifile_x, &matX);

    if (ifile_y != NULL)
        read_mm_new_vector(ifile_y, &b);
    else
        b = NULL;
#else
    convert_Rdata(pm,pn,pnz,pval,pjdx,prdx,pb,psolution,ptflag,ppflag,pqflag,
                        &pflag,&qflag,&tflag,&matX,&b);
#endif
    if (!qflag) summary_prob(matX, tflag);

#ifndef Rpackage
    get_mm_info(ifile_model, &model_m, &model_n, &model_nz);
    if (matX->n != model_m-1)
    {
        /* test examples : n = coefficients,
           model         : m = intercept+coefficients. */
        fprintf(stderr, "ERROR: number of features are different\n");
        fprintf(stderr, "       %d features in test examples, but %d in model.\n",
                        matX->n, model_m-1);
        exit(1);
    }

    if (model_n == 1)   /* model for one lambda */
    {
        read_mm_new_vector(ifile_model, &solution);

        /* run classifier */
        if (!qflag) fprintf(stderr,"Running classifier...\n");
        clock_sol = clock();

        res = malloc(sizeof(double)*(matX->m));
        ret = l1_logreg_classify(matX, b, solution, pflag, res, &error_count);

        if (!qflag) summary_result(matX, res, error_count);

        /*
         *  write solution
         */
        clock_wri = clock();
        if (ofile != NULL)
        {
            sprintf(line, comment1, ifile_x, PACKAGE_NAME, VERSION);
            if (pflag == TRUE)
                write_mm_vector(ofile, matX->m, res, line, TYPE_E);
            else
                write_mm_vector(ofile, matX->m, res, line, TYPE_G);
        }

        clock_end = clock();
        if (!qflag) summary_time(clock_pre, clock_sol, clock_wri, clock_end);

        dmat_free(matX);
        free(b);
        free(solution);
        free(res);
        return EXIT_SUCCESS;
    }
    else    /* model for multiple lambdas */
    {
        int i;
        double *lambda_vec, *error_vec;
        dmatrix *mat_model;

        clock_sol = clock();

        /* read lambda vector */
        sprintf(name, "%s_lambda", ifile_model);
        read_mm_new_vector(name, &lambda_vec);

        read_mm_new_matrix_transpose(ifile_model, &mat_model);
        /* each row is intercept+coefficients */

        res = malloc(sizeof(double)*(matX->m));
        error_vec = malloc(sizeof(double)*mat_model->m);

        solution = malloc(sizeof(double)*mat_model->n);
        for (i = 0; i < mat_model->m; i++)
        {
            if (!qflag) fprintf(stderr,"    lambda = %e\n",lambda_vec[i]);

            dmat_get_row(mat_model, i, solution);

            ret = l1_logreg_classify(matX, b, solution, pflag,
                                     res, &error_count);
            if (!qflag) summary_result(matX, res, error_count);
            error_vec[i] = error_count;

        }
        /* write solution */
        clock_wri = clock();
        if (ofile != NULL)
        {
            sprintf(line, comment2, ifile_x, PACKAGE_NAME, VERSION);
            write_mm_vector(ofile, mat_model->m, error_vec, line, TYPE_G);
        }
        clock_end = clock();
        if (!qflag) summary_time(clock_pre, clock_sol, clock_wri, clock_end);

        dmat_free(matX);
        dmat_free(mat_model);
        free(b);
        free(res);
        free(solution);
        free(error_vec);
        free(lambda_vec);
        return EXIT_SUCCESS;
    }
#else
    if (TYPEOF(psolution)==REALSXP)   /* model for one lambda */
    {
        SEXP Rres;
        solution =  REAL(psolution);

        /* run classifier */
        if (!qflag) fprintf(stderr,"Running classifier...\n");
        //clock_sol = clock();

        PROTECT(Rres = allocVector(REALSXP,matX->m));
        res = REAL(Rres);

        ret = l1_logreg_classify(matX, b, solution, pflag, res, &error_count);
        if (!qflag) summary_result(matX, res, error_count);

        if (matX->nz >= 0) {
            free(matX->idx);
            free(matX->jdx);
            free(matX->rdx);
        }
        free(matX);

        UNPROTECT(1);
        return Rres;
    }
    else    /* model for multiple lambdas */
    {
        int i;
        SEXP Rerror_vec;
        double *lambda_vec, *error_vec;
        dmatrix *mat_model;

        mat_model       = malloc(sizeof(dmatrix));
        mat_model->m    = *INTEGER(VECTOR_ELT(psolution,0));
        mat_model->n    = *INTEGER(VECTOR_ELT(psolution,1));
        mat_model->nz   = *INTEGER(VECTOR_ELT(psolution,2));
        mat_model->val  = REAL(VECTOR_ELT(psolution,3));
        mat_model->idx  = malloc(sizeof(int)*mat_model->nz);
        mat_model->jdx  = malloc(sizeof(int)*mat_model->nz);
        mat_model->rdx  = malloc(sizeof(int)*(mat_model->m+1));
        lambda_vec      = REAL(VECTOR_ELT(psolution,6));

        if (mat_model->nz >= 0)
        {
            int *pjdx, *prdx;
            pjdx = INTEGER(VECTOR_ELT(psolution,4));
            prdx = INTEGER(VECTOR_ELT(psolution,5));
            /* C uses zero-based indexing while R uses one-based indexing */
            for (i = 0; i < mat_model->nz; i++) {
                mat_model->jdx[i] = pjdx[i]-1;
            }
            for (i = 0; i < mat_model->m+1; i++) {
                mat_model->rdx[i] = prdx[i]-1;
            }
            /* build idx from rdx and jdx */
            dmat_build_idx(mat_model);
        }
        /* each row is intercept+coefficients */

        res = malloc(sizeof(double)*(matX->m));
        //error_vec = malloc(sizeof(double)*mat_model->m);
        PROTECT(Rerror_vec = allocVector(REALSXP,mat_model->m));
        error_vec = REAL(Rerror_vec);

        solution = malloc(sizeof(double)*mat_model->n);
        for (i = 0; i < mat_model->m; i++)
        {
            if (!qflag) fprintf(stderr,"    lambda = %e\n",lambda_vec[i]);

            dmat_get_row(mat_model, i, solution);

            ret = l1_logreg_classify(matX, b, solution, pflag, res,
                                                        &error_count);
            if (!qflag) summary_result(matX, res, error_count);
            error_vec[i] = error_count;

        }
        free(mat_model->idx);
        free(mat_model->jdx);
        free(mat_model->rdx);
        free(mat_model);
        free(res);
        free(solution);
        if (matX->nz >= 0) {
            free(matX->idx);
            free(matX->jdx);
            free(matX->rdx);
        }
        free(matX);

        UNPROTECT(1);
        return Rerror_vec;
    }
#endif

}
