/** \file   train.c
 *  \brief  Train over training data.
 *
 *  Commandline executable wrapper for the function l1_logreg.
 *  Reads and standardizes data, solves the problem, and
 *  writes the result 
 *
 */

#if HAVE_CONFIG_H
#   include "config.h"
#endif
#include "pkgdef.h"

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef Rpackage
#include <R.h>
#include <Rinternals.h>
#endif

#include "def.h"
#include "util.h"
#include "l1_logreg.h"

#define BUFFER_SIZE     1024


static const char comment1[] = {
"%% \n"
"%% This is a model file of %s,\n"
"%%    generated by %s_train ver.%s\n"
"%% Contents of model:\n"
"%%    entry   1   : intercept\n"
"%%    entries 2..m: coefficients\n"
"%% \n"
};


static void summary_prob(const dmatrix *M, const int sflag,
                         const double lambda_max, const double lambda)
{
    fprintf(stderr,"Problem summary:\n");
    if (M->nz >= 0)
        fprintf(stderr,"    [feature matrix]        sparse matrix of "
            "(%d examples x %d features), %d non-zeros\n", M->m, M->n, M->nz);
    else
        fprintf(stderr,"    [feature matrix]        dense matrix of "
            "(%d examples x %d features)\n", M->m, M->n);

    if (sflag == TRUE)
        fprintf(stderr,"    [standardization]       yes\n");
    else
        fprintf(stderr,"    [standardization]       no\n");

    fprintf(stderr,"    [lambda_max]            %g\n", lambda_max);
    fprintf(stderr,"    [lambda]                %g", lambda);
    fprintf(stderr,"        (ratio = %.4f)\n\n", lambda/lambda_max);
}


static void summary_time(clock_t t0, clock_t t1, clock_t t2, clock_t t3)
{
    fprintf(stderr,"Timing:\n");
    fprintf(stderr,"    [read data]         %10.3g (sec)\n",
        (double)(t1-t0)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [solve problem]     %10.3g (sec)\n",
        (double)(t2-t1)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [write solution]    %10.3g (sec)\n",
        (double)(t3-t2)/CLOCKS_PER_SEC);
    fprintf(stderr,"    [total time]        %10.3g (sec)\n\n",
        (double)(t3-t0)/CLOCKS_PER_SEC);
}


static void summary_all(const int m, const int n,
                        const double lambda, const double lambda_max,
                        const clock_t tt, const int ntiter, const int pcgiter, 
                        const double *coeffs)
{
    int i;
    int card;
    card = 0;
    for (i = 0; i < n; i++)
        if (coeffs[i] != 0.0) card++;

    /* lambda : real lambda value */
    fprintf(stderr,"(%d examples x %d features) lambda=%9.4e (lambda_max=%9.4e),  card= %d,  NTiter= %d,  PCGiter= %d, sol time= %10.3g (sec)\n", m, n, lambda, lambda_max, card, ntiter, pcgiter, (double)tt/CLOCKS_PER_SEC);
}


#ifndef Rpackage
static void usage(char *package, char *version)
{
    static const char *help[] = 
    {
"\n%s version %s\n",
"\nUsage: %s_train [options] feature_file class_file lambda model_file\n",
"Solve an l1-regularized logistic regression problem.\n\n",
"   Arguments:\n",
"       feature_file        - feature matrix\n",
"       class_file          - output vector\n",
"       lambda              - regularization parameter\n",
"       model_file          - store model data to file model_file\n",
"   Options:\n",
"       -q                  - quiet mode (same as -v0) \n",
"       -v [0..3]           - set verbosity level (default 1)\n",
"                               0 : quiet mode\n",
"                               1 : show one line summary\n",
"                               2 : show simple log\n",
"                               3 : show detailed log\n",
"       -r                  - use relative lambda\n",
"                               if used,     lambda := lambda*lambda_max\n",
"                               if not used, lambda := lambda\n",
"       -s                  - standardize data\n",
"   Advanced options:\n"
"       -h                  - show coefficients histogram and trim\n",
"       -k <double>         - set tolerance for zero coefficients from KKT\n",
"                             condition\n",
"       -t <double>         - set tolerance for duality gap\n",
"\n",
NULL,
    };

    const char **p;
    for (p = help; *p != NULL; p++)
        fprintf(stderr,*p, package, version);
    exit(EXIT_SUCCESS);
}


static void parse_command_line_args(int argc, char *argv[], double *lambda,
                                    train_opts *to, int *rflag, int *hflag,
                                    char **ifile_x, char **ifile_y, char **ofile)
{
    int  c;
    int  quiet = FALSE;

    /* default values */
    to->sflag         = FALSE;
    to->cflag         = FALSE;
    to->verbose_level = 2;
    to->tolerance     = 1.0e-8;
    to->ktolerance    = 0.999;
    *rflag            = FALSE;
    *hflag            = FALSE;

    while ((c = getopt(argc, argv, "hk:qrst:v:")) != -1)
    {
        switch (c)
        {
        case 'h': *hflag            = TRUE;         break;
        case 'r': *rflag            = TRUE;         break;
        case 's': to->sflag         = TRUE;         break;
        case 'k': to->ktolerance    = atof(optarg); break;
        case 't': to->tolerance     = atof(optarg); break;
        case 'v': to->verbose_level = atoi(optarg); break;
        case 'q': quiet             = TRUE;         break;
        case '?':
        default : abort();
        }
    }
    if (quiet == TRUE) to->verbose_level = 0;
    argc -= optind;
    argv += optind;

    switch (argc)
    {
    case 4:
        *lambda  = atof(argv[2]);
        *ofile   = argv[3];
        *ifile_y = argv[1];
        *ifile_x = argv[0];
        break;
    default:
        usage(PACKAGE_NAME, VERSION);
        exit (EXIT_SUCCESS);
    }
}
#endif


void show_status(int verbose_level, int ret, int total_nt, int total_pcg)
{

    if (verbose_level>=2)
    {
        switch(ret)
        {
        case STATUS_SOLUTION_FOUND:
            fprintf(stderr,"Solution found.\n\n");
            break;
        case STATUS_MAX_NT_ITER_EXCEEDED:
            fprintf(stderr,"MAX_NT_ITER exceeded.\n\n");
            break;
        case STATUS_MAX_LS_ITER_EXCEEDED:
            fprintf(stderr,"MAX_LS_ITER exceeded in linesearch.\n\n");
            break;
        default:
            fprintf(stderr,"no case.\n");
            exit(EXIT_SUCCESS);
        }
        fprintf(stderr,"    [NT  iterations]    %10d\n",total_nt);
        if (total_pcg >= 0)
            fprintf(stderr,"    [PCG iterations]    %10d\n",total_pcg);
        fprintf(stderr,"\n");
    }
}

#ifdef Rpackage
static void convert_Rdata(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx,
                        SEXP prdx, SEXP pb, SEXP plambda, SEXP pqflag,
                        SEXP prflag, SEXP psflag, SEXP phflag, SEXP pvval,
                        SEXP ptol, SEXP pktol,
                        dmatrix **matX, double **b,
                        double *lambda, train_opts *to, int *hflag, int *rflag)
{
    dmatrix *X;

    *b               = REAL(pb);
    *lambda          = *REAL(plambda);
    *hflag           = *INTEGER(phflag);
    *rflag           = *INTEGER(prflag);

    to->cflag        =  FALSE;
    to->sflag        = *INTEGER(psflag);
    to->verbose_level= *INTEGER(pvval);
    to->tolerance    = *REAL(ptol);
    to->ktolerance   = *REAL(pktol);

    if (*INTEGER(pqflag)) to->verbose_level = 0;

    X      = malloc(sizeof(dmatrix));
    X->m   = *INTEGER(pm);
    X->n   = *INTEGER(pn);
    X->nz  = *INTEGER(pnz);
    X->val = REAL(pval);
    if (X->nz >= 0)
    {
        int i;
        X->idx = malloc(sizeof(int)*(X->nz));
        X->jdx = malloc(sizeof(int)*(X->nz));
        X->rdx = malloc(sizeof(int)*(X->m+1));

        /* C uses zero-based indexing while R uses one-based indexing */
        for (i = 0; i < X->nz; i++ ) X->jdx[i] = INTEGER(pjdx)[i]-1;
        for (i = 0; i < X->m+1; i++) X->rdx[i] = INTEGER(prdx)[i]-1;
        /* build idx from rdx and jdx */
        dmat_build_idx(X);
    }
    else
    {
        X->idx = NULL;
        X->jdx = NULL;
        X->rdx = NULL;
    }
    *matX   = X;
}


static SEXP create_Rdata_to_return(const int n, const double lambda, const double *solution)
{
    SEXP res, Rintercept, Rcoefficients, Rabs_lambda, names;

    PROTECT(res          = allocVector(VECSXP,3));
    PROTECT(Rintercept   = allocVector(REALSXP,1));
    PROTECT(Rcoefficients= allocVector(REALSXP,n));
    PROTECT(Rabs_lambda  = allocVector(REALSXP,1));

    REAL(Rintercept)[0]  = solution[0];
    REAL(Rabs_lambda)[0] = lambda;
    memcpy(REAL(Rcoefficients), solution+1, n*sizeof(double));

    SET_VECTOR_ELT(res, 0, Rintercept   );
    SET_VECTOR_ELT(res, 1, Rcoefficients);
    SET_VECTOR_ELT(res, 2, Rabs_lambda  );

    PROTECT(names = allocVector(STRSXP,3));
    SET_STRING_ELT(names, 0, mkChar("intercept"));
    SET_STRING_ELT(names, 1, mkChar("coefficients"));
    SET_STRING_ELT(names, 2, mkChar("abs.lambda"));
    setAttrib(res, R_NamesSymbol, names);

    UNPROTECT(5);

    return res;
}
#endif

/*
 *  Standardize data and solve l_1-regularized logistic regression problem
 *  by calling l1_logreg().
 *
 */
#ifndef Rpackage
int main(int argc, char *argv[])
#else
SEXP trainR(SEXP pm, SEXP pn, SEXP pnz, SEXP pval, SEXP pjdx, SEXP prdx,
        SEXP pb, SEXP plambda, SEXP pqflag, SEXP prflag, SEXP psflag,
        SEXP phflag, SEXP pvval, SEXP ptol, SEXP pktol)
#endif
{
    /* problem data */
    dmatrix *matX;          /* feature matrix               */
    double  *b;             /* class vector                 */
    double  lambda;         /* regularization parameter     */
    train_opts  to;         /* training options             */

    double  *solution;      /* entry   1     : intercept,   */
                            /* entries 2~n+1 : coefficients */

    char *ifile_x, *ifile_y, *ofile;
    int  rflag;             /* relative lambda flag */
    int  hflag;             /* histogram & threshold flag */

    double  lambda_max;
    clock_t clock_pre, clock_sol, clock_wri, clock_end;
    int     total_nt, total_pcg, ret;


    clock_pre = clock();

#ifndef Rpackage
    parse_command_line_args(argc, argv, &lambda, &to, &rflag, &hflag,
                                        &ifile_x, &ifile_y, &ofile);

    /* read data file */
    if (to.verbose_level>=2) fprintf(stderr,"\nReading data...\n\n");
    read_mm_new_matrix(ifile_x, &matX);
    read_mm_new_vector(ifile_y, &b);
#else
    convert_Rdata(pm,pn,pnz,pval,pjdx,prdx,pb,plambda,pqflag,prflag,psflag,
          phflag,pvval,ptol,pktol,&matX,&b,&lambda,&to,&hflag,&rflag);
#endif

    lambda_max = find_lambdamax(matX, b, to.sflag);
    if (rflag) lambda = lambda*lambda_max;

    if (to.verbose_level>=2)
        summary_prob(matX, to.sflag, lambda_max, lambda);

    /* intercept(1) + coefficients(2..n+1) */
    solution = malloc(sizeof(double)*(matX->n+1));

    /* run solver */
    if (to.verbose_level>=2) fprintf(stderr,"Running solver...\n");
    clock_sol = clock();

    ret = l1_logreg_train(matX, b, lambda, to, NULL, NULL,
                          solution, &total_nt, &total_pcg);

    //dmat_profile();
    
    /* show status */
    if (matX->nz < 0) total_pcg = -1;

    show_status(to.verbose_level, ret, total_nt, total_pcg);

    /* write solution */
    clock_wri = clock();
    if (hflag == TRUE) /* manual thresholding */
    {
        show_histogram(matX->n, solution+1);
        thresholding(matX->n, solution+1, userinput_threshold());
        show_histogram(matX->n, solution+1);
    }
        
#ifndef Rpackage
    if (ofile != NULL)
    {
        char linebf[BUFFER_SIZE];
        sprintf(linebf, comment1,ifile_x, PACKAGE_NAME, VERSION);
        write_mm_vector(ofile, matX->n+1, solution, linebf, TYPE_E);
    }
    /* print info */
    clock_end = clock();
    if (to.verbose_level>=2)
        summary_time(clock_pre,clock_sol,clock_wri,clock_end);

    if (to.verbose_level==1)
        summary_all(matX->m, matX->n, lambda, lambda_max, clock_wri-clock_sol,
                    total_nt, total_pcg, solution+1);
    free(solution);
    free(b);
    dmat_free(matX);

    return EXIT_SUCCESS;
#else
    {
        SEXP res;
        res =  create_Rdata_to_return(matX->n, lambda, solution);
        if (matX->nz >= 0) {
            free(matX->idx);
            free(matX->jdx);
            free(matX->rdx);
        }
        free(matX);
        free(solution);

        return res;
    }
#endif

}


