function mask = quick_label(invol)
% Rapidly manually label a volume with the help of logistic regression
% 
% 
% output = QUICK_LABEL
% output = QUICK_LABEL(2/3D matrix)
% output = QUICK_LABEL(input_file.nii/.nii.gz)
% 
% INPUT:
%   QUICK_LABEL with no input arguments brings up a file selection GUI
%   QUICK_LABEL(2/3D matrix) assumes the third dimension is the slice direction
%     It does not allow you to specify pixel dimensions.
%   QUICK_LABEL(input_file.nii/.nii.gz) attempts to open the .nii or .nii.gz file
%     and orient the data sensibly.
%     2D images in the GUI are axial slices.
%     Anterior (nose-side of head) be up in the 2D slice
%     Left and Right will be left and righ in the GUI
%     Anterior->superior (foot->head) be increasing slice number
%     It will also use pixel dimension data to get the correct aspect ratio
%
% 
% OUTPUT:
%   If the input file is a nifti (either from gui or argument), 
%   then QUICK_LABEL can output a nifty file.
%
%   If no output argument is given (nargout == 0), it will save a nifty and 
%       .mat file in the same place as the input file.
% 
%   If output is given, it will not do this.
%
%
% Instructions for use after image is loaded:
%
%  THESE INSTRUCTIONS ARE MOSTLY RIGHT BUT DON'T REFLECT ALL OF THE AVAILABLE
%  FEATURES. I'LL FIX THEM SOON!
%
% GUI will pop up.
% Navigate to the top of the anatomy (or one slice before) using the slider.
% Outline the anatomy in question.
% Continue for the rest of the anatomy.
% When you've enclosed the anatomy completely (100% enclosed)
%     hit the Logisticize Me! button.
%     We then try to classify the image based on the information you provided.
% If you are happy with this classification, hit Finish! to return the logical
%     variable lab_vol to the matlab workspace.
% If not, modify the image by:
%     1. Clipping of stuff that shouldn't be included: start outside the
%        included area and end outside.
%     2. Including stuff that should be included: Start inside the included
%        area and end inside.
%     Edge-cases: If a slice is totally wrong, "Redo-Slice" to draw a new
%        polygon,
%
% At any point, you can hit 3D render and see a 3D rendering of the current
%     segmentation in a new figure window.
%
% Notes:
% Homology is enforced.
% Nothing you exclude will ever be included.


Banner = ...
    ['====================================================================\n'...
     '====================================================================\n'...
     '===============                            =========================\n'...
     '===============    QUICK LABEL             =========================\n'...
     '===============    Version 1               =========================\n'...
     '===============                            =========================\n'...
     '===============    Frederick Bryan, 2013   =========================\n'...
     '===============                            =========================\n'...
     '====================================================================\n'...
     '====================================================================\n'];
 
fprintf(['\n' Banner '\n\n']);

% startime
st  = cputime; 

% error checking
if nargin<1;
    fprintf('No input argument given. Use GUI to select input file.\n\n');
    [fname pathname ~] = uigetfile({'*.nii;*.nii.gz','NIFTI files'},'Select file');
    if ~isa(fname,'char') && fname == 0 && pathname == 0;
        disp('Cancelled');
        return;
    else
        invol = [pathname fname];
    end
    
    
end

wasnifty = 0;
if isa(invol,'char')
    if isequal(invol(end-3:end),'.nii');
        disp('Loading .nii');
        vol = load_nii(invol);
        output_filename = [invol(1:end-4) '_QuickMask.mat'];
        wasnifty = 1;
    elseif isequal(invol(end-6:end),'.nii.gz')
        disp('Loading .nii.gz');
        vol = load_nii_gz(invol);
        output_filename = [invol(1:end-7) '_mask.mat'];
        wasnifty = 2;
    else
        fprintf('\nInput was string, but not .nii or .nii.gz. Exiting\n\n');
        return;
    end
    
    invol = vol.img;
end

if isnumeric(invol)  % a matrix
    if ~(ndims(invol)>1 && ndims(invol)<4)
        fprintf('\nVolumes must be 2-3D. Exiting\n\n');
        return;
    end
else
    fprintf('\nVolume was non-numeric. Exiting\n\n');
    return;
end

% pre-processing
invol = img_normalize(invol);

% prepare to plot
% fix image dimensions if information is available.
if wasnifty;
    pixdim = vol.hdr.dime.pixdim(2:4);
    % flip flop dimensions until happy
    invol = flipdim(permute(invol,[2 1 3]),1);
    pixdim = pixdim([2 1 3]);
else
    pixdim = [1 1 1];
end


fprintf('\nData accepted, bringing up GUI\n\n');

% handle/catch ctrl-c so the GUIDE window closes even in that case
cleanupObj = onCleanup(@()closereq);

% bring up the GUIDE gui - may error
% THIS DOES WAYY MOST OF THE WORK
[mask,handles] = labelbox(invol,pixdim);

% check if mask empty, exit if so
if strcmp(mask,'empty');
    disp('GUI closed. Any data lost.')
    return;
end

% from here on, we assume mask not 'empty' (see return statement above)

% get quickclassify time outputs (passed through labelbox)
if isfield(handles,'timespent') % an array of times
    tsp = sum(handles.timespent);
else
    tsp = 0;
end

% reformat data into original space if was nifty,
if wasnifty;
    mask = ipermute(flipdim(mask,1), [2 1 3]);
end

% save output data
if nargout == 0 && wasnifty; 
    save(output_filename,'mask');
    fprintf('\nNo Output Argument given!');
    fprintf('\nOutput saved as .mat file: \n%s \n\n',output_filename);
    
    try
        if wasnifty == 1;
            % remove '.mat' and add '.nii'
            output_filename = [output_filename(1:end-4)  '.nii'];
            
            out_nii = vol;
            out_nii.img = int16(mask);
            out_nii.hdr.dime.datatype = 4;
            out_nii.hdr.dime.bitpix = 16;

            save_nii(out_nii, output_filename);
            fprintf('\nOutput saved as .nii file:\n%s \n\n',output_filename);

        elseif wasnifty == 2;
            % remove '.mat' and add '.nii.gz'
            output_filename = [output_filename(1:end-4) '.nii.gz'];
            
            out_nii = vol;
            out_nii.img = int16(mask);
            out_nii.hdr.dime.datatype = 4;
            out_nii.hdr.dime.bitpix = 16;

            save_nii(out_nii, output_filename);
            fprintf('\nOutput saved as .nii file:\n%s \n\n',output_filename);
        end

        
    catch err
        disp('was not able to save nifty file, .mat file WAS saved!');
    end
    
    mask = ['check input directory. If you want the output in the '...
            'workspace, give an output arg.'];
    
end

% elapsed time
et = cputime -st; 
fprintf('Total Elapsed time: %5.2fs (%02d:%02d:%02d) \n\n',...
    et,floor(et/3600),floor(mod(et,3600)/60), floor(mod(et,60)));
fprintf('Elapsed time - Processing Time: %5.2fs (%02d:%02d:%02d) \n\n',...
    (et-tsp),floor((et-tsp)/3600),floor(mod((et-tsp),3600)/60), ...
    floor(mod((et-tsp),60)));

function out = img_normalize(in)

% cast to double
in = double(in);

% saturate btwen .5 and 99.5 percentile
lowval = prctile(in(:), .5);
highval = prctile(in(:), 99.5);
out = in;
out(in<lowval) = lowval;
out(in>highval) = highval;

% scale to one;
out = (out-lowval)/(highval-lowval);








