function [mask_out varargout] = ...
    quickclassify(im, mask, pixdim, feature_struct, reg_opts)
% QUICKCLASSIFY takes a rough outline of an object and tries to determine the
% object of interest. This is a sub-function of quick_label, which combines this
% stuff with a sweet gui.
%
% mask_out = QUICKCLASSIFY(im, mask, pixdim, feature_struct)
% mask_out = QUICKCLASSIFY(im, mask, pixdim, feature_struct, reg_opts)
%
% [mask_out, elapsed_time] = QUICKCLASSIFY( ... );
% [mask_out, elapsed_time, beta_hat] = QUICKCLASSIFY( ... );
%
% reg_opts can have these fields:
%  l1_lamb: l1 regression regularization parameter, [0 inf) [default = 0]
%  l2_lamb: l2 regression regularization parameter, [0 inf) [default = 0.01]
%
% Notes:



% -------- Things to fix ------------
% better preallocation of X
%

% debugging

% % Check 2d
% mask = mask(:,:,82);
% im = im(:,:,82);

% % allow truth to come in as third arguement
% truth = feature_struct;
%

% feature_struct.crop.status =1;
% feature_struct.normalize.status=1;
%
% feature_struct.displayims.status=1;
%
% feature_struct.intensity.status=1;
% feature_struct.gaussian.status=[1 2 3];
% feature_struct.median.status=[1 2];
% feature_struct.distance.status=1;
%
% feature_struct.filterprobabilities.status=1;
% feature_struct.guaranteedenclosed.status=1;
% feature_struct.homo_singleobject.status=1;
% feature_struct.homo_noenclosedholes.status=1;
% feature_struct.homo_enforcehomo.status=1;
% feature_struct.usesmallregion.status=1;
% feature_struct.nospurs.status=1;

% % Not sure why this is here.
% mask = flipdim(ipermute(mask,[2 1 3]),1);

% ============================================================================
% =========== INITIALIZATION AND BEGINNING OF FUNCTION STUFF =================
% ============================================================================

% time this function
st = cputime;

% defaults in one place for easy change
l1_lamb_def = 0;
l2_lamb_def = 0.01;

% parse inputs
if nargin<4
    error('not enough args');
end
% handle optional regression options struct
if nargin<5; % no struct given at all
    reg_opts.l1_lamb = l1_lamb_def;
    reg_opts.l2_lamb = l2_lamb_def;
end
if ~isfield(reg_opts,'l1_lamb'); % no l1_lamb given
    reg_opts.l1_lamb = l1_lamb_def;
else
    % check user input
    if ~isa(reg_opts.l1_lamb,'numeric');
        error('reg_opts.l1_lamb must be numeric');
    end
end
if ~isfield(reg_opts,'l2_lamb'); % no l2_lamb given
    reg_opts.l2_lamb = l2_lamb_def;
else
    if ~isa(reg_opts.l2_lamb,'numeric');
        error('reg_opts.l2_lamb must be numeric');
    end
end
% check sizes
if ~isequal(size(im),size(mask));
    error('Intensity and mask volumes need to be same size');
elseif ndims(im)>3  || ndims(im)<2
    error('Need to have 2 or 3 dimensional objects');
end
if ~all(ismember(mask(:),[0 1]))
    error('Mask must be logical or have elements in the set: { 0,1 }');
end

% parse feature struct
parse_yourself(feature_struct);

% keep a copy of originals
% orig_im = im;
orig_mask = mask;

% normalize pixdims
pdscale = pixdim/min(pixdim); % pixdim scaled so min() = 1

% check for display image "feature"
if isfield(feature_struct,'displayims') && feature_struct.displayims.status
    displayims = true;
    
    sl = display_im(im,mask); title('Initial Image');
    feature_struct = rmfield(feature_struct,'displayims');
else
    displayims = false;
end

% ============================================================================
% =========== PREPROCESSING ==================================================
% ============================================================================

fprintf('\n --- --- Preprocessing --- ---\n');

if isfield(feature_struct,'crop') && feature_struct.crop.status;
    disp('Cropping');
    
    
    if ~isempty(mask)
        % crop image way way down
        r1 = max(find(sum(sum(mask,3),2)>0,1,'first')-3,1);
        r2 = min(find(sum(sum(mask,3),2)>0,1,'last')+3,size(mask,1));
        c1 = max(find(sum(sum(mask,3),1)>0,1,'first')-3,1);
        c2 = min(find(sum(sum(mask,3),1)>0,1,'last')+3,size(mask,2));
        
        if ~ismatrix(mask) % 3d case
            z1 = max(find(sum(sum(mask,2),1)>0,1,'first')-3,1);
            z2 = min(find(sum(sum(mask,2),1)>0,1,'last')+3,size(mask,3));
            im = im(r1:r2,c1:c2,z1:z2);
            mask = mask(r1:r2,c1:c2,z1:z2);
        elseif ismatrix(mask)
            im = im(r1:r2,c1:c2);
            mask = mask(r1:r2,c1:c2);
        else
            error('check dims of data, can''t crop. dummmy?');
        end
        
    else
        disp('Mask totally empty. No way to crop. Not cropping.');
    end
    
    if displayims
        sl = display_im(im,mask); title('After Cropping');
    end
    feature_struct = rmfield(feature_struct,'crop');
end

if isfield(feature_struct,'normalize')  && feature_struct.normalize.status
    disp('Normalizing')
    
    % normalize w/in masked region
    im = img_normalize(im,[],mask);
    
    if displayims
        sl = display_im(im,mask); title('After Normalization');
    end
    feature_struct = rmfield(feature_struct,'normalize');
end

% inhomogeneity correction?

% make non-mask uninteresting
if isfield(feature_struct,'uninteresting')  && ...
        feature_struct.uninteresting.status;
    
    fprintf('Making "uninteresting" image');
    
    
    imu = make_im_uninteresting(im,mask); % Cropped IMage Uninteresting
    
    if displayims
        display_im(imu,mask,sl); title('After "Making Uninteresting"');
    end
    feature_struct = rmfield(feature_struct,'uninteresting');
else
    imu = im; % this doesn't waste memory because matlab doesn't allocate
    % for unchanged arrays (this case)
end

% ============================================================================
% =========== FEATURE SELECTION ==============================================
% ============================================================================
fprintf('\n --- --- Feature Selection --- ---\n');

% intialize feature matrix
% add ones column as first "feature"
X = ones(numel(mask),1);
% this needs to be fixed.****************
% should be able to figure out size from the begginning and preallocated this
% thing correctly

if isfield(feature_struct,'intensity')  && feature_struct.intensity.status
    disp('Including raw intensity as feature');
    
    
    X = [X,imu(:)];
    
    if displayims
        display_im([],imu,sl); title('Feature: intensity');
    end
    feature_struct = rmfield(feature_struct,'intensity');
end

if isfield(feature_struct,'gaussian') && any(feature_struct.gaussian.status)
    % gaussian smooth
    

        

    sizes = [3 5 7];
    sizes_chosen = feature_struct.gaussian.status; % an array % FIXME

    if max(sizes_chosen) > length(sizes) || min(sizes_chosen)<1
        error('sizes_chosen elements must be >1, <%i',length(sizes));
    end
    
    fprintf('Feature: Gaussian Smoothing aka "Scale-Space" ');
    fprintf(': Sizes chosen :'); fprintf(' %i ',sizes(sizes_chosen));
    fprintf('\n');
    
    disp('    remember to fix this!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!');

    for ii = 1:length(sizes_chosen);
        s = sizes(sizes_chosen(ii));

        k = gkernel([s s s],floor(s/2)./pdscale);
        gim = imfilter(imu,k,'replicate');

        X = [X, gim(:)];

        if displayims
            display_im([],gim,sl); title('Feature: gaussian');
        end
    end
    
    
    %     s=3; k1 = gkernel([s s s],1./pdscale);
    %     s=5; k2 = gkernel([s s s],2./pdscale);
    %     s=7; k3 = gkernel([s s s],3./pdscale);
    %     s=9; k4 = gkernel([s s s],4./pdscale);
    
    %     gcim1 = imfilter(cimu,k1,'replicate');
    %     gcim2 = imfilter(cimu,k2,'replicate');
    %     gcim3 = imfilter(cimu,k3,'replicate');
    %     gcim4 = imfilter(cimu,k4,'replicate');
    
    % Fourier Domain is faster, but it causes a wierd shift in the 3rd
    % dimension that I can't account for...
    % cimf=fftn(cimu);
    % h1 = zeros(size(cim)); h2=h1; h3=h1; h4=h1;
    % s=3; h1(1:s,1:s,1:s) = k1;
    % s=5; h2(1:s,1:s,1:s) = k2;
    % s=7; h3(1:s,1:s,1:s) = k3;
    % s=9; h4(1:s,1:s,1:s) = k4;
    % % fourier domain is faster than imfilter (by seconds)
    % gcim1f = (cimf.*fftn(h1));
    % gcim2f = (cimf.*fftn(h2));
    % gcim3f = (cimf.*fftn(h3));
    % gcim4f = (cimf.*fftn(h4));
    % gcim1 = ifftn(gcim1f);
    % gcim2 = ifftn(gcim2f);
    % gcim3 = ifftn(gcim3f);
    % gcim4 = ifftn(gcim4f);
    feature_struct = rmfield(feature_struct,'gaussian');
end

if isfield(feature_struct,'median') && any(feature_struct.median.status)
    
    % median smooth
    sizes = [3 5];
    sizes_chosen = feature_struct.median.status; % an array % FIXME
    
    if max(sizes_chosen) > length(sizes) || min(sizes_chosen)<1
        error('sizes_chosen elements must be >1, <%i',length(sizes));
    end
    
    fprintf('Feature: Median Smoothing')
    fprintf(': Sizes chosen :'); fprintf(' %i ',sizes(sizes_chosen));
    fprintf('\n');
    
    disp('    remember to fix this!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!');
    
    for ii = 1:length(sizes_chosen);
        s = sizes(sizes_chosen(ii));
        
        mcim = medfilt3(imu,[s s s]);
        
        X = [X,mcim(:)]; % add to feature matrix
        
        if displayims
            display_im([],mcim,sl); title('Feature: median');
        end
    end
    
    %         s=3; mcim1 = medfilt3(cimu,[s s s]);
    %         s=5; mcim2 = medfilt3(cimu,[s s s]);
    feature_struct = rmfield(feature_struct,'median');
end

if isfield(feature_struct,'distance')  && feature_struct.distance.status
    %  distance metric from center of image (Probalistic Distance)
    disp('Feature: Distance Metric');
    
    
    pd = zeros(size(im)); mid = round(size(pd)/2);
    pd(mid(1),mid(2),mid(3))=1;
    pd = -bwdistX(pd,pixdim);
    pd = mat2gray(pd);
    
    X = [X,pd(:)];
    
    if displayims
        display_im([],pd,sl); title('Feature: distance');
    end
    feature_struct = rmfield(feature_struct,'distance');
end

% if isfield(feature_struct,'gabor') && feature_struct.gabor.status
%     %     % disp('Feature: Gabor Filters');
%     %     % some gabor filters
%     %     % do it with 2d filters
%     %     % only really need to do  it on unfiltered image?
%     %     %
%     %     % in xy plane
%     %     % do theta = [0,45,90,135]
%     %     % do bw = [2 5 10 20]
%     %     % do aspect = [1 .5 2]
%     %     % do sigmoid = bw/1.5;
% end

    
if isfield(feature_struct,'gonuts') && feature_struct.gonuts.status>0
    
    fprintf('Feature: Going nuts for features!\n');

    % this was nice for debugging     
%     imu = randn(100,100,20);
%     X = [ones(numel(imu),1),imu(:)];
%     pdscale = [1 1 5];
%     
    sz = size(X);
    fprintf('\tSize X before go nuts: %i by %i\n',sz(1), sz(2));
    
    % gaussians
    s=[10 10 3]; % size of thing
    sizesx = [3,5,7,10,15,20];
    sizesy = [3,5,7,10,15,20];
    sizesz = [3:3:7];
        
    for x=1:numel(sizesx);
        for y=1:numel(sizesy);
            for z=1:numel(sizesz)
                s=round([sizesx(x) sizesy(y) sizesz(z)]./pdscale); % size of thing
                s = max(s,ones(size(s))); % no zeros
                k = gkernel(s,...
                    [sizesx(x) sizesy(y) sizesz(z)]);
                gim = imfilter(imu,k,'replicate');
                X = [X, gim(:)];
            end
        end
    end
    fprintf('\tSize X after gaussian: %i by %i\n',size(X,1), size(X,2));
    fprintf('\t                added: %i features\n',size(X,2)-sz(2));
    sz = size(X);

    if feature_struct.gonuts.status > 1;
        % 2D gabors
        s = [20 20];
        for theta=deg2rad([0:45:135]);
            for bw = [2,5,9];
                for aspect = [.25,1,2];
                    for sig = bw*[.5,1];
                        gb=gabor_fwb(aspect,theta,bw,0,sig,s);
    %                     imagesc(gb);drawnow; pause(0.1); 
                        gbim = imfilter(imu,gb,'replicate');
    %                     keyboard;s
                        X = [X, gbim(:)];
                    end
                end
            end
        end
        fprintf('\tSize X after gabors  : %i by %i\n',size(X,1), size(X,2));
        fprintf('\t                added: %i features\n',size(X,2)-sz(2));
        sz = size(X);
    end
    
    if feature_struct.gonuts.status > 2
        % random feature vectors
        for i = 1:30; 
            t = randn(5,5,5);
            tt = imfilter(imu,t,'replicate');
            X = [X,tt(:)];
        end
        fprintf('\tSize X after rnd text: %i by %i\n',size(X,1), size(X,2));
        fprintf('\t                added: %i features\n',size(X,2)-sz(2));
        sz = size(X);
    end 
    
    if feature_struct.gonuts.status > 3
        % totally random features
        X = [X,randn(size(X,1),30)];
        fprintf('\tSize X after randoms : %i by %i\n',size(X,1), size(X,2));
        fprintf('\t                added: %i features\n',size(X,2)-sz(2));

        if displayims;
            figure; 
            for i = 1:size(X,2);
                x =(reshape(X(:,i),size(imu))); 
                imagesc(x(:,:,round(size(x,3)/2))); colorbar;
                drawnow; pause(0.1); 
            end
        end
    end
end

% ============================================================================
% =========== REGRESSION =====================================================
% ============================================================================

fprintf('\n --- --- Regression --- ---\n');

Y = mask(:);

if isfield(feature_struct,'usesmallregion') && ...
        feature_struct.usesmallregion.status
    
    disp('Using only small region around mask for training');
    
    
    inds = find(bwdistX(mask,pixdim)<5);
    
    Y = Y(inds);
    Xfull = X;
    X = X(inds,:);
    
    if displayims
        indsm = zeros(size(mask));
        indsm(inds)=1;
        display_im(imu,indsm+mask,sl);
        title('Limiting training to mask + this extra region');
    end
    feature_struct = rmfield(feature_struct,'usesmallregion');
else
    inds = 1:numel(Y);
    Xfull = X;
end

if isfield(feature_struct,'autoselectfeature') && ...
        feature_struct.autoselectfeature.status
    % use rSLR for feature selection
    disp('Running l1 sparse logistic regression to try to pick features');
    
    
    logreg_opts.verbosity = 1; % default anyway
    % note: l1_default = quickclassify_default = '0.01'
    logreg_opts.lambda = num2str(reg_opts.l1_lamb);
    logreg_opts.standardize = 1; % default anyway
    
    [Bsparse,~] = l1_logreg_wrapper(X,Y,logreg_opts);
    % Yest = 1./(1+exp(-Xall*Bsparse));
    %  Yest = reshape(Yest,size(cim));
    X = X(:,(Bsparse~=0));
    
    %     tmp = Yhatsparse;
    %     Yhatsparse = zeros(size(mask));
    %     Yhatsparse(inds) = tmp;
    feature_struct = rmfield(feature_struct,'autoselectfeature');
end

% set up options structs for rlogistic function
opt1.verbose = 1;
opt1.l2_lamb = reg_opts.l2_lamb;
opt1.l1_lamb = reg_opts.l1_lamb;

% standarization = for each row, subract mean and divide by var
if numel(Xfull)< (100*2^20)/4 % less than 100 MB (faster method)
    Xfull = (Xfull-repmat(mean(Xfull),[size(Xfull,1) 1])) ./ ...
        repmat(std(Xfull),[size(Xfull,1) 1]);
    X = (X-repmat(mean(X),[size(X,1) 1])) ./ repmat(std(X),[size(X,1) 1]);
    
    % fix ones column. divide by zero stddev causes NaNs
    Xfull(:,1)=1; 
    X(:,1)=1;
else % more memory conservative method
    fprintf('Very large matrix (>100MB) being used\n');
    fprintf('\tXfull: %i MB, Xused: %i MB\n',...
        round(numel(Xfull)*4/2^20),round(numel(X)*4/2^20))
    for i = 2:size(X,2); % leave off ones column
        Xfull(:,i) = (Xfull(:,i) - mean(Xfull(:,i))) / std(Xfull(:,i));
        X(:,i) = (X(:,i) - mean(X(:,i))) / std(X(:,i));
    end
end

if isfield(feature_struct,'gonuts') && feature_struct.gonuts.status > 0;
    opt1.wentnuts = 1; % this option causes rlogistic to run just the first 10
                        %   features to get a good ininitialization, then run
                        %   all billion features. should be more stable
    feature_struct = rmfield(feature_struct,'gonuts');
end

% use robust logistic
[Bhat,Yhat] = rlogistic(X,Y,opt1);

% put Y back into full size (accounts for uses small region)
if numel(Yhat) ~= numel(im)
    Yhat = 1./(1+exp(-Xfull*Bhat));
end

% reshape into 3d image from vector
Yhat = reshape(Yhat,size(mask));

if displayims
    display_im([],Yhat,sl); title('Probabilities After regression');
    %     time = clock; time = ['Created -' num2str(time(4)) '.' num2str(time(5))];
    %     figure('name',time); slice_slider(im,Yhat); colormap jet;
    %     figure('name',time); slice_slider(Yhat); colormap jet;
end

% ============================================================================
% =========== POST=PROCESSING ================================================
% ============================================================================

fprintf('\n --- --- Postprocessing --- ---\n');

if isfield(feature_struct,'brightspotremoval') && ...
        feature_struct.brightspotremoval.status
    
    disp('Bright Spot Removal (Beta)');
    
    % the better way to do this is by doing -im and constrain the Betas to be
    % positive (Betas go negative if im is just inverted, doing nothing)
    
    %     % what did work
    %     [Bhat,YY] = rlogistic([ones(numel(im),1),(-im(:)+1).*mask(:)],Y,opt1);
    
    Yhat = Yhat(inds); % this either does nothing or makes Yhat small again
    size(X)
    size(Yhat)
    
    Xinv = [X(:,1),(-X(:,2:end)+1).*repmat(Yhat(:),[1 size(X,2)-1])];
    
    [Bhat2,Yhat2] = rlogistic(Xinv,Y,opt1);
    
    % put Y back into full size (accounts for uses small region)
    if numel(Yhat2) ~= numel(im)
        Yhat2 = 1./(1+exp(-Xinv*Bhat2));
    end
    
    Yhat = Yhat.*reshape(Yhat2,size(im));
    
    if displayims
        display_im([],Yhat,sl); title('PostProc: After brightspotremoval');
    end
    feature_struct = rmfield(feature_struct,'brightspotremoval');
end

if isfield(feature_struct,'guaranteedenclosed')  && ...
        feature_struct.guaranteedenclosed.status
    disp('Constraining results to only labeled areas');
    
    % constrain to be in masked region
    Yhat = Yhat.*double(mask);
    
    if displayims
        display_im([],Yhat,sl); title('PostProc: After guaranteedenclosed');
    end
    feature_struct = rmfield(feature_struct,'guaranteedenclosed');
else
    %nothing
end

if isfield(feature_struct,'filterprobabilities')  && ...
        feature_struct.filterprobabilities.status
    disp('Filtering/Denoising Probabilities');
    
    % filter probabilities and threshold
    Yhat = medfilt3(Yhat)>.5;
    
    if displayims
        display_im([],Yhat,sl); title('PostProc: After filterprobabilities');
    end
    feature_struct = rmfield(feature_struct,'filterprobabilities');
else
    Yhat = Yhat>.5;
end

if displayims
    display_im(imu,Yhat,sl); title('Before Applying Homology Operations');
end

if isfield(feature_struct,'homo_singleobject') && ...
        feature_struct.homo_singleobject.status
    
    disp('Homology: Eliminating other blobs');
    
    % eliminate other blobs in 3D
    Yhat = clear_blobs(Yhat);
    % now what has been enforced is only 3D homology (1 object)
    
    if displayims
        display_im(imu,Yhat,sl); title('Homology: After homo\_singleobject');
    end
    feature_struct = rmfield(feature_struct,'homo_singleobject');
else
    % nothing
end

if isfield(feature_struct,'homo_noenclosedholes') && ...
        feature_struct.homo_noenclosedholes.status
    
    disp('Homology: Closing holes in main blob');
    
    % close holes in 3D
    Yhat = imfill(Yhat,'holes');
    
    % coffee cups and donuts could still exist
    if displayims
        display_im(imu,Yhat,sl); title('Homology: After homo\_noenclosedholes');
    end
    feature_struct = rmfield(feature_struct,'homo_noenclosedholes');
else
    %nothing
end


if isfield(feature_struct,'homo_enforcehomo')  && ...
        feature_struct.homo_enforcehomo.status
    % enforce a "convex" (no little pockets or spurs) shape
    disp('Homology: Enforcing homology (no coffee cups or donuts allowed)');
    
    Yhat = enforce_no_coffee_cups(Yhat);
    
    % now donuts and coffee cups shouldn't have holes.
    if displayims
        display_im(imu,Yhat,sl); title('Homology: After homo\_enforcehomo');
    end
    feature_struct = rmfield(feature_struct,'homo_enforcehomo');
end

if isfield(feature_struct,'nospurs')  && feature_struct.nospurs.status
    
    disp('Trying to eliminate spurs of object to produce smoother object');
    
    % define structuring elements needed for a few things
    sesz2 = 3; % smaller structing element
    if ~ismatrix(im)  % 3D
        if round(pdscale(1))==round(pdscale(2));
            % usual case: same in-plane pixdims
            R = sesz2 * max(1,round(1/pdscale(1)));
            H = max(1,sesz2 * round(1/pdscale(3)));
            se1 = strel('ball',R+1,H);
            se2 = strel('ball',R,H);
        else % diff in-plane pixdims (take average)
            R = sesz2*max(1,mean([1/pdscale(1), 1/pdscale(2)]));
            H = max(1,sesz2 * round(1/pdscale(3)));
            se1 = strel('ball',R+1,H);
            se2 = strel('ball',R,H);
        end
    elseif ismatrix(im) % 2D
        R = sesz2*max(1,mean([1/pdscale(1), 1/pdscale(2)]));
        se1 = strel('disk',R+1);
        se2 = strel('disk',R);
    end
    
    % weakly eliminate spurs (erode, fill,dilate)
    Yhat = imdilate(imerode(double(Yhat),se2),se1);
    % cast to double elimates erros when imerode tries to dilate binary
    % binary imags
    
    if displayims
        display_im(imu,Yhat,sl); title('Homology: After nospurs');
    end
    feature_struct = rmfield(feature_struct,'nospurs');
end

% enforce b-spline surface?

% maybe do something with the edge detection here?
% i like that watershed idea.
% foo = double(watershed(pe4));
% if isfield(feature_struct,'edge_detection');
%     disp('Detecting Edges');
%     % edge detection at different scales (used zeroed image)
%     % not sure yet, some ideas below
%     [e1, pe1] = edge3_fwb(im1,'nofilter');
%     [e2, pe2] = edge3_fwb(gcim4, 'nofilter');
%     [e3, pe3] = edge3_fwb(mcim1, 'nofilter');
%     [e4, pe4] = edge3_fwb(mcim2, 'nofilter');
%     % ideas for use:
%     % dot-mult with bwdistX & threshold (at end)
%     % use watershed function somehow
% end

if displayims
    display_im(imu,Yhat,sl); title('Final');
end

% Check if feature_struct still has any .status == 1 fields and report to
% user so that they can fix their misspelling or whatever
f = fields(feature_struct); % get remaining fields
foundone = 0;
if ~isempty(f); % go through once to see if any fields have status==1
    for ii = 1:length(f); % parse status for each remaining fields
        if isfield( feature_struct.(f{ii}),'status') % note the space in this 
                                                     % to avoid being parsed!
            if ~isequal(feature_struct.(f{ii}).status,0)
                foundone = foundone+1;
            end
        else
            fprintf(['\n%s remained a field of feature_struct, '...
                'but did not have a status field. Contact developer.\n'],f{ii});
        end   
    end
    if foundone>0; % print those fields
        fprintf('\n');
        fprintf('The following feature tags had status==1, but were not used.\n');
        fprintf('Please if feature implemented, spelling, and FEATURES.txt file.\n\n');
        for ii = 1:length(f);
            if ~isequal(feature_struct.(f{ii}).status,0)
                fprintf(['    ' f{ii} '\n']);
            end
        end
    end
end

% elapsed time
et = cputime - st;

% output time as optional output
if nargout>1;
    varargout{1} = et;
end
if nargout>2;
    varargout{2} = Bhat;
end


fprintf('\n --- --- quickclassify DONE! --- --- \n');
fprintf(' Elapsed time: %5.2fs (%02d:%02d:%02d) \n\n',...
    et,floor(et/3600),floor(mod(et,3600)/60), floor(mod(et,60)));


% put back in original image space
mask_out = false(size(orig_mask));
if ~ismatrix(mask) % 3d
    mask_out(r1:r2,c1:c2,z1:z2)=Yhat>.5;
elseif ismatrix(mask) % 2d
    mask_out(r1:r2,c1:c2)=Yhat;
end


mask_out = logical(mask_out);


function out = make_im_uninteresting(im,mask)
% this is a quirky little idea that I had to get rid of some
% of the 'more interesting than my feature' features

% get rid of these bright spots
% repeat until image doesn't change

s = 5; % median filter param
se = strel('disk',s); % param for image dilation
max_its = 3;

% preallocate/initialize
Y = mask(:);
out = im;
out_old = zeros(size(im));
its=0;

while ~isequal(out,out_old) && its < max_its;
    
    % medial filter/denoise
    filt = medfilt3(out,[s s s]);
    
    % do a quick intensity segmentation (non robust);
    B = glmfit(filt(:),Y(:),'binomial','link','logit');
    Y = 1./(1+exp(-[ones(numel(im),1) filt(:)]*B))>.5;
    Y = reshape(Y,size(im));
    
    % set these out-of-mask areas = 0
    out_old = out;
    out(imdilate(Y,se) & ~mask)=0;
    its = its+1; fprintf('.');
end
fprintf('\n');

function out = enforce_no_coffee_cups(in)
% morphological 'no-pockets' function
% applies imfill in all 3 dims
% applies imopen in all 3 dims

out = in;

% fill in any holes
for ii = 1:size(in,3);
    out(:,:,ii) = imfill(out(:,:,ii),'holes')>0.5;
end
for ii = 1:size(in,2);
    out(:,ii,:) = imfill(squeeze(out(:,ii,:)),'holes')>0.5;
end
if ~ismatrix(in)
    for ii = 1:size(in,1);
        out(ii,:,:) = imfill(squeeze(out(ii,:,:)),'holes')>0.5;
    end
end

function varargout = display_im(im,seg,sl,handle)
% im can be [] to ignore it

if nargin<4
    figure;
else
    figure(handle);
end

if nargin<3;
    % show middle slice
    sl = find(squeeze(sum(sum(seg,1),2)));
    sl = sl(round(length(sl)/2));
end

if ~isempty(im)
    plot_segmentation_overlay(im,seg,[],[],[],sl);
else
    imagesc(seg(:,:,sl)); colormap jet; colorbar; 
end

axis off;

drawnow; 


if nargout;
    varargout{1} = sl;
end

function parse_yourself(feature_struct)
% given feat_struct see if every field in this function has a corresponding
% handle

% count lines
fid = fopen('quickclassify.m');
Nrows = numel(cell2mat(textscan(fid,'%1c%*[^\n]')));
fclose(fid);

lines = cell(Nrows,1);

fid = fopen('quickclassify.m');
i=1;
while 1
    tmp = fgetl(fid);
    if ~ischar(tmp), break, end
    lines{i} = strtrim(tmp);
    i=i+1;
end
fclose(fid);
lines = lines(1:end-1);
inds = find(strncmp(lines,'if isfield(feature_struct,',25));
lines = lines(inds);
for ii=1:length(lines)
    ind = strfind(lines(ii),''''); % aline with a quote
    feature = lines{ii}(ind{1}(1)+1:ind{1}(2)-1);
    if ~isfield(feature_struct,feature);
        fprintf(['Feature: "%s" is available in quickclassify, '...
            'but is not included as a field of feature_struct.\n'...
            'Check line %i\n'],feature,inds(ii));
    end
end


% remove blobs
% while ~isequal(out_old, out)
%     out_old = out;
%     for ii = 1:size(in,3);
%         out(:,:,ii) = clear_blobs(out(:,:,ii));
%     end
%     for ii = 1:size(in,2);
%         out(:,ii,:) = clear_blobs(out(:,ii,:));
%     end
%     if ~ismatrix(in)
%         for ii = 1:size(in,1);
%             out(ii,:,:) = clear_blobs(out(ii,:,:));
%         end
%     end
% end



% % attempt to use lassplore
%
% XR = cim(:); % matrix of size mxn, each row a sample
% [m n] = size(XR);
% YR = Y; YR(YR==0)=-1; % response vector, with -1/1 elements
% z = 500; % radius of l1 ball
% rho = 0; % two norm reg param
% x0 = zeros(n,1); % initial guess of weights params
% c0 = 0; % init guess of intercept
% gamma = 1; % param for adaptive line search
% mu = rho; % lower bound of strongly convex param (safeguarded init)
% L = 1/m; % step size
% flag = 4; % stpping criterion (4 is run max its)
% tol = 1e-4; % relative gap (not used with flag =4)
% maxIter = 500; % max iteratiosn (used with flag =4)
%
% [x2,c2,L_val,fun_val] = lassplore(XR,YR,z,rho,...
%     x0,c0,...
%     gamma, mu, L, flag, tol, maxIter);
