function obs = apply_voxelwise_random_model(truth, cm)
% APPLY_VOXELWISE_RANDOM_MODEL - uses a confusion matrix to observe a given
%                                truth model in a voxelwise random sense
%
% obs = apply_voxelwise_random_model(truth, cm)
%
% Input: truth - the N-D truth model
%        cm - the L x L confusion matrix, the sum of all columns should equal 1
%
% Output: obs - the N-D observation of the truth model (truth)
%               given the confusion matrix (cm)
%
% Bennett Landman, bennett.landman@vanderbilt.edu
% Andrew Asman, andrew.j.asman@vanderbilt.edu
% May 4, 2011

% fix the labels if they begin with 0
minlabel = min(truth(:));
if minlabel == 0
    truth = truth + 1;
end

% Step 1: Check inputs
if(min(truth(:))<1)
    error('Minimum truth label must be >0');
end
L = size(cm,1);
if(L~=size(cm,2))
    error('Confusion Matrix must be square');
end
if(max(truth(:))>L)
    error('Maximum label must be less than or equal to the size of the confusion matrix');
end

% Step 2: Create an random number for each vx uniform in [0 1]
vxRand = rand(size(truth(:)));

% Step 3: Generate a CDF for each voxel
cmCDF = cumsum(cm,1);
vxCMCDF = cmCDF(:,truth(:))';

% Step 4: Map Uniform distribution to discrete distribution
obs =reshape(1+sum(repmat(vxRand,[1 L])>vxCMCDF,2),size(truth));

% reset the labels to zero if applicable
if minlabel == 0
    obs = obs - 1;
end
