function [obs cms truth] = consensus_simulation_COLLATE_model(...
                                                  num_raters, num_slices, ...
                                                  num_labels, length_im, ...
                                                  length_im_confusion, ...
                                                  lower_lim_diag, ...
                                                  upper_lim_diag)
% CONSENSUS_SIMULATION_COLLATE_MODEL - observe using the COLLATE model
% - This is mainly meant to be used to compare the STAPLE and COLLATE models
%
% [obs cms truth] = consensus_simulation_COLLATE_model(...
%                                                 num_raters, num_slices, ...
%                                                 num_labels, length_im, ...
%                                                 length_im_confusion, ...
%                                                 lower_lim_diag, ...
%                                                 upper_lim_diag);
%
% Input: num_raters - number of raters to observe
%        num_slices - number of slices for each volume
%        num_labels - number of labels on each slice
%        length_im - the length of square slices
%        length_im_confusion - size of the confusion region
%        lower_lim_diag - diagonal value for worst rater
%        upper_lim_diag - diagonal value for best rater
%
% Output: obs - the observation struct
%         cms - the true confusion matrices
%         truth - the true "segmentation"

%
% create the confusion matrices
%
diag_vals = linspace(lower_lim_diag, upper_lim_diag, num_raters);

% allocate space for the confusion matrices
cms = zeros([num_labels num_labels num_raters]);

% iterate over all of the raters
for i = 1:num_raters
    % set the diagonal values
    cm = diag(diag_vals(i) * ones([1 num_labels]));

    % fill in the off-diagonal values
    for j = 1:num_labels
        % generate (num_labels - 1) values to fill in the off-diagonals
        nums = rand([1 (num_labels-1)]);

        % calculate how much probability is available
        ava = 1 - diag_vals(i);

        % normalize these values so that we can just slap them into the cm
        nums = ava * (nums / sum(nums));

        count = 0;
        for k = 1:num_labels
            if (j ~= k)
                count = count + 1;
                cm(k, j) = nums(count);
            end
        end
    end
    cms(:,:,i) = cm;
end

% generate the initial truth images
truth = zeros([length_im length_im num_slices]);
x = round(linspace(0, length_im, num_labels));
for i = 1:num_slices
    label_perm = randperm(num_labels - 1);
    for j = 1:num_labels-1
        truth(:, x(j)+1:x(j+1), i) = label_perm(j);
    end
end

% create a section in the middle which is where all of the variation occurs
truth_section = zeros([length_im_confusion length_im_confusion]);

% create the section where the variation will be applied
%x = round(linspace(0, length_im_confusion, num_labels+1));
x = round(linspace(1, length_im_confusion+1, num_labels+1))-1;
for i = 1:num_labels
    truth_section(x(i)+1:x(i+1), :) = i-1;
end

% find where we want to place the truth_section in the truth images
bind_l = floor(length_im / 2) - floor(length_im_confusion / 2);
eind_l = bind_l + length_im_confusion - 1;
bind_w = floor(length_im / 2) - floor(length_im_confusion / 2);
eind_w = bind_w + length_im_confusion - 1;

% apply the truth section to each image
for i = 1:num_slices
    truth(bind_l:eind_l, bind_w:eind_w, i) = truth_section;
end

% create the user data
data = zeros(length_im, length_im, num_slices, num_raters);

% iterate over all raters
for r = 1:num_raters
    % iterate over all slices
    for s = 1:num_slices

        % apply the confusion matrix to the truth section
        obs_section = apply_voxelwise_random_model(truth_section+1, ...
                                                   squeeze(cms(:,:,r))) - 1;

        % apply the observed section to the truth data
        obs = squeeze(truth(:,:,s));
        obs(bind_l:eind_l, bind_w:eind_w) = obs_section;

        data(:,:,s,r) = obs;
    end
end

% create the observations
obs = create_obs('volume', [length_im length_im num_slices]);
for r = 1:num_raters
    obs = add_obs(obs, data(:, :, :, r));
end

