function [obs truth] = consensus_simulation_boundary_model(num_raters, ...
                                        num_slices, length_im, shiftiness);
% CONSENSUS_SIMULATION_BOUNDARY_MODEL - observe using a boundary only model
% - This is mainly meant to be used to compare the STAPLE and COLLATE models
%
% [obs truth] = consensus_simulation_boundary_model(num_raters, num_slices, ...
%                                                   length_im, shiftiness);
%
% Input: num_raters - number of raters to observe
%        num_slices - number of slices for each volume
%        length_im - the length of square slices
%        shiftiness - the amount of boundary error (e.g. 1.6)
% Output: obs - the observation struct
%         truth - the true "segmentation"
%

% initial settings
num_labels = 5; % don't change
filt_size = round(length_im / 8);
filt_std = 2;
edge_thresh = 0.01;
show_ims = 0;
first_vertical = length_im / 2;
second_vertical = 5 * length_im / 8;

label_nums = zeros(num_slices, 8);
for i = 1:num_slices
    x = [randperm(2)-1, randperm(3)+1];
    label_nums(i, :) = [x(1) x(2) x(3) x(2) x(4) x(1) x(5) x(2)]; %[x x(1) x(1) x(1)];
end

%
% create the truth
%
truth = zeros(length_im, length_im, num_slices); % num_slices);
c = zeros(length_im, length_im , num_slices);

for n = 1:size(label_nums, 1)
    % define the locations for each label
    ind_l1 = [...
              1, ...
              floor(first_vertical) + 1, ...
              1, ...
              1, ...
              floor(first_vertical) + 1, ...
              floor(first_vertical) + 1, ...
              floor(second_vertical) + 1, ...
              floor(second_vertical) + 1, ...
             ];
    ind_l2 = [...
              floor(first_vertical), ...
              length_im, ...
              floor(first_vertical), ...
              floor(first_vertical), ...
              floor(second_vertical), ...
              floor(second_vertical), ...
              length_im, ...
              length_im, ...
             ];
    ind_w1 = [...
              1, ...
              1, ...
              floor(first_vertical) + 1, ...
              floor(second_vertical) + 1, ...
              floor(first_vertical) + 1, ...
              floor(second_vertical) + 1, ...
              floor(first_vertical) + 1, ...
              floor(second_vertical) + 1, ...
             ];
    ind_w2 = [...
              floor(first_vertical), ...
              floor(first_vertical), ...
              floor(second_vertical), ...
              length_im, ...
              floor(second_vertical), ...
              length_im, ...
              floor(second_vertical), ...
              length_im, ...
             ];

    % set the values
    for i = 1:length(ind_l1)
        truth(ind_l1(i):ind_l2(i), ind_w1(i):ind_w2(i), n) = label_nums(n, i);
    end

    % find the edges
    edge_im = double(edge(squeeze(truth(:,:,n)), edge_thresh));

    % create a gaussian blurring filter
    gaussian_filter = fspecial('gaussian', [filt_size filt_size], filt_std);

    % apply the blurring filter
    gauss_edge_im = imfilter(edge_im, gaussian_filter, 'replicate', 'conv');

    % set the consensus values
    %gauss_edge_im(gauss_edge_im > 0.01) = 1;
    c(:, :, n) = gauss_edge_im / max(max(gauss_edge_im));

end

% create the user data
data = zeros(length_im, length_im, num_slices, num_raters);

% iterate over all raters
for r = 1:num_raters
    % iterate over all slices
    for s = 1:num_slices

        im = squeeze(truth(:, :, s));

        shift_factor = shiftiness * r^(1/3);

        %
        % first handle the vertical lines
        %

        % middle vertical line
        x = floor(first_vertical);
        for y = 1:length_im
            shift = round(randn * shift_factor);
            if (shift < 0)
                im(y, max([(x+shift) 1]):x) = im(y, min([x-shift length_im]));
            else
                im(y, x:min([(x+shift) length_im])) = im(y, max([x-shift 1]));
            end
        end

        % 3rd quartile vertical line
        x = floor(second_vertical);
        for y = 1:length_im
            shift = round(randn * shift_factor);
            if (shift < 0)
                im(y, max([(x+shift) 1]):x) = im(y, min([x-shift length_im]));
            else
                im(y, x:min([(x+shift) length_im])) = im(y, max([x-shift 1]));
            end
        end

        %
        % second handle the horizontal lines
        %

        % middle vertical line
        y = floor(first_vertical);
        for x = 1:length_im
            shift = round(randn * shift_factor);
            if (shift < 0)
                im(max([(y+shift) 1]):y, x) = im(min([y-shift length_im]), x);
            else
                im(y:min([(y+shift) length_im]), x) = im(max([y-shift 1]), x);
            end
        end

        % 3rd quartile vertical line
        y = floor(second_vertical);
        for x = ceil(first_vertical):length_im
            shift = round(randn * shift_factor);
            if (shift < 0)
                im(max([(y+shift) 1]):y, x) = im(min([y-shift length_im]), x);
            else
                im(y:min([(y+shift) length_im]), x) = im(max([y-shift 1]), x);
            end
        end
        % apply the confusion matrix to the truth section
        data(:,:,s,r) = im;

    end
end

% create the observations
obs = create_obs('volume', [length_im length_im num_slices]);
for r = 1:num_raters
    obs = add_obs(obs, data(:, :, :, r));
end

