function [obs truth] = spatial_atlas_simulation(num_raters);
% SPATIAL_ATLAS_SIMULATION - Spatial varying observation simulation
% - This is mainly meant to be used to compare STAPLE and Spatial STAPLE
%
% [obs truth] = spatial_atlas_simulation(num_raters);
%
% Input: num_raters - number of raters to observe
% Output: obs - the observation struct
%         truth - the true "segmentation"
%

num_labels = 15;
dims = [80 60 60];
shiftiness_min = 35 * 0.05;
shiftiness_max = 45 * 0.05;
num_shifties = 40;
shiftiness = linspace(shiftiness_min, shiftiness_max, num_shifties);

truth = zeros(dims);
data = zeros([dims num_raters]);

% create the truth cube

for i = 1:num_labels-1
    % fill in label i
    f = num_labels * 2;
    xstart = round(i * dims(1) / f);
    xend = dims(1) - round(i * dims(1) / f);
    ystart = round(i * dims(2) / f);
    yend = dims(2) - round(i * dims(2) / f);
    zstart = round(i * dims(3) / f);
    zend = dims(3) - round(i * dims(3) / f);
    truth(xstart:xend, ystart:yend, zstart:zend) = i;
end

truth = uint8(truth);

% initially assign the truth to each rater
for j = 1:num_raters
    data(:, :, :, j) = uint8(truth);
end

% create the shiftiness 4D matrix
shiftiness_matrix = zeros([dims num_raters]);

for j = 1:num_raters
    shrp = randperm(num_shifties);

    x1 = 1:round(dims(1)/4);
    x2 = round(dims(1)/4):round(dims(1)/2);
    x3 = round(dims(1)/2):round(dims(1)*(3/4));
    x4 = round(dims(1)*(3/4)):dims(1);
    y1 = 1:round(dims(2)/2);
    y2 = round(dims(2)/2):dims(2);
    z1 = 1:round(dims(3)/2);
    z2 = round(dims(3)/2):dims(3);

    shiftiness_matrix(x1, y1, z1, j) = shiftiness(shrp(1));
    shiftiness_matrix(x2, y1, z1, j) = shiftiness(shrp(2));
    shiftiness_matrix(x3, y1, z1, j) = shiftiness(shrp(3));
    shiftiness_matrix(x4, y1, z1, j) = shiftiness(shrp(4));
    shiftiness_matrix(x1, y2, z1, j) = shiftiness(shrp(5));
    shiftiness_matrix(x2, y2, z1, j) = shiftiness(shrp(6));
    shiftiness_matrix(x3, y2, z1, j) = shiftiness(shrp(7));
    shiftiness_matrix(x4, y2, z1, j) = shiftiness(shrp(8));
    shiftiness_matrix(x1, y1, z2, j) = shiftiness(shrp(1));
    shiftiness_matrix(x2, y1, z2, j) = shiftiness(shrp(2));
    shiftiness_matrix(x3, y1, z2, j) = shiftiness(shrp(3));
    shiftiness_matrix(x4, y1, z2, j) = shiftiness(shrp(4));
    shiftiness_matrix(x1, y2, z2, j) = shiftiness(shrp(5));
    shiftiness_matrix(x2, y2, z2, j) = shiftiness(shrp(6));
    shiftiness_matrix(x3, y2, z2, j) = shiftiness(shrp(7));
    shiftiness_matrix(x4, y2, z2, j) = shiftiness(shrp(8));
end


% only perform the shifting on the x and y components
% % the z-plane might be a little bit too thin to do this accurately

xshift_list_t = zeros(10000, 5);
yshift_list_t = zeros(10000, 5);

xs = 1:dims(1)-1;
xn = 2:dims(1);

ys = 1:dims(2)-1;
yn = 2:dims(2);

% get the xshift_list
c = 0;
for z = 1:dims(3)
    for y = 1:dims(2);
        for x = 1:(dims(1)-1)
            if (truth(x, y, z) ~= truth(x+1, y, z))
                c = c + 1;
                vals = [truth(x, y, z), truth(x+1, y, z)];
                if x < dims(1)/2
                    xshift_list_t(c, :) = [x+1, y, z, vals(1), vals(2)];
                else
                    xshift_list_t(c, :) = [x, y, z, vals(1), vals(2)];
                end
            end
        end
    end
end
xshift_list = xshift_list_t(1:c, :);

% get the yshift_list
c = 0;
for z = 1:dims(3)
    for x = 1:dims(1)
        for y = 1:(dims(2)-1);
            if (truth(x, y, z) ~= truth(x, y+1, z))
                c = c + 1;
                vals = [truth(x, y, z), truth(x, y+1, z)];
                if y < dims(2)/2
                    yshift_list_t(c, :) = [x, y+1, z, vals(1), vals(2)];
                else
                    yshift_list_t(c, :) = [x, y, z, vals(1), vals(2)];
                end
            end
        end
    end
end
yshift_list = yshift_list_t(1:c, :);

% iterate over each rater
for j = 1:num_raters

    % apply the x-shifts
    for i = 1:size(xshift_list, 1)
        x = xshift_list(i, 1);
        y = xshift_list(i, 2);
        z = xshift_list(i, 3);
        v1 = xshift_list(i, 4);
        v2 = xshift_list(i, 5);

        shift = round(randn * shiftiness_matrix(x, y, z, j));

        if shift < 0
            data(max([x+shift 1]):x, y, z, j) = v2;
        elseif shift > 0
            data(x:min([x+shift dims(1)]), y, z, j) = v1;
        end
    end

    % apply the y-shifts
    for i = 1:size(yshift_list, 1)
        x = yshift_list(i, 1);
        y = yshift_list(i, 2);
        z = yshift_list(i, 3);
        v1 = yshift_list(i, 4);
        v2 = yshift_list(i, 5);

        shift = round(randn * shiftiness_matrix(x, y, z, j));

        if shift < 0
            data(x, max([y+shift 1]):y, z, j) = v2;
        elseif shift > 0
            data(x, y:min([y+shift dims(2)]), z, j) = v1;
        end
    end
end

% make one rater perfect in each region
for i = 1:16

    % define all regions
    x1 = 1:round(dims(1)/4);
    x2 = round(dims(1)/4):round(dims(1)/2);
    x3 = round(dims(1)/2):round(dims(1)*(3/4));
    x4 = round(dims(1)*(3/4)):dims(1);
    y1 = 1:round(dims(2)/2);
    y2 = round(dims(2)/2):dims(2);
    z1 = 1:round(dims(3)/2);
    z2 = round(dims(3)/2):dims(3);

    if i == 1
        xr = x1; yr = y1; zr = z1;
    elseif i == 2
        xr = x2; yr = y1; zr = z1;
    elseif i == 3
        xr = x3; yr = y1; zr = z1;
    elseif i == 4
        xr = x4; yr = y1; zr = z1;
    elseif i == 5
        xr = x1; yr = y2; zr = z1;
    elseif i == 6
        xr = x2; yr = y2; zr = z1;
    elseif i == 7
        xr = x3; yr = y2; zr = z1;
    elseif i == 8
        xr = x4; yr = y2; zr = z1;
    elseif i == 9
        xr = x1; yr = y1; zr = z2;
    elseif i == 10
        xr = x2; yr = y1; zr = z2;
    elseif i == 11
        xr = x3; yr = y1; zr = z2;
    elseif i == 12
        xr = x4; yr = y1; zr = z2;
    elseif i == 13
        xr = x1; yr = y2; zr = z2;
    elseif i == 14
        xr = x2; yr = y2; zr = z2;
    elseif i == 15
        xr = x3; yr = y2; zr = z2;
    elseif i == 16
        xr = x4; yr = y2; zr = z2;
    end

    data(xr, yr, zr, i) = truth(xr, yr, zr);
end

% create the observations
obs = create_obs('volume', dims);
for r = 1:num_raters
    obs = add_obs(obs, data(:, :, :, r));
end



