#=========================================================================
#
#   Program:   draw3d
#   Copyright 2006-2011 Rupert Brooks
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#
#=========================================================================*/
package require vtk
package require vtkinteraction

#################################################################################
proc printhelp {} {
   puts "draw3d data rendering program"
   puts ""
   puts "Copyright 2006-2011 Rupert Brooks"
   puts "licensed under the Apache License, Version 2.0"
   puts ""
   puts "Usage: draw3d {commands, datafiles}"
   puts "the commands are of 3 kinds"
   puts "1. Commands that change program operation "
   puts "    help               : print this help"
   puts "    render <filename>  : dont display, render to file and exit"
   puts ""
   puts "2. Commands that change expected data type"
   puts "   After one of these appears on the command line, subsequent data files"
   puts "   Are treated in this manner untill the next data type command is found"
   puts "    polydata       : vtk polydata files"
   puts "    volume         : volumetric images, volume rendered" 
   puts "    planes         : volumetric images, rendered with vtkImagePlaneWidgets" 
   puts "    surface        : volumetric images, rendered as an isosurface" 
   puts "    slice          : a 1-pixel thick volume, e.g. an Ultrasound slice" 
   puts "    outline        : render just the bounding box of a volumetric image"
   puts "    polyoutline    : render just the bounding box of a vtk polydata"
   puts "    points         : expects a text file of x y z coordinates"
   puts "    lines          : as above, but draws as a connected polyline" 
   puts "    vectors        : expects file of x1 y1 z1 x2 y2 z2 and draws lines"
   puts "                   : from the first point to the second"
   puts "    tensors        : expects a vtk poly data with tensor attributes"
   puts "    defgrid        : draws a vtkpolydata as lines and dots" 
   puts "    text           : puts text in the bottom left"
   puts "    axes           : draw coordinate system axes"
   puts ""
   puts "3. Commands that change drawing parameters"
   puts "    pointsize       : All subsequent points will be drawn with spheres of this size"
   puts "    linewidth       : All subsequent lines will be drawn with this width"
   puts "    axessize        : All subsequent axes will be drawn at this length"
   puts "    isolevel        : All subsequent surface extraction will be done at this isolevel"
   puts "    color <color>   : draw subsequent items in this color. <color> is a string of 3"
   puts "                      floats ranging 0 to 1 as a single argument, eg '1 0.5 0.5' "
   puts "                      is pinkish"
   puts "    red/blue/yellow/green/purple/white/black: set next color to these named colors"
   puts "    azimuth <num>   : Set Camera azimuth to num before other settings"
   puts "    zoom    <num>   : Set Camera zoom to num"
   puts "    opacity <num>   : Set opacity for subsequent items (only partially supported)" 
   puts "    size <w h>      : Set window/image size to w x h "
   puts "    fazimuth <num>  : Set Camera azimuth to num after all other settings"
   puts "    roll <num>      : Set Camera roll to num"
   puts "    elevation <num> : Set Camera elevation to num"
   puts "    stereo          : Set stereo rendering to on (Default off)"
   puts "    on              : Subsequent datasets will start visible"
   puts "    off             : Subsequent datasets will start invisible"
   puts "The datafiles must conform to the expected data type" 
}

#################################################################################
proc addobjtolist {filename obj cmdstring outline visibility} {
   global color
   global check$obj
   global seton
   global renWin
   global lastObject
   set check$obj $seton
   set cs .top.f3.canvas.f
   set lastObject $cs.check$obj
   checkbutton $cs.check$obj -variable check$obj -text $filename -command $cmdstring
   pack $cs.check$obj

   eval $cmdstring

   bind $cs.check$obj <Leave> "
   global check$obj
   \[$outline GetProperty\] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   if {$visibility} {
      eval $outline SetVisibility \$check$obj
   } else {
      $outline SetVisibility 0
   }
   \$renWin Render
   "
   bind $cs.check$obj <Enter> "
   \[$outline GetProperty\] SetColor  1 0.41 0.71
   $outline SetVisibility 1
   \$renWin Render"
}

#################################################################################
proc readdata {filename obj} {
   global reader-$obj
   global transmat-$obj

   vtkMatrix4x4 transmat-$obj
   switch -glob "$filename" {
   *.vtk {
      # puts "Reading as vtk structured points"
      vtkStructuredPointsReader reader-$obj
      reader-$obj SetFileName $filename
      reader-$obj Update
      transmat-$obj SetElement 0 0 1
      transmat-$obj SetElement 0 1 0
      transmat-$obj SetElement 0 2 0
      transmat-$obj SetElement 0 3 0
      transmat-$obj SetElement 1 0 0
      transmat-$obj SetElement 1 1 1
      transmat-$obj SetElement 1 2 0
      transmat-$obj SetElement 1 3 0
      transmat-$obj SetElement 2 0 0
      transmat-$obj SetElement 2 1 0
      transmat-$obj SetElement 2 2 1
      transmat-$obj SetElement 2 3 0
      transmat-$obj SetElement 3 0 0
      transmat-$obj SetElement 3 1 0
      transmat-$obj SetElement 3 2 0
      transmat-$obj SetElement 3 3 1
   }
   *.mh[da] {
      #puts "Reading as UNC meta image"
         vtkMetaImageReader metareader-$obj
         metareader-$obj SetFileName $filename
         vtkImageChangeInformation reader-$obj
         reader-$obj SetInputConnection [metareader-$obj GetOutputPort]
         reader-$obj SetOutputOrigin 0 0 0
     # cause we set the origin below...
     #reader-$obj Update
     # ok, this is irritating, the vtk mha reader does
     # not read in the transform matrix, so, read it in
     # myself
     #
         set file [open $filename r]
     # initialize matrix and offset, so if not
     # found behavior is reasonable
         set a 1
         set b 0
         set c 0
         set d 0
         set e 1
         set f 0
         set g 0
         set h 0
         set i 1
         set offx 0
         set offy 0
         set offz 0

         while { [gets $file line] != -1 } {
            scan $line "%s" firstToken
            if { $firstToken == "TransformMatrix" } {
            scan $line "%s %s %f %f %f %f %f %f %f %f %f" firstToken eqtoken a b c d e f g h i
    #puts "Matrix is:"
    #puts "$a $b $c"
    #puts "$d $e $f"
    #puts "$g $h $i"
         }
         if { $firstToken == "Offset" } {
            scan $line "%s %s %f %f %f" firstToken eqtoken offx offy offz
    #puts "$offx $offy $offz"
         }
      }
      transmat-$obj SetElement 0 0 $a
      transmat-$obj SetElement 0 1 $b
      transmat-$obj SetElement 0 2 $c
      transmat-$obj SetElement 0 3 0
      transmat-$obj SetElement 1 0 $d
      transmat-$obj SetElement 1 1 $e
      transmat-$obj SetElement 1 2 $f
      transmat-$obj SetElement 1 3 0
      transmat-$obj SetElement 2 0 $g
      transmat-$obj SetElement 2 1 $h
      transmat-$obj SetElement 2 2 $i
      transmat-$obj SetElement 2 3 0
      transmat-$obj SetElement 3 0 0
      transmat-$obj SetElement 3 1 0
      transmat-$obj SetElement 3 2 0
      transmat-$obj SetElement 3 3 1
      transmat-$obj Invert
      transmat-$obj SetElement 0 3 $offx
      transmat-$obj SetElement 1 3 $offy
      transmat-$obj SetElement 2 3 $offz

   }


}

}
#################################################################################
proc drawpolyoutline {filename color obj} {
global aRenderer
global reader-$obj
global transmat-$obj
global outline-$obj
global linewidth

vtkPolyDataReader vpreader-$obj
vpreader-$obj SetFileName $filename

vtkOutlineFilter outlineData-$obj
outlineData-$obj SetInputConnection  [vpreader-$obj GetOutputPort]


   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [outline-$obj GetProperty] SetLineWidth $linewidth

   aRenderer AddActor outline-$obj
   global check$obj
   set check$obj 1
   set cmdstring "outline-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 1

}

#################################################################################
proc drawoutline {filename color obj} {
   global aRenderer
   global reader-$obj
   global transmat-$obj
   global outline-$obj
   global linewidth

   readdata $filename $obj

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [reader-$obj GetOutputPort]


   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [outline-$obj GetProperty] SetLineWidth $linewidth

   outline-$obj SetUserMatrix transmat-$obj
   aRenderer AddActor outline-$obj
   global check$obj
   set check$obj 1
   set cmdstring "outline-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 1

}
#################################################################################
proc drawsurface {filename color obj} {
   global aRenderer
   global reader-$obj
   global transmat-$obj
   global opacity
   global isolevel
   global wireframe

   readdata $filename $obj

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [reader-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   outline-$obj SetUserMatrix transmat-$obj
   aRenderer AddActor outline-$obj



   vtkContourFilter skinExtractor-$obj
   skinExtractor-$obj SetInputConnection  [reader-$obj GetOutputPort]
   skinExtractor-$obj SetValue 0 $isolevel
   skinExtractor-$obj Update
   set verts [[skinExtractor-$obj GetOutput] GetNumberOfVerts]
   set lines [[skinExtractor-$obj GetOutput] GetNumberOfLines]
   set polys [[skinExtractor-$obj GetOutput] GetNumberOfPolys]
   set strips [[skinExtractor-$obj GetOutput] GetNumberOfStrips]
#puts "verts are $verts"
#puts "lines are $lines"
#puts "polys are $polys"
#puts "strips are $strips"
   vtkTriangleFilter tri-$obj
   tri-$obj SetInputConnection [skinExtractor-$obj GetOutputPort]
   tri-$obj Update
   set verts [[tri-$obj GetOutput] GetNumberOfVerts]
   set lines [[tri-$obj GetOutput] GetNumberOfLines]
   set polys [[tri-$obj GetOutput] GetNumberOfPolys]
   set strips [[tri-$obj GetOutput] GetNumberOfStrips]
#puts "verts are $verts"
#puts "lines are $lines"
#puts "polys are $polys"
#puts "strips are $strips"
   vtkQuadricDecimation deci-$obj
   deci-$obj SetInputConnection [tri-$obj GetOutputPort]
   deci-$obj SetTargetReduction 0.95
# deci-$obj PreserveTopologyOn
   deci-$obj Update
   set verts [[deci-$obj GetOutput] GetNumberOfVerts]
   set lines [[deci-$obj GetOutput] GetNumberOfLines]
   set polys [[deci-$obj GetOutput] GetNumberOfPolys]
   set strips [[deci-$obj GetOutput] GetNumberOfStrips]
#puts "verts are $verts"
#puts "lines are $lines"
#puts "polys are $polys"
#puts "strips are $strips"
   vtkPolyDataNormals skinNormals-$obj
#  skinNormals SetInputConnection [skinExtractor GetOutputPort]
   skinNormals-$obj SetInputConnection [deci-$obj GetOutputPort]
   skinNormals-$obj SetFeatureAngle 60.0
   vtkPolyDataMapper skinMapper-$obj
   skinMapper-$obj SetInputConnection [skinNormals-$obj GetOutputPort]
#  skinMapper SetInputConnection [deci GetOutputPort]
   skinMapper-$obj ScalarVisibilityOff
   vtkActor skin-$obj
   skin-$obj SetMapper skinMapper-$obj
   [skin-$obj GetProperty]  SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   if { $wireframe > 0 } then {
       puts "Wireframe"
      [skin-$obj GetProperty] SetRepresentationToWireframe
   }
   #[skin-$obj GetProperty]  SetEdgeColor 1 0 0
   #[skin-$obj GetProperty]  EdgeVisibilityOn
   [skin-$obj GetProperty]  SetOpacity $opacity


   skin-$obj SetUserMatrix transmat-$obj
   aRenderer AddActor skin-$obj

   set cmdstring "outline-$obj SetVisibility \$check$obj; skin-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 1

}
#################################################################################
proc drawslice {filename color obj} {
   global aRenderer
   global reader-$obj
   global transmat-$obj

   readdata $filename $obj
   vtkImageChangeInformation changer-$obj
   changer-$obj SetInputConnection [reader-$obj GetOutputPort]
#changer-$obj SetOutputOrigin 0 0 0

   set data [changer-$obj GetOutput]
#$data SetOrigin $offx $offy $offz


   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [changer-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]

   outlineData-$obj Update
   set pts [outlineData-$obj GetOutput]
   #$pts Update
   set p0 [$pts GetPoint 0]
   set p1 [$pts GetPoint 1]
   set p2 [$pts GetPoint 2]
   set p3 [$pts GetPoint 3]

   outline-$obj SetUserMatrix transmat-$obj
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0


   vtkImageMapToColors colormap-$obj
   colormap-$obj SetInputConnection [ changer-$obj GetOutputPort ]
   colormap-$obj SetLookupTable graymapsolid
   colormap-$obj SetOutputFormatToRGBA;
   colormap-$obj PassAlphaToOutputOn;

   vtkTexture texture-$obj
   texture-$obj SetInputConnection [ colormap-$obj GetOutputPort ]
   texture-$obj InterpolateOn
   texture-$obj SetLookupTable graymapsolid
   vtkPiecewiseFunction opacityTransferFunction2-$obj
   opacityTransferFunction2-$obj AddSegment  0   0.0 255  0.5

   set imdata [changer-$obj GetOutput]
   set spacing [$imdata GetSpacing]
   set origin [$imdata GetOrigin]
   set extent [$imdata GetExtent]

   vtkPlaneSource imageplane-$obj
   imageplane-$obj SetXResolution 1
   imageplane-$obj SetYResolution 1
   imageplane-$obj SetPoint1 [lindex $p1 0] [lindex $p1 1] [lindex $p1 2]
   imageplane-$obj SetOrigin [lindex $p0 0] [lindex $p0 1] [lindex $p0 2]
   imageplane-$obj SetPoint2 [lindex $p2 0] [lindex $p2 1] [lindex $p2 2]
   vtkPolyDataMapper map-$obj
   map-$obj SetInputConnection [ imageplane-$obj GetOutputPort ]
   vtkActor slice-$obj
   slice-$obj SetMapper map-$obj
   slice-$obj SetTexture texture-$obj
   slice-$obj SetUserMatrix transmat-$obj
   aRenderer AddActor slice-$obj

   set cmdstring "slice-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0

}
#################################################################################
proc drawvolume {filename color obj} {
   global aRenderer
   global reader-$obj
   global transmat-$obj

   readdata $filename $obj

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [reader-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]

   outline-$obj SetUserMatrix transmat-$obj

   aRenderer AddActor outline-$obj


   vtkImageResample resampler1-$obj
   resampler1-$obj SetAxisMagnificationFactor 0 0.5
   resampler1-$obj SetAxisMagnificationFactor 1 0.5
   resampler1-$obj SetAxisMagnificationFactor 2 0.5

   vtkImageResample resampler2-$obj
   resampler2-$obj SetAxisMagnificationFactor 0 0.25
   resampler2-$obj SetAxisMagnificationFactor 1 0.25
   resampler2-$obj SetAxisMagnificationFactor 2 0.25

   resampler1-$obj SetInputConnection [reader-$obj GetOutputPort]
   resampler2-$obj SetInputConnection [resampler1-$obj GetOutputPort]

# Create transfer mapping scalar value to opacity
   vtkPiecewiseFunction opacityTransferFunction-$obj
   opacityTransferFunction-$obj AddSegment  0 0.0 100  0.5

   vtkPiecewiseFunction opacityTransferFunction2-$obj
   opacityTransferFunction2-$obj AddSegment  0   0.0 255  0.5

# Create transfer mapping scalar value to color
   vtkColorTransferFunction colorTransferFunction-$obj
   colorTransferFunction-$obj AddRGBPoint      0.0 0.0 0.0 0.0
   colorTransferFunction-$obj AddRGBPoint     64.0 1.0 0.0 0.0
   colorTransferFunction-$obj AddRGBPoint    128.0 0.0 0.0 1.0
   colorTransferFunction-$obj AddRGBPoint    192.0 0.0 1.0 0.0
   colorTransferFunction-$obj AddRGBPoint    255.0 0.0 0.2 0.0

# The property describes how the data will look
   vtkVolumeProperty volumeProperty-$obj
   volumeProperty-$obj SetColor colorTransferFunction-$obj
   volumeProperty-$obj SetScalarOpacity opacityTransferFunction-$obj
   volumeProperty-$obj ShadeOn
   volumeProperty-$obj SetInterpolationTypeToLinear

   vtkVolumeProperty volumeProperty2-$obj
   volumeProperty2-$obj SetColor colorTransferFunction-$obj
   volumeProperty2-$obj SetScalarOpacity opacityTransferFunction2-$obj
   volumeProperty2-$obj ShadeOn
   volumeProperty2-$obj SetInterpolationTypeToLinear

# The mapper / ray cast function know how to render the data
   vtkVolumeRayCastCompositeFunction  compositeFunction-$obj
   vtkVolumeRayCastMapper volumeMapperlo-$obj
   volumeMapperlo-$obj SetVolumeRayCastFunction compositeFunction-$obj
   vtkVolumeRayCastMapper volumeMappermed-$obj
   volumeMappermed-$obj SetVolumeRayCastFunction compositeFunction-$obj
   vtkVolumeRayCastMapper volumeMapperhi-$obj
   volumeMapperhi-$obj SetVolumeRayCastFunction compositeFunction-$obj
   volumeMapperlo-$obj SetInputConnection [resampler2-$obj GetOutputPort]
   volumeMappermed-$obj SetInputConnection [resampler1-$obj GetOutputPort]
   volumeMapperhi-$obj SetInputConnection [reader-$obj GetOutputPort]

   vtkLODProp3D volumeLOD-$obj
   volumeLOD-$obj AddLOD volumeMapperlo-$obj volumeProperty-$obj 0.0
   volumeLOD-$obj AddLOD volumeMappermed-$obj volumeProperty-$obj 0.0
   volumeLOD-$obj AddLOD volumeMapperhi-$obj volumeProperty2-$obj 0.0

   volumeLOD-$obj SetUserMatrix transmat-$obj
   aRenderer AddProp volumeLOD-$obj

   set cmdstring "outline-$obj SetVisibility \$check$obj; volumeLOD-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 1

}
#################################################################################
proc drawplanes {filename color obj} {
   global aRenderer
   global reader-$obj
   global transmat-$obj
   global renWin

   readdata $filename $obj

   vtkImageChangeInformation changer-$obj
   changer-$obj SetInputConnection [reader-$obj GetOutputPort]
   changer-$obj SetOutputOrigin [transmat-$obj GetElement 0 3] [transmat-$obj GetElement 1 3] [transmat-$obj GetElement 2 3]

   vtkImageChangeInformation changer2-$obj
   changer2-$obj SetInputConnection [reader-$obj GetOutputPort]

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [changer2-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]

   outline-$obj SetUserMatrix transmat-$obj

   outline-$obj SetVisibility 0

   aRenderer AddActor outline-$obj

   reader-$obj Update
   set imdata [reader-$obj GetOutput]
   set extent [$imdata GetExtent]
   set xsize [expr abs ( [lindex $extent 0] - [lindex $extent 3]  ) + 1 ]
   set ysize [expr abs ( [lindex $extent 1] - [lindex $extent 4]  ) + 1 ]
   set zsize [expr abs ( [lindex $extent 2] - [lindex $extent 5]  ) + 1 ]



   vtkCellPicker picker-$obj
   picker-$obj SetTolerance 0.005

   if { $ysize > 1 && $zsize > 1 } then {
      vtkImagePlaneWidget planeWidgetX-$obj
      planeWidgetX-$obj DisplayTextOn
      planeWidgetX-$obj SetInputConnection [changer-$obj GetOutputPort]
     
      #planeWidgetX-$obj SetWindowLevel 255 127 0
      planeWidgetX-$obj SetPlaneOrientationToXAxes
      planeWidgetX-$obj SetSliceIndex 0
      planeWidgetX-$obj SetPicker picker-$obj
      planeWidgetX-$obj SetKeyPressActivationValue "x"
      planeWidgetX-$obj TextureInterpolateOn
      planeWidgetX-$obj SetResliceInterpolate 1
      set prop1 [planeWidgetX-$obj GetPlaneProperty]
      $prop1 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
      set prop2 [planeWidgetX-$obj GetTextProperty]
      $prop2 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]

      planeWidgetX-$obj SetLookupTable graymap

      planeWidgetX-$obj SetInteractor [$renWin GetInteractor]

      planeWidgetX-$obj On
      #puts "Window LEvel: [planeWidgetX-$obj GetWindow] [planeWidgetX-$obj GetLevel]"
      set cmdstring "planeWidgetX-$obj SetEnabled \$check$obj\_x ; \$renWin Render"
      addobjtolist "[lindex [file split $filename] end] (X)" "$obj\_x" $cmdstring outline-$obj 0
   }

   if { $xsize > 1 && $zsize > 1} then {
      vtkImagePlaneWidget planeWidgetY-$obj
      planeWidgetY-$obj DisplayTextOn
      planeWidgetY-$obj SetInputConnection [changer-$obj GetOutputPort]
      planeWidgetY-$obj SetPlaneOrientationToYAxes
      planeWidgetY-$obj SetSliceIndex 0
      planeWidgetY-$obj SetPicker picker-$obj
      planeWidgetY-$obj SetKeyPressActivationValue "y"
      planeWidgetY-$obj TextureInterpolateOn
      planeWidgetY-$obj SetResliceInterpolate 1
      set prop1 [planeWidgetY-$obj GetPlaneProperty]
      $prop1 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
      set prop2 [planeWidgetY-$obj GetTextProperty]
      $prop2 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]

      planeWidgetY-$obj SetInteractor [$renWin GetInteractor]
      planeWidgetY-$obj On
      if {$ysize > 1  } then {
      planeWidgetY-$obj SetLookupTable [planeWidgetX-$obj GetLookupTable]
      } else {
         planeWidgetY-$obj SetLookupTable graymap
         planeWidgetY-$obj SetWindowLevel 255 127 0
      }
      set cmdstring "planeWidgetY-$obj SetEnabled \$check$obj\_y ; \$renWin Render"
      addobjtolist "[lindex [file split $filename] end] (Y)" "$obj\_y" $cmdstring outline-$obj 0
   }

   if { $xsize > 1 && $ysize > 1 } then {
      vtkImagePlaneWidget planeWidgetZ-$obj
      planeWidgetZ-$obj DisplayTextOn
      planeWidgetZ-$obj SetInputConnection [changer-$obj GetOutputPort]
      planeWidgetZ-$obj SetPlaneOrientationToZAxes
      planeWidgetZ-$obj SetSliceIndex 0
      planeWidgetZ-$obj SetPicker picker-$obj
      planeWidgetZ-$obj SetKeyPressActivationValue "z"
      planeWidgetZ-$obj TextureInterpolateOn
      planeWidgetZ-$obj SetResliceInterpolate 1
      set prop1 [planeWidgetZ-$obj GetPlaneProperty]
      $prop1 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
      set prop2 [planeWidgetZ-$obj GetTextProperty]
      $prop2 SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]

      planeWidgetZ-$obj SetInteractor [$renWin GetInteractor]

      planeWidgetZ-$obj On

      if {$zsize > 1} then {
         planeWidgetZ-$obj SetLookupTable [planeWidgetX-$obj GetLookupTable]
      } else {
         planeWidgetZ-$obj SetLookupTable graymap
         planeWidgetZ-$obj SetWindowLevel 255 127 0
      }

      set cmdstring "planeWidgetZ-$obj SetEnabled \$check$obj\_z ; \$renWin Render"
      addobjtolist "[lindex [file split $filename] end] (Z)" "$obj\_z" $cmdstring outline-$obj 0

   }
}
#################################################################################

proc writebutton {} {
   wm deiconify .dlg
}

#################################################################################
proc write {} {
   puts [ writer GetFileName ]
   w2if Modified
   writer Write
}

#################################################################################
proc showcamera {} {
   set cam [aRenderer GetActiveCamera]
   puts [$cam GetZoom]
   puts [$cam GetAzimuth]
   puts [$cam GetElevation]
   puts [$cam GetRoll]
}

#################################################################################
proc drawpoints {filename color obj} {
   global aRenderer
   global pointsize
   vtkSphereSource ball-$obj
   ball-$obj SetRadius $pointsize
   ball-$obj SetThetaResolution 12
   ball-$obj SetPhiResolution 12


   vtkPoints points-$obj
   set file [open $filename r]
   while { [gets $file line] != -1 } {
      if {[scan $line "%f %f %f" x y z] } {
         points-$obj InsertNextPoint $x $y $z
      }
   }

   vtkPolyData poly-$obj
   poly-$obj SetPoints points-$obj

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputData poly-$obj
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0

   vtkGlyph3D balls-$obj
   balls-$obj SetInputData poly-$obj
   balls-$obj SetSourceConnection [ball-$obj GetOutputPort]
   vtkPolyDataMapper mapBalls-$obj
   mapBalls-$obj SetInputConnection [balls-$obj GetOutputPort]
   vtkActor ballActor-$obj
   ballActor-$obj SetMapper mapBalls-$obj
   eval [ballActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [ballActor-$obj GetProperty] SetSpecularColor 1 1 1
   [ballActor-$obj GetProperty] SetSpecular 0.3
   [ballActor-$obj GetProperty] SetSpecularPower 20
   [ballActor-$obj GetProperty] SetAmbient 0.2
   [ballActor-$obj GetProperty] SetDiffuse 0.8
   aRenderer AddActor ballActor-$obj
   set cmdstring "ballActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0
}
#################################################################################
proc drawlines {filename color obj} {
   global aRenderer
   global linewidth

   vtkPoints points-$obj
   set file [open $filename r]
   set numberOfOutputPoints 0
   while { [gets $file line] != -1 } {
      if {[scan $line "%f %f %f" x y z] } {
         points-$obj InsertNextPoint $x $y $z
         incr numberOfOutputPoints 1
      }
   }
   vtkCellArray lines-$obj
   lines-$obj InsertNextCell $numberOfOutputPoints
   for {set i 0} {$i < $numberOfOutputPoints} {incr i 1} {
      lines-$obj InsertCellPoint $i
   }

   vtkPolyData poly-$obj
   poly-$obj SetPoints points-$obj
   poly-$obj SetLines lines-$obj
   
   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputData  poly-$obj
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0

   vtkPolyDataMapper mapLines-$obj
   mapLines-$obj SetInputData poly-$obj
   vtkActor lineActor-$obj
   lineActor-$obj SetMapper mapLines-$obj
   eval [lineActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [lineActor-$obj GetProperty] SetSpecularColor 1 1 1
   [lineActor-$obj GetProperty] SetSpecular 0.3
   [lineActor-$obj GetProperty] SetSpecularPower 20
   [lineActor-$obj GetProperty] SetAmbient 0.2
   [lineActor-$obj GetProperty] SetDiffuse 0.8
   [lineActor-$obj GetProperty] SetLineWidth $linewidth
   aRenderer AddActor lineActor-$obj
   set cmdstring "lineActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0
}
#################################################################################
proc drawaxes {} {
   global aRenderer
   global axesOn
   global axessize
   global linewidth
   
   if { $axesOn == 0 } then {
       set obj "Xaxis"
           vtkPoints points-$obj
           points-$obj InsertNextPoint 0 0 0
           points-$obj InsertNextPoint $axessize 0 0
           vtkCellArray lines-$obj
           lines-$obj InsertNextCell 2
           lines-$obj InsertCellPoint 0
           lines-$obj InsertCellPoint 1

           vtkPolyData poly-$obj
           poly-$obj SetPoints points-$obj
           poly-$obj SetLines lines-$obj
           
           vtkOutlineFilter outlineData-$obj
           outlineData-$obj SetInputData  poly-$obj
           vtkPolyDataMapper mapOutline-$obj
           mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
           vtkActor outline-$obj
           outline-$obj SetMapper mapOutline-$obj
           [outline-$obj GetProperty] SetColor 1 0 0
           aRenderer AddActor outline-$obj
           outline-$obj SetVisibility 0

           vtkPolyDataMapper mapLines-$obj
           mapLines-$obj SetInputData poly-$obj
           vtkActor lineActor-$obj
           lineActor-$obj SetMapper mapLines-$obj
           eval [lineActor-$obj GetProperty] SetColor 1 0 0
           [lineActor-$obj GetProperty] SetSpecularColor 1 0 0
           [lineActor-$obj GetProperty] SetSpecular 0.3
           [lineActor-$obj GetProperty] SetSpecularPower 20
           [lineActor-$obj GetProperty] SetAmbient 0.2
           [lineActor-$obj GetProperty] SetDiffuse 0.8
           [lineActor-$obj GetProperty] SetLineWidth $linewidth
           aRenderer AddActor lineActor-$obj
           set cmdstring "lineActor-$obj SetVisibility \$check$obj ; \$renWin Render"
           addobjtolist "Xaxis" $obj $cmdstring outline-$obj 0

       set obj "Yaxis"
           vtkPoints points-$obj
           points-$obj InsertNextPoint 0 0 0
           points-$obj InsertNextPoint 0 $axessize 0
           vtkCellArray lines-$obj
           lines-$obj InsertNextCell 2
           lines-$obj InsertCellPoint 0
           lines-$obj InsertCellPoint 1

           vtkPolyData poly-$obj
           poly-$obj SetPoints points-$obj
           poly-$obj SetLines lines-$obj
           
           vtkOutlineFilter outlineData-$obj
           outlineData-$obj SetInputData  poly-$obj
           vtkPolyDataMapper mapOutline-$obj
           mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
           vtkActor outline-$obj
           outline-$obj SetMapper mapOutline-$obj
           [outline-$obj GetProperty] SetColor 0 1 0
           aRenderer AddActor outline-$obj
           outline-$obj SetVisibility 0

           vtkPolyDataMapper mapLines-$obj
           mapLines-$obj SetInputData poly-$obj
           vtkActor lineActor-$obj
           lineActor-$obj SetMapper mapLines-$obj
           eval [lineActor-$obj GetProperty] SetColor 0 1 0
           [lineActor-$obj GetProperty] SetSpecularColor 0 1 0
           [lineActor-$obj GetProperty] SetSpecular 0.3
           [lineActor-$obj GetProperty] SetSpecularPower 20
           [lineActor-$obj GetProperty] SetAmbient 0.2
           [lineActor-$obj GetProperty] SetDiffuse 0.8
           [lineActor-$obj GetProperty] SetLineWidth $linewidth
           aRenderer AddActor lineActor-$obj
           set cmdstring "lineActor-$obj SetVisibility \$check$obj ; \$renWin Render"
           addobjtolist "Yaxis" $obj $cmdstring outline-$obj 0

       set obj "Zaxis"
           vtkPoints points-$obj
           points-$obj InsertNextPoint 0 0 0
           points-$obj InsertNextPoint 0 0 $axessize
           vtkCellArray lines-$obj
           lines-$obj InsertNextCell 2
           lines-$obj InsertCellPoint 0
           lines-$obj InsertCellPoint 1

           vtkPolyData poly-$obj
           poly-$obj SetPoints points-$obj
           poly-$obj SetLines lines-$obj
           
           vtkOutlineFilter outlineData-$obj
           outlineData-$obj SetInputData  poly-$obj
           vtkPolyDataMapper mapOutline-$obj
           mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
           vtkActor outline-$obj
           outline-$obj SetMapper mapOutline-$obj
           [outline-$obj GetProperty] SetColor 0 0 1
           aRenderer AddActor outline-$obj
           outline-$obj SetVisibility 0

           vtkPolyDataMapper mapLines-$obj
           mapLines-$obj SetInputData poly-$obj
           vtkActor lineActor-$obj
           lineActor-$obj SetMapper mapLines-$obj
           eval [lineActor-$obj GetProperty] SetColor 0 0 1
           [lineActor-$obj GetProperty] SetSpecularColor 0 0 1
           [lineActor-$obj GetProperty] SetSpecular 0.3
           [lineActor-$obj GetProperty] SetSpecularPower 20
           [lineActor-$obj GetProperty] SetAmbient 0.2
           [lineActor-$obj GetProperty] SetDiffuse 0.8
           [lineActor-$obj GetProperty] SetLineWidth $linewidth
           aRenderer AddActor lineActor-$obj
           set cmdstring "lineActor-$obj SetVisibility \$check$obj ; \$renWin Render"
           addobjtolist "Zaxis" $obj $cmdstring outline-$obj 0

      set axesOn 1
   }
}
#################################################################################
proc drawvectors {filename color obj} {
   global aRenderer
   global linewidth

   vtkPoints points-$obj
   set file [open $filename r]
   set numberOfOutputVectors 0
   while { [gets $file line] != -1 } {
      if {[scan $line "%f %f %f %f %f %f" x1 y1 z1 x2 y2 z2] } {
         points-$obj InsertNextPoint $x1 $y1 $z1
         points-$obj InsertNextPoint $x2 $y2 $z2
         incr numberOfOutputVectors 1
      }
   }
   vtkCellArray lines-$obj
   for {set i 0} {$i < $numberOfOutputVectors} {incr i 1} {
   lines-$obj InsertNextCell 2
   lines-$obj InsertCellPoint [expr $i * 2 ]
      lines-$obj InsertCellPoint [expr $i * 2 + 1 ]
   }
   vtkPolyData poly-$obj
   poly-$obj SetPoints points-$obj
   poly-$obj SetLines lines-$obj

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputData  poly-$obj
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0

   vtkPolyDataMapper mapLines-$obj
   mapLines-$obj SetInputData poly-$obj
   vtkActor lineActor-$obj
   lineActor-$obj SetMapper mapLines-$obj
   eval [lineActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [lineActor-$obj GetProperty] SetSpecularColor 1 1 1
   [lineActor-$obj GetProperty] SetSpecular 0.3
   [lineActor-$obj GetProperty] SetSpecularPower 20
   [lineActor-$obj GetProperty] SetAmbient 0.2
   [lineActor-$obj GetProperty] SetDiffuse 0.8
   [lineActor-$obj GetProperty] SetLineWidth $linewidth
   aRenderer AddActor lineActor-$obj
   set cmdstring "lineActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0
}
#################################################################################
proc drawtensors {filename color obj} {
   global aRenderer
   vtkSphereSource ball-$obj
   ball-$obj SetRadius 1
   ball-$obj SetThetaResolution 12
   ball-$obj SetPhiResolution 12


   vtkPolyDataReader vpreader-$obj
   vpreader-$obj SetFileName $filename

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [vpreader-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0

   vtkTensorGlyph tensors-$obj
   tensors-$obj SetInputConnection [vpreader-$obj GetOutputPort]
   tensors-$obj SetSourceConnection [ball-$obj GetOutputPort]
   vtkPolyDataMapper mapBalls-$obj
   mapBalls-$obj SetInputConnection [tensors-$obj GetOutputPort]
   vtkActor ballActor-$obj
   ballActor-$obj SetMapper mapBalls-$obj
   eval [ballActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [ballActor-$obj GetProperty] SetSpecularColor 1 1 1
   [ballActor-$obj GetProperty] SetSpecular 0.3
   [ballActor-$obj GetProperty] SetSpecularPower 20
   [ballActor-$obj GetProperty] SetAmbient 0.2
   [ballActor-$obj GetProperty] SetDiffuse 0.8
   aRenderer AddActor ballActor-$obj
   set cmdstring "ballActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0
}

#################################################################################
proc drawpolydata {filename color obj} {
   global aRenderer
   global renWin
   global opacity

   vtkPolyDataReader vpreader-$obj
   vpreader-$obj SetFileName $filename

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [vpreader-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0


   vtkPolyDataMapper mapPD-$obj
   mapPD-$obj SetInputConnection [vpreader-$obj GetOutputPort]

   vtkActor PDActor-$obj
   PDActor-$obj SetMapper mapPD-$obj
   [PDActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [PDActor-$obj GetProperty] SetEdgeColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [PDActor-$obj GetProperty]  SetOpacity $opacity
   aRenderer AddActor PDActor-$obj

   set cmdstring "PDActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0


}
#################################################################################
proc drawdefgrid {filename color obj} {
   global aRenderer
   global pointsize
   global linewidth
   vtkSphereSource smallball-$obj
   smallball-$obj SetRadius [expr $pointsize / 6 ]
   smallball-$obj SetThetaResolution 12
   smallball-$obj SetPhiResolution 12

   vtkLookupTable colormap-$obj
   colormap-$obj SetNumberOfColors 256
   for { set i 0 } { $i < 256 } { incr i } {
   set v [ expr $i /255.0 ]
      set a [ expr 0.1 +$i/280.0]
      colormap-$obj SetTableValue $i $v [expr 1.0 - $v] 0 $a
   }

   vtkPolyDataReader vpreader-$obj
   vpreader-$obj SetFileName $filename

   vtkOutlineFilter outlineData-$obj
   outlineData-$obj SetInputConnection  [vpreader-$obj GetOutputPort]
   vtkPolyDataMapper mapOutline-$obj
   mapOutline-$obj SetInputConnection [outlineData-$obj GetOutputPort]
   vtkActor outline-$obj
   outline-$obj SetMapper mapOutline-$obj
   [outline-$obj GetProperty] SetColor  [lindex $color 0] [lindex $color 1] [lindex $color 2]
   aRenderer AddActor outline-$obj
   outline-$obj SetVisibility 0

   vtkPolyDataMapper mapPD-$obj
   mapPD-$obj SetInputConnection [vpreader-$obj GetOutputPort]

   #vtkFloatArray dumbarray-$obj
   #dumbarray-$obj SetName "dumb"
   #for {set i 0 } {$i<26} {incr i} {
   #   dumbarray-$obj InsertNextValue $i
   #}
   vpreader-$obj Update
   set range [[vpreader-$obj GetOutput] GetScalarRange]

   mapPD-$obj SetScalarRange [lindex $range 0] [lindex $range 1]
   mapPD-$obj SetLookupTable colormap-$obj
   vtkActor PDActor-$obj
   PDActor-$obj SetMapper mapPD-$obj
   [PDActor-$obj GetProperty] SetColor [lindex $color 0] [lindex $color 1] [lindex $color 2]
   [PDActor-$obj GetProperty] SetPointSize 5
   [PDActor-$obj GetProperty] SetLineWidth $linewidth
   aRenderer AddActor PDActor-$obj

   vtkGlyph3D balls-$obj
   balls-$obj SetInputConnection [vpreader-$obj GetOutputPort]
   balls-$obj SetSourceConnection [smallball-$obj GetOutputPort]
   balls-$obj SetScaleModeToDataScalingOff
   balls-$obj SetColorModeToColorByScalar
   vtkPolyDataMapper mapBalls-$obj
   mapBalls-$obj SetScalarRange 0 11
   mapBalls-$obj SetInputConnection [balls-$obj GetOutputPort]
   vtkActor ballActor-$obj
   ballActor-$obj SetMapper mapBalls-$obj
   mapBalls-$obj SetLookupTable colormap-$obj

   set cmdstring "PDActor-$obj SetVisibility \$check$obj ; \$renWin Render"
   addobjtolist [lindex [file split $filename] end] $obj $cmdstring outline-$obj 0

}
#################################################################################
proc drawtext {filename color obj} {
   global aRenderer
   global textstring
   set file [open $filename r]
   while { [gets $file line] != -1 } {
      set textstring "$textstring\n$line"
   }
}


#################################################################################
proc parseArgs {argc argv} {
   puts "In parse args"
   global action
   global obj
   global aRenderer
   global renWin
   global writer
   global color
   global opacity
   global wireframe
   global seton
   global zoom
   global azimuth
   global fazimuth
   global roll
   global elevation
   global xsize
   global ysize
   global stereo
   global rendertofile
   global renderfile
   global pointsize
   global linewidth
   global axessize
   global isolevel
   global axesOn

   set axesOn 0
   set seton 1
   set pointsize 3
   set linewidth 1
   set axessize 5
   set isolevel 100

   for {set i 0} {$i < $argc} {incr i} {
   set arg [lindex $argv $i]
      puts "processing $arg"
      switch -exact "$arg" {
      help   {
         printhelp
      }
      volume {
         set action "volume"
      }
      planes {
         set action "planes"
      }
      surface {
         set action "surface"
      }
      slice {
         set action "slice"
      }
      outline {
         set action "outline"
      }
      polyoutline {
         set action "polyoutline"
      }
      points {
         set action "points"
      }
      lines {
         set action "lines"
      }
      vectors {
         set action "vectors"
      }
      tensors {
         set action "tensors"
      }
      polydata {
         set action "polydata"
      }
      defgrid {
         set action "defgrid"
      }
      text {
         set action "text"
      }
      stereo      {
         set stereo 1
      }
      on      {
         set seton 1
      }
      off     {
         set seton 0
      }
      axes {
         drawaxes
      }
      pointsize {
         if {$i < $argc} then {
            incr i
            set pointsize  [lindex $argv $i]
         }
      }
      linewidth {
         if {$i < $argc} then {
            incr i
            set linewidth  [lindex $argv $i]
         }
      }
      axessize {
         if {$i < $argc} then {
            incr i
            set axessize  [lindex $argv $i]
         }
      }
      isolevel {
         if {$i < $argc} then {
            incr i
            set isolevel  [lindex $argv $i]
         }
     }
     background {
        aRenderer SetBackground [lindex $color 0] [lindex $color 1] [lindex $color 2]
     }
     color {
        if {$i < $argc} then {
        incr i
        set color [lindex $argv $i]
        }
     }
     red       {		set color "1 0 0"	        }
     blue      {		set color "0 0 1"	        }
     green    {		set color "0 1 0"	        }
     yellow    {		set color "1 1 0"	        }
     magenta {		set color "1 0 1"	        }
     cyan     {		set color "0 1 1"	        }
     purple    {		set color "0.5 0 0.5"	        }
     orange   {		set color "1 0.5 0"	        }
     brown    {		set color "0.6 0.5 0.1"	        }
     black     {		set color "0 0 0"	        }
     white     {		set color "0 0 0"	        }
     gray     {		set color "0.5 0.5 0.5"	        }
     wireframe  {       set wireframe 1      }
     opacity  {
        if {$i < $argc} then {
           incr i
           set opacity [lindex $argv $i]
        }
     }
     zoom {
        if {$i < $argc} then {
           incr i
           set zoom [lindex $argv $i]
        }
     }
     size {
        if {$i < $argc} then {
        incr i
        set xsize [lindex $argv $i]
           incr i
           set ysize [lindex $argv $i]
        }
     }
     azimuth {
        if {$i < $argc} then {
           incr i
           set azimuth [lindex $argv $i]
        }
     }
     fazimuth {
        if {$i < $argc} then {
           incr i
           set fazimuth [lindex $argv $i]
        }
     }
     roll {
        if {$i < $argc} then {
           incr i
           set roll [lindex $argv $i]
        }
     }
     elevation {
        if {$i < $argc} then {
           incr i
           set elevation [lindex $argv $i]
        }
     }
     render {
        if {$i < $argc} then {
           incr i
           set renderfile [lindex $argv $i]
           set rendertofile 1
        }
     }
      default {
        set filelist [glob -nocomplain "$arg"]
        set sz [llength $filelist]
        if { $sz == 0 } then {
           puts "WARNING: $arg expanded to nothing"
        }
        for {set f 0} {$f < $sz} {incr f} {
            set farg [lindex $filelist $f]
               if { [file exists $farg ] } {
                  switch -exact "$action" {
                  volume {
                     drawvolume $farg "$color" $obj
                     incr obj
                  }
                  planes {
                     drawplanes $farg "$color" $obj
                     incr obj
                  }
                  surface {
                     drawsurface $farg "$color" $obj
                     incr obj
                  }
                  slice {
                     drawslice $farg "$color" $obj
                     incr obj
                  }
                  outline {
                     drawoutline $farg "$color" $obj
                     incr obj
                  }
                  polyoutline {
                     drawpolyoutline $farg "$color" $obj
                     incr obj
                  }
                  points {
                     drawpoints $farg "$color" $obj
                     incr obj
                  }
                  lines {
                     drawlines $farg "$color" $obj
                     incr obj
                  }
                  vectors {
                     drawvectors $farg "$color" $obj
                     incr obj
                  }
                  tensors {
                     drawtensors $farg "$color" $obj
                     incr obj
                  }
                  polydata {
                     drawpolydata $farg "$color" $obj
                     incr obj
                  }
                  defgrid {
                     drawdefgrid $farg "$color" $obj
                     incr obj
                  }
                  text {
                     drawtext $farg "$color" $obj
                     incr obj
                  }
               default {
                     puts "Unknown action $action"
                     exit
                  }
               }
            } else {
               puts "WARNING: cant read $farg, ignoring"
            }
         }

      }
   }
}
}





#################################################################################
#                        MAIN DRAW3D
vtkLookupTable graymap
graymap SetNumberOfColors 256
graymap SetTableRange 0 255
for { set i 0 } { $i < 256 } { incr i } {
set v [ expr $i /255.0 ]
   set a [ expr $i > 0 ]
   graymap SetTableValue $i $v $v $v $a
}
vtkLookupTable graymapsolid
graymapsolid SetNumberOfColors 256
graymapsolid SetTableRange 0 255
for { set i 0 } { $i < 256 } { incr i } {
set v [ expr $i /255.0 ]
   graymapsolid SetTableValue $i $v $v $v 1
}

vtkFileOutputWindow fow
fow SetFileName "draw3dErrors.txt"
fow SetFlush 0
fow SetInstance fow


global lastObject

toplevel .top
wm withdraw .top
wm protocol .top WM_DELETE_WINDOW "exit"
wm title .top "draw3d"
frame .top.f1


menu .top.menubar -type menubar
.top configure -menu .top.menubar
.top.menubar add cascade -label "File" -menu .top.menubar.filemenu -underline 0
menu .top.menubar.filemenu

.top.menubar add cascade -label "Edit" -menu .top.menubar.editmenu -underline 0
menu .top.menubar.editmenu
.top.menubar add cascade -label "View" -menu .top.menubar.viewmenu -underline 0
menu .top.menubar.viewmenu
.top.menubar add cascade -label "Help" -menu .top.menubar.helpmenu -underline 0
menu .top.menubar.helpmenu

.top.menubar.filemenu add command -label "Exit" -command {exit} -underline 1

.top.menubar.editmenu add command -label "Render to file" -command {writebutton} -underline 0

.top.menubar.viewmenu add command -label "Show Camera Parameters" -command {showcamera} -underline 6

.top.menubar.helpmenu add command -label "About" -command {aboutBox} -underline 0
frame .top.status
label .top.status.status -relief sunken -borderwidth 1 -text "Initializing..."
pack .top.status.status -side left -expand true -fill x
pack .top.status -side bottom -fill x


vtkTkRenderWidget .top.f1.r1 -width 400 -height 400

frame .top.f3
canvas .top.f3.canvas -width 70 -height 10 -yscrollcommand [list .top.f3.yscroll set] -xscrollcommand [list .top.f3.xscroll set]
scrollbar .top.f3.yscroll -orient vertical -command [list .top.f3.canvas yview]
scrollbar .top.f3.xscroll -orient horizontal -command [list .top.f3.canvas xview]
pack .top.f3.yscroll -side right -fill y
pack .top.f3.xscroll -side bottom -fill x
pack .top.f3.canvas -side left -fill both -expand true

pack .top.f3 -side left -fill y

set f [frame .top.f3.canvas.f ]
.top.f3.canvas create window 0 0 -anchor nw -window $f -tags scrolledcanvas


pack .top.f1.r1 -side right -padx 3 -pady 3 -fill both -expand t
pack .top.f1 -fill both -expand t
toplevel .dlg -class Dialog
wm title .dlg "Output image"
wm withdraw .dlg
label .dlg.l -text "Enter the (bmp) file to write"
entry .dlg.t
button .dlg.b -text "Write" -command "writer SetFileName \[.dlg.t get\]; wm withdraw .dlg;write"
pack .dlg.l
pack .dlg.t
pack .dlg.b



# Get the render window associated with the widget
set renWin [.top.f1.r1 GetRenderWindow]
vtkRenderer aRenderer
aRenderer SetBackground 1 1 1
$renWin AddRenderer aRenderer

vtkRenderWindowInteractor iren
vtkInteractorStyleTrackballCamera style
iren SetRenderWindow $renWin
iren SetInteractorStyle style

vtkWindowToImageFilter w2if
w2if SetInput $renWin
vtkBMPWriter writer
writer SetInputConnection [w2if GetOutputPort]



# bind the mouse events
::vtk::bind_tk_render_widget .top.f1.r1

# Initialize state
set action outline
set color {0 0 1}

set obj 0
global opacity
set opacity 1.0
global wireframe
set wireframe 0
global zoom
set zoom 1
global azimuth
set azimuth 45
global fazimuth
set fazimuth 0
global elevation
set elevation 45
global roll
set roll 0
global xsize
set xsize 0
global ysize
set ysize 0
global stereo
set stereo 0
global renderfile
global rendertofile
set rendertofile 0

set textstring ""
global textstring

parseArgs $argc "$argv"

vtkTextActor textActor
textActor SetTextScaleModeToProp
textActor SetDisplayPosition 90 50
textActor SetInput "$textstring"

[textActor GetPosition2Coordinate] SetCoordinateSystemToNormalizedViewport
[textActor GetPosition2Coordinate] SetValue 0.2 0.2

set tprop [textActor GetTextProperty]
$tprop SetFontSize 20
$tprop SetFontFamilyToArial
$tprop SetJustificationToCentered
#$tprop BoldOn
#$tprop ItalicOn
#$tprop ShadowOn
$tprop SetColor 0.5 0 0

aRenderer AddActor textActor




if {$xsize > 0} {
   $renWin SetSize $xsize $ysize
}

$renWin SetStereoTypeToRedBlue
if {$stereo > 0} {
  $renWin StereoRenderOn
}
aRenderer ResetCameraClippingRange
aRenderer ResetCamera
[aRenderer GetActiveCamera] Zoom $zoom
[aRenderer GetActiveCamera] Azimuth $azimuth
[aRenderer GetActiveCamera] Elevation $elevation
[aRenderer GetActiveCamera] OrthogonalizeViewUp
[aRenderer GetActiveCamera] Azimuth $fazimuth
[aRenderer GetActiveCamera] Roll $roll

if { ![info exists lastObject] } {
   puts "No datafiles input"
   printhelp
   exit
}


if {$rendertofile>0} {
wm withdraw .
$renWin OffScreenRenderingOn
writer SetFileName $renderfile
write
exit
} else {
wm deiconify .top
tkwait visibility $lastObject
set width [winfo reqwidth $f]
   set height [winfo reqheight $f]
   puts "Width $width Height $height"
   .top.f3.canvas config -width $width -height $height
   .top.f3.canvas config -scrollregion "0 0 $width $height"
   .top.f3.canvas config -yscrollincrement 10
   .top.f3.canvas itemconfigure scrolledcanvas -width $width -height $height
   wm withdraw .
}

