#ifndef FIBERORIENTATION_H
#define FIBERORIENTATION_H

/*
 *	Author: Ravikiran J
 *	Email : ravikirn@cs.unc.edu
 */
#include <iostream>
#include <fstream>
#include <algorithm>
#include <math.h>
#include <string.h>

#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkMaskImageFilter.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionConstIterator.h"
#include "itkImageRegionIterator.h"

#include <DisplayFunc.h>
#include <VolumeCounter.h>
#include <FiberSegmentExtractor.h>
#include <vtkPolyDataReader.h>
#include <Counter.h>

#define DEBUG false
#define COUNTDISP false 

//Fiber Orientation Image meta info struct 
struct FOMeta{
    double origin[3];
    double spacing[3];
    unsigned int size[3];
};

//HistoPlot of voxel fiber distances
class histoPlot{
    public:
        unsigned int x;
        unsigned int y;
        unsigned int z;

        std::string type;
        std::string filename;

    public:
        void setParams(unsigned int x1, unsigned int y1, unsigned int z1, std::string t, std::string path){
            x = x1;
            y = y1;
            z = z1;
            type = t;
            std::stringstream s;
            s<<path<<t<<"_("<<x<<","<<y<<","<<z<<")_voxel_hist.txt";
            filename = s.str();
        }
};


// FA Image
const unsigned int FADimension = 3;
typedef double FAPixelType;
typedef itk::Image< FAPixelType, FADimension > FAImageType;
typedef itk::ImageFileReader<FAImageType> FAReaderType;

class FiberOrientation{
	public:
		explicit FiberOrientation(std::string fiberFile, std::string dwiFile, std::string faFile, std::string maskFile);
		//explicit FiberOrientation(std::string dwiFile, std::string faFile, std::string maskFile);

        int computeFiberSegmentsPerVoxelImage(std::string fsImgFileName);

        int writeFiberOrientationsPerVoxelTextFile(std::string foImgFileName);

        void computeMaxFiberSegments(std::string fsImgFileName);

        int computeColorEntropyImage(std::string foImgFileName, std::string entropyImgFileName, std::string cEntropyImgFileName);

        int computeVolumeCounter(std::string vCounterImgFileName);

        int readFOMeta(std::string foImgFileName, FOMeta *m);

        int writeFiberDistancesPerVoxelTextFile(std::string volumeCounterFileName, std::string foImgFileName, std::string dImgFileName);

        int writeFiberDistanceHistogram(std::string volumeCounterFileName, std::string foImgFileName, histoPlot h[], int size);

        int computePercentileFiberDistanceImage(std::string dImgFileName, std::string path, double percentile);

        int computeEarthMoversDistanceImage(std::string dImgFileName, std::string enImgFileName, std::string emdImgFileName, std::string path);

        int writeEMDTextToImageFile(std::string inputFileName, std::string outputFileName, int max);

        std::vector <double> convertCartesianToPhiTheta(double x, double y, double z);

        double getSphericalDistance(std::vector <double>, std::vector <double>);
        
        template <class ImagePointer>
        int getFOImageHandler(ImagePointer &foImage, std::string foImageFileName);

        template <class ImagePointer>
        int updatePixelValue(ImagePointer &foImage, PositionInVolume p, geometry::Vector direction);

		virtual ~FiberOrientation(){}
		
	protected:
        //Filenames
        std::string fiberImgFileName;
        std::string dwiImgFileName;
        std::string faImgFileName;
        std::string maskImgFileName;
        std::string fiberSegmentsPerVoxelImgFileName;

        //FA Image
        FAReaderType::Pointer faReader;
        FAImageType::Pointer faImage;
        FAImageType::PointType faOrigin;
        FAImageType::SpacingType faSpacing;
        FAImageType::SizeType faSize;

        //Mask Image
        FAReaderType::Pointer maskReader;
        FAImageType::Pointer maskImage;

        //Max segments in fiber file
        long int maxFiberSegments;

        //Subdivision Level
        int subdivisionLevel;
} ;

FiberOrientation::FiberOrientation(std::string fiberFile, std::string dwiFile, std::string faFile, std::string maskFile){
//FiberOrientation::FiberOrientation(std::string dwiFile, std::string faFile, std::string maskFile){
    //Assign default values for variables
    maxFiberSegments = 0;
    //Assign the subdivisionLevel
    subdivisionLevel = 6;

    //Assign filenames for future use
    fiberImgFileName = fiberFile;
    dwiImgFileName = dwiFile;
    faImgFileName = faFile;
    maskImgFileName = maskFile;

    faReader = ReaderType::New();
    faReader->SetFileName(faImgFileName.c_str());
    faReader->Update();
    faImage = faReader->GetOutput();
    
    maskReader = ReaderType::New();
    maskReader->SetFileName(maskImgFileName.c_str());
    maskReader->Update();
    maskImage = maskReader->GetOutput();

    faOrigin = faImage->GetOrigin();
    faSpacing = faImage->GetSpacing();
    faSize = faImage->GetLargestPossibleRegion().GetSize();
}

int FiberOrientation::computeFiberSegmentsPerVoxelImage(std::string fsImgFileName){
    /* Pass the vtk polydata fiber file which is generated from ukf tractography */
    output("Reading fiber file started");
    FiberSegmentExtractor fse(fiberImgFileName.c_str()) ;
    output("Reading fiber file finished");
    // Pass the original DWI file 
    output("Reading DWI file started");
    VolumeCounter<Counter_WeightedVertices> vcc(6, dwiImgFileName.c_str()) ;
    //VolumeCounter<Counter_WeightedVertices> vcc(6, argv[2]) ;
    output("Reading DWI file finished");
    // Compute the total number of fibers 
    int totalfibers = (fse.GetNumberOfFibers());
    output2("Number of fibers = ", totalfibers);
    // Compute the total number of segments 
    int totalSegments = fse.GetNumberOfSegments();
    output2("Number of segments = ", totalSegments);

    long int segmentCount = 1, zeroVectorCount = 0, wildCount = 0;
    FiberSegment segment;
    PositionInVolume p1,p2;

    const unsigned int Dimension = 3;
    typedef long int PixelType;
    typedef itk::Image< PixelType, Dimension > ImageType;
    typedef itk::ImageFileReader<ImageType> ReaderType;
    typedef itk::ImageFileWriter< ImageType > WriterType;

    ImageType::PointType fsOrigin;
    ImageType::SpacingType fsSpacing;
    ImageType::SizeType fsSize;

	for (int i = 0; i < VOLUME_DIMENSION; i++){
		fsOrigin[i]  = faOrigin[i];
        fsSpacing[i] = faSpacing[i];
        fsSize[i] = faSize[i];
    }

    ImageType::Pointer fsImage = ImageType::New();
    ImageType::RegionType fsRegion;
    fsRegion.SetSize(fsSize);
    fsImage->SetSpacing(fsSpacing);
    fsImage->SetOrigin(fsOrigin);
    fsImage->SetRegions(fsRegion);
    fsImage->Allocate();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType pixelValue;
    if(COUNTDISP){
        cout<<"No. of Fiber Segments read = ";
    }
    while(fse){
        segment = fse.Next();
        PositionInVolume p1 = vcc.GetVoxelPositionInVolume(segment.p1) ;
        PositionInVolume p2 = vcc.GetVoxelPositionInVolume(segment.p2) ;

        if (!p1.isValid() || !p2.isValid()){
            wildCount++ ;
            continue ;
        }
        //Print debug
        if(DEBUG){
            displayPoint(p1, "p1 = ");
            displayPoint(p2, "p2 = ");
            displayPoint(segment.p1, "segment.p1 = ");
            displayPoint(segment.p2, "segment.p2 = ");
        }
        geometry::Vector direction = (segment.p1 - segment.p2);
        if(direction.isZero()){
            zeroVectorCount++;
        }else{
            direction.normalize();
            if(DEBUG){
                displayVect(direction, "Direction Vector = ");
            }
            //Retrieve the pixel value, increment by 1 and write back
            for(int i = 0; i < VOLUME_DIMENSION; i++){
                pixelIndex[i] = (int)p1[i];
            }
            pixelValue = fsImage->GetPixel(pixelIndex);
            pixelValue += 1;
            fsImage->SetPixel(pixelIndex, pixelValue);

            if(!(p1 == p2)){
                //Retrieve the pixel value, increment by 1 and write back
                for(int i = 0; i < VOLUME_DIMENSION; i++){
                    pixelIndex[i] = (int)p2[i];
                }
                pixelValue = fsImage->GetPixel(pixelIndex);
                pixelValue += 1;
                fsImage->SetPixel(pixelIndex, pixelValue);
            }
        }
        segmentCount++;
        if(COUNTDISP){
            display(segmentCount);
        }
    }
    //Compute Max Fiber Segments before writing out
    for(unsigned int i = 0; i < fsSize[0]; i++){
        for(unsigned int j = 0; j < fsSize[1]; j++){
            for(unsigned int k = 0; k < fsSize[2]; k++){
                pixelIndex[0] = i, pixelIndex[1] = j; pixelIndex[2] = k;
                pixelValue = fsImage->GetPixel(pixelIndex);
                if(maxFiberSegments < pixelValue){
                    maxFiberSegments = pixelValue;
                }
            }
        }
    }

    output2("Max Fiber Segments = ", maxFiberSegments);

    output("Writing out the Fiber Segments Per Voxel Image");
    WriterType::Pointer fsWriter = WriterType::New();
    //Only MHD and VTK formats support writing multiple components per pixel
    fsWriter->SetFileName(fsImgFileName.c_str());
    fsWriter->SetInput(fsImage);
    try{
        fsWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

void FiberOrientation::computeMaxFiberSegments(std::string fsImgFileName){
    //Check if file already exists
    ifstream ifile(fsImgFileName.c_str());
    if(ifile){
        output("Fiber Segments Per Voxel Image exists, computing Max Fiber Segments");
        const unsigned int Dimension = 3;
        typedef long int PixelType;
        typedef itk::Image< PixelType, Dimension > ImageType;
        typedef itk::ImageFileReader<ImageType> ReaderType;
        typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;

        ReaderType::Pointer fsReader;
        ImageType::Pointer fsImage;
        fsReader = ReaderType::New();
        fsReader->SetFileName(fsImgFileName.c_str());
        fsReader->Update();
        fsImage = fsReader->GetOutput();

        ConstIteratorType fs_it( fsImage, fsImage->GetLargestPossibleRegion());
        fs_it.GoToBegin();

        ImageType::IndexType pixelIndex;
        ImageType::PixelType pixelValue;

        while(!fs_it.IsAtEnd()){
            pixelIndex = fs_it.GetIndex();
            pixelValue = fs_it.Get();
            if(maxFiberSegments < pixelValue){
                maxFiberSegments = pixelValue;
            }
            ++fs_it; 
        }
        output2("Max Fiber Segments = ", maxFiberSegments);
    }else{
        computeFiberSegmentsPerVoxelImage(fsImgFileName);
    }
}

int FiberOrientation::writeFiberOrientationsPerVoxelTextFile(std::string foImgFileName){
    const unsigned int Dimension = 3;
    typedef std::vector< double > PixelType;
    typedef itk::Image< PixelType, Dimension > ImageType;
    typedef itk::ImageFileWriter< ImageType > WriterType;
    typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;

    ImageType::PointType foOrigin;
    ImageType::SpacingType foSpacing;
    ImageType::SizeType foSize;

	for (int i = 0; i < VOLUME_DIMENSION; i++){
		foOrigin[i]  = faOrigin[i];
        foSpacing[i] = faSpacing[i];
        foSize[i] = faSize[i];
    }

    ImageType::Pointer foImage = ImageType::New();
    ImageType::RegionType foRegion;
    foRegion.SetSize(foSize);
    foImage->SetSpacing(foSpacing);
    foImage->SetOrigin(foOrigin);
    foImage->SetRegions(foRegion);
    foImage->Allocate();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType pixelValue;

    // Pass the vtk polydata fiber file which is generated from ukf tractography 
    output("Reading fiber file started");
    FiberSegmentExtractor fse(fiberImgFileName.c_str()) ;
    output("Reading fiber file finished");
    // Pass the original DWI file 
    output("Reading DWI file started");
    VolumeCounter<Counter_WeightedVertices> vcc(6, dwiImgFileName.c_str()) ;
    output("Reading DWI file finished");
    // Compute the total number of fibers 
    int totalfibers = (fse.GetNumberOfFibers());
    output2("Number of fibers = ", totalfibers);
    // Compute the total number of segments 
    int totalSegments = fse.GetNumberOfSegments();
    output2("Number of segments = ", totalSegments);

    output("Computing Fiber Orientations per voxel");
    if(COUNTDISP){
        cout<<"No. of Fiber Segments read = ";
    }
    long int segmentCount = 1, wildCount = 0;
    bool fiberStart = true, fiberEnd = false;
    FiberSegment currSegment, prevSegment, nextSegment;
    PositionInVolume point1, point2;

    while(fse){
        currSegment = fse.Next(fiberEnd);
        point1 = vcc.GetVoxelPositionInVolume(currSegment.p1);
        point2 = vcc.GetVoxelPositionInVolume(currSegment.p2);

        if (!point1.isValid() || !point2.isValid()){
            wildCount++ ;
            continue ;
        }
        //Print debug
        if(DEBUG){
            displayPoint(point1, "p1 = ");
            displayPoint(point2, "p2 = ");
            displayFiberSegment(currSegment);
        }
        
        //Handle special case of fiberStart
        if(fiberStart){
            geometry::Vector direction = (currSegment.p2 - currSegment.p1);
            updatePixelValue(foImage, point1, direction);
            fiberStart = false;
            prevSegment = currSegment;
            segmentCount++;
            continue;
        }

        // Normal case of computing tangential vector at current fiber point using neighbor points
        geometry::Vector direction = (currSegment.p2 - prevSegment.p1);
        updatePixelValue(foImage, point1, direction);
        prevSegment = currSegment;
        
        //Handle special case of fiberEnd
        if(fiberEnd){
            geometry::Vector direction = (currSegment.p2 - currSegment.p1);
            updatePixelValue(foImage, point2, direction);
            fiberStart = true;
            fiberEnd = false;
        }
        segmentCount++ ;
        if(COUNTDISP){
            display(segmentCount);
        }
    }

    if(COUNTDISP){
        cout<<endl;
    }
    // Format of Fiber Orientation File
    // ================================
    // begin header
    // dimension=3
    // type=double
    // size=maxx maxy maxz
    // origin=ox oy oz
    // spacing=sx sy sz
    // end header
    //
    // begin data
    // voxel=0 0 0
    // size=3
    // 0.1
    // 0.8
    // 0.1
    // voxel=1 0 0
    // ...
    // ...
    // end data
    
    // Write the image to a text file
    output("Writing the image to text file...please be patient");
 	std::ofstream fo(foImgFileName.c_str());
    fo<<"begin header"<<endl;
    fo<<"dimension="<<VOLUME_DIMENSION<<endl;
    fo<<"type=double"<<endl;
    fo<<"size="<<foSize[0]<<" "<<foSize[1]<<" "<<foSize[2]<<endl;
    fo<<"origin="<<foOrigin[0]<<" "<<foOrigin[1]<<" "<<foOrigin[2]<<endl;
    fo<<"spacing="<<foSpacing[0]<<" "<<foSpacing[1]<<" "<<foSpacing[2]<<endl;
    fo<<"end header"<<endl;
    fo<<endl;
    fo<<"begin data"<<endl;

    long int size = 0;
    ConstIteratorType fo_it( foImage, foImage->GetLargestPossibleRegion());
    fo_it.GoToBegin();
    while(!fo_it.IsAtEnd()){
        pixelIndex = fo_it.GetIndex();
        pixelValue = fo_it.Get();
        size = pixelValue.size();
        fo<<"voxel="<<pixelIndex[0]<<" "<<pixelIndex[1]<<" "<<pixelIndex[2]<<endl;
        fo<<"size="<<size<<endl;
        for(unsigned int i = 0; i < size; i++){
            fo<<pixelValue[i]<<endl;
        }
        ++fo_it;
    }
    fo<<"end data"<<endl;
    return 0;
}

int FiberOrientation::computeColorEntropyImage(std::string foImgFileName, std::string entropyImgFileName, std::string cEntropyImgFileName){
    //Read FOImage
    output("Reading Fiber Orientations per Voxel Image");
    const unsigned int FODimension = 3;
    typedef std::vector< double > FOPixelType;
    typedef itk::Image< FOPixelType, FODimension > FOImageType;
    typedef itk::ImageRegionConstIterator< FOImageType > FOConstIteratorType;

    FOImageType::Pointer foImage = FOImageType::New();
    getFOImageHandler(foImage, foImgFileName); 
    FOConstIteratorType fo_it( foImage, foImage->GetLargestPossibleRegion());


    //Read Entropy Image
    output("Reading Entropy Image");
    const unsigned int enDimension = 3;
    typedef double enPixelType;
    typedef itk::Image< enPixelType, enDimension > enImageType;
    typedef itk::ImageFileReader<enImageType> enReaderType;
    typedef itk::ImageRegionConstIterator< enImageType > enConstIteratorType;

    enReaderType::Pointer enReader;
    enImageType::Pointer enImage;

    enReader = enReaderType::New();
    enReader->SetFileName(entropyImgFileName.c_str());
    enReader->Update();
    enImage = enReader->GetOutput();
    enConstIteratorType en_it( enImage, enImage->GetLargestPossibleRegion());

    //Read Mask Image
    output("Reading Mask Image");
    const unsigned int mDimension = 3;
    typedef double mPixelType;
    typedef itk::Image< mPixelType, mDimension > mImageType;
    typedef itk::ImageFileReader<mImageType> mReaderType;
    typedef itk::ImageRegionConstIterator< mImageType > mConstIteratorType;

    mReaderType::Pointer maskReader;
    mImageType::Pointer maskImage;

    maskReader = mReaderType::New();
    maskReader->SetFileName(maskImgFileName.c_str());
    maskReader->Update();
    maskImage = maskReader->GetOutput();
    mConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    //Initialize Color Entropy Image
    typedef itk::RGBPixel< double > RGBPixelType;
    typedef itk::Image< RGBPixelType, 3 >   RGBImageType;
    typedef itk::ImageFileWriter< RGBImageType > RGBWriterType;
    RGBImageType::Pointer cEntropy = RGBImageType::New();
    RGBImageType::RegionType cEntropyRegion;
    cEntropyRegion.SetSize(faSize);
    cEntropy->SetSpacing(faSpacing);
    cEntropy->SetOrigin(faOrigin);
    cEntropy->SetRegions(cEntropyRegion);
    cEntropy->Allocate();

    //enImageType::IndexType enIndex;
    enImageType::PixelType enValue;

    mImageType::IndexType maskIndex;
    mImageType::PixelType maskValue;

    //FOImageType::IndexType foIndex;
    FOImageType::PixelType foValue;

    RGBImageType::IndexType cEntropyIndex;
    RGBImageType::PixelType cEntropyValue;
 
    en_it.GoToBegin();
    mask_it.GoToBegin();
    fo_it.GoToBegin();

    double temp[3];
    unsigned int count;

    double maxVal[] = {0, 0, 0};
    double minVal[] = {1000, 1000, 1000};

    output("Computing Color Entropy per voxel");
    while(!en_it.IsAtEnd() && !mask_it.IsAtEnd() && !fo_it.IsAtEnd()){
        maskIndex = mask_it.GetIndex();
        cEntropyIndex[0] = maskIndex[0] , cEntropyIndex[1] = maskIndex[1], cEntropyIndex[2] = maskIndex[2];
        maskValue = mask_it.Get();
        enValue = en_it.Get();
        if(maskValue > 0 && enValue != 0){
            foValue = fo_it.Get();
            temp[0] = 0.0, temp[1] = 0.0, temp[2] = 0.0;
            count = foValue.size();
            if(count > 0){
                for(unsigned int i = 0; i < count; i += 3){
                    if(foValue[i] != 0 && foValue[i+1] != 0 && foValue[i+2] != 0){
                        temp[0] += log(fabs(foValue[i]));
                        temp[1] += log(fabs(foValue[i+1]));
                        temp[2] += log(fabs(foValue[i+2]));
                    }
                }
                cEntropyValue[0] = exp(temp[0]/(count/3)) * enValue;
                cEntropyValue[1] = exp(temp[1]/(count/3)) * enValue;
                cEntropyValue[2] = exp(temp[2]/(count/3)) * enValue;

                if(cEntropyValue[0] < minVal[0]){
                    minVal[0] = cEntropyValue[0];
                }
                if(cEntropyValue[0] > maxVal[0]){
                    maxVal[0] = cEntropyValue[0];
                }
                if(cEntropyValue[1] < minVal[1]){
                    minVal[1] = cEntropyValue[1];
                }
                if(cEntropyValue[1] > maxVal[1]){
                    maxVal[1] = cEntropyValue[1];
                }
                if(cEntropyValue[2] < minVal[2]){
                    minVal[2] = cEntropyValue[2];
                }
                if(cEntropyValue[2] > maxVal[2]){
                    maxVal[2] = cEntropyValue[2];
                }
            }else{
                cEntropyValue[0] = 0.0;
                cEntropyValue[1] = 0.0;
                cEntropyValue[2] = 0.0;
            }
        }else{
            cEntropyValue[0] = 0.0;
            cEntropyValue[1] = 0.0;
            cEntropyValue[2] = 0.0;
        }
        cEntropy->SetPixel(cEntropyIndex, cEntropyValue);
        ++en_it;
        ++mask_it;
        ++fo_it;
    }

    output("Scaling the RGB values to 0..255");
    mask_it.GoToBegin();    
    en_it.GoToBegin();

    while(!mask_it.IsAtEnd() && !en_it.IsAtEnd()){
        maskIndex = mask_it.GetIndex(); 
        cEntropyIndex[0] = maskIndex[0] , cEntropyIndex[1] = maskIndex[1], cEntropyIndex[2] = maskIndex[2]; 
        maskValue = mask_it.Get(); 
        enValue = en_it.Get();

        if(maskValue > 0 && enValue != 0.0){
            cEntropyValue = cEntropy->GetPixel(cEntropyIndex);
            cEntropyValue[0] = ((cEntropyValue[0] - minVal[0]) / (maxVal[0] - minVal[0])) * 255;
            cEntropyValue[1] = ((cEntropyValue[1] - minVal[1]) / (maxVal[1] - minVal[1])) * 255;
            cEntropyValue[2] = ((cEntropyValue[2] - minVal[2]) / (maxVal[2] - minVal[2])) * 255;
            cEntropy->SetPixel(cEntropyIndex, cEntropyValue);
        }
        ++mask_it;
        ++en_it;
    } 
    output("Writing out the Color Entropy Image");
    RGBWriterType::Pointer cEntropyWriter = RGBWriterType::New();
    cEntropyWriter->SetFileName(cEntropyImgFileName.c_str());
    cEntropyWriter->SetInput(cEntropy);
    try{
        cEntropyWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

template <class ImagePointer>
int FiberOrientation::getFOImageHandler(ImagePointer &foImage, std::string foImgFileName){
    //Read the Fiber Orientation Image
    const unsigned int FODimension = 3;
    typedef std::vector< double > FOPixelType;
    typedef itk::Image< FOPixelType, FODimension > FOImageType;

    typedef itk::ImageFileWriter< FOImageType > FOWriterType;
    typedef itk::ImageRegionConstIterator< FOImageType > FOConstIteratorType;

    FOImageType::PointType foOrigin;
    FOImageType::SpacingType foSpacing;
    FOImageType::SizeType foSize;

    FOMeta *meta = new FOMeta();
    readFOMeta(foImgFileName, meta);

    for(int i = 0; i < VOLUME_DIMENSION; i++){
        foOrigin[i] = meta->origin[i];
        foSpacing[i] = meta->spacing[i];
        foSize[i] = meta->size[i];
    }
    
   // FOImageType::Pointer foImage = FOImageType::New();
    FOImageType::RegionType foRegion;
    foRegion.SetSize(foSize);
    foImage->SetSpacing(foSpacing);
    foImage->SetOrigin(foOrigin);
    foImage->SetRegions(foRegion);
    foImage->Allocate();

    FOImageType::IndexType pixelIndex;
    FOImageType::PixelType pixelValue;

    ifstream inFile(foImgFileName.c_str());
    std::string line;
    bool headerFinish = false;
    bool keyFinish = false;
    unsigned int i, count;
    int j, k;
    char value[100];

    while(getline(inFile, line)){
        if(line.find("end header") != std::string::npos){
            headerFinish = true; 
        }
        if(!headerFinish){
            continue;
        }
        
        //Process voxels
        if(line.find("voxel") != std::string::npos){
            keyFinish = false;
            j = 0;
            k = 0;
            for(i = 0; i < line.length(); i++){
               if(line[i] == '='){
                   keyFinish = true;
                   continue;
               }
               if(!keyFinish){
                   continue;
               }
               if(line[i] == ' '){
                   value[j] = '\0';
                   j = 0;
                   pixelIndex[k++] = (unsigned int)atoi(value);
               }else{
                   value[j++] = line[i];
               }
            }
            value[j] = '\0';
            pixelIndex[k] = (unsigned int)atoi(value);

            pixelValue = foImage->GetPixel(pixelIndex);

            //Compute size 
            getline(inFile, line);
            if(line.find("size") != std::string::npos){
                keyFinish = false;
                j = 0;
                for(i = 0; i < line.length(); i++){
                   if(line[i] == '='){
                       keyFinish = true;
                       continue;
                   }
                   if(!keyFinish){
                       continue;
                   }
                   value[j++] = line[i];
                }
                value[j] = '\0';
                count = (unsigned int)atoi(value);
            }
            while(count > 0){
                getline(inFile, line);
                pixelValue.push_back(strtod(line.c_str(), NULL));
                count--;
            }
            foImage->SetPixel(pixelIndex, pixelValue);
        }
    }
    return 0;
}

int FiberOrientation::readFOMeta(std::string foImgFileName, FOMeta* m){
    ifstream inFile(foImgFileName.c_str());
    std::string line;
    bool keyFinish = false;
    unsigned int i;
    int j, k;
    char value[100];

    while(getline(inFile, line)){
        if(line.find("size") != std::string::npos){
            keyFinish = false; 
            j = 0;
            k = 0;
            for(i = 0; i < line.length(); i++){
               if(line[i] == '='){
                   keyFinish = true;
                   continue;
               }
               if(!keyFinish){
                   continue;
               }
               if(line[i] == ' '){
                   value[j] = '\0';
                   j = 0;
                   m->size[k++] = (unsigned int)atoi(value);
               }else{
                   value[j++] = line[i];
               }
            }
            value[j] = '\0';
            m->size[k] = (unsigned int)atoi(value);
        }

        if(line.find("origin") != std::string::npos){
            keyFinish = false; 
            j = 0;
            k = 0;
            for(i = 0; i<line.length(); i++){
               if(line[i] == '='){
                   keyFinish = true;
                   continue;
               }
               if(!keyFinish){
                   continue;
               }
               if(line[i] == ' '){
                   value[j] = '\0';
                   j = 0;
                   m->origin[k++] = strtod(value, NULL);
               }else{
                   value[j++] = line[i];
               }
            }
            value[j] = '\0';
            m->origin[k] = strtod(value, NULL);
        }
        if(line.find("spacing") != std::string::npos){
            keyFinish = false; 
            j = 0;
            k = 0;
            for(i = 0; i < line.length(); i++){
               if(line[i] == '='){
                   keyFinish = true;
                   continue;
               }
               if(!keyFinish){
                   continue;
               }
               if(line[i] == ' '){
                   value[j] = '\0';
                   j = 0;
                   m->spacing[k++] = strtod(value, NULL);
               }else{
                   value[j++] = line[i];
               }
            }
            value[j] = '\0';
            m->spacing[k] = strtod(value, NULL);
        }
        if(line.find("end header") != std::string::npos){
            break;
        }
    }
    return 0;
}

template <class ImagePointer>
int FiberOrientation::updatePixelValue(ImagePointer &foImage, PositionInVolume p, geometry::Vector direction){
    const unsigned int Dimension = 3;
    typedef std::vector< double > PixelType;
    typedef itk::Image< PixelType, Dimension > ImageType;

    ImageType::IndexType pixelIndex;
    ImageType::PixelType pixelValue;

    if(!direction.isZero()){
        direction.normalize();
        if(DEBUG){
            displayVect(direction, "Direction Vector = ");
        }
        //Compute the pixel position, retrieve the vector, update it and set the pixel value again
        for(int i = 0; i < VOLUME_DIMENSION; i++){
            pixelIndex[i] = (int)p[i];
        }
        pixelValue = foImage->GetPixel(pixelIndex);
        for(int i = 0; i < VOLUME_DIMENSION; i++){
            pixelValue.push_back(direction[i]);
        }
        foImage->SetPixel(pixelIndex, pixelValue);
    }
    return 0;
}


int FiberOrientation::computeVolumeCounter(std::string vCounterImgFileName){
    // Pass the vtk polydata fiber file which is generated from ukf tractography 
    output("Reading fiber file started");
    FiberSegmentExtractor fse(fiberImgFileName.c_str()) ;
    output("Reading fiber file finished");
    // Pass the original DWI file 
    output("Reading DWI file started");
    VolumeCounter<Counter_WeightedVertices> vcc(subdivisionLevel, dwiImgFileName.c_str()) ;
    //VolumeCounter<Counter_WeightedVertices> vcc(6, argv[2]) ;
    output("Reading DWI file finished");
    // Compute the total number of fibers 
    //
    int totalfibers = (fse.GetNumberOfFibers());
    output2("Number of fibers = ", totalfibers);
    // Compute the total number of segments 
    int totalSegments = fse.GetNumberOfSegments();
    output2("Number of segments = ", totalSegments);

    int count = 0;
    int wildCount = 0;
    output("Computing Volume Counter");
    if(COUNTDISP){
        cout<<"No. of segments read = ";
    }
    bool fiberStart = true, fiberEnd = false;
    FiberSegment currSegment, prevSegment, nextSegment;
    PositionInVolume point1, point2;
    while(fse){
        currSegment = fse.Next(fiberEnd);
        point1 = vcc.GetVoxelPositionInVolume(currSegment.p1);
        point2 = vcc.GetVoxelPositionInVolume(currSegment.p2);
        if (!point1.isValid() || !point2.isValid()){
            wildCount++ ;
            continue ;
        }
        
        //Handle special case of fiberStart
        if(fiberStart){
            Counter &c1 = vcc.GetCounterForResidingVoxel(currSegment.p1);
            c1.Add(currSegment.p2 - currSegment.p1);
            c1.Add(currSegment.p1 - currSegment.p2);
            fiberStart = false;
            prevSegment = currSegment;
            count++;
            continue;
        }

        // Normal case of computing tangential vector at current fiber point using neighbor points
        Counter &c1 = vcc.GetCounterForResidingVoxel(currSegment.p1);
        c1.Add(currSegment.p2 - prevSegment.p1);
        c1.Add(prevSegment.p1 - currSegment.p2);
        prevSegment = currSegment;
        
        //Handle special case of fiberEnd
        if(fiberEnd){
            Counter &c1 = vcc.GetCounterForResidingVoxel(currSegment.p2);
            c1.Add(currSegment.p2 - currSegment.p1);
            c1.Add(currSegment.p1 - currSegment.p2);
            fiberStart = true;
            fiberEnd = false;
        }
        count++ ;
        if(count % 1000000 == 0){
            output2("Current Fiber Track Status = ", count);
        }
        if(COUNTDISP){
            display(count);
        }
    }
    output2(count, "\nTask complete\n");
    output("Writing Volume Counter to Nrrd File");    
    vcc.WriteVolumeCounterToNrrdFile(vCounterImgFileName.c_str()) ;
    return 0;
}

int FiberOrientation::writeFiberDistancesPerVoxelTextFile(std::string volumeCounterFileName, std::string foImgFileName, std::string dImgFileName){
    typedef std::vector<double>	VectorType ;
    //Get the histogram bin coordinates by instantiating an Icosahedron and retrieving coords
    SphereIkosahedronType::Pointer ico = SphereIkosahedronType::New() ;
	ico->SetSubdivisionLevel(subdivisionLevel);
	ico->Initialize();
    std::vector < VectorType > cartesianCoords = ico->GetCoordinateTable();
    std::vector < VectorType > phiThetaCoords = ico->GetPhiThetaTable();
    std::vector < double > phiThetaVal1, phiThetaVal2, phiThetaVal3;

    //Read the volume counter
    output("Reading the Volume Counter");
    VolumeCounter<Counter_WeightedVertices> vc(volumeCounterFileName.c_str());
    int dims[3];
    // Initialize the dimensions, origin and axes
    for(int i = 0; i < VOLUME_DIMENSION; i++){
        dims[i] = vc.GetNumVoxelsAlongAxis(i);
    }

    //Read FO Image and its meta information
    const unsigned int FODimension = 3;
    typedef std::vector< double > FOPixelType;
    typedef itk::Image< FOPixelType, FODimension > FOImageType;
    typedef itk::ImageRegionConstIterator< FOImageType > FOConstIteratorType;

    FOImageType::Pointer foImage = FOImageType::New();
    getFOImageHandler(foImage, foImgFileName); 
    FOConstIteratorType fo_it(foImage, foImage->GetLargestPossibleRegion());

    FOImageType::IndexType foPixelIndex;
    FOImageType::PixelType foPixelValue;

    //Initialize Distance Image
    const unsigned int dDimension = 3;
    typedef std::vector< double > dPixelType;
    typedef itk::Image< dPixelType, dDimension > dImageType;

    typedef itk::ImageFileWriter< dImageType > dWriterType;
    typedef itk::ImageRegionConstIterator< dImageType > dConstIteratorType;

    dImageType::PointType dOrigin;
    dImageType::SpacingType dSpacing;
    dImageType::SizeType dSize;

    //Copy the size, origin and spacing from FOImage
    FOMeta *meta = new FOMeta();
    readFOMeta(foImgFileName, meta);

    for(int i = 0; i < VOLUME_DIMENSION; i++){
        dOrigin[i] = meta->origin[i];
        dSpacing[i] = meta->spacing[i];
        dSize[i] = meta->size[i];
    }
    
   // FOImageType::Pointer foImage = FOImageType::New();
    dImageType::Pointer dImage = dImageType::New();
    dImageType::RegionType dRegion;
    dRegion.SetSize(dSize);
    dImage->SetSpacing(dSpacing);
    dImage->SetOrigin(dOrigin);
    dImage->SetRegions(dRegion);
    dImage->Allocate();

    dImageType::IndexType dPixelIndex;
    dImageType::PixelType dPixelValue;
    std::vector < double > temp;

    double maxVal = 0.0;
    unsigned int maxPos = 0;
    double d1 = 0.0, d2 = 0.0;
    output("Computing Fiber Distance Image ... please be patient");
    for (int i = 0; i < dims[0]; i++){
        for (int j = 0; j < dims[1]; j++){
            for (int k = 0; k < dims[2]; k++){
                Counter &c = vc.GetCounterForVoxel(i, j, k) ;
                if (!c.IsEmpty()){
                    //Compute max bin position
                    maxVal = 0.0, maxPos = 0;
                    std::vector<double> res = c.GetBins() ;
                    for (unsigned int l = 0; l < res.size(); l++){
                        if(res[l] > maxVal){
                            maxVal = res[l];
                            maxPos = l;
                        }
                    }
                    phiThetaVal1 = phiThetaCoords[maxPos];

                    foPixelIndex[0] = dPixelIndex[0] = i;
                    foPixelIndex[1] = dPixelIndex[1] = j;
                    foPixelIndex[2] = dPixelIndex[2] = k;

                    foPixelValue = foImage->GetPixel(foPixelIndex);
                    dPixelValue = dImage->GetPixel(dPixelIndex);
                    temp.clear();
                    for(unsigned int l = 0; l < foPixelValue.size(); l += 3){
                        //Compute distances to both possible fiber vectors
                        phiThetaVal2 = convertCartesianToPhiTheta(foPixelValue[l], foPixelValue[l+1], foPixelValue[l+2]);
                        phiThetaVal3 = convertCartesianToPhiTheta(-1 * foPixelValue[l], -1 * foPixelValue[l+1], -1 * foPixelValue[l+2]);
                        d1 = getSphericalDistance(phiThetaVal1, phiThetaVal2);
                        d2 = getSphericalDistance(phiThetaVal1, phiThetaVal3);
                        if(d1 < 0 || d2 < 0){
                            output("Error Occurred, encountered negative distance");
                            displayPoint(dPixelIndex);
                            return -1;
                        }
                        if(d1 <= d2){
                            temp.push_back(d1);
                        }else{
                            temp.push_back(d2);
                        }
                    }
                    sort(temp.begin(), temp.end());
                    for(unsigned int l = 0; l < temp.size(); l++){
                        dPixelValue.push_back(temp[l]);
                    }
                    dImage->SetPixel(dPixelIndex, dPixelValue);
                }
            }
        }
    }

    // Format of Fiber Distance File
    // ================================
    // begin header
    // dimension=3
    // type=double
    // size=maxx maxy maxz
    // origin=ox oy oz
    // spacing=sx sy sz
    // end header
    //
    // begin data
    // voxel=0 0 0
    // size=2
    // 0.1
    // 0.6
    // voxel=1 0 0
    // ...
    // ...
    // end data
    
    // Write the image to a text file
    output("Writing the distance image to text file...please be patient");
 	std::ofstream dist(dImgFileName.c_str());
    dist<<"begin header"<<endl;
    dist<<"dimension="<<VOLUME_DIMENSION<<endl;
    dist<<"type=double"<<endl;
    dist<<"size="<<dSize[0]<<" "<<dSize[1]<<" "<<dSize[2]<<endl;
    dist<<"origin="<<dOrigin[0]<<" "<<dOrigin[1]<<" "<<dOrigin[2]<<endl;
    dist<<"spacing="<<dSpacing[0]<<" "<<dSpacing[1]<<" "<<dSpacing[2]<<endl;
    dist<<"end header"<<endl;
    dist<<endl;
    dist<<"begin data"<<endl;

    long int size = 0;
    dConstIteratorType d_it( dImage, dImage->GetLargestPossibleRegion());
    d_it.GoToBegin();
    while(!d_it.IsAtEnd()){
        dPixelIndex = d_it.GetIndex();
        dPixelValue = d_it.Get();
        size = dPixelValue.size();
        dist<<"voxel="<<dPixelIndex[0]<<" "<<dPixelIndex[1]<<" "<<dPixelIndex[2]<<endl;
        dist<<"size="<<size<<endl;
        for(unsigned int i = 0; i < size; i++){
            dist<<dPixelValue[i]<<endl;
        }
        ++d_it;
    }
    dist<<"end data"<<endl;
    return 0;
}

std::vector <double> FiberOrientation::convertCartesianToPhiTheta(double x, double y, double z){
    //Refer http://en.wikipedia.org/wiki/Spherical_coordinate_system
    std::vector <double> phiTheta;

    double temp_phi = atan2(y, x);
    while(temp_phi >= M_PI)
    {
        temp_phi -= 2*M_PI;
    }
    while(temp_phi < -M_PI)
    {
        temp_phi += 2*M_PI;
    }
    phiTheta.push_back(temp_phi);//phi (phi = atan2(y/x))
    phiTheta.push_back(acos(z));//theta (theta = acos(z/r)) , unit sphere, so r = 1

    return phiTheta;
}

double FiberOrientation::getSphericalDistance(std::vector <double> phiTheta1, std::vector <double> phiTheta2){
    //Refer http://en.wikipedia.org/wiki/Great-circle_distance
    double num = 0.0, den = 0.0, d = 0.0, sigma = 0.0;

    double phi1 = phiTheta1[0];
    double theta1 = phiTheta1[1];
    double phi2 = phiTheta2[0];
    double theta2 = phiTheta2[1];
    double deltaTheta = theta2 - theta1;
    //double deltaPhi = phi2 - phi1;
    double r = 1.0;

    //Using Haversines
    //d = 2 * asin(sqrt((sin(deltaPhi/2) * sin(deltaPhi/2)) + (cos(phi1) * cos(phi2) * sin(deltaTheta/2) * sin(deltaTheta/2))));

    //Better way to handle all cases
    num = sqrt(pow((cos(phi2) * sin(deltaTheta)), 2) + pow((cos(phi1) * sin(phi2) - sin(phi1) * cos(phi2) * cos(deltaTheta)), 2));
    den = sin(phi1) * sin(phi2) + cos(phi1) * cos(phi2) * cos(deltaTheta);
    sigma = atan2(num, den);
    d = r * sigma;
    if(DEBUG){
        output2("Distance = ", d);
    }
    return d;
}


int FiberOrientation::computePercentileFiberDistanceImage(std::string dImgFileName, std::string path, double percentile){
    //Read Fiber Distance Image
    output("Reading Fiber Distances per Voxel Image");
    const unsigned int dDimension = 3;
    typedef std::vector< double > dPixelType;
    typedef itk::Image< dPixelType, dDimension > dImageType;
    typedef itk::ImageRegionConstIterator< dImageType > dConstIteratorType;

    dImageType::Pointer dImage = dImageType::New();
    //Use the same FOImageHandler to read fiber distance image
    getFOImageHandler(dImage, dImgFileName); 
    dConstIteratorType dist_it( dImage, dImage->GetLargestPossibleRegion());

    dImageType::PixelType dPixelValue;

    //Read Mask Image
    output("Reading Mask Image");
    const unsigned int mDimension = 3;
    typedef double mPixelType;
    typedef itk::Image< mPixelType, mDimension > mImageType;
    typedef itk::ImageFileReader<mImageType> mReaderType;
    typedef itk::ImageRegionConstIterator< mImageType > mConstIteratorType;

    mReaderType::Pointer maskReader;
    mImageType::Pointer maskImage;

    maskReader = mReaderType::New();
    maskReader->SetFileName(maskImgFileName.c_str());
    maskReader->Update();
    maskImage = maskReader->GetOutput();
    mConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    mImageType::IndexType maskIndex;
    mImageType::PixelType maskValue;

    //Percentile Image
    typedef double pPixelType;
    const unsigned int pDimension = 3;
    typedef itk::Image< pPixelType, pDimension > pImageType;
    typedef itk::ImageFileWriter< pImageType > pWriterType;

    pImageType::PointType pOrigin;
    pImageType::SpacingType pSpacing;
    pImageType::SizeType pSize;

    //Use same dimensions as FA image
	for (int i = 0; i < VOLUME_DIMENSION; i++){
		pOrigin[i]  = faOrigin[i];
        pSpacing[i] = faSpacing[i];
        pSize[i] = faSize[i];
    }

    pImageType::Pointer pImage = pImageType::New();
    
    pImageType::RegionType pRegion;
    pRegion.SetSize(pSize);
    pImage->SetSpacing(pSpacing);
    pImage->SetOrigin(pOrigin);
    pImage->SetRegions(pRegion);
    pImage->Allocate();
     
    pImageType::IndexType pPixelIndex;
    pImageType::PixelType pPixelValue;

    cout<<"Computing "<<percentile * 100 <<" Percentile Distance Image"<<endl;
    mask_it.GoToBegin();
    dist_it.GoToBegin();
    long int size = 0;
    int distIndex;

    while(!mask_it.IsAtEnd() && !dist_it.IsAtEnd()){
        maskIndex = mask_it.GetIndex();
        pPixelIndex[0] = maskIndex[0] , pPixelIndex[1] = maskIndex[1], pPixelIndex[2] = maskIndex[2];
        maskValue = mask_it.Get();
        if(maskValue > 0){
            dPixelValue = dist_it.Get();
            size = dPixelValue.size();
            if(size > 0){
                distIndex = (int)round((size-1) * percentile);
                pPixelValue = dPixelValue[distIndex];
            }
        }else{
            pPixelValue = 0.0;
        }
        pImage->SetPixel(pPixelIndex, pPixelValue);
        ++mask_it;
        ++dist_it;
    }

    // Update the writer 
    pWriterType::Pointer pWriter = pWriterType::New();
    std::ostringstream fn;
    fn << path <<"FiberDistance_" << percentile*100 << "_Percentile.nrrd";
    std::string fileName = fn.str();

    pWriter->SetFileName(fileName.c_str());
    pWriter->SetInput(pImage);
    try{
        pWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

int FiberOrientation::computeEarthMoversDistanceImage(std::string dImgFileName, std::string enImgFileName, std::string emdImgFileName, std::string path){
    std::string tempEmdTextFile = path + "emd_temp.txt";

    /*
    s.writeImageToTextFile(maskImgFileName, maskTextFile);

    output("Computing EMD for all the pixels");
    std::string cmd = "./run_compute_EMD.sh /playpen/MATLAB/R2011a/ " + dImgFileName + " " + maskTextFile + " " + tempEmdTextFile;
    output(cmd);
    int status = system(cmd.c_str());
    if(status != 0){
        output("Error executing matlab code, exiting ... ");
        exit(1);
    }
    */
    output("Writing EMD Image");
    int max = 500;
    writeEMDTextToImageFile(tempEmdTextFile, emdImgFileName, max);
    return 0; 
}

int FiberOrientation::writeFiberDistanceHistogram(std::string volumeCounterFileName, std::string foImgFileName, histoPlot h[], int size){
    typedef std::vector<double>	VectorType;
    //Get the histogram bin coordinates by instantiating an Icosahedron and retrieving coords
    SphereIkosahedronType::Pointer ico = SphereIkosahedronType::New() ;
	ico->SetSubdivisionLevel(subdivisionLevel);
	ico->Initialize();
    std::vector < VectorType > cartesianCoords = ico->GetCoordinateTable();
    std::vector < VectorType > phiThetaCoords = ico->GetPhiThetaTable();
    std::vector < double > phiThetaVal1, phiThetaVal2, phiThetaVal3;

    //Read the volume counter
    output("Reading the Volume Counter");
    VolumeCounter<Counter_WeightedVertices> vc(volumeCounterFileName.c_str());

    //Read FO Image and its meta information
    const unsigned int FODimension = 3;
    typedef std::vector< double > FOPixelType;
    typedef itk::Image< FOPixelType, FODimension > FOImageType;
    typedef itk::ImageRegionConstIterator< FOImageType > FOConstIteratorType;

    FOImageType::Pointer foImage = FOImageType::New();
    getFOImageHandler(foImage, foImgFileName); 
    FOConstIteratorType fo_it(foImage, foImage->GetLargestPossibleRegion());

    FOImageType::IndexType foPixelIndex;
    FOImageType::PixelType foPixelValue;

    for(int i = 0; i < size; i++){
        std::vector < double > temp;
        double maxVal = 0.0;
        unsigned int maxPos = 0;
        double d1 = 0.0, d2 = 0.0;
        unsigned int x = h[i].x;
        unsigned int y = h[i].y;
        unsigned int z = h[i].z;
        Counter &c = vc.GetCounterForVoxel(x, y, z);
        c.WriteCounterToVTKFile(h[i].filename.c_str());
        return 0;
        if(!c.IsEmpty()){
            //Compute max bin position
            std::ofstream hist(h[i].filename.c_str());
            maxVal = 0.0, maxPos = 0;
            std::vector<double> res = c.GetBins() ;
            for (unsigned int l = 0; l < res.size(); l++){
                if(res[l] > maxVal){
                    maxVal = res[l];
                    maxPos = l;
                }
            }
            phiThetaVal1 = phiThetaCoords[maxPos];

            foPixelIndex[0] = x;
            foPixelIndex[1] = y;
            foPixelIndex[2] = z;

            foPixelValue = foImage->GetPixel(foPixelIndex);
            temp.clear();
            for(unsigned int l = 0; l < foPixelValue.size(); l += 3){
                //Compute distances to both possible fiber vectors
                phiThetaVal2 = convertCartesianToPhiTheta(foPixelValue[l], foPixelValue[l+1], foPixelValue[l+2]);
                phiThetaVal3 = convertCartesianToPhiTheta(-1 * foPixelValue[l], -1 * foPixelValue[l+1], -1 * foPixelValue[l+2]);
                d1 = getSphericalDistance(phiThetaVal1, phiThetaVal2);
                d2 = getSphericalDistance(phiThetaVal1, phiThetaVal3);
                if(d1 < 0 || d2 < 0){
                    output("Error Occurred, encountered negative distance");
                    displayPoint(foPixelIndex);
                    return -1;
                }
                if(d1 <= d2){
                    temp.push_back(d1);
                }else{
                    temp.push_back(d2);
                }
            }
            for(unsigned int l = 0; l < temp.size(); l++){
                hist << temp[l] << endl;
            }
        }else{
            output2("Counter bin is empty, exiting, unable to write out the histogram = ", h[i].filename.c_str());
        }
        cout<<"Finished writing out the file = "<<h[i].filename<<endl;
    }
    return 0;
}

int FiberOrientation::writeEMDTextToImageFile(std::string inputFileName, std::string outputFileName, int max){
    typedef itk::ImageFileWriter< ImageType > WriterType;
    ImageType::Pointer outImage = ImageType::New();
    ImageType::RegionType outRegion;
    outRegion.SetSize(faSize);
    outImage->SetSpacing(faSpacing);
    outImage->SetOrigin(faOrigin);
    outImage->SetRegions(outRegion);
    outImage->Allocate();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType pixelValue;
    ifstream inFile(inputFileName.c_str());
    std::string line;
    unsigned int i, x, y, z;
    int j, k;
    char value[100];
    double val;

    while(getline(inFile, line)){
        k = j = 0;
        x = y = z = 0;
        val = 0.0;
        for(i = 0; i < line.length(); i++){
            if(line[i] == ','){
                value[j] = '\0';
                j = 0;
                if(k == 0){
                    x = (unsigned int)atoi(value);
                }else if(k == 1){
                    y = (unsigned int)atoi(value);
                }else if(k == 2){
                    z = (unsigned int)atoi(value);
                }
                k++;
            }else{
                value[j++] = line[i];
            }
        }
        value[j] = '\0';
        if(k == 3){
            val = strtod(value, NULL);
        }
        pixelIndex[0] = x;
        pixelIndex[1] = y;
        pixelIndex[2] = z;
        if(val > max){
            val = max;
        }
        pixelValue = val;
        outImage->SetPixel(pixelIndex, pixelValue);
    }

    output("Writing out the Image");
    WriterType::Pointer outWriter = WriterType::New();
    outWriter->SetFileName(outputFileName.c_str());
    outWriter->SetInput(outImage);
    try{
        outWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

#endif
