#ifndef STATS_H
#define STATS_H

/*
 *	Author: Ravikiran J
 *	Email : ravikirn@cs.unc.edu
 */
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkMaskImageFilter.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionConstIterator.h"
#include "itkImageRegionIterator.h"
#include "itkGaussianDistribution.h"
#include <iostream>
#include <fstream>
#include <math.h>
#include <DisplayFunc.h>

const unsigned int Dimension = 3;
typedef double PixelType;
typedef itk::Image< PixelType, Dimension > ImageType;
typedef itk::ImageFileReader<ImageType> ReaderType;
typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;
typedef itk::ImageFileWriter< ImageType > WriterType;

struct imgStats{
    double mean;
    double sd;
    double min;
    double max;
    double median;
    double medianSD;
};

class Stats{
	public:
		explicit Stats(std::string faImgFileName, std::string enImgFileName, std::string maskImgFileName);

        double computePearsonCorrelation();

        int writeNormalizedAndDiffImages(std::string faNormImgFileName, std::string enNormImgFileName);

        int writeNormalizedCrossCorrImage();
        
        int writeNormalizedCrossCorrImage(std::string imgFileName1, std::string imgFileName2, std::string outputFileName);
        
        int computeCombinedResult(std::string imgFileName1, std::string imgFileName2, std::string imgFileName3, std::string outputFileName);

        int writeImageToTextFile(std::string inputFileName, std::string outputFileName);

        int writeTextToImageFile(std::string inputFileName, std::string outputFileName);

        int writeHistogramOfIntensities(std::string inputFileName, std::string outputFileName, int skipZero);

        int displayImageStats(std::string inputFileName, int nonzero);

        int computeImageStats(std::string inputFileName, imgStats* s, int nonzero);

        int computeErfValuedImage(std::string inputFileName, std::string outputFileName, std::string type);

        double computeNormalizedCrossCorrelation();

		virtual ~Stats(){}
		
	protected:
        ReaderType::Pointer faReader;
        ImageType::Pointer faImage;
        ImageType::PointType faOrigin;
        ImageType::SpacingType faSpacing;
        ImageType::SizeType faSize;

        ReaderType::Pointer enReader;
        ImageType::Pointer enImage;
        ImageType::PointType enOrigin;
        ImageType::SpacingType enSpacing;
        ImageType::SizeType enSize;

        ReaderType::Pointer maskReader;
        ImageType::Pointer maskImage;
        ImageType::PointType maskOrigin;
        ImageType::SpacingType maskSpacing;
        ImageType::SizeType maskSize;

        double meanFA;
        double sdFA;

        double meanEn;
        double sdEn;

        double correlation;
        double normCrossCorr;

        long int N;
} ;

Stats::Stats(std::string faImgFileName, std::string enImgFileName, std::string maskImgFileName){
    meanFA = meanEn = 0.0;
    sdFA = sdEn = 0.0;
    correlation = 0.0;
    normCrossCorr = 0.0;
    N = 0;

    faReader = ReaderType::New();
    faReader->SetFileName(faImgFileName.c_str());
    faReader->Update();
    enReader = ReaderType::New();
    enReader->SetFileName(enImgFileName.c_str());
    enReader->Update();
    maskReader = ReaderType::New();
    maskReader->SetFileName(maskImgFileName.c_str());
    maskReader->Update();

    
    faImage = faReader->GetOutput();
    enImage = enReader->GetOutput();
    maskImage = maskReader->GetOutput();

    faOrigin = faImage->GetOrigin();
    faSpacing = faImage->GetSpacing();
    faSize = faImage->GetLargestPossibleRegion().GetSize();

    enOrigin = enImage->GetOrigin();
    enSpacing = enImage->GetSpacing();
    enSize = enImage->GetLargestPossibleRegion().GetSize();

    maskOrigin = maskImage->GetOrigin();
    maskSpacing = maskImage->GetSpacing();
    maskSize = maskImage->GetLargestPossibleRegion().GetSize();
        
    imgStats *faStats = new imgStats();
    imgStats *enStats = new imgStats();
    computeImageStats(faImgFileName, faStats, 0);
    computeImageStats(enImgFileName, enStats, 0);
    meanFA = faStats->mean;
    sdFA = faStats->sd;
    meanEn = enStats->mean;
    sdEn = enStats->sd;
}

double Stats::computePearsonCorrelation(){
    // Pearson Correlation is computed as per the definition found at http://en.wikipedia.org/wiki/Correlation_and_dependence
    //Initialize iterators
    ConstIteratorType fa_it( faImage, faImage->GetLargestPossibleRegion());
    ConstIteratorType en_it( enImage, enImage->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    fa_it.GoToBegin();
    en_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType enValue;
    ImageType::PixelType faValue;
    ImageType::PixelType maskValue;
    ImageType::PixelType xy = 0.0, x = 0.0, y = 0.0, xpow2 = 0.0, ypow2 = 0.0;

    while(!fa_it.IsAtEnd() && !en_it.IsAtEnd() && !mask_it.IsAtEnd()){
        pixelIndex = fa_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            faValue = fa_it.Get();
            enValue = en_it.Get();
            x += faValue;
            y += enValue;
            xy += faValue * enValue;
            xpow2 += pow(faValue,2);
            ypow2 += pow(enValue,2);
        }
        ++fa_it;
        ++en_it;
        ++mask_it;
    }
    correlation = ((N * xy) - x * y) / (sqrt(N * xpow2 - pow(x,2)) * sqrt(N * ypow2 - pow(y,2)));
    return correlation;
}


int Stats::writeNormalizedCrossCorrImage(){
    //Initialize iterators
    ConstIteratorType fa_it( faImage, faImage->GetLargestPossibleRegion());
    ConstIteratorType en_it( enImage, enImage->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    // Set same as FA image
    ImageType::Pointer ncc = ImageType::New();
    ImageType::RegionType nccRegion;
    nccRegion.SetSize(faSize);
    ncc->SetSpacing(faSpacing);
    ncc->SetOrigin(faOrigin);
    ncc->SetRegions(nccRegion);
    ncc->Allocate();

    fa_it.GoToBegin();
    en_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex, newIndex;
    ImageType::PixelType enValue;
    ImageType::PixelType faValue;
    ImageType::PixelType maskValue;
    ImageType::PixelType corrValue;
    ImageType::PixelType xy = 0.0, x = 0.0, y = 0.0, xpow2 = 0.0, ypow2 = 0.0;
    int count = 0, neighborLevel = 3;
    output2("Computing Normalized Cross Correlation with neighborhood level of ", neighborLevel);

    while(!mask_it.IsAtEnd()){
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            xy = 0.0, x = 0.0, y = 0.0, xpow2 = 0.0, ypow2 = 0.0, count = 0;
            for(int i = -neighborLevel/2; i <= neighborLevel/2 ; i++){
                for(int j = -neighborLevel/2; j <= neighborLevel/2; j++){
                    for(int k = -neighborLevel/2; k <= neighborLevel/2; k++){
                        newIndex[0] = pixelIndex[0]+i;
                        newIndex[1] = pixelIndex[1]+j;
                        newIndex[2] = pixelIndex[2]+k;
                        if(newIndex[0] < 0 || newIndex[0] > (unsigned int)faSize[0] || newIndex[1] < 0 || newIndex[1] > (unsigned int)faSize[1] || newIndex[2] < 0 || newIndex[2] > (unsigned int)faSize[2]){
                            faValue = 0.0;
                            enValue = 0.0;
                        }else{
                            faValue = faImage->GetPixel(newIndex);
                            enValue = enImage->GetPixel(newIndex);
                        }
                        x += faValue;
                        y += enValue;
                        xy += faValue * enValue;
                        xpow2 += pow(faValue,2);
                        ypow2 += pow(enValue,2);
                        count++;
                    }
                }
            }
            if( (xpow2 == 0.0 && x == 0.0) || (ypow2 == 0.0 && y == 0.0) ){
                corrValue = 0.0;
            }else{
                corrValue = fabs(((count * xy) - x * y) / (sqrt(count * xpow2 - pow(x,2)) * sqrt(count * ypow2 - pow(y,2))));
            }
        }else{
            corrValue = 0.0;
        }
        ncc->SetPixel(pixelIndex, corrValue);
        ++mask_it;
    }
    output("Writing Normalized Cross Correlation Image");
    WriterType::Pointer nccWriter = WriterType::New();
    nccWriter->SetFileName("results/normalizedCrossCorrImage.nrrd");
    nccWriter->SetInput(ncc);
    try{
        nccWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

int Stats::writeNormalizedCrossCorrImage(std::string imgFileName1, std::string imgFileName2, std::string outputFileName){
    ReaderType::Pointer img1Reader;
    ReaderType::Pointer img2Reader;
    ImageType::Pointer img1;
    ImageType::Pointer img2;

    img1Reader = ReaderType::New();
    img1Reader->SetFileName(imgFileName1.c_str());
    img1Reader->Update();
    
    img2Reader = ReaderType::New();
    img2Reader->SetFileName(imgFileName2.c_str());
    img2Reader->Update();
    
    img1 = img1Reader->GetOutput();
    img2 = img2Reader->GetOutput();

    ConstIteratorType img1_it( img1, img1->GetLargestPossibleRegion());
    ConstIteratorType img2_it( img2, img2->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    // Set same as FA image
    ImageType::Pointer ncc = ImageType::New();
    ImageType::RegionType nccRegion;
    nccRegion.SetSize(faSize);
    ncc->SetSpacing(faSpacing);
    ncc->SetOrigin(faOrigin);
    ncc->SetRegions(nccRegion);
    ncc->Allocate();

    img1_it.GoToBegin();
    img2_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex, newIndex;
    ImageType::PixelType img1Value;
    ImageType::PixelType img2Value;
    ImageType::PixelType maskValue;
    ImageType::PixelType corrValue;
    ImageType::PixelType xy = 0.0, x = 0.0, y = 0.0, xpow2 = 0.0, ypow2 = 0.0;
    int count = 0, neighborLevel = 3;
    output2("Computing Normalized Cross Correlation with neighborhood level of ", neighborLevel);

    while(!mask_it.IsAtEnd()){
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            xy = 0.0, x = 0.0, y = 0.0, xpow2 = 0.0, ypow2 = 0.0, count = 0;
            for(int i = -neighborLevel/2; i <= neighborLevel/2 ; i++){
                for(int j = -neighborLevel/2; j <= neighborLevel/2; j++){
                    for(int k = -neighborLevel/2; k <= neighborLevel/2; k++){
                        newIndex[0] = pixelIndex[0]+i;
                        newIndex[1] = pixelIndex[1]+j;
                        newIndex[2] = pixelIndex[2]+k;
                        if(newIndex[0] < 0 || newIndex[0] > (unsigned int)faSize[0] || newIndex[1] < 0 || newIndex[1] > (unsigned int)faSize[1] || newIndex[2] < 0 || newIndex[2] > (unsigned int)faSize[2]){
                            img1Value = 0.0;
                            img2Value = 0.0;
                        }else{
                            img1Value = img1->GetPixel(newIndex);
                            img2Value = img2->GetPixel(newIndex);
                        }
                        x += img1Value;
                        y += img2Value;
                        xy += img1Value * img2Value;
                        xpow2 += pow(img1Value,2);
                        ypow2 += pow(img2Value,2);
                        count++;
                    }
                }
            }
            if( (xpow2 == 0.0 && x == 0.0) || (ypow2 == 0.0 && y == 0.0) || (count * xpow2 - pow(x,2) == 0) || (count * ypow2 - pow(y,2) == 0)){
                corrValue = 0.0;
            }else{
                corrValue = fabs(((count * xy) - x * y) / (sqrt(count * xpow2 - pow(x,2)) * sqrt(count * ypow2 - pow(y,2))));
            }
        }else{
            corrValue = 0.0;
        }
        ncc->SetPixel(pixelIndex, corrValue);
        ++mask_it;
    }
    output("Writing Normalized Cross Correlation Image");
    WriterType::Pointer nccWriter = WriterType::New();
    nccWriter->SetFileName(outputFileName.c_str());
    nccWriter->SetInput(ncc);
    try{
        nccWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}
// Cubic root of the product of the 3 image values
int Stats::computeCombinedResult(std::string imgFileName1, std::string imgFileName2, std::string imgFileName3, std::string outputFileName){
    ReaderType::Pointer img1Reader;
    ReaderType::Pointer img2Reader;
    ReaderType::Pointer img3Reader;
    ImageType::Pointer img1;
    ImageType::Pointer img2;
    ImageType::Pointer img3;

    img1Reader = ReaderType::New();
    img1Reader->SetFileName(imgFileName1.c_str());
    img1Reader->Update();
    
    img2Reader = ReaderType::New();
    img2Reader->SetFileName(imgFileName2.c_str());
    img2Reader->Update();

    img3Reader = ReaderType::New();
    img3Reader->SetFileName(imgFileName3.c_str());
    img3Reader->Update();
    
    img1 = img1Reader->GetOutput();
    img2 = img2Reader->GetOutput();
    img3 = img3Reader->GetOutput();

    ConstIteratorType img1_it(img1, img1->GetLargestPossibleRegion());
    ConstIteratorType img2_it(img2, img2->GetLargestPossibleRegion());
    ConstIteratorType img3_it(img3, img3->GetLargestPossibleRegion());
    ConstIteratorType mask_it(maskImage, maskImage->GetLargestPossibleRegion());

    // Set same as FA image
    ImageType::Pointer ncc = ImageType::New();
    ImageType::RegionType nccRegion;
    nccRegion.SetSize(faSize);
    ncc->SetSpacing(faSpacing);
    ncc->SetOrigin(faOrigin);
    ncc->SetRegions(nccRegion);
    ncc->Allocate();

    img1_it.GoToBegin();
    img2_it.GoToBegin();
    img3_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType img1Value;
    ImageType::PixelType img2Value;
    ImageType::PixelType img3Value;
    ImageType::PixelType maskValue;
    ImageType::PixelType nccValue;

    while(!mask_it.IsAtEnd()){
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            img1Value = img1->GetPixel(pixelIndex);
            img2Value = img2->GetPixel(pixelIndex);
            img3Value = img3->GetPixel(pixelIndex);
            nccValue = pow(img1Value * img2Value * img3Value, 1/3.0);
            //nccValue = pow(img1Value * img2Value, 1/2.0);
        }else{
            nccValue = 0.0;
        }
        ncc->SetPixel(pixelIndex, nccValue);
        ++mask_it;
    }

    output("Writing Cubic Root of Products of Images");
    WriterType::Pointer nccWriter = WriterType::New();
    nccWriter->SetFileName(outputFileName.c_str());
    nccWriter->SetInput(ncc);
    try{
        nccWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

int Stats::computeErfValuedImage(std::string inputFileName, std::string outputFileName, std::string type){
    ReaderType::Pointer inReader;
    ImageType::Pointer in;

    inReader = ReaderType::New();
    inReader->SetFileName(inputFileName.c_str());
    inReader->Update();

    in = inReader->GetOutput();

    imgStats *inStats = new imgStats(); 
    computeImageStats(inputFileName, inStats, 0);
    displayImageStats(inputFileName, 0);

    ConstIteratorType in_it(in, in->GetLargestPossibleRegion());
    ConstIteratorType mask_it(maskImage, maskImage->GetLargestPossibleRegion());

    // Set same as FA image
    ImageType::Pointer erf = ImageType::New();
    ImageType::RegionType erfRegion;
    erfRegion.SetSize(faSize);
    erf->SetSpacing(faSpacing);
    erf->SetOrigin(faOrigin);
    erf->SetRegions(erfRegion);
    erf->Allocate();

    in_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType inValue;
    ImageType::PixelType maskValue;

    ImageType::PixelType erfValue;
    itk::Statistics::GaussianDistribution::Pointer gaussian = itk::Statistics::GaussianDistribution::New();
    while(!mask_it.IsAtEnd()){
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            inValue = in->GetPixel(pixelIndex);
            if(type == "fs"){
                erfValue = gaussian->EvaluateCDF(inValue, (inStats->mean + inStats->sd), pow(inStats->sd, 2));
            }else if(type == "en"){
                erfValue = gaussian->EvaluateCDF(inValue, inStats->mean, pow(inStats->sd, 2));
            }else if(type == "emd"){
                  erfValue = gaussian->EvaluateCDF(inValue, inStats->mean-inStats->sd, pow(inStats->sd, 2));
            }else if(type == "threshold"){
                erfValue = inValue <= inStats->mean ? 0.0 : inValue;
            }
        }else{
            erfValue = 0.0;
        }
        erf->SetPixel(pixelIndex, erfValue);
        ++mask_it;
    }
    /* //DEBUG
    if(type == "fs"){
        cout<<"sd  = "<<inStats->sd<<", sd2 = "<<inStats->max/15.0<<endl;
    }else if(type == "en"){
        cout<<"sd  = "<<inStats->sd<<", sd2 = "<<inStats->max/6.0<<endl;
    }else if(type == "emd"){
        cout<<"sd  = "<<inStats->sd<<", sd2 = "<<inStats->max/10.0<<endl;
    }
    */

    output("Writing the erf-valued Images");
    WriterType::Pointer erfWriter = WriterType::New();
    erfWriter->SetFileName(outputFileName.c_str());
    erfWriter->SetInput(erf);
    try{
        erfWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

int Stats::writeNormalizedAndDiffImages(std::string faNormImgFileName, std::string enNormImgFileName){
    //Initialize iterators
    ConstIteratorType fa_it( faImage, faImage->GetLargestPossibleRegion());
    ConstIteratorType en_it( enImage, enImage->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    ImageType::Pointer faNormal = ImageType::New();
    ImageType::RegionType faRegion;
    faRegion.SetSize(faSize);
    faNormal->SetSpacing(faSpacing);
    faNormal->SetOrigin(faOrigin);
    faNormal->SetRegions(faRegion);
    faNormal->Allocate();

    //Set origin same as FA - hack
    ImageType::Pointer enNormal = ImageType::New();
    ImageType::RegionType enRegion;
    enRegion.SetSize(enSize);
    enNormal->SetSpacing(enSpacing);
    enNormal->SetOrigin(faOrigin);
    enNormal->SetRegions(enRegion);
    enNormal->Allocate();
    
    // Set same as FA image
    ImageType::Pointer diffNormal = ImageType::New();
    ImageType::RegionType diffRegion;
    diffRegion.SetSize(faSize);
    diffNormal->SetSpacing(faSpacing);
    diffNormal->SetOrigin(faOrigin);
    diffNormal->SetRegions(diffRegion);
    diffNormal->Allocate();

    fa_it.GoToBegin();
    en_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType enValue;
    ImageType::PixelType faValue;
    ImageType::PixelType maskValue;
    ImageType::PixelType diffValue;
    output("Computing Normalized FA, Entropy and Diff Images");
    while(!fa_it.IsAtEnd() && !en_it.IsAtEnd() && !mask_it.IsAtEnd()){
        pixelIndex = fa_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            faValue = (fa_it.Get()-meanFA) / sdFA;
            enValue = (en_it.Get()-meanEn) / sdEn;
            diffValue = fabs(faValue-enValue);
        }else {
            faValue = 0.0;
            enValue = 0.0;
            diffValue = 0.0;
        }
        faNormal->SetPixel(pixelIndex, faValue);
        enNormal->SetPixel(pixelIndex, enValue);
        diffNormal->SetPixel(pixelIndex, diffValue);
        ++fa_it;
        ++en_it;
        ++mask_it;
    }
    output("Writing out Normalized FA, Entropy and Diff Images");
    WriterType::Pointer faWriter = WriterType::New();
    WriterType::Pointer enWriter = WriterType::New();
    WriterType::Pointer diffWriter = WriterType::New();

    faWriter->SetFileName(faNormImgFileName.c_str());
    enWriter->SetFileName(enNormImgFileName.c_str());
    diffWriter->SetFileName("../diff_fa_en.nrrd");

    faWriter->SetInput(faNormal);
    enWriter->SetInput(enNormal);
    diffWriter->SetInput(diffNormal);

    try{
        faWriter->Update();
        enWriter->Update();
        diffWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}

double Stats::computeNormalizedCrossCorrelation(){
    //Initialize iterators
    ConstIteratorType fa_it( faImage, faImage->GetLargestPossibleRegion());
    ConstIteratorType en_it( enImage, enImage->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());

    fa_it.GoToBegin();
    en_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType enValue;
    ImageType::PixelType faValue;
    ImageType::PixelType maskValue;
    normCrossCorr = 0.0;
    while(!fa_it.IsAtEnd() && !en_it.IsAtEnd() && !mask_it.IsAtEnd()){
        pixelIndex = fa_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            faValue = (fa_it.Get()-meanFA) / sdFA;
            enValue = (en_it.Get()-meanEn) / sdEn;
            normCrossCorr += faValue * enValue;  
        }
        ++fa_it;
        ++en_it;
        ++mask_it;
    }
    normCrossCorr = normCrossCorr / (N-1);
    return normCrossCorr;
}

int Stats::writeImageToTextFile(std::string inputFileName, std::string outputFileName){
    ReaderType::Pointer imgReader;
    ImageType::Pointer image;

    imgReader = ReaderType::New();
    imgReader->SetFileName(inputFileName.c_str());
    imgReader->Update();
    image = imgReader->GetOutput();

    ImageType::PixelType pixelValue;
    ImageType::IndexType mIndex;
    ImageType::PixelType mValue;

    typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;

    ConstIteratorType img_it( image, image->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());
    img_it.GoToBegin();
    mask_it.GoToBegin();

 	std::ofstream out(outputFileName.c_str());
    output("Writing to a text file");    
    while(!img_it.IsAtEnd() && !mask_it.IsAtEnd()){
        mIndex = mask_it.GetIndex();
        mValue = mask_it.Get();
        if(mValue > 0){
            pixelValue = img_it.Get();
            out << mIndex[0] << "," << mIndex[1] << "," << mIndex[2] << ","<< pixelValue << std::endl ;
        }else{
            out << mIndex[0] << "," << mIndex[1] << "," << mIndex[2] << ","<< 0 << std::endl ;
        }
        ++mask_it; 
        ++img_it;
    }
    out.close();
    return 0;
}

int Stats::writeTextToImageFile(std::string inputFileName, std::string outputFileName){
    typedef itk::ImageFileWriter< ImageType > WriterType;
    ImageType::Pointer outImage = ImageType::New();
    ImageType::RegionType outRegion;
    outRegion.SetSize(faSize);
    outImage->SetSpacing(faSpacing);
    outImage->SetOrigin(faOrigin);
    outImage->SetRegions(outRegion);
    outImage->Allocate();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType pixelValue;
    ifstream inFile(inputFileName.c_str());
    std::string line;
    unsigned int i, x, y, z;
    int j, k;
    char value[100];
    double val;

    while(getline(inFile, line)){
        k = j = 0;
        x = y = z = 0;
        val = 0.0;
        for(i = 0; i < line.length(); i++){
            if(line[i] == ','){
                value[j] = '\0';
                j = 0;
                if(k == 0){
                    x = (unsigned int)atoi(value);
                }else if(k == 1){
                    y = (unsigned int)atoi(value);
                }else if(k == 2){
                    z = (unsigned int)atoi(value);
                }
                k++;
            }else{
                value[j++] = line[i];
            }
        }
        value[j] = '\0';
        if(k == 3){
            val = strtod(value, NULL);
        }
        pixelIndex[0] = x;
        pixelIndex[1] = y;
        pixelIndex[2] = z;
        pixelValue = val;
        outImage->SetPixel(pixelIndex, pixelValue);
    }

    output("Writing out the Image");
    WriterType::Pointer outWriter = WriterType::New();
    outWriter->SetFileName(outputFileName.c_str());
    outWriter->SetInput(outImage);
    try{
        outWriter->Update();
    }
    catch( itk::ExceptionObject & err ){
        std::cerr << "ExceptionObject caught !" << std::endl;
        std::cerr << err << std::endl;
        return EXIT_FAILURE;
    }
    return 0;
}


int Stats::writeHistogramOfIntensities(std::string inputFileName, std::string outputFileName, int skipZero = 0){
    ReaderType::Pointer imgReader;
    ImageType::Pointer image;

    imgReader = ReaderType::New();
    imgReader->SetFileName(inputFileName.c_str());
    imgReader->Update();
    image = imgReader->GetOutput();

    ImageType::PixelType pixelValue;
    ImageType::IndexType mIndex;
    ImageType::PixelType mValue;

    typedef itk::ImageRegionConstIterator< ImageType > ConstIteratorType;

    ConstIteratorType img_it( image, image->GetLargestPossibleRegion());
    ConstIteratorType mask_it( maskImage, maskImage->GetLargestPossibleRegion());
    img_it.GoToBegin();
    mask_it.GoToBegin();
 	std::ofstream out(outputFileName.c_str());
    int count = 0;
    double maxVal = 1e+7, minVal = 1e-4;
    output("Writing Histogram to a text file");    
    while(!img_it.IsAtEnd() && !mask_it.IsAtEnd()){
        mIndex = mask_it.GetIndex();
        mValue = mask_it.Get();
        if(mValue > 0){
            pixelValue = img_it.Get();
            if(skipZero){
                if(pixelValue > 0){
                    out<<pixelValue<<endl;
                }
            }else{
                if(pixelValue >= minVal && pixelValue <= maxVal){
                    out<<pixelValue<<endl;
                }else{
                    count++;
                }
            }
        }
        ++mask_it;
        ++img_it;
    }
    if(count > 0){
        output2("Values lesser than minVal or greater than maxVal = ", count);
    }
   return 0;
}

int Stats::displayImageStats(std::string inputFileName, int nonzero = 0){
    imgStats *imgSt = new imgStats();
    computeImageStats(inputFileName, imgSt, nonzero);
    output2("Filename = ", inputFileName);
    cout<<"mean = "<<imgSt->mean<<", sd = "<<imgSt->sd<<", min = "<<imgSt->min<<", max = "<<imgSt->max<<", median = "<<imgSt->median<<", median absolute deviation = "<<imgSt->medianSD<<endl;
    return 0;
}

int Stats::computeImageStats(std::string inputFileName, imgStats* s, int nonzero = 0){
    ReaderType::Pointer imgReader;
    ImageType::Pointer img;

    imgReader = ReaderType::New();
    imgReader->SetFileName(inputFileName.c_str());
    imgReader->Update();
    
    img = imgReader->GetOutput();

    ConstIteratorType img_it(img, img->GetLargestPossibleRegion());
    ConstIteratorType mask_it(maskImage, maskImage->GetLargestPossibleRegion());

    img_it.GoToBegin();
    mask_it.GoToBegin();

    ImageType::IndexType pixelIndex;
    ImageType::PixelType imgValue;
    ImageType::PixelType maskValue;

    double N = 0.0, count = 0.0;
    double imgSum = 0.0, imgMean = 0.0, imgSD = 0.0, imgMin = 1000.0, imgMax = 0.0, imgMedian = 0.0, imgMedianDeviation = 0.0;
    std::vector <double> imgVals;

    while(!mask_it.IsAtEnd()){
        count++;
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            imgValue = img->GetPixel(pixelIndex);
            if(nonzero){
                if(imgValue > 0){
                    if(imgValue < imgMin){
                        imgMin = imgValue;
                    }
                    if(imgValue > imgMax){
                        imgMax = imgValue;
                    }
                    imgVals.push_back(imgValue);
                    imgSum += imgValue;
                    N += 1.0;
                }
            }else{
                if(imgValue < imgMin){
                    imgMin = imgValue;
                }
                if(imgValue > imgMax){
                    imgMax = imgValue;
                }
                imgVals.push_back(imgValue);
                imgSum += imgValue;
                N += 1.0;
            }
        }
        ++mask_it;
    }
    imgMean = imgSum / N;

    img_it.GoToBegin();
    mask_it.GoToBegin();
    imgSum = 0.0;
    while(!mask_it.IsAtEnd()){
        pixelIndex = mask_it.GetIndex();
        maskValue = mask_it.Get();
        if(maskValue > 0){
            if(nonzero){
                if(imgValue > 0){
                    imgValue = img->GetPixel(pixelIndex);
                    imgSum += pow(imgValue-imgMean,2);
                }
            }else{
                imgValue = img->GetPixel(pixelIndex);
                imgSum += pow(imgValue-imgMean,2);
            }
        }
        ++mask_it;
    }
    imgSum = imgSum / N;
    imgSD = sqrt(imgSum);

    //Find median
    sort(imgVals.begin(), imgVals.end());
    unsigned int medianPos = imgVals.size() / 2;
    imgMedian = imgVals[medianPos];

    for(unsigned int i = 0; i < imgVals.size(); i++){
        imgVals[i] = fabs(imgVals[i]-imgMedian);
    }
    sort(imgVals.begin(), imgVals.end());
    imgMedianDeviation = imgVals[medianPos];

    s->mean = imgMean;
    s->sd = imgSD;
    s->min = imgMin;
    s->max = imgMax;
    s->median = imgMedian;
    s->medianSD = imgMedianDeviation;
    return 0;
}
#endif
