package edu.jhu.ece.iacl.algorithms.MGDM;

import java.util.Arrays;

import edu.jhu.ece.iacl.algorithms.MGDM.forces.*;

/**
 * Class implementing the evolution of the MGDM functions according to speeds or "forces."
 * 
 * @author John Bogovic
 *
 */
public class MgdmEvolution {
	
	protected MgdmDecomposition 	mgdmDecomp;
	protected MgdmNarrowband 		nb;
	protected MgdmForceSpec			mgdmForces;
	protected MgdmDataRepository	mgdmRepo;
	
	protected double				stepsize=0.2;
	protected boolean 				adaptStepSize = true;
	protected boolean 				instabilityWarning = false;
	protected double				maxSpeed = 0;
	
	protected final static double 	stepEps = 0.001;
	protected final static double 	maxStep = 10;
	
	protected float					lowlevel = 0.05f;
	

	/**
	 * Default constructor
	 */
	public MgdmEvolution(){
	}
	
	/**
	 * Constructor for a 2D Mgdm decomposition/evolution
	 * @param seg initial segmentation
	 */
	public MgdmEvolution(int[][] seg){
		mgdmDecomp = new MgdmDecomposition(seg);
	}
	
	/**
	 * Constructor for a 3D Mgdm decomposition/evolution
	 * @param seg initial segmentation
	 */
	public MgdmEvolution(int[][][] seg){
		mgdmDecomp = new MgdmDecomposition(seg);
	}
	
	public MgdmEvolution(MgdmDecomposition mgdmDecomp){
		this.mgdmDecomp=mgdmDecomp;
	}
	
	public void setDataRepository(MgdmDataRepository mgdmRepo){
		this.mgdmRepo=mgdmRepo;
	}
	
	public void setForce(MgdmForceSpec mgdmForces){
		this.mgdmForces=mgdmForces;
	}
	public void setStepSize(double stepSize){
		this.stepsize=stepSize;
	}
	public void setAdaptStepSize(boolean adaptStepSize){
		this.adaptStepSize=adaptStepSize;
	}
	public MgdmDecomposition getDecomposition(){
		return mgdmDecomp;
	}
	
	public void evolveNarrowBand(int maxIterations, int maxItersBeforeReinit){
		
		
		// initialize narrow band
		nb = new MgdmNarrowband(mgdmDecomp);
		
		
		boolean reinit = false;	// flat that forces reinitialization
		int itersSinceReinit = 0;
		
		// keeps track of number of points with a label change at each level
		// of the mgdm decomposition
		int[] nswap = new int[mgdmDecomp.nmgdm]; 
		
		// update forces
		mgdmForces.initializeForces(mgdmDecomp, mgdmRepo);
		
		// * ******************* * //
		// * MAIN ITERATION LOOP * //
		// * ******************* * //
		int t = 0;
		
		//ArrayList<Integer> changed = new ArrayList<Integer>();
		for (t = 0; t < maxIterations; t++) {
			
//			changed.clear();
			
			Arrays.fill(nswap, 0); // reset array
			
			// loop through the narrow band
			MgdmNarrowband.MgdmNarrowbandIterator it = nb.iterator();
			
			// check to see if the CFL condition is broken
			instabilityWarning = false;
			maxSpeed= 0;
						
			it.iteratorReset();
			while(it.hasNext()){
				int xyz 	= it.next();
				int nbidx 	= it.nextNarrowBandIndex();
			
				// get the labels and distances from the narrow band
				int[] mgdmlabels  = nb.getMgdmLabels(nbidx);
				float[] mgdmdists = nb.getMgdmFunctions(nbidx);
				
				if(mgdmDecomp.isDebug(xyz)){
					System.out.println("Before:");
					System.out.print(nb.printDecomposition(nbidx));
					System.out.println(" ");
				}
				
				// evolve each distance function in the decomposition
				for(int n = mgdmDecomp.nmgdm-1; n >= 0; n--){
					
					int nbr = -1;
					if (n == mgdmDecomp.nmgdm-1) {
						nbr = mgdmDecomp.getLastNeighbor(xyz);
					}else{
						nbr = mgdmlabels[n+1];
					}


					// if there is not a valid label at this level - do nothing and continue
					if (mgdmlabels[n] == MgdmDecomposition.EMPTY) {
						continue;
					}
					
					// if the competing label is not valid - do nothing and continue
					if (nbr == MgdmDecomposition.EMPTY) {
						continue;
					}
					
					
					double speed = levelSetSpeed(xyz, mgdmlabels[n], nbr);
					mgdmdists[n] += speed;
					
					if(mgdmDecomp.isDebug(xyz)){
						System.out.println("computing speed for boundary " + mgdmlabels[n] + " - " + nbr);
						System.out.println("Speed at level["+n+"]:" +speed);
					}
					
					// if the distance function is negative, there has been a label change
					if(mgdmdists[n]<0){
						nswap[n]++; 
						
						if (n == mgdmDecomp.nmgdm-1) {
							
							mgdmlabels[n] = nbr;
							
							nb.setLabel(nbidx, n, nbr);
							nb.setDistance(nbidx, n, -1*mgdmdists[n]);
							
						}else if(n == 0){
							
							// test topology
							if(homeomorphicLabeling(xyz)){ // if the change is allowed by the topology rules
								
								
								// swap label and neighbor in hierarchy
								nb.setLabel(nbidx, 	n+1,	mgdmlabels[n]); // demote former closer neighbor
								nb.setLabel(nbidx, 	n, 	nbr);			 // promote former further neighbor
								
								// update the segmentation array
								mgdmDecomp.setSegmentation(xyz, nbr);
								
//								changed.add(xyz);
								
							}else{ // otherwise, set the distance to a an epsilon
								
								nb.setDistance(nbidx, n, lowlevel);
								
							}
							
							// test reinitialization conditions
							
							// reinitialize if a "landmine" is hit
							// i.e. a label change occurs close to the far edge of the narrow band
							if(nb.isLandmine(xyz)){
								reinit = true;
							}
							
							
						}else{
							// swap label and neighbor in hierarchy
							nb.setLabel(nbidx, 	n+1,	mgdmlabels[n]); // demote former closer neighbor
							nb.setLabel(nbidx, 	n, 	nbr);			 // promote former further neighbor
							nb.setDistance(nbidx, n, -1*mgdmdists[n]);
						}

						
					// end label change
						
					}else{ // no label change, just update the distance
						
						nb.setDistance(nbidx, n, mgdmdists[n]);
						
					}
					
					
				} // end loop over mgdm levels
				
				if(mgdmDecomp.isDebug(xyz)){
					System.out.println("After:");
					System.out.print(nb.printDecomposition(nbidx));
					System.out.println(" ");
				}
				
			} // end loop over space
			
//			Iterator<Integer> changedit = changed.iterator();
//			
//			System.out.println("swapped: " + nswap[0]);
//			System.out.println("changed: " + changed.size());
//			while(changedit.hasNext()){
//				
//				
//				int[] coords = MgdmUtils.indexToCoordinates(changedit.next(),
//						mgdmDecomp.nxPad, mgdmDecomp.nyPad, mgdmDecomp.nzPad);
//				
//				System.out.println("("+coords[0]+","+coords[1]+","+coords[2]+")");
//			}
			
			// Copy new results from narrow band into the main decomposition object
			// only if there is no instability warning
			if(!instabilityWarning ){
				mgdmDecomp.updateFromNarrowBand(nb);
			}else{
				reinit = true;
			}
			
			if(itersSinceReinit>maxItersBeforeReinit){
				reinit = true;
			}
			itersSinceReinit++;
			
			// * ************ * //
			// * REINITIALIZE * //
			// * ************ * //
			if(reinit){
				
				System.out.print("\nReinitializing...");
				
				if (adaptStepSize) {
					updateStepSize();
				}
				
				mgdmDecomp.fastMarchingReinitialization(); 	// reinitialize
				nb.buildFromDecomposition(mgdmDecomp); 		// copy results to narrow band
				
				// update forces
				mgdmForces.updateForces(mgdmDecomp, mgdmRepo);
				
				
				// reset reinitialization counters
				reinit = false;
				itersSinceReinit = 0;
				
				System.out.print("done.\n\n");
				
			}// end reinitialization
			
			
			System.out.println("Iteration: " + t);
			for(int n=0; n<mgdmDecomp.nmgdm; n++){
				System.out.println("Number of voxels with a label change at level ["+n+"]: " + nswap[n] );
			}
			
			
			// * **************** * //
			// * TEST CONVERGENCE * //
			// * **************** * //
			
			
			
		} // end iteration loop
	
	}
	
	public double levelSetSpeed(int xyz, int lb, int nbr){
		
		//TODO: make 2d compatible
		float[][][] phi = mgdmDecomp.signedDistanceFunctionNeighborhood(xyz, lb);
		
		GdmDerivatives derivative = GdmDerivatives.computeDerivatives(phi, mgdmDecomp.rx, mgdmDecomp.ry, mgdmDecomp.rz);
		
		double speed =  mgdmForces.getForce(xyz, lb, nbr, derivative, 0, mgdmRepo, mgdmDecomp.isDebug(xyz));
		
		// check for potential instability.
		// and throw up a flag if its possible - this will signal a change in the step size
		if ( adaptStepSize ) {
			double absspeed = Math.abs(speed);
			if( absspeed > maxSpeed){
				maxSpeed = absspeed;
				double maxStepsize = maxStepSize(stepsize);
				// check to see if instability could be a problem [xa/t]
				if (maxSpeed > maxStepsize ) {
					instabilityWarning = true;
				}
			}
		}
		
		speed *= stepsize;

		return speed;
	}
	
	public final void updateStepSize() {
		updateStepSize(0.85);
	}
	
	public final double maxStepSize(double speed){
		double absSpeed = Math.abs(speed);
		
		if(absSpeed > stepEps){
//			System.out.println("absSpeed: " + absSpeed);
			return Math.min(mgdmDecomp.rz, Math.min(mgdmDecomp.rx, mgdmDecomp.ry)) / absSpeed;
		}else{
			return maxStep;
		}
	}
	
	public final void updateStepSize(double fraction) {
		System.out.print("\nstep size was: " + stepsize);
		double maxstepsize = maxStepSize(maxSpeed);
		stepsize = fraction * maxstepsize;
		System.out.print("; now is: " + stepsize + "\n\n");
	}
	
	//TODO: fill in later
	public boolean homeomorphicLabeling(int xyz){
		return true;
	}

}
