package edu.jhu.ece.iacl.algorithms.MGDM;

import java.util.BitSet;
import java.util.Iterator;

import edu.jhmi.rad.medic.utilities.Numerics;

/**
 * Implements the narrow band for MGDM for more
 * efficient level set computations.  The narrow band contains
 * a subset of all pixels/voxels 
 * 
 * The <i>width</i> of the narrow band specifies the maximum
 *
 * 
 * @author John Bogovic
 * @version $Revision: $
 * @see MgdmDecomposition
 */
public class MgdmNarrowband {
	
	
	protected 	int[] 	id;			
	protected int[][] 	mgdmlabels; 	// MGDM's label maps
	protected float[][] mgdmfunctions; 	// MGDM's label maps
	protected int 		nmgdm;
	
	protected 	double	narrowBandDist=6.5;
	protected 	double	landmineDist=5.0;
	
	protected 	BitSet	landmines;
	protected 	int 	capacity;
	protected 	int 	update;
	protected 	int 	currentsize;
	
	// for debug 
	protected int 		debugXYZ;
	protected boolean 	debug = true;

	/**
	 * Default constructor.
	 */
	public MgdmNarrowband(){
	}
	
	/**
	 * Constructor that builds this narrow band from 
	 * an existing MGDM decomposition.
	 * 
	 * @param mgdmdecomp
	 * @see MgdmDecomposition
	 */
	public MgdmNarrowband(MgdmDecomposition mgdmdecomp){
		buildFromDecomposition(mgdmdecomp);
	}
	
	/**
	 * Constructor that builds this narrow band from 
	 * an existing MGDM decomposition, given a particular
	 * narrow band width.
	 * 
	 * @param mgdmdecomp
	 * @param narrowBandDist
	 * @see MgdmDecomposition
	 */
	public MgdmNarrowband(MgdmDecomposition mgdmdecomp, int narrowBandDist){
		this.narrowBandDist=narrowBandDist;
		buildFromDecomposition(mgdmdecomp);
	}
	
	private void initialize(int size, int increase, int nfuncs) {
		capacity = size;
		update = increase;
		currentsize = 0;
		nmgdm = nfuncs;

		id = new int[capacity];
		mgdmlabels = new int[capacity][nmgdm];
		mgdmfunctions = new float[capacity][nmgdm];
		landmines = new BitSet(Numerics.ceil(0.2f * size));
	}
	
	/**
	 * Builds this narrow band from an existing MGDM decomposition. 
	 * 
	 * @param mgdmdecomp the MGDMDecomposition
	 * @see MgdmDecomposition
	 */
	public void buildFromDecomposition(MgdmDecomposition mgdmdecomp){
		
		this.debugXYZ = mgdmdecomp.getDebugXYZ();
		this.nmgdm = mgdmdecomp.nmgdm;
		
		int size = 0;
		// first estimate the narrow band size
		
		MgdmDecomposition.MgdmDecompositionIterator it = mgdmdecomp.iterator();
		
		while(it.hasNext()) {
			
			int xyz = it.next();
			if (mgdmdecomp.getMask(xyz)){
				if (mgdmdecomp.getDistance(xyz, 0) < narrowBandDist) {
					size++;
				}
			}
		}
		
		if(id==null){
			initialize((int)Math.ceil(1.25*size),(int)Math.ceil(0.1*size), mgdmdecomp.nmgdm);
		}
		reset();
		landmines.clear();
		
		it = mgdmdecomp.iterator();
		while(it.hasNext()) {
			int xyz = it.next();
			
			if (mgdmdecomp.getMask(xyz)){
				if (mgdmdecomp.getDistance(xyz, 0) < narrowBandDist) {
					
					addPoint(xyz,mgdmdecomp);
					
					if (mgdmdecomp.getDistance(xyz, 0) >= landmineDist) {
						landmines.set(xyz, true);
					}
				}
			}
			
			
		}
		
		System.out.println("\nNarrowband updated.  Size: " + currentsize);
		
	}
	
	private final void addPoint(int xyz, MgdmDecomposition mgdmdecomp) {
		// check for size
		if (currentsize >= capacity - 1) {
			capacity += update;

			int[] oldid = id;
			int[][] oldlabels = mgdmlabels;
			float[][] oldfunctions = mgdmfunctions;

			id = new int[capacity];
			mgdmlabels = new int[capacity][nmgdm];
			mgdmfunctions = new float[capacity][nmgdm];

			for (int n = 0; n < currentsize; n++) {
				id[n] = oldid[n];
				for (int l = 0; l < nmgdm; l++) {
					mgdmlabels[n][l] = oldlabels[n][l];
					mgdmfunctions[n][l] = oldfunctions[n][l];
				}
			}

			oldid = null;
			oldlabels = null;
			oldfunctions = null;
		}

		// add the new point (use the MGDM variables)
		id[currentsize] = xyz;
		if (mgdmdecomp.getLabel(xyz, 0) == -1) {
			// System.out.println("UNKNOWN!!");
		}
		
		for (int l = 0; l < nmgdm; l++) {
			mgdmlabels[currentsize][l] 		= mgdmdecomp.getLabel(xyz, l ); 
			mgdmfunctions[currentsize][l] 	= mgdmdecomp.getDistance(xyz, l ); 
		}
		currentsize++;
	}
	
	public void setLabel(int xyz, int level, int value){
		mgdmlabels[xyz][level] = value;
	}
	public void setDistance(int xyz, int level, float value){
		mgdmfunctions[xyz][level] = value;
	}
	
	public void setLandmine(int xyz){
		landmines.set(xyz,true);
	}
	
	public boolean isLandmine(int xyz){
		return landmines.get(xyz);
	}
	
	public boolean isDebug(int xyz){
		return (xyz==debugXYZ);
	}
	
	public void reset() {
		currentsize = 0;
	}
	
	public void finalize() {
		capacity = -1;
		id = null;
		mgdmfunctions=null;
		mgdmlabels=null;
	}
	
	
	public int[] getMgdmLabels(int xyz){
		return mgdmlabels[xyz];
	}
	
	public float[] getMgdmFunctions(int xyz){
		return mgdmfunctions[xyz];
	}
	
	/**
	 * Returns an iterator for this narrow band.
	 * 
	 * @return the iterator
	 */
	public MgdmNarrowbandIterator iterator(){
		return new MgdmNarrowbandIterator();
	}
	
	public String printDecomposition(int xyz){
		
		String out = "";
		for(int n=0; n<nmgdm; n++){
			out += ("Label/Distance ["+n+"]: " + mgdmlabels[xyz][n] + "\t" + mgdmfunctions[xyz][n] + "\n") ;
		}
		
		return out;
	}
	/**
	 * An iterator for the MGDM narrow band. 
	 * 
	 * @author John Bogovic
	 *
	 */
	public class MgdmNarrowbandIterator implements Iterator<Integer>{
		
		protected int iterIdx;
		
		public MgdmNarrowbandIterator(){
			iterIdx=0;
		}
		
		@Override
		public boolean hasNext() {
			return (iterIdx<(currentsize-1));
		}

		@Override
		public Integer next() {
			iterIdx++;
			int out = id[iterIdx];		
			return out;
		}

		
		public void remove() {
			// do nothing
		}

		public void iteratorReset(){
			iterIdx=0;
		}
		
		public int nextNarrowBandIndex(){
			return iterIdx;
		}
		
	}

}
