package edu.jhu.ece.iacl.algorithms.MGDM;

import edu.jhmi.rad.medic.utilities.Numerics;

/**
 * Class that computes derivatives using finite differences.  Primarily used
 * for level set computations.  Works for images of either two or three spatial dimensions.
 * <p>
 * Typical usage involves use of the static {@link #computDerivatives(phi) computeDerivatives} methods
 * to return an instance of GdmDerivatives containing derivatives of interest.
 * 
 * @author John Bogovic
 * @version $Revision$
 */
public class GdmDerivatives {
	
	protected double Dmx; // backward difference x
	protected double Dmy; // backward difference y
	protected double Dmz; // backward difference z
	
	protected double Dpx; // forward difference  x
	protected double Dpy; // forward difference  y
	protected double Dpz; // forward difference  z
	
	protected double D0x; // central difference  x
	protected double D0y; // central difference  y
	protected double D0z; // central difference  z
	
	protected double Dxx; // second derivative d^2(f)/d(x^2)
	protected double Dyy; // second derivative d^2(f)/d(y^2)
	protected double Dxy; // second derivative d^2(f)/(dxdy)
	protected double Dxz; // second derivative d^2(f)/(dxdz)
	protected double Dyz; // second derivative d^2(f)/(dydz)
	protected double Dzz; // second derivative d^2(f)/(dzdz)
	
	protected double DeltaP=Double.NaN;
	protected double DeltaM=Double.NaN;
	
	protected double K;
	protected double GPhi;
	
	protected spatialDimensions dims;			// number of spatial dimensions
	
	public enum spatialDimensions {TWO, THREE};
	
	public GdmDerivatives(){
	}
	
	public double meanCurvature(){
		if(dims==spatialDimensions.TWO){
			K = (Dyy)*(D0x*D0x) + (Dxx)*(D0y*D0y)  - 2.0*(D0x*D0y*Dxy);
		}else if(dims==spatialDimensions.THREE){
			K = (Dyy + Dzz)*(D0x*D0x) 
			  + (Dxx + Dzz)*(D0y*D0y)
			  + (Dxx + Dyy)*(D0z*D0z) 
			  - 2.0*(D0x*D0y*Dxy + D0y*D0z*Dyz + D0z*D0x*Dxz);
		}
		return K;
	}
	
	/**
	 * Returns the magnitude of the gradient computed using central differences.
	 *  
	 * @return the magnitude of the gradient
	 */
	public double gradientMagnitude(){
		if(dims==spatialDimensions.TWO){
			GPhi = Math.sqrt((D0x*D0x) + (D0y*D0y));
		}else if(dims==spatialDimensions.THREE){
			GPhi = Math.sqrt((D0x*D0x) + (D0y*D0y) + (D0z*D0z));
		}
		return GPhi;
	}
	
	public double meanCurvatureFlow(boolean debug){
		gradientMagnitude();
		meanCurvature();
		if(debug){
			System.out.println("GPhi: " + GPhi);
			System.out.println("K  : " + K);
			System.out.println("Dxx: " + Dxx);
			System.out.println("Dxy: " + Dxy);
			System.out.println("Dxz: " + Dxz);
			System.out.println("Dyy: " + Dyy);
			System.out.println("Dyz: " + Dyz);
			System.out.println("Dzz: " + Dzz);
		}
		if (GPhi > 0.0000001) {
			K = K / (GPhi * GPhi * GPhi);
		} else {
			K = 0;
		}
		return GPhi*K;
	}
	
	public double meanCurvatureFlow(){
		return meanCurvatureFlow(false);
	}
	
	/**
	 * @return the backward derivative for x
	 */
	public double Dmx(){
		return Dmx;
	}
	
	/**
	 * 
	 * @return the backward derivative for y
	 */
	public double Dmy(){
		return Dmy;
	}
	
	/**
	 * 
	 * @return the backward derivative for z
	 */
	public double Dmz(){
		return Dmz;
	}
	
	/**
	 * @return the forward derivative for x
	 */
	public double Dpx(){
		return Dpx;
	}
	
	/**
	 * @return the forward derivative for y
	 */
	public double Dpy(){
		return Dpy;
	}
	
	/**
	 * @return the forward derivative for z
	 */
	public double Dpz(){
		return Dpz;
	}
	
	/**
	 * @return the central derivative for x
	 */
	public double D0x(){
		return D0x;
	}
	
	/**
	 * @return the central derivative for y
	 */
	public double D0y(){
		return D0y;
	}
	
	/**
	 * @return the central derivative for z
	 */
	public double D0z(){
		return D0z;
	}
	
	public double deltaP(){
		if(Double.isNaN(DeltaP)){
			if(dims==spatialDimensions.TWO){
				DeltaP = Math.sqrt(Numerics.square(Numerics.max(Dmx, 0.0))
						+ Numerics.square(Numerics.min(Dpx, 0.0))
						+ Numerics.square(Numerics.max(Dmy, 0.0))
						+ Numerics.square(Numerics.min(Dpy, 0.0)));
			}else if(dims==spatialDimensions.THREE){
				DeltaP = Math.sqrt(Numerics.square(Numerics.max(Dmx, 0.0))
						+ Numerics.square(Numerics.min(Dpx, 0.0))
						+ Numerics.square(Numerics.max(Dmy, 0.0))
						+ Numerics.square(Numerics.min(Dpy, 0.0))
						+ Numerics.square(Numerics.max(Dmz, 0.0))
						+ Numerics.square(Numerics.min(Dpz, 0.0)));
			}
		}
		return DeltaP;

	}
	public double deltaM(){
		if(Double.isNaN(DeltaM)){
			if(dims==spatialDimensions.TWO){
				DeltaM = Math.sqrt(Numerics.square(Numerics.max(Dpx, 0.0))
						+ Numerics.square(Numerics.min(Dmx, 0.0))
						+ Numerics.square(Numerics.max(Dpy, 0.0))
						+ Numerics.square(Numerics.min(Dmy, 0.0)));
			}else if(dims==spatialDimensions.THREE){
				DeltaM = Math.sqrt(Numerics.square(Numerics.max(Dpx, 0.0))
						+ Numerics.square(Numerics.min(Dmx, 0.0))
						+ Numerics.square(Numerics.max(Dpy, 0.0))
						+ Numerics.square(Numerics.min(Dmy, 0.0))
						+ Numerics.square(Numerics.max(Dpz, 0.0))
						+ Numerics.square(Numerics.min(Dmz, 0.0)));

			}
		}
		return DeltaM;
	}

	
	public String toString(){
		
		System.out.println("Dmx: " + Dmx);
		System.out.println("D0x: " + D0x);
		System.out.println("Dpx: " + Dpx);
		
		System.out.println("D0y: " + D0y);
		System.out.println("Dmy: " + Dmy);
		System.out.println("Dpy: " + Dpy);
		
		System.out.println("D0z: " + D0z);
		System.out.println("Dmz: " + Dmz);
		System.out.println("Dpz: " + Dpz);
		
		System.out.println("DeltaM: " + DeltaM);
		System.out.println("DeltaP: " + DeltaP);
		
		System.out.println("Dxx: " + Dxx);
		System.out.println("Dxy: " + Dxy);
		System.out.println("Dxz: " + Dxz);
		System.out.println("Dyy: " + Dyy);
		System.out.println("Dyz: " + Dyz);
		System.out.println("Dzz: " + Dzz);
		
		return "";
	}

	/**
	 * Computes derivatives for a 2D neighborhood and returns 
	 * a GdmDerivatives object containing the results of the computations.
	 * Assumes sample spacing of 1 in both x and y. 
	 * 
	 * @param phi the 2D neighborhood
	 * @return the GdmDerivatives object
	 */
	public static GdmDerivatives computeDerivatives(float[][] phi){
		return computeDerivatives(phi,1f,1f);
	}
	
	/**
	 * Computes derivatives for a 3D neighborhood and returns 
	 * a GdmDerivatives object containing the results of the computations.
	 * Assumes sample spacing of 1 in both x, y, and z. 
	 * 
	 * @param phi the 3D neighborhood
	 * @return the GdmDerivatives object
	 */
	public static GdmDerivatives computeDerivatives(float[][][] phi){
		return computeDerivatives(phi,1f,1f,1f);
	}
	
	/**
	 * Computes derivatives for a 2D neighborhood with specified sample spacing and returns 
	 * a GdmDerivatives object containing the results of the computations. 
	 * 
	 * @param phi the 2D neighborhood
	 * @param rx the sample spacing in x
	 * @param ry the sample spacing in y
	 * @return the GdmDerivatives object
	 */
	public static GdmDerivatives computeDerivatives(float[][] phi, float rx, float ry){
	
		GdmDerivatives derivs = new GdmDerivatives();
		derivs.dims = spatialDimensions.TWO;
		
		derivs.Dmx = (phi[1][1] - phi[0][1])/rx;
		derivs.Dmy = (phi[1][1] - phi[1][0])/ry;

		derivs.Dpx = (phi[2][1] - phi[1][1])/rx;
		derivs.Dpy = (phi[1][2] - phi[1][1])/ry;

		derivs.D0x = (phi[2][1] - phi[0][1]) / (2.0*rx);
		derivs.D0y = (phi[1][2] - phi[1][0]) / (2.0*ry);

		derivs.Dxx = (phi[0][1] + phi[2][1]- 2.0 * phi[1][1])/(rx*rx);
		derivs.Dyy = (phi[1][0] + phi[1][2] - 2.0 * phi[1][1])/(ry*ry);
		derivs.Dxy = (phi[0][0] + phi[2][2] - phi[0][2] - phi[2][0]) / (4.0*rx*ry);
		
		return derivs;
	}
	
	/**
	 * Computes derivatives for a 3D neighborhood with specified sample spacing and returns 
	 * a GdmDerivatives object containing the results of the computations. 
	 * 
	 * @param phi the 3D neighborhood
	 * @param rx the sample spacing in x
	 * @param ry the sample spacing in y
	 * @param rz the sample spacing in z
	 * @return the GdmDerivatives object
	 */
	public static GdmDerivatives computeDerivatives(float[][][] phi, float rx, float ry, float rz){
		
		GdmDerivatives derivs = new GdmDerivatives();
		derivs.dims = spatialDimensions.THREE;
		
		derivs.Dmx = (phi[1][1][1] - phi[0][1][1]) / rx;
		derivs.Dmy = (phi[1][1][1] - phi[1][0][1]) / ry;
		derivs.Dmz = (phi[1][1][1] - phi[1][1][0]) / rz;

		derivs.Dpx = (phi[2][1][1] - phi[1][1][1]) / rx;
		derivs.Dpy = (phi[1][2][1] - phi[1][1][1]) / ry;
		derivs.Dpz = (phi[1][1][2] - phi[1][1][1]) / rz;

		derivs.D0x = (phi[2][1][1] - phi[0][1][1]) / (2.0 * rx);
		derivs.D0y = (phi[1][2][1] - phi[1][0][1]) / (2.0 * ry);
		derivs.D0z = (phi[1][1][2] - phi[1][1][0]) / (2.0 * rz);
		
		derivs.Dxx = (phi[0][1][1] + phi[2][1][1] - 2.0 * phi[1][1][1]) / (rx * rx);
		derivs.Dyy = (phi[1][0][1] + phi[1][2][1] - 2.0 * phi[1][1][1]) / (ry * ry);
		derivs.Dzz = (phi[1][1][0] + phi[1][1][2] - 2.0 * phi[1][1][1]) / (rz * rz);

		derivs.Dxy = (phi[0][0][1] + phi[2][2][1] - phi[0][2][1] - phi[2][0][1]) / (4.0 * rx * ry);
		derivs.Dyz = (phi[1][0][0] + phi[1][2][2] - phi[1][0][2] - phi[1][2][0]) / (4.0 * ry * rz);
		derivs.Dxz = (phi[0][1][0] + phi[2][1][2] - phi[2][1][0] - phi[0][1][2]) / (4.0 * rx * rz);

		return derivs;
	}
	
}
