package edu.jhu.ece.iacl.algorithms.MGDM.forces;

import edu.jhu.ece.iacl.algorithms.MGDM.GdmDerivatives;
import edu.jhu.ece.iacl.algorithms.MGDM.MgdmDecomposition;
import edu.jhu.ece.iacl.algorithms.MGDM.forces.MgdmConstants.ForceType;

/**
 * This abstract class represents a boundary force used by MGDM.
 * 
 * @author John Bogovic
 * @author Bhaskar Kishore
 */
public abstract class MgdmForce implements Cloneable{
	protected	int 			m_ID;			// Force ID
	protected	ForceType		m_ForceType;	// The type of force
	protected	double			m_Weight;		// The weight for this force
	protected	MgdmBoundary	m_Boundary;		// The boundary this object is associated with	
	protected	String[]		m_RequiredData;	// A collection of string tags indicating what data is needed by this force object
	
	protected	String[]		parameters;
	
	/**
	 * Constructor for Mgdm Force
	 * @param id		The unique id assigned to this force
	 * @param weight	The weight term for this object
	 * @param type		The type of force
	 * @param boundary  The boundary this force acts on
	 */
	public MgdmForce(ForceType type, MgdmBoundary boundary, double weight, int id) {
		m_ID		= id;
		m_Weight	= weight;
		m_ForceType	= type;
		m_Boundary = boundary;
	}
	
	/**
	 * Constructor for Mgdm Force
	 * @param id		The unique id assigned to this force
	 * @param weight	The weight term for this object
	 * @param type		The type of force
	 */
	public MgdmForce(ForceType type, double weight, int id) {
		m_ID		= id;
		m_Weight	= weight;
		m_ForceType	= type;
		m_Boundary	= (MgdmBoundary) MgdmConstants.ANYBOUNDARY.clone();
	}
	
	/**
	 * Default  Constructor for Mgdm Force
	 * @see	ForceType
	 * @see	MgdmConstants
	 */
	public MgdmForce()	{
		m_ID		= MgdmConstants.INVALID;
		m_ForceType	= ForceType.CUSTOM;
		m_Boundary = MgdmConstants.NOBOUNDARY;
	}
	
	/**
	 * An initialization function used to cache data from the repository instead of
	 * calling the repository repeatedly in the getForce function
	 * @param repo	The active Mgdm repository
	 * @see	MgdmDataRepository
	 */
	public abstract void initForce(MgdmDataRepository repo);
	
	
	/**
	 * Returns and object whose and whose initializeRepoForForce will be called
	 * before Mgdm evolution begins and whose updateRepoForForce method will be called at reinitialization.
	 * 
	 *  Defaults to returning a null object (no initializing or updating needed by this force).
	 *  Subclasses can override this method if additional functionality is needed.  See
	 *  @see MgdmForceIntensity
	 */
	public MgdmForceInitializerUpdater getInitializerUpdater(){
		return null;
	}
	
	/**
	 * Called by mgdm to compute extrinsic forces
	 * @param xyz	The array index
	 * @param lbl 	The current label
	 * @param nbr	The competing neighbor
	 * @param dr	The object containing the derivatives
	 * @param repo	The active Mgdm repository
	 * @param flipValue	A negative value indicates that boundary indices had to be flipped
	 * @return		The force computed. Double.NaN is returned if the force cannot be computed.
	 * @see	MgdmDerivatives
	 * @see	MgdmDataRepository
	 */
	public abstract double getForce(int xyz, int lbl, int nbr, GdmDerivatives dr, int iteration, MgdmDataRepository repo, float flipValue, boolean debug);
	
	/**
	 * Called by mgdm to compute extrinsic forces
	 * @param xyz	The array index
	 * @param lbl 	The current label
	 * @param nbr	The competing neighbor
	 * @param dr	The object containing the derivatives
	 * @param repo	The active Mgdm repository
	 * @param flipValue	A negative value indicates that boundary indices had to be flipped
	 * @return		The force computed. Double.NaN is returned if the force cannot be computed.
	 * @see	MgdmDerivatives
	 * @see	MgdmDataRepository
	 */
	public double getForce(int xyz, int lbl, int nbr, GdmDerivatives dr, int iteration, MgdmDataRepository repo, float flipValue){
		return getForce(xyz, lbl, nbr, dr,iteration,repo,flipValue,false);
	}
	
	
	/**
	 * Set the weight associated with this force object
	 * @param weight	The new weight
	 */
	public void setWeight(double weight) {
		m_Weight = weight;
	}
	
	/**
	 * Gets the weight associated with this force object
	 * @return	The current weight
	 */
	public double getWeight() {
		return m_Weight;
	}
	
	/**
	 * Sets the type of force this object represents
	 * @param type	The new type of force
	 */
	public void setForceType(ForceType type) {
		m_ForceType = type;
	}
	
	/**
	 * Gets the type of the force represented by this object
	 * @return	The type of the force
	 */
	public ForceType getForceType() {
		return m_ForceType;
	}
	
	/**
	 *  Sets the unique ID for this force object
	 * @param id	The ID value
	 */
	public void setID(int id) {
		m_ID = id;
	}
	
	/**
	 * Gets the unique ID for this force object
	 * @return	The ID value
	 */
	public int getID() {
		return m_ID;
	}
	
	/**
	 * Sets the boundary for which this force is to be associated with
	 * @param boundary	The new boundary to be associated with
	 */
	public void setBoundary(MgdmBoundary boundary) {
		m_Boundary.m_b1 = boundary.m_b1;
		m_Boundary.m_b2 = boundary.m_b2;
	}
	
	/**
	 * Gets the boundary which the force object is associated with
	 * @return	The boundary
	 */
	public MgdmBoundary getBoundary() {
		return m_Boundary;
	}
	
	/**
	 * Option : Gets a string array of the required data this object needs
	 * @return	String array
	 */
	public String [] getRequiredData() {
		return m_RequiredData;
	}
	
	/**
	 * Sets the required data for this object
	 * @param reqData	The new String array hold tags for required data
	 */
	
	protected void setRequiredData(String [] reqData) {
		m_RequiredData = reqData;
	}
	
	/**
	 * A generic method to set parameter values of a subclass.
	 * @param params String representations of the parameters
	 */
	public abstract void fromParameters(String[] params);
	
	public String[] getParameters(){
		return parameters;
	}
	
	
	/**
	 * Shallow copies fields from this object into the input MgdmForce
	 * @param f
	 */
	public abstract MgdmForce clone();
	
	public String toString() {
		String rep = "Force (Type - " + m_ForceType.toString() + ", Weight: " + m_Weight + 
			", Boundary <" + m_Boundary+  ">)";
		return rep;
	}
	
	/**
	 * Subclasses of this abstract class are called to add and update information used by an MgdmForce to the MgdmDataRepository. 
	 * 
	 * @author John Bogovic
	 * @see MgdmForceIntensity
	 * @see MgdmIntensityForceInitializer
	 *
	 */
	public abstract class MgdmForceInitializerUpdater{
		
		public abstract void initializeRepoForForce(MgdmDecomposition mgdmDecomp, MgdmDataRepository repo);
		
		public abstract void updateRepoForForce(MgdmDecomposition mgdmDecomp, MgdmDataRepository repo);
		
	}
}
