package edu.jhu.ece.iacl.algorithms.MGDM.forces;


import java.util.*;
import java.io.*;

import javax.vecmath.*;

import edu.jhu.ece.iacl.algorithms.MGDM.*;
import edu.jhu.ece.iacl.algorithms.MGDM.forces.MgdmForce.MgdmForceInitializerUpdater;

/**
 * Structure for representing the mapping from boundary pairs to their associated forces.
 * 
 * 
 * @author John Bogovic
 * @author Bhaskar Kishore
 *
 */
public class MgdmForceSpec {
	private HashMap<MgdmBoundary,List<MgdmForce>>	m_ForceMap;				// Map object translating boundaries --> list of forces
	private	HashMap<String ,List<MgdmForce>>		m_ForceProfiles;		// Map object storing name but unassociated list of forces 
	private HashSet<MgdmForceInitializerUpdater>	m_ForceInitializers;	
	
	public MgdmForceSpec(){
		m_ForceMap		= new HashMap<MgdmBoundary,List<MgdmForce>>();
		m_ForceProfiles	= new HashMap<String ,List<MgdmForce>>();
	}
	
	public MgdmForceSpec(String f){
		this(new File(f));
	}
	
	public MgdmForceSpec(File f){
		MgdmForceSpec tmp = MgdmForceReaderWriter.getInstance().readObject(f);
		this.m_ForceInitializers 	= tmp.m_ForceInitializers;
		this.m_ForceProfiles 		= tmp.m_ForceProfiles;
		this.m_ForceMap 			= tmp.m_ForceMap;
	}
	
	/**
	 * Adds a list of forces as a profile to this force spec object
	 * @param name		The name of the profile, must be unique
	 * @param forces	The list of forces in the profile
	 */
	public void addProfile(String name, List<MgdmForce> forces) {
		if(name == null || forces == null || name.isEmpty())
			return;
		if(!m_ForceProfiles.containsKey(name))
			m_ForceProfiles.put(name, forces);
	}
	
	/**
	 * Creates a new profile of forces
	 * @param name		The name of the profile to create
	 */
	public void createProfile(String name) {
		if(name == null || name.isEmpty())
			return;
		if(!m_ForceProfiles.containsKey(name))
			m_ForceProfiles.put(name, new ArrayList<MgdmForce>());
	}
	
	/**
	 * Appends a new force to an existing profile.
	 * @param name		The name of the profile
	 * @param force		The force to append.
	 */
	public void appendForceToProfile(String name, MgdmForce force) {
		if(name == ""  || force == null || name.isEmpty())
			return;
		if(m_ForceProfiles.containsKey(name))
			m_ForceProfiles.get(name).add(force);
	}
	
	/**
	 * Adds a MgdmForceInitializerUpdater object whose and whose initializeRepoForForce will be called
	 * before Mgdm evolution begins and whose updateRepoForForce method will be called at reinitialization.
	 * 
	 * @param iniUper the MgdmForceInitializerUpdater object
	 */
	public void addInitializerUpdater(MgdmForceInitializerUpdater iniUper){
		if(m_ForceInitializers==null){
			m_ForceInitializers = new HashSet<MgdmForceInitializerUpdater>();
		}
		if(iniUper!=null){
			if(m_ForceInitializers.add(iniUper)){
				System.out.println("Added force updater: " + iniUper);
				System.out.println("Now have a total of: " + m_ForceInitializers.size());
			}
		}
	}
	
	// TODO Should we be sorting the boundary indexes?  
	//	maybe ... HashMap is "smart" and can fetch the forces for points
	//	quickly even if they're not sorted (right?).
	//  Are there any other advantages to sorting? - JB
	//  -- Well that depends on the key implementation of the Point2i class,
	//	i was suggesting sorting to ensure we bring some consistency - BK
	/**
	 * Adds a force 
	 * @param bnd the force
	 * @param force the boundary
	 */
	public void addForce(MgdmBoundary bnd, MgdmForce force){
		if(!m_ForceMap.containsKey(bnd)){
			m_ForceMap.put(bnd, new ArrayList<MgdmForce>());
		}
		// TODO: is this cloning / changing of boundary info worth it?  maybe will make more
		// sense after "Profiling"
		// -- Would make sense to clone it if we plan on allowing run time changes to force
		// objects. Wouldn't want all copies to get updated if only 1 has to change - BK
		MgdmForce addme = force.clone();
		addme.setBoundary(bnd);
		m_ForceMap.get(bnd).add(addme);
		
		// add initializers / updaters that the force requests
		addInitializerUpdater(addme.getInitializerUpdater());
	}
	
	/**
	 * Adds a force to a boundary specified by object indexes
	 * 
	 * @param obj1 the first object
	 * @param obj2 the second object
	 * @param force the force
	 */
	public void addForce(int obj1, int obj2, MgdmForce force){
		addForce(new MgdmBoundary(obj1,obj2),force);
	}
	
	public void setForce(MgdmBoundary bnd, List<MgdmForce> forcelist){
		m_ForceMap.put(bnd, forcelist);
	}
	
	/**
	 * Computes the change in the MGDM distance functions that is
	 * used to optimize the boundary locations during evolution. 
	 * 
	 * @param xyz the index of spatial position
	 * @param lbl the current label 
	 * @param nbr the competing label
	 * @param dr the derivatives computed at that point
	 * @param iteration the iteration number
	 * @param repo a repository containing the volumes/data necessary for all forces
	 * @return the change in the distance function
	 */
	public double getForce(int xyz, int lbl, int nbr, GdmDerivatives dr, int iteration, MgdmDataRepository repo){
		return getForce(xyz,lbl,nbr,dr,iteration,repo,false);
	}
	
	/**
	 * 
	 * @param xyz the index of spatial position
	 * @param lbl the current label 
	 * @param nbr the competing label
	 * @param dr the derivatives computed at that point
	 * @param iteration the iteration number
	 * @param repo a repository containing the volumes/data necessary for all forces
	 * @param debug whether debug information be printed
	 * @return the change in the distance function
	 */
	public double getForce(int xyz, int lbl, int nbr, GdmDerivatives dr, int iteration, MgdmDataRepository repo, boolean debug){
		double f = 0.0;
		if(lbl == nbr)	// TODO Should never happen (Check why?)
			return f;
		
		float flipValue = 1.0f;
		MgdmBoundary bnd = null;
		if(lbl < nbr)
			bnd = new MgdmBoundary(lbl,nbr);
		else {
			bnd = new MgdmBoundary(nbr, lbl);
			flipValue = -1.0f;
		}
		List<MgdmForce> forcelist = m_ForceMap.get(bnd);
		if(forcelist == null) {
			//System.out.println("Boundary was not found!!! <" + lbl + ", " + nbr + ">");
			return 0;
		}
		
		if(debug){
			System.out.println("Computing speed for boundary: " + bnd);
			System.out.println("Found " + forcelist.size() + " forces");
		}
		
		Iterator<MgdmForce> it = forcelist.iterator();
		while(it.hasNext()) {
			MgdmForce force = it.next();
			
			double compForce = force.getForce(xyz, lbl, nbr, dr, iteration, repo, flipValue, debug);
			if(compForce == Double.NaN) {
				continue;
			}
			
			if(debug){
				System.out.println("Value of " + force + " at " + xyz + " was " + compForce);
				System.out.println("\n");
			}
			
			f += compForce;
		}
		
//		System.out.println(" ");
		
		return f;
	}
	
	
	/**
	 * Converts from the old method of storing force weights to the new
	 * @param weights - a MgdmBoundaryForces object
	 * @return the equivalent MgdmForceSpec object
	 */
//	public static MgdmForceSpec MgdmWeights2Spec(MgdmBoundaryForces weights){
//		MgdmForceSpec spec = new MgdmForceSpec();
//		
//		Set<Point2i> bndset = weights.getBoundarySet();
//		Iterator<Point2i> it = bndset.iterator();
//		System.out.println("num bnds: " + bndset.size());
//		while(it.hasNext()){ 
//			Point2i pbnd = it.next();
//			MgdmBoundary thisbnd = new MgdmBoundary(pbnd.x, pbnd.y);
//			System.out.println("thisbnd: " + thisbnd);
//			
//			if(weights.hasCurvatureWeight()){
//				double cw  = weights.getCurvatureWeight(thisbnd.m_b2, thisbnd.m_b1);
//				if(cw!=0){
//					spec.addForce(thisbnd, new MgdmForceCurvature(cw));
//				}
//			}
//			
//			double iw = 0;
//			for(int ni = 0; ni<weights.numIntensity(); ni++){
//				iw= weights.getIntensityWeight(thisbnd.m_b2, thisbnd.m_b1, ni);
//				if(iw!=0){
//					spec.addForce(thisbnd, new MgdmForceIntensity(iw));
//				}
//			}
//			
//			double rw = 0;
//			double rt = 0;
//			for(int nr = 0; nr<weights.numRegion(); nr++){
//				rw = weights.getRegionWeight(thisbnd.m_b2, thisbnd.m_b1, nr);
//				rt = weights.getRegionThresh(thisbnd.m_b2, thisbnd.m_b1, nr);
//				if(rw!=0){
//					spec.addForce(thisbnd, new MgdmForceRegion(rw,nr,rt));
//				}
//			}
//			
//			double fw = 0;
//			for(int nf = 0; nf<weights.numField(); nf++){
//				fw = weights.getFieldWeight(thisbnd.m_b2, thisbnd.m_b1, nf);
//				if(fw!=0){
//					spec.addForce(thisbnd, new MgdmForceField(fw,nf));
//				}
//			}
//			
//			if(weights.hasPressureWeight()){
//				System.out.println("tb: " + thisbnd);
//				double pw = weights.getPressureWeight(thisbnd.m_b2, thisbnd.m_b1);
//				if(pw!=0){
//					spec.addForce(thisbnd, new MgdmForcePressure(pw));
//					System.out.println("added pressure force");
//				}
//			}
//			System.out.println(spec);
//		}
//		
//		return spec;		
//	}
	
	public void initializeForces(MgdmDecomposition mgdmDecomp, MgdmDataRepository repo){
		
		Iterator<MgdmForceInitializerUpdater> it = m_ForceInitializers.iterator();
		
		while(it.hasNext()){
			it.next().initializeRepoForForce(mgdmDecomp, repo);
		}
	}
	
	public void updateForces(MgdmDecomposition mgdmDecomp, MgdmDataRepository repo){
		
		Iterator<MgdmForceInitializerUpdater> it = m_ForceInitializers.iterator();
		
		while(it.hasNext()){
			it.next().updateRepoForForce(mgdmDecomp, repo); 
		}
	}
	
	
	
	public String toString(){
//		Set<Point2i> ks = m_ForceMap.keySet();
//		String s = "";
//		
//		Iterator<Point2i> ksit = ks.iterator();
//		int i = 0;
//		while(ksit.hasNext()){
//			Point2i thiskey = ksit.next();
//			s+= "{" + thiskey + ", " + m_ForceMap.get(thiskey) + "}\n";
//			
//			i++;
//		}
//		
//		return ks.toString();
		
		return m_ForceMap.toString();
	}
}
