package edu.jhu.ece.iacl.algorithms.MGDM;

import java.util.*;

import edu.jhmi.rad.medic.libraries.ObjectProcessing;
import edu.jhmi.rad.medic.structures.BinaryHeap2D;
import edu.jhmi.rad.medic.utilities.Numerics;
import edu.jhu.ece.iacl.utility.ArrayReshape;

/**
 * This class implements the MGDM level set decomposition for 2D and 3D images.
 * 
 * Given a discrete labeling, a "joint" fast marching simultaneously computes the
 * nearest neighbors and distance functions for each object.
 * 
 * @author John Bogovic
 * @version $Revision: $
 * 
 */
public class MgdmDecomposition {
	
	public final static float UNKNOWN = -1f;
	public final static int   EMPTY = -1;
	protected final static double SQR2 = Math.sqrt(2.0);
	protected final static double SQR3 = Math.sqrt(3.0);
	
	protected int[]   	dimSize;		// size of each spatial dimension
	protected float[] 	dimRes;		// grid spacing for each spatial dimension
	protected int 		N; 				// total number of spatial points
	protected boolean[] mask; 		// masking regions not used in computations
	protected int[] 	dimsize; 		// images dimensions
	protected float[] 	res;			// repeated for fast marching
	
	protected int[] segmentation; // MGDM's segmentation
	protected int[][] mgdmlabels; // MGDM's label maps
	protected int[] otherlabels; 		// MGDM's segmentation
	protected float[][] mgdmfunctions; // MGDM's pseudo level set mgdmfunctions
	protected float[]   tempObjLvlset; // temporary single-object levelset
	
	protected int nmgdm=3;	// number of mgdm functions to store
	protected int nobj;   // number of objects
	
	public int 		nx, 	ny,		nz;		// image spatial dimensions
	public int		nxPad, 	nyPad,  nzPad;
	public float  	rx=1f, 	ry=1f, 	rz=1f; 	// image resolutions
	
	protected byte[] dimoff = new byte[] { 3, 3, 2, 2, 1, 1 };
	protected int[] xoff;
	protected int[] yoff;
	protected int[] zoff;
	protected float[] resoff;
	
	protected int[] objLabel; 	// list of original labels
	
	protected BinaryHeap2D heap; // the heap used in fast marching
	
	
	
	// for debug 
	protected int 		debugX = -1;
	protected int 		debugY = -1;
	protected int 		debugZ = -1;
	protected int 		debugXYZ;
	protected boolean 	debug = true;
	
	/**
	 * Default constructor
	 */
	public MgdmDecomposition(){
	}
	
	
	/**
	 * Constructor for a 2D Mgdm decomposition/evolution
	 * 
	 * @param segvol initial segmentation
	 * @param nmgdm the number of MGDM distance functions to store
	 */
	public MgdmDecomposition(int[][] segvol, int nmgdm){
		
		this.nmgdm=nmgdm;
		
		nx = segvol.length;
		ny = segvol[0].length;
		nz = 0;
		
		int[] segPadReshape = ArrayReshape.reshapePad(segvol, 2, 2, 2, 2, true);
		
		nxPad = nx+4;
		nyPad = ny+4;
		nzPad = 0;

	
		fastMarchingInitializationFromSegmentation(segPadReshape);
	}
	
	/**
	 * Constructor for a 3D Mgdm decomposition/evolution
	 * 
	 * @param segvol initial segmentation
	 * @param nmgdm the number of MGDM distance functions to store
	 */
	public MgdmDecomposition(int[][][] segvol, int nmgdm){
		this.nmgdm=nmgdm;
		
		nx = segvol.length;
		ny = segvol[0].length;
		nz = segvol[0][0].length;
		
		int[] segPadReshape = ArrayReshape.reshapePad(segvol, 2, 2, 2, 2, 2, 2, true);
		
		System.out.println("segPadReshape: " + segPadReshape.length);
		
		nxPad = nx+4;
		nyPad = ny+4;
		nzPad = nz+4;
		
		fastMarchingInitializationFromSegmentation(segPadReshape);
	}
	
	/**
	 * Constructor for a 2D Mgdm decomposition/evolution
	 * @param seg initial segmentation
	 */
	public MgdmDecomposition(int[][] segvol){
		this(segvol,2);
	}
	
	/**
	 * Constructor for a 3D Mgdm decomposition/evolution
	 * @param seg initial segmentation
	 */
	public MgdmDecomposition(int[][][] segvol){
		this(segvol,3);
	}
	
	/**
	 * 
	 * @param rx the sample spacing in x
	 * @param ry the sample spacing in y
	 */
	public void setResolutions(float rx, float ry){
		this.rx=rx;
		this.ry=ry;
	}
	
	/**
	 * 
	 * @param rx the sample spacing in x
	 * @param ry the sample spacing in y
	 * @param rz the sample spacing in z
	 */
	public void setResolutions(float rx, float ry, float rz){
		this.rx=rx;
		this.ry=ry;
		this.rz=rz;
	}
	
	public final void toOriginalLabels() {
		for (int xyz = 0; xyz < N; xyz++) {
			if (segmentation[xyz] < 0)
				continue;

			segmentation[xyz] = objLabel[segmentation[xyz]];

			for (int n = 0; n < nmgdm; n++) {
				if (mgdmlabels[n][xyz] < 0)
					continue;

				mgdmlabels[n][xyz] = objLabel[mgdmlabels[n][xyz]];
			}
		}
	}
	
	public int getNumLabels(){
		return nobj;
	}
	
	public boolean getMask(int xyz){
		return mask[xyz];
	}
	
	public float getDistance(int xyz, int n){
		if(n<nmgdm){
			return mgdmfunctions[n][xyz];
		}else{
			return Float.NaN;
		}
	}
	
	public int getLabel(int xyz, int n){
		if(n<nmgdm){
			return mgdmlabels[n][xyz];
		}else{
			return -2;
		}
	}
	
	public int getLastNeighbor(int xyz){
		return otherlabels[xyz];
	}
	
	public void setSegmentation(int xyz, int value){
		segmentation[xyz]=value;
	}
	
	/**
	 * Copies the values of the mgdm label and distance functions from the narrow band into this object.
	 * 
	 * @param nb the narrow band.
	 */
	public void updateFromNarrowBand(MgdmNarrowband nb){
		
		MgdmNarrowband.MgdmNarrowbandIterator it = nb.iterator();
		
		it.iteratorReset();
		while(it.hasNext()){
			int xyz 		= it.next();
			int nbidx  		= it.nextNarrowBandIndex();
			
			int[] lbls  	= nb.getMgdmLabels(nbidx);
			float[] fncs  	= nb.getMgdmFunctions(nbidx);
			
			for(int n=0; n<nmgdm; n++){
				
				mgdmlabels[n][xyz] 		= lbls[n];
				mgdmfunctions[n][xyz]	= fncs[n];
				
			}
			
		}
		
	}
	
	public final int[][][] exportAllLabels2d(){
		toOriginalLabels();
		int[][][] labelsout = new int[nx][ny][nmgdm+1];
		
		for(int n=0; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
				int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
				
				labelsout[x-2][y-2][n]=mgdmlabels[n][xyz];
				
			}
		}
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
			int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2][nmgdm]=otherlabels[xyz];
		}
		
		return labelsout;
	}
	
	public final int[][] exportSegmentation2d(){
		toOriginalLabels();
		int[][] labelsout = new int[nx][ny];
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
			int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2]=segmentation[xyz];
		}
		
		return labelsout;
	}
	
	public final int[][][] exportNeighbors2d(){
		toOriginalLabels();
		int[][][] labelsout = new int[nx][ny][nmgdm];
		
		for(int n=1; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
				int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
				
				labelsout[x-2][y-2][n-1]=mgdmlabels[n][xyz];
				
			}
		}
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
			int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2][nmgdm-1]=otherlabels[xyz];
		}
		
		return labelsout;
	}
	

	public final float[][][] exportDistances2d(){
		toOriginalLabels();
		float[][][] labelsout = new float[nx][ny][nmgdm];
		
		for(int n=0; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++){
				int xyz = ArrayReshape.coordsToIndex(x, y, 0, nxPad, nyPad, nzPad);
				labelsout[x-2][y-2][n]=mgdmfunctions[n][xyz];
			}
		}
		
		return labelsout;
	}
	
	public final int[][][][] exportAllLabels3d(){
		toOriginalLabels();
		int[][][][] labelsout = new int[nx][ny][nz][nmgdm+1];
		
		for(int n=0; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
				int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
				
				labelsout[x-2][y-2][z-2][n]=mgdmlabels[n][xyz];
				
			}
		}
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
			int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2][z-2][nmgdm]=otherlabels[xyz];
		}
		
		return labelsout;
	}
	
	public final int[][][] exportSegmentation3d(){
		toOriginalLabels();
		int[][][] labelsout = new int[nx][ny][nz];
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
			int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2][z-2]=segmentation[xyz];
		}
		
		return labelsout;
	}
	
	public final int[][][][] exportNeighbors3d(){
		toOriginalLabels();
		int[][][][] labelsout = new int[nx][ny][nz][nmgdm];
		
		for(int n=1; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
				int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
				
				labelsout[x-2][y-2][z-2][n-1]=mgdmlabels[n][xyz];
				
			}
		}
		
		for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
			int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
			labelsout[x-2][y-2][z-2][nmgdm-1]=otherlabels[xyz];
		}
		
		return labelsout;
	}
	

	public final float[][][][] exportDistances3d(){
		toOriginalLabels();
		float[][][][] labelsout = new float[nx][ny][nz][nmgdm];
		
		for(int n=0; n<nmgdm; n++){
			for(int x=2; x<nx+2; x++)for(int y=2; y<ny+2; y++)for(int z=2; z<nz+2; z++){
				int xyz = ArrayReshape.coordsToIndex(x, y, z, nxPad, nyPad, nzPad);
				labelsout[x-2][y-2][z-2][n]=mgdmfunctions[n][xyz];
			}
		}
		
		return labelsout;
	}

	
//	public void exportAllLabels(){
//		
//	}
//	
//	public <T> T exportSegmentation();
//	
//	public <T> T exportNeighbors();
//	
//	public <T> T exportDistances();
	
	
	private void initOffset(){
		// initialize offset arrays
		if(nzPad!=0){
			xoff = new int[] { 0, 0, 0, 0, nzPad * nyPad, -nzPad * nyPad };
			yoff = new int[] { 0, 0, nzPad, -nzPad, 0, 0 };
			zoff = new int[] { 1, -1, 0, 0, 0, 0 };
			resoff = new float[] { rz, rz, ry, ry, rx, rx };
		}else{
			xoff = new int[] { 0, 0, nyPad, -nyPad };
			yoff = new int[] { 1, -1, 0, 0 };
			zoff = new int[] { 0,0,0,0};
			resoff = new float[] { ry, ry, rx, rx };
		}
	}
	
	private void initFunctions(){
		if(mgdmlabels==null){
			mgdmlabels = new int[nmgdm][N];
		}
		if(mgdmfunctions==null){
			mgdmfunctions = new float[nmgdm][N];
		}
		if(segmentation==null){
			segmentation = new int[N];
		}
		if(otherlabels==null){
			otherlabels = new int[N];
		}
	}
	
	/**
	 * 
	 * Returns the signed distance function (phi) for an object, reconstructed
	 * from the MGDM decomposition
	 * 
	 * 
	 * @param xyz the spatial index of the center point
	 * @param lb the object
	 * @return
	 */
	public final float[][][] signedDistanceFunctionNeighborhood(int xyz, int lb){
		float[][][] phi = new float[3][3][3];
		
		// do the center point first
		if (mgdmlabels[0][xyz] == lb)
			phi[1][1][1] = -mgdmfunctions[0][xyz];
		else
			phi[1][1][1] = 0.0f;
		for (int l = 0; l < nmgdm && mgdmlabels[l][xyz] != lb; l++) {
			phi[1][1][1] += mgdmfunctions[l][xyz];
		}
		
		// neighbors
		for (int i = -1; i <= 1; i++)	for (int j = -1; j <= 1; j++)for (int k = -1; k <= 1; k++){
			if (i * i > 0 || j * j > 0 || k * k > 0) {
				
				int xyzn = xyz + ArrayReshape.coordsToIndex(i,j,k,nxPad,nyPad,nzPad);
				
				// are we in the mask
				if (mask[xyzn]) { 
					if (mgdmlabels[0][xyzn] == lb){ // the object is the current label
						phi[i + 1][j + 1][k + 1] = -mgdmfunctions[0][xyzn];
					}else{ // the object is a neighbor
						phi[i + 1][j + 1][k + 1] = 0.0f;
					}
					
					// add to the sdf until we reach the object or run out of mgdm functions
					for (int l = 0; l < nmgdm && mgdmlabels[l][xyzn] != lb; l++) {
						phi[i + 1][j + 1][k + 1] += mgdmfunctions[l][xyzn];
					}
					
				} else {
					// filling in values outside the mask?? center value
					phi[i + 1][j + 1][k + 1] = phi[1][1][1];
				}
			}
		}
		
		return phi;
	}
	
	
	public final int fastMarchingInitializationFromSegmentation(int[] init, float rx, float ry, float rz) {
		return fastMarchingInitializationFromSegmentation(init,nmgdm,rx,ry,rz);
	}
	
	public final int fastMarchingInitializationFromSegmentation(int[] init) {
		return fastMarchingInitializationFromSegmentation(init,nmgdm,1f,1f,1f);
	}
	
	public final int fastMarchingInitializationFromSegmentation(int[] init, int nmgdm) {
		return fastMarchingInitializationFromSegmentation(init,nmgdm,1f,1f,1f);
	}
	
	
	protected void fastMarchHeap(byte[] processed, double maxMarchDist ){
		
		float[] nbdist = new float[xoff.length];
		boolean[] nbflag = new boolean[xoff.length];
		
		float maxdist = 0.0f;
		// grow the labels and functions
		while (heap.isNotEmpty() && maxdist <= maxMarchDist ) {
			// extract point with minimum distance
			float dist = heap.getFirst();
			int xyz = heap.getFirstId();
			int lb = heap.getFirstState();
			heap.removeFirst();

			// if more than nmgdm labels have been found already, this is done
			if (processed[xyz] >= nmgdm)
				continue;

			// if there is already a label for this object, this is done
			boolean done = false;
			for (int n = 0; n < processed[xyz]; n++)
				if (mgdmlabels[n][xyz] == lb)
					done = true;
			if (done)
				continue;

			// update the distance functions at the current level
			mgdmfunctions[processed[xyz]][xyz] = dist;
			mgdmlabels[processed[xyz]][xyz] = lb;
			processed[xyz]++; // update the current level

			// keep track of distance if stopping at the narrow band
			maxdist = dist;

			// find new neighbors
			for (int k = 0; k < xoff.length; k++) {
				int xyzn = xyz + xoff[k] + yoff[k] + zoff[k];

				if (mask[xyzn]) {
					// must be in outside the object or its processed neighborhood
					boolean isprocessed = false;
					if (segmentation[xyzn] == lb)
						isprocessed = true;
					else {
						for (int n = 0; n < processed[xyzn]; n++)
							if (mgdmlabels[n][xyzn] == lb)
								isprocessed = true;
					}

					if (!isprocessed) {
						// compute new distance based on processed neighbors for
						// the same object
						for (int l = 0; l < xoff.length; l++) {
							
							nbdist[l] = UNKNOWN;
							nbflag[l] = false;
							
							int xyznb = xyzn + xoff[l] + yoff[l] + zoff[l];
							// note that there is at most one value used here
							for (int n = 0; n < processed[xyznb]; n++) {
								if (mask[xyznb])
									if (mgdmlabels[n][xyznb] == lb) {
										nbdist[l] = mgdmfunctions[n][xyznb];
										nbflag[l] = true;
									}
							}
						}
						

						float newdist = minimumMarchingDistance(nbdist, nbflag, resoff);

						// add to the heap
						heap.addValue(newdist, xyzn, lb);
					}
				}
			}
		}

		// to create the MGDM functions, we need to copy the segmentation,forget
		// the last labels
		// and compute differences between distance functions
		// if(debug) System.out.println("transform into MGDM functions\n");

		for (int xyz = 0; xyz < N; xyz++)
			if (mask[xyz]) {
				// distance function difference
				for (int n = nmgdm - 1; n > 0; n--) {
					if (mgdmlabels[n][xyz] != EMPTY) {
						mgdmfunctions[n][xyz] = mgdmfunctions[n][xyz] - mgdmfunctions[n - 1][xyz];
					}
				}

				// label permutation
				otherlabels[xyz] = mgdmlabels[nmgdm - 1][xyz];
				for (int n = nmgdm - 1; n > 0; n--) {
					mgdmlabels[n][xyz] = mgdmlabels[n - 1][xyz];
				}
				mgdmlabels[0][xyz] = segmentation[xyz];

			}

	}
	
	/**
	 * Performs the joint fast marching on a discrete labeling, reshaped to 1D.
	 * @param init - the discrete labels
	 * @return zero if the fast marching was successful, non-zero if an error occured
	 */
	public final int fastMarchingInitializationFromSegmentation(int[] init, int nmgdm, float rx, float ry, float rz) {

		System.out.println("MGDM INITIALIZATION!!");
		
		this.nmgdm=nmgdm;
		
		this.rx=rx;
		this.ry=ry;
		this.rz=rz;
		
		N = init.length;
		
		if(nzPad==0){
			if(nxPad*nyPad != N ){
				System.err.println("Supplied dimensions (" +nx +" " + ny  + ") are incompatible with length of input label: " + N);
				return 1;
			}
		}else{
			if(nxPad*nyPad*nzPad != N ){
				System.err.println("Supplied dimensions (" +nx +" " + ny + " " + nz  + ") are incompatible with length of input label: " + N);
				return 1;
			}
		}
		
		System.out.println("Padded dimensions: (" + nxPad +"," + nyPad +"," + nzPad +")");
		System.out.println("Number of spatial locations: " + N);
		
		if(nzPad==0){
			// find labels in image
			objLabel = ObjectProcessing.listOrderedLabels(init, nxPad, nyPad);
			nobj = objLabel.length;
		}else{
			objLabel = ObjectProcessing.listOrderedLabels(init, nxPad, nyPad, nzPad);
			nobj = objLabel.length;
		}
		
		System.out.println("Found " + nobj + " objects");
		System.out.println(Arrays.toString(objLabel));
		
		// initialize heap
		if(heap==null){
			heap=new BinaryHeap2D(N,-1);
		}

		// default to mask everywhere except for pad of 2 pixels/voxels on each side of the image
		if(mask==null){
			mask = new boolean[N];
			int n = 0;
			for (int x = 0; x < nxPad; x++)for (int y = 0; y < nyPad; y++){
				if(nzPad!=0){
					for (int z = 0; z < nzPad; z++) {
						if (x > 1 && x < nxPad - 2 && y > 1 && y < nyPad - 2 && z > 1	&& z < nzPad - 2){
							mask[n] = true;
						}else{
							mask[n] = false;
						}
						n++;
					}
				}else{
					if (x > 1 && x < nxPad - 2 && y > 1 && y < nyPad - 2 ){
						mask[n] = true;
					}else{
						mask[n] = false;
					}
					n++;
				}
				
			}
			
		}
		
		// initialize mgdm functions
		initFunctions();
		initOffset();
		
		// initialize the quantities
		// this replaces the labels with their rank indices (useful for making
		// things fast)
		for (int xyz = 0; xyz < N; xyz++) {
			// mgdm functions
			for (int n = 0; n < nmgdm; n++) {
				mgdmfunctions[n][xyz] = UNKNOWN;
				mgdmlabels[n][xyz] = EMPTY;
			}
			// segmentation
			
			int nlb = EMPTY;

			nlb = Arrays.binarySearch(objLabel, init[xyz]);
//			for (int n = 0; n < nobj; n++) {
//				if (objLabel[n] == init[xyz]) {
//					nlb = n;
//					continue;
//				}
//			}

			segmentation[xyz] = nlb;
			otherlabels[xyz] = EMPTY;
		}

		// computation variables
		byte[] processed = new byte[N]; // note: using a byte

		
		// compute the neighboring labels and corresponding distance functions
		// (! not the MGDM functions !)
		
		heap.reset();
		// initialize the heap from boundaries
		for (int xyz = 0; xyz < N; xyz++){
			if (mask[xyz]) {
				processed[xyz] = 0;
				// search for boundaries
				for (int k = 0; k < xoff.length; k++) {
					int xyzn = xyz + xoff[k] + yoff[k] + zoff[k];
					if (segmentation[xyzn] != segmentation[xyz])
						if (mask[xyzn]) {
							// add to the heap
							heap.addValue(0.5f * resoff[k], xyzn,
									segmentation[xyz]);
						}
				}
			}
		}

		fastMarchHeap(processed, Double.POSITIVE_INFINITY);

		return 0;
	}
	
	
	public final void fastMarchingReinitialization(){
		
		resetIsosurfaceBoundary();
		
		// computation variables
		byte[] processed = new byte[N]; // note: using a byte


		// compute the neighboring labels and corresponding distance functions
		// (! not the MGDM functions !)
		heap.reset();

		// initialize the heap from boundaries
		for (int xyz = 0; xyz < N; xyz++)
			if (mask[xyz]) {
				processed[xyz] = 0;
				segmentation[xyz] = mgdmlabels[0][xyz];
				// search for boundaries
				for (int k = 0; k < xoff.length; k++) {
					int xyzn = xyz + xoff[k] + yoff[k] + zoff[k];
					if (mgdmlabels[0][xyzn] != mgdmlabels[0][xyz])
						if (mask[xyzn]) {

							// add to the heap with previous value
							heap.addValue(mgdmfunctions[0][xyzn], xyzn,
									mgdmlabels[0][xyz]);
						}
				}
			}

		for (int xyz = 0; xyz < N; xyz++) {
			if (mask[xyz]) {
				for (int n = 0; n < nmgdm; n++) {
					mgdmlabels[n][xyz] = EMPTY;
				}
				otherlabels[xyz] = EMPTY;
			}
		}

		fastMarchHeap(processed, Double.POSITIVE_INFINITY);

		return;
	}
	
	protected final void resetIsosurfaceBoundary() {

		float[] nbdist = new float[xoff.length];
		boolean[] nbflag = new boolean[xoff.length];
		boolean boundary;

		float[] tmp = new float[N];
		boolean[] processed = new boolean[N];
		for (int xyz = 0; xyz < N; xyz++)
			if (mask[xyz]) {

				boundary = false;
				for (int l = 0; l < xoff.length; l++) {
					nbdist[l] = UNKNOWN;
					nbflag[l] = false;

					int xyznb = xyz + xoff[l] + yoff[l] + zoff[l];
					if (mgdmlabels[0][xyznb] != mgdmlabels[0][xyz]
							&& mask[xyznb]) {
						// compute new distance based on processed neighbors for
						// the same object
						nbdist[l] = Numerics.abs(mgdmfunctions[0][xyznb]);
						nbflag[l] = true;
						boundary = true;
					}
				}
				if (boundary) {
					tmp[xyz] = isoSurfaceDistance(mgdmfunctions[0][xyz],nbdist, nbflag, resoff);
					processed[xyz] = true;
				}
			}
		// once all the new values are computed, copy into original GDM function
		// (sign is not important here)
		for (int xyz = 0; xyz <N; xyz++) {
			if (processed[xyz])
				mgdmfunctions[0][xyz] = tmp[xyz];
			else
				mgdmfunctions[0][xyz] = UNKNOWN;
		}

		return;
	}
	

	
	/**
	 * The iso-surface distance computation (assumes a 6D array with opposite
	 * coordinates stacked one after the other) (the input values are all
	 * positive, the flags are true only if the iso-surface crosses)
	 * 
	 * @param cur
	 * @param val
	 * @param flag
	 * @param res sample spacing in each dimension
	 */
	public static final float isoSurfaceDistance(float cur, float[] val,
			boolean[] flag, float[] res) {

		if (cur == 0)
			return 0;

		float s;
		double dist;
		float tmp;
		s = 0;
		dist = 0;

		for (int n = 0; n < val.length; n += 2) {
			float ressqr = res[n] * res[n];
			if (flag[n] && flag[n + 1]) { // if both are across the boundary
				// Take the largest distance (aka the boundary surface is closer
				// to CURRENT point)
				tmp = Numerics.max(val[n], val[n + 1]);
				s = cur / (cur + tmp);
				dist += 1.0 / (s * s * ressqr);
			} else if (flag[n]) {
				s = cur / (cur + val[n]); // Else, take the boundary point
				dist += 1.0 / (s * s * ressqr);
			} else if (flag[n + 1]) {
				s = cur / (cur + val[n + 1]);
				dist += 1.0 / (s * s * ressqr);
			}
		}
		// triangular (tetrahedral?) relationship of height in right triangles
		// gives correct distance
		tmp = (float) Math.sqrt(1.0 / dist);

		// The larger root
		return tmp;
	}
	
	/**
	 * the Fast marching distance computation (!assumes a 6D array with opposite
	 * coordinates stacked one after the other, and res[0]==res[1],
	 * res[2]==res[3], res[4]==res[5]) This method differs from the above in
	 * that it incorporates image resolution into the distance computation
	 * 
	 * Solves for the distance value at 'this' voxel, x that is consistent with
	 * the eikonal equation: \nabla T = 1;
	 * 
	 * this is done by solving for the larger root of: ((x-phi1)/resx)^2 +
	 * ((y-phi2)/resy)^2 + ((z-phi3)/resz)^2 = 1
	 * 
	 * @param val
	 * @param flag
	 * @param res sample spacing in each dimension
	 */
	public static final float minimumMarchingDistance(float[] val,
			boolean[] flag, float[] res) {

		float s, s2; // s = rx*a + ry*b + rz*c; s2 = rx*rx*a*a + ry*ry*b*b +
		// rz*rz*c*c
		float tmp;
		float count;
		s = 0;
		s2 = 0;
		count = 0;

		for (int n = 0; n < val.length; n += 2) {
			float ressqr = res[n] * res[n];
			if (flag[n] && flag[n + 1]) {
				tmp = Numerics.min(val[n], val[n + 1]); // Take the smaller one
				// if both are processed
				s += tmp / ressqr;
				s2 += tmp * tmp / ressqr;
				count += (1.0 / ressqr);
			} else if (flag[n]) {
				s += val[n] / ressqr; // Else, take the processed one
				s2 += val[n] * val[n] / ressqr;
				count += (1.0 / ressqr);
			} else if (flag[n + 1]) {
				s += val[n + 1] / ressqr;
				s2 += val[n + 1] * val[n + 1] / ressqr;
				count += (1.0 / ressqr);
			}
		}

		// count must be greater than zero since there must be at least one
		// processed pt in the neighbors
		tmp = (s + (float) Math.sqrt((double) (s * s - count * (s2 - 1.0f))))
				/ (count);

		// The larger root
		return tmp;
	}
	
	public boolean isDebug(int xyz){
		return (xyz==debugXYZ);
	}
	
	public int getDebugXYZ(){
		return debugXYZ;
	}
	
	
	public void setDebugPoint(int debugX, int debugY, int debugZ){
		this.debugX=debugX+2;
		this.debugY=debugY+2;
		this.debugZ=debugZ+2;
		
		// init debug parameters
		debugXYZ = ArrayReshape.coordsToIndex(this.debugX, this.debugY,this.debugZ, nxPad, nyPad, nzPad);
		
	}
	
	public String printDecomposition(int xyz){
		String out = "";
		for(int n=0; n<nmgdm; n++){
			out += ("Label/Distance ["+n+"]: " + mgdmlabels[n][xyz] + "\t" + mgdmfunctions[n][xyz] + "\n") ;
		}
		out+="Last neighbor: " + otherlabels[xyz];
		
		return out;
	}
	
	public MgdmDecompositionIterator iterator(){
		return new MgdmDecompositionIterator();
	}
	
	public void finalize(){
		mgdmfunctions=null;
		mgdmlabels=null;
		segmentation=null;
	}

	/**
	 * An iterator for the MGDM decomposition.
	 * 
	 * @author John Bogovic
	 *
	 */
	public class MgdmDecompositionIterator implements Iterator<Integer>{
		
		// for iterating
		protected int iterIdx;
		protected int nextIdx;
		
		@Override
		public boolean hasNext() {
			nextIdx = iterIdx + 1;

			if(nextIdx==N){ return false; } 

			while(!mask[nextIdx]){

				nextIdx++;

				if(nextIdx==N){ // return false - out of bounds
					return false;
				}
			}

			return true;
		}

		@Override
		public Integer next() {

			if(nextIdx>=0){
				iterIdx = nextIdx;
			}else if(hasNext()){
				iterIdx = nextIdx;
			}else{
				return -1;
			}
			return iterIdx;
		}

		@Override
		public void remove() {
			// do nothing
		}

		public void iteratorReset(){
			iterIdx=0;
		}
	}
}

