package edu.jhu.ece.iacl.utility;

/**
 * 
 * Class to reshape 2D, 3D and 4D arrays.
 * <p>
 * The most common methods of reshaping involve converting two or three spatial dimensions to 
 * a one-dimensional array.
 * <p>
 * Throughout the code and comments, we will refer to the first, second, and third (if present) 
 * spatial dimensions as "x", "y", and "z", respectively. 
 * <p>
 * The spatial dimensions can optionally be padded.  
 * <p>
 * For example, if a 2D array of size 10x10 is reshaped
 * to a 1D array with no padding, the resulting 1D array will be of length 100.
 * 
 * If the 2D array is reshaped with a pad of 2 in each dimension, it will yield an array of
 * length 196 (14*14, since the padding of two is applied to both the lower and upper ends of each dimension.
 * 
 * @author John Bogovic
 * @version $Revision$
 */
public class ArrayReshape{
	
	protected int dim;
	protected int pad;
	
	/**
	 * Constructor specifying the type of reshape operation this ArrayReshape object will 
	 * perform.
	 * 
	 * @param dim the number of spatial dimensions
	 * @param pad the number of units to pad to each spatial dimension during reshaping
	 */
	public ArrayReshape(int dim, int pad){
		setDim(dim);
		setPad(pad);
	}
	
	/**
	 * 
	 * @param dim the number of spatial dimensions
	 */
	public void setDim(int dim){
		this.dim=dim;
	}
	
	/**
	 * 
	 * @param pad the number of units to pad to each spatial dimension during reshaping
	 */
	public void setPad(int pad){
		this.pad=pad;
	}

	/**
	 * Reshapes a 3D float array to 1D
	 * <p>
	 * Calls reshape(in,true).
	 * 
	 * @param rowwise true if x varies most slowly, false if x varies most quickly 
	 * @return the input 3D array reshaped to 1D
	 */
	public static float[] reshape(float[][][] in){
		return reshape(in,true);
	}
	
	/**
	 *Reshapes a 3D float array to 1D
	 * 
	 * @param in a 3D float array
	 * @param rowwise true if x varies most slowly, false if x varies most quickly 
	 * @return the input 3D array reshaped to 1D 
	 */
	public static float[] reshape(float[][][] in, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		float[] out = new float[nx*ny*nz];
		int i=0;
		if(rowwise){
			for(int x=0; x<nx; x++)  for(int y=0; y<ny; y++) for(int z=0; z<nz; z++) {
				out[i]=in[x][y][z];
				i++;
			}
		}else{
			for(int z=0; z<nz; z++)  for(int y=0; y<ny; y++) for(int x=0; x<nx; x++){
				out[i]=in[x][y][z];
				i++;
			}
		}
		return out;
	}
	
	/**
	 * Reshapes a 3D int array to 1D
	 * <p>
	 * Calls reshape(in,true).
	 * 
	 * @param rowwise true if x varies most slowly, false if x varies most quickly 
	 * @return the input 3D array reshaped to 1D
	 */
	public static int[] reshape(int[][][] in){
		return reshape(in,true);
	}
	
	/**
	 *Reshapes a 3D int array to 1D
	 * 
	 * @param in a 3D int array
	 * @param rowwise true if x varies most slowly, false if x varies most quickly 
	 * @return the input 3D array reshaped to 1D 
	 */
	public static int[] reshape(int[][][] in, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		int[] out = new int[nx*ny*nz];
		int i=0;
		if(rowwise){
			for(int x=0; x<nx; x++)  for(int y=0; y<ny; y++) for(int z=0; z<nz; z++) {
				out[i]=in[x][y][z];
				i++;
			}
		}else{
			for(int z=0; z<nz; z++)  for(int y=0; y<ny; y++) for(int x=0; x<nx; x++){
				out[i]=in[x][y][z];
				i++;
			}
		}
		return out;
	}
	
	/**
	 * Reshapes a 1D int array to 3D while cropping spatial dimensions by specified amounts
	 * 
	 * @param in the 1D int array
	 * @param nx size of x
	 * @param ny size of y
	 * @param nz size of z
	 * @param padxlo amount to crop the low indices of x
	 * @param padxhi amount to crop the high indices of x
	 * @param padylo amount to crop the low indices of y
	 * @param padyhi amount to crop the high indices of y
	 * @param padzlo amount to crop the low indices of z
	 * @param padzhi amount to crop the high indices of z
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array reshaped to 3D 
	 */
	public static final int[][][] reshapeCrop(int[] in, int nx, int ny, int nz, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise ){
		int[][][] out = new int[nx][ny][nz];

		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		if(rowwise){
			for(int x=0; x<nx; x++)for(int y=0; y<ny; y++)for(int z=0; z<nz; z++){
				out[x][y][z] = in[coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
			}
		}else{
			for(int z=0; z<nz; z++)for(int y=0; y<ny; y++)for(int x=0; x<nx; x++){
				out[x][y][z] = in[coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
			}
		}
		
		return out;
	}
	
	/**
	 * Reshapes a 1D float array to 3D while cropping spatial dimensions by specified amounts
	 * 
	 * @param in the 1D float array
	 * @param nx size of x
	 * @param ny size of y
	 * @param nz size of z
	 * @param padxlo amount to crop the low indices of x
	 * @param padxhi amount to crop the high indices of x
	 * @param padylo amount to crop the low indices of y
	 * @param padyhi amount to crop the high indices of y
	 * @param padzlo amount to crop the low indices of z
	 * @param padzhi amount to crop the high indices of z
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array reshaped to 3D 
	 */
	public static final float[][][] reshapeCrop(float[] in, int nx, int ny, int nz, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise ){
		float[][][] out = new float[nx][ny][nz];

		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		if(rowwise){
			for(int x=0; x<nx; x++)for(int y=0; y<ny; y++)for(int z=0; z<nz; z++){
				out[x][y][z] = in[coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
			}
		}else{
			for(int z=0; z<nz; z++)for(int y=0; y<ny; y++)for(int x=0; x<nx; x++){
				out[x][y][z] = in[coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
			}
		}
		
		return out;
	}
	
	/**
	 * Reshapes a 2D float array to 4D while cropping spatial dimensions by specified amounts.
	 * <p>
	 * The input 2D array must have the first array dimension correspond to components, and the second
	 * spatial dimension correspond to space.  The output array will have the first three dimensions
	 * correspond to space and the last dimension correspond to components.
	 * 
	 * @param in the 2D float array
	 * @param nx size of x
	 * @param ny size of y
	 * @param nz size of z
	 * @param padxlo amount to crop the low indices of x
	 * @param padxhi amount to crop the high indices of x
	 * @param padylo amount to crop the low indices of y
	 * @param padyhi amount to crop the high indices of y
	 * @param padzlo amount to crop the low indices of z
	 * @param padzhi amount to crop the high indices of z
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 2D array reshaped to 4D 
	 */
	public static final float[][][][] reshapeMultiCrop(float[][] in, int nx, int ny, int nz, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise ){
		int nc = in.length;
		float[][][][] out = new float[nx][ny][nz][nc];

		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		if(rowwise){
			for(int c=0; c<nc; c++){
				for(int x=0; x<nx; x++)for(int y=0; y<ny; y++)for(int z=0; z<nz; z++){
					out[x][y][z][c] = in[c][coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				for(int z=0; z<nz; z++)for(int y=0; y<ny; y++)for(int x=0; x<nx; x++){
					out[x][y][z][c] = in[c][coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
				}
			}
		}
		
		return out;
	}
	
	/**
	 * Reshapes a 2D int array to 4D while cropping spatial dimensions by specified amounts.
	 * <p>
	 * The input 2D array must have the first array dimension correspond to components, and the second
	 * spatial dimension correspond to space.  The output array will have the first three dimensions
	 * correspond to space and the last dimension correspond to components.
	 * 
	 * @param in the 2D int array
	 * @param nx size of x
	 * @param ny size of y
	 * @param nz size of z
	 * @param padxlo amount to crop the low indices of x
	 * @param padxhi amount to crop the high indices of x
	 * @param padylo amount to crop the low indices of y
	 * @param padyhi amount to crop the high indices of y
	 * @param padzlo amount to crop the low indices of z
	 * @param padzhi amount to crop the high indices of z
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 2D array reshaped to 4D 
	 */
	public static final int[][][][] reshapeMultiCrop(int[][] in, int nx, int ny, int nz, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise ){
		int nc = in.length;
		int[][][][] out = new int[nx][ny][nz][nc];

		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		if(rowwise){
			for(int c=0; c<nc; c++){
				for(int x=0; x<nx; x++)for(int y=0; y<ny; y++)for(int z=0; z<nz; z++){
					out[x][y][z][c] = in[c][coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				for(int z=0; z<nz; z++)for(int y=0; y<ny; y++)for(int x=0; x<nx; x++){
					out[x][y][z][c] = in[c][coordsToIndex(x+padxlo, y+padylo, z+padzlo, nx_new, ny_new, nz_new, rowwise)];
				}
			}
		}
		
		return out;
	}
	
	
	/**
	 * Reshapes a 3D array (two spatial dimensions and one component direction)
	 *  <p>
	 *  Calls reshapeMultiPad(in,padxlo,padxhi,padylo,padyhi,rowwise,true);
	 *  
	 * @param in the 3D int array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the reshaped 2D array
	 */
	public static final int[][] reshapeMultiPad(int[][][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise){
		return reshapeMultiPad(in, padxlo, padxhi, padylo, padyhi, rowwise, true);
	}
	
	/**
	 * Reshapes a 3D array (two spatial dimensions and one component direction - formatted [x][y][z]) to a 2D array
	 * (formatted [c][xy] or [xy][c]).
	 *  
	 *  
	 * @param in the 3D int array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @param compFirst true if the output 2D array should have its first dimension correspond to components
	 * @return the reshaped 2D array
	 */
	public static final int[][] reshapeMultiPad(int[][][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise, boolean compFirst){
		int nx=in.length;
		int ny=in[0].length;
		int nc=in[0][0].length;
		
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;

		int xx=-1, yy=-1; // coordinate in original image

		int[][] out = null;
		if(compFirst){
			out = new int[nc][nx_new*ny_new];
		}else{
			out = new int[nx_new*ny_new][nc];
		}
		
		
		int k = 0;
		if(rowwise){
			for(int c=0; c<nc; c++){
				k = 0;
				for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

					if(compFirst){
						out[c][k] = in[xx][yy][c];
					}else{
						out[k][c] = in[xx][yy][c];
					}
					k++;
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				k = 0;
				for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

					if(compFirst){
						out[c][k] = in[xx][yy][c];
					}else{
						out[k][c] = in[xx][yy][c];
					}
					
					k++;
				}
			}
		}

		return out;
	}
	
	/**
	 * Reshapes a 3D array (two spatial dimensions and one component direction)
	 *  <p>
	 *  Calls reshapeMultiPad(in,padxlo,padxhi,padylo,padyhi,rowwise,true);
	 *  
	 * @param in the 3D int array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the reshaped 2D array
	 */
	public static final float[][] reshapeMultiPad(float[][][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise){
		return reshapeMultiPad(in, padxlo, padxhi, padylo, padyhi, rowwise, true);
	}
	
	/**
	 * Reshapes a 3D array (two spatial dimensions and one component direction - formatted [x][y][z]) to a 2D array
	 * (formatted [c][xy] or [xy][c]).
	 *  
	 *  
	 * @param in the 3D int array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @param compFirst true if the output 2D array should have its first dimension correspond to components
	 * @return the reshaped 2D array
	 */
	public static final float[][] reshapeMultiPad(float[][][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise, boolean compFirst){
		int nx=in.length;
		int ny=in[0].length;
		int nc=in[0][0].length;
		
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;

		int xx=-1, yy=-1; // coordinate in original image

		float[][] out = null;
		if(compFirst){
			out = new float[nc][nx_new*ny_new];
		}else{
			out = new float[nx_new*ny_new][nc];
		}
		int k = 0;
		if(rowwise){
			for(int c=0; c<nc; c++){
				k = 0;
				for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

					if(compFirst){
						out[c][k] = in[xx][yy][c];
					}else{
						out[k][c] = in[xx][yy][c];
					}
					k++;
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				k = 0;
				for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

					if(compFirst){
						out[c][k] = in[xx][yy][c];
					}else{
						out[k][c] = in[xx][yy][c];
					}
					k++;
				}
			}
		}

		return out;
	}
	
	/**
	 * Reshapes a 2D array with two spatial dimensions to a 1D array.
	 * 
	 * @param in the 2D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array
	 */
	public static final float[] reshapePad(float[][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;

		int xx=-1, yy=-1; // coordinate in original image

		float[] out = new float[nx_new*ny_new];
		int k = 0;
		if(rowwise){
			k = 0;
			for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

				out[k] = in[xx][yy];
				k++;
			}

		}else{
			k = 0;
			for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

				out[k] = in[xx][yy];
				k++;
			}

		}

		return out;
	}
	
	/**
	 * Reshapes a 2D array with two spatial dimensions to a 1D array.
	 * 
	 * @param in the 2D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array
	 */
	public static final int[] reshapePad(int[][] in, int padxlo, int padxhi, int padylo, int padyhi, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;

		int xx=-1, yy=-1; // coordinate in original image

		int[] out = new int[nx_new*ny_new];
		int k = 0;
		if(rowwise){
			k = 0;
			for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

				out[k] = in[xx][yy];
				k++;
			}

		}else{
			k = 0;
			for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);

				out[k] = in[xx][yy];
				k++;
			}

		}

		return out;
	}
	
	/**	
	 * Reshapes a 3D array with three spatial dimensions to a 1D array.
	 * 
	 * @param in the 3D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param padzlo amount to pad the high indices of y
	 * @param padzhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array
	 */
	public static final float[] reshapePad(float[][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;

		int xx=-1, yy=-1, zz=-1; // coordinate in original image

		float[] out = new float[nx_new*ny_new*nz_new];
		int k = 0;
		if(rowwise){
			k = 0;
			for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++)for(int z=0; z<nz_new; z++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
				zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

				out[k] = in[xx][yy][zz];
				k++;
			}
			

		}else{
			k = 0;
			for(int z=0; z<nz_new; z++)for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
				zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

				out[k] = in[xx][yy][zz];
				k++;
			}

		}

		return out;
	}
	
	/**	
	 * Reshapes a 3D array with three spatial dimensions to a 1D array.
	 * 
	 * @param in the 3D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param padzlo amount to pad the high indices of y
	 * @param padzhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the 1D array
	 */
	public static final int[] reshapePad(int[][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;

		int xx=-1, yy=-1, zz=-1; // coordinate in original image

		int[] out = new int[nx_new*ny_new*nz_new];
		int k = 0;
		if(rowwise){
			k = 0;
			for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++)for(int z=0; z<nz_new; z++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
				zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

				out[k] = in[xx][yy][zz];
				k++;
			}

		}else{
			k = 0;
			for(int z=0; z<nz_new; z++)for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
				xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
				yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
				zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

				out[k] = in[xx][yy][zz];
				k++;
			}

		}

		return out;
	}
	
	
	/**
	 * Reshapes a 4D array with three spatial dimensions and one component dimension (formatted
	 * [x][y][z][c]) to a 2D array (formatted either [xyz][c] or [c][xyz]).
	 * <p>
	 * Calls reshapeMultiPad(in, padxlo, padxhi, padylo, padyhi, padzlo, padzhi, rowwise, true)
	 * 
	 * @param in the 2D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param padzlo amount to pad the high indices of y
	 * @param padzhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @param compFirst true if the output 2D array should have its first dimension correspond to components
	 * @return the 2D array
	 */
	public static final int[][] reshapeMultiPad(int[][][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise){
		return reshapeMultiPad(in, padxlo, padxhi, padylo, padyhi, padzlo, padzhi, rowwise, true);
	}
	
	/**	
	 * Reshapes a 4D array with three spatial dimensions and one component dimension (formatted
	 * [x][y][z][c]) to a 2D array (formatted either [xyz][c] or [c][xyz]).
	 * 
	 * @param in the 2D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param padzlo amount to pad the high indices of y
	 * @param padzhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @param compFirst true if the output 2D array should have its first dimension correspond to components
	 * @return the 2D array
	 */
	public static final int[][] reshapeMultiPad(int[][][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise, boolean compFirst){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		int nc=in[0][0][0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		int xx=-1, yy=-1, zz=-1; // coordinate in original image
		
		int[][] out = null;
		if(compFirst){
			out = new int[nc][nx_new*ny_new*nz_new];
		}else{
			out = new int[nx_new*ny_new*nz_new][nc];
		}
		
		int k = 0;
		if(rowwise){
			for(int c=0; c<nc; c++){
				k = 0;
				for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++)for(int z=0; z<nz_new; z++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
					zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

					if(compFirst){
						out[c][k] = in[xx][yy][zz][c];
					}else{
						out[k][c] = in[xx][yy][zz][c];
					}
					
					k++;
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				k = 0;
				for(int z=0; z<nz_new; z++)for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
					zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

					if(compFirst){
						out[c][k] = in[xx][yy][zz][c];
					}else{
						out[k][c] = in[xx][yy][zz][c];
					}
					
					k++;
				}
			}
		}
		
		return out;
	}
	
	public static final float[][] reshapeMultiPad(float[][][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise){
		return reshapeMultiPad(in, padxlo, padxhi, padylo, padyhi, padzlo, padzhi, rowwise, true);
	}
	
	/**	
	 * Reshapes a 4D array with three spatial dimensions and one component dimension (formatted
	 * [x][y][z][c]) to a 2D array (formatted either [xyz][c] or [c][xyz]).
	 * 
	 * @param in the 2D array
	 * @param padxlo amount to pad the low indices of x
	 * @param padxhi amount to pad the high indices of x
	 * @param padylo amount to pad the low indices of y
	 * @param padyhi amount to pad the high indices of y
	 * @param padzlo amount to pad the high indices of y
	 * @param padzhi amount to pad the high indices of y
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @param compFirst true if the output 2D array should have its first dimension correspond to components
	 * @return the 2D array
	 */
	public static final float[][] reshapeMultiPad(float[][][][] in, int padxlo, int padxhi, int padylo, int padyhi, int padzlo, int padzhi, boolean rowwise, boolean compFirst){
		int nx=in.length;
		int ny=in[0].length;
		int nz=in[0][0].length;
		int nc=in[0][0][0].length;
		int nx_new = nx+padxlo+padxhi;
		int ny_new = ny+padylo+padyhi;
		int nz_new = nz+padzlo+padzhi;
		
		int xx=-1, yy=-1, zz=-1; // coordinate in original image
		
		float[][] out = null;
		if(compFirst){
			out = new float[nc][nx_new*ny_new*nz_new];
		}else{
			out = new float[nx_new*ny_new*nz_new][nc];
		}
		int k = 0;
		if(rowwise){
			for(int c=0; c<nc; c++){
				k = 0;
				for(int x=0; x<nx_new; x++)for(int y=0; y<ny_new; y++)for(int z=0; z<nz_new; z++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
					zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

					if(compFirst){
						out[c][k] = in[xx][yy][zz][c];
					}else{
						out[k][c] = in[xx][yy][zz][c];
					}
					k++;
				}
			}
		}else{
			for(int c=0; c<nc; c++){
				k = 0;
				for(int z=0; z<nz_new; z++)for(int y=0; y<ny_new; y++)for(int x=0; x<nx_new; x++){
					xx = getReplicatePadCoordinate(x, nx, padxlo, padxhi);
					yy = getReplicatePadCoordinate(y, ny, padylo, padyhi);
					zz = getReplicatePadCoordinate(z, nz, padzlo, padzhi);

					if(compFirst){
						out[c][k] = in[xx][yy][zz][c];
					}else{
						out[k][c] = in[xx][yy][zz][c];
					}
					k++;
				}
			}
		}
		
		return out;
	}
	
	/**
	 * Method used in padding a spatial dimension by repeating 
	 * the value at the first or last index of that spatial dimension.
	 * <p>
	 * Given a spatial coordinate after padding, this method returns the
	 * index in the original (before padding) coordinate system from which 
	 * to take the value.
	 * 
	 * @param x index in a spatial dimension for the new coordinates(after padding) 
	 * @param nxOrig size of the original dimension (before padding)
	 * @param padxlo amount to pad the low indices
	 * @param padxhi amount to pad the high indices
	 * @return the index into the old (before padding) coordinates
	 */
	public static final int getReplicatePadCoordinate(int x, int nxOrig, int padxlo, int padxhi){
		int xx = -1;
		if(x<padxlo){ xx = 0;
		}else if( (x-padxlo) >= nxOrig){ xx = nxOrig - 1;
		}else{ xx=x-padxlo; }
		
		return xx;
	}
	
	/**
	 * Method used in padding a spatial dimension by mirroring 
	 * the values near the first or last index of that spatial dimension.
	 * <p>
	 * Given a spatial coordinate after padding, this method returns the
	 * index in the original (before padding) coordinate system from which 
	 * to take the value.
	 * 
	 * @param x index in a spatial dimension for the new coordinates(after padding) 
	 * @param nxOrig size of the original dimension (before padding)
	 * @param padxlo amount to pad the low indices
	 * @param padxhi amount to pad the high indices
	 * @return the index into the old (before padding) coordinates
	 */
	public static final int getMirrorPadCoordinate(int x, int nxOrig, int padxlo, int padxhi){
		int xx = -1;
		if(x<padxlo){ xx = (padxlo - x -1);
		}else if( (x-2) >= nxOrig){ xx = 2*nxOrig -x + 1;
		}else{ xx=x-padxlo; }
		
		return xx;
	}
	
	/**
	 * Returns the indices into a 3D array of size (nx) x (ny) x (nz)
	 * given the index xyz in the corresponding 1D array. 
	 *  
	 * @param xyz the index in the 1D array
	 * @param nx size in dimension x
	 * @param ny size in dimension y
	 * @param nz size in dimension z
	 * @return 1D array of length 3 containing the indicies for the 3D array
	 */
	public static int[] indexToCoordinates(int xyz, int nx, int ny, int nz){
		
		if(xyz>nx*ny*nz){
			return null;
		}else if(xyz<0){
			return null;
		}
		
		int[] coord = new int[3];
		
		coord[2] = xyz%nz;
		xyz -= coord[2];
		xyz/=nz;
		
		coord[1] = xyz%ny;
		xyz -= coord[1];
		xyz/=ny;
		
		coord[0] = xyz%nx;
		
		return coord;
	}
	
	/**
	 * Returns the indices into a 2D array of size (nx) x (ny)
	 * given the index xyz in the corresponding 1D array. 
	 *  
	 * @param xyz the index in the 1D array
	 * @param nx size in dimension x
	 * @param ny size in dimension x
	 * @return 1D array of length 2 containing the indicies for the 2D array
	 */
	public static int[] indexToCoordinates(int xyz, int nx, int ny){
		
		if(xyz>nx*ny){
			return null;
		}else if(xyz<0){
			return null;
		}
		
		int[] coord = new int[2];
		
		coord[1] = xyz%ny;
		xyz -= coord[1];
		xyz/=ny;
		
		coord[0] = xyz%nx;
		
		return coord;
	}
	
	/**
	 * Returns the index into a 1D array given the index into the corresponding 3D array.
	 * Assumes x varies most slowly.
	 * <p> 
	 * Calls coordsToIndex(x,y,z,nx,ny,nz,true)
	 * 
	 * @param x
	 * @param y
	 * @param z
	 * @param nx
	 * @param ny
	 * @param nz
	 * @return
	 */
	public static int coordsToIndex(int x, int y, int z, int nx, int ny, int nz){
		return coordsToIndex(x,y,z,nx,ny,nz,true);
	}
	
	/**
	 * Returns the index into a 1D array given the index into the corresponding 2D array.
	 * Assumes x varies most slowly.
	 * <p>
	 * Calls coordsToIndex(x,y,nx,ny,true)
	 * 
	 * @param x
	 * @param y
	 * @param nx
	 * @param ny
	 * @return
	 */
	public static int coordsToIndex(int x, int y, int nx, int ny){
		return coordsToIndex(x,y,nx,ny,true);
	}
	
	
	/**
	 * Returns the index into a 1D array given the index into the corresponding 3D array. 
	 * 
	 * @param x
	 * @param y
	 * @param z
	 * @param nx
	 * @param ny
	 * @param nz
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the index into the 1D array
	 */
	public static int coordsToIndex(int x, int y, int z, int nx, int ny, int nz, boolean rowwise){
		if(rowwise)	return z + nz*y + nz*ny*x;
		else  		return x + nx*y + nx*ny*z;
	}
	
	/**
	 * Returns the index into a 1D array given the index into the corresponding 2D array. 
	 * 
	 * @param x
	 * @param y
	 * @param nx
	 * @param ny
	 * @param rowwise true if x varies most slowly, false if x varies most quickly
	 * @return the index into the 1D array
	 */
	public static int coordsToIndex(int x, int y,int nx, int ny, boolean rowwise){
		if(rowwise)	return y + ny*x;
		else  		return x + nx*y;
	}
	

}
