function helpReg = procStreamRegHelp()
helpReg = {...
{...
'dod = hmrIntensityNormalized( d )',...
'UI NAME',...
'Intensity_Normalized',...
'INPUT',...
'd - intensity data (#time points x #data channels)',...
'OUTPUT',...
'dod - the intensity divided by the mean'...
},...

{...
'dod = hmrIntensity2OD( d )',...
'UI NAME:',...
'Intensity_to_OD ',...
'Converts internsity (raw data) to optical density',...
'INPUT',...
'd - intensity data (#time points x #data channels',...
'OUTPUT',...
'dod - the change in optical density'...
},...

{...
'dod = hmrFlowInput( d )',...
'UI NAME',...
'Relative Flow Change',...
'INPUT',...
'd - intensity data (#time points x #data channels)',...
'OUTPUT',...
'dod - the intensity divided by the mean'...
},...

{...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'UI NAME:',...
'PCA_Filter',...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'Perform a PCA filter on the data matrix y. ',...
'INPUT:',...
'y: This is the data matrix where rows are time points. If y is wavelength',...
'data, then the columns are channels as described in SD.MeasList. If',...
'y is concentration data, then the third dimension is channels and the',...
'second dimension indicates HbO and HbR.',...
'SD: This is the source detector structure.',...
'tInc: This is a vector of length number of time points and is 1 to',...
'indicate that a time point is included in the analysis and 0 if it is to',...
'be excluded. This is useful for ignoring periods of time with strong',...
'artifacts.',...
'nSV: This is the number of principle components to remove filter from the',...
'data. This can be an integer to indicate the number of components to',...
'remove. Or it can be a fraction less than 1 to indicate that enough',...
'components should be removed to remove up to that fraction of the',...
'variance in the data. If nSV is a single number it is applied to all',...
'channels of data. This is useful for filtering motion artifacts. If it is',...
'2 or more numbers, then it is applied to each wavelength or concentration',...
'separately using the corresponding number.',...
'OUTPUT:',...
'yc: The filtered data matrix.',...
'svs: The singuler value spectrum from the PCA.',...
'nSV: This is the number of components filtered from the data.'...
},...

{...
'SD = enPruneChannels(d,SD,tInc,dRange,SNRthresh,SDrange,reset)',...
'UI NAME:',...
'Prune_Channels',...
'Prune channels from the measurement list if their signal is too weak, too',...
'strong, or their standard deviation is too great. This function',...
'updates SD.MeasListAct based on whether data ''d''',...
'as specified by''dRange'' and ''SNRthresh''',...
'INPUTS:',...
'd - data (nTpts x nChannels )',...
'SD - data structure describing the probe',...
'dRange - if mean(d) < dRange(1) or > dRange(2) then it is excluded as an',...
'active channel',...
'SNRthresh - if mean(d)/std(d) < SNRthresh then it is excluded as an',...
'active channel',...
'SDrange - will prune channels with a source-detector separation <',...
'SDrange(1) or > SDrange(2)',...
'reset - reset previously pruned channels (automatic and manual)',...
'OUTPUTS:',...
'SD - data structure describing the probe',...
'TO DO:',...
'consider Conc as well as wavelength data'...
},...

{...
'tInc = hmrMotionArtifact(d, fs, SD, tIncMan, tMotion, tMask, STDEVthresh, AMPthresh)',...
'UI NAME:   ',...
'Motion_Artifacts',...
'Identifies motion artifacts in an input data matrix d. If any active ',...
'data channel exhibits a signal change greater than std_thresh or',...
'amp_thresh, then a segment of data around that time point is marked as a',...
'motion artifact.',...
'INPUTS:',...
'd: data matrix, timepoints x sd pairs',...
'fs: sample frequency in Hz. You can send the time vector and fs will be',...
'calculated',...
'SD: Source Detector Structure. The active data channels are indicated in',...
'SD.MeasListAct.',...
'tIncMan: Data that has been manually excluded. 0-excluded. 1-included.',...
'Vector same length as d.',...
'tMotion: Check for signal change indicative of a motion artifact over',...
'time range tMotion. Units of seconds. ',...
'Typical value ranges from 0.1 to 0.5.',...
'tMask: Mark data over +/- tMask seconds around the identified motion ',...
'artifact as a motion artifact. Units of seconds. ',...
'Typical value ranges from 0.5 to 1.',...
'STDEVthresh: If the signal d for any given active channel changes by more',...
'that stdev_thresh * stdev(d) over the time interval tMotion, then',...
'this time point is marked as a motion artifact. The standard deviation is',...
'determined for each channel independently.',...
'Typical value ranges from 5 to 20. Use a value of 100 or greater if',...
'you wish for this condition to not find motion artifacts.',...
'AMPthresh: If the signal d for any given active channel changes by more',...
'that amp_thresh over the time interval tMotion, then this time point',...
'is marked as a motion artifact.',...
'Typical value ranges from 0.01 to 0.3 for optical density units. Use',...
'a value greater than 100 if you wish for this condition to not find',...
'motion artifacts.',...
'OUTPUTS:',...
'tInc: a vector of length time points with 1''',...
'and 0''',...
'LOG:',...
'K. Perdue',...
'kperdue@nmr.mgh.harvard.edu',...
'Sept. 23, 2010',...
'Modified by DAB 3/17/2011 to not act on a single channel by default',...
'DAB 4/20/2011 added tMotion and tMask and hard coded stdev option.',...
'DAB 8/4/2011 added amp_thresh to work at same time as std_thresh',...
'TO DO:',...
'Consider tIncMan'...
},...

{...
'[tInc,tIncCh] = hmrMotionArtifactByChannel(d, fs, SD, tIncMan, tMotion, tMask, STDEVthresh, AMPthresh)',...
'UI NAME:   ',...
'Motion_Artifacts_By_Channel',...
'Identifies motion artifacts in an input data matrix d. If any active ',...
'data channel exhibits a signal change greater than std_thresh or',...
'amp_thresh, then a segment of data around that time point is marked as a',...
'motion artifact. The channel-wise motion artifacts are recorded in the',...
'output matrix tIncCh. If any channel has a motion artifact, it is',...
'indicated by the vector tInc.',...
'INPUTS:',...
'd: data matrix, timepoints x sd pairs',...
'fs: sample frequency in Hz. You can send the time vector and fs will be',...
'calculated',...
'SD: Source Detector Structure. The active data channels are indicated in',...
'SD.MeasListAct.',...
'tIncMan: Data that has been manually excluded. 0-excluded. 1-included.',...
'Vector same length as d.',...
'tMotion: Check for signal change indicative of a motion artifact over',...
'time range tMotion. Units of seconds.',...
'tMask: Mark data over +/- tMask seconds around the identified motion ',...
'artifact as a motion artifact. Units of seconds.',...
'STDEVthresh: If the signal d for any given active channel changes by more',...
'that stdev_thresh * stdev(d) over the time interval tMotion, then',...
'this time point is marked as a motion artifact. The standard deviation is',...
'determined for each channel independently.',...
'AMPthresh: If the signal d for any given active channel changes by more',...
'that amp_thresh over the time interval tMotion, then this time point',...
'is marked as a motion artifact.',...
'OUTPUTS:',...
'tInc: a vector of length time points with 1''',...
'and 0''',...
'tIncCh: a matrix with #time points x #channels, with 1''',...
'included and 0''',...
'channel basis',...
'LOG:',...
'K. Perdue',...
'kperdue@nmr.mgh.harvard.edu',...
'Sept. 23, 2010',...
'Modified by DAB 3/17/2011 to not act on a single channel by default',...
'DAB 4/20/2011 added tMotion and tMask and hard coded stdev option.',...
'DAB 8/4/2011 added amp_thresh to work at same time as std_thresh',...
'TO DO:',...
'Consider tIncMan'...
},...

{...
'function dodWavelet = hmrMotionCorrectWavelet(dod,SD,iqr)',...
'UI NAME:',...
'Wavelet_Motion_Correction',...
'Perform a wavelet transformation of the dod data and computes the',...
'distribution of the wavelet coefficients. It sets the coefficient',...
'exceeding iqr times the interquartile range to zero, because these are probably due',...
'to motion artifacts. set iqr<0 to skip this function.',...
'The algorithm follows in part the procedure described by',...
'Molavi et al.,Physiol Meas, 33, 259-270 (2012).',...
'INPUTS:',...
'dod -  delta_OD ',...
'SD -   SD structure',...
'iqr -  parameter used to compute the statistics (iqr = 1.5 is 1.5 times the',...
'interquartile range and is usually used to detect outliers). ',...
'Increasing it, it will delete fewer coefficients.',...
'If iqr<0 then this function is skipped. ',...
'OUTPUTS:',...
'dodWavelet - dod after wavelet motion correction, same',...
'size as dod (Channels that are not in the active ml remain unchanged)',...
'LOG:',...
'Script by Behnam Molavi bmolavi@ece.ubc.ca adapted for Homer2 by RJC',...
'modified 10/17/2012 by S. Brigadoi'...
},...

{...
'function dodKWavelet = hmrMotionCorrectKurtosisWavelet(dod,SD,kurt)',...
'UI NAME:',...
'Wavelet_Motion_Correction',...
'Performs a wavelet transformation of the dod data and computes the',...
'distribution of the wavelet coefficients. It iteratively eliminates the',...
'biggest (as absolute value) wavelet coefficient until the kurtosis (the',...
'kurt parameter), of the wavelet coefficients is below a certain',...
'threshold. It does so for each level of the wavelet decomposition.',...
'The algorithm follows the procedure described by',...
'Chiarelli, A. M. et al. (2015). A kurtosis-based wavelet algorithm for motion artifact correction of fNIRS data. Neuroimage, 112, 128-137.',...
'INPUTS:',...
'dod:   delta_OD ',...
'SD:    SD structure',...
'kurt    kurtosis threshold of the wavelet coefficients.',...
'the kurtosis level is selected to test the gaussianity of the wavelet coefficients.',...
'lower level of the kurtosis reflects in stricter gaussianity',...
'assumption about the signal. Kurtosis needs to me strictly above',...
'3',...
'If kurt<=3 then this function is skipped.',...
'SUGGESTED KURTOSIS LEVEL: 3.3',...
'OUTPUTS:',...
'dodKWavelet:  dod after wavelet motion correction, same',...
'size as dod (Channels that are not in the active ml remain unchanged)',...
'LOG:',...
'Script by Antonio Maria Chiarelli chiarell@illinois.edu adapted for Homer2 by',...
'Antonio Maria Chiarelli the 10/18/2016'...
},...

{...
'[dN,svs,nSV] = hmrMotionCorrectPCA( SD, d, tInc, nSV )',...
'UI NAME:',...
'PCA_Motion_Correction',...
'This function uses PCA to filter only the segments of data identified as',...
'a motion artifact. The motion artifacts are indicated in the tInc vector',...
'by the value of 0.',...
'INPUTS',...
'SD:   Source Detector Structure. The active data channels are indicated in',...
'SD.MeasListAct.',...
'd:    data matrix, timepoints x sd pairs',...
'tInc: a vector of length time points with 1''',...
'and 0''',...
'nSV: This is the number of principal components to remove from the data.',...
'If this number is less than 1, then the filter removes the first n',...
'components of the data that removes a fraction of the variance',...
'up to nSV.',...
'OUTPUTS',...
'dN: This the the motion corrected data.',...
'svs: the singular values of the PCA',...
'nSV: number of singular values removed from the data.'...
},...

{...
'[dN,tInc,dstd,svs,nSV,tInc0] = hmrMotionCorrectPCArecurse( d, fs, SD, tIncMan, tMotion, tMask, std_thresh, amp_thresh, nSV, turn on)',...
'UI NAME',...
'Motion_Correct_PCA_Recurse',...
'Identified motion artifacts in an input data matrix d. If any active ',...
'data channel exhibits a signal change greater than std_thresh or',...
'amp_thresh, then a segment of data around that time point is marked as a',...
'motion artifact. Set maxIter=0 to skip this function.',...
'INPUTS:',...
'd: data matrix, timepoints x sd pairs',...
'fs: sample frequency in Hz. You can send the time vector and fs will be',...
'calculated',...
'SD: Source Detector Structure. The active data channels are indicated in',...
'SD.MeasListAct.',...
'tIncMan: Data that has been manually excluded. 0-excluded. 1-included.',...
'Vector same length as d.',...
'tMotion: Check for signal change indicative of a motion artifact over',...
'time range tMotion. Units of seconds. Typically tMotion=0.5.',...
'tMask: Mark data over +/- tMask seconds around the identified motion ',...
'artifact as a motion artifact. Units of seconds. Typicall tMask=1.',...
'STDEVthresh: If the signal d for any given active channel changes by more',...
'that stdev_thresh * stdev(d) over the time interval tMotion, then',...
'this time point is marked as a motion artifact. The standard deviation is',...
'determined for each channel independently.',...
'AMPthresh: If the signal d for any given active channel changes by more',...
'that amp_thresh over the time interval tMotion, then this time point',...
'is marked as a motion artifact.',...
'nSV: This is the number of principal components to remove from the data.',...
'If this number is less than 1, then the filter removes the first n',...
'components of the data that removes a fraction of the variance',...
'up to nSV. Yucel et al uses nSV=0.97.',...
'maxIter: Maximum number of iterations. Yucel et al uses maxIter=5;',...
'OUTPUTS:',...
'dN: This is the the motion corrected data.',...
'tInc: a vector of length time points with 1''',...
'and 0''',...
'artifacts',...
'svs: the singular values of the PCA for each iteration in each column',...
'vector',...
'nSV: number of singular values removed from the data.',...
'tInc0: a vector of length time points with 1''',...
'and 0''',...
'artifacts'...
},...

{...
'[dN,tInc,dstd,svs,nSV,tInc0] = hmrMotionCorrectPCArecurse( d, fs, SD, tIncMan, tMotion, tMask, std_thresh, amp_thresh, nSV )',...
'UI NAME',...
'Motion_Correct_PCA_Recurse',...
'Identified motion artifacts in an input data matrix d. If any active ',...
'data channel exhibits a signal change greater than std_thresh or',...
'amp_thresh, then a segment of data around that time point is marked as a',...
'motion artifact. Set maxIter=0 to skip this function.',...
'INPUTS:',...
'd: data matrix, timepoints x sd pairs',...
'fs: sample frequency in Hz. You can send the time vector and fs will be',...
'calculated',...
'SD: Source Detector Structure. The active data channels are indicated in',...
'SD.MeasListAct.',...
'tIncMan: Data that has been manually excluded. 0-excluded. 1-included.',...
'Vector same length as d.',...
'tMotion: Check for signal change indicative of a motion artifact over',...
'time range tMotion. Units of seconds. Typically tMotion=0.5.',...
'tMask: Mark data over +/- tMask seconds around the identified motion ',...
'artifact as a motion artifact. Units of seconds. Typicall tMask=1.',...
'STDEVthresh: If the signal d for any given active channel changes by more',...
'that stdev_thresh * stdev(d) over the time interval tMotion, then',...
'this time point is marked as a motion artifact. The standard deviation is',...
'determined for each channel independently.',...
'AMPthresh: If the signal d for any given active channel changes by more',...
'that amp_thresh over the time interval tMotion, then this time point',...
'is marked as a motion artifact.',...
'nSV: This is the number of principal components to remove from the data.',...
'If this number is less than 1, then the filter removes the first n',...
'components of the data that removes a fraction of the variance',...
'up to nSV. Yucel et al uses nSV=0.97.',...
'maxIter: Maximum number of iterations. Yucel et al uses maxIter=5;',...
'OUTPUTS:',...
'dN: This is the the motion corrected data.',...
'tInc: a vector of length time points with 1''',...
'and 0''',...
'artifacts',...
'svs: the singular values of the PCA for each iteration in each column',...
'vector',...
'nSV: number of singular values removed from the data.',...
'tInc0: a vector of length time points with 1''',...
'and 0''',...
'artifacts',...
'LOGS:',...
'This function performs tPCA and splice the segments of data together in',...
'two different ways (either using one tInc for all channels or using',...
'tIncCh), compares the results and picks the good one for a specific channel. Meryem Oct 2017'...
},...

{...
'This function first finds baseline shifts and corrects them with tPCA_Ch',...
'and then corrects the remaining spikes with Savitzky-Golay smoothing.',...
'Sahar Jahani, Oct 2017',...
'added turn on/off option Meryem, Nov 2017'...
},...

{...
'function dodSpline = hmrMotionCorrectSpline(dod, t, SD, tInc, p)',...
'UI NAME:',...
'Spline_Motion_Correction',...
'dodSpline = hmrMotionCorrectSpline(dod, t, SD, tInc, p)',...
'Perform a cubic spline correction of the motion artifacts identified in',...
'tInc. The algorithm follows the procedure describe by',...
'Scholkmann et al., Physiol. Meas. 31, 649-662 (2010). Set p = -1 to skip',...
'this function.',...
'INPUTS:',...
'dod:   delta_OD',...
't:     time vector',...
'SD:    SD structure',...
'tInc:  Matrix of included time points (1=included; 0=not included (movement)',...
'The matrix is #time points x #channels and usually comes from',...
'hmrMotionArtifactByChannel()',...
'p:     Parameter p used in the spline interpolation. The value',...
'recommended in the literature is 0.99. Use -1 if you want to skip this',...
'motion correction.',...
'OUTPUTS:',...
'dodSpline:  dod after spline interpolation correction, same size as dod',...
'(Channels that are not in the active ml remain unchanged)',...
'LOG:',...
'created 01-26-2012, J. Selb',...
'TO DO:'...
},...

{...
'Sahar Jahani, October 2017',...
'Added turn on/off option Meryem Nov 2017'...
},...

{...
'function dcCbsi = hmrMotionCorrectCbsi(dc,SD,flagSkip)',...
'UI NAME:',...
'Cbsi_Motion_Correction',...
'Perform a correlation-based signal improvement of the concentration',...
'changes in order to correct for motion artifacts.  ',...
'The algorithm follows the procedure described by',...
'Cui et al.,NeuroImage, 49(4), 3039-46 (2010).',...
'INPUTS:',...
'dc:    Concentration changes (it works with HbO and HbR)',...
'SD:    SD structure',...
'flagSkip:  Skip this function if flagSkip=1. Otherwise execute function. ',...
'Default is to execute function if this does not exist.',...
'OUTPUTS:',...
'dcSpline:  dc after correlation-based signal improvement correction, same',...
'size as dc (Channels that are not in the active ml remain unchanged)',...
'LOG:',...
'created 10-17-2012, S. Brigadoi'...
},...

{...
'Meryem Yucel, Oct, 2017',...
'span = 0.02 (default)',...
'Added turn on/off option Meryem Nov 2017'...
},...

{...
'Savitzky_Golay filter',...
'K: polynomial order (default = 3)',...
'FrameSize_sec: Frame size in sec (default = 10 sec)',...
'Meryem Yucel, Oct, 2017',...
'Added turn on/off option Meryem Nov 2017'...
},...

{...
'[s,tRange] = enStimRejection(t,s,tIncAuto,tIncMan,tRange)',...
'UI NAME:',...
'Stim_Exclude',...
'Excludes stims that fall within the time points identified as ',...
'motion artifacts from HRF calculation.',...
'INPUT:',...
't:        the time vector (#time points x 1)',...
's:        s matrix (#time points x #conditions) containing 1 for ',...
'each time point and condition that has a stimulus and ',...
'zeros otherwise.',...
'tIncAuto: time points (#time points x 1) identified as motion ',...
'artifacts by processing stream.',...
'tIncMan:  time points (#time points x 1) identified as motion ',...
'artifacts by user.',...
'tRange:   an array of 2 numbers [t1 t2] specifying how many ',...
'seconds surrounding motion artifacts, tIncMan and tIncAuto, ',...
'to consider as excluded data and therefore exclude any stimuli ',...
'which fall within those buffers.',...
'Typically values are t1=-2 and t2 equal to the stimulus',...
'duration.',...
'OUTPUT:',...
's:        s matrix (#time points x #conditions) containing 1 for ',...
'each time point and condition that has a stimulus that is ',...
'included in the HRF calculation, -1 for a stimulus that is ',...
'excluded automatically in the processing stream, -2 ',...
'for each stimulus excluded by a manually set patch and ',...
'zeros otherwise.',...
'tRange:   same tRange array as in the input'...
},...

{...
'y2 = hmrBandpassFilt( y, fs, hpf, lpf )',...
'UI NAME:',...
'Bandpass_Filter',...
'y2 = hmrBandpassFilt( y, fs, hpf, lpf )',...
'Perform a bandpass filter',...
'INPUT:',...
'y - data to filter #time points x #channels of data',...
'fs - sample frequency (Hz). If length(fs)>1 then this is assumed to be a time',...
'vector from which fs is estimated',...
'hpf - high pass filter frequency (Hz)',...
'Typical value is 0 to 0.02.',...
'lpf - low pass filter frequency (Hz)',...
'Typical value is 0.5 to 3.',...
'OUTPUT:',...
'y2 - filtered data'...
},...

{...
''...
},...

{...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'UI NAME:',...
'PCA_Filter',...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'Perform a PCA filter on the data matrix y. ',...
'INPUT:',...
'y: This is the data matrix where rows are time points. If y is wavelength',...
'data, then the columns are channels as described in SD.MeasList. If',...
'y is concentration data, then the third dimension is channels and the',...
'second dimension indicates HbO and HbR.',...
'SD: This is the source detector structure.',...
'tInc: This is a vector of length number of time points and is 1 to',...
'indicate that a time point is included in the analysis and 0 if it is to',...
'be excluded. This is useful for ignoring periods of time with strong',...
'artifacts.',...
'nSV: This is the number of principle components to remove filter from the',...
'data. This can be an integer to indicate the number of components to',...
'remove. Or it can be a fraction less than 1 to indicate that enough',...
'components should be removed to remove up to that fraction of the',...
'variance in the data. If nSV is a single number it is applied to all',...
'channels of data. This is useful for filtering motion artifacts. If it is',...
'2 or more numbers, then it is applied to each wavelength or concentration',...
'separately using the corresponding number.',...
'OUTPUT:',...
'yc: The filtered data matrix.',...
'svs: The singuler value spectrum from the PCA.',...
'nSV: This is the number of components filtered from the data.'...
},...

{...
'dc = hmrOD2Conc( dod, SD, ppf )',...
'UI NAME:',...
'OD_to_Conc',...
'dc = hmrOD2Conc( dod, SD, ppf )',...
'Convert OD to concentrations',...
'INPUTS:',...
'dod: the change in OD (#time points x #channels)',...
'SD:  the SD structure. A spatial unit of mm is assumed, but if',...
'SD.SpatialUnit = ''cm''',...
'ppf: partial pathlength factors for each wavelength. If there are 2',...
'wavelengths of data, then this is a vector ot 2 elements.',...
'Typical value is ~6 for each wavelength if the absorption change is ',...
'uniform over the volume of tissue measured. To approximate the',...
'partial volume effect of a small localized absorption change within',...
'an adult human head, this value could be as small as 0.1.',...
'OUTPUTS:',...
'dc: the concentration data (#time points x 3 x #SD pairs',...
'3 concentrations are returned (HbO, HbR, HbT)'...
},...

{...
'dod = hmrConc2OD( dc, SD, ppf )',...
'UI NAME:',...
'Conc_to_OD',...
'dod = hmrConc2OD( dc, SD, ppf )',...
'Convert concentrations to OD',...
'INPUTS:',...
'dc: the concentration data (#time points x 3 x #SD pairs',...
'3 concentrations are returned (HbO, HbR, HbT)',...
'SD:  the SD structure',...
'ppf: partial pathlength factors for each wavelength. If there are 2',...
'wavelengths of data, then this is a vector ot 2 elements.',...
'Typical value is ~6 for each wavelength if the absorption change is ',...
'uniform over the volume of tissue measured. To approximate the',...
'partial volume effect of a small localized absorption change within',...
'an adult human head, this value could be as small as 0.1.',...
'OUTPUTS:',...
'dod: the change in OD (#time points x #channels)'...
},...

{...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'UI NAME:',...
'PCA_Filter',...
'[yc, svs, nSV] = enPCAFilter( y, SD, tInc, nSV )',...
'Perform a PCA filter on the data matrix y. ',...
'INPUT:',...
'y: This is the data matrix where rows are time points. If y is wavelength',...
'data, then the columns are channels as described in SD.MeasList. If',...
'y is concentration data, then the third dimension is channels and the',...
'second dimension indicates HbO and HbR.',...
'SD: This is the source detector structure.',...
'tInc: This is a vector of length number of time points and is 1 to',...
'indicate that a time point is included in the analysis and 0 if it is to',...
'be excluded. This is useful for ignoring periods of time with strong',...
'artifacts.',...
'nSV: This is the number of principle components to remove filter from the',...
'data. This can be an integer to indicate the number of components to',...
'remove. Or it can be a fraction less than 1 to indicate that enough',...
'components should be removed to remove up to that fraction of the',...
'variance in the data. If nSV is a single number it is applied to all',...
'channels of data. This is useful for filtering motion artifacts. If it is',...
'2 or more numbers, then it is applied to each wavelength or concentration',...
'separately using the corresponding number.',...
'OUTPUT:',...
'yc: The filtered data matrix.',...
'svs: The singuler value spectrum from the PCA.',...
'nSV: This is the number of components filtered from the data.'...
},...

{...
'[cc,ml,cc_thresh] = enCrossCorrelation( SD, dc0, cc_thresh )',...
'UI NAME:',...
'Cross_Correlation'...
},...

{...
'[dc,iNearestSS] = enAdaptiveFilteringSS(SD,t,dc,M,mu)',...
'UI NAME:',...
'Adaptive_Filtering_ShortSep'...
},...

{...
'[s,nFuncParam,funcParam,funcParamFormat,funcParamVal] = ',...
'enStimIncData_varargin(s,t,userdata,varargin)',...
'UI NAME:',...
'Stim_Include_UserDefVar',...
'Excludes stims based on user-defined variables',...
'INPUTS:',...
's -        s matrix (#time points x #conditions) containing 1 for ',...
'each time point and condition that has a stimulus and ',...
'zeros otherwise.',...
't -        the time vector (#time points x 1)',...
'userdata - table of user-defined variable values based on which stims can ',...
'be excluded.  (# of stims  x  # of user defined variables)',...
'varargin - ranges for each of the user defined variables from userdata.',...
'OUTPUTS:',...
's -             s matrix (#time points x #conditions) ',...
'nFuncParam -    # of user-defined variables from userdata used ',...
'when function is first added to processing stream. ',...
'funcParam -     Names of user-defined variables from userdata',...
'when function is first added to processing stream. ',...
'funcParamFormat - format of user-defined variables from userdata',...
'when function is first added to processing stream. ',...
'funcParamVal -  values of user-defined variables from userdata',...
'when function is first added to processing stream. '...
},...

{...
''...
},...

{...
'[yavg, ystd, tHRF, nTrials, ysum2, yTrials] = hmrBlockAvg( y, s, t, trange )',...
'UI NAME:',...
'Block_Average',...
'Calculate the block average given the stimulation conditions in s over',...
'the time range trange. The baseline of the average is set to zero by',...
'subtracting the mean of the average for t<0. If a stimulus occurs too',...
'close to the start or end of the data such that trange extends outside of',...
'the data range, then the trial is excluded from the average.',...
'INPUTS:',...
'y: this is the data wavelength #time points x #channels',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels',...
's: stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't: time vector corresponding with y and s',...
'trange: defines the range for the block average [tPre tPost]',...
'OUTPUTS:',...
'yavg: the averaged results',...
'ystd: the standard deviation across trials',...
'tHRF: the time vector',...
'nTrials: the number of trials averaged for each condition',...
'yTrials: a structure containing the individual trial responses'...
},...

{...
'[yavg, ystd, tHRF, nTrials, ysum2, yTrials] = hmrBlockAvg( y, s, t, trange )',...
'UI NAME:',...
'Block_Average',...
'Calculate the block average given the stimulation conditions in s over',...
'the time range trange. The baseline of the average is set to zero by',...
'subtracting the mean of the average for t<0. If a stimulus occurs too',...
'close to the start or end of the data such that trange extends outside of',...
'the data range, then the trial is excluded from the average.',...
'INPUTS:',...
'y: this is the data wavelength #time points x #channels',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels',...
's: stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't: time vector corresponding with y and s',...
'trange: defines the range for the block average [tPre tPost]',...
'OUTPUTS:',...
'yavg: the averaged results',...
'ystd: the standard deviation across trials',...
'tHRF: the time vector',...
'nTrials: the number of trials averaged for each condition',...
'yTrials: a structure containing the individual trial responses'...
},...

{...
'[yavg, ystd, tHRF, nTrials, ysum2, yTrials] = hmrBlockAvg( y, s, t, trange )',...
'UI NAME:',...
'Block_Average',...
'Calculate the block average given the stimulation conditions in s over',...
'the time range trange. The baseline of the average is set to zero by',...
'subtracting the mean of the average for t<0. If a stimulus occurs too',...
'close to the start or end of the data such that trange extends outside of',...
'the data range, then the trial is excluded from the average.',...
'INPUTS:',...
'y: this is the data wavelength #time points x #channels',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels',...
's: stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't: time vector corresponding with y and s',...
'trange: defines the range for the block average [tPre tPost]',...
'OUTPUTS:',...
'yavg: the averaged results',...
'ystd: the standard deviation across trials',...
'tHRF: the time vector',...
'nTrials: the number of trials averaged for each condition',...
'yTrials: a structure containing the individual trial responses'...
},...

{...
''...
},...

{...
'[s,yAvg,yStd,nTrials,ysum2,yTrials] = hmrFindHrfOutlier( s, tHRF, yTrials, tRange, stdThresh, minNtrials )',...
'UI NAME:',...
'Find_Outlier_Trials',...
'Find trials that are outliers with respect to the average HRF. Remove',...
'those trials from the stimulus vector s and reaverage the results. The',...
'mean and standard deviation of the trials are found over the time range',...
'specified by tRange. Outliers are trials with a mean deviating more',...
'than stdThresh standard deviations from the mean.',...
'INPUTS:',...
's: The stimulus vector',...
'tHRF: The time vector for the HRF. This is generally returned with the',...
'HRF from hmrBlockAvg() or comparable functions.',...
'yTrials: This structure contains the response to each trial for each',...
'stimulus condition. This structure is obtained from hmrBlockAvg().',...
'tRange: The time range over which the mean is estimated.',...
'stdThresh: The number of standard deviations that a trial must deviate',...
'to be considered an outlier.',...
'minNtrials: Only remove outliers if number of trials for the given ',...
'condition is equal to our greater than this limit.',...
'OUTPUTS:',...
's: the stimulus vector revised for any trials removed as outliers',...
'yAvg: the averaged results',...
'yStd: the standard deviation across trials',...
'nTrials: the number of trials averaged for each condition',...
'ysum2: A variable that enables calculation of the standard deviation',...
'across multiple runs for a single subject.',...
'yTrials: a structure containing the individual trial responses',...
'TO DO:',...
'This function assumes the data is hemoglobin concentrations. The code',...
'should be modified to handle either wavelength data or hemoglobin',...
'concentration data.'...
},...

{...
'function [yresid, yhrf, tHRF, nTrials] = hmrRegressors(yo, yreg, t, trange, nStim )',...
'UI NAME:',...
'Regressors',...
'Calculate the impulse response to different regressors',...
'INPUT',...
'yo - this is the concentration data (#time points x [HbO/HbR/HbT] x',...
'#channels)',...
'yreg - these are the regressors (#time points x #regressors)',...
'the first nStim columns are assumed to be stimulus marks for each ',...
'of nStim independent stimulus conditions',...
't - the time vector (#time points x 1)',...
'trange - the time range for the impulse response function [tpre tpost]',...
'nStim - number of different stimulus conditions presented in yreg',...
'This is optional. If not included then nStim = size(yreg,2)',...
'OUTPUT',...
'yresid - residual of the regressor fit to the data (#time points x',...
'[HbO/Hbr/HbT] x #channels)',...
'yhrf - impulse response to the stimuli (#time points x [HbO/Hbr/HbT] x',...
'#channels x #nStim) ',...
'tHRF - This is the time vector for the estimted HRF',...
'nTrials - number of trials for each stim condition in yreg'...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_3rd(y, s, t, Aaux, trange, gstd, gms)',...
'UI NAME: ',...
'GLM_HRF_Drift',...
'This script estimates the HRF with a set a temporal basis without any',...
'preprocessing but uses a 3rd order polynomial drift regressor',...
'LOG:',...
'created 1-14-2011 L. Gagnon'...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_SS3rd(y, s, t, SD, Aaux, trange, gstd, gms, rhoSD_ssThresh)',...
'UI NAME: ',...
'GLM_HRF_ShortSep_Drift',...
'This script estimates the HRF with a set a temporal basis functions,',...
'regresses the nearest short separation measurement,',...
'and uses a 3rd order polynomial regressor.',...
'The command line function call is',...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_SS3rd(y, s, t, SD, Aaux, trange, gstd, gms, rhoSD_ssThresh)',...
'INPUTS:',...
'y - this is the data wavelength #time points x #channels',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels',...
's - stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't - time vector corresponding with y and s',...
'SD - source detector stucture (units should be consistent with rhoSD_ssThresh)',...
'trange - defines the range for the block average [tPre tPost]',...
'gstd - std for gaussian shape temporal basis function (sec)',...
'gms - mean for gaussian shape temporal basis function (sec)',...
'rhoSD_ssThresh: max distance for a short separation measurement (cm)',...
'OUTPUTS:',...
'yavg - the averaged results',...
'ystd - the standard deviation across trials',...
'tHRF - the time vector',...
'nTrials - the number of trials averaged for each condition',...
'LOG',...
'created 1-14-2011 L. Gagnon'...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_SS3rd(y, s, t, SD, Aaux, trange, gstd, gms, rhoSD_ssThresh)',...
'UI NAME: ',...
'GLM_HRF_ShortSep_Drift',...
'This script estimates the HRF with a set a temporal basis functions,',...
'regresses the nearest short separation measurement,',...
'and uses a 3rd order polynomial regressor.',...
'The command line function call is',...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_SS3rd(y, s, t, SD, Aaux, trange, gstd, gms, rhoSD_ssThresh)',...
'INPUTS:',...
'y - this is the data wavelength #time points x #channels',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels',...
's - stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't - time vector corresponding with y and s',...
'SD - source detector stucture (units should be consistent with rhoSD_ssThresh)',...
'trange - defines the range for the block average [tPre tPost]',...
'gstd - std for gaussian shape temporal basis function (sec)',...
'gms - mean for gaussian shape temporal basis function (sec)',...
'rhoSD_ssThresh: max distance for a short separation measurement (cm)',...
'OUTPUTS:',...
'yavg - the averaged results',...
'ystd - the standard deviation across trials',...
'tHRF - the time vector',...
'nTrials - the number of trials averaged for each condition',...
'LOG',...
'created 1-14-2011 L. Gagnon'...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_SS3rd_Highest(y, s, t, SD, cc, Aaux, trange, gstd, gms, rhoSD_ssThresh)',...
'UI NAME:  ',...
'GLM_HRF_ShortSep_Drift_Highest  ',...
'This script estimates the HRF with a set a temporal basis without any  ',...
'preprocessing but uses a 3rd order polynomial regressor. Also it finds short separations with highest correlation.  ',...
'INPUTS:  ',...
'y: this is the data wavelength #time points x #channels ',...
'concentration data is #time points x [HbO/HbR/HbT] x #channels  ',...
's: stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0  ',...
't: time vector corresponding with y and s .',...
'SD: source detector stucture (units should be consistent with rhoSD_ssThresh)  ',...
'trange: defines the range for the block average [tPre tPost]  ',...
'gstd: std for gaussian shape temporal basis function (sec)  ',...
'gms: mean for gaussian shape temporal basis function (sec)  ',...
'rhoSD_ssThresh: max distance for a short separation measurement (cm)  ',...
'OUTPUTS:  ',...
'yavg: the averaged results  ',...
'ystd: the standard deviation across trials  ',...
'tHRF: the time vector  ',...
'nTrials: the number of trials averaged for each condition ',...
'LOG ',...
'created 1-14-2011 L. Gagnon'...
},...

{...
''...
},...

{...
''...
},...

{...
''...
},...

{...
''...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ysum2] = hmrDeconvTB_3rd(y, s, t, Aaux, trange, gstd, gms)',...
'UI NAME: ',...
'GLM_HRF_Drift',...
'This script estimates the HRF with a set a temporal basis without any',...
'preprocessing but uses a 3rd order polynomial drift regressor',...
'LOG:',...
'created 1-14-2011 L. Gagnon'...
},...

{...
'[yavg, yavgstd, tHRF, nTrials, ynew, yresid, ysum2, beta, R] =',...
'hmrDeconvHRF_DriftSS(y, s, t, SD, Aaux, tIncAuto, trange, glmSolveMethod, idxBasis, paramsBasis, rhoSD_ssThresh, flagSSmethod, driftOrder, flagMotionCorrect )',...
'UI NAME: ',...
'GLM_HRF_Drift_SS',...
'This script estimates the HRF with options to specify the temporal basis',...
'function type and corresponding parameters, whether or not to perform',...
'simultaneous regression of short separation channels, drift order, and',...
'whether or not to correct for motion artifacts. You can also choose the',...
'method for solving the GLM matrix equation.',...
'INPUTS:',...
'y - this is the concentration data with dimensions #time points x [HbO/HbR/HbT] x #channels',...
's - stimulation vector (# time points x #conditions)=1 at stim onset otherwise =0',...
't - time vector corresponding with y and s',...
'SD - source detector stucture (units should be consistent with rhoSD_ssThresh)',...
'Aaux - A matrix of auxilliary regressors (#time points x #Aux channels)',...
'tIncAuto - a vector (#time points x 1) indicating which data time points',...
'are motion (=0) or not (=1)',...
'trange - defines the range for the block average [tPre tPost]',...
'glmSolveMethod - this specifies the GLM solution method to use',...
'1 - use ordinary least squares (Ye et al (2009). NeuroImage, 44(2), 428?447.)',...
'2 - use iterative weighted least squares (Barker,',...
'Aarabi, Huppert (2013). Biomedical optics express, 4(8), 1366?1379.)',...
'Note that we suggest driftOrder=0 for this method as',...
'otherwise it can produce spurious results.',...
'idxBasis - this specifies the type of basis function to use for the HRF',...
'1 - a consecutive sequence of gaussian functions',...
'2 - a modified gamma function convolved with a square-wave of',...
'duration T. Set T=0 for no convolution.',...
'The modified gamma function is',...
'(exp(1)*(t-tau).^2/sigma^2) .* exp(-(tHRF-tau).^2/sigma^2)',...
'3 - a modified gamma function and its derivative convolved',...
'with a square-wave of duration T. Set T=0 for no convolution.',...
'4-  GAM function from 3dDeconvolve AFNI convolved with',...
'a square-wave of duration T. Set T=0 for no convolution.',...
'(t/(p*q))^p * exp(p-t/q) ',...
'Defaults: p=8.6 q=0.547 ',...
'The peak is at time p*q.  The FWHM is about 2.3*sqrt(p)*q.',...
'paramsBasis - Parameters for the basis function depends on idxBasis',...
'idxBasis=1 - [stdev step] where stdev is the width of the',...
'gaussian and step is the temporal spacing between',...
'consecutive gaussians',...
'idxBasis=2 - [tau sigma T] applied to both HbO and HbR',...
'or [tau1 sigma1 T1 tau2 sigma2 T2] ',...
'where the 1 (2) indicates the parameters for HbO (HbR).',...
'idxBasis=3 - [tau sigma T] applied to both HbO and HbR',...
'or [tau1 sigma1 T1 tau2 sigma2 T2] ',...
'where the 1 (2) indicates the parameters for HbO (HbR).',...
'idxBasis=4 - [p q T] applied to both HbO and HbR',...
'or [p1 q1 T1 p2 q2 T2] ',...
'where the 1 (2) indicates the parameters for HbO (HbR).',...
'rhoSD_ssThresh - max distance for a short separation measurement. Set =0',...
'if you do not want to regress the short separation measurements.',...
'Follows the static estimate procedure described in Gagnon et al (2011). ',...
'NeuroImage, 56(3), 1362?1371.',...
'flagSSmethod - 0 if short separation regression is performed with the nearest ',...
'short separation channel.',...
'1 if performed with the short separation channel with the ',...
'greatest correlation. ',...
'2 if performed with average of all short separation channels.',...
'driftOrder - Polynomial drift correction of this order',...
'flagMotionCorrect - set to 1 to baseline correct between motion epochs indicated in tIncAuto, otherwise set to 0 ',...
'gstd - std for gaussian shape temporal basis function (sec)',...
'gms - mean for gaussian shape temporal basis function (sec)',...
'OUTPUTS:',...
'yavg - the averaged results',...
'ystd - the standard deviation across trials',...
'tHRF - the time vector',...
'nTrials - the number of trials averaged for each condition',...
'ynew - the model of the HRF with the residual. That is, it is the data y',...
'with the nuasance model parameters removed.',...
'yresid - the residual between the data y and the GLM fit',...
'ysum2 - an intermediate matrix for calculating stdev across runs',...
'beta - the coefficients of the temporal basis function fit for the HRF',...
'(#coefficients x HbX x #Channels x #conditions)',...
'R - the correlation coefficient of the GLM fit to the data ',...
'(#Channels x HbX)',...
'LOG:'...
}...
};
