#include "vtkKWMeshVisu.h"
#include "KWMeshVisuAttributeIO.h"

#include "vtkKWApplication.h"
#include "vtkKWFrame.h"
#include "vtkKWColorTransferFunctionEditor.h"
#include "vtkKWMenuButton.h"
#include "vtkKWMenuButtonWithSpinButtons.h"
#include "vtkKWMenuButtonWithSpinButtonsWithLabel.h"
#include "vtkKWNotebook.h"
#include "vtkKWRenderWidget.h"
#include "vtkKWWindow.h"
#include "vtkObjectFactory.h"
#include "vtkKWMenu.h"
#include "vtkToolkits.h"
#include "vtkColorTransferFunction.h"
#include "vtkKWPushButtonSet.h"
#include "vtkKWPushButton.h"
#include "vtkKWLoadSaveDialog.h"
#include "vtkKWScaleWithEntry.h"
#include "vtkKWScale.h"
#include "vtkKWMessageDialog.h"
#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWListBoxWithScrollbarsWithLabel.h"
#include "vtkKWListBoxWithScrollbars.h"
#include "vtkKWListBox.h"
#include "vtkKWFrameWithLabel.h"
#include "vtkKWLabel.h"
#include <vtksys/SystemTools.hxx>
#include <vtksys/CommandLineArguments.hxx>

// ITK-includes
#include <itkDefaultDynamicMeshTraits.h>
#include <itkSceneSpatialObject.h>
#include <itkMetaMeshConverter.h>
#include <itkMeshTovtkPolyData.h>

// VTK-includes
#include "vtkPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkActor.h"
#include "vtkPointData.h"
#include "vtkFloatArray.h"
#include "vtkBitArray.h"
#include "vtkPolyDataNormals.h"
#include "vtkGlyph3D.h"
#include "vtkTensorGlyph.h"
#include "vtkArrowSource.h"
#include "vtkSphereSource.h"
#include "vtkAssignAttribute.h"
#include "vtkProperty.h"
#include "vtkTransform.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkRenderer.h"
#include "vtkCamera.h"
#include "vtkBMPWriter.h"
#include "vtkWindowToImageFilter.h"
#include "vtkPolyLine.h"
#include "vtkUnstructuredGrid.h"
#include "vtkDataSetMapper.h"
#include "vtkPolyDataNormals.h"
#include "vtkPolyDataWriter.h"
#include "vtkPolyDataReader.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkCommand.h"
#include "vtkCallbackCommand.h"
#include "vtkRenderWindow.h"
#include "vtkPointPicker.h"
#include "vtkXMLPolyDataReader.h"

#include <assert.h>
#include <string.h>

#include "argio.hh"

//----------------------------------------------------------------------------
vtkStandardNewMacro( vtkKWMeshVisu );
vtkCxxRevisionMacro(vtkKWMeshVisu, "$Revision: 1.45 $");

//----------------------------------------------------------------------------

vtkKWMeshVisu::vtkKWMeshVisu ()
{
  this->vtkMesh = 0 ;
  this->meshMapper = vtkPolyDataMapper::New();
  this->meshActor = 0 ;
  this->normalFilter = vtkPolyDataNormals::New ();
  
  this->arrowGlyphMapper = vtkPolyDataMapper::New();
  this->arrowGlyphActor = 0 ;
  this->arrowGlyphFilter = vtkGlyph3D::New();
  
  this->arrowSource = vtkArrowSource::New();
  
  this->ellipsoidGlyphMapper = vtkPolyDataMapper::New();
  this->ellipsoidGlyphActor = 0 ;
  this->ellipsoidGlyphFilter = vtkTensorGlyph::New();
  this->ellipsoidSource = vtkSphereSource::New();
  
  this->opacityScale = vtkKWScaleWithEntry::New();
  this->featureScale = vtkKWScaleWithEntry::New();
  
  this->attributeDimension = 0 ;
  this->FixedTFunc = 0 ;
  
  this->commandLine = false ;

  this->aPolyLineActor = vtkActor::New() ;

  this->ROIMaskEnabled = false ;
  this->ROIBackPickEnabled = false ; 
  this->ROIMask = vtkBitArray::New () ;
  this->ROIMask->SetNumberOfTuples ( 0 ) ;
  this->ROImaskedScalars = vtkFloatArray::New () ;
  this->ROImaskedScalars->SetNumberOfTuples ( 0 ) ;
  this->currentScalarRange[0] = -1 ;
  this->currentScalarRange[1] = 1 ;
  
  this->nParticleSets = 0 ;
  this->particleSets = new vtkPolyData * ;
  this->particleMapper = vtkPolyDataMapper::New();
  this->particleActor = vtkActor::New();
  this->particleActor->SetMapper (this->particleMapper);
  this->particleScalars = vtkFloatArray::New()  ;
}

vtkKWMeshVisu::~vtkKWMeshVisu()
{
  if ( this->vtkMesh )
    this->vtkMesh->Delete () ;
  
  if ( this->meshActor )
    this->meshActor->Delete() ;
  
  if ( this->arrowGlyphActor )
    this->arrowGlyphActor->Delete ();
  
  if ( this->ellipsoidGlyphActor )
    this->ellipsoidGlyphActor->Delete ();
  
  if ( this->particleActor )
    this->particleActor->Delete ();
  
  if ( this->aPolyLineActor )
    this->aPolyLineActor->Delete() ;

  this->meshMapper->Delete();
  this->normalFilter->Delete();
  this->arrowGlyphMapper->Delete();
  this->arrowGlyphFilter->Delete();
  this->arrowSource->Delete();
  this->ellipsoidGlyphMapper->Delete();
  this->ellipsoidGlyphFilter->Delete();
  this->ellipsoidSource->Delete();
  this->particleMapper->Delete();
  
  this->opacityScale->Delete();
  this->featureScale->Delete();
  
  if ( this->FixedTFunc ) 
    this->FixedTFunc->Delete();

  this->ROIMask->Delete () ;

  if ( this->commandLine )
  {
    this->win->Close();
    this->RenderWidget->Delete();
    this->win->Delete();
    this->DistanceMapTFunc->Delete();
    this->PValueTFunc->Delete();
    this->SignedDistanceMapTFunc->Delete () ;
    this->ROIMapTFunc->Delete() ;
  }

}

int vtkKWMeshVisu::Run(int argc, char *argv[])
{
  vtkKWChangeColorButton *ccb ;
  vtkKWMenuButtonWithSpinButtonsWithLabel *colormode_menubutton;
  vtkKWApplication *app ;

  // Process some command-line arguments 
  // The --test option here is used to run this example as a non-interactive 
  // test for software quality purposes. You can ignore it.
  
  int option_test = 0;
  vtksys::CommandLineArguments args;
  args.Initialize(argc, argv);
  args.AddArgument(
    "--test", vtksys::CommandLineArguments::NO_ARGUMENT, &option_test, "");
  if ( argc == 1 )  args.Parse();
  
  // Create the application
  // If --test was provided, ignore all registry settings, and exit silently
  // Restore the settings that have been saved to the registry, like
  // the geometry of the user interface so far.
  app = this->GetApplication();
  app->SetName("KWMeshVisu");
  if (option_test)
  {
    app->SetRegistryLevel(0);
    app->PromptBeforeExitOff();
  }
  app->RestoreApplicationSettingsFromRegistry();
  

  // Add a window
  // Set 'SupportHelp' to automatically add a menu entry for the help link
  
  this->win = vtkKWWindow::New();
  this->win->SupportHelpOn();
  app->AddWindow(this->win);
  this->win->Create();
  this->win->SecondaryPanelVisibilityOff();
 
  if ( this->commandLine ) 
  {
     this->win->MainPanelVisibilityOff() ;
  } 

  if ( !this->commandLine )
  {
  
     this->win->MainPanelVisibilityOn() ;
    // Set a help link. Can be a remote link (URL), or a local file
    app->SetHelpDialogStartingPage("http://www.kwwidgets.org");
  }
  
  // Populate file menu
  if ( !this->commandLine )
  {
    vtkKWMenu *file_menu = this->win->GetFileMenu() ;
    file_menu->InsertCommand(0, "Load Mesh", this, "LoadMeshCallback");
    file_menu->InsertCommand(1, "Load 1D Attribute", this, "Load1DAttributeCallback");
    file_menu->InsertCommand(2, "Load 3D Attribute", this, "Load3DAttributeCallback");
    file_menu->InsertCommand(3, "Load Ellipsoid Field", this, "LoadEllipsoidFieldCallback");
    file_menu->InsertCommand(4, "Load Curve", this, "LoadCurveCallback");
    file_menu->InsertCommand(5, "Load ROI", this, "LoadROICallback");
    file_menu->InsertCommand(6, "Save ROI", this, "SaveROICallback");
    file_menu->InsertCommand(7, "Clean Attributes", this, "CleanAttributeCallback");
    file_menu->InsertCommand(8, "Load Attribute Mesh", this, "LoadAttributeMeshCallback");
    file_menu->InsertCommand(9, "Save Attribute Mesh", this, "SaveAttributeMeshCallback");
    file_menu->InsertCommand(10, "Get Camera", this, "GetCameraCallback");
    file_menu->InsertCommand(11, "Save Colored Surface", this, "SaveColorsCallback");
    file_menu->InsertCommand(12, "Load Particle Set", this, "LoadParticleSetCallback");
    file_menu->InsertCommand(13, "Make Video", this, "MakeVideoCallback");

    file_menu->InsertSeparator (14) ;
  }
  
  // Add a render widget, attach it to the view frame, and pack
  
  this->RenderWidget = vtkKWRenderWidget::New();
  this->RenderWidget->SetParent(this->win->GetViewFrame());
  this->RenderWidget->Create();
  this->RenderWidget->SetRendererBackgroundColor(0, 0, 0);
  /*if ( argc != 1 )
  {
    this->RenderWidget->OffScreenRenderingOn();
  }*/

  if ( ipExistsArgument ( (const char **)argv, "--quit" ) )
  {
    this->RenderWidget->OffScreenRenderingOn() ;
  }
  this->SetupInteractor () ;

  app->Script("pack %s -expand y -fill both -anchor c -expand y", 
    this->RenderWidget->GetWidgetName());
  
  if ( !this->commandLine )
  {
    // Create a menu button to control the orientation
    
    colormode_menubutton = vtkKWMenuButtonWithSpinButtonsWithLabel::New();
    
    colormode_menubutton->SetParent(this->win->GetMainPanelFrame());
    colormode_menubutton->Create();
    colormode_menubutton->SetLabelText("Visualization Mode:");
    colormode_menubutton->SetPadX(2);
    colormode_menubutton->SetPadY(2);
    colormode_menubutton->SetBorderWidth(2);
    colormode_menubutton->SetReliefToGroove();
    
    app->Script("pack %s -side top -anchor nw -expand n -fill x -padx 2 -pady 2",
      colormode_menubutton->GetWidgetName());
    
    this->ColorMode = colormode_menubutton->GetWidget()->GetWidget();
    this->ColorMode->GetMenu()->AddRadioButton("P Value", this, "PValueCallback");
    this->ColorMode->GetMenu()->AddRadioButton("Distance Map", this, "DistanceMapCallback");
    this->ColorMode->GetMenu()->AddRadioButton("Signed Distance", this, "SignedDistanceMapCallback");
    this->ColorMode->GetMenu()->AddRadioButton("ROI", this, "ROIMapCallback") ;
    //this->ColorMode->GetMenu()->SetItemStateToDisabled ( roiIndex ) ;

    // Create a transfer function editor
    
    this->TFuncEditor = vtkKWColorTransferFunctionEditor::New();
    this->TFuncEditor->SetParent(this->win->GetMainPanelFrame());
    this->TFuncEditor->Create();
    this->TFuncEditor->SetPadX(2);
    this->TFuncEditor->SetPadY(2);
    this->TFuncEditor->SetBorderWidth(2);
    this->TFuncEditor->SetReliefToGroove();
    
    this->TFuncEditor->DisableAddAndRemoveOn();
    this->TFuncEditor->MidPointVisibilityOff();
    this->TFuncEditor->ParameterTicksVisibilityOn();
    this->TFuncEditor->SetRangeLabelPositionToTop();
    this->TFuncEditor->SetLabelPositionToTop();
    
    this->TFuncEditor->SetFunctionChangingCommand (this->RenderWidget, "Render" );
    this->TFuncEditor->SetFunctionChangedCommand (this->RenderWidget, "Render" );
    
    app->Script("pack %s -side top -anchor nw -expand n -fill x  -padx 2 -pady 2",
      this->TFuncEditor->GetWidgetName());


  }

  // Create default color transfer functions for pvalue and distance map
  
  this->PValueTFunc = vtkColorTransferFunction::New();
  this->PValueTFunc->SetColorSpaceToHSV();
  this->DistanceMapTFunc = vtkColorTransferFunction::New();
  this->DistanceMapTFunc->SetColorSpaceToHSV();
  this->SignedDistanceMapTFunc = vtkColorTransferFunction::New();
  this->SignedDistanceMapTFunc->SetColorSpaceToHSV();
  this->ROIMapTFunc = vtkColorTransferFunction::New () ;
  this->ROIMapTFunc->SetColorSpaceToHSV() ;

  if ( !this->commandLine ) 
  {
    this->PValueTFunc->AddHSVPoint(0, 0, 1, 1);
    this->PValueTFunc->AddHSVPoint(0.05, 0.27, 0.6, 0.97, 0, 1);
    //this->PValueTFunc->AddHSVPoint(0.05, 0.27, 0.6, 0.97);
    this->PValueTFunc->AddHSVPoint(1, 0.6, 1, 1);
    
    this->UpdateTFunc();

    // Set default color mode
    this->ColorMode->SetValue("P Value");
    this->PValueCallback ();
    
    this->opacityScale->SetParent(this->win->GetMainPanelFrame());
    this->opacityScale->Create();
    this->opacityScale->SetRange(0.0, 1.0);
    this->opacityScale->SetResolution(0.01);
    this->opacityScale->GetScale()->SetLength(350);
    this->opacityScale->SetValue(1.0);
    this->opacityScale->SetLabelText("Mesh Opacity:");
    this->opacityScale->SetLabelPositionToTop();
    this->opacityScale->SetEntryPositionToTop();
    this->opacityScale->RangeVisibilityOn();
    this->opacityScale->SetCommand(this, "OpacityCallback");
    
    app->Script("pack %s -side top -anchor nw -expand n -fill none -padx 2 -pady 6", 
      this->opacityScale->GetWidgetName());
    
    this->featureScale->SetParent(this->win->GetMainPanelFrame());
    this->featureScale->Create();
    this->featureScale->SetRange(0.0, 20.0);
    this->featureScale->SetResolution(0.01);
    this->featureScale->GetScale()->SetLength(350);
    this->featureScale->SetValue(1.0);
    this->featureScale->SetLabelText("Feature scale:");
    this->featureScale->SetLabelPositionToTop();
    this->featureScale->SetEntryPositionToTop();
    this->featureScale->RangeVisibilityOn();
    this->featureScale->SetCommand(this, "FeatureScaleCallback");
    
    app->Script("pack %s -side top -anchor nw -expand n -fill none -padx 2 -pady 6", 
      this->featureScale->GetWidgetName());
    
    // change background button
    ccb = vtkKWChangeColorButton::New();
    ccb->SetParent(this->win->GetMainPanelFrame());
    ccb->Create();
    ccb->SetColor(1.0, 0.0, 0.0);
    ccb->SetLabelPositionToLeft();
    ccb->SetLabelText("Set Background Color");
    ccb->SetCommand(this->RenderWidget, "SetRendererBackgroundColor");
    ccb->SetColor(0, 0, 0);
    
    app->Script(
      "pack %s -side top -anchor nw -expand n -padx 2 -pady 2", 
      ccb->GetWidgetName());

    //ROI selection
    vtkKWFrameWithLabel *roiFrame = vtkKWFrameWithLabel::New () ;
    roiFrame->SetParent ( this->win->GetMainPanelFrame() ) ;
    roiFrame->Create () ;
    roiFrame->SetLabelText ( "ROI Selection" ) ;
    roiFrame->SetWidth ( 300 ) ;
    roiFrame->SetHeight ( 100 ) ;
    app->Script ( "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", roiFrame->GetWidgetName() ) ;

    vtkKWLabel *roiLabel = vtkKWLabel::New () ;
    roiLabel->SetParent ( roiFrame->GetFrame () ) ;
    roiLabel->Create () ;
    roiLabel->SetText ( "Press 'Q' to add the vertex closest to the mouse cursor to the selected ROI" ) ;
    roiLabel->AdjustWrapLengthToWidthOn () ;
    app->Script ( "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", roiLabel->GetWidgetName() ) ;

    this->ROIMaskCheckBox = vtkKWCheckButton::New () ; 
    this->ROIMaskCheckBox->SetParent ( roiFrame->GetFrame() ) ;
    this->ROIMaskCheckBox->Create() ;
    this->ROIMaskCheckBox->SetText ( "Mask using ROI" ) ;
    this->ROIMaskCheckBox->SetCommand ( this, "ROIMaskCheckBoxCallback" ) ;
    app->Script ( "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", this->ROIMaskCheckBox->GetWidgetName () ) ;

    roiFrame->AddCallbackCommandObserver ( this->ROIMaskCheckBox, vtkKWCheckButton::SelectedStateChangedEvent ) ;
    
    this->ROIDisableBackPickCheckBox = vtkKWCheckButton::New () ;
    this->ROIDisableBackPickCheckBox->SetParent ( roiFrame->GetFrame () ) ;
    this->ROIDisableBackPickCheckBox->Create () ;
    this->ROIDisableBackPickCheckBox->SetText ( "Back-picking" ) ;
    this->ROIDisableBackPickCheckBox->SetBalloonHelpString ( "Enable/disable picking points from the backward-facing surfaces" ) ;
    this->ROIDisableBackPickCheckBox->SetCommand ( this, "ROIDisableBackPickCheckBoxCallback" ) ;
    app->Script ( "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", this->ROIDisableBackPickCheckBox->GetWidgetName () ) ;

    roiFrame->AddCallbackCommandObserver ( this->ROIDisableBackPickCheckBox, vtkKWCheckButton::SelectedStateChangedEvent ) ;

    // Particle Visualization
    vtkKWFrameWithLabel *particleFrame = vtkKWFrameWithLabel::New();
    particleFrame->SetParent(this->win->GetMainPanelFrame());
    particleFrame->Create();
    particleFrame->SetLabelText("Particle Visualization");
    particleFrame->SetWidth(300);
    particleFrame->SetHeight(100);
    
    app->Script(
      "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", 
      particleFrame->GetWidgetName());
    
    this->ParticleVisualizationCheckBox = vtkKWCheckButton::New();
    this->ParticleVisualizationCheckBox->SetParent(particleFrame->GetFrame());
    this->ParticleVisualizationCheckBox->Create();
    this->ParticleVisualizationCheckBox->SetText("Particle Visualization Enabled");
    this->ParticleVisualizationCheckBox->SetCommand (this, "ParticleVisualizationCheckBoxCallback") ;

    particleFrame->AddCallbackCommandObserver(this->ParticleVisualizationCheckBox, vtkKWCheckButton::SelectedStateChangedEvent);

    app->Script(
      "pack %s -side top -anchor nw -expand n -padx 2 -pady 2", 
      this->ParticleVisualizationCheckBox->GetWidgetName());

    
    this->ParticleListBox = vtkKWListBoxWithScrollbarsWithLabel::New();
    this->ParticleListBox->SetParent(particleFrame->GetFrame());
    this->ParticleListBox->Create();
    this->ParticleListBox->SetBorderWidth(2);
    this->ParticleListBox->SetReliefToGroove();
    this->ParticleListBox->SetPadX(2);
    this->ParticleListBox->SetPadY(2);
    this->ParticleListBox->GetWidget()->GetWidget()->SetSelectionModeToSingle () ;
    this->ParticleListBox->GetWidget()->GetWidget()->ExportSelectionOff();
    this->ParticleListBox->SetLabelText("Active Particle Set") ;
    this->ParticleListBox->SetLabelPositionToTop() ;
    this->ParticleListBox->ExpandWidgetOn () ;
    this->ParticleListBox->GetWidget()->GetWidget()->SetSelectionCommand ( this, "ParticleListBoxCallback" ) ;

    app->Script(
      "pack %s -side top -anchor nw -expand n -padx 2 -pady 6", 
      this->ParticleListBox->GetWidgetName());
    
    vtkKWPushButton *pb = vtkKWPushButton::New();
    pb->SetParent(particleFrame->GetFrame());
    pb->Create();
    pb->SetText("Load Particle Scalars");
    pb->SetCommand(this, "LoadParticleScalarsCallback");    

    app->Script(
      "pack %s -side top -anchor nw -expand n -padx 2 -pady 2", 
      pb->GetWidgetName());
  }
  // Start the application
  // If --test was provided, do not enter the event loop and run this example
  // as a non-interactive test for software quality purposes.

  int ret = 0;
  this->win->Display();
  if ((!option_test) && (!this->commandLine) && ( argc == 1 ) )
  {
    app->Start(argc, argv);
    ret = app->GetExitStatus();
    this->win->Close();
  
    // Deallocate and exit
    ccb->Delete();
    colormode_menubutton->Delete();
    this->RenderWidget->Delete();
    this->TFuncEditor->Delete();
    this->SignedDistanceMapTFunc->Delete();
    this->DistanceMapTFunc->Delete();
    this->PValueTFunc->Delete();
    this->ROIMapTFunc->Delete() ;
    this->win->Delete();
  }
  
  return ret;
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadMeshCallback()
{
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Mesh");
  dialog->SetFileTypes ("{{ITK Mesh File} {.meta}} {{VTK PolyData File} {.vtk}} {{VTK XML PolyData File} {.vtp}}");
  dialog->SetDefaultExtension (".meta");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadMesh (dialog->GetFileName());
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::Load1DAttributeCallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the attributes 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load 1D Attributes");
  dialog->SetFileTypes ("{{Attribute File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadScalarField ( dialog->GetFileName() );
  dialog->Delete ();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::Load3DAttributeCallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the attributes 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load 3D Attributes");
  dialog->SetFileTypes ("{{Attribute File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadVectorField ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadCurveCallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the attributes 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Curve");
  dialog->SetFileTypes ("{{Attribute File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadCurve( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadROICallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the points 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load ROI");
  dialog->SetFileTypes ("{{ROI File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadROI ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

void vtkKWMeshVisu::SaveROICallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the points 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Save ROI");
  dialog->SetFileTypes ("{{ROI File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  dialog->SaveDialogOn () ;
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->SaveROI ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadEllipsoidFieldCallback()
{
  if ( this->vtkMesh == 0 ) return ;
  
  // get the file name for reading the attributes 
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Ellipsoid Field");
  dialog->SetFileTypes ("{{Attribute File} {.txt}}");
  dialog->SetDefaultExtension (".txt");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadEllipsoidField ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::CleanAttributesCallback()
{
  if ( this->attributeDimension == 0 ) 
    return ;
  
  this->vtkMesh->GetPointData()->Reset();
  
  this->meshMapper->ScalarVisibilityOff ();
  
  vtkDataArray *vectors, *tensors ;
  vectors = this->vtkMesh->GetPointData()->GetVectors() ;
  tensors = this->vtkMesh->GetPointData()->GetTensors() ;
  if ( vectors )
  {
    this->RenderWidget->RemoveViewProp(this->arrowGlyphActor);
  }
  if ( tensors )
  {
    this->RenderWidget->RemoveViewProp(this->ellipsoidGlyphActor);
  }

  this->ParticleVisualizationCheckBox->Deselect () ;
  
  this->attributeDimension = 0 ;
  this->featureScale->SetValue(1) ;
  
  this->ColorMode->SetValue("P Value");
  this->PValueCallback ();
  
  this->RenderWidget->Render();
}


//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadAttributeMeshCallback()
{
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Attributed Mesh");
  dialog->SetFileTypes ("{{VTK PolyData File} {.vtk}}");
  dialog->SetDefaultExtension (".vtk");
  
  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  
  vtkPolyDataReader *reader = vtkPolyDataReader::New () ;
  reader->SetFileName ( dialog->GetFileName() ) ;
  reader->Update () ;
  vtkPolyData *tempMesh = reader->GetOutput() ;
  if ( this->vtkMesh ) 
    this->vtkMesh->Delete () ;
  this->vtkMesh = vtkPolyData::New () ;
  this->vtkMesh->DeepCopy ( tempMesh ) ;
  reader->Delete () ;
    
  // make the final connections
  this->normalFilter->SetInput (this->vtkMesh);
  
  // setup vtk rendering
  this->meshMapper->SetInput(this->normalFilter->GetOutput());
  this->meshMapper->InterpolateScalarsBeforeMappingOff () ;
  this->meshMapper->Update() ;

  if ( this->meshActor ) 
    this->meshActor->Delete () ;
  this->meshActor = vtkActor::New();
  this->meshActor->SetMapper(this->meshMapper);
  
  // setup the vector field visualization
  this->arrowGlyphFilter->SetInput(this->meshMapper->GetInput());
  this->arrowGlyphFilter->SetSource(this->arrowSource->GetOutput());
  this->arrowGlyphFilter->SetScaleModeToScaleByVector();
  this->arrowGlyphFilter->SetVectorModeToUseVector();
  this->arrowGlyphFilter->SetColorModeToColorByVector();
  this->arrowGlyphFilter->OrientOn ();
  this->arrowGlyphMapper->SetInput(this->arrowGlyphFilter->GetOutput());
  if ( this->arrowGlyphActor )
    this->arrowGlyphActor->Delete () ;
  this->arrowGlyphActor = vtkActor::New();
  this->arrowGlyphActor->SetMapper (this->arrowGlyphMapper);
  
  // setup the ellipsoid field visualization
  this->ellipsoidGlyphFilter->SetInput(this->meshMapper->GetInput());
  this->ellipsoidGlyphFilter->SetSource(this->ellipsoidSource->GetOutput());
  this->ellipsoidGlyphFilter->SetScaleFactor (1);
  this->ellipsoidGlyphFilter->ThreeGlyphsOff () ;
  this->ellipsoidGlyphFilter->ColorGlyphsOn () ;
  this->ellipsoidGlyphFilter->SetColorModeToEigenvalues();
  this->ellipsoidGlyphFilter->ExtractEigenvaluesOff();

  // make sure the normals for the ellipsoids are in the correct direction, 
  // to be able to color them properly
  vtkPolyDataNormals *normals = vtkPolyDataNormals::New() ;
  normals->SetInput (this->ellipsoidGlyphFilter->GetOutput()) ;
  this->ellipsoidGlyphMapper->SetInput(normals->GetOutput());
  normals->Delete () ;
  
  if ( this->ellipsoidGlyphActor ) 
    this->ellipsoidGlyphActor->Delete () ;
  this->ellipsoidGlyphActor = vtkActor::New();
  this->ellipsoidGlyphActor->SetMapper (this->ellipsoidGlyphMapper);
  
  this->RenderWidget->AddViewProp (this->meshActor) ;
  
  this->RenderWidget->ResetCamera();
  
  this->attributeDimension = 1 ;
  
  this->UpdateTFunc();  

  this->meshMapper->ScalarVisibilityOn ();
  
  // make sure we're not drawing arrows or spheres
  this->RenderWidget->RemoveViewProp(this->arrowGlyphActor);
  this->RenderWidget->RemoveViewProp(this->ellipsoidGlyphActor);
  
  if ( !this->commandLine )
  {
    this->ColorMode->SetValue("P Value");
  }
  this->PValueCallback ();
  
  
  this->RenderWidget->Render();
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::SaveAttributeMeshCallback()
{
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Attributed Mesh");
  dialog->SetFileTypes ("{{VTK PolyData File} {.vtk}}");
  dialog->SetDefaultExtension (".vtk");
  dialog->SaveDialogOn() ;

  if ( dialog->Invoke () == 0 )
  {  
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  
  vtkPolyDataWriter *writer = vtkPolyDataWriter::New () ;
  writer->SetInput ( this->vtkMesh ) ;
  writer->SetFileName ( dialog->GetFileName() ) ;
  writer->Update () ;
  
  dialog->Delete();
}


//----------------------------------------------------------------------------
void vtkKWMeshVisu::SaveColorsCallback()
{
  if ( ! this->vtkMesh ) 
    return ;

  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Save Colored Surface");
  dialog->SetFileTypes ("{{VTK PolyData File} {.vtk}}");
  dialog->SetDefaultExtension (".vtk");
  dialog->SaveDialogOn() ;

  if ( dialog->Invoke () == 0 )
  {  
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->SaveColors ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadParticleSetCallback()
{
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Particle Set");
  dialog->SetFileTypes ("{{Particle Set} {.lpts}} {{Particle Set} {.wpts}}");
  dialog->SetDefaultExtension (".lpts");

  if ( dialog->Invoke () == 0 )
  {  
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadParticleSet ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::LoadParticleScalarsCallback()
{
  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;
  
  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Load Particle Scalars");
  dialog->SetFileTypes ("{{Particle Scalars} {.txt}}");
  dialog->SetDefaultExtension (".txt");

  if ( dialog->Invoke () == 0 )
  {  
    dialog->Delete();
    return ;
  }
  
  dialog->SaveLastPathToRegistry("DataPath");
  
  this->LoadParticleScalars ( dialog->GetFileName() ) ;
  
  dialog->Delete();
}

void vtkKWMeshVisu::ROIMaskCheckBoxCallback ( int state ) 
{
  if ( this->commandLine ) return ;
  this->ROIMaskEnabled = this->ROIMaskCheckBox->GetSelectedState() ;
  if ( this->ROIMaskEnabled ) 
    {
      this->vtkMesh->GetPointData()->SetScalars ( this->ROImaskedScalars ) ;
      this->ColorMode->SetValue ( "ROI" ) ;
      this->ROIMapCallback () ;
    }
  else
    {
      this->vtkMesh->GetPointData()->SetScalars ( this->unmaskedScalars ) ;
    }
  this->UpdateTFunc () ;
  this->RenderWidget->Render () ;
}

void vtkKWMeshVisu::ROIDisableBackPickCheckBoxCallback ( int state ) 
{
  this->ROIBackPickEnabled = this->ROIDisableBackPickCheckBox->GetSelectedState () ;
}

void vtkKWMeshVisu::ResetROIMask () 
{
  this->ROIMask->Resize ( 0 ) ; 
  if ( !this->vtkMesh ) 
    return ;
  bool scalars = this->attributeDimension == 1 ;
  unsigned long int nVert = this->vtkMesh->GetNumberOfPoints () ;
  this->ROIMask->SetNumberOfValues ( nVert ) ;
  this->ROImaskedScalars->SetNumberOfValues ( nVert ) ;
  for ( unsigned long int i = 0 ; i < nVert ; i++ )
    {
      this->ROIMask->SetValue ( i, 0 ) ;
      if ( scalars ) 
	{
	  this->ROImaskedScalars->SetTuple1 ( i, this->unmaskedScalars->GetTuple1 ( i ) ) ; 
	}
      else
	{
	  this->ROImaskedScalars->SetTuple1 ( i, -1 ) ;
	}
    }
}

void vtkKWMeshVisu::UpdateROIMaskedScalars ( int id ) 
{
  bool inROI = this->ROIMask->GetValue ( id ) ;
  bool scalars = this->attributeDimension == 1 ;

  if ( !scalars ) 
    {
      if ( inROI ) 
	this->ROImaskedScalars->SetTuple1 ( id, 1 ) ;
      else
	this->ROImaskedScalars->SetTuple1 ( id, -1 ) ;
    }
  else
    {
      double f, d ;
      f = this->unmaskedScalars->GetTuple1 ( id ) ;
      d = this->currentScalarRange[1] - this->currentScalarRange[0] ;
      if ( inROI ) 
	this->ROImaskedScalars->SetTuple1 ( id, f - d ) ;
      else
	this->ROImaskedScalars->SetTuple1 ( id, f ) ;
    }
  this->ROImaskedScalars->Modified () ;
}


//----------------------------------------------------------------------------
void vtkKWMeshVisu::ParticleVisualizationCheckBoxCallback (int state)
{
  if ( this->ParticleVisualizationCheckBox->GetSelectedState () )
  {
    this->RenderWidget->AddViewProp(this->particleActor);
    this->RenderWidget->ResetCamera();
  }
  else
  {
    this->RenderWidget->RemoveViewProp ( this->particleActor ) ;
  }
  this->RenderWidget->Render() ;
}

void vtkKWMeshVisu::ParticleListBoxCallback () 
{
  int nOldPoints ;
  nOldPoints = this->particleScalars->GetNumberOfTuples () ;

  this->particleMapper->SetInput(this->particleSets[this->ParticleListBox->GetWidget()->GetWidget()->GetSelectionIndex()]);
  
  int nPoints = this->particleSets[this->ParticleListBox->GetWidget()->GetWidget()->GetSelectionIndex()]->GetNumberOfVerts() ;

  if ( nPoints != nOldPoints ) 
    this->ResetParticleScalars () ;

  this->RenderWidget->Render() ;
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::PValueCallback()
{
  if ( this->commandLine ) 
  {
    return ;
  }
  this->TFuncEditor->SetColorTransferFunction (this->PValueTFunc) ;
  this->TFuncEditor->LockEndPointsParameterOn();
  this->TFuncEditor->SetLabelText ("P-Value Color Map");
  
  // if we are not in 1D attribute mode, leave the 1D visualization alone
  if ( this->attributeDimension == 1 )
  {
    this->meshMapper->SetLookupTable (this->PValueTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
  }
  
  this->arrowGlyphMapper->SetLookupTable (this->PValueTFunc) ;
  this->ellipsoidGlyphMapper->SetLookupTable (this->PValueTFunc) ;
  this->particleMapper->SetLookupTable (this->PValueTFunc) ;
  //this->meshMapper->ScalarVisibilityOn ();
  
  
  this->TFuncEditor->SetWholeParameterRange(0.0, 1.0);
  this->RenderWidget->Render();
  
}

void vtkKWMeshVisu::SignedDistanceMapCallback()
{
  if ( this->commandLine ) 
  {
    return ;
  }
  this->TFuncEditor->SetColorTransferFunction (this->SignedDistanceMapTFunc) ;
  this->TFuncEditor->LockEndPointsParameterOff();
  this->TFuncEditor->SetLabelText ("Signed Distance Color Map");
  
  // if we are not in 1D attribute mode, leave the 1D visualization alone
  if ( this->attributeDimension == 1 )
  {
    this->meshMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
  }
  
  this->arrowGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
  this->ellipsoidGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
  this->particleMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
  //this->meshMapper->ScalarVisibilityOn ();
  
  double range[2] ;
  this->GetAttributeRange ( range ) ;
  double dist = range[1] - range[0] ;
  this->TFuncEditor->SetWholeParameterRange(range[0]-dist, range[1]+dist) ;
  this->TFuncEditor->SetVisibleParameterRange (range);
  //this->TFuncEditor->SetWholeParameterRange(0.0, 1.0);
  
  this->RenderWidget->Render();
}

//----------------------------------------------------------------------------
void vtkKWMeshVisu::DistanceMapCallback()
{
  if ( this->commandLine ) 
  {
    return ;
  }
  this->TFuncEditor->SetColorTransferFunction (this->DistanceMapTFunc) ;
  this->TFuncEditor->LockEndPointsParameterOff();
  this->TFuncEditor->SetLabelText ("Distance Color Map");

  this->arrowGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
  this->ellipsoidGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
  this->particleMapper->SetLookupTable (this->DistanceMapTFunc) ;

  // if we are not in 1D attribute mode, leave the 1D visualization alone
  if ( this->attributeDimension == 1 )
  {
    this->meshMapper->SetLookupTable (this->DistanceMapTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
  }
  
  // add some cushioning to the parameter range
  double range[2] ;
  this->GetAttributeRange ( range ) ;
  double dist = range[1] - range[0] ;
  this->TFuncEditor->SetWholeParameterRange(range[0]-dist, range[1]+dist) ;
  this->TFuncEditor->SetVisibleParameterRange (range);

  this->RenderWidget->Render();
}


//----------------------------------------------------------------------------
void vtkKWMeshVisu::ROIMapCallback()
{
  if ( this->commandLine ) 
  {
    return ;
  }
  this->TFuncEditor->SetColorTransferFunction (this->ROIMapTFunc) ;
  this->TFuncEditor->LockEndPointsParameterOff();
  this->TFuncEditor->SetLabelText ("ROI Color Map");

  this->arrowGlyphMapper->SetLookupTable (this->ROIMapTFunc) ;
  this->ellipsoidGlyphMapper->SetLookupTable (this->ROIMapTFunc) ;
  this->particleMapper->SetLookupTable (this->ROIMapTFunc) ;

  // if we are not in 1D attribute mode, leave the 1D visualization alone
  if ( this->attributeDimension == 1 )
  {
    this->meshMapper->SetLookupTable (this->ROIMapTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
  }
  
  // add some cushioning to the parameter range
  double range[2] ;
  //this->GetAttributeRange ( range ) ;
  if ( this->attributeDimension == 1 ) 
    this->unmaskedScalars->GetRange ( range ) ;
  else
    {
      range[0] = -1 ;
      range[1] = 1 ;
    }
  double dist = range[1] - range[0] ;
  double padding = dist / 2 ;
  this->TFuncEditor->SetWholeParameterRange(range[0]-dist-padding, range[1]+padding) ;
  this->TFuncEditor->SetVisibleParameterRange ( range[0]-dist, range[1] );

  this->RenderWidget->Render();
}



//----------------------------------------------------------------------------
void vtkKWMeshVisu::OpacityCallback(double d)
{
  double val = this->opacityScale->GetValue() ;
  this->meshActor->GetProperty()->SetOpacity(val);
  this->RenderWidget->Render();
}

void vtkKWMeshVisu::FeatureScaleCallback(double d)
{
  double val = this->featureScale->GetValue() ;
  this->arrowGlyphFilter->SetScaleFactor(val);
  this->ellipsoidGlyphFilter->SetScaleFactor(val);
  this->particleActor->GetProperty()->SetPointSize ( val ) ;
  this->RenderWidget->Render();
}

void vtkKWMeshVisu::UpdateROIMapTFunc ()
{
  if ( this->commandLine )
  {
    return ;
  }
  this->ROIMapTFunc->RemoveAllPoints () ;
  double min = this->currentScalarRange[0] ;
  double d = this->currentScalarRange[1] - min ;
  this->ROIMapTFunc->AddHSVPoint( min - d,       0, 1, 1);
  this->ROIMapTFunc->AddHSVPoint( min - 0.00001, 0, 1, 0.5);
  this->ROIMapTFunc->AddHSVPoint( min,           0.6, 1, 1);
  this->ROIMapTFunc->AddHSVPoint( min + d,       0.6, 1, 0.5);
}

void vtkKWMeshVisu::UpdateDistanceMapTFunc ()
{
  double range[2];
  if ( this->commandLine )
  {
    return ;
  }
  this->GetAttributeRange ( range ) ;
  this->DistanceMapTFunc->RemoveAllPoints () ;
  this->DistanceMapTFunc->AddHSVPoint(range[0], 0.3, 1, 1);
  this->DistanceMapTFunc->AddHSVPoint(range[1], 0, 1, 1);
}

void vtkKWMeshVisu::UpdateSignedDistanceMapTFunc ()
{
  double range[2];
  if ( this->commandLine )
  {
    return ;
  }
  this->GetAttributeRange ( range ) ;
  
  this->SignedDistanceMapTFunc->RemoveAllPoints () ;
  
  this->SignedDistanceMapTFunc->AddHSVPoint(range[0], 0, 1, 1);
  this->SignedDistanceMapTFunc->AddHSVPoint(0, 0.27 , 0, 1);
  this->SignedDistanceMapTFunc->AddHSVPoint(0.0001, 0.27 , 0, 1);
  this->SignedDistanceMapTFunc->AddHSVPoint(range[1], 0.6, 1, 1);

}
void vtkKWMeshVisu::GetAttributeRange (double *r )
{
  double temp[2] ;
  if ( ( this->attributeDimension == 1 ) || ( this->ROIMaskEnabled ) ) 
  {
    this->vtkMesh->GetScalarRange (temp) ;
    r[0] = temp[0] ;
    r[1] = temp[1] ;
  }
  else if ( this->attributeDimension == 3 )
  {
    double range ;
    r[0] = r[1] = 0 ;
    for ( int i = 0 ; i < 3 ; i++ )
    {
      this->vtkMesh->GetPointData()->GetVectors()->GetRange(temp, i) ;
      if (temp[0]<0) 
        temp[0] *= -1 ;
      if (temp[1]<0) 
        temp[1] *= -1 ;
      if ( temp[0] > temp[1] )
      {
        range = temp[0] ;
      }
      else
      {
        range = temp[1] ;
      }
      r[1] += range * range;
    }
    r[1] = sqrt ( r[1] ) ;
  }
  else if ( this->attributeDimension == 9 )
  {
    double range ;
    r[0] = r[1] = 0 ;
    for ( int i = 0 ; i < 9 ; i++ )
    {
      this->vtkMesh->GetPointData()->GetTensors()->GetRange(temp, i) ;
      if (temp[0]<0) 
        temp[0] *= -1 ;
      if (temp[1]<0) 
        temp[1] *= -1 ;
      if ( temp[0] > temp[1] )
      {
        range = temp[0] ;
      }
      else
      {
        range = temp[1] ;
      }
      r[1] += range * range;
    }
    r[1] = sqrt ( r[1] ) ;
  }
  else if ( this->nParticleSets )
  {
    std::cout << particleScalars->GetValue(150) << std::endl ;
    this->particleScalars->GetRange ( temp ) ;
    r[0] = temp[0] ;
    r[1] = temp[1] ;
    std::cout << "now im here" << r[0] << " " << r[1] << std::endl ;
  }
  else // at least don't leave the range uninitialized
    {
      r[0] = 0 ;
      r[1] = 1 ;
    }
}

void vtkKWMeshVisu::Stabilize1DVisualization ()
{
  if ( this->commandLine ) 
    return ;

  if ( this->FixedTFunc ) 
    this->FixedTFunc->Delete();
  
  this->FixedTFunc = vtkColorTransferFunction::New();
  
  if ( strcmp (this->ColorMode->GetValue(), "P Value") == 0 )
  {
    this->FixedTFunc->DeepCopy (this->PValueTFunc) ;
  }
  else
  {
    this->FixedTFunc->DeepCopy (this->DistanceMapTFunc) ;
  }
  this->meshMapper->SetLookupTable ( this->FixedTFunc ) ;
}

void vtkKWMeshVisu::GetCameraCallback ()
{
  vtkCamera *camera ;
  double pos[3], dir[3], up[3], scale ;
  
  camera = this->RenderWidget->GetRenderer()->GetActiveCamera();
  camera->GetPosition(pos);
  camera->GetFocalPoint(dir);
  camera->GetViewUp(up);
  scale = camera->GetParallelScale();
  
  vtkKWMessageDialog *dlg = vtkKWMessageDialog::New();
  dlg->SetParent(this->RenderWidget->GetParent());
  dlg->SetMasterWindow(this->RenderWidget->GetParentTopLevel());
  dlg->SetStyleToOkCancel();
  dlg->Create();
  dlg->SetTitle("Current camera parameters");
  
  char text[256] ;
  sprintf (text, "Position: %f %f %f\nFocal Point: %f %f %f\nUp: %f %f %f\nScale:%f", pos[0], pos[1], pos[2], dir[0], dir[1], dir[2], up[0], up[1], up[2], scale);
  dlg->SetText(text);
  
  this->RenderWidget->Render();
  dlg->Invoke();
  dlg->Delete();
}

//------------------------------------------------------------------------
void vtkKWMeshVisu::MakeVideoCallback()
{
  if ( ! this->vtkMesh )
    return ;

  vtkKWLoadSaveDialog *dialog = vtkKWLoadSaveDialog::New() ;

  dialog->SetParent(this->RenderWidget->GetParentTopLevel()) ;
  dialog->Create();
  dialog->RetrieveLastPathFromRegistry("DataPath");
  dialog->SetTitle ("Create 'video' by rotating around the vertical axis");
  //dialog->SetFileTypes ("{{BMP bitmap images} {.bmp}}");
  //dialog->SetDefaultExtension (".bmp");
  dialog->SaveDialogOn() ;

  if ( dialog->Invoke () == 0 )
  {
    dialog->Delete();
    return ;
  }

  dialog->SaveLastPathToRegistry("DataPath");

  this->MakeVideo ( dialog->GetFileName() ) ;

  dialog->Delete();
}

void vtkKWMeshVisu::LoadMesh (const char *fileName)
{
  typedef double real ;

  bool itkFile, vtkFile, vtpFile ;
  const char *extension ;
  extension = strrchr ( fileName, '.' ) ;
  itkFile = ! strcmp ( extension, ".meta" ) ;
  vtkFile = ! strcmp ( extension, ".vtk" ) ;
  vtpFile = ! strcmp ( extension, ".vtp" ) ;

  assert ( itkFile || vtkFile || vtpFile ) ;

  if ( itkFile ) // itk format, has to be converted
  {
    typedef itk::DefaultDynamicMeshTraits < real, 3, 3, real, real > MeshTraitsType ; 
    typedef itk::Mesh < real, 3, MeshTraitsType > itkMeshType ;
    typedef itk::MeshSpatialObject < itkMeshType > itkMeshSOType ;
    typedef itk::MetaMeshConverter < 3, real, MeshTraitsType > MeshConverterType ;
    
    // read the data in itk format
    MeshConverterType * itkConverter = new MeshConverterType() ;
    itkMeshSOType::Pointer meshSO = itkConverter->ReadMeta (fileName) ;
    itkMeshType::Pointer mesh = meshSO->GetMesh() ;
    delete (itkConverter);
    
    // convert to vtk format
    itkMeshTovtkPolyData * ITKVTKConverter = new itkMeshTovtkPolyData;
    ITKVTKConverter->SetInput ( mesh ) ;
    if ( this->vtkMesh ) 
    {
      this->vtkMesh->Delete() ;
    }
    this->vtkMesh = ITKVTKConverter->GetOutput () ;
    delete (ITKVTKConverter);
  }
  else if ( vtkFile ) // already in vtk format
  {
    if ( this->vtkMesh ) 
    {
      this->vtkMesh->Delete() ;
    }

    vtkPolyDataReader *meshReader = vtkPolyDataReader::New() ;
    meshReader->SetFileName ( fileName ) ;
    meshReader->Update () ;
    vtkPolyData *tempMesh = meshReader->GetOutput() ;
    this->vtkMesh = vtkPolyData::New () ;
    this->vtkMesh->DeepCopy ( tempMesh ) ;

    meshReader->Delete();
  }
  else if ( vtpFile ) // already in vtk format
  {
    if ( this->vtkMesh ) 
    {
      this->vtkMesh->Delete() ;
    }

    vtkXMLPolyDataReader *meshReader = vtkXMLPolyDataReader::New() ;
    meshReader->SetFileName ( fileName ) ;
    meshReader->Update () ;
    vtkPolyData *tempMesh = meshReader->GetOutput() ;
    this->vtkMesh = vtkPolyData::New () ;
    this->vtkMesh->DeepCopy ( tempMesh ) ;

    meshReader->Delete();
  }

  this->normalFilter->SetInput (this->vtkMesh);
  
  // setup vtk rendering
  this->meshMapper->SetInput(this->normalFilter->GetOutput());
  this->meshMapper->Update() ;
  
  if ( ! this->meshActor )
  {
    this->meshActor = vtkActor::New();
    this->meshActor->SetMapper(this->meshMapper);
  }
  
  // setup the vector field visualization
  this->arrowGlyphFilter->SetInput(this->meshMapper->GetInput());
  this->arrowGlyphFilter->SetSource(this->arrowSource->GetOutput());
  this->arrowGlyphFilter->SetScaleModeToScaleByVector();
  //this->arrowGlyphFilter->SetScaleModeToDataScalingOff();
  //this->arrowGlyphFilter->SetScaleFactor(1);
  this->arrowGlyphFilter->SetVectorModeToUseVector();
  this->arrowGlyphFilter->SetColorModeToColorByVector();
  this->arrowGlyphFilter->OrientOn ();
  
  this->arrowGlyphMapper->SetInput(this->arrowGlyphFilter->GetOutput());
  
  if ( !this->arrowGlyphActor )
  {
    this->arrowGlyphActor = vtkActor::New();
    this->arrowGlyphActor->SetMapper (this->arrowGlyphMapper);
  }

  // setup the ellipsoid field visualization
  this->ellipsoidGlyphFilter->SetInput(this->meshMapper->GetInput());
  this->ellipsoidGlyphFilter->SetSource(this->ellipsoidSource->GetOutput());
  
  this->ellipsoidGlyphFilter->SetScaleFactor (1);
  this->ellipsoidGlyphFilter->ThreeGlyphsOff () ;
  this->ellipsoidGlyphFilter->ColorGlyphsOn () ;
  this->ellipsoidGlyphFilter->SetColorModeToEigenvalues();
  this->ellipsoidGlyphFilter->ExtractEigenvaluesOff();
  
  
  //vtkReverseSense *reverse = vtkReverseSense::New() ;
  //reverse->SetInput (this->ellipsoidGlyphFilter->GetOutput()) ;
  //this->ellipsoidGlyphMapper->SetInput(this->ellipsoidGlyphFilter->GetOutput());

  // make sure the normals for the ellipsoids are in the correct direction, 
  // to be able to color them properly
  vtkPolyDataNormals *normals = vtkPolyDataNormals::New() ;
  normals->SetInput (this->ellipsoidGlyphFilter->GetOutput()) ;
  
  this->ellipsoidGlyphMapper->SetInput(normals->GetOutput());
  normals->Delete () ;
  
  if ( !this->ellipsoidGlyphActor )
  {
    this->ellipsoidGlyphActor = vtkActor::New();
    this->ellipsoidGlyphActor->SetMapper (this->ellipsoidGlyphMapper);
  }

  this->ResetROIMask () ;

  this->RenderWidget->AddViewProp (this->meshActor) ;
  
  this->RenderWidget->ResetCamera();

  this->RenderWidget->Render();  
  
  this->attributeDimension = 0 ;
  //meshReader->Delete() ;
  this->win->SetTitle ( fileName ) ;
}

void vtkKWMeshVisu::LoadScalarField (const char *fileName)
{
  KWMeshVisuAttributeIO *attributeReader = new ( KWMeshVisuAttributeIO ) ;
  attributeReader->SetFileName ( fileName ) ;
  attributeReader->ReadAttributes () ;
  vtkFloatArray *scalars = attributeReader->GetAttributes () ; 
  if ( ( scalars->GetNumberOfComponents () != 1 ) || ( scalars->GetNumberOfTuples () != this->vtkMesh->GetNumberOfPoints() ) )  
    return ; 

  this->unmaskedScalars = scalars ;
  // make the final connections
  
  this->vtkMesh->GetPointData()->SetScalars ( scalars ) ;
  this->attributeDimension = 1 ;

  for ( int i = 0 ; i < scalars->GetNumberOfTuples () ; i++ )
    {
      this->UpdateROIMaskedScalars ( i ) ;
    }
  this->unmaskedScalars->GetRange ( this->currentScalarRange ) ;

  this->UpdateTFunc();  

  this->meshMapper->ScalarVisibilityOn ();
  this->ROIMaskCheckBoxCallback ( 0 ) ;
  
  // make sure we're not drawing arrows or spheres
  this->RenderWidget->RemoveViewProp(this->arrowGlyphActor);
  this->RenderWidget->RemoveViewProp(this->ellipsoidGlyphActor);
  
  if ( !this->commandLine )
  {
    if ( this->ROIMaskEnabled ) 
      this->ColorMode->SetValue ( "ROI" ) ;
    else 
      this->ColorMode->SetValue("P Value") ;
  }
  if ( !this->ROIMaskEnabled ) 
    this->PValueCallback () ;

  this->RenderWidget->Render();
  char title[1000] ;
  sprintf ( title, "%s; %s", this->win->GetTitle(), fileName ) ;
  this->win->SetTitle ( title ) ;
  //scalars->Delete ();
}

void vtkKWMeshVisu::LoadVectorField (const char *fileName)
{
  KWMeshVisuAttributeIO *attributeReader = new ( KWMeshVisuAttributeIO ) ;
  attributeReader->SetFileName ( fileName ) ;
  attributeReader->ReadAttributes () ;
  vtkFloatArray *vectors = attributeReader->GetAttributes () ; 
  if ( ( vectors->GetNumberOfComponents () != 3 ) || ( vectors->GetNumberOfTuples () != this->vtkMesh->GetNumberOfPoints() ) )  
    return ; 
  
  this->vtkMesh->GetPointData()->SetVectors ( vectors ) ;
  
  if ( this->attributeDimension == 1 )
  {
    this->Stabilize1DVisualization();
  }
  
  this->attributeDimension = 3 ;
  this->UpdateTFunc () ;

  this->RenderWidget->AddViewProp(this->arrowGlyphActor);
  //this->meshMapper->ScalarVisibilityOff ();
  if ( !this->commandLine ) 
  {
    this->opacityScale->SetValue(0.5);
  }
  this->meshActor->GetProperty()->SetOpacity(0.5) ;
  
  this->RenderWidget->Render();
  
  vectors->Delete();
  
}

void vtkKWMeshVisu::LoadCurve (const char *fileName)
{
  // read in the file
  std::ifstream attrFile ;
  attrFile.open ( fileName ) ;
    // This is a quite sloppy implementation of reading keyword/value pairs!
  bool found;
  char * valuePtr;
  int nPts, nDim ;
  char typeString[1001], line[1001];

  attrFile.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !attrFile.eof())
  { attrFile.getline ( line, 1000 ) ;
    if (line[0] != '#' && strstr ( line, "NUMBER_OF_POINTS" )) found = true;
  }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nPts);

  attrFile.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !attrFile.eof())
  { attrFile.getline ( line, 1000 ) ;
    if (line[0] != '#' && strstr ( line, "DIMENSION" )) found = true;
  }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nDim);

  attrFile.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !attrFile.eof())
  { attrFile.getline ( line, 1000 ) ;
    if (line[0] != '#' && strstr ( line, "TYPE" )) found = true;
  }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return;
  valuePtr++;
  sscanf(valuePtr, " %s ", typeString);

  assert ( nPts > 0 ) ;
  assert ( nDim == 3 ) ;
  assert ( strcmp ( typeString, "Curve" ) == 0 ) ;

  attrFile.seekg(0,std::ios::beg);
  const int numEntries = 3;
  int counter = 0;
  while ( counter < numEntries && !attrFile.eof())
  { attrFile.getline ( line, 1000 ) ;
    if ((line[0] != '#')) counter++;
  }

  int i ;
  double x[3] ;
  vtkPoints *polyLinePoints = vtkPoints::New() ;
  polyLinePoints->SetNumberOfPoints (nPts) ;

  vtkPolyLine *aPolyLine = vtkPolyLine::New() ;
  aPolyLine->GetPointIds()->SetNumberOfIds (nPts) ;
  
  for ( i = 0 ; i < nPts ; i++ )
  {
    attrFile >> x[0] >> x[1] >> x[2] ;
    polyLinePoints->InsertPoint(i, x[0], x[1], x[2]);
    aPolyLine->GetPointIds()->SetId(i, i) ;
  }

  attrFile.close () ;

  vtkUnstructuredGrid *aPolyLineGrid = vtkUnstructuredGrid::New() ;
  aPolyLineGrid->Allocate(1, 1) ;
  aPolyLineGrid->InsertNextCell(aPolyLine->GetCellType(), aPolyLine->GetPointIds());
  aPolyLineGrid->SetPoints(polyLinePoints) ;

  vtkDataSetMapper *aPolyLineMapper = vtkDataSetMapper::New() ;
  aPolyLineMapper->SetInput(aPolyLineGrid);

  this->aPolyLineActor->SetMapper(aPolyLineMapper) ;
  this->aPolyLineActor->GetProperty()->SetColor(1, 0, 0) ;
  this->aPolyLineActor->GetProperty()->SetLineWidth (3) ;
  this->RenderWidget->AddViewProp (this->aPolyLineActor) ;
  this->RenderWidget->Render();
  
}

void vtkKWMeshVisu::LoadROI (const char *fileName)
{
  // read in the file
  std::ifstream roiFile ;
  roiFile.open ( fileName ) ;
    // This is a quite sloppy implementation of reading keyword/value pairs!
  bool found;
  char * valuePtr;
  int nPts ;
  char line[1001];

  roiFile.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !roiFile.eof())
  { roiFile.getline ( line, 1000 ) ;
    if (line[0] != '#' && strstr ( line, "NUMBER_OF_POINTS" )) found = true;
  }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nPts);

  assert ( nPts == this->vtkMesh->GetNumberOfPoints() ) ;
  
  roiFile.seekg(0,std::ios::beg);
  const int numEntries = 3;
  int counter = 0;
  while ( counter < numEntries && !roiFile.eof())
  { roiFile.getline ( line, 1000 ) ;
    if ((line[0] != '#')) counter++;
  }

  int i ;
  bool current ;
  this->ResetROIMask () ;
  for ( i = 0 ; i < nPts ; i++ )
  {
    roiFile >> current ;
    if ( current ) 
      {
	this->AddBall ( i ) ;
      }
  }

  roiFile.close () ;
  this->ROIMaskCheckBox->SelectedStateOn () ;
  this->UpdateTFunc () ;
  this->RenderWidget->Render();
}

void vtkKWMeshVisu::SaveROI (const char *fileName)
{
  // open the file
  std::ofstream roiFile ;
  
  roiFile.open ( fileName ) ;
  int nPoints = this->vtkMesh->GetNumberOfPoints() ;
  roiFile << "NUMBER_OF_POINTS=" << nPoints << std::endl ;
  roiFile << "DIMENSION=1" << std::endl << "TYPE=ROI" << std::endl ;

  for ( int i = 0 ; i < nPoints ; i++ )
  {
    roiFile << this->ROIMask->GetValue(i) << std::endl ;
  }
  
  roiFile.close () ;
}

void vtkKWMeshVisu::LoadEllipsoidField (const char *fileName)
{
  KWMeshVisuAttributeIO *attributeReader = new ( KWMeshVisuAttributeIO ) ;
  attributeReader->SetFileName ( fileName ) ;
  attributeReader->ReadAttributes () ;
  vtkFloatArray *vectors = attributeReader->GetAttributes () ; 
  if ( ( vectors->GetNumberOfComponents () != 9 ) || ( vectors->GetNumberOfTuples () != this->vtkMesh->GetNumberOfPoints() ) )  
    return ; 

  
  if ( this->attributeDimension == 1 )
  {
    this->Stabilize1DVisualization();
  }
  
  this->vtkMesh->GetPointData()->SetTensors ( vectors ) ;
  this->attributeDimension = 9 ;
  this->UpdateTFunc () ;

  this->RenderWidget->AddViewProp(this->ellipsoidGlyphActor);
  if ( !this->commandLine ) 
  {
    this->opacityScale->SetValue(0.5);
  }
  this->meshActor->GetProperty()->SetOpacity(0.5) ;
  
  this->RenderWidget->Render();
  vectors->Delete();
  
}

void vtkKWMeshVisu::LoadParticleSet(const char *fileName)
{
  // load file
  std::ifstream particleFile ;
  particleFile.open ( fileName ) ;
  
  int nOldPoints ;
  if ( this->nParticleSets )
    nOldPoints = this->particleSets[this->nParticleSets-1]->GetNumberOfVerts() ;
  else
    nOldPoints = 0 ;

  this->particleSets[this->nParticleSets] = vtkPolyData::New () ;
  vtkPolyData *pd = this->particleSets[this->nParticleSets] ;
  this->nParticleSets++ ;

  double x, y, z ; 
  int nPoints = 0 ;
  vtkPoints *points = vtkPoints::New () ;
  vtkCellArray *cells = vtkCellArray::New () ;
  vtkIdList *currentCell = vtkIdList::New () ;
  currentCell->SetNumberOfIds ( 1 ) ;
  while ( 1 ) 
  {
    particleFile >> x >> y ;
    if ( particleFile.eof () ) 
      break ;
    
    particleFile >> z ;
    points->InsertNextPoint ( x, y, z ) ;
    currentCell->SetId ( 0, nPoints ) ;
    cells->InsertNextCell ( currentCell ) ;
    nPoints++ ;
    
  }
  
  pd->SetPoints ( points ) ;
  pd->SetVerts ( cells ) ;
  
  points->Delete () ;
  cells->Delete () ;
  
  // add this file to particle combobox
  std::string fileNameString ( fileName ) ;
  int i = fileNameString.rfind( '\\' ) ;
  fileNameString = fileNameString.substr( i + 1 ); 
  this->ParticleListBox->GetWidget()->GetWidget()->Append ( fileNameString.c_str() ) ;

  // make this file current particle set
  for ( int i = 0 ; i < this->nParticleSets - 1 ; i++ )
    this->ParticleListBox->GetWidget()->GetWidget()->SetSelectState ( i, 0 ) ;
  this->ParticleListBox->GetWidget()->GetWidget()->SetSelectionIndex ( this->nParticleSets - 1 ) ;
  this->ParticleListBoxCallback () ;

  pd->GetPointData()->SetScalars ( this->particleScalars ) ;

  // enable particle visualization
  this->ParticleVisualizationCheckBox->Select() ;
  this->ParticleVisualizationCheckBoxCallback ( 1 ) ;

  // separate the particle color map from the surface scalar color map
  this->Stabilize1DVisualization () ;

  // refresh visualization
  this->RenderWidget->Render();
}

void vtkKWMeshVisu::LoadParticleScalars (const char *fileName)
{
  int currentParticleSet = this->ParticleListBox->GetWidget()->GetWidget()->GetSelectionIndex ( ) ;
  if ( currentParticleSet < 0 ) 
    return ;

  KWMeshVisuAttributeIO *attributeReader = new ( KWMeshVisuAttributeIO ) ;
  attributeReader->SetFileName ( fileName ) ;
  attributeReader->ReadAttributes () ;
  //vtkFloatArray *
  this->particleScalars = attributeReader->GetAttributes () ;
  //if ( ( particleScalars->GetNumberOfComponents () != 1 ) || ( particleScalars->GetNumberOfTuples () != this->vtkMesh->GetNumberOfPoints() ) )  
  if ( ( this->particleScalars->GetNumberOfComponents () != 1 ) || ( this->particleScalars->GetNumberOfTuples () != this->particleSets[currentParticleSet]->GetNumberOfPoints() ) )
    {
      std::cout << "number of tuples: " << this->particleScalars->GetNumberOfTuples() << std::endl ;
      std::cout << "number of particles: " << this->particleSets[currentParticleSet]->GetNumberOfPoints() << std::endl ;
      return ;
    }

  // make the final connections
  
  this->particleSets[currentParticleSet]->GetPointData()->SetScalars ( this->particleScalars ) ;

  this->UpdateTFunc();  

  this->meshMapper->ScalarVisibilityOn ();
  this->meshMapper->Update() ;  
  this->particleMapper->ScalarVisibilityOn () ;
  this->particleMapper->Update() ;
  this->RenderWidget->Render();
  this->RenderWidget->Render() ;

}

void vtkKWMeshVisu::ResetParticleScalars () 
{
  int currentParticleSet = this->ParticleListBox->GetWidget()->GetWidget()->GetSelectionIndex () ;
  int nPoints = this->particleSets[currentParticleSet]->GetNumberOfVerts() ;
  this->particleScalars->SetNumberOfTuples ( nPoints ) ;
  for ( int i = 0 ; i < nPoints ; i++ )
  {
    this->particleScalars->InsertTuple1 ( i, 1 ) ;
  }
}

int vtkKWMeshVisu::CommandLine (int argc, const char *argv[])
{
  char *meshFile, *scalarFile, *vectorFile, *ellipsoidFile, *particleFile, 
    *inputFile, *outputFile,
    *imageFile, *visMode, *roiFile;
  
  float opacity, scale, pvalueThreshold, distMapMin, distMapMax ;
  float posx, posy, posz, focx, focy, focz, upx, upy, upz, zoom ;
  float signedDistMin, signedDistMax, signedDistAlpha ;
 
  float r, g, b ;
  
  bool quit ;

  this->commandLine = ipExistsArgument ( argv, "--quit" ) ;
  this->Run (argc, (char **)argv) ;

  meshFile = ipGetStringArgument(argv, "--mesh", NULL);  
  scalarFile = ipGetStringArgument(argv, "--scalar", NULL);  
  vectorFile = ipGetStringArgument(argv, "--vector", NULL);  
  ellipsoidFile = ipGetStringArgument(argv, "--ellipsoid", NULL);  
  particleFile = ipGetStringArgument(argv, "--particle", NULL);
  inputFile = ipGetStringArgument(argv, "--input", NULL);  
  outputFile = ipGetStringArgument(argv, "--output", NULL);  
  imageFile = ipGetStringArgument(argv, "--image", NULL);  
  roiFile = ipGetStringArgument(argv, "--roi", NULL) ;
  visMode = ipGetStringArgument (argv, "--visMode", "PValue") ;

  opacity = ipGetFloatArgument(argv, "--opacity", 1);
  scale = ipGetFloatArgument(argv, "--scale", 1);
  pvalueThreshold = ipGetFloatArgument(argv, "--pvalue", -1);
  distMapMin = ipGetFloatArgument(argv, "--distMapMin", -1);
  distMapMax = ipGetFloatArgument(argv, "--distMapMax", -1);
  signedDistMin = ipGetFloatArgument ( argv, "--signedDistMin", -1) ;
  signedDistMax = ipGetFloatArgument ( argv, "--signedDistMax", -1) ;
  signedDistAlpha  = ipGetFloatArgument ( argv, "--signedDistAlpha", -1) ;
   
  posx = ipGetFloatArgument(argv, "--posx", -1);
  posy = ipGetFloatArgument(argv, "--posy", -1);
  posz = ipGetFloatArgument(argv, "--posz", -1);
  focx = ipGetFloatArgument(argv, "--focx", -1);
  focy = ipGetFloatArgument(argv, "--focy", -1);
  focz = ipGetFloatArgument(argv, "--focz", -1);
  upx = ipGetFloatArgument(argv, "--upx", -1);
  upy = ipGetFloatArgument(argv, "--upy", -1);
  upz = ipGetFloatArgument(argv, "--upz", -1);
  zoom = ipGetFloatArgument(argv, "--zoom", -1);
  
  r = ipGetFloatArgument(argv, "--br", 0);
  g = ipGetFloatArgument(argv, "--bg", 0);
  b = ipGetFloatArgument(argv, "--bb", 0);
  
  quit = ipExistsArgument ( argv, "--quit" ) ;
  bool cameraOn = ipExistsArgument ( argv, "--cameraSwitch" ) ;

  if ( ( !meshFile && !inputFile ) || ipExistsArgument(argv, "--help") )
  {
    cout << "Usage:" << endl ;
    cout << "KWMeshVisu " << endl ;
    cout << "[--mesh Meshfile [--scalar 1dfile] [--vector 3dfile] [--ellipsoid 9dfile] [--particle particleFile]]" << endl ;
    cout << "[--input attributedMeshfile]" << endl ;
    cout << "[--output attributedMeshfile]" << endl ;
    cout << "[--roi roiFile]" << endl ;
    cout << "[--image bmpfile [--opacity o] [--scale s] [--br BackgroundR] [--bg BackgroundG] [--bb BackgroundB]]" << endl ;
    cout << "[--visMode DistanceMap] [--distMapMin min --distMapMax max]" << endl ;
    cout << "[--visMode Pvalue] [--pvalue threshold]" << endl;
    cout << "[--visMode SignedDistMap] [--signedDistMin min --signedDistMax max --signedDistAlpha alpha]" << endl ;
    cout << "[--posx x --posy x --posz x --focx x --focy x --focz x --upx x --upy x --upz x --zoom x --cameraSwitch]" << endl ;
    cout << "[--quit]" << endl ;
    return 0 ;
  }
  
  if (inputFile)
  {
  }
  else
  {
    this->LoadMesh (meshFile) ;
    if ( scalarFile && ( strcmp ( scalarFile, "NULL" ) ) )
    {
      this->LoadScalarField ( scalarFile ) ;
    }
    if ( vectorFile && ( strcmp ( vectorFile, "NULL" ) ) ) 
    {
      this->LoadVectorField ( vectorFile ) ;
    }
    if ( ellipsoidFile && ( strcmp ( ellipsoidFile, "NULL" ) ) ) 
    {
      this->LoadEllipsoidField ( ellipsoidFile ) ;
    }
    if ( particleFile && ( strcmp ( particleFile, "NULL" ) ) ) 
    {
      this->LoadParticleSet ( particleFile ) ;
    }    
    if ( roiFile && ( strcmp ( roiFile, "NULL" ) ) ) 
    {
      this->LoadROI ( roiFile ) ;
    }
  }

  this->RenderWidget->SetRendererBackgroundColor(r, g, b) ;
  this->meshActor->GetProperty()->SetOpacity(opacity);

  this->arrowGlyphFilter->SetScaleFactor(scale);
  this->ellipsoidGlyphFilter->SetScaleFactor(scale);

  if ( pvalueThreshold != -1 )
  {
    this->PValueTFunc->RemoveAllPoints () ;
    this->PValueTFunc->AddHSVPoint(0, 0, 1, 1);
    this->PValueTFunc->AddHSVPoint(pvalueThreshold, 0.27, 0.6, 0.97, 0, 1);
    //this->PValueTFunc->AddHSVPoint(pvalueThreshold, 0.27, 0.6, 0.97);
    this->PValueTFunc->AddHSVPoint(1, 0.6, 1, 1);
    this->meshMapper->SetLookupTable (this->PValueTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
    this->arrowGlyphMapper->SetLookupTable (this->PValueTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->PValueTFunc) ;
    if ( !this->commandLine )
      this->ColorMode->SetValue("P Value");
  }

  if ( ( distMapMin != -1 ) || ( distMapMax != -1 ) )
  {
    this->DistanceMapTFunc->RemoveAllPoints () ;
    this->DistanceMapTFunc->AddHSVPoint(distMapMin, 0.3, 1, 1);
    this->DistanceMapTFunc->AddHSVPoint(distMapMax, 0, 1, 1);
    this->meshMapper->SetLookupTable (this->DistanceMapTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
    this->arrowGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
    if ( !this->commandLine )
      this->ColorMode->SetValue("Distance Map");
  }

  if ( ( signedDistMin != -1 ) || ( signedDistMax != -1 ) || ( signedDistAlpha != -1  ) )
  {
    this->SignedDistanceMapTFunc->RemoveAllPoints () ;
    this->SignedDistanceMapTFunc->AddHSVPoint(signedDistMin, 0, 1, 1);
    this->SignedDistanceMapTFunc->AddHSVPoint(-signedDistAlpha, 0.27, 0, 1);
    this->SignedDistanceMapTFunc->AddHSVPoint(signedDistAlpha, 0.27, 0, 1);
    this->SignedDistanceMapTFunc->AddHSVPoint(signedDistMax, 0.6, 1, 1);
    this->meshMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    this->meshMapper->SetColorModeToMapScalars ();
    this->arrowGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    if ( !this->commandLine )
      this->ColorMode->SetValue("Signed Distance");
  }

  if ( strcmp ( visMode, "PValue" ) == 0 )
  {
    this->meshMapper->SetLookupTable (this->PValueTFunc) ;
    this->arrowGlyphMapper->SetLookupTable (this->PValueTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->PValueTFunc) ;
    if ( ! this->commandLine ) 
    {
      this->TFuncEditor->SetColorTransferFunction (this->PValueTFunc) ;
      this->TFuncEditor->LockEndPointsParameterOn();
      this->TFuncEditor->SetLabelText ("P-Value Color Map");  
      this->ColorMode->SetValue("P Value");
    }
  }
  else if ( strcmp ( visMode, "DistanceMap" ) == 0 ) 
  {
    this->meshMapper->SetLookupTable (this->DistanceMapTFunc) ;
    this->arrowGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->DistanceMapTFunc) ;
    if ( ! this->commandLine ) 
    {
      this->TFuncEditor->SetColorTransferFunction (this->DistanceMapTFunc) ;
      this->TFuncEditor->LockEndPointsParameterOff();
      this->TFuncEditor->SetLabelText ("Distance Color Map");
      this->ColorMode->SetValue("Distance Map");
    }
  }
  else if ( strcmp ( visMode, "SignedDistanceMap" ) == 0 )
  {
    this->meshMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    this->arrowGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    this->ellipsoidGlyphMapper->SetLookupTable (this->SignedDistanceMapTFunc) ;
    if ( ! this->commandLine )
    {
      this->TFuncEditor->SetColorTransferFunction (this->SignedDistanceMapTFunc) ;
      this->TFuncEditor->LockEndPointsParameterOff();
      this->TFuncEditor->SetLabelText ("Signed Distance Color Map");
      this->ColorMode->SetValue("Signed Distance");
    }
  }
  if ( cameraOn )
  {
    vtkCamera *camera = this->RenderWidget->GetRenderer()->GetActiveCamera();
    camera->SetPosition(posx, posy, posz);
    camera->SetFocalPoint(focx, focy, focz);
    camera->SetViewUp(upx, upy, upz);
    camera->SetParallelScale(zoom);
  }
  this->RenderWidget->Render () ;

  if ( imageFile && quit ) 
  {
    this->SaveRenderAsBMP ( imageFile ) ;
  }

  
  if ( quit ) 
    return 0 ;

  // dont quit until the user kills the application
  vtkKWApplication *app = this->GetApplication();

  int ret ;

  app->Start(argc, (char **)argv);
  if ( imageFile ) 
  {
    this->RenderWidget->OffScreenRenderingOn () ;
    this->RenderWidget->Render () ;
    this->SaveRenderAsBMP ( imageFile ) ;
  }

  ret = app->GetExitStatus();
  this->win->Close();
  
  this->win->Delete();
  return ret ;
  
}

void vtkKWMeshVisu::SaveRenderAsBMP(const char *filename)
{
  // mostly using the example in SOV
  vtkWindowToImageFilter* exporter = vtkWindowToImageFilter::New();
  exporter->SetInput((vtkWindow *)this->RenderWidget->GetRenderWindow());
  exporter->Modified() ;
  exporter->Update();

  vtkBMPWriter* writer = vtkBMPWriter::New();
  writer->SetInput(exporter->GetOutput());
  writer->SetFileName(filename);
  writer->Write();
  writer->Delete();
  exporter->Delete();
}

void vtkKWMeshVisu::AddBall (int vertexId)
{
  this->ROIMask->SetValue ( vertexId, true ) ;
  this->UpdateROIMaskedScalars ( vertexId ) ;

  this->RenderWidget->Render () ;
}

void vtkKWMeshVisu::RemoveBall ( int vertexId ) 
{
  this->ROIMask->SetValue ( vertexId, false ) ;
  this->UpdateROIMaskedScalars ( vertexId ) ;
}

bool vtkKWMeshVisu::HasBall ( int vertexId ) 
{
  return this->ROIMask->GetValue ( vertexId ) ;
}

void vtkKWMeshVisu::SwitchBall (int vertexId) 
{
  bool hasBall = this->HasBall ( vertexId ) ;
  if ( hasBall  )
  {
    this->RemoveBall ( vertexId ) ;
  }
  else
  {
    this->AddBall ( vertexId ) ;
  }
  this->RenderWidget->Render () ;
}

void vtkKWMeshVisu::SetupInteractor ()
{
  vtkRenderWindowInteractor *interactor = this->RenderWidget->GetRenderWindowInteractor () ;
  vtkCallbackCommand* callback = vtkCallbackCommand::New();
  callback->SetCallback(vtkKWMeshVisu::PointSelector);
  callback->SetClientData ( (void *) this) ;
  //interactor->AddObserver (vtkCommand::LeftButtonPressEvent, callback) ;
  interactor->AddObserver (vtkCommand::KeyPressEvent, callback) ;
  callback->Delete();
}

void vtkKWMeshVisu::PointSelector (class vtkObject *obj,unsigned long eid, void* clientdata, void *calldata)
{
  vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::SafeDownCast(obj);
  vtkKWMeshVisu *self = (vtkKWMeshVisu *) clientdata ;
  if ( !self->ROIMaskEnabled ) 
    return ; 

  if ( iren->GetKeyCode () == 'q' )
  {
    if ( self->vtkMesh == 0 )
      return ;
    
    int eventPos[2] ;
    iren->GetEventPosition (eventPos) ;

    vtkRenderer *renderer = iren->FindPokedRenderer (eventPos[0], eventPos[1]) ;
    if ( self->attributeDimension == 3 )
      renderer->RemoveActor ( self->arrowGlyphActor ) ;
    if ( self->attributeDimension == 9 )
      renderer->RemoveActor ( self->ellipsoidGlyphActor ) ;

    vtkPointPicker *picker = vtkPointPicker::New () ;
    if ( picker->Pick (eventPos[0], eventPos[1], 0, renderer) )
    {
      if ( picker->GetMapper() == self->meshMapper )
      {
        int point = picker->GetPointId () ;
	
	if ( !self->ROIBackPickEnabled ) 
	  {
	    // check that point is on front surface, not back
	    double cameraPos[3], cameraFoc[3], cameraDir[3], *pointNormal, dotproduct ;
	    vtkCamera *camera = renderer->GetActiveCamera() ;
	    camera->GetPosition ( cameraPos ) ;
	    camera->GetFocalPoint ( cameraFoc ) ;
	    cameraDir[0] = cameraFoc[0] - cameraPos[0] ;
	    cameraDir[1] = cameraFoc[1] - cameraPos[1] ;
	    cameraDir[2] = cameraFoc[2] - cameraPos[2] ;

	    pointNormal = self->normalFilter->GetOutput()->GetPointData()->GetNormals()->GetTuple3 ( point ) ;
	    dotproduct = cameraDir[0] * pointNormal[0] + cameraDir[1] * pointNormal[1] + cameraDir[2] * pointNormal[2] ;
	    if ( dotproduct < 0 )
	      {
		self->SwitchBall ( point ) ;
	      }
	  }
	else
	  self->SwitchBall ( point ) ;
      }
    }
    picker->Delete () ;

    if ( self->attributeDimension == 3 )
      renderer->AddActor ( self->arrowGlyphActor ) ;
    if ( self->attributeDimension == 9 ) 
      renderer->AddActor ( self->ellipsoidGlyphActor ) ;

    self->RenderWidget->Render() ;
  }
}

void vtkKWMeshVisu::SaveColors(const char *fileName)
{
  vtkPolyData *coloredMesh = vtkPolyData::New () ;
  coloredMesh->DeepCopy (this->vtkMesh) ;
  coloredMesh->GetPointData()->SetScalars ( this->meshMapper->MapScalars (this->opacityScale->GetValue()) ) ;

  vtkPolyDataWriter *writer = vtkPolyDataWriter::New () ;
  writer->SetInput ( coloredMesh ) ;
  writer->SetFileName ( fileName ) ;
  writer->Update () ;

  coloredMesh->Delete () ;
}

void vtkKWMeshVisu::MakeVideo ( const char *fileName ) 
{
   // there are some problems for this code, but it works for now
   // the camera's beginning configuration should be such that the focal point is at 0, and the up direction is (0,0,1). it also shouldn't be such that the camera position is (0,0,z)
   // i might also need to switch to off screen rendering, although this does seem to work properly right now.
   const double DEGREES_PER_FRAME = 1.0 ;
   const double pi = 3.141593 ;
   const double two_pi = 2.0 * pi ;
   const double radians_per_frame = DEGREES_PER_FRAME * pi / 180.0 ;

   char frameName[1000] ;
   double pos[3], currentPos[3], radius, alpha = 0.0 ;
   
   vtkCamera *camera = this->RenderWidget->GetRenderer()->GetActiveCamera();
   camera->GetPosition ( pos ) ;
   radius =  sqrt ( pos[0] * pos[0] + pos[1] * pos[1] ) ;
   currentPos[2] = pos[2] ;

   for ( int i = 0 ; alpha < two_pi ; i++ ) 
   {
      currentPos[0] = radius * cos ( alpha ) ;
      currentPos[1] = radius * sin ( alpha ) ;     
 
      camera->SetPosition ( currentPos ) ;
     
      this->RenderWidget->Modified () ;
      this->RenderWidget->Render () ;  
      
      sprintf ( frameName, "%s%03i.bmp", fileName, i ) ;
      this->SaveRenderAsBMP ( frameName ) ;   
     
      alpha += radians_per_frame ;
   }
}

void vtkKWMeshVisu::GenerateXMLDescription () 
{
  // this function generates the xml description of the command line parameters for slicer3
  
  cout << "<?xml version=\"1.0\" encoding=\"utf-8\"?>" << endl ;
  cout << "<executable>" << endl ;
  cout << "<category>UNC</category>" << endl ;
  cout << "<title>KWMeshVisu</title>" << endl ;
  cout << "<description>KWMeshVisu enables visualizing various data sets associated with surfaces. \
Given a surface mesh, KWMeshVisu can create customizable visualizations of scalar maps, vector fields, ellipsoid fields. \
Multiple attribute sets can also be layered in a single visualization.</description>" << endl ;
  cout << "<acknowledgements></acknowledgements>" << endl ;
  cout << "<contributor>Ipek Oguz, Martin Styner, Sebastien Barre, James Miller\nThis work is supported by the NIH Roadmap \
for Medical Research Grant U54 EB005149-01, UNC Neurodevelopmental Disorders Research Center HD 03110, and the NIH NIBIB \
grant P01 EB002779.</contributor>" << endl ;
  cout << "<parameters>" << endl ;
  cout << "<label>Input/Output Files</label>" << endl ;
  cout << "<description>" << "The surface mesh and the attribute files, as well as\
 the image file to be created." << "</description>" << endl ;
  
  cout << "<geometry>" << endl ;
  cout << "<name>mesh</name>" << endl ;
  cout << "<label>Surface mesh</label>" << endl ;
  cout << "<channel>input</channel>" << endl ;
  cout << "<longflag>mesh</longflag>" << endl ;
  cout << "</geometry>" << endl ;
  
  cout << "<file>" << endl ;
  cout << "<name>scalar</name>" << endl ;
  cout << "<label>1D Attribute</label>" << endl ;
  cout << "<longflag>scalar</longflag>" << endl ;
  cout << "<default>NULL</default>" << endl ;
  cout << "</file>" << endl ;
  
  cout << "<file>" << endl ;
  cout << "<name>vector</name>" << endl ;
  cout << "<label>3D Attribute</label>" << endl ;
  cout << "<longflag>vector</longflag>" << endl ;
  cout << "<default>NULL</default>" << endl ;
  cout << "</file>" << endl ;

  cout << "<file>" << endl ;
  cout << "<name>ellipsoid</name>" << endl ;
  cout << "<label>Ellipsoid Attribute</label>" << endl ;
  cout << "<longflag>ellipsoid</longflag>" << endl ;
  cout << "<default>NULL</default>" << endl ;
  cout << "</file>" << endl ;
  
  cout << "<file>" << endl ;
  cout << "<name>image</name>" << endl ;
  cout << "<label>Output image</label>" << endl ;
  cout << "<longflag>image</longflag>" << endl ;
  cout << "<default>NULL</default>" << endl ;
  cout << "</file>" << endl ;

  cout << "</parameters>" << endl ;

  cout << "<parameters>" << endl ;
  cout << "<label>Visualization Parameters</label>" << endl ;
  cout << "<description>Parameters regarding the visualization options</description>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>opacity</name>" << endl ;
  cout << "<label>Mesh Opacity</label>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>1.0</maximum>" << endl ;
  cout << "<default>1.0</default>" << endl ;
  cout << "<longflag>opacity</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>scale</name>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>20.0</maximum>" << endl ;
  cout << "<default>1.0</default>" << endl ;
  cout << "<label>Feature Scale</label>" << endl ;
  cout << "<longflag>scale</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>br</name>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>1.0</maximum>" << endl ;
  cout << "<default>0.0</default>" << endl ;
  cout << "<label>Background Color - Red</label>" << endl ;
  cout << "<longflag>br</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>bg</name>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>1.0</maximum>" << endl ;
  cout << "<default>0.0</default>" << endl ;
  cout << "<label>Background Color - Green</label>" << endl ;
  cout << "<longflag>bg</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>bb</name>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>1.0</maximum>" << endl ;
  cout << "<default>0.0</default>" << endl ;
  cout << "<label>Background Color - Blue</label>" << endl ;
  cout << "<longflag>bb</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<string-enumeration>" << endl ;
  cout << "<name>visMode</name>" << endl ;
  cout << "<element>PValue</element>" << endl ;
  cout << "<element>DistanceMap</element>" << endl ;
  cout << "<default>PValue</default>" << endl ;
  cout << "<label>Visualization Mode</label>" << endl ;
  cout << "<longflag>visMode</longflag>" << endl ;
  cout << "</string-enumeration>" << endl ;
  
  cout << "<float>" << endl ;
  cout << "<name>pvalue</name>" << endl ;
  cout << "<minimum>0.0</minimum>" << endl ;
  cout << "<maximum>1.0</maximum>" << endl ;
  cout << "<default>0.05</default>" << endl ;
  cout << "<label>P Value Threshold</label>" << endl ;
  cout << "<longflag>pvalue</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>distMapMin</name>" << endl ;
  cout << "<default>0.0</default>" << endl ;
  cout << "<label>Distance Map - Min</label>" << endl ;
  cout << "<longflag>distMapMin</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>distMapMax</name>" << endl ;
  cout << "<default>1.0</default>" << endl ;
  cout << "<label>Distance Map - Max</label>" << endl ;
  cout << "<longflag>distMapMax</longflag>" << endl ;
  cout << "</float>" << endl ;

  cout << "</parameters>" << endl ;
 
  cout << "<parameters>" << endl ;
  cout << "<label>Camera Parameters</label>" << endl ;
  cout << "<description>Camera's position, focal point, up-vector and zoom amount</description>" << endl ;
  cout << "<boolean>" << endl ;
  cout << "<name>cameraSwitch</name>" << endl ;
  cout << "<label>Use Camera Parameters</label>" << endl ;
  cout << "<longflag>cameraSwitch</longflag>" << endl ;
  cout << "<default>false</default>" << endl ;
  cout << "</boolean>" << endl ;

  cout << "<float>" << endl ;
  cout << "<name>posx</name>" << endl ;
  cout << "<label>Position - X</label>" << endl ;
  cout << "<longflag>posx</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>posy</name>" << endl ;
  cout << "<label>Position - Y</label>" << endl ;
  cout << "<longflag>posy</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>posz</name>" << endl ;
  cout << "<label>Position - Z</label>" << endl ;
  cout << "<longflag>posz</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>focx</name>" << endl ;
  cout << "<label>Focal Point - X</label>" << endl ;
  cout << "<longflag>focx</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>focy</name>" << endl ;
  cout << "<label>Focal Point - Y</label>" << endl ;
  cout << "<longflag>focy</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>focz</name>" << endl ;
  cout << "<label>Focal Point - Z</label>" << endl ;
  cout << "<longflag>focz</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>upx</name>" << endl ;
  cout << "<label>Up Vector - X</label>" << endl ;
  cout << "<longflag>upx</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>upy</name>" << endl ;
  cout << "<label>Up Vector - Y</label>" << endl ;
  cout << "<longflag>upy</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>upz</name>" << endl ;
  cout << "<label>Up Vector - Z</label>" << endl ;
  cout << "<longflag>upz</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "<float>" << endl ;
  cout << "<name>zoom</name>" << endl ;
  cout << "<label>Zoom</label>" << endl ;
  cout << "<longflag>zoom</longflag>" << endl ;
  cout << "</float>" << endl ;
  cout << "</parameters>" << endl ;
    
  cout << "</executable>" << endl ;
  
}

