#include "ScanConvCLP.h"
#include <iostream>
#include <fstream>
#include <vector>
#include <string>

#include "itkImage.h"
#include "itkImageFileWriter.h"
#include "itkImageFileReader.h"
#include "itkSignedDanielssonDistanceMapImageFilter.h"
#include "itkDanielssonDistanceMapImageFilter.h"
#include "itkDiscreteGaussianImageFilter.h"

#include "vtkPolyData.h"
#include "vtkPolyDataReader.h"
#include "vtkFloatArray.h"
#include "vtkPointData.h"
#include "vtkMetaImageWriter.h"
#include "vtkImageData.h"
#include "vtkFSSurfaceReader.h"
#include "vtkFSSurfaceScalarReader.h"
#include "vtkWindowedSincPolyDataFilter.h" // Pengdong Xiao

#include "vtkAttributedPolyDataToImage.h"
#include "KWMeshVisuAttributeIO.h"

#include "MeshIO.h"

typedef itk::Image < double, 3 > DoubleImageType ;
typedef DoubleImageType::Pointer DoubleImagePointer ;
typedef itk::ImageFileWriter < DoubleImageType > DoubleImageWriterType ; 

typedef itk::Image < int, 3 > ImageType ;
typedef ImageType::Pointer ImagePointer ;
typedef itk::ImageFileWriter < ImageType > ImageWriterType ; 

typedef itk::ImageFileReader < ImageType > ImageReaderType ;
typedef itk::SignedDanielssonDistanceMapImageFilter < ImageType, DoubleImageType > SignedDDMapType ;
typedef itk::DanielssonDistanceMapImageFilter <DoubleImageType, DoubleImageType > DDMapType ;

typedef itk::DiscreteGaussianImageFilter < DoubleImageType, DoubleImageType > gaussFilterType ;

std::vector < std::string > subjectNames ;
std::vector < std::vector < std::string > > attributeNames ;
std::vector < double > weights ;
std::vector < ImagePointer > binaryVolumes ;
std::vector < std::vector < vtkImageData * > > attributeVolumes ;

void WriteVTKImage ( vtkImageData *image, std::string fileName ) 
{
  vtkMetaImageWriter *imWriter = vtkMetaImageWriter::New () ;
  imWriter->SetInput ( image ) ;
  imWriter->SetFileName ( fileName.c_str() ) ;
  imWriter->Write() ;
  std::cout << "Image written: " << fileName << std::endl ;
  imWriter->Delete () ;
}

void WriteITKImage ( ImagePointer image, std::string fileName ) 
{
  ImageWriterType::Pointer itkWriter = ImageWriterType::New() ;
  itkWriter->SetFileName ( fileName ) ;
  itkWriter->SetInput ( image ) ;
  //std::cout << "blergh" << std::endl ;
  ImageType::PointType o = image->GetOrigin() ;
  //std::cout << o[0] << " " << o[1] << " " << o[2] << std::endl ;
  itkWriter->Write() ;
  std::cout << "Image written: " << fileName << std::endl ;
}

void WriteITKDoubleImage ( DoubleImagePointer image, std::string fileName ) 
{
  DoubleImageWriterType::Pointer itkWriter = DoubleImageWriterType::New() ;
  itkWriter->SetFileName ( fileName ) ;
  itkWriter->SetInput ( image ) ;
  itkWriter->Write() ;
  std::cout << "Image written: " << fileName << std::endl ;
}

vtkImageData *ITK2VTK ( ImagePointer itkImage ) 
{
  vtkImageData *vtkImage = vtkImageData::New () ;
  ImageType::PointType origin = itkImage->GetOrigin () ;
  vtkImage->SetOrigin ( origin[0], origin[1], origin[2] ) ;
  ImageType::SpacingType spacing = itkImage->GetSpacing () ;
  vtkImage->SetSpacing ( spacing[0], spacing[1], spacing[2] ) ;
  ImageType::SizeType size = itkImage->GetLargestPossibleRegion().GetSize() ;
  vtkImage->SetDimensions ( size[0], size[1], size[2] ) ;
  vtkImage->SetScalarTypeToLong() ;
  vtkImage->AllocateScalars();
  vtkImage->Update() ;

  ImageType::IndexType index ; 
  long pixel ;
  for ( index[0] = 0 ; index[0] < size[0] ; index[0]++ )
    {
      for ( index[1] = 0 ; index[1] < size[1] ; index[1]++ )
	{
	  for ( index[2] = 0 ; index[2] < size[2] ; index[2]++ )
	    {
	      pixel = itkImage->GetPixel ( index ) ;
	      //vtkImage->SetScalarComponentFromDouble ( size[0] - 1 - index[0], size[1] - 1 - index[1], index[2], 0, pixel ) ;
	      vtkImage->SetScalarComponentFromDouble ( index[0], index[1], index[2], 0, pixel ) ;
	    }
	}
    }
  return vtkImage ;
}

ImagePointer VTK2BinaryITK ( vtkImageData *vtkImage, int *dims, double *origin, double spacing ) 
{
  // convert the vtk image to an itk image
  ImageType::SizeType size ;
  size[0] = dims[0] ;
  size[1] = dims[1] ;
  size[2] = dims[2] ;
  
  ImagePointer itkImage = ImageType::New () ;
  
  ImageType::RegionType region;
  region.SetSize( size );
  ImageType::IndexType start;
  start[0] = start[1] = start[2] = 0;  
  region.SetIndex( start );
  itkImage->SetRegions( region );
  itkImage->SetOrigin ( origin ) ;

  double temp[3] ;
  temp[0] = temp[1] = temp[2] = spacing ;
  itkImage->SetSpacing ( spacing ) ;
  itkImage->Allocate () ;
  
  ImageType::IndexType index ;
  int pixel ;

  for ( index[0] = 0 ; index[0] < dims[0] ; index[0]++ )
  {
    for ( index[1] = 0 ; index[1] < dims[1] ; index[1]++ )
    {
      for ( index[2] = 0 ; index[2] < dims[2] ; index[2]++ )
      {
        //pixel = vtkImage->GetScalarComponentAsFloat ( dims[0] - 1 - index[0], dims[1] - 1 - index[1], index[2], 0 );            
	pixel = vtkImage->GetScalarComponentAsFloat ( index[0], index[1], index[2], 0 ) ;
	if ( pixel != 128 )
	  pixel = 0 ;
        itkImage->SetPixel ( index, pixel ) ;
      }
    }
  }
  
  //std::cout << "VTK to ITK conversion completed." << std::endl ;
  return itkImage ;
}

DoubleImagePointer VTK2DoubleITK ( vtkImageData *vtkImage, int *dims, double *origin, double spacing ) 
{
  // convert the vtk image to an itk image
  DoubleImageType::SizeType size ;
  size[0] = dims[0] ;
  size[1] = dims[1] ;
  size[2] = dims[2] ;
  
  DoubleImagePointer itkImage = DoubleImageType::New () ;
  
  DoubleImageType::RegionType region;
  region.SetSize( size );
  DoubleImageType::IndexType start;
  start[0] = start[1] = start[2] = 0;  
  region.SetIndex( start );
  itkImage->SetRegions( region );
  itkImage->SetOrigin ( origin ) ;

  double temp[3] ;
  temp[0] = temp[1] = temp[2] = spacing ;
  itkImage->SetSpacing ( spacing ) ;
  itkImage->Allocate () ;

  DoubleImageType::IndexType index ;
  //long int face ;
  double face ;
  for ( index[0] = 0 ; index[0] < dims[0] ; index[0]++ )
  {
    for ( index[1] = 0 ; index[1] < dims[1] ; index[1]++ )
    {
      for ( index[2] = 0 ; index[2] < dims[2] ; index[2]++ )
      {
        //face = vtkImage->GetScalarComponentAsFloat ( dims[0] - 1 - index[0], dims[1] - 1 - index[1], index[2], 0 );            
	face = vtkImage->GetScalarComponentAsFloat ( index[0], index[1], index[2], 0 ) ;
	itkImage->SetPixel ( index, face ) ;
      }
    }
  }
  
  //std::cout << "VTK to double ITK conversion completed." << std::endl ;
  return itkImage ;
}



bool parseParameterFile ( std::string fileName, int &nShapes, int &nAttributes, int &nTemplateIndex )
{
  std::ifstream input ;
  input.open ( fileName.c_str () ) ;

  bool found;
  char * valuePtr;
  char line[1001];

  input.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !input.eof())
    { input.getline ( line, 1000 ) ;
      if (line[0] != '#' && strstr ( line, "NUMBER_OF_SHAPES" )) found = true;
    }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return false ;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nShapes);
  //std::cout << "nShapes: " << nShapes << std::endl ;

  input.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !input.eof())
    { input.getline ( line, 1000 ) ;
      if (line[0] != '#' && strstr ( line, "NUMBER_OF_ATTRIBUTES" )) found = true;
    }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return false ;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nAttributes);
  //std::cout << "nAttributes: " << nAttributes << std::endl ;

  input.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !input.eof())
    { input.getline ( line, 1000 ) ;
      if (line[0] != '#' && strstr ( line, "TEMPLATE_INDEX" )) found = true;
    }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return false ;
  valuePtr++;
  sscanf(valuePtr, " %d ", &nTemplateIndex);
  //std::cout << "Template index: " << nTemplateIndex << std::endl ;

  input.seekg(0,std::ios::beg);
  found = false ;
  while ( !found && !input.eof())
    { input.getline ( line, 1000 ) ;
      if (line[0] != '#' && strstr ( line, "WEIGHTS" )) found = true;
    }
  valuePtr=strchr(line, '=');
  if (!valuePtr) return false;
  weights.resize ( nAttributes ) ;
  //std::cout << "Weights: " ;
  float temp ;
  for ( int i = 0 ; i < nAttributes; i++ )
  {
    valuePtr++;
    sscanf(valuePtr, "%f", &temp);
    weights[i] = temp ;
    valuePtr = strchr(valuePtr, ' ') ;
    //std::cout << weights[i] << " " ; 
  } 
  //std::cout << std::endl ;

  input.seekg(0,std::ios::beg);
  const int numEntries = 4;
  int counter = 0;
  while ( counter < numEntries && !input.eof())
    { input.getline ( line, 1000 ) ;
      if ((line[0] != '#')) counter++;
    }
  
  subjectNames.resize ( nShapes ) ;
  attributeNames.resize ( nShapes ) ;
  for (int i = 0 ; i < nShapes ; i++ )
    {
      input >> subjectNames[i] ;
      //std::cout << i << ":" << subjectNames[i] << ":" << std::endl ;
      attributeNames[i].resize ( nAttributes ) ;
      for ( int j = 0 ; j < nAttributes ; j++ )
	{
	  input >> attributeNames[i][j] ; 
	  //std::cout << i << ":" << j << ":" << attributeNames[i][j] << ":" << std::endl ;
        }
    }

  input.close () ;
  return true ;
}

vtkFloatArray *ReadScalars ( std::string fileName ) 
{
  vtkFloatArray *scalars = vtkFloatArray::New () ;
  int lastPoint = fileName.rfind ( '.' ) ;
  std::string extension ;
  if ( lastPoint > 0 ) 
    {
      extension = fileName.substr ( lastPoint ) ;
    }
  if ( extension.compare ( ".txt" ) == 0 )
    {
      // KWMeshVisu style scalars
      KWMeshVisuAttributeIO *attributeReader = new ( KWMeshVisuAttributeIO ) ;
      attributeReader->SetFileName ( fileName.c_str() ) ;
      attributeReader->ReadAttributes () ;
      scalars = attributeReader->GetAttributes () ;
      delete ( attributeReader ) ;
      if ( ( scalars->GetNumberOfComponents () != 1 ) )
	return 0 ;
    }
  else
    {
      // FreeSurfer style scalars
      vtkFSSurfaceScalarReader *fsScalarReader = vtkFSSurfaceScalarReader::New () ;
      fsScalarReader->SetFileName ( fileName.c_str() ) ;
      fsScalarReader->SetOutput ( scalars ) ;
      fsScalarReader->ReadFSScalars() ; 
      scalars = fsScalarReader->GetOutput () ;
      fsScalarReader->Delete () ;
    }

  return scalars ;
}

// Pengdong Xiao -- start -- end
void ScanConvert ( int nShapes, int nAttributesPerShape, double scale, std::string directory, bool center, double extraBorder, bool RelaxPolygonFlag) 
{
   // we'll need to compute the bounding box for the entire population  
   double largestBoundaries[6] ;
   largestBoundaries[0] = largestBoundaries[2] = largestBoundaries[4] = 999999 ;
   largestBoundaries[1] = largestBoundaries[3] = largestBoundaries[5] = -999999 ;

   std::vector < vtkPolyData* > meshes ;
   meshes.resize ( nShapes ) ;

   // ************ VARIANCE COMPUTING *************
   std::vector < std::vector <float> > Mean;
   std::vector < std::vector <float> > Variance;

   Mean.resize ( nShapes ) ;
   Variance.resize ( nShapes ) ;
   for (int i = 0 ; i < nShapes ; i++ )
   {
      Mean[i].resize ( nAttributesPerShape ) ;
      Variance[i].resize ( nAttributesPerShape ) ;
      for ( int j = 0 ; j < nAttributesPerShape ; j++ )
	{
	  Mean[i][j] = 0.0; 
	  Variance[i][j] = 0.0;
        }
    }

   // ************ VARIANCE COMPUTING *************

   for ( int i = 0 ; i < nShapes ; i++ )
     {
       double bb[6] ;
       MeshIO *meshIOtool = new ( MeshIO ) ;

       if ( meshIOtool->Read ( subjectNames[i] ) )
	 {
	   meshes[i] = meshIOtool->GetMesh () ;
	 }

       // Pengdong Xiao -- start
       if (RelaxPolygonFlag == true)
         {
	   //std::cout << " Pengdong Xiao:: relax polygon start." << std::endl;
           vtkWindowedSincPolyDataFilter* smoother = vtkWindowedSincPolyDataFilter::New();
           smoother->SetInput( meshes[i] );
           smoother->SetNumberOfIterations(20);//20
           smoother->SetFeatureAngle(60);//60
           smoother->SetPassBand(0.05);//0.05
           smoother->FeatureEdgeSmoothingOff();//false
           smoother->BoundarySmoothingOff();//false
           smoother->NonManifoldSmoothingOn();//true
           smoother->NormalizeCoordinatesOn();//true
	   smoother->GenerateErrorScalarsOn();//true
           smoother->GenerateErrorVectorsOn();//true
           smoother->Update();

           meshes[i] = smoother->GetOutput();
	   //std::cout << " Pengdong Xiao:: relax polygon end." << std::endl;
         }
       // Pengdong Xiao -- end

       meshes[i]->GetBounds ( bb ) ;
       std::cout << "BB: " << bb[0] << " " << bb[1] << " " << bb[2] << " " << bb[3] << " " << bb[4] << " " << bb[5] << std::endl ;

       if ( center ) 
	 {
	   double bbSize[3], centerOffset[3] ;

	   bbSize[0] = bb[1] - bb[0] ;
	   bbSize[1] = bb[3] - bb[2] ;
	   bbSize[2] = bb[5] - bb[4] ;
	   centerOffset[0] = -bbSize[0] * 0.5 - bb[0] ;
	   centerOffset[1] = -bbSize[1] * 0.5 - bb[2] ;
	   centerOffset[2] = -bbSize[2] * 0.5 - bb[4] ;
	   bb[0] = -bbSize[0] * 0.5 ;
	   bb[1] = bbSize[0] * 0.5 ;
	   bb[2] = -bbSize[1] * 0.5 ;
	   bb[3] = bbSize[1] * 0.5 ;
	   bb[4] = -bbSize[2] * 0.5 ;
	   bb[5] = bbSize[2] * 0.5 ;
           std::cout << "ReadMesh completed for " << subjectNames[i] << " " ;
	   std::cout << "Offsetting by " << centerOffset[0] << " " << centerOffset[1] << " " << centerOffset[2] << std::endl ;
	 }

       std::cout << "BBNEW: " << bb[0] << " " << bb[1] << " " << bb[2] << " " << bb[3] << " " << bb[4] << " " << bb[5] << std::endl ;



       if ( bb[0] < largestBoundaries[0] )
         largestBoundaries[0] = bb[0] ;
       if ( bb[2] < largestBoundaries[2] )
         largestBoundaries[2] = bb[2] ;
       if ( bb[4] < largestBoundaries[4] )
         largestBoundaries[4] = bb[4] ;
       if ( bb[1] > largestBoundaries[1] )
         largestBoundaries[1] = bb[1] ;
       if ( bb[3] > largestBoundaries[3] )
         largestBoundaries[3] = bb[3] ;
       if ( bb[5] > largestBoundaries[5] )
         largestBoundaries[5] = bb[5] ;
       delete ( meshIOtool ) ;
     }

  

 /* if (extraBorder != 0 )
   {
	largestBoundaries[0] += extraBorder ;
        largestBoundaries[1] += extraBorder ;
        largestBoundaries[2] += extraBorder ;
	largestBoundaries[3] += extraBorder ;
        largestBoundaries[4] += extraBorder ;
        largestBoundaries[5] += extraBorder ;
	std::cout << "Extra border added! " << std::endl ; 
   } */

 std::cout << "Largest boundaries: " << largestBoundaries[0] << " " << largestBoundaries[1] << " " << largestBoundaries[2] << " " << largestBoundaries[3] << " " << largestBoundaries[4] << " " << largestBoundaries[5] << std::endl ;

   // compute image size, origin and spacing given the common bounding box
   double  origin[3], voxelSize[3] ;
   int size[3] ;
   voxelSize[0] = voxelSize[1] = voxelSize[2] = scale;
   for ( int i = 0 ; i < 3 ; i++ )
     {
       origin[i] = floor ( largestBoundaries[2*i] ) - 1  ;
       size[i] = ceil ( ( ceil ( largestBoundaries[2*i+1] ) - origin[i] ) / voxelSize[i] ) + 1 ;
     }

if (extraBorder != 0 )
   {
	size[0] += extraBorder ;
        size[1] += extraBorder ;
        size[2] += extraBorder ;
	origin[0] -= extraBorder/2 ;
        origin[1] -= extraBorder/2 ;
        origin[2] -= extraBorder/2 ;
	std::cout << "Extra border added! " << std::endl ; 
   }

   std::cout << "Origin: " << origin[0] << " " << origin[1] << " " << origin[2] << std::endl ;
   std::cout << "Size: " << size[0] << " " << size[1] << " " << size[2] << std::endl ;
     
   binaryVolumes.resize ( nShapes ) ;
   attributeVolumes.resize ( nShapes ) ;

   double shapeOrigin[3] ;  
   for ( int i = 0 ; i < nShapes ; i++ )
     {
       // Scan-convert the mesh
       vtkAttributedPolyDataToImage *scanConverter = vtkAttributedPolyDataToImage::New () ;
       scanConverter->SetTolerance ( 0.0 ) ;
       scanConverter->ReleaseDataFlagOn () ;
       scanConverter->SetInput ( meshes[i] ) ; 

       if ( center ) 
	 {
	   double bbSize[3], centerOffset[3], bb[6] ;
	   meshes[i]->GetBounds ( bb ) ;

	   bbSize[0] = bb[1] - bb[0] ;
	   bbSize[1] = bb[3] - bb[2] ;
	   bbSize[2] = bb[5] - bb[4] ;
	   centerOffset[0] = -bbSize[0] * 0.5 - bb[0] ;
	   centerOffset[1] = -bbSize[1] * 0.5 - bb[2] ;
	   centerOffset[2] = -bbSize[2] * 0.5 - bb[4] ;
	   shapeOrigin[0] = origin[0] - centerOffset[0] ;
	   shapeOrigin[1] = origin[1] - centerOffset[1] ;
	   shapeOrigin[2] = origin[2] - centerOffset[2] ;
	   scanConverter->SetOutputOrigin ( shapeOrigin ) ;
	 }
       else
	 scanConverter->SetOutputOrigin ( origin ) ;

       scanConverter->SetOutputSpacing ( voxelSize ) ;

       scanConverter->SetOutputWholeExtent ( 0, size[0] - 1, 0, size[1] - 1, 0, size[2] - 1 ) ;
       scanConverter->Update () ;

       vtkImageData *vtkBinaryVolume = scanConverter->GetBinaryVolume () ;
       binaryVolumes[i] = VTK2BinaryITK ( vtkBinaryVolume, size, origin, voxelSize[0] ) ;
       
       if ( nAttributesPerShape ) 
	 {
           attributeVolumes[i].resize ( nAttributesPerShape ) ;
           char surfaceName[2000] ;
           for ( int j = 0 ; j < nAttributesPerShape ; j++ )
	     {
	       vtkFloatArray *scalars = ReadScalars ( attributeNames[i][j] ) ;
	       if ( meshes[i]->GetNumberOfPoints() != scalars->GetNumberOfTuples () )
	         {
	           std::cout << "Different number of vertices between mesh and attribute files" << std::endl ;
	           std::cout << "Mesh has: " << meshes[i]->GetNumberOfPoints() << " vertices" << std::endl ;
	           std::cout << "Attribute file has " << scalars->GetNumberOfTuples ()  << " vertices" << std::endl ;
	           exit (0) ;
	         }

	       scanConverter->SetAttributes ( scalars ) ;
	       attributeVolumes[i][j] = scanConverter->GetAttributeVolume () ;

	       DDMapType::Pointer ddmap = DDMapType::New () ;
	       DoubleImagePointer attributes = VTK2DoubleITK ( attributeVolumes[i][j], size, origin, voxelSize[0] ) ;
	       
	       ddmap->SetInput ( attributes ) ;
	       ddmap->Update () ; 

	       sprintf ( surfaceName, "%s/subject_%03d_Attribute_%02d.mha", directory.c_str(), i, j ) ;
	       WriteITKDoubleImage ( ddmap->GetVoronoiMap (), surfaceName ) ;
	       //WriteITKDoubleImage ( attributes, surfaceName ) ;
	       //scalars->Delete () ;
	     } //EndFOR j # of attributes

	  }
       
       scanConverter->Delete () ;
       //vtkBinaryVolume->Delete () ;
    } // EndFOR i # of shapes

   for ( int i = 0 ; i < nShapes ; i++ )
     {
       meshes[i]->Delete () ;
     }
}

int main ( int argc, char *argv[] )
{
  PARSE_ARGS ;
  
  int nAttributesPerShape, nShapes, templateIndex ;
  
  // parse the parameter file
  if ( !parseParameterFile ( parameterFileName, nShapes, nAttributesPerShape, templateIndex ) )
     return EXIT_FAILURE ;

  std::cout << nShapes << " " << nAttributesPerShape << " " << templateIndex << std::endl ;
  
  // figure out where to put all the output files
  std::string directory ; 
  if ( outputDirectory.compare ( "" ) == 0 ) 
    {
      int lastSlash = subjectNames[0].rfind ( '/' ) ;
      std::string path ;
      if ( lastSlash > 0 )
	{
	  directory = subjectNames[0].substr ( 0, lastSlash ) ;
	}
      else
	{
	  directory = "." ;
	}
    }
  else
    {
      directory = outputDirectory ;
    }
  std::cout << "Output Directory: " << directory << std::endl ;

  // start by reading the polydata files and converting them to volumes 
  ScanConvert ( nShapes, nAttributesPerShape, scale, directory, CenterBoundingBoxes, extraBorder, RelaxPolygonFlag) ; // Pengdong Xiao
 // std::cout << "So far so good" << std::endl ;

  // The following of writing out binary volumes is set to always true. Pengdong Xiao
  char surfaceName[2000] ; 
  for ( int i = 0 ; i < nShapes ; i++ )
    {
      sprintf ( surfaceName, "%s/subject_%03d_Binary.mha", directory.c_str(), i ) ;
      std::cout << "Writing out " << surfaceName << std::endl ;
      //  std::cout << binaryVolumes[i] << std::endl ;
      WriteITKImage ( binaryVolumes[i], surfaceName ) ;
    }

  return EXIT_SUCCESS ;
} 
