/**
 * @file  test_utilities.cxx
 * @brief Test of imaging utility functions.
 *
 * Copyright (c) 2012 University of Pennsylvania. All rights reserved.<br />
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */


#include <stdexcept>

#include <basis/test.h>       // unit testing framework

#include "common/general.h"   // round()
#include "common/utilities.h" // testee


using namespace std;
using namespace dramms;


// ===========================================================================
// constants
// ===========================================================================

// size of test image
const int dim[3] = {13, 11, 7};

// voxel size of test image
const float pixdim[3] = {1, 1.5, 2};

// number of foreground voxels in test image
const int npos = 9;

// foreground voxel positions in test image
const int pos[npos][3] = {
    { 0,  0, 0}, // 8 corners
    { 0,  0, 6},
    { 0, 10, 6},
    { 0, 10, 0},
    {12,  0, 0},
    {12,  0, 6},
    {12, 10, 0},
    {12, 10, 6},
    { 6,  5, 3}, // center
    // add further positions if desired below and increase npos,
    // otherwise it will likely break existing tests
};

// ===========================================================================
// helper
// ===========================================================================

// ---------------------------------------------------------------------------
// Generate test image.
Image* generate_test_image(int x_orient = NIFTI_L2R,
                           int y_orient = NIFTI_P2A,
                           int z_orient = NIFTI_S2I)
{
    Image* image = new Image(dim[0], dim[1], dim[2],
                             DT_UNSIGNED_CHAR, 1,
                             Image::FORMAT_ITK);
                             // TODO Test using image in FORMAT_DRAMMS.
                             //      Yet, the tests fail when this format
                             //      is used instead...
    // orientation
    image->hdr.qform_code = NIFTI_XFORM_SCANNER_ANAT;
    mat44 q;
    for (int r = 0; r < 4; r++) {
        for (int c = 0; c < 4; c++) q.m[r][c] = 0;
    }
    if (x_orient == NIFTI_L2R) {
        q.m[0][0] = pixdim[0];
    } else {
        q.m[0][0] = - pixdim[0];
    }
    if (y_orient == NIFTI_P2A) {
        q.m[1][1] = pixdim[1];
    } else {
        q.m[1][1] = - pixdim[1];
    }
    if (z_orient == NIFTI_S2I) {
        q.m[2][2] = - pixdim[2];
    } else {
        q.m[2][2] = pixdim[2];
    }
    // origin
    q.m[3][0] = 33;
    q.m[3][1] = -42;
    q.m[3][2] = 123;
    // set quaternion and offset representation
    nifti_mat44_to_quatern(q, &image->hdr.quatern_b,
                              &image->hdr.quatern_c,
                              &image->hdr.quatern_d,
                              &image->hdr.qoffset_x,
                              &image->hdr.qoffset_y,
                              &image->hdr.qoffset_z,
                              &image->hdr.pixdim[1],
                              &image->hdr.pixdim[2],
                              &image->hdr.pixdim[3],
                              &image->hdr.pixdim[0]);
    // set some sform transformation
    image->hdr.sform_code = NIFTI_XFORM_ALIGNED_ANAT;
    image->hdr.srow_x[0] = 0;
    image->hdr.srow_x[1] = -0.83867;
    image->hdr.srow_x[2] = 0.544639;
    image->hdr.srow_x[3] = -56;
    image->hdr.srow_y[0] = 0.707107;
    image->hdr.srow_y[1] = 0.385118;
    image->hdr.srow_y[2] = 0.322987;
    image->hdr.srow_y[3] = 200;
    image->hdr.srow_z[0] = -0.707107;
    image->hdr.srow_z[1] = 0.385118;
    image->hdr.srow_z[2] = 0.54303;
    image->hdr.srow_z[3] = 77;
    // assign indices to voxels
    for (int i = 0; i < npos; i++) {
        image->set(pos[i][0], pos[i][1], pos[i][2], i + 1);
    }
    return image;
}

// ---------------------------------------------------------------------------
// Get qform transform from ijk voxel coordinates to xyz coordinates.
mat44 qto_xyz(const Image* image)
{
    return GetQFormTransform(image);
}

// ---------------------------------------------------------------------------
mat44 qto_ijk(const Image* image)
{
    return nifti_mat44_inverse(qto_xyz(image));
}

// ---------------------------------------------------------------------------
// Get sform transform from ijk voxel coordinates to xyz coordinates.
mat44 sto_xyz(const Image* image)
{
    return GetSFormTransform(image);
}

// ---------------------------------------------------------------------------
mat44 sto_ijk(const Image* image)
{
    return nifti_mat44_inverse(sto_xyz(image));
}

// ---------------------------------------------------------------------------
// Transform voxel coordinate to anatomical coordinate.
Fvector3d to_xyz(const mat44& m, const Ivector3d& v)
{
    Fvector3d x;
    x.x = v.x * m.m[0][0] + v.y * m.m[0][1] + v.z * m.m[0][2] + m.m[0][3];
    x.y = v.x * m.m[1][0] + v.y * m.m[1][1] + v.z * m.m[1][2] + m.m[1][3];
    x.z = v.x * m.m[2][0] + v.y * m.m[2][1] + v.z * m.m[2][2] + m.m[2][3];
    return x;
}

// ---------------------------------------------------------------------------
// Transform anatomical coordinate to voxel coordinate.
Ivector3d to_ijk(const mat44& m, const Fvector3d& x)
{
    Ivector3d v;
    v.x = static_cast<int>(round(x.x * m.m[0][0] + x.y * m.m[0][1] + x.z * m.m[0][2] + m.m[0][3]));
    v.y = static_cast<int>(round(x.x * m.m[1][0] + x.y * m.m[1][1] + x.z * m.m[1][2] + m.m[1][3]));
    v.z = static_cast<int>(round(x.x * m.m[2][0] + x.y * m.m[2][1] + x.z * m.m[2][2] + m.m[2][3]));
    return v;
}

// ===========================================================================
// image grid manipulations
// ===========================================================================

// ---------------------------------------------------------------------------
// Test invalid permutation.
TEST(Utilities, PermuteAxesInvalid)
{
    Image* test_image = generate_test_image();
    int order1[3] = {0, 0, 1};
    EXPECT_FALSE(PermuteAxes(test_image, order1));
    int order2[3] = {3, 0, 1};
    EXPECT_FALSE(PermuteAxes(test_image, order1));
    delete test_image;
}

// ---------------------------------------------------------------------------
// Test permutation of image axes.
TEST(Utilities, PermuteAxes)
{
    // create test image
    Image* test_image = generate_test_image();
    // set some more complicated qform transformation
    test_image->hdr.quatern_b = 0;
    test_image->hdr.quatern_c = 0;
    test_image->hdr.quatern_d = 0.707106781;
    // coordinates of test points
    mat44 qm = qto_xyz(test_image);
    mat44 sm = sto_xyz(test_image);
    Ivector3d v1, v2, v3, v4;
    v1.x = v1.y = v1.z = 0;
    v2.x =  0; v2.y = 10; v2.z = 6;
    v3.x = 12; v3.y = 10; v3.z = 6;
    v4.x =  6; v4.y =  5; v4.z = 3;
    const Fvector3d qp1 = to_xyz(qm, v1);
    const Fvector3d qp2 = to_xyz(qm, v2);
    const Fvector3d qp3 = to_xyz(qm, v3);
    const Fvector3d qp4 = to_xyz(qm, v4);
    const Fvector3d sp1 = to_xyz(sm, v1);
    const Fvector3d sp2 = to_xyz(sm, v2);
    const Fvector3d sp3 = to_xyz(sm, v3);
    const Fvector3d sp4 = to_xyz(sm, v4);
    // null operation
    int order1[3] = {0, 1, 2};
    ASSERT_TRUE(PermuteAxes(test_image, order1));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // permute x and y
    int order2[3] = {1, 0, 2};
    ASSERT_TRUE(PermuteAxes(test_image, order2));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[1], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[0], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get(10,  0, 6));
    EXPECT_EQ(8, test_image->get(10, 12, 6));
    EXPECT_EQ(9, test_image->get( 5,  6, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ(12, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 6, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ(12, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 6, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(PermuteAxes(test_image, order2));
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    // permute x and z
    int order3[3] = {2, 1, 0};
    ASSERT_TRUE(PermuteAxes(test_image, order3));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[2], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[0], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0,  0,  0));
    EXPECT_EQ(3, test_image->get( 6, 10,  0));
    EXPECT_EQ(8, test_image->get( 6, 10, 12));
    EXPECT_EQ(9, test_image->get( 3,  5,  6));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(6, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(6, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ(3, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ( 6, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(6, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(6, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ(3, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ( 6, v4.z);
    // revert
    ASSERT_TRUE(PermuteAxes(test_image, order3));
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    // permute y and z
    int order4[3] = {0, 2, 1};
    ASSERT_TRUE(PermuteAxes(test_image, order4));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0, 0,  0));
    EXPECT_EQ(3, test_image->get( 0, 6, 10));
    EXPECT_EQ(8, test_image->get(12, 6, 10));
    EXPECT_EQ(9, test_image->get( 6, 3,  5));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(6, v2.y); EXPECT_EQ(10, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(6, v3.y); EXPECT_EQ(10, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ(3, v4.y); EXPECT_EQ( 5, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(6, v2.y); EXPECT_EQ(10, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(6, v3.y); EXPECT_EQ(10, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ(3, v4.y); EXPECT_EQ( 5, v4.z);
    // revert
    ASSERT_TRUE(PermuteAxes(test_image, order4));
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    // permute all axes in two steps
    int order5[3] = {2, 1, 0};
    ASSERT_TRUE(PermuteAxes(test_image, order5));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(6, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(6, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ(3, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ( 6, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(6, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(6, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ(3, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ( 6, v4.z);
    int order6[3] = {1, 0, 2};
    ASSERT_TRUE(PermuteAxes(test_image, order6));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[1], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[0], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0, 0,  0));
    EXPECT_EQ(3, test_image->get(10, 6,  0));
    EXPECT_EQ(8, test_image->get(10, 6, 12));
    EXPECT_EQ(9, test_image->get( 5, 3,  6));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 6, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ( 6, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 3, v4.y); EXPECT_EQ( 6, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 6, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ( 6, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 3, v4.y); EXPECT_EQ( 6, v4.z);
    // revert
    ASSERT_TRUE(PermuteAxes(test_image, order6));
    ASSERT_TRUE(PermuteAxes(test_image, order5));
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    // permute all axes in one step
    int order7[3] = {1, 2, 0};
    ASSERT_TRUE(PermuteAxes(test_image, order7));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[1], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[0], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0, 0,  0));
    EXPECT_EQ(3, test_image->get(10, 6,  0));
    EXPECT_EQ(8, test_image->get(10, 6, 12));
    EXPECT_EQ(9, test_image->get( 5, 3,  6));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 6, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ( 6, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 3, v4.y); EXPECT_EQ( 6, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ( 0, v1.z);
    EXPECT_EQ(10, v2.x); EXPECT_EQ( 6, v2.y); EXPECT_EQ( 0, v2.z);
    EXPECT_EQ(10, v3.x); EXPECT_EQ( 6, v3.y); EXPECT_EQ(12, v3.z);
    EXPECT_EQ( 5, v4.x); EXPECT_EQ( 3, v4.y); EXPECT_EQ( 6, v4.z);
    // destroy test image
    delete test_image;
}

// ---------------------------------------------------------------------------
// Test flip of image axes.
TEST(Utilities, FlipAxes)
{
    // create test image
    Image* test_image = generate_test_image();
    // set some more complicated qform transformation
    test_image->hdr.quatern_b = 0;
    test_image->hdr.quatern_c = 0;
    test_image->hdr.quatern_d = 0.707106781;
    // coordinates of test points
    mat44 qm = qto_xyz(test_image);
    mat44 sm = sto_xyz(test_image);
    Ivector3d v1, v2, v3, v4;
    v1.x = v1.y = v1.z = 0;
    v2.x =  0; v2.y = 10; v2.z = 6;
    v3.x = 12; v3.y = 10; v3.z = 6;
    v4.x =  6; v4.y =  5; v4.z = 3;
    const Fvector3d qp1 = to_xyz(qm, v1);
    const Fvector3d qp2 = to_xyz(qm, v2);
    const Fvector3d qp3 = to_xyz(qm, v3);
    const Fvector3d qp4 = to_xyz(qm, v4);
    const Fvector3d sp1 = to_xyz(sm, v1);
    const Fvector3d sp2 = to_xyz(sm, v2);
    const Fvector3d sp3 = to_xyz(sm, v3);
    const Fvector3d sp4 = to_xyz(sm, v4);
    // null operation
    bool flip1[3] = {false, false, false};
    ASSERT_TRUE(FlipAxes(test_image, flip1));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // flip x axis
    bool flip2[3] = {true, false, false};
    ASSERT_TRUE(FlipAxes(test_image, flip2));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(5, test_image->get( 0,  0, 0));
    EXPECT_EQ(8, test_image->get( 0, 10, 6));
    EXPECT_EQ(3, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip2));
    // flip y axis
    bool flip3[3] = {false, true, false};
    ASSERT_TRUE(FlipAxes(test_image, flip3));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(4, test_image->get( 0,  0, 0));
    EXPECT_EQ(2, test_image->get( 0, 10, 6));
    EXPECT_EQ(6, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);;
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip3));
    // flip z axis
    bool flip4[3] = {false, false, true};
    ASSERT_TRUE(FlipAxes(test_image, flip4));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(2, test_image->get( 0,  0, 0));
    EXPECT_EQ(4, test_image->get( 0, 10, 6));
    EXPECT_EQ(7, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip4));
    // flip x and y axis
    bool flip5[3] = {true, true, false};
    ASSERT_TRUE(FlipAxes(test_image, flip5));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(7, test_image->get( 0,  0, 0));
    EXPECT_EQ(6, test_image->get( 0, 10, 6));
    EXPECT_EQ(2, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(0, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(6, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(6, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip5));
    // flip x and z axis
    bool flip6[3] = {true, false, true};
    ASSERT_TRUE(FlipAxes(test_image, flip6));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(6, test_image->get( 0,  0, 0));
    EXPECT_EQ(7, test_image->get( 0, 10, 6));
    EXPECT_EQ(4, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ( 0, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ(10, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ(10, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip6));
    // flip y and z axis
    bool flip7[3] = {false, true, true};
    ASSERT_TRUE(FlipAxes(test_image, flip7));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(3, test_image->get( 0,  0, 0));
    EXPECT_EQ(1, test_image->get( 0, 10, 6));
    EXPECT_EQ(5, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ( 0, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ( 0, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ(12, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // revert
    ASSERT_TRUE(FlipAxes(test_image, flip7));
    // flip x, y, and z axis
    bool flip8[3] = {true, true, true};
    ASSERT_TRUE(FlipAxes(test_image, flip8));
    EXPECT_EQ(test_image->hdr.dim[1], test_image->region.nx);
    EXPECT_EQ(test_image->hdr.dim[2], test_image->region.ny);
    EXPECT_EQ(test_image->hdr.dim[3], test_image->region.nz);
    EXPECT_EQ(dim[0], test_image->hdr.dim[1]);
    EXPECT_EQ(dim[1], test_image->hdr.dim[2]);
    EXPECT_EQ(dim[2], test_image->hdr.dim[3]);
    EXPECT_EQ(pixdim[0], test_image->hdr.pixdim[1]);
    EXPECT_EQ(pixdim[1], test_image->hdr.pixdim[2]);
    EXPECT_EQ(pixdim[2], test_image->hdr.pixdim[3]);
    EXPECT_EQ(8, test_image->get( 0,  0, 0));
    EXPECT_EQ(5, test_image->get( 0, 10, 6));
    EXPECT_EQ(1, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    qm = qto_ijk(test_image);
    v1 = to_ijk(qm, qp1);
    v2 = to_ijk(qm, qp2);
    v3 = to_ijk(qm, qp3);
    v4 = to_ijk(qm, qp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    sm = sto_ijk(test_image);
    v1 = to_ijk(sm, sp1);
    v2 = to_ijk(sm, sp2);
    v3 = to_ijk(sm, sp3);
    v4 = to_ijk(sm, sp4);
    EXPECT_EQ(12, v1.x); EXPECT_EQ(10, v1.y); EXPECT_EQ(6, v1.z);
    EXPECT_EQ(12, v2.x); EXPECT_EQ( 0, v2.y); EXPECT_EQ(0, v2.z);
    EXPECT_EQ( 0, v3.x); EXPECT_EQ( 0, v3.y); EXPECT_EQ(0, v3.z);
    EXPECT_EQ( 6, v4.x); EXPECT_EQ( 5, v4.y); EXPECT_EQ(3, v4.z);
    // destroy test image
    delete test_image;
}

// ===========================================================================
// test re-orientation of image data
// ===========================================================================

// ---------------------------------------------------------------------------
// RAI -> RAI (null operation)
TEST(Utilities, OrientImageRAI2RAI)
{
    Image* test_image = generate_test_image(NIFTI_R2L, NIFTI_A2P, NIFTI_I2S);
    EXPECT_TRUE(OrientImage(test_image, NIFTI_R2L, NIFTI_A2P, NIFTI_I2S));
    EXPECT_EQ(1, test_image->get( 0,  0, 0));
    EXPECT_EQ(3, test_image->get( 0, 10, 6));
    EXPECT_EQ(8, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    delete test_image;
}

// ---------------------------------------------------------------------------
// LPS -> RAI
TEST(Utilities, OrientImageLPS2RAI)
{
    Image* test_image = generate_test_image(NIFTI_L2R, NIFTI_P2A, NIFTI_S2I);
    EXPECT_TRUE(OrientImage(test_image, NIFTI_R2L, NIFTI_A2P, NIFTI_I2S));
    EXPECT_EQ(8, test_image->get( 0,  0, 0));
    EXPECT_EQ(5, test_image->get( 0, 10, 6));
    EXPECT_EQ(1, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    delete test_image;
}

// ---------------------------------------------------------------------------
// RAI -> RPI
TEST(Utilities, OrientImageRAI2RPI)
{
    Image* test_image = generate_test_image(NIFTI_R2L, NIFTI_A2P, NIFTI_I2S);
    EXPECT_TRUE(OrientImage(test_image, NIFTI_R2L, NIFTI_P2A, NIFTI_I2S));
    EXPECT_EQ(4, test_image->get( 0,  0, 0));
    EXPECT_EQ(2, test_image->get( 0, 10, 6));
    EXPECT_EQ(6, test_image->get(12, 10, 6));
    EXPECT_EQ(9, test_image->get( 6,  5, 3));
    delete test_image;
}

// ---------------------------------------------------------------------------
// LPS -> PSL
TEST(Utilities, OrientImageLPS2PSL)
{
    Image* test_image = generate_test_image(NIFTI_L2R, NIFTI_P2A, NIFTI_S2I);
    EXPECT_TRUE(OrientImage(test_image, NIFTI_P2A, NIFTI_S2I, NIFTI_L2R));
    EXPECT_EQ(1, test_image->get( 0, 0,  0));
    EXPECT_EQ(3, test_image->get(10, 6,  0));
    EXPECT_EQ(8, test_image->get(10, 6, 12));
    EXPECT_EQ(9, test_image->get( 5, 3,  6));
    delete test_image;
}
