#!/usr/bin/python2

################################################################################
# This script is used to ingest and upload image data to a remote or local site.
# It will take care of setting up the correct environmental variables specific
# to the local scanner data you will upload. 
#
# You must run the <Install.sh> script prior to running this script.
#  
# This script will traverse the <NATIVEDir>, create the XCEDE XML header, gather
# metadata from the XCEDE XML header and upload the images to the appropriate
# remote directory.  The script will further convert the <NATIVEDir> format to
# NIfTI-1 format and upload that to the remote site.
#
# If the <k-SpaceDir> is passed, the script will also upload the k-Space data 
#  If the user passes a KSPACE data directory to the script it will also upload
# those files.
#
#
# Authors:
#    David Keator (dbkeator@uci.edu)
#    Karen Pease (karen-pease@uiowa.edu)
#
################################################################################


import copy
import datetime
import glob
import imp
import logging
import logging.handlers
import math
import os
import re
import shutil
import stat
import string
import sys
import threading
import time
import tokenize
import traceback
import types
import urlparse
import xml
import xml.dom.ext.Printer
import xml.xpath
import xml.utils.boolean

from xml.dom.ext.reader import Sax2

from UploadClasses import _Constants
from UploadClasses import _StagingPaths
from UploadClasses import _RemotePaths
from UploadClasses import _FIPSPaths
from UploadClasses import _SiteInfo
from UploadClasses import _BXHInfo
from UploadClasses import _LoadedDataCoreElem
from UploadClasses import _LoadedDataElem
from UploadClasses import _UploadConfig
from UploadClasses import _DataPaths
from UploadClasses import _RunConfig
from UploadClasses import _Database
from UploadClasses import _RLS
from UploadClasses import _Schematron

from UploadFunctions import *

from SeriesDataDirectory import SeriesDataDirectory

try:
  import cx_Oracle
except:
  pass

if 'ScriptInstallDir' in os.environ:
  sys.path.append(os.environ['ScriptInstallDir'])

################################################################################
# Let the user know how to use the program.
################################################################################
def Usage():
  logging.warn("Human Usage: " + sys.argv[0] + "<ROOT_IMAGE_DIR> <XML Config File> <Target Directory> [<Local_REMOTE_HOME_BIRN>]")
  logging.warn("Nonhuman Usage: " + sys.argv[0] + " -nonhuman <Protocol Description File> <Native Directory> [<KSpaceDir>] ")
  print
  logging.warn("Optional arguments (general):")
  logging.warn("  -debug: print out more debugging messages")
  logging.warn("  -scanner <PATH>: Specify the scanner configuration file you wish to use.")
  logging.warn("  -remoteBaseURL: Specify the absolute URL to use as the base for uploads.")
  print
  logging.warn("Optional arguments (nonhuman only):")
  logging.warn("  -phantomType <geometric | stability>: Specify which kind of phantom to use.")
  logging.warn("  -qaAnalysis <Y | N>: Specify whether or not to conduct QA analysis.")
  logging.warn("  -project: Specify the project name, as it will appear in the remote path of the uploaded data.  Is overridden by -remoteBaseURL")
  logging.warn("  -phantomID: Specify the phantom ID -- should be SITEID_NUM (such as 0010_001)")
  print
  logging.warn("You must run the <Install.sh> script prior to running this script or you must set up")
  logging.warn("the required ENV variables yourself")
  print
  logging.warn("This shell script is the entry way for \"uploading\" or basically reorganizing your human imaging data")
  logging.warn("into a directory structure appropriate for BIRN tools")
  print
  logging.warn("The script will traverse the sub-directories of <ROOT_IMAGE_DIR> and copy the original data to a directory called \"Native\" ")
  logging.warn("in the <Target_DIR> directory you specified during the script installation.")
  print
  logging.warn("This script will also wrap your original data by creating an XML file called ImageWrapper.xml in the staging directory.")
  logging.warn("The script will then convert your original data into NIfTI 1.0 and Analyze 7.5 format, storing both in the staging directory")
  print
  logging.warn("You must specify an XML Configuration File (See PhaseI_Template.xml or PhaseII_Template.xml in the installation directory for an examples) which")
  logging.warn("will be used to get the project name/id, visit name/id, and study name/id and other information about your data.")
  print
  logging.warn("You must specify a <Target_DIR> this directory is where the actual data hierarchy will be copied")
  print
  logging.warn("If you specify another directory on the command line [<LOCAL_REMOTE_HOME_BIRN>], if the directory is different from the <Target_DIR>")
  logging.warn("symbolic links will be created in the <LOCAL_REMOTE_HOME_BIRN> to the data physically copied to the <Target_DIR>.  This can be used if")
  logging.warn("your filesystem is running out of space.  FIPS analysis software looks in the <Local_REMOTE_HOME_BIRN> directory for data.")
  print
  logging.warn("See Readme.htm file for help!!")

################################################################################
# Dynamically load a module ("hook")
################################################################################
def call_hook(hookdir, hookname, *args):
  logging.info("Looking for '" + hookname + "' hook in " + hookdir)
  hookfile = None
  hookpath = None
  hookdesc = None
  try:
    (hookfile, hookpath, hookdesc) = imp.find_module(hookname, [hookdir])
  except ImportError, e:
    pass
  if hookpath == None:
    logging.info(" -- didn't find it, skipping...")
  else:
    logging.info(" -- found it, loading...")
    try:
      hook = imp.load_module(hookname, hookfile, hookpath, hookdesc)
    except ImportError, e:
      logging.error("Error loading project-specific module!")
      raise
    hook.run(*args)

################################################################################
# Run the external program dicom2bxh
################################################################################
def dicom2bxh(flags, TMPListFile, XMLFile):
  ret = run_cmd('dicom2bxh ' + flags + ' --inputsfromfile=' + TMPListFile + ' "' + XMLFile + '"')
  try:
    os.stat(XMLFile)
    return ret
  except OSError:
    # No XMLFile exists; dicom command failed.
    logging.error("ERROR: dicom2bxh command failed.  Command was:")
    logging.error('  dicom2bxh ' + flags + ' --inputsfromfile=' + TMPListFile + ' "' + XMLFile + '"')
    logging.error('Output was:')
    logging.error(" " + str(ret[1]))
    logging.error('Return value was:')
    logging.error(" " + str(ret[0]))
    logging.error('TMPListFile was:')
    logging.error( "----------------")
    fp = open(TMPListFile, "r")
    logging.error(fp.read())
    fp.close()
    logging.error("----------------")
    logging.error("Exiting program.")
    sys.exit(1)
    
  
################################################################################
# Given a series of dimensions, determine the orientation.
################################################################################
def get_ras_flag(arg):
  split = map(lambda x: string.atof(x), arg.split())
  if math.fabs(split[0]) > math.fabs(split[1]) and math.fabs(split[0]) > math.fabs(split[2]):
    if split[0] < 0:
      return 'L'
    else:
      return 'R'
  if math.fabs(split[1]) > math.fabs(split[0]) and math.fabs(split[1]) > math.fabs(split[2]):
    if split[1] < 0:
      return 'P'
    else:
      return 'A'
  if math.fabs(split[2]) > math.fabs(split[0]) and math.fabs(split[2]) > math.fabs(split[1]):
    if split[2] < 0:
      return 'I'
    else:
      return 'S'


################################################################################
# Strip out important information from an XCEDE file
################################################################################
def ParseXMLHeader(path, BXHInfo, SiteInfo):
  
  logging.info('################################################################################')
  logging.info('# Parsing XML header for Metadata       ')
  logging.info('################################################################################')
  
  # Load the xml file
  reader = Sax2.Reader()
  doc = reader.fromUri(path)
  
  # Strip out variables using xpath statements.
  try:
    BXHInfo.set_SeriesDesc(XPathEval('/serieslevel/expProtocol/name',doc)[0].encode('ascii'))
  except:
    BXHInfo.set_SeriesDesc('')
  try:
    BXHInfo.set_StudyDate(XPathEval('/serieslevel/acqProtocol/acqParam[@name="scandate"]',doc)[0].encode('ascii'))
  except:
    BXHInfo.set_StudyDate('UnknownDate')
  try:
    BXHInfo.set_StudyTime(XPathEval('/serieslevel/acqProtocol/acqParam[@name="scantime"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_StudyTime('UnknownTime')
  logging.info("SiteInfo.get_ScannerManufacturer(): " + SiteInfo.get_ScannerManufacturer())
  BXHInfo.set_Manufacturer(SiteInfo.get_ScannerManufacturer())
  try:
    BXHInfo.set_Modality(XPathEval('/bxh/acquisitiondata/psdname', doc)[0].encode('ascii'))
  except IndexError:
    BXHInfo.set_Modality("")
  try:
    BXHInfo.set_Model(XPathEval('/serieslevel/scanner/model', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_Model('Unknown')
  try:
    BXHInfo.set_MagneticFieldStrength(XPathEval('/serieslevel/acqProtocol/acqParam[@name="magneticfield"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_MagneticFieldStrength('Unknown')
  try:
    BXHInfo.set_SeriesNumber(XPathEval('/serieslevel/acqProtocol/acqParam[@name="seriesnumber"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_SeriesNumber('Unknown')
  try:
    BXHInfo.set_ScanningSequence(XPathEval('/serieslevel/acqProtocol/acqParam[@name="scanningsequence"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_ScanningSequence('Unknown')
  try:
    BXHInfo.set_SequenceVariant(XPathEval('/serieslevel/acqProtocol/acqParam[@name="sequencevariant"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_SequenceVariant('Unknown')
  try:
    BXHInfo.set_SliceOrder(XPathEval('/serieslevel/acqProtocol/acqParam[@name="sliceorder"]', doc)[0].encode('ascii'))
  except:
    pass
  try:
    BXHInfo.set_SeriesTime(XPathEval('/serieslevel/acqProtocol/acqParam[@name="scantime"]', doc)[0].encode('ascii'))
  except:
    pass
  try:
    BXHInfo.set_AcqPlane(grep(path,".*Orientation.*").split()[4] )
  except AttributeError:
    BXHInfo.set_AcqPlane("None")
  BXHInfo.set_Matrix_X(XPathEval('/serieslevel/datarec/dimension[@type="x"]/size', doc)[0].encode('ascii'))
  BXHInfo.set_Matrix_Y(XPathEval('/serieslevel/datarec/dimension[@type="y"]/size', doc)[0].encode('ascii'))
  try:
    BXHInfo.set_Matrix_Z(XPathEval('/serieslevel/datarec/dimension[@type="z"]/size', doc)[0].encode('ascii'))
  except IndexError:
    # for Siemens data
    zsplit1 = int(XPathEval('/serieslevel/datarec/dimension[@type="z-split1"]/size', doc)[0].encode('ascii'))
    zsplit2 = int(XPathEval('/serieslevel/datarec/dimension[@type="z-split2"]/size', doc)[0].encode('ascii'))
    zsize = str(zsplit1 * zsplit2)
    osel = XPathEval('/serieslevel/datarec/dimension[@type="z-split2"]/@outputselect', doc)[0].encode('ascii')
    if osel != None:
      osellist = osel.split()
      zsize = str(int(osellist[-1]) + 1)
    BXHInfo.set_Matrix_Z(zsize)
    
  BXHInfo.set_PixelSpacing_X(XPathEval('/serieslevel/datarec/dimension[@type="x"]/spacing', doc)[0].encode('ascii'))
  BXHInfo.set_PixelSpacing_Y(XPathEval('/serieslevel/datarec/dimension[@type="y"]/spacing', doc)[0].encode('ascii'))
  try:
    BXHInfo.set_PixelSpacing_Z(XPathEval('/serieslevel/datarec/dimension[@type="z"]/spacing', doc)[0].encode('ascii'))
  except IndexError:
    BXHInfo.set_PixelSpacing_Z ( XPathEval('/serieslevel/datarec/dimension[@type="z-split2"]/spacing', doc)[0].encode('ascii'))
  try:
    BXHInfo.set_FlipAngle(XPathEval('/serieslevel/acqProtocol/acqParam[@name="flipangle"]', doc)[0].encode('ascii'))
  except:
    BXHInfo.set_FlipAngle('Unknown')
  BXHInfo.set_RASFlag_X(get_ras_flag(XPathEval('/serieslevel/datarec/dimension[@type="x"]/direction', doc)[0].encode('ascii')))
  BXHInfo.set_RASFlag_Y(get_ras_flag(XPathEval('/serieslevel/datarec/dimension[@type="y"]/direction', doc)[0].encode('ascii')))
  try:
    BXHInfo.set_RASFlag_Z(XPathEval('/serieslevel/datarec/dimension[@type="z"]/direction', doc)[0].encode('ascii'))
  except IndexError:
    BXHInfo.set_RASFlag_Z ( XPathEval('/serieslevel/datarec/dimension[@type="z-split2"]/direction', doc)[0].encode('ascii'))
  BXHInfo.set_RASFlag_Z ( get_ras_flag(BXHInfo.get_RASFlag_Z()))
      
  #Output the results
  
  logging.info("SeriesDesc="+BXHInfo.get_SeriesDesc())
  logging.info("StudyDate="+BXHInfo.get_StudyDate())
  logging.info("StudyTime="+BXHInfo.get_StudyTime())
  logging.info("StudyDate_StudyTime="+BXHInfo.get_StudyDate_StudyTime())
  logging.info("Modality="+BXHInfo.get_Modality())
  logging.info("Manufacturer="+BXHInfo.get_Manufacturer())
  logging.info("Model="+BXHInfo.get_Model())
  logging.info("MagneticFieldStrength="+BXHInfo.get_MagneticFieldStrength())
  logging.info("SeriesNumber="+BXHInfo.get_SeriesNumber())
  logging.info("ScanningSequence="+BXHInfo.get_ScanningSequence())
  logging.info("AcqPlane="+BXHInfo.get_AcqPlane())
  logging.info("Matrix_X="+BXHInfo.get_Matrix_X())
  logging.info("Matrix_Y="+BXHInfo.get_Matrix_Y())
  logging.info("Matrix_Z="+BXHInfo.get_Matrix_Z())
  logging.info("PixelSpacing_X="+BXHInfo.get_PixelSpacing_X())
  logging.info("PixelSpacing_Y="+BXHInfo.get_PixelSpacing_Y())
  logging.info("PixelSpacing_Z="+BXHInfo.get_PixelSpacing_Z())
  logging.info("RASFlag_X="+BXHInfo.get_RASFlag_X())
  logging.info("RASFlag_Y="+BXHInfo.get_RASFlag_Y())
  logging.info("RASFlag_Z="+BXHInfo.get_RASFlag_Z())
    
###############################################################################
# Helper function to replace '$' variables (params) in a string with the
# values given in the dictionary argument
###############################################################################
def replaceParams(s, d):
  components = s.split('$')
  tokenre = re.compile('^(([a-zA-Z_][a-zA-Z_0-9.-]*):)?([a-zA-Z_][a-zA-Z_0-9.-]*)')
  for i in range(len(components)):
    if i == 0:
      continue
    component = components[i]
    tokenmatch = tokenre.match(component)
    if tokenmatch == None:
      continue
    prefix = tokenmatch.group(2)
    token = tokenmatch.group(3)
    if token == None:
      continue
    tokenspan = tokenmatch.span(3)
    if d.has_key((prefix,token)):
      components[i] = d[(prefix,token)] + component[tokenspan[1]:]
  return ''.join(components)


###############################################################################
# Parse the whole upload XML file
###############################################################################
def ParseUploadXML(XMLDoc, UploadConfig):
  ParseXMLProject(XMLDoc, UploadConfig)
  ParseXMLVisit(XMLDoc, UploadConfig)
  ParseXMLStudy(XMLDoc, UploadConfig)
  ParseXMLBIRNID(XMLDoc, UploadConfig)
  ParseXMLSubjectGroup(XMLDoc, UploadConfig)
  ParseXMLRecruitAcqID(XMLDoc, UploadConfig)
  ParseXMLScannerManufacturer(XMLDoc, UploadConfig)
  ParseXMLSeries(XMLDoc, UploadConfig)

################################################################################
# Strip out important information from an XCEDE file
################################################################################
def ParseXMLSeries(doc, UploadConfig):

  logging.info('################################################################################')
  logging.info('# Parsing XML File for Series Names               ')
  logging.info('################################################################################')

  temp=XPathEval('/FIPS/series/nameStandard',doc)

  logging.info("Found " + str(len(temp)) + " Series in XML File")

  # initialize all arrays we plan to overwrite
  UploadConfig.set_SeriesList([])
  UploadConfig.set_SeriesDirLocal([])
  UploadConfig.set_SeriesType([])
  UploadConfig.set_SeriesSliceOrder([])
  UploadConfig.set_SeriesTime([])
  UploadConfig.set_SeriesSkipInitialVols([])
  UploadConfig.set_SeriesSourceData([])
  UploadConfig.set_SeriesSourceID([])
  UploadConfig.set_SeriesParadigm([])
  UploadConfig.set_SeriesAnalysisLevelLevel([])
  UploadConfig.set_SeriesAnalysisLevelName([])
  UploadConfig.set_Skip([])
  UploadConfig.set_SeriesXMLString([])

  # Iterate over the nameLocal and type entries in each nameStandard from
  # the xml file.  Build up a list of them.
  for serName in temp:
    serName2=serName.encode('ascii')
    UploadConfig.get_SeriesList().append(serName2)
    serNode = xml.xpath.Evaluate('/FIPS/series[nameStandard=\''+serName+'\']',doc)[0]

    UploadConfig.get_SeriesXMLString().append(xml2string(serNode, canonicalize=True))
    
    temp=XPathEval('nameLocal', serNode)[0].encode('ascii')
    UploadConfig.get_SeriesDirLocal().append(temp)
    
    temp=XPathEval('type', serNode)[0].encode('ascii')
    UploadConfig.get_SeriesType().append(temp)

    try:
      temp=XPathEval('sliceorder', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesSliceOrder().append(temp)
    except:
      UploadConfig.get_SeriesSliceOrder().append(None)

    try:
      temp=XPathEval('seriesTime', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesTime().append(temp)
    except:
      UploadConfig.get_SeriesTime().append(None)

    try:
      temp=XPathEval('skipInitialVols', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesSkipInitialVols().append(temp)
    except:
      UploadConfig.get_SeriesSkipInitialVols().append(None)

    try:
      temp=XPathEval('source', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesSourceData().append(temp)
    except:
      UploadConfig.get_SeriesSourceData().append(serName2)

    try:
      temp=XPathEval('ID', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesSourceID().append(temp.split(":"))
    except:
      UploadConfig.get_SeriesSourceID().append(None)

    try:
      temp=XPathEval('paradigm', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesParadigm().append(temp)
    except:
      UploadConfig.get_SeriesParadigm().append(None)

    try:
      temp=XPathEval('analysislevel/level', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesAnalysisLevelLevel().append(temp)
    except:
      UploadConfig.get_SeriesAnalysisLevelLevel().append(None)

    try:
      temp=XPathEval('analysislevel/name', serNode)[0].encode('ascii')
      UploadConfig.get_SeriesAnalysisLevelName().append(temp)
    except:
      UploadConfig.get_SeriesAnalysisLevelName().append(None)

    try:
      if xml.xpath.Evaluate('skip', serNode):
        logging.info("Setting Skip for " + serName)
        UploadConfig.get_Skip().append(True)
      else:
        UploadConfig.get_Skip().append(False)
    except:
      UploadConfig.get_Skip().append(False)


################################################################################
# Extract ProjectDir and ProjectID from the upload config file.
################################################################################
def ParseXMLProject(doc, UploadConfig):

  logging.info('################################################################################')
  logging.info("# Parsing XML File for Project Name and ID               ")
  logging.info('################################################################################')

  projectNode = xml.xpath.Evaluate('/FIPS/project', doc)[0]
  
  UploadConfig.set_ProjectXMLString(xml2string(projectNode, canonicalize=True))

  UploadConfig.m_ProjectName.set_value(XPathEval('name', projectNode)[0].encode('ascii'))
  UploadConfig.m_ProjectID.set_value(XPathEval('ID', projectNode)[0].encode('ascii'))
  
  logging.info("ProjectName="+UploadConfig.m_ProjectName.get_value())
  logging.info("ProjectID="+UploadConfig.m_ProjectID.get_value())


################################################################################
# Extract VisitName and VisitID from the upload config file.
################################################################################
def ParseXMLVisit(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for Visit Name and ID               ")
  logging.info('################################################################################')
  
  visitNode = xml.xpath.Evaluate('/FIPS/visit', doc)[0]
  
  UploadConfig.set_VisitXMLString(xml2string(visitNode, canonicalize=True))

  UploadConfig.m_VisitName.set_value(XPathEval('name', visitNode)[0].encode('ascii'))
  UploadConfig.m_VisitID.set_value(XPathEval('ID', visitNode)[0].encode('ascii'))
  try:
    UploadConfig.m_VisitDate.set_value(XPathEval('/FIPS/visit/visitDate',doc)[0].encode('ascii'))
  except:
    pass
  
  logging.info("VisitName="+UploadConfig.m_VisitName.get_value())
  logging.info("VisitID="+UploadConfig.m_VisitID.get_value())
  logging.info("VisitDate="+str(UploadConfig.m_VisitDate.get_value()))


################################################################################
# Extract StudyName and StudyID from the upload config file.
################################################################################
def ParseXMLStudy(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for Study Name and ID               ")
  logging.info('################################################################################')
  
  studyNode = xml.xpath.Evaluate('/FIPS/study', doc)[0]
  
  UploadConfig.set_StudyXMLString(xml2string(studyNode, canonicalize=True))

  UploadConfig.m_StudyName.set_value(XPathEval('name', studyNode)[0].encode('ascii'))
  UploadConfig.m_StudyID.set_value(XPathEval('ID', studyNode)[0].encode('ascii'))
  try:
    UploadConfig.m_StudyTime.set_value(XPathEval('studyTime', studyNode)[0].encode('ascii'))
  except:
    pass
  
  logging.info("StudyName="+UploadConfig.m_StudyName.get_value())
  logging.info("StudyID="+UploadConfig.m_StudyID.get_value())
  logging.info("StudyTime="+str(UploadConfig.m_StudyTime.get_value()))


################################################################################
# Extract SubjectID from the upload config file.
################################################################################
def ParseXMLBIRNID(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for BIRNID               ")
  logging.info('################################################################################')
  
  UploadConfig.m_SubjectID.set_value(XPathEval('/FIPS/BIRNID',doc)[0].encode('ascii'))
  
  logging.info("BIRNID = "+UploadConfig.m_SubjectID.get_value())

################################################################################
# Extract SubjectGroup from the upload config file.
################################################################################
def ParseXMLSubjectGroup(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for SubjectGroup               ")
  logging.info('################################################################################')
  
  try:
    UploadConfig.m_SubjectGroup.set_value(XPathEval('/FIPS/subjectGroup',doc)[0].encode('ascii'))
    logging.info("Subject Group = "+UploadConfig.m_SubjectGroup.get_value())
  except:
    pass


################################################################################
# Extract RecruitID and SiteID from the upload config file.
################################################################################
def ParseXMLRecruitAcqID(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for Recruiting and Acquisition IDs      ")
  logging.info('################################################################################')
  
  try:
    UploadConfig.set_RecruitSiteID(XPathEval('/FIPS/recruitID',doc)[0].encode('ascii'))
  except IndexError:
    pass
  try:
    UploadConfig.set_SiteID(XPathEval('/FIPS/acquisitionSiteID',doc)[0].encode('ascii'))
  except IndexError:
    pass
  
  try:
    logging.info("RecruitID = "+UploadConfig.get_RecruitSiteID())
  except TypeError:
    pass
  try:
    logging.info("SiteID = "+UploadConfig.get_SiteID())
  except TypeError:
    pass

  
################################################################################
# Extract Scanner Manufacturer from the upload config file.
################################################################################
def ParseXMLScannerManufacturer(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for Scanner Manufacturer      ")
  logging.info('################################################################################')
  
  try:
    UploadConfig.set_ScannerManufacturer(XPathEval('/FIPS/scannerManufacturer',doc)[0].encode('ascii'))
  except IndexError:
    pass
  
  try:
    logging.info("ScannerManufacturer = "+UploadConfig.get_ScannerManufacturer())
  except TypeError:
    pass

  
################################################################################
# Extract the FileSystem (Remote, Local, Local-Remote, Local-In-Place) from the upload
# config file.
################################################################################
def ParseXMLFileSystem(doc, UploadConfig):
  logging.info('################################################################################')
  logging.info("# Parsing XML File for Type of Upload (Local/Remote)               ")
  logging.info('################################################################################')
  
  result = XPathEval('/FIPS/FileSystem',doc)[0].encode('ascii')
  if result != "Remote" and result != "Local" and result != "Local-Remote" and result != "Local-In-Place":
    logging.error("ERROR: <FileSystem> field must be one of: Local, Remote, Local-Remote, Local-In-Place\n")
    sys.exit(1)
  UploadConfig.set_FileSystem(result)
  logging.info("FileSystem "+UploadConfig.get_FileSystem())


################################################################################
# Get rid of non-relevant series in fips-process.xml
################################################################################
def IsolateSeries(doc, name, outpath):

#  for result in xml.xpath.Evaluate('/FIPS/series/nameStandard',doc):
#    if result.firstChild.nodeValue == name:
#      result.parentNode.removeChild(result)
  newdoc = doc.cloneNode(True)
  for result in xml.xpath.Evaluate('/FIPS/series/nameStandard',newdoc):
    if result.firstChild != None and result.firstChild.nodeValue != name:
      result.parentNode.parentNode.removeChild(result.parentNode)
  
  SaveXML(outpath, newdoc)


################################################################################
# Change the value of a node entry in an xml file
################################################################################
def ChangeNode(path, outpath, xpath, value):

  reader = Sax2.Reader()
  doc = reader.fromUri(path)

  XPathReplace(xpath, value, doc)

#  for i in xml.xpath.Evaluate(xpath, doc):
#    i.firstChild.nodeValue = value
  
  SaveXML(outpath, doc)
        

################################################################################
# Add a new series of nodes to an xml file.  "args" is a list of tuples.
# Tuple entries:
#   xpath - XPath to prospective parent of new node
#   name  - tag name of new node
#   value - string content of new node
#   mode  - If 0, append node without checking for any other elements.
#           If 1, the new node will not be appended if there is already
#             a matching node.
#           If 2, remove any existing child node(s) with the same tag
#             name before appending the new node.
#   attrname (optional) - attach a "name" attribute with this value
#   attrtype (optional) - attach a "type" attribute with this value
################################################################################
def AppendXML(path, outpath, args):
  reader = Sax2.Reader()
  doc = reader.fromUri(path)
  
  for arg in args:	#Iterate over each node that we want to add
    xpath=arg[0]
    element=arg[1]
    value=arg[2]
    mode=arg[3]
    name=None
    typ=None
    if len(arg)==6:
      name=arg[4]
      typ=arg[5]

    modestr = ('APPEND', 'SKIP_IF_EXISTS', 'REPLACE_IF_EXISTS')[mode]
    typestr = 'None'
    if typ != None:
      typestr = "'%s'" % typ
    logging.info("AppendXML: path='%s' outpath='%s' xpath='%s' element='%s' value='%s' mode='%s' name=%s typ=%s" % (path, outpath, xpath, element, value, modestr, name, typestr))
  
    for node in xml.xpath.Evaluate(xpath, doc):
      if mode != 0:
        elemxpath = element
        if name != None and typ != None:
          elemxpath = elemxpath + ('[@name="%s"]' % name)
        logging.debug(" elemxpath=" + elemxpath)
        childnodes = xml.xpath.Evaluate(elemxpath, node)
        logging.debug("  matched %d nodes" % len(childnodes))
        if mode == 1 and len(childnodes) > 0:
          # skip this, child element with this name already exists
          logging.debug(" child element " + elemxpath + " already exists, skipping...")
          continue
        if mode == 2:
          for childnode in childnodes:
            logging.debug(" removing child element " + elemxpath)
            node.removeChild(childnode)
      new_node=doc.createElement(element)
      contents_node=doc.createTextNode(value)
      new_node.appendChild(contents_node)
      if len(arg)==6:
        new_node.setAttribute("name", name)
        new_node.setAttribute("type", typ)
      if len(arg)==6:
        logging.debug(" appending new element %s/%s[@name='%s' @type='%s']='%s'" % (xpath, element, name, typ, value))
      else:
        logging.debug(" appending new element %s/%s='%s'" % (xpath, element, value))
      node=node.appendChild(new_node)

  SaveXML(outpath, doc)

     
################################################################################
# Clean up and exit
################################################################################
def Clean(UploadConfig, arg):
  if UploadConfig.get_FileSystem() == "Remote":
    shutil.rmtree(UploadConfig.get_CurStudyDir())

################################################################################
#  Delete records from a data table, $1, dataURI matches the pattern, $2   
################################################################################
def dbdelete(table, uri, Database):
  
  query="delete from "+table+" where datauri like '"+uri+"'"
  Database.execute(query)
#  logging.info("dbdelete "+table+" %d" % dataCount1)


################################################################################
# Import the Upload configuration file's data.
################################################################################
def ImportConfig(StagingPaths, SiteInfo):
  config = SiteInfo.get_ScannerConfigFile()
  logging.info("Importing config file: " + config)
  f=open(config, "r")
  lines=f.readlines()
  f.close()
  results={}
  
  # Read in all of the lines and break into key-value pairs. Store those
  # in the dictionary "results".
  for line in lines:
    if re.match("^[ \t]*\#.*$", line) != None:
      continue
    if line=="\n":
      continue
    split=line[:-1].split("=")
    key=split[0]
    val=split[1]
    if val == "":
      continue
    if val[0]=='"' and val[-1]=='"':
      val=val[1:-1]
    else:
      val=string.atoi(val)
    results[key]=val

  
  # Use try-catch blocks, because not everything will be in there.
  try:
    SiteInfo.set_ScriptInstallDir(results["ScriptInstallDir"])
  except: 
    SiteInfo.set_ScriptInstallDir(".")
  try:
    SiteInfo.set_NativeFormat(results["NativeFormat"])
  except: 
    pass
  try:
    SiteInfo.set_GeometricNativeFormat(results["GeometricNativeFormat"])
  except: 
    pass
  try:
    SiteInfo.set_StabilityNativeFormat(results["StabilityNativeFormat"])
  except: 
    pass
  try:
    SiteInfo.set_StructFormat(results["StructFormat"])
  except: 
    pass
  try:
    SiteInfo.set_StructAnonFormat(results["StructAnonFormat"])
  except: 
    pass
  try:
    SiteInfo.set_BehavioralFormat(results["BehavioralFormat"])
  except: 
    pass
  try:
    SiteInfo.set_StructAnonFormat(results["SegmentFormat"])
  except: 
    pass
  try:
    SiteInfo.set_Institution(results["Institution"])
  except: 
    pass
  try:
    SiteInfo.set_InstitutionID(results["InstitutionID"])
  except:
    pass
  try:
    SiteInfo.set_AcquisitionSite(results["AcquisitionSite"])
  except:
    SiteInfo.set_AcquisitionSite(SiteInfo.get_Institution())
  try:
    SiteInfo.set_AcquisitionSiteID(results["AcquisitionSiteID"])
  except:
    SiteInfo.set_AcquisitionSiteID(SiteInfo.get_InstitutionID())
  try:
    SiteInfo.set_ScannerManufacturer(results["ScannerManufacturer"])
  except: 
    pass
  try:
    SiteInfo.set_ScannerID(results["ScannerID"])
  except: 
    pass
  try:
    SiteInfo.set_BehavioralEquipment(results["BehavioralEquipment"])
  except: 
    pass
  try:
    SiteInfo.set_BehavioralEquipmentID(results["BehavioralEquipmentID"])
  except: 
    pass
  try:
    StagingPaths.set_TempDIR(results["TempDIR"])
  except: 
    pass
  try:
    SiteInfo.set_dbHost(results["DBHost"])
  except: 
    pass
  try:
    SiteInfo.set_dbPort(results["DBPort"])
  except: 
    pass
  try:
    SiteInfo.set_dbInstance(results["DBInstance"])
  except: 
    pass
  try:
    SiteInfo.set_dbUser(results["DBUser"])
  except: 
    pass
  try:
    SiteInfo.set_dbPass(results["DBPass"])
  except: 
    pass
#IBO
  try:
    SiteInfo.set_dbType(results["DBType"])
  except: 
    pass
  try:
    SiteInfo.set_User(results["User"])
  except: 
    pass
  try:
    SiteInfo.set_UseCXOracle(results["UseCXOracle"])
  except: 
    pass
  try:
    SiteInfo.set_MyProxyHost(results["MyProxyHost"])
  except: 
    pass
  try:
    SiteInfo.set_MyProxyPort(results["MyProxyPort"])
  except: 
    pass
  try:
    SiteInfo.set_GridFTPHost(results["GridFTPHost"])
  except: 
    pass
  try:
    SiteInfo.set_GridFTPPort(results["GridFTPPort"])
  except: 
    pass


################################################################################
# Import a FIPS config file file if present (.fipsrc)
################################################################################
def ImportFIPSConfig(config, FIPSPaths, RemotePaths):

  f=open(config, "r")
  lines=f.readlines()
  f.close()
  results={}
  for line in lines:
    line=line[:-1]
    if re.match("^[ \t]*\#.*$", line) != None:
      continue
    split=line[:-1].split(" ")
    key=split[1]
    val=split[2]
    results[key]=val
  
  try:
    RemotePaths.set_LocalRemoteHome(results["LOCAL_REMOTE_HOME_BIRN"])
  except: 
    pass
  try:
    FIPSPaths.set_FIPSFLACDir(results["FIPS_FLAC_DIR"])
  except: 
    pass
  try:
    FIPSPaths.set_MyFIPSDir(results["MY_FIPS_DIR"])
  except: 
    pass
  try:
    FIPSPaths.set_FIPSDefaultDatabase(results["FIPS_DEFAULT_DATABASE"])
  except: 
    pass
  try:
    FIPSPaths.set_FIPSDefaultProjectName(results["FIPS_DEFAULT_PROJECT_NAME"])
  except: 
    pass
  try:
    FIPSPaths.set_FIPSDefaultProjectID(results["FIPS_DEFAULT_PROJECT_ID"])
  except: 
    pass


################################################################################
# Auto-detect image format in a directory
###############################################################################
def AutoDetectImageDirectory(LocalPath, initial_ignore_types=[]):
  sdd = SeriesDataDirectory(LocalPath, ignore_types=initial_ignore_types)
  if len(sdd.types) == 0:
    sdd = SeriesDataDirectory(LocalPath)
  if len(sdd.types) == 0:
    logging.info("Can't auto-detect type of image data in '" + LocalPath + "'.  Choose from the following:")
    while True:
      for i in range(len(_Constants.m_DataSpecifiers)):
        logging.info("%-6u   %s" % (i, _Constants.m_DataSpecifiers[i]))
      logging.info("")
      logging.info("Please choose by number (0 - %u):" % (len(_Constants.m_DataSpecifiers) - 1))
      myImageFormats = [int(sys.stdin.readline()[:-1])]
      if myImageFormats[0] < 0 or myImageFormat[0] >= len(_Constants.m_DataSpecifiers):
        logging.info("Number %u out of range!" % myImageFormat)
        continue
      else:
        break
  else:
    Constants = _Constants()
    myImageFormats = []
    for format in [Constants.m_DATASPEC_DICOM,Constants.m_DATASPEC_AFNI,Constants.m_DATASPEC_PFILE,Constants.m_DATASPEC_SIGNA5,Constants.m_DATASPEC_IOWASIGNA5,Constants.m_DATASPEC_FFILE,Constants.m_DATASPEC_UCIRAW,Constants.m_DATASPEC_MINC,Constants.m_DATASPEC_NIFTI,Constants.m_DATASPEC_ANALYZE,Constants.m_DATASPEC_XCEDE,Constants.m_DATASPEC_BXH]:
      if format in sdd.types:
        myImageFormats.append(format)
    logging.info("Detected image formats " + ', '.join(map(lambda x: _Constants.m_DataSpecifiers[x], myImageFormats)) + " for " + LocalPath)
  return [myImageFormats,sdd]

################################################################################
# Merge acquisition parameters from multiple XCEDE files
################################################################################
def MergeXCEDEAcqParams(origFile, mergeFile, outFile):
  logging.info("Attempting to merge acqParam elements from " + mergeFile + " into " + origFile)
  reader = Sax2.Reader()
  origDoc = reader.fromUri(origFile)
  mergeDoc = reader.fromUri(mergeFile)
  # if gaps are zero in original document and not zero in merge document,
  # replace them.
  replacegaps = False
  origGaps = {}
  for gapnode in xml.xpath.Evaluate('/serieslevel/datarec/dimension/gap', origDoc):
    try:
      dimtype = xml.xpath.Evaluate('string(../@type)', gapnode)
      gapval = xml.xpath.Evaluate('string(.)', gapnode)
      if float(gapval) == 0 or gapval == '':
        replacegaps = True
      origGaps[dimtype] = (gapnode, gapval)
    except:
      pass
  if replacegaps:
    mergeGaps = {}
    for gapnode in xml.xpath.Evaluate('/serieslevel/datarec/dimension/gap', mergeDoc):
      try:
        dimtype = xml.xpath.Evaluate('string(../@type)', gapnode)
        gapval = xml.xpath.Evaluate('string(.)', gapnode)
        mergeGaps[dimtype] = (gapnode, gapval)
      except:
        pass
    for dimtype in origGaps:
      (origGapNode, origGapVal) = origGaps[dimtype]
      if float(origGapVal) != 0 and origGapVal != '':
        continue
      if dimtype not in mergeGaps:
        continue
      (mergeGapNode, mergeGapVal) = mergeGaps[dimtype]
      if float(mergeGapVal) == 0 or mergeGapVal == '':
        continue
      logging.info(" replacing empty gap element in dimension %s" % dimtype)
      newGapNode = origDoc.importNode(mergeGapNode, True)
      origGapNode.parentNode.insertBefore(newGapNode, origGapNode)
      origGapNode.parentNode.removeChild(origGapNode)
  # do acqParam elements
  origAcqParams = xml.xpath.Evaluate('/serieslevel/acqProtocol/acqParam', origDoc)
  origDatarecs = xml.xpath.Evaluate('/serieslevel/datarec', origDoc)
  mergeAcqParams = xml.xpath.Evaluate('/serieslevel/acqProtocol/acqParam', mergeDoc)
  if len(origAcqParams) == 0 and len(mergeAcqParams) > 0:
    logging.info(" merging in complete acqProtocol element")
    # remove any existing acqProtocol elements (which should be empty)
    origAcqProtocols = xml.xpath.Evaluate('/serieslevel/acqProtocol', origDoc)
    for origAcqProtocol in origAcqProtocols:
      origAcqProtocol.parentNode.removeChild(origAcqProtocol)
    newAcqProtocol = origDoc.importNode(mergeAcqParams[0].parentNode, True)
    origDatarecs[0].parentNode.insertBefore(newAcqProtocol, origDatarecs[0])
  elif len(origAcqParams) != 0:
    origMap = {}
    for paramNode in origAcqParams:
      prevTextNode = paramNode.previousSibling
      if prevTextNode != None and prevTextNode.nodeType != paramNode.TEXT_NODE:
        prevTextNode = None
      name = paramNode.getAttribute('name')
      if name:
        origMap[name] = (paramNode, prevTextNode)
    mergeMap = {}
    for paramNode in mergeAcqParams:
      prevTextNode = paramNode.previousSibling
      if prevTextNode != None and prevTextNode.nodeType != paramNode.TEXT_NODE:
        prevTextNode = None
      name = paramNode.getAttribute('name')
      if name:
        mergeMap[name] = (paramNode, prevTextNode)
    origAcqProtocol = origAcqParams[-1].parentNode
    for name in mergeMap.keys():
      if name not in origMap:
        (mergeNode, mergeTextNode) = mergeMap[name]
        logging.info(" merging in param '" + name + "'")
        if mergeTextNode != None:
          newText = origDoc.importNode(mergeTextNode, True)
          origAcqProtocol.appendChild(newText)
        newParam = origDoc.importNode(mergeNode, True)
        origAcqProtocol.appendChild(newParam)
  provenances = xml.xpath.Evaluate('/serieslevel/provenance', origDoc)
  XCEDENS = 'http://nbirn.net/Resources/Users/Applications/xcede/'
  newStep = origDoc.createElementNS(XCEDENS, 'processStep')
  programName = origDoc.createElementNS(XCEDENS, 'programName')
  programName.appendChild(origDoc.createTextNode('Upload.py (function MergeXCEDEAcqParams)'))
  newStep.appendChild(programName)
  programArgument = origDoc.createElementNS(XCEDENS, 'programArgument')
  programArgument.appendChild(origDoc.createTextNode('origFile="' + origFile + '", mergeFile="' + mergeFile + '", outFile="' + outFile + '"'))
  newStep.appendChild(programArgument)
  timeStamp = origDoc.createElementNS(XCEDENS, 'timeStamp')
  timeStamp.appendChild(origDoc.createTextNode(time.strftime("%Y-%m-%d %H:%M:%S")))
  newStep.appendChild(timeStamp)
  provenances[0].appendChild(newStep)
  SaveXML(outFile, origDoc)
  return 1
  

################################################################################
# Create NIFTI files.
################################################################################
def CreateBXHFiles(myImageFormats, LocalPath, TempPath, BXHInfo, SiteInfo, NIFTIDir, xcede, example_files=None, seriestype=None):
  Constants = _Constants()
  
  flags = ""
  if xcede:
    flags = "--xcede"

  for formatind in range(len(myImageFormats)):
    myImageFormat = myImageFormats[formatind]
    myImageExample = None
    if example_files != None and len(example_files) > 0:
      myImageExample = example_files[formatind]
    if myImageFormat == Constants.m_DATASPEC_DICOM:
      logging.info('################################################################################')
      logging.info("# Creating DICOM XML header: "+LocalPath)
      logging.info('################################################################################')
      if which("dicom2bxh") == None:
        logging.info("Binary <dicom2bxh> Not Found! Check Your Path Environmental Variable")
        sys.exit(-1)
        
      TMPListFile = os.path.join(TempPath, ("_tmplist.%d" % time.mktime(time.localtime())))
      try:
        os.unlink(TMPListFile)
      except OSError:
        pass
      listfh = open(TMPListFile,"w")
  
      DICOMFileFound = False
      DICOMFile = ""
      logging.info("Loading: "+TempPath)
      sortedfilelist = os.listdir(TempPath)
      sortedfilelist.sort()
      for file in sortedfilelist:
        file2 = os.path.join(TempPath, file)
        try:
          f = open(file2,"r")
        except IOError:
          continue
        f.seek(128)
        magic = f.read(4)
        f.close()
        if magic == "DICM" or len(filter(lambda x: file[-len(x):] == x, [".dcm", ".DCM", ".IMA", ".ima"])) > 0:
          DICOMFile = file2
          DICOMFileFound = True
          listfh.write(file2 + "\n")
      listfh.close()
  
      if not DICOMFileFound:
        logging.error("No DICOM files found in "+TempPath)
        os.unlink(TMPListFile)
        sys.exit(-1)
  
      dicom2bxh(flags, TMPListFile, os.path.join(TempPath, 'ImageWrapper-orig.xml'))
      os.unlink(TMPListFile)
      
    elif myImageFormat == Constants.m_DATASPEC_PFILE:
      logging.info('################################################################################')
      logging.info("# Creating GE PFile XML header: "+LocalPath)
      logging.info('################################################################################')
      if which("pfile2bxh") == None:
        logging.error("Binary <pfile2bxh> Not Found! Check Your Path Environmental Variable")
        sys.exit(-1)
    
      MAGFileFound = False
      PFileFound = False
      PFile = None
      for file in os.listdir(TempPath):
        if MAGFileFound and PFileFound:
          break
        if file.find(".img") != -1 or file.find(".IMG") != -1:
          MAGFile = file  
          MAGFileFound = True
        elif file[0:3] == 'pfh' or file.find(".pfh") != -1 or file.find(".PFH") != -1 or re.match("P.*\.7", file) != None:
          PFile = os.path.join(TempPath, file)
          PFileFound = True
  
      if MAGFileFound and PFileFound:
        run_cmd("pfile2bxh " + flags + " " + PFile + " *." + file + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
      elif PFileFound:
        run_cmd("pfile2bxh " + flags + " " + PFile + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
      else:
        logging.error("No MAG File / P-file set found!")
        sys.exit(-1)
        
    elif myImageFormat == Constants.m_DATASPEC_FFILE:
      
      logging.info('################################################################################')
      logging.info("# Creating Stanford F-File XML header: "+LocalPath)
      logging.info('################################################################################')
      if which("ffile2bxh") == None:
        MAGFileFound = False
        FFileFound = False
        FFile=None
        TMPListFile=os.path.join(TempPath,("_tmplist.%d" % time.mktime(time.localtime())))
        shutil.rmtree(TMPListFile)
        fp.open(TMPListFile, "a").close()
        for file in filter(lambda x: x.find("SERHDR")!=-1, os.listdir(TempPath)):
          if re.match('^F.*\.7$', file) != None:
            FFile = os.path.join(TempPath, file)
            FFileFound = True
          if re.match('\.img$', file) != None:
            MAGFileFound = True
            f = open(TMPListFile, "a")
            f.write(file + "\n")
            f.close()
    
        if MAGFileFound and FFileFound:
          f = open(TMPListFile)
          data = f.read()
          f.close()
          run_cmd("ffile2bxh " + flags + " " + FFile + " " + data+" " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
          os.unlink(TMPListFile)
          
        else:
          logging.error("No .img File / F-file set found!")
          return 1
      
      else:
        logging.error("Binary <ffile2bxh> Not Found! Check Your Path Environmental Variable")
        return 1
    
    elif myImageFormat == Constants.m_DATASPEC_UCIRAW:
    
      logging.info('################################################################################')
      logging.info("# Searching for UCI XML header: "+LocalPath)
      logging.info('################################################################################')
    
      BXHFileFound = False
      BXHFile = ""
      for file in os.listdir(TempPath):
        if BXHFileFound:
          break
        if file.find("xml")!=-1 or file.find("XML")!=-1:
          BXHFile=file  
          BXHFileFound = True
          logging.info("****** Generating " + os.path.join(TempPath, "ImageWrapper-orig.xml") + " *******")
          shutil.copy(BXHFile, os.path.join(TempPath, "ImageWrapper-orig.xml"))
    
      if not BXHFileFound:
        logging.error("BXH XML Header File Not Found! Please Create XML Header for RAW Data!")
        return 1
    elif myImageFormat == Constants.m_DATASPEC_IOWASIGNA5:
      
      logging.info('################################################################################')
      logging.info("# Creating Iowa GE5.x XML header: "+LocalPath)
      logging.info('################################################################################')
      logging.info("Please enter the study number for this study:")
      BXHInfo.set_StudyNum(sys.stdin.readline()[:-1])
      
      IowaConcatDir()
      
      if which("iowa-signafive2bxh") != None:
        run_cmd("iowa-signafive2bxh " + flags + " " + LocalPath + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
        
      else:
        logging.error("Binary <iowa-signafive2bxh> Not Found! Check Your Path Environmental Variable")
        return 1
      
    elif myImageFormat == Constants.m_DATASPEC_ANALYZE:
      logging.info('################################################################################')
      logging.info("# Cleaning up Analyze 7.5 Format Data: "+LocalPath)
      logging.info('################################################################################')
    
      BXHFileFound = False
      BXHFile=""
      for file in os.listdir(TempPath):
        if BXHFileFound:
          break
        if file.find("xml")!=-1 or file.find("XML")!=-1:
          BXHFile=file  
          BXHFileFound = True
          logging.info("****** Generating " + os.path.join(TempPath, "ImageWrapper-orig.xml") + " *******")
          shutil.copy(BXHFile, os.path.join(TempPath, "ImageWrapper-orig.xml"))
      if not BXHFileFound:
        logging.info('################################################################################')
        logging.info("Making XML Wrapper for Analyze Files")
        logging.info('################################################################################')
        run_cmd("analyze2bxh " + flags + " " + os.path.join(TempPath, "*.img") + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
    
    elif myImageFormat == Constants.m_DATASPEC_NIFTI:
      logging.info('################################################################################')
      logging.info("# Creating NIFTI XML header: "+LocalPath)
      logging.info('################################################################################')
    
      NIFTIFileFound = False
      NIFTIFile = None
      for file in os.listdir(TempPath):
        if NIFTIFileFound:
          break
        if re.match('^.*\.nii(.gz)?$', file) != None:
          NIFTIFile = os.path.join(TempPath, file)
          NIFTIFileFound = True
  
      if NIFTIFileFound:
        run_cmd("analyze2bxh " + flags + " " + NIFTIFile + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
    
    elif myImageFormat == Constants.m_DATASPEC_MINC:
      logging.info('################################################################################')
      logging.info("# Cleaning up MINC Format Data: "+LocalPath)
      logging.info('################################################################################')
    
      MINCFileFound = False
      for file in os.listdir(TempPath):
        if MINCFileFound:
          break
        if re.match('^.*\.mnc$', file) != None:
          MINCFileFound = True
  
      if MINCFileFound:
        logging.info('################################################################################')
        logging.info("Making XML Wrapper for MINC Files")
        logging.info('################################################################################')
        run_cmd("minc2bxh " + flags + " " + os.path.join(TempPath, "*.mnc") + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
        
      else:
        logging.error('################################################################################')
        logging.error('# ERROR! NO MINC FILE FOUND!')
        logging.error('################################################################################')
    
    elif myImageFormat == Constants.m_DATASPEC_SIGNA5:
    
      if which("signafive2bxh") != None:
        logging.info('################################################################################')
        logging.info("# Creating GE Signa 5.x XML Wrapper: "+LocalPath)
        logging.info('################################################################################')
      
        SIGNAFileFound = False
        TMPListFile=os.path.join(TempPath,("_tmplist.%d" % time.mktime(time.localtime())))
        os.unlink(TMPListFile)
        open(TMPListFile, "a").close()
        for file in filter(lambda x: x.find("SERHDR")!=-1, os.listdir(TempPath)):
          if len(filter(lambda x: x.find("IMGF")!=-1, open(file, "r").readlines(4))) > 0:
            SIGNAFileFound = True
            f=open(TMPListFile, "a")
            f.write(os.path.join(TempPath, file) + "\n")
            f.close()
    
        run_cmd('signafive2bxh ' + flags + ' --inputsfromfile='+TMPListFile+' "'+os.path.join(TempPath,'ImageWrapper-orig.xml')+'"')
        os.unlink(TMPListFile)
        
        if not SIGNAFileFound:
          logging.error("No GE SIGNA 5.x files found in <`pwd`>")
      else:
        logging.error("Binary <signafive2bxh> Not Found! Check Your Path Environmental Variable")
        return 1
        
    elif myImageFormat == Constants.m_DATASPEC_BXH:
      BXHFileFound = False
      BXHFile = None
      for file in os.listdir(TempPath):
        if re.match('^.*\.bxh$', file) != None:
          BXHFileFound = True
          BXHFile = os.path.join(LocalPath, file)
          break
      if not BXHFileFound:
        logging.error('###############################################################################')
        logging.error('# ERROR! NO BXH FILE FOUND!')
        logging.error('###############################################################################')
        sys.exit(-1)
  
      if not xcede:
        shutil.copy(BXHFile, os.path.join(TempPath,"ImageWrapper-orig.xml"))
      else:
        if which("bxh2xcede") == None:
          logging.error("Binary <bxh2xcede> Not Found! Check Your Path Environmental Variable")
          sys.exit(-1)
  
        logging.info('################################################################################')
        logging.info("Making XML Wrapper for BXH Files")
        logging.info('################################################################################')
        run_cmd("bxh2xcede " + " " + BXHFile + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
  
    elif myImageFormat == Constants.m_DATASPEC_XCEDE:
      XCEDEFileFound = False
      XCEDEFile = myImageExample
      # check LocalPath because "linked" ImageWrapper.xml may have been removed
      for file in os.listdir(LocalPath):
        if re.match('^.*\.xml$', file) != None:
          XCEDEFileFound = True
          XCEDEFile = os.path.join(LocalPath, file)
          break
      if not XCEDEFileFound:
        logging.error('###############################################################################')
        logging.error('# ERROR! NO XCEDE FILE FOUND!')
        logging.error('###############################################################################')
        sys.exit(-1)
  
      if xcede:
        shutil.copy(XCEDEFile, os.path.join(TempPath, "ImageWrapper-orig.xml"))
      else:
        if which("xcede2bxh") == None:
          logging.error("Binary <xcede2bxh> Not Found! Check Your Path Environmental Variable")
          sys.exit(-1)
  
        logging.info('################################################################################')
        logging.info("Making XML Wrapper for XCEDE Files")
        logging.info('################################################################################')
        run_cmd("xcede2bxh " + " " + XCEDEFile + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
    elif myImageFormat == Constants.m_DATASPEC_AFNI:
      logging.info('################################################################################')
      logging.info("# Cleaning up AFNI Format Data: "+LocalPath)
      logging.info('################################################################################')
    
      AFNIFileFound = False
      for file in os.listdir(TempPath):
        if AFNIFileFound:
          break
        if re.match('^.*\.HEAD$', file) != None:
          AFNIFileFound = True
  
      if AFNIFileFound:
        logging.info('################################################################################')
        logging.info("Making XML Wrapper for AFNI Files")
        logging.info('################################################################################')
        run_cmd("afni2bxh " + flags + " " + os.path.join(TempPath, "*.HEAD") + " " + os.path.join(TempPath, "ImageWrapper-orig.xml"))
        
      else:
        logging.error('################################################################################')
        logging.error('# ERROR! NO AFNI .HEAD FILE FOUND!')
        logging.error('################################################################################')
        sys.exit(-1)
    if os.path.isfile(os.path.join(TempPath, "ImageWrapper.xml")):
      # merge new info into existing file
      os.rename(os.path.join(TempPath, "ImageWrapper.xml"), os.path.join(TempPath, "ImageWrapper-previous.xml"))
      if not MergeXCEDEAcqParams(os.path.join(TempPath, "ImageWrapper-previous.xml"),
                                 os.path.join(TempPath, "ImageWrapper-orig.xml"),
                                 os.path.join(TempPath, "ImageWrapper.xml")):
        logging.error('ERROR: Merging acquisition params from ' + os.path.join(TempPath, 'ImageWrapper-previous.xml') + ' and ' + os.path.join(TempPath, 'ImageWrapper-orig.xml') + ' to ' + os.path.join(TempPath, 'ImageWrapper.xml') + ' failed.')
        sys.exit(-1)
      os.unlink(os.path.join(TempPath, "ImageWrapper-previous.xml"))
      os.unlink(os.path.join(TempPath, "ImageWrapper-orig.xml"))
    else:
      os.rename(os.path.join(TempPath, "ImageWrapper-orig.xml"), os.path.join(TempPath, "ImageWrapper.xml"))
  
  ParseXMLHeader(os.path.join(TempPath, "ImageWrapper.xml"), BXHInfo, SiteInfo)
  
  # set DICOM flip if needed
  BXHInfo.set_DICOMFlipZ(False)
  if (seriestype == "functional" or seriestype == "functional-with-behavioral") and myImageFormat == Constants.m_DATASPEC_DICOM:
    # this is for Siemens DICOM files which introduce an errant flip
    if BXHInfo.get_RASFlag_Z() == "R" or BXHInfo.get_RASFlag_Z() == "L":
      BXHInfo.set_DICOMFlipZ(len(grep(DICOMFile, ".*sSliceArray.ucImageNumbSag.*")) > 0)
    elif BXHInfo.get_RASFlag_Z() == "A" or BXHInfo.get_RASFlag_Z() == "P":
      BXHInfo.set_DICOMFlipZ(len(grep(DICOMFile, ".*sSliceArray.ucImageNumbCor.*")) > 0)
    elif BXHInfo.get_RASFlag_Z() == "S" or BXHInfo.get_RASFlag_Z() == "I":
      BXHInfo.set_DICOMFlipZ(len(grep(DICOMFile, ".*sSliceArray.ucImageNumbTra.*")) > 0)
    logging.info("DICOMFlipZ=%d" % BXHInfo.get_DICOMFlipZ())
    if BXHInfo.get_DICOMFlipZ():
      if BXHInfo.get_RASFlag_Z()=="R":
        newz="L"
      elif BXHInfo.get_RASFlag_Z()=="L":
        newz="R"
      elif BXHInfo.get_RASFlag_Z()=="A":
        newz="P"
      elif BXHInfo.get_RASFlag_Z()=="P":
        newz="A"
      elif BXHInfo.get_RASFlag_Z()=="S":
        newz="I"
      elif BXHInfo.get_RASFlag_Z()=="I":
        newz="S"
      DICOMFlipOrientation = BXHInfo.get_RASFlag_X() + BXHInfo.get_RASFlag_Y() + newz
      logging.info("DICOMFlipOrientation="+DICOMFlipOrientation)
        
      
  logging.info('########################################################')
  logging.info("Making NIfTI Files")
  logging.info('########################################################')

  if which("bxh2analyze") == None:
    logging.error("Binary <bxh2analyze> Not Found! Check Your Path Environmental Variable")
    sys.exit(-1)

  try:
    shutil.rmtree(NIFTIDir)
  except OSError:
    pass
  os.makedirs(NIFTIDir)
  run_cmd("bxh2analyze -s --preferanalyzetypes --niigz --xcede " + os.path.join(TempPath, "ImageWrapper.xml") + " " + os.path.join(NIFTIDir, "f"))
  os.rename(os.path.join(NIFTIDir, "f.xml"), os.path.join(NIFTIDir, "ImageWrapper.xml"))

  if myImageFormat == Constants.m_DATASPEC_DICOM:
    if BXHInfo.get_DICOMFlipZ():
      logging.info('################################################################################')
      logging.info("Flipping Z direction")
      logging.info('################################################################################')
      run_cmd("bxhreorient --orientation " + DICOMFlipOrientation + " " + os.path.join(NIFTIDir, "ImageWrapper.xml") + " " + os.path.join(NIFTIDir, "_reorienttemp.xml"))
      os.unlink(os.path.join(NIFTIDir, "f.nii.gz"))
      os.unlink(os.path.join(NIFTIDir, "ImageWrapper.xml"))
      run_cmd("bxh2analyze -s --preferanalyzetypes --niigz --xcede " + os.path.join(NIFTIDir, "_reorienttemp.xml") + " " + os.path.join(NIFTIDir, "f"))
      os.rename(os.path.join(NIFTIDir, "f.xml"), os.path.join(NIFTIDir, "ImageWrapper.xml"))
      os.unlink(os.path.join(NIFTIDir, "_reorienttemp.xml"))
      os.unlink(os.path.join(NIFTIDir, "_reorienttemp.dat"))
        


################################################################################
# Prepare the directory for remote upload (big)
################################################################################
def LocalHumanTransfer(ConfigFile, XMLFile, SiteInfo, Database, UploadConfig, DataPaths, StagingPaths, RemotePaths, Constants, BXHInfo, RunConfig):
  global OPT_debug
  global OPT_update
  global OPT_nolinks
  
  logging.info('################################################################################')
  logging.info('#Collect format of Native_Dir & remote resource specifier')
  logging.info('################################################################################')
  logging.info("ScannerManufacturer is "+SiteInfo.get_ScannerManufacturer())
  logging.info("Native image format is "+Constants.m_DataSpecifiers[SiteInfo.get_NativeFormat()])
  logging.info("Structural format is "+Constants.m_DataSpecifiers[SiteInfo.get_StructFormat()])
  logging.info("Structural Anonymized format is "+Constants.m_DataSpecifiers[SiteInfo.get_StructAnonFormat()])
  logging.info('################################################################################')
  logging.info('#Collect XML config metadata')
  logging.info('################################################################################')

  #IBO 
  btestMode=False

  # Load the xml file
  reader = Sax2.Reader()
  XMLDoc = reader.fromUri(XMLFile)

  ParseUploadXML(XMLDoc, UploadConfig) # (will do series again since we may modify XMLDoc below)

  if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place" and SiteInfo.get_dbInstance() == None:
    logging.warn("Trying to upload in " + UploadConfig.get_FileSystem() + " mode but HID database support was not configured during installation!!")
    logging.warn("If you continue, you'll be asked to specify the HID database connection string later or the upload will fail!")
    if not read_yn("Are you sure you want to continue [y/n]?", "y"):
      logging.error("Either change the <FileSystem> tag in your XML Template file to \"Local\" or re-run script Install.sh and configure HID Database!")
      sys.exit(1)

  # Create a temporary directory.
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    StagingPaths.set_TempRoot(DataPaths.get_DataRoot())
  elif btestMode:
    StagingPaths.set_TempRoot(os.path.join(StagingPaths.get_TempDIR(), "_tempUpload.1222294747"))
  else:
    StagingPaths.set_TempRoot(os.path.join(StagingPaths.get_TempDIR(), ("_tempUpload.%d.%d" % (os.getpid(), time.mktime(time.localtime())))))
    if os.path.isdir(StagingPaths.get_TempRoot()):
      logging.info("Staging area exists, cleaning up....")
      shutil.rmtree(StagingPaths.get_TempRoot())
  
    os.makedirs(StagingPaths.get_TempRoot())
  
  if not SiteInfo.get_BehavioralEquipmentID():
    reader = Sax2.Reader()
    temp = XPathEval('/FIPS/series[type=\'functional-with-behavioral\']/type', XMLDoc)
    if len(temp) > 0:
      logging.error("BehavioralEquipID not found, and the configuration file contains functional-with-behavioral series type")
      sys.exit(3)
    temp = XPathEval('/FIPS/series[type=\'behavioral\']/type', doc)
    if len(temp) > 0:
      logging.error("BehavioralEquipID not found, and the configuration file contains behavioral series type" )
      sys.exit(3)

  logging.info('################################################################################')
  logging.info('#Checking for missing, required metadata from XML template file')
  logging.info('################################################################################')
  
  if UploadConfig.get_SiteID() == None:
    logging.warn("Site IDs:")
    for i in Constants.m_InstitutionsOrder:
      logging.warn("    "+Constants.m_Institutions[i]+"   =   %d" % i)
    logging.warn("Please enter the ID for the site that acquired this data: ")
    AcqID=string.atoi(sys.stdin.readline()[:-1])
    UploadConfig.set_SiteID(Constants.m_SiteIDs[AcqID])
  
  #make sure studyID has three padding zeros and have name and id separate for matching
  TruncatedStudyID=re.match("^0*(.*?)$", UploadConfig.m_StudyID.get_value()).group(1)
  UploadConfig.m_StudyID.set_value("000"+TruncatedStudyID)

  ###
  # Before we write anything to disk, let's check in with database and
  # make sure nothing is amiss
  ###

  ################################################################################
  # Get the host string and username for the HID part
  if SiteInfo.get_dbInstance() != None:
    if SiteInfo.get_ScannerID() == None:
      logging.info("Querying equipment information...")
      query="select uniqueid, make, model from nc_collectionequipment"
      assert SiteInfo.get_dbType() != None, "dbType was null"
      Database.execute(query)
      EquipmentInfo = Database.fetchall()
      
      logging.info("Number of Devices is %d" % len(EquipmentInfo))
      # logging.info(all of the devices.)
      index=0
      for row in EquipmentInfo:
        logging.warn("//////////////////////////////////////////////////////////////////////")
        logging.warn("%d) Device: %d; Manufacturer is %s and scanner name is %s" % (index, row[0], row[1], row[2]))
        index+=1
    
      while True:
        logging.warn("Please select a scanner that you used for this collection")
        Ans = string.atoi(sys.stdin.readline()[:-1])
        if Ans < len(EquipmentInfo) and Ans > 0:
          SiteInfo.set_ScannerID( EquipmentInfo[Ans][0] )
          logging.warn("ScannerID is " + str(SiteInfo.get_ScannerID()))
          break
        else:
          logging.warn("Invalid choice.")
    
    #check the validity of the equipment id
    #only if it returns 1 then we continue, otherwise we report an error and exit
    query="select count(*) from nc_collectionequipment where uniqueid = %d" % SiteInfo.get_ScannerID()
    Database.execute(query)
    result = Database.fetchone()[0]
    logging.info([result])
    
    if result != 1:
      logging.error("Equipment ID specified in the configuration file does not exist in the HID.  Please correct this.")
      return 3
    
    query="select count(*) from nc_collectionequipment where uniqueid = %d" % SiteInfo.get_BehavioralEquipmentID()
    Database.execute(query)
    result = Database.fetchone()[0]
    
    if result != 1:
      logging.error("Behavioral Equipment ID specified in the configuration file does not exist in the HID, please make sure it's in the HID. ")
      return 3
    
    if UploadConfig.get_RecruitSiteID() == None:
      logging.info(UploadConfig.m_SubjectID.get_value())
      logging.info(UploadConfig.m_SubjectID.get_value()[:4])
      logging.info(string.atoi(UploadConfig.m_SubjectID.get_value()[:4]))
      logging.info(Constants.m_SiteIDs[string.atoi(UploadConfig.m_SubjectID.get_value()[:4])])
      UploadConfig.set_RecruitSiteID(Constants.m_SiteIDs[string.atoi(UploadConfig.m_SubjectID.get_value()[:4])])
    logging.info("SubjectID is " + UploadConfig.m_SubjectID.get_value())
    logging.info("RecruitSiteID is " + UploadConfig.get_RecruitSiteID())

    logging.info("curProjectDir " + UploadConfig.get_CurProjectName())
    ################################################################################
    #checking against HID
    # 1. the project id and name first
    ################################################################################
    logging.info("The project name from XML file is " + UploadConfig.get_Project())
    query="select uniqueid from nc_experiment where name = '" + UploadConfig.get_Project() + "'"
    Database.execute(query)
    result = Database.fetchall()
    
    if len(result) == 1:
      UploadConfig.set_dbProjectID(result[0][0])
    else:
      query="select name from nc_experiment"
      Database.execute(query)
      result = Database.fetchall()
      
      logging.warn(UploadConfig.get_Project()+" is not in the HID.  Please select one from the following list:")
      listall(result)
      
      number = selectone("Project Name", len(result))
      dbProjectName=result[number-1][0]
      query="select uniqueid from nc_experiment where name like '"+dbProjectName+"%'"
      Database.execute(query)
      result = Database.fetchall()
      UploadConfig.set_dbProjectID(result[0][0])
  
    query="select baseuri from nc_experiment where uniqueid = %d" % UploadConfig.get_dbProjectID()
    Database.execute(query)
    result = Database.fetchall()
   
    basepath = None 
    if len(result) == 1:
      basepath = result[0][0]
    else:
      logging.error("Please update your HID with the project information first.")
      return 3
    
    query="select storagetype from nc_experiment where uniqueid = %d" % UploadConfig.get_dbProjectID()
    Database.execute(query)
    result = Database.fetchall()

    scheme = None
    hostname = ''
    if len(result) == 1:
      scheme = result[0][0]
      if scheme == 'local':
        scheme = 'file'
      elif scheme == 'gridftp':
        scheme = 'gsiftp'
        hostname = 'hostname'
    if scheme == None:
      logging.error("storagetype missing.  Please update your HID with project information first.")
      return 3

    baseuri = scheme + '://' + hostname + basepath
    RemotePaths.set_RemoteBaseURLstr(baseuri)
    logging.info("baseuri is "+urlparse.urlunparse(RemotePaths.get_RemoteBaseURL()))

    ################################################################################
    # Checking against HID
    # 2. the SubjectID , which is the BIRNID
    ################################################################################
    if UploadConfig.m_SubjectID.get_missing():
      logging.info("The BIRN ID is missing from the XML file, will use query result from the HID" )
    
    logging.info("Querying the HID for Subject ....")
    query="select distinct subjectid from nc_humansubject where subjectid=trim('"+UploadConfig.m_SubjectID.get_value()+"')"
    Database.execute(query)
    result = Database.fetchall()
    
    if len(result) == 0:
      logging.error("There are no subjects in the HID with the subjectID %s." % UploadConfig.m_SubjectID.get_value())
      return 3

    logging.info("Queryin the HID for Subject Group ....")
    query = "select rg.name from nc_researchgroup rg, nc_subjexperiment se where rg.uniqueid=se.nc_researchgroup_uniqueid and se.subjectid='%s' and se.nc_experiment_uniqueid=%d" % (UploadConfig.m_SubjectID.get_value(), UploadConfig.get_dbProjectID());
    Database.execute(query)
    result = Database.fetchall()

    if len(result) == 0:
      logging.error("Can't find subject group in the HID for the subjectID %s." % UploadConfig.m_SubjectID.get_value())
      return 3
    logging.info(result)
    hidSubjectGroup = result[0][0]
    if UploadConfig.m_SubjectGroup.get_missing():
      UploadConfig.m_SubjectGroup.set_value(hidSubjectGroup)
      projectnodes = xml.xpath.Evaluate('/FIPS/project',XMLDoc)
      newelem = XMLDoc.createElement('subjectGroup')
      newelem.appendChild(XMLDoc.createTextNode(hidSubjectGroup))
      projectnodes[0].parentNode.insertBefore(newelem, projectnodes[0])
    if hidSubjectGroup != UploadConfig.m_SubjectGroup.get_value():
      logging.error("Subject group for subject %s was specified as %s but in the HID it is %s!" % (UploadConfig.m_SubjectID.get_value(), UploadConfig.m_SubjectGroup.get_value(), hidSubjectGroup))
      return 3
  
  ###
  # Also get the remote uploader started up, just in case we are missing
  # anything
  ###
  RemoteUploader = None
  if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
    RemoteUploader = RemotePaths.get_RemoteUploader(SiteInfo, OPT_debug);
    if RemoteUploader == None:
      logging.error("ERROR getting upload class for " + RemotePaths.get_RemoteBaseURL())
      logging.error("  (reminder: it must be a URL)")
      return 1

  ###
  # Check to make sure (local) study directory does not exist
  ###
  if os.path.isdir(UploadConfig.get_CurStudyDir()):
    if OPT_update == False and UploadConfig.get_FileSystem() != "Local-In-Place":
      logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
      logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
      logging.error("ERROR! Duplicate Study for Subject in local <TARGET_DIR>:")
      logging.error(UploadConfig.get_CurStudyDir())
      logging.error("Please Remove Study and Re-Run Scripts")
      logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
      logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
      logging.error("....Removing Temporary Data....Please Wait.....")
      shutil.rmtree(StagingPaths.get_TempRoot())
      return 1

  #############################################################################
  # Collect output (local and/or remote) and input timestamps, and earlier
  # upload XML file (if any), for later use
  #############################################################################
  currentUploadTimestamp = time.time()
  outputTimestampStruct = {}
  earlierUploadConfig = copy.deepcopy(UploadConfig)
  if OPT_update:
    logging.info("Collecting timestamps from previous uploads.")
    # Time stamp dictionaries map the standard series name to a timestamp.
    # Input timestamps are the timestamp of the most-recently updated input
    # file or directory associated with that standard series name.
    # Output timestamps are read from a local and/or remote XML file and if the
    # local and remote output timestamps are different, the older one is used.
    # On upload, the output timestamps for any uploaded series will be the time
    # of the start of the upload process, and this time will be put in the
    # uploaded XML timestamp file.
    # These choices favor re-uploading data if there is any confusion.
    LocalTimestampFile = os.path.join(UploadConfig.get_CurStudyDir(), "Timestamps.xml")
    if os.path.exists(LocalTimestampFile):
      outputTimestampStruct = ReadTimestampFile(LocalTimestampFile)
    if RemoteUploader != None:
      remotebasedir = RemotePaths.get_RemotePath()
      remotesubjdir = os.path.join(remotebasedir, UploadConfig.m_SubjectID.get_value())
      remotevisitdir = os.path.join(remotesubjdir, UploadConfig.get_NewVisitName())
      remotestudydir = os.path.join(remotevisitdir, UploadConfig.get_NewStudyName())
      RemoteTimestampFile = os.path.join(StagingPaths.get_TempRoot(), "Timestamps.xml")
      try:
        RemoteUploader.get(os.path.join(remotestudydir, "Timestamps.xml"), RemoteTimestampFile)
      except:
        pass
      if os.path.exists(RemoteTimestampFile):
        remoteOutputTimestampStruct = ReadTimestampFile(RemoteTimestampFile)
        for key in outputTimestampStruct.keys():
          if key not in remoteOutputTimestampStruct:
            del outputTimestampStruct[key]
          else:
            localstruct = outputTimestampStruct[key]
            remotestruct = remoteOutputTimestampStruct[key]
            if remotestruct["timeStamp"] < localstruct["timeStamp"]:
              outputTimestampStruct[key] = remoteOutputTimestampStruct[key]
    logging.info("Collecting original upload XML file from previous uploads.")
    RemoteUploadFile = os.path.join(StagingPaths.get_TempRoot(), "Upload-earlier.xml")
    try:
      RemoteUploader.get(os.path.join(remotestudydir, "Upload-orig.xml"), RemoteUploadFile)
    except:
      pass
    LocalUploadFile = os.path.join(UploadConfig.get_CurStudyDir(), "Upload-orig.xml")
    for uploadfile in (LocalUploadFile, RemoteUploadFile):
      if OPT_update == '':
        continue
      if os.path.exists(uploadfile):
        logging.info("Reading earlier upload config file %s" % uploadfile)
        reader = Sax2.Reader()
        doc = reader.fromUri(uploadfile)
        ParseUploadXML(doc, earlierUploadConfig)
        if (UploadConfig.get_SiteID() != earlierUploadConfig.get_SiteID() or
            UploadConfig.m_SubjectID.get_value() != earlierUploadConfig.m_SubjectID.get_value() or
            UploadConfig.get_RecruitSiteID() != earlierUploadConfig.get_RecruitSiteID() or
            UploadConfig.get_ScannerManufacturer() != earlierUploadConfig.get_ScannerManufacturer() or
            UploadConfig.get_FileSystem() != earlierUploadConfig.get_FileSystem() or
            UploadConfig.get_ProjectXMLString() != earlierUploadConfig.get_ProjectXMLString() or
            UploadConfig.get_VisitXMLString() != earlierUploadConfig.get_VisitXMLString() or
            UploadConfig.get_StudyXMLString() != earlierUploadConfig.get_StudyXMLString()):
          logging.info("Important info in upload XML file has changed since previous upload.  Will re-upload everything.")
          # this is a special value meaning overwrite everything, even if it already exists.
          OPT_update = ''
          outputTimestampStruct = {} # make believe everything is new
          break
        for seriesind in range(len(UploadConfig.get_SeriesList())):
          standardName = UploadConfig.get_SeriesList(seriesind)
          if standardName not in outputTimestampStruct:
            continue
          earlierseriesind = None
          for tmpseriesind in range(len(earlierUploadConfig.get_SeriesList())):
            if earlierUploadConfig.get_SeriesList(tmpseriesind) == standardName:
              earlierseriesind = tmpseriesind
          if earlierseriesind != None:
            if UploadConfig.get_SeriesXMLString(seriesind) != earlierUploadConfig.get_SeriesXMLString(seriesind):
              outputTimestampStruct[standardName]['timeStamp'] = 0
              logging.info("Important info in upload XML file for series '%s' has changed since previous upload.  Will re-upload this series." % standardName)
      
  ###
  # These store the various protocol validation errors we encounter
  ###
  protocolErrs = []
  numProtocolErrs = 0

  ###
  # call BeforeSeries hook, if it exists
  ###
  hookProtocolErrList = [0, []]
  call_hook(os.path.join(SiteInfo.get_ScriptInstallDir(), "hooks", UploadConfig.m_ProjectName.get_value()), "BeforeSeries", XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct, hookProtocolErrList)
  if hookProtocolErrList[0] > 0:
    logging.error("=== Errors reported when running hooks: ===")
    logging.error('\n'.join(map(lambda x: x[0], hookProtocolErrList[1])) + '\n')
    if not read_yn("Are you sure you want to continue? [y/n]", "y"):
      logging.error("Exiting...")
      sys.exit(1)
    protocolErrs.append("=== Errors in project-specific hooks ===")
    protocolErrs.extend(hookProtocolErrList[1])
    numProtocolErrs += hookProtocolErrList[0]

  #############################################################################
  # Enter NATIVE_DIR, and gather preliminary image metadata (currently only
  # series number for automated behavioral data association, slice order,
  # series time, study time, and visit date)
  #############################################################################
  logging.info('Extracting preliminary image metadata...')

  # get series info from XML upload file (will do this again since we may
  # modify XMLDoc below to add slice order)
  ParseXMLSeries(XMLDoc, UploadConfig)
  numseries = len(UploadConfig.get_SeriesDirLocal())

  autodetectdata = [None] * numseries
  UploadConfig.set_SeriesSeriesNumber([None] * numseries)
  seriesnodes = xml.xpath.Evaluate('/FIPS/series[nameStandard]',XMLDoc)

  tmpvisitdate = None
  tmpstudytime = None
  for seriesind in range(numseries):
    if UploadConfig.get_Skip(seriesind):
      continue
    seriesnode = seriesnodes[seriesind]
    StandardName = UploadConfig.get_SeriesList(seriesind)
    LocalDir = UploadConfig.get_SeriesDirLocal(seriesind)
    LocalPath = DataPaths.get_LocalPath(LocalDir)
    if UploadConfig.get_SeriesType(seriesind) in [ "functional", "functional-with-behavioral", "structural", "structural-anon" ]:
      autodetectdata[seriesind] = AutoDetectImageDirectory(LocalPath)
      [formats, sdd] = autodetectdata[seriesind]
      sdd.extract_data()
      if 'seriesNumber' in sdd.metadata.keys() and sdd.metadata['seriesNumber'] != None:
        # save series number
        logging.info("Series with standard name " + StandardName + " has series number " + sdd.metadata['seriesNumber'])
        UploadConfig.set_SeriesSeriesNumber(seriesind, sdd.metadata['seriesNumber'])
      # extract a couple fields from the preliminary metadata and double-check that they match what's already in the upload config file.
      if 'seriesDate' in sdd.metadata:
        if tmpvisitdate == None or sdd.metadata['seriesDate'] < tmpvisitdate:
          tmpvisitdate = sdd.metadata['seriesDate']
      if 'seriesTime' in sdd.metadata:
        if tmpstudytime == None or sdd.metadata['seriesTime'] < tmpstudytime:
          tmpstudytime = sdd.metadata['seriesTime']
      for (sddname, ucarray) in ( ('sliceorder', UploadConfig.get_SeriesSliceOrder()),
                                  ('seriesTime', UploadConfig.get_SeriesTime()) ):
        if sddname in sdd.metadata.keys() and sdd.metadata[sddname] != None:
          fieldval = ucarray[seriesind]
          if fieldval != None and fieldval != '':
            # if exists already in upload XML file, skip, but double-check that the values match
            if fieldval != sdd.metadata[sddname]:
              if sddname == 'sliceorder':
                logging.critical("ERROR: %s for standard name %s specified in upload XML file (%s) does not match %s extracted from image data (%s)!" % (sddname, StandardName, fieldval, sddname, sdd.metadata[sddname]))
                sys.exit(1)
              else:
                logging.warning("WARNING: %s for standard name %s specified in upload XML file (%s) does not match %s extracted from image data (%s)!" % (sddname, StandardName, fieldval, sddname, sdd.metadata[sddname]))
            continue
          # add extracted field to upload config XML doc, since it doesn't exist yet (or was empty)
          logging.info("Series with standard name %s has %s %s" % (StandardName, sddname, sdd.metadata[sddname]))
          # make sure to get rid of any existing (presumably empty) nodes
          matchednodes = xml.xpath.Evaluate(sddname, seriesnode)
          for matchednode in matchednodes:
            matchednode.parentNode.removeChild(matchednode)
          newelem = XMLDoc.createElement(sddname)
          newelem.appendChild(XMLDoc.createTextNode(sdd.metadata[sddname]))
          seriesnode.appendChild(newelem)
  if tmpvisitdate != None and UploadConfig.m_VisitDate.get_missing():
    visitnodes = xml.xpath.Evaluate('/FIPS/visit',XMLDoc)
    if len(visitnodes) != 1:
      logging.error("ERROR: did not find exactly one <FIPS><visit> node in upload XML file!\n")
      return 1
    # make sure to get rid of any existing (presumably empty) nodes
    matchednodes = xml.xpath.Evaluate('visitDate', visitnodes[0])
    for matchednode in matchednodes:
      matchednode.parentNode.removeChild(matchednode)
    newelem = XMLDoc.createElement('visitDate')
    newelem.appendChild(XMLDoc.createTextNode(tmpvisitdate))
    visitnodes[0].appendChild(newelem)
  if tmpstudytime != None and UploadConfig.m_StudyTime.get_missing():
    studynodes = xml.xpath.Evaluate('/FIPS/study',XMLDoc)
    if len(studynodes) != 1:
      logging.error("ERROR: did not find exactly one <FIPS><study> node in upload XML file!\n")
      return 1
    # make sure to get rid of any existing (presumably empty) nodes
    matchednodes = xml.xpath.Evaluate('studyTime', studynodes[0])
    for matchednode in matchednodes:
      matchednode.parentNode.removeChild(matchednode)
    newelem = XMLDoc.createElement('studyTime')
    newelem.appendChild(XMLDoc.createTextNode(tmpstudytime))
    studynodes[0].appendChild(newelem)

  #############################################################################
  # Do behavior data search
  #############################################################################
  # Search through behavioral root directories, if provided
  UploadConfig.set_SeriesBehaviorDirs([[]] * numseries)
  if SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_CIGAL and len(DataPaths.get_BehaviorRoots()) > 0:
    logging.info('################################################################################')
    logging.info('# Finding behavioral directories and associating them with')
    logging.info('# the appropriate series directory')
    logging.info('################################################################################')
    # series2behav is a list of [rootdir,behavpath,copyrecursive]
    series2behav = {}
    pathqueue = []
    for behavroot in DataPaths.get_BehaviorRoots():
      while behavroot[-1] == os.sep:
        behavroot = behavroot[0:-1]
      (roothead, roottail) = os.path.split(behavroot)
      pathqueue.append((roothead, roottail))
    while len(pathqueue) > 0:
      (curroot, curpath) = pathqueue[0]
      pathqueue[0:1] = []
      (pathhead, pathtail) = os.path.split(curpath)
      if len(pathtail) == 0:
        (pathhead, pathtail) = os.path.split(pathhead)
      logging.info('checking ' + os.path.join(curroot,curpath))
      matchobj = re.search("^bhv_s(\d+)", pathtail)
      if matchobj:
        curseriesnum = matchobj.group(1)
        if int(curseriesnum) < 100:
          logging.info(' adding to series ' + curseriesnum)
          if curseriesnum not in series2behav:
            series2behav[curseriesnum] = []
          series2behav[curseriesnum].append([os.path.join(curroot,pathhead),pathtail,True])
        # everything gets added to "extra" series too
        logging.info(' adding to "extra" series')
        if series2behav.has_key(None):
          series2behav[None].append([curroot,os.path.join(pathhead,pathtail),True])
        else:
          series2behav[None] = [[curroot,os.path.join(pathhead,pathtail),True]]
      else:
        logging.info(' adding to "extra" series')
        if None not in series2behav:
          series2behav[None] = []
        series2behav[None].append([curroot,curpath,False])
        if os.path.isdir(os.path.join(curroot,curpath)):
          for file in os.listdir(os.path.join(curroot,curpath)):
            pathqueue.append((curroot, os.path.join(curpath, file)))
    # associate any found behavioral directories with the series they claim
    # to be a part of (this match will be overridden if behavioral data is
    # found in the image directory)
    logging.info("Associating found behavioral directories with the series they claim to be a part of (may be overridden later)")
    for seriesind in range(numseries):
      if UploadConfig.get_Skip(seriesind):
        continue
      if UploadConfig.get_SeriesType(seriesind) == "extra" and UploadConfig.get_SeriesParadigm(seriesind) == "cigal_data":
        seriesnumber = None
      elif UploadConfig.get_SeriesType(seriesind) != "functional-with-behavioral":
        continue
      else:
        seriesnumber = UploadConfig.get_SeriesSeriesNumber(seriesind)
        if seriesnumber == None:
          continue
      if series2behav.has_key(seriesnumber):
        logging.debug(" checking series number " + str(seriesnumber))
        behavdirs = series2behav[seriesnumber]
        UploadConfig.set_SeriesBehaviorDirs(seriesind, map(lambda x: (os.path.join(x[0], x[1]), x[1], x[2]), behavdirs))

  # Look for behavioral data in image directories
  for seriesind in range(numseries):
    if UploadConfig.get_Skip(seriesind):
      continue
    logging.info("Prepping to glob behavioral files: " + DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind)))
    mydirs=[]
    myglobs=[]
    if SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_EPRIME:
      myglobs.append(".*\.edat")
      myglobs.append(".*\.txt")
    elif SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_CIGAL:
      myglobs.append("bhv_.*")
      
    logging.info("Globbing with: " + ' '.join(myglobs))

    if btestMode: myglobs=[]

    localseriespath = DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind))

    foundpaths = [] # list of (path, destname, flag if recursive copy) tuples
    for myglob in myglobs:
      for file in filter(lambda x: re.match(myglob, x) != None, os.listdir(localseriespath)):
        found = True
        newpath = os.path.join(localseriespath, file)
        logging.info("File: " + file)
        foundpaths.append((newpath, file, os.path.isdir(newpath)))
    if len(foundpaths) > 0:
      # these paths (actually found in the image directory) override any
      # previously associated behavioral data for this series
      UploadConfig.set_SeriesBehaviorDirs(seriesind, foundpaths)
    else:
      logging.info('Did not find any CIGAL behavioral data in the image directory ' + localseriespath)

  # Determine whether multiple series are claiming a particular behavioral
  # directory
  logging.info("Checking to see if any behavioral directories are claimed by multiple series")
  claimedbehavdirs = {}
  for seriesind in range(numseries):
    if UploadConfig.get_Skip(seriesind):
      continue
    seriesnumber = UploadConfig.get_SeriesSeriesNumber(seriesind)
    if seriesnumber == None:
      continue
    for (copypath, destname, recursive) in UploadConfig.get_SeriesBehaviorDirs(seriesind):
      if copypath not in claimedbehavdirs:
        claimedbehavdirs[copypath] = []
      claimedbehavdirs[copypath].append(seriesind)
  behaverror = False
  for (behavdir, seriesindlist) in claimedbehavdirs.iteritems():
    if len(seriesindlist) > 1:
      logging.error("ERROR: The following series:\n %s\nare all laying claim to the behavioral directory '%s'!'" % ('\n '.join(map(lambda x: ("standardName=%s seriesnumber=%s" % (UploadConfig.get_SeriesList(x), UploadConfig.get_SeriesSeriesNumber(x))), seriesindlist)), behavdir))
      behaverror = True
  if behaverror:
    return 1

  # Flag missing behavioral data
  numBehavErrs = 0
  behavErrs = []
  for seriesind in range(numseries):
    if UploadConfig.get_Skip(seriesind):
      continue
    if UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral":
      if len(UploadConfig.get_SeriesBehaviorDirs(seriesind)) == 0:
        errMsg = "Error: no behavioral files found in 'functional-with-behavioral' series " + DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind)) + "\n"
        numBehavErrs += 1
        behavErrs.append((errMsg, None))
  if numBehavErrs > 0:
    logging.error("Behavioral data check resulted in " + ("%d" % len(behavErrs)) + " errors.")
    logging.error(''.join(map((lambda x: x[0]), behavErrs)))
    if not read_yn("Are you sure you want to continue? [y/n]", "y"):
      logging.error("Exiting...")
      sys.exit(1)
  numProtocolErrs += numBehavErrs
  protocolErrs.extend(behavErrs)
 

  #############################################################################
  # Go through output (local and/or remote) and input timestamps to know what
  # we can skip.
  #############################################################################
  if OPT_update:
    logging.info("Update mode enabled.  Will look at timestamps to see what we don't need to re-upload.")
    # go through input data
    for seriesind in range(numseries):
      if UploadConfig.get_Skip(seriesind):
        continue
      LocalName = UploadConfig.get_SeriesNameLocal(seriesind)
      LocalDir = UploadConfig.get_SeriesDirLocal(seriesind)
      LocalPath = DataPaths.get_LocalPath(LocalDir)
      StandardName = UploadConfig.get_SeriesList(seriesind)
      TempPath = None
      if UploadConfig.get_FileSystem() != "Local-In-Place":
        TempPath = StagingPaths.get_TempPath(LocalName)
      else:
        TempPath = os.path.join(UploadConfig.get_TargetDir(), LocalDir)
      # recursively look at each file/directory in this series (including any
      # behavioral directories associated with this series) and find the
      # newest modification time
      localinode = os.stat(LocalPath)[stat.ST_INO]
      queue = []
      if os.path.exists(TempPath) and localinode == os.stat(TempPath)[stat.ST_INO]:
        # don't check mtime of top-level temp directories
        # (created by hooks, for example)
        queue.append( (LocalPath, False, True) )
      else:
        queue.append( (LocalPath, True, True) )
      for (copypath, destname, recursive) in UploadConfig.get_SeriesBehaviorDirs(seriesind):
        # add this behavioral directory to the queue
        queue.append( (copypath, True, recursive) )
      inputMTime = MaxMTime(queue)
      if StandardName in outputTimestampStruct.keys():
        if inputMTime < outputTimestampStruct[StandardName]["timeStamp"]:
          logging.info("Will not re-upload series " + LocalName + "/" + StandardName + ".  Timestamps: input(" + time.ctime(inputMTime) + ") < output(" + time.ctime(outputTimestampStruct[StandardName]["timeStamp"]) + ")")
          seriesnodes[seriesind].appendChild(XMLDoc.createElement('skip'))
          UploadConfig.set_Skip(seriesind, True)
        else:
          logging.info("Will re-upload series " + LocalName + "/" + StandardName + ".  Timestamps: input(" + time.ctime(inputMTime) + ") >= output(" + time.ctime(outputTimestampStruct[StandardName]["timeStamp"]) + ")")


  # replace "skipped" series with data from earlier uploads
  RemoteUploadFile = os.path.join(StagingPaths.get_TempRoot(), "Upload-updated-earlier.xml")
  try:
    RemoteUploader.get(os.path.join(remotestudydir, "Upload-updated.xml"), RemoteUploadFile)
  except:
    pass
  LocalUploadFile = os.path.join(UploadConfig.get_CurStudyDir(), "Upload-updated.xml")
  oldseriesnodemap = {}
  for uploadfile in (LocalUploadFile, RemoteUploadFile):
    if os.path.exists(uploadfile):
      reader = Sax2.Reader()
      doc = reader.fromUri(uploadfile)
      oldseriesnodes = xml.xpath.Evaluate('/FIPS/series[nameStandard]', XMLDoc)
      for oldseriesnode in oldseriesnodes:
        try:
          nameStandard = xml.xpath.Evaluate('string(nameStandard)', oldseriesnode)
          oldseriesnodemap[nameStandard] = oldseriesnode
        except:
          continue
  for seriesind in range(numseries):
    if UploadConfig.get_Skip(seriesind):
      StandardName = UploadConfig.get_SeriesList(seriesind)
      if StandardName in oldseriesnodemap:
        logging.info('Since we are skipping series with standard name "%s", using metadata from earlier upload.' % StandardName)
        newseriesnode = XMLDoc.importNode(oldseriesnodemap[StandardName], True)
        seriesnodes[seriesind].parentNode.insertBefore(newseriesnode, seriesnodes[seriesind])
        seriesnodes[seriesind].parentNode.removeChild(seriesnodes[seriesind])
        seriesnodes[seriesind] = oldseriesnodemap[StandardName]

  ###
  # We now may have an updated upload XML file (via the BeforeSeries hook and
  # preliminary image metadata extraction) so now actually do things with it
  ###

  ###
  # Do initial protocol validation
  ###
  protocolfile = os.path.join(SiteInfo.get_ScriptInstallDir(), "protocols", UploadConfig.m_ProjectName.get_value()+".schematron")
  schematron = _Schematron(protocolfile)
  if OPT_debug:
    [retval, dummy] = schematron.printtraverse()
    if retval < 0:
      logging.error("Error traversing Schematron file.\n")
      sys.exit(1)
  logging.info("Looking for protocol file: " + protocolfile)
  tmpProtocolErrs = []
  tmpNumProtocolErrs = 0
  if os.path.isfile(protocolfile):
    scannermanufacturer = UploadConfig.get_ScannerManufacturer()
    for phase in [ "FIPS", "FIPS (%s)" % scannermanufacturer, "FIPS UploadScript", "FIPS UploadScript (%s)" % scannermanufacturer]:
      tmpProtocolErrs.append(("==== Phase: " + phase + "\n", None))
      [numerrs, errorMsgs] = schematron.applyToDoc(XMLDoc, XMLFile, phase, debug=OPT_debug)
      if numerrs < 0:
        logging.error("Error applying Schematron file.\n")
        sys.exit(1)
      # filter out errors whose context nodes are in "skipped" series
      for errstruct in errorMsgs:
        (msg, contextnodes) = errstruct
        if type(contextnodes) == types.ListType:
          skipthis = 0
          logging.debug("checking %d context nodes to see whether we should skip them" % len(contextnodes))
          for contextnode in contextnodes:
            for seriesind in range(numseries):
              if UploadConfig.get_Skip(seriesind):
                seriesnode = seriesnodes[seriesind]
                nameStandard = xml.xpath.Evaluate('string(nameStandard)', seriesnode)
                logging.debug("  against skipped '%s' series node %s" % (nameStandard, str(seriesnode)))
                skipresult = xml.xpath.Evaluate('count(ancestor-or-self::series[nameStandard="%s"]|.) = count(ancestor-or-self::series[nameStandard="%s"])' % (nameStandard, nameStandard), contextnode)
                if skipresult == xml.utils.boolean.true:
                  logging.debug("   yes, skip this error")
                  skipthis = 1
                  break
          if skipthis == 1:
            continue
        tmpNumProtocolErrs += 1
        tmpProtocolErrs.append(errstruct)
  if int(SiteInfo.get_AcquisitionSiteID()) != int(UploadConfig.get_SiteID()):
    errMsg = "Warning: acquisition site ID from upload XML (" + str(int(UploadConfig.get_SiteID())) + ") does not match acquisition site ID from upload script install (" + str(int(SiteInfo.get_AcquisitionSiteID())) + ")\n"
    tmpNumProtocolErrs += 1
    tmpProtocolErrs.append((errMsg, None))
  if tmpNumProtocolErrs > 0:
    logging.error("Protocol validation of upload XML file resulted in %d errors." % tmpNumProtocolErrs)
    logging.error(''.join(map((lambda x: x[0]), tmpProtocolErrs)))
    if not read_yn("Are you sure you want to continue? [y/n]", "y"):
      logging.error("Exiting...")
      sys.exit(1)
  numProtocolErrs += tmpNumProtocolErrs
  protocolErrs.extend(tmpProtocolErrs)

  # get updated info from XML upload file
  ParseUploadXML(XMLDoc, UploadConfig)

  #############################################################################
  # Do a check to make sure we actually have data to upload
  #############################################################################
  if OPT_update:
    found = False
    for seriesind in range(numseries):
      if not UploadConfig.get_Skip(seriesind):
        found = True
        break
    if found == False:
      logging.info("Did not find any (updated) data to upload!  Leaving things as they are and exiting...")
      return 0
  
  # Add series info to database
  newXMLFile = os.path.join(StagingPaths.get_TempRoot(), "tmpUploadFile.xml")
  logging.info("Writing out new upload XML file to " + newXMLFile + "...")
  SaveXML(newXMLFile, XMLDoc)
  logging.info(" ...and updating database")
  [retval, retdata] = run_cmd("AddVisit2HID.sh " + newXMLFile);
  if retval != 0:
    logging.error("ERROR: Adding visit to HID failed with exit status " + str(retval) + "!")
    logging.error("  output:\n" + retdata)
    return 1

  # double check that subject has been enrolled
  if SiteInfo.get_dbInstance() != None:
    logging.info("Querying the HID for Subject ....")
    query="select distinct subjectid from nc_expcomponent where nc_experiment_uniqueid = %d" % UploadConfig.get_dbProjectID() +" and subjectid=trim('"+UploadConfig.m_SubjectID.get_value()+"')"
    Database.execute(query)
    result = Database.fetchall()
    
    if len(result) == 1:
      logging.info(result)
      dbSubjectID = result[0][0]
      logging.info("the BIRN ID is "+dbSubjectID)
    else:
      query="select distinct subjectid from nc_expcomponent where nc_experiment_uniqueid = %d" % UploadConfig.get_dbProjectID()
      Database.execute(query)
      result = Database.fetchall()
      if len(result) == 0:
             logging.error("There are no subjects that have any visits with this experiment in the HID." )
             return 3
      else:
        logging.info(UploadConfig.m_SubjectID.get_value()+" specified in the XML file is not in the HID, please select a subject from the HID:")
        listall(result)
        number = selectone("BIRNID", len(result))
        UploadConfig.m_SubjectID.set_value(result[number-1][0])
        logging.info("the BIRN ID you selected is " + UploadConfig.m_SubjectID.get_value())
  
  ###
  # OK, now we can start doing stuff on the file system
  ###
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    logging.info("################################################################################")
    logging.info("#Checking input directories.......")
    logging.info("################################################################################")
    numerrs = 0
    for i in range(numseries):
      NativeDir = os.path.join(UploadConfig.get_CurStudyDir(),UploadConfig.get_SeriesList(i),"Native")
      NiftiDir = os.path.join(UploadConfig.get_CurStudyDir(),UploadConfig.get_SeriesList(i),"NiftiFormat")
      WrapperFile = os.path.join(UploadConfig.get_CurStudyDir(),UploadConfig.get_SeriesList(i),"ImageWrapper.xml")
      for checkfile in [NativeDir, NiftiDir, WrapperFile]:
        if os.path.isdir(checkfile) or os.path.isfile(checkfile):
          if numerrs == 0:
            logging.error("ERROR: The following files/directories exist.  This script (in Local-In-Place\nmode) needs to create or write files or directories with those names.\nPlease rename or remove these before running the script again:")
          logging.error("  " + checkfile)
          numerrs = numerrs + 1
    if numerrs > 0:
      sys.exit(1)

  logging.info("################################################################################")
  logging.info("#Analyzing Root Image Directory.......")
  logging.info("################################################################################")
  
  CorrectConfig=0
  if re.match(".*"+UploadConfig.get_Project()+".*", DataPaths.get_DataRoot()):
    logging.info("Root image directory may already be in correct remote configuration....inspecting...")
  
    if re.match(".*"+UploadConfig.get_SubjectDir()+".*", DataPaths.get_DataRoot()):
      logging.info("Subject directory found, continuing inspection.....")
  
      tempdir=os.path.join(DataPaths.get_DataRoot(),UploadConfig.get_SubjectDir())
      logging.info("tempdir = "+tempdir)
  
      if re.match(".*"+UploadConfig.get_VisitDir()+".*", DataPaths.get_DataRoot()):
  
        logging.info("Visit directory found, continuing inspection.....")
  
        tempdir = os.path.join(tempdir,UploadConfig.get_VisitDir())
        logging.info("tempdir = "+tempdir)
  
        if re.match(".*"+UploadConfig.get_Study()+".*", DataPaths.get_DataRoot()):
          logging.info("Study directory found, directories seem to be in the correct configuration.....")
          CorrectConfig=1
        elif os.path.isdir(os.path.join(tempdir,UploadConfig.m_StudyName.get_value()+"__"+TruncatedStudyID)):
          os.rename(os.path.join(tempdir,UploadConfig.m_StudyName.get_value()+"__"+TruncatedStudyID, tempdir,UploadConfig.Study))
          logging.info("Study directory found, directories seem to be in the correct configuration.....")
          CorrectConfig=1    
        else:
          logging.info("Study directory not found but Visit already exists!!!  Please delete previous incomplete upload")
          logging.info("from: "+os.path.join(tempdir,UploadConfig.get_VisitDir())+" and re-run the scripts!")
          return 1
      else:
        logging.info("Visit directory not found!  Continuing with regular upload....")
    else:
      logging.info("Subject directory not found!  Continuing with regular upload....")

  logging.info('################################################################################')
  logging.info('# Checking for missing, required information.....')
  logging.info('################################################################################')
  for i in range(numseries):
    if UploadConfig.get_SeriesSkipInitialVols(i) == None and (UploadConfig.get_SeriesType(i) == "functional" or UploadConfig.get_SeriesType(i) == "functional-with-behavioral"):
      logging.error("ERROR! Skip Initial Vols Not Set for: "+UploadConfig.get_SeriesList(i))
      sys.exit(1)
    
    if UploadConfig.get_SeriesType(i) == None:
      logging.warn("Series Type not set for: "+UploadConfig.get_SeriesList(i)+"!")
      while True:
        logging.warn("Please enter structural, functional, functional-with-behavioral, structural-anon, exclude, behavioral, extra, or segmentation: ")
        UploadConfig.set_SeriesType(i, sys.stdin.readline()[:-1])
        if UploadConfig.IsSeriesTypeValid(i):
          break
  
    if UploadConfig.get_SeriesDirLocal(i) == None:
      logging.warn("Local Directory for Series Not Defined!: " + UploadConfig.get_SeriesList(i))
      logging.warn("Please enter local directory for series:")
      UploadConfig.set_SeriesDirLocal(i, sys.stdin.readline()[:-1])
  
  for i in range(numseries):
    if (UploadConfig.get_SeriesType(i) == "functional" or UploadConfig.get_SeriesType(i) == "functional-with-behavioral") and UploadConfig.get_SeriesAnalysisLevelLevel(i) == None:
      logging.error("ERROR: Analysis Level level (analysislevel/level element) does not exist for functional run: "+UploadConfig.get_SeriesList(i)+"!")
      sys.exit(1)
    if (UploadConfig.get_SeriesType(i) == "functional" or UploadConfig.get_SeriesType(i) == "functional-with-behavioral") and UploadConfig.get_SeriesAnalysisLevelName(i) == None:
      logging.error("ERROR: Analysis Level name (analysislevel/name element) does not exist for functional run: "+UploadConfig.get_SeriesList(i)+"!")
      sys.exit(1)
  
  #the native_dir is the found runname[] directory under the rootimagedir
  #temp = numseries - 1


  ################################################################################
  # Enter NATIVE_DIR, link data into temp path, make XML header, and gather
  # more metadata
  ################################################################################
  UploadConfig.set_DICOMFlipZ([0] * numseries)
  BXHInfo.set_Model([])
  
  if CorrectConfig == 0:
    UploadConfig.set_SeriesImageFormat([None] * numseries)
    for seriesind in range(numseries):
      if UploadConfig.get_Skip(seriesind):
        continue
      
      LocalName = UploadConfig.get_SeriesNameLocal(seriesind)
      LocalDir = UploadConfig.get_SeriesDirLocal(seriesind)
      LocalPath = DataPaths.get_LocalPath(LocalDir)
      TempPath = None
      if UploadConfig.get_FileSystem() != "Local-In-Place":
        TempPath = StagingPaths.get_TempPath(LocalName)
      else:
        TempPath = os.path.join(UploadConfig.get_TargetDir(), LocalDir)
      NIFTIDir = StagingPaths.get_NIFTIDir(LocalName)
      try:
        os.makedirs(TempPath)
      except OSError:
        pass

      myImageFormats=[]
      myImageExamples=[]
      if UploadConfig.IsSeriesTypeValid(seriesind):
        if UploadConfig.get_SeriesType(seriesind) == "functional":
          myImageFormats=[SiteInfo.get_NativeFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral":
          myImageFormats=[SiteInfo.get_NativeFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "structural":
          myImageFormats=[SiteInfo.get_StructFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "structural-anon":
          myImageFormats=[SiteInfo.get_StructAnonFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "behavioral":
          myImageFormats=[SiteInfo.get_BehavioralFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "segmentation":
          myImageFormats=[SiteInfo.get_SegmentFormat()]
        elif UploadConfig.get_SeriesType(seriesind) == "extra":
          myImageFormats=[Constants.m_DATASPEC_UNKNOWN]

        if len(myImageFormats) == 0 or myImageFormats[0] == Constants.m_DATASPEC_AUTODETECT:
          if autodetectdata[seriesind] != None:
            (myImageFormats,sdd) = autodetectdata[seriesind]
            if len(sdd.example_files) == len(myImageFormats):
              myImageExamples = sdd.example_files
        if len(myImageFormats) == 0 or myImageFormats[0] == None:
          myImageFormats = [Constants.m_DATASPEC_UNKNOWN]

        UploadConfig.set_SeriesImageFormat(seriesind, myImageFormats[0])
      else:
        logging.error("ERROR: Series type '%s' in upload XML file is invalid!" % UploadConfig.get_SeriesType(seriesind))
        return 1

      # soft link data files into temp directory (unless it's already a
      # temp directory created by hooks!)
      localinode = os.stat(LocalPath)[stat.ST_INO]
      tempinode = os.stat(TempPath)[stat.ST_INO]
      if localinode != tempinode:
        for file in ls(LocalPath):
          if btestMode: continue
          StrippedFile = re.match("^.*/([^/]+?)$", file).group(1)
          if OPT_nolinks:
            try:
              shutil.copyfile(file, os.path.join(TempPath,StrippedFile))
            except OSError, e:
              if UploadConfig.get_FileSystem() != "Local-In-Place":
                logging.error("ERROR: Unable to copy file" + str([file, os.path.join(TempPath,StrippedFile)]))
                raise e
          else:
            try:
              os.symlink(file, os.path.join(TempPath,StrippedFile))
            except OSError, e:
              if UploadConfig.get_FileSystem() != "Local-In-Place":
                logging.error("ERROR: Unable to create symlink" + str([file, os.path.join(TempPath,StrippedFile)]))
                raise e
       
      if UploadConfig.IsSeriesTypeValid(seriesind):
        # just in case we linked in files/dirs we're going to create anyway
        if Constants.m_DATASPEC_XCEDE in myImageFormats:
          try:
            os.unlink(os.path.join(TempPath, "ImageWrapper.xml"))
          except OSError:
            pass

        if UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral" or UploadConfig.get_SeriesType(seriesind) == "structural" or UploadConfig.get_SeriesType(seriesind) == "structural-anon" or UploadConfig.get_SeriesType(seriesind) == "segmentation":
          pass
        else:
          logging.info('################################################################################')
          logging.info("# Excluding " + LocalDir)
          logging.info('################################################################################')
          continue
        if not btestMode:
          BXHInfo = _BXHInfo()
          CreateBXHFiles(myImageFormats, LocalPath, TempPath, BXHInfo, SiteInfo, NIFTIDir, True, example_files=myImageExamples, seriestype=UploadConfig.get_SeriesType(seriesind))
          UploadConfig.set_SeriesSeriesNumber(seriesind, BXHInfo.get_SeriesNumber())
          UploadConfig.set_DICOMFlipZ(seriesind, BXHInfo.get_DICOMFlipZ())
        # Check slice order field
        if (UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral") and UploadConfig.get_SeriesSliceOrder(seriesind) == None and BXHInfo.get_SliceOrder() == None:
          logging.error("ERROR! Slice Order Not Set for: "+UploadConfig.get_SeriesList(seriesind))
          sys.exit(1)
        # Verify that slice order fields are correct
        if not btestMode and (UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral"):
          localname = UploadConfig.get_SeriesNameLocal(seriesind)
          standardname = UploadConfig.get_SeriesList(seriesind)
          seriesidentifier = localname + "/" + standardname
          sliceorder = []
          if UploadConfig.get_SeriesSliceOrder(seriesind) != None:
            sliceorder = map(int, UploadConfig.get_SeriesSliceOrder(seriesind).split(","))
          elif BXHInfo.get_SliceOrder() != None:
            sliceorder = map(int, BXHInfo.get_SliceOrder().split(","))
          numslices = int(BXHInfo.get_Matrix_Z())
          sortedsliceorder = list(sliceorder) # copy the list for sorting
          sortedsliceorder.sort()
          if sortedsliceorder[0] < 1 or sortedsliceorder[-1] > numslices:
            logging.error("ERROR: all items in " + seriesidentifier + " <sliceorder> element must be between 1 and " + str(numslices) + "!")
            sys.exit(1)
          if len(sliceorder) != numslices:
            logging.error("ERROR: " + seriesidentifier + " <sliceorder> element must have " + str(numslices) + " slice indices (has " + str(len(sliceorder)) + ")!")
            sys.exit(1)
          for x in range(len(sortedsliceorder)-1):
            if sortedsliceorder[x] == sortedsliceorder[x+1]:
              logging.error("ERROR: found repeated index (" + str(sortedsliceorder[x]) + ") in " + seriesidentifier + " <sliceorder> element!")
              sys.exit(1)
          increasing = range(1, numslices+1)
          decreasing = range(numslices, 0, -1)
          oddevenincreasing = range(1, numslices+1, 2) + range(2, numslices+1, 2)
          evenoddincreasing = range(2, numslices+1, 2) + range(1, numslices+1, 2)
          lastodd = numslices - ((numslices + 1) % 2)
          lasteven = numslices - (numslices % 2)
          oddevendecreasing = range(lastodd, 0, -2) + range(lasteven, 0, -2)
          evenodddecreasing = range(lasteven, 0, -2) + range(lastodd, 0, -2)
          sliceorderdesc = None
          if sliceorder == increasing:
            sliceorderdesc = 'increasing from 1'
          elif sliceorder == decreasing:
            sliceorderdesc = 'decreasing to 1'
          elif sliceorder == oddevenincreasing:
            sliceorderdesc = 'odds then evens, increasing from 1'
          elif sliceorder == evenoddincreasing:
            sliceorderdesc = 'evens then odds, increasing from 1'
          elif sliceorder == oddevendecreasing:
            sliceorderdesc = 'odds then evens, decreasing to 1'
          elif sliceorder == evenodddecreasing:
            sliceorderdesc = 'evens then odds, decreasing to 1'
          else:
            sliceorderdesc = '<unrecognized>'
          logging.info("You specified '" + sliceorderdesc + "' slice order for series " + seriesidentifier)

        # apply protocol validation to image wrapper file
        numProtocolErrs2 = 0
        protocolErrs2 = []
        if not btestMode and os.path.isfile(protocolfile):
          # Load the xml file
          reader = Sax2.Reader()
          NIFTIXMLFile = os.path.join(NIFTIDir, "ImageWrapper.xml")
          NIFTIXMLDoc = reader.fromUri(NIFTIXMLFile)
          protocolErrs2.append(("==== Phase: XCEDE " + UploadConfig.get_SeriesType(seriesind) + " " + UploadConfig.get_SeriesParadigm(seriesind) + "\n", None))
          [numerrs, errorMsgs] = schematron.applyToDoc(NIFTIXMLDoc, NIFTIXMLFile, "XCEDE " + UploadConfig.get_SeriesType(seriesind) + " " + UploadConfig.get_SeriesParadigm(seriesind), debug=OPT_debug)
          if numerrs < 0:
            logging.error("Error applying Schematron file.\n")
            sys.exit(1)
          numProtocolErrs2 += numerrs
          protocolErrs2.extend(errorMsgs)
          protocolErrs2.append(("==== Phase: XCEDE " + UploadConfig.get_SeriesType(seriesind) + " " + UploadConfig.get_SeriesParadigm(seriesind) + " (" + UploadConfig.get_ScannerManufacturer() + ")\n", None))
          [numerrs, errorMsgs] = schematron.applyToDoc(NIFTIXMLDoc, NIFTIXMLFile, "XCEDE " + UploadConfig.get_SeriesType(seriesind) + " " + UploadConfig.get_SeriesParadigm(seriesind) + " (" + UploadConfig.get_ScannerManufacturer() + ")", debug=OPT_debug)
          if numerrs < 0:
            logging.error("Error applying Schematron file.\n")
            sys.exit(1)
          numProtocolErrs2 += numerrs
          protocolErrs2.extend(errorMsgs)
        if not btestMode and BXHInfo.get_StudyDate() != 'UnknownDate' and BXHInfo.get_StudyDate() != UploadConfig.m_VisitDate.get_value():
          errMsg = "Error reported: Date from image (" + str(BXHInfo.get_StudyDate()) + ") does not match date from upload config file (" + str(UploadConfig.m_VisitDate.get_value()) + ")\n"
          numProtocolErrs2 += 1
          protocolErrs2.append((errMsg, None))
        if numProtocolErrs2 > 0:
          logging.error("Protocol validation of upload XML file resulted in " + ("%d" % numProtocolErrs2) + " errors.")
          logging.error(''.join(map((lambda x: x[0]), protocolErrs2)))
          if not read_yn("Are you sure you want to continue? [y/n]", "y"):
            logging.error("Exiting...")
            sys.exit(1)
        numProtocolErrs += numProtocolErrs2
        protocolErrs.extend(protocolErrs2)

    
    ###########################################################################
    # Local Collection Creation and Uploading
    # *** This is where the "local" uploaded data is first created/updated ***
    ###########################################################################
  
    logging.info('################################################################################')
    logging.info('#Preparing Local Collection Hierarchy....')
    logging.info('################################################################################'    )
    
    #Series Directory
      
    try:
      os.makedirs(UploadConfig.get_TargetDir())
    except OSError:
      pass
  
    SetSymLink = False
    if UploadConfig.get_TargetDir() != RemotePaths.get_LocalRemoteHome():
      SetSymLink = True
      logging.info("***********************************")
      logging.info("<LOCAL_REMOTE_HOME> parameter not equal to <TARGET_DIR>, data being symbolically linked.")
      logging.info("***********************************")
      try:
        os.makedirs(RemotePaths.get_LocalRemoteHome())
      except OSError:
        pass
    else:
      SetSymLink = False
    
    if SetSymLink:
      Symroot=os.path.join(RemotePaths.get_LocalRemoteHome(),UploadConfig.get_Project())
    
    logging.info("Making Sub-Collection: "+os.path.join(UploadConfig.get_PathRoot(),UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir()))
    try:
      os.makedirs(os.path.join(UploadConfig.get_PathRoot(),UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir()))
    except OSError:
      pass

    if SetSymLink:
      logging.info("Making directory for sym links: "+os.path.join(Symroot,UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir()))
      try:
        os.makedirs(os.path.join(Symroot,UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir()))
      except OSError:
        pass
  
    if os.path.isdir(UploadConfig.get_CurStudyDir()):
      # check explicitly for False, because OPT_update could be the empty string, which means upload everything (again)
      if OPT_update == False and UploadConfig.get_FileSystem() != "Local-In-Place":
        logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
        logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
        logging.error("ERROR! Duplicate Study for Subject in local <TARGET_DIR>:")
        logging.error(UploadConfig.get_CurStudyDir())
        logging.error("Please Remove Study and Re-Run Scripts")
        logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
        logging.error("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
        logging.error("....Removing Temporary Data....Please Wait.....")
        shutil.rmtree(StagingPaths.get_TempRoot())
        return 1
#      else:
#        logging.info("Recreating Sub-Collection: "+UploadConfig.get_CurStudyDir())
#        try:
#          shutil.rmtree(UploadConfig.get_CurStudyDir())
#          os.makedirs(UploadConfig.get_CurStudyDir())
#        except OSError:
#          pass
    else:
      logging.info("Making Sub-Collection: "+UploadConfig.get_CurStudyDir())
      try:
        os.makedirs(UploadConfig.get_CurStudyDir())
      except OSError:
        pass
      if SetSymLink:
          logging.info("Making directory for sym links: "+os.path.join(Symroot,UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir(),UploadConfig.get_Study()))
          try:
            os.makedirs(os.path.join(Symroot,UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir(),UploadConfig.get_Study()))
          except OSError:
            pass

    ####
    # Write out protocol errors
    ####
    if numProtocolErrs > 0:
      oldProtocolDeviations = []
      if OPT_update:
        # read in old version of ProtocolDeviations file and append to it
        oldProtocolDeviationsFile = os.path.join(UploadConfig.get_CurStudyDir(), "ProtocolDeviations")
        if RemoteUploader != None:
          remoteProtocolDeviationsFile = os.path.join(StagingPaths.get_TempRoot(), "ProtocolDeviations")
          remotebasedir = RemotePaths.get_RemotePath()
          remotesubjdir = os.path.join(remotebasedir, UploadConfig.m_SubjectID.get_value())
          remotevisitdir = os.path.join(remotesubjdir, UploadConfig.get_NewVisitName())
          remotestudydir = os.path.join(remotevisitdir, UploadConfig.get_NewStudyName())
          try:
            RemoteUploader.get(os.path.join(remotestudydir, "ProtocolDeviations"), remoteProtocolDeviationsFile)
          except:
            pass
          if os.path.exists(remoteProtocolDeviationsFile):
            # trust remote version over local one
            oldProtocolDeviationsFile = remoteProtocolDeviationsFile
        if os.path.exists(oldProtocolDeviationsFile):
          opdfile = open(oldProtocolDeviationsFile, "r")
          oldProtocolDeviations = opdfile.readlines()
          opdfile.close()
      pdfile = open(os.path.join(UploadConfig.get_CurStudyDir(), "ProtocolDeviations"), "w")
      pdfile.write("This file lists potential deviations in this study from the\n'" + UploadConfig.m_ProjectName.get_value() + "' protocol as identified by the data upload script\nusing the schematron file:\n '" + protocolfile + "'\n(plus other non-protocol-specific issues flagged as errors)\n")
      if len(oldProtocolDeviations) > 0:
        pdfile.write(",- FROM EARLIER UPLOAD:\n")
        pdfile.writelines(map(lambda x: "| " + x, oldProtocolDeviations))
        pdfile.write("FROM UPDATED UPLOAD:\n")
      pdfile.writelines(map((lambda x: x[0]), protocolErrs))
      pdfile.close()
    
    del schematron

    ####
    # Copy upload XML file(s)
    ####
    shutil.copy(XMLFile, os.path.join(UploadConfig.get_CurStudyDir(), "Upload-orig.xml"))
    # strip out <skip/> imperatives inserted by hooks
    XPathRemove('/FIPS/series/skip', XMLDoc)
    SaveXML(os.path.join(UploadConfig.get_CurStudyDir(), "Upload-updated.xml"), XMLDoc)

    logging.info('################################################################################')
    logging.info('# Copying Local Dir Space to Standard Dir Space' )
    logging.info('################################################################################')
    for seriesind in range(numseries):
      if UploadConfig.get_Skip(seriesind):
        continue
      
      Path=UploadConfig.get_CurStudyDir()
#      if SetSymLink:
#        SymPath=os.path.join(Symroot,UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir(),UploadConfig.get_Study())
      
      #for each series
      logging.info("Local Directory = "+UploadConfig.get_SeriesDirLocal(seriesind))
      TempPath = StagingPaths.get_TempPath(UploadConfig.get_SeriesNameLocal(seriesind))
    
      logging.info("Standard Directory: "+UploadConfig.get_SeriesList(seriesind))
      if UploadConfig.get_FileSystem() != "Local-In-Place":
        targetseriesdir = os.path.join(Path,UploadConfig.get_SeriesList(seriesind))
        if os.path.exists(targetseriesdir):
          shutil.rmtree(targetseriesdir)
        os.makedirs(targetseriesdir)
#        if SetSymLink:
#          if os.path.islink(os.path.join(SymPath,UploadConfig.get_SeriesList(seriesind))):
#            logging.info("Info: "+os.path.join(SymPath,UploadConfig.get_SeriesList(seriesind))+" symbolic link already exists")
#          else:
#            try:
#              os.symlink(os.path.join(Path,UploadConfig.get_SeriesList(seriesind)), os.path.join(SymPath,UploadConfig.get_SeriesList(seriesind)))
#            except OSError:
#              pass
#            logging.info("Making sym link for study directory: "+os.path.join(SymPath,UploadConfig.get_SeriesList(seriesind)))
        
                
      ################################################################################    
      #Copying NIFTI Format Directory Data
      ################################################################################
      try:
        os.makedirs(os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native"))
      except OSError:
        pass
      try:
        os.makedirs(os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001"))
      except OSError:
        pass
      if UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral" or UploadConfig.get_SeriesType(seriesind) == "structural" or UploadConfig.get_SeriesType(seriesind) == "structural-anon" or UploadConfig.get_SeriesType(seriesind) == "segmentation":
        logging.info("Making NIFTI dir: " + os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001",RunConfig.get_StandardDirectoryNameNifti()))
        os.makedirs(os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001",RunConfig.get_StandardDirectoryNameNifti()))

      DirNameNifti=None
      if UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral" or UploadConfig.get_SeriesType(seriesind) == "structural" or UploadConfig.get_SeriesType(seriesind) == "structural-anon" or UploadConfig.get_SeriesType(seriesind) == "segmentation":
        logging.info("Copying files from " + os.path.join(TempPath, "NiftiFormat") + " to " + os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001",RunConfig.get_StandardDirectoryNameNifti()))
        for file in os.listdir(os.path.join(TempPath, "NiftiFormat")):
          shutil.copy(os.path.join(TempPath, "NiftiFormat", file), os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001",RunConfig.get_StandardDirectoryNameNifti()))
        
        logging.info('################################################################################')
        logging.info('# Editing and Changing Nodes in fips-process.xml' )
        logging.info('################################################################################')
        DirNameNifti=os.path.join(Path,UploadConfig.get_SeriesList(seriesind),"Native","Original__0001",RunConfig.get_StandardDirectoryNameNifti())
        if not btestMode and UploadConfig.get_SeriesType(seriesind) == "structural" or UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral" or UploadConfig.get_SeriesType(seriesind) == "structural-anon":
          IsolateSeries(XMLDoc, UploadConfig.get_SeriesList(seriesind), os.path.join(DirNameNifti, "fips-process.xml"))
        
          ChangeNode(os.path.join(DirNameNifti, "fips-process.xml"),
                     os.path.join(DirNameNifti, "fips-process.xml"),
                     "FIPS//series[nameStandard='"+UploadConfig.get_SeriesList(seriesind)+"']//analysislevel//name",
                     RunConfig.get_StandardDirectoryNameNifti())
          
          if SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_EPRIME:
            for edat_ver in os.listdir(DirNameNifti):
              if re.match('.*?((wait_)?ver\d*(_mod)?).*\.edat', edat_ver) == None:
                continue
              AppendXML(os.path.join(DirNameNifti, "fips-process.xml"),
                        os.path.join(DirNameNifti, "temp.xml"),
                        [
                         ( "FIPS//series[nameStandard='"+UploadConfig.get_SeriesList(seriesind)+"']",
                           "eprime_ver",
                           edat_ver,
                           0
                         )
                        ])
              os.rename(os.path.join(DirNameNifti, "temp.xml"), os.path.join(DirNameNifti, "fips-process.xml"))
          
        if UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral":
          logging.info("################################################################################")
          logging.info("#Adding sliceorder and skipInitialVols to image XML wrapper file")
          logging.info("################################################################################")
          params = [];
          params.append(
            ( "/serieslevel/acqProtocol",
              "acqParam",
              UploadConfig.get_SeriesSkipInitialVols(seriesind),
              0,
              "skipInitialVols",
              "varchar" ) )
          params.append(
            ( "/serieslevel/acqProtocol",
              "acqParam",
              UploadConfig.get_SeriesSliceOrder(seriesind),
              1, # only add if it doesn't exist already
              "sliceorder",
              "varchar" ) )
          if UploadConfig.m_VisitDate.get_value() != None:
            params.append(
              ( "/serieslevel/acqProtocol",
                "acqParam",
                UploadConfig.m_VisitDate.get_value(),
                1, # only add if it doesn't exist already
                "scandate",
                "varchar" ) )
          if UploadConfig.get_SeriesTime(seriesind) != None:
            params.append(
              ( "/serieslevel/acqProtocol",
                "acqParam",
                UploadConfig.get_SeriesTime(seriesind),
                1, # only add if it doesn't exist already
                "scantime",
                "varchar" ) )
          AppendXML(os.path.join(TempPath, "NiftiFormat", "ImageWrapper.xml"),
                    os.path.join(DirNameNifti, "ImageWrapper.xml"),
                    params)
          
        elif UploadConfig.get_SeriesType(seriesind) == "structural":
          shutil.copy(os.path.join(TempPath, "NiftiFormat", "ImageWrapper.xml"), os.path.join(DirNameNifti, "ImageWrapper.xml"))
          
      myImageFormat=UploadConfig.get_SeriesImageFormat(seriesind)
      
      Type=None
      if myImageFormat==Constants.m_DATASPEC_DICOM:
        Type="DICOM"
      elif myImageFormat==Constants.m_DATASPEC_PFILE:
        Type="GE_PFILE"
      elif myImageFormat==Constants.m_DATASPEC_FFILE:
        Type="STANFORD_FFILE"
      elif myImageFormat==Constants.m_DATASPEC_UCIRAW:
        Type="UCI_RAW"
      elif myImageFormat==Constants.m_DATASPEC_IOWASIGNA5:
        Type="GE5X_SIGNA"
      elif myImageFormat==Constants.m_DATASPEC_ANALYZE:
        Type="ANALYZE_7.5"
      elif myImageFormat==Constants.m_DATASPEC_MINC:
        Type="MINC"
      elif myImageFormat==Constants.m_DATASPEC_SIGNA5:
        Type="GE5X_SIGNA"
      elif myImageFormat==Constants.m_DATASPEC_EPRIME:
        Type="EPRIME"
      elif myImageFormat==Constants.m_DATASPEC_CIGAL:
        Type="CIGAL"
      elif myImageFormat==Constants.m_DATASPEC_XCEDE:
        Type="XCEDE"
      elif myImageFormat==Constants.m_DATASPEC_BXH:
        Type="BXH"
      elif myImageFormat==Constants.m_DATASPEC_AFNI:
        Type="AFNI"
      elif myImageFormat==Constants.m_DATASPEC_NIFTI:
        Type="NIFTI"
      elif myImageFormat==Constants.m_DATASPEC_UNKNOWN:
        Type="UNKNOWN"
        
      if Type == None:
        logging.error("ERROR: Invalid series image format %s for series %s\n" % (myImageFormat, str(seriesind)))
        sys.exit(1)

      SchedulePath = os.path.join(Path, UploadConfig.get_SeriesList(seriesind), "Analysis", "Original__0001", "StimulusSchedule")
      BehaviorPath = os.path.join(Path, UploadConfig.get_SeriesList(seriesind), "Analysis", "Original__0001", "Behavior")

      if myImageFormat == Constants.m_DATASPEC_ANALYZE or myImageFormat == Constants.m_DATASPEC_NIFTI:
        Path = os.path.join(Path, UploadConfig.get_SeriesList(seriesind), "Native", "Original__0001", "Original_" + Type)
      elif myImageFormat == None:
        pass
      else:
        Path = os.path.join(Path, UploadConfig.get_SeriesList(seriesind), "Native", "Original__0001", Type)

      
      try:
        os.makedirs(Path)
      except OSError:
        pass
     
      if not btestMode and myImageFormat == SiteInfo.get_BehavioralFormat() and myImageFormat == Constants.m_DATASPEC_EPRIME:
        for edat_ver in os.listdir(Path):
          if re.match('.*?((wait_)?ver\d*(_mod)?).*\.edat', edat_ver) == None:
            continue
          logging.info("AppendXML(" + os.path.join(Path, "fips-process.xml") + " " + os.path.join(Path, "temp.xml") + ", FIPS//series[nameStandard='"+UploadConfig.get_SeriesList(seriesind)+"'], eprime_ver, "+edat_ver+", 0)")
          AppendXML(os.path.join(Path, "fips-process.xml"),
                    os.path.join(Path, "temp.xml"),
                    [
                     ( "FIPS//series[nameStandard='"+UploadConfig.get_SeriesList(seriesind)+"']",
                       "eprime_ver", 
                       edat_ver,
                       0
                     )
                    ])
          os.rename(os.path.join(Path, "temp.xml"), os.path.join(Path, "fips-process.xml"))
        
      dir=DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind))
      logging.info("Copying (3," + str(myImageFormat) + "," + UploadConfig.get_SeriesType(seriesind) + ") files from " + dir + " to " + Path)
      for file in ls(dir):
        shutil.copy(file, Path) 

      logging.info("################################################################################")
      logging.info("#Copying XML File For FIPS Pipeline")
      logging.info("################################################################################")
    
      logging.info("IsolateSeries(" + XMLFile + ", " + UploadConfig.get_SeriesList(seriesind) + ", " + os.path.join(Path, "fips-process.xml") + ")")
      if not btestMode:
         IsolateSeries(XMLDoc, UploadConfig.get_SeriesList(seriesind), os.path.join(Path, "fips-process.xml"))
      
      logging.info("################################################################################")
      logging.info("#Copying Behavioral Data")
      logging.info("################################################################################")
    
      logging.info("Prepping to glob behavioral files: " + DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind)))
      mydirs=[]
      myglobs=[]
      if SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_EPRIME:
        myglobs.append(".*\.edat")
        myglobs.append(".*\.txt")
      elif SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_CIGAL:
        myglobs.append("bhv_.*")
        
      logging.info("Globbing behavioral data files: " + DataPaths.get_LocalPath(UploadConfig.get_SeriesDirLocal(seriesind)))

      if btestMode: myglobs=[]

      for (copypath, destname, recursive) in UploadConfig.get_SeriesBehaviorDirs(seriesind):
        logging.info("Will copy " + copypath + " to " + os.path.join(Path, destname))
        if DirNameNifti != None:
          logging.info("Will copy " + copypath + " to " + os.path.join(DirNameNifti, destname))
        (desthead, desttail) = os.path.split(destname)
        if desthead:
          if not os.path.exists(os.path.join(Path, desthead)):
            os.mkdir(os.path.join(Path, desthead))
          if DirNameNifti != None:
            if not os.path.exists(os.path.join(DirNameNifti, desthead)):
              os.mkdir(os.path.join(DirNameNifti, desthead))
        if recursive:
          shutil.copytree(copypath, os.path.join(Path, destname))
          if DirNameNifti != None:
            shutil.copytree(copypath, os.path.join(DirNameNifti, destname))
        else:
          if os.path.isdir(copypath):
            if not os.path.exists(os.path.join(Path, destname)):
              os.mkdir(os.path.join(Path, destname))
          else:
            shutil.copyfile(copypath, os.path.join(Path, destname))
          if DirNameNifti != None:
            if os.path.isdir(copypath):
              if not os.path.exists(os.path.join(DirNameNifti, destname)):
                os.mkdir(os.path.join(DirNameNifti, destname))
            else:
              shutil.copyfile(copypath, os.path.join(DirNameNifti, destname))
            

      if not btestMode and SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_CIGAL and UploadConfig.get_SeriesType(seriesind) != "extra":
        logging.info("Copying stimulus schedule files:")
        for (copypath, destname, recursive) in UploadConfig.get_SeriesBehaviorDirs(seriesind):
          if os.path.isdir(copypath):
            for file in filter(lambda x: re.match("sch_.*", x) != None, os.listdir(copypath)):
              logging.info("File: " + file)
              try:
                os.makedirs(SchedulePath)
              except OSError:
                pass
              if os.path.isdir(os.path.join(copypath, file)):
                shutil.copytree(os.path.join(copypath, file), os.path.join(SchedulePath, file))
              else:
                shutil.copyfile(os.path.join(copypath, file), os.path.join(SchedulePath, file))
        logging.info("Copying behavior files:")
        for (copypath, destname, recursive) in UploadConfig.get_SeriesBehaviorDirs(seriesind):
          if os.path.isdir(copypath):
            copylist = []
            filelist = os.listdir(copypath)
            for file in filelist:
              # match standard identifier (group 1), subject/visit identifier
              # (group 3), and extension (group 4)
              matchobj = re.match("^(cardiac|events|[^_]*_(summary|trials)|pdigm|respir|show)(.*?)((\.[^.]*)?)$", file)
              if matchobj != None:
                # create a standard name using the standard identifier and
                # prepend "cigal_" to all cigal files
                standardname = "cigal_" + matchobj.group(1) + matchobj.group(4)
                copylist.append((os.path.join(copypath,file), standardname))
              if re.match("^sch_.*$", file):
                copylist.append((os.path.join(copypath,file), file))
            try:
              os.makedirs(BehaviorPath)
            except OSError:
              pass
            for (frompath, toname) in copylist:
              logging.info("Copying: " + frompath + " => " + os.path.join(BehaviorPath, toname))
              if os.path.isdir(frompath):
                shutil.copytree(frompath, os.path.join(BehaviorPath, toname))
              else:
                shutil.copyfile(frompath, os.path.join(BehaviorPath, toname))

      if not btestMode and (UploadConfig.get_SeriesType(seriesind) == "functional" or UploadConfig.get_SeriesType(seriesind) == "functional-with-behavioral"):
        # this is for Siemens DICOM files which introduce an errant flip
        oldsliceorder=UploadConfig.get_SeriesSliceOrder(seriesind)
        newsliceorder=oldsliceorder
        if Type == "DICOM" and UploadConfig.get_DICOMFlipZ(seriesind) != 0:
          numslices = string.atoi(BXHInfo.get_Matrix_Z())
          temp2=oldsliceorder.split(',')
          for i in range(len(temp2)):
            temp2[i] = "%d" % ((numslices + 1) - string.atoi(temp2[i]))
          newsliceorder = string.join(temp2, ',')
        params = []
        params.append(
          ( "/serieslevel/acqProtocol",
            "acqParam",
            UploadConfig.get_SeriesSkipInitialVols(seriesind),
            2, # overwrite any existing skipInitialVols
            "skipInitialVols",
            "varchar"
            ) )
        if UploadConfig.m_VisitDate.get_value() != None:
          params.append(
            ( "/serieslevel/acqProtocol",
              "acqParam",
              UploadConfig.m_VisitDate.get_value(),
              1, # only add if it doesn't exist already
              "scandate",
              "varchar" ) )
        if UploadConfig.get_SeriesTime(seriesind) != None:
          params.append(
            ( "/serieslevel/acqProtocol",
              "acqParam",
              UploadConfig.get_SeriesTime(seriesind),
              1, # only add if it doesn't exist already
              "scantime",
              "varchar" ) )
        if oldsliceorder != newsliceorder:
          params.append(
            ( "/serieslevel/acqProtocol",
              "acqParam",
              newsliceorder,
              1, # only add if it doesn't exist already
              "sliceorder",
              "varchar"
              ) )
        try:
          AppendXML(os.path.join(TempPath, "ImageWrapper.xml"),
                    os.path.join(Path, "ImageWrapper.xml"),
                    params)
        except ValueError:
          logging.warn("WARNING: Unable to append XML for " + os.path.join(TempPath, "ImageWrapper.xml") + "; file does not exist.")
          pass
        
        ChangeNode(os.path.join(Path, "fips-process.xml"),
                   os.path.join(Path, "temp.xml"),
                   "FIPS//series[nameStandard='"+UploadConfig.get_SeriesList(seriesind)+"']//analysislevel//name",
                   Type)
        os.rename(os.path.join(Path, "temp.xml"), os.path.join(Path, "fips-process.xml"))
      
      elif not btestMode and UploadConfig.get_SeriesType(seriesind) == "structural":
        logging.info("################################################################################")
        logging.info("#Copying XML File For FIPS Pipeline")
        logging.info("################################################################################")
        
        shutil.copy(os.path.join(TempPath, "ImageWrapper.xml"), os.path.join(Path, "ImageWrapper.xml"))

      # store current upload timestamp for this series
      if not btestMode:
        standardName = UploadConfig.get_SeriesList(seriesind)
        if standardName not in outputTimestampStruct.keys():
          outputTimestampStruct[standardName] = {}
        outputTimestampStruct[standardName]['timeStamp'] = currentUploadTimestamp

  #############################################################################
  # Write out timestamps to local upload
  #############################################################################
  WriteTimestampFile(outputTimestampStruct, os.path.join(UploadConfig.get_CurStudyDir(), "Timestamps.xml"))
  

  ################################################################################
  # Physical upload
  ################################################################################
  
  if SiteInfo.get_dbInstance() != None:
    ret=Rollback(os.path.join(UploadConfig.get_TargetDir(),UploadConfig.get_Project()), UploadConfig, RemotePaths, Database, SiteInfo, RemoteUploader)
    if ret != 0:
      logging.error("################################################################################" )
      if UploadConfig.get_FileSystem() == "Remote" or UploadConfig.get_FileSystem() == "Local-Remote":
        logging.error("#         db and remote cleanup failed" )
      else:
        logging.error("#         db cleanup failed" )
      logging.error("################################################################################" )
      return 3

  ret=RemoteHumanTransfer(os.path.join(UploadConfig.get_TargetDir(),UploadConfig.get_Project()), SiteInfo, StagingPaths, UploadConfig, RemotePaths, Database, RunConfig, RemoteUploader)
  if ret == 0: 
    if CorrectConfig == 0:
      logging.info("################################################################################")
      logging.info("# Cleaning up temporary data")
      logging.info("################################################################################")
      shutil.rmtree(StagingPaths.get_TempRoot())
    logging.info("Upload finished successfully!")
  else:
    logging.error("Sorry uploading failed+" "+some files in "+StagingPaths.get_TempRoot()+" might give you information for debugging")
  if UploadConfig.get_FileSystem() == "Remote":
    shutil.rmtree(os.path.join(UploadConfig.get_TargetDir(),UploadConfig.get_Project(),UploadConfig.get_SubjectDir(),UploadConfig.get_VisitDir(),UploadConfig.get_Study()))

  logging.info("ALL DONE!")
  logging.info(str(datetime.datetime.now()))
  if RemoteUploader != None:
    logging.info("Deleting RemoteUploader...")
    del RemoteUploader
    logging.info(" ...done")
  return 0

################################################################################
# Do the actual remote upload
################################################################################
def RemoteHumanTransfer(Path, SiteInfo, StagingPaths, UploadConfig, RemotePaths, Database, RunConfig, RemoteUploader):

  global OPT_update

  Constants = _Constants()

  numseries = len(UploadConfig.get_SeriesDirLocal())

  ################################################################################
  #checking against HID
  # 3. the VisitName(VisitName) and VisitID
  ################################################################################
  logging.info("Querying the HID for Visit ....")
  
  #take out the leading L
  VID=UploadConfig.m_VisitID.get_value()
  if UploadConfig.m_VisitID.get_value()[0]=='L':
    VID=UploadConfig.m_VisitID.get_value()[1:]
  if VID != UploadConfig.m_VisitID.get_value():
    VID=-999 
  
  SID=UploadConfig.m_StudyID.get_value()
  if SID[0]=='L':
    SID=SID[1:]

  dbStudyID = None
  dbVisitID = None
  dbVisitName = None
  dbStudyName = None

  if SiteInfo.get_dbInstance() == None:
    dbStudyID = UploadConfig.m_StudyID.get_value()
    dbVisitID = UploadConfig.m_VisitID.get_value()
    dbStudyName = UploadConfig.m_StudyName.get_value()
    dbVisitName = UploadConfig.m_VisitName.get_value()
  else:
    query = "select count(*) from nc_expcomponent where nc_experiment_uniqueid = %d" % UploadConfig.get_dbProjectID() + " and subjectid = '"+UploadConfig.m_SubjectID.get_value()+"' and componentid="+VID+" and name ='"+UploadConfig.m_VisitName.get_value()+"'"
    Database.execute(query)
    result = Database.fetchone()
    
    if result[0] != 1:
      logging.info("There are no matching visits in the HID.")
      if UploadConfig.m_VisitID.get_missing():
        logging.info("The VisitID is missing from the XML file")
      elif UploadConfig.m_VisitName.get_missing():
        logging.info("The VisitName is missing from the XML file")
    
      query = "select componentid, name from nc_expcomponent where nc_experiment_uniqueid = %d" % UploadConfig.get_dbProjectID() + " and subjectid = '"+UploadConfig.m_SubjectID.get_value()+"' order by componentid";
      Database.execute(query)
      result = Database.fetchall()
    
      if len(result) == 0:
        logging.error("There are no visits for "+UploadConfig.m_SubjectID.get_value()+" in the HID.")
        return 3
    
      logging.warn("Please select a visit from the HID:")
      listall(result)
      
      number = selectone("Visit", len(result))
      dbVisitID=result[number-1][0]
      dbVisitName=result[number-1][1]
      if string.atoi(UploadConfig.m_VisitID.get_value()) != dbVisitID:
        UploadConfig.m_VisitID.set_value("%04d" % dbVisitID)
      if UploadConfig.m_VisitName.get_value() != dbVisitName:
        UploadConfig.m_VisitName.set_value(dbVisitName)
    
    ################################################################################
    ################################################################################
    #checking against HID
    # 4. the Study and StudyID and Name
    ################################################################################
    ################################################################################
    #  Get studyID
    ################################################################################
    logging.info("Querying the HID for Study ID....")
    
    query="select studyid from nc_expstudy where experimentid = %d" % UploadConfig.get_dbProjectID()+" and subjectid = '"+UploadConfig.m_SubjectID.get_value()+"' and componentId = %d" % UploadConfig.m_VisitID.get_int_value() +" and studyid="+SID+" and name='"+UploadConfig.m_StudyName.get_value()+"'"
    Database.execute(query)
    result = Database.fetchall()
    
    logging.info("VisitID is %d" % UploadConfig.m_VisitID.get_int_value())
    if len(result) == 1:
      dbStudyID="%d" % result[0][0]
    else:
      logging.info("No matching study found in the HID")
      query="select studyid, name from nc_expstudy where experimentid = %d" % UploadConfig.get_dbProjectID() + " and subjectid = '" + UploadConfig.m_SubjectID.get_value() + "' and componentId = %d" % UploadConfig.m_VisitID.get_int_value() + " order by studyid"
      Database.execute(query)
      result = Database.fetchall()
    
      if len(result) == 0:
        logging.error("No matching study found in the HID for the specified visit (" + UploadConfig.m_VisitID.get_int_value() + ") and subject (" + UploadConfig.m_SubjectID.get_value() + ")")
        return 3
    
      logging.warn("Please select one study from the HID")
      listall(result)
      number = selectone("Study ID", len(result))
      dbStudyID="%d" % result[number-1][0]
      logging.warn("the Study ID you selected is "+dbStudyID)
    
    query="select name from nc_expstudy where experimentid = %d" % UploadConfig.get_dbProjectID() + " and subjectid = '" + UploadConfig.m_SubjectID.get_value() + "' and componentId = %d" % UploadConfig.m_VisitID.get_int_value() + " and studyid = "+dbStudyID
    Database.execute(query)
    result = Database.fetchone()
    
    dbStudyName=result[0]  
  
  if UploadConfig.m_StudyID.get_missing():
    logging.info("The Study ID is missing from the XML file")
    UploadConfig.m_StudyID.set_value("%04d" % string.atoi(dbStudyID))
  elif UploadConfig.m_StudyID.get_value() != dbStudyID:
    UploadConfig.m_StudyID.set_value("%04d" % string.atoi(dbStudyID))
  
  if UploadConfig.m_StudyName.get_missing():
    logging.info("The Study Name is missing from the XML file")
    UploadConfig.m_StudyName.set_value(dbStudyName)
  elif UploadConfig.m_StudyName.get_value() != dbStudyName:
    UploadConfig.m_StudyName.set_value(dbStudyName)
  
  
  ################################################################################
  # since the site ids are not in the HID, so if it is not in the
  # XML file, then the only way to get it is to ask the user for it
  if UploadConfig.get_RecruitSiteID() == None:
    logging.warn("Site IDs:")
    for i in Constants.m_InstitutionsOrder:
      logging.warn("          "+Constants.m_Institutions[i]+"         =         %d" % i)
    logging.warn("Please enter the recruiting site ID:")
    RecruitID = string.atoi(sys.stdin.readline()[:-1])
    UploadConfig.set_RecruitSiteID(Constants.m_SiteIDs[RecruitID])
  
  if UploadConfig.get_SiteID() == None:
    logging.warn("Site IDs:")
    for i in Constants.m_InstitutionsOrder:
      logging.warn("          "+Constants.m_Institutions[i]+"         =         %d" % i)
    logging.warn("Please enter the site that acquired this data:")
    UploadConfig.set_SiteID(sys.stdin.readline()[:-1])
  
  
  ################################################################################
  # try to get segment id by the name
  
  segmentID = [None] * numseries
  if SiteInfo.get_dbInstance() != None:
    for i in range(len(UploadConfig.get_SeriesType())):
      if UploadConfig.get_Skip(i):
        continue
      
      if UploadConfig.get_SeriesType(i) == "exclude"  or UploadConfig.get_SeriesType(i) == "extra" :
        continue
  
      if (UploadConfig.get_SeriesList(i) != UploadConfig.get_SeriesSourceData(i)) and (UploadConfig.get_SeriesType(i) != "structural-anon"):
        continue
      
      query="select segmentid from nc_expsegment where nc_experiment_uniqueid = %d" % UploadConfig.get_dbProjectID() + " and subjectid = \'" + UploadConfig.m_SubjectID.get_value() + "\' and componentId = %d" % UploadConfig.m_VisitID.get_int_value() + " and studyid = %d" % UploadConfig.m_StudyID.get_int_value() + " and name = \'" + UploadConfig.get_SeriesSourceData(i) + "\'";
      Database.execute(query)
      result = Database.fetchone()
    
      if len(result) == 0:
        logging.error("The HID doesn't have a segment with name of " + UploadConfig.get_SeriesSourceData(i) + " for subject "+UploadConfig.m_SubjectID.get_value() + " in project " + UploadConfig.m_ProjectID.get_value() + ", visit \"%d" % UploadConfig.m_VisitID.get_int_value() + "\", study %d" % UploadConfig.m_StudyID.get_int_value() + ", please make sure that the name specified in the XML file matches that in HID.")
        sys.exit(3)
      segmentID[i] = result[0]
  else:
    logging.error("No series directories found.")
    sys.exit(1)
  
  
  # Now check if any xml information needs change, or directory name needs change
  logging.info("TempPaths.NewStudyName is " + UploadConfig.get_NewStudyName())
  
  if UploadConfig.get_FileSystem() != "Local-In-Place":
    logging.info("old vs new "+UploadConfig.get_CurProjectDir()+" "+UploadConfig.get_NewProjectDir())
    if UploadConfig.get_CurProjectDir() != UploadConfig.get_NewProjectDir():
      os.rename(UploadConfig.get_CurProjectDir(), UploadConfig.get_NewProjectDir())
  
    logging.info("old vs new "+UploadConfig.get_CurVisitDir()+" "+UploadConfig.get_NewVisitDir())
    if UploadConfig.get_CurVisitDir() != UploadConfig.get_NewVisitDir():
      os.rename(UploadConfig.get_CurVisitDir(), UploadConfig.get_NewVisitDir())
  
    logging.info("study dir old vs new "+UploadConfig.get_CurStudyDir()+" "+UploadConfig.get_NewStudyDir())
    if UploadConfig.get_CurStudyDir() != UploadConfig.get_NewStudyDir():
      os.rename(UploadConfig.get_CurStudyDir(), UploadConfig.get_NewStudyDir())
  

  # Now we need to update the information for every fips-process.xml
  results = []
  if UploadConfig.m_ProjectID.get_changed() or UploadConfig.m_VisitName.get_changed() or UploadConfig.m_StudyName.get_changed() or UploadConfig.m_SubjectID.get_changed() or UploadConfig.m_VisitID.get_changed() or UploadConfig.m_StudyID.get_changed():
    results=find(os.path.join(UploadConfig.get_NewProjectDir(),UploadConfig.get_NewVisitName(),UploadConfig.get_NewStudyName()), "fips*.\.xml")
  for result in results:
    logging.info(result)
    # Change information in each myLine
    
    reader = Sax2.Reader()
    doc = reader.fromUri(result)
  
    if UploadConfig.m_ProjectID.get_changed():
      temp=XPathReplace('/FIPS/project/ID', UploadConfig.m_ProjectID.get_value(), doc)
  
    if UploadConfig.m_VisitName.get_changed():
      temp=XPathReplace('/FIPS/visit/name', UploadConfig.get_NewVisitName(), doc)
    
    if UploadConfig.m_StudyName.get_changed():
      temp=XPathReplace('/FIPS/study/name', UploadConfig.m_StudyName.get_value(), doc)
  
    if UploadConfig.m_SubjectID.get_changed():
      temp=XPathReplace('/FIPS/BIRNID', UploadConfig.m_SubjectID.get_value(), doc)
    
    if UploadConfig.m_VisitID.get_changed():
      temp=XPathReplace('/FIPS/visit/ID', UploadConfig.m_VisitID.get_value(), doc)
    if UploadConfig.m_StudyID.get_changed():
      temp=XPathReplace('/FIPS/study/ID', UploadConfig.m_StudyID.get_value(), doc)
          
    SaveXML(result, doc)

  # before uploading, tar up DICOMs (can be thousands of files)
  if UploadConfig.get_tarDICOM() and UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
    for i in range(len(UploadConfig.get_SeriesType())):
      if UploadConfig.get_Skip(i):
        continue
      
      if UploadConfig.get_SeriesImageFormat(i) == Constants.m_DATASPEC_DICOM:
        tmpnativedir = os.path.join(UploadConfig.get_NewProjectDir(), UploadConfig.get_NewVisitName(), UploadConfig.get_NewStudyName(), UploadConfig.get_SeriesList(i), "Native", "Original__0001");
        cwd = os.getcwd()
        os.chdir(tmpnativedir)
        logging.info("tarring up in " + tmpnativedir)
        [retval, rdata] = run_cmd("tar cf DICOM.tar DICOM")
        # exit status 1 (i.e. retval (1 >> 8) == 256) means files have changed.
        # Sometimes you get this on CIFS file systems, so ignore this.
        # We won't be changing files during the tar anyway.
        if retval != 0 and retval != 256:
          logging.error("ERROR tar-ing up DICOMs!  Return code=" + str(retval))
          return 1
        [retval, rdata] = run_cmd("gzip DICOM.tar")
        if retval != 0:
          logging.error("ERROR running gzip!")
          return 1
        shutil.rmtree("DICOM")
        os.chdir(cwd)

  if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
    logging.info("Uploading....")
    remotebasedir = RemotePaths.get_RemotePath()
    remotesubjdir = os.path.join(remotebasedir, UploadConfig.m_SubjectID.get_value())
    remotevisitdir = os.path.join(remotesubjdir, UploadConfig.get_NewVisitName())
    remotestudydir = os.path.join(remotevisitdir, UploadConfig.get_NewStudyName())
    try:
      RemoteUploader.mkdir(remotebasedir)
    except StandardError, e:
      logging.info("Creating directory '" + remotebasedir + "' failed (if it already existed, then this is OK)")
    try:
      RemoteUploader.mkdir(remotesubjdir)
    except StandardError, e:
      logging.info("Creating directory '" + remotesubjdir + "' failed (if it already existed, then this is OK)")
    try:
      RemoteUploader.mkdir(remotevisitdir)
    except StandardError, e:
      logging.info("Creating directory '" + remotevisitdir + "' failed (if it already existed, then this is OK)")
    try:
      RemoteUploader.mkdir(remotestudydir)
    except StandardError, e:
      if OPT_update:
        logging.info("Creating directory '" + remotestudydir + "' failed (if it already existed, then this is OK)")
      else:
        raise e
    RemoteUploader.addDirMeta(remotevisitdir, "UploadStatus", "InProgress")
    localstudydir = os.path.join(UploadConfig.get_NewProjectDir(), UploadConfig.get_NewVisitName(), UploadConfig.get_NewStudyName())
    if OPT_update:
      for i in range(numseries):
        if UploadConfig.get_Skip(i):
          continue
        localseriesdir = os.path.join(localstudydir, UploadConfig.get_SeriesList(i))
        RemoteUploader.put(localseriesdir, remotestudydir)
      for filename in os.listdir(localstudydir):
        fullfilepath = os.path.join(localstudydir, filename)
        if os.path.isdir(fullfilepath):
          continue
        RemoteUploader.put(fullfilepath, remotestudydir)
    else:
      RemoteUploader.put(localstudydir, remotevisitdir)

    # Need to remove the excluded Ser directory , which means to check the 
    # the SeriesType[].  If it is structural-anon, then we remove them from the Remote using ( Srm -r testtemp ).
    # First, we form the directory/collection path
    for i in range(numseries):
      if UploadConfig.get_Skip(i):
        continue
      
      if UploadConfig.get_SeriesType(i) == "exclude":
        tempCollection=os.path.join(remotestudydir,UploadConfig.get_SeriesList(i))
        RemoteUploader.rmdirtree(tempCollection)

    projectgroupname = UploadConfig.get_RemoteProjectName()
    sitegroupname = UploadConfig.get_LocalRemoteProjectName(SiteInfo)
    RemoteUploader.addAccessToGroup(remotesubjdir, projectgroupname, "a")
    RemoteUploader.addAccessToGroupRecursive(remotevisitdir, projectgroupname, "r")
    RemoteUploader.addAccessToGroupRecursive(remotestudydir, sitegroupname, "a")
    RemoteUploader.updateDirMeta(remotevisitdir, "UploadStatus", "Complete")

  # Now verify listings of uploaded (or local) directories and add stuff we
  # find to the database
  
  curDir = os.path.join(UploadConfig.m_SubjectID.get_value(),UploadConfig.get_NewVisitName(),UploadConfig.get_NewStudyName())
  tempRootCollection = None
  if UploadConfig.get_FileSystem() == "Local" or UploadConfig.get_FileSystem() == "Local-In-Place":
    tempRootCollection = UploadConfig.get_NewStudyDir()
  else:
    tempRootCollection = remotestudydir
    
  logging.info("tempRootCollection is "+tempRootCollection)
    
  rawDataHeaders = [
    "dataID",
    "TABLE_ID",
    "dataURI",
    "ScannerID",
    "isBad",
    "segmentID",
    "VisitID",
    "ProjectID",
    "SubjectID"
  ]
  
  derivedDataHeaders = [
    "dataID",
    "TABLE_ID",
    "dataURI",
    "isBad",
    "segmentID",
    "VisitID",
    "ProjectID",
    "SubjectID"
  ]
  
  researchDataHeaders = [
    "dataID",
    "TABLE_ID",
    "dataURI",
    "isBad"
  ]
  
  rawData = []
  derivedData = []
  researchData = []
  
  if SiteInfo.get_dbInstance() != None:
    # IBO
    query = None
    if SiteInfo.get_dbType() == 'postgres':
       query = "select get_table_id('nc_rawdata'), get_table_id('nc_deriveddata'), get_table_id('nc_researchdata')"
    else:
       query="select get_table_id('nc_rawdata'), get_table_id('nc_deriveddata'), get_table_id('nc_researchdata') from dual"
    Database.execute(query)
    TableIDs=Database.fetchone()
  
    researchTableID=TableIDs[2]
    derivedTableID=TableIDs[1]
    rawTableID=TableIDs[0]
  
  for i in range(len(UploadConfig.get_SeriesType())):
    if UploadConfig.get_Skip(i):
      continue
    
    # Excluded series do nothing
    if UploadConfig.get_SeriesType(i) == "exclude":
        continue
  
    if SiteInfo.get_dbInstance() != None:
      curSegmentID = segmentID[i]
    else:
      curSegmentID = None
    logging.info("SeriesName is "+UploadConfig.get_SeriesList(i)+" , VisitID is "+UploadConfig.m_VisitID.get_value()+" , PID is " + UploadConfig.m_ProjectID.get_value()+", SubjectID is "+UploadConfig.m_SubjectID.get_value()+", StudyID is " + UploadConfig.m_StudyID.get_value())
    if curSegmentID != None:
      logging.info("segmentID is %d" % curSegmentID)
    isBad=0
    if SiteInfo.get_dbType() == 'postgres': 
	isBad = 'false'
  
    tempLocal=os.path.join(UploadConfig.get_NewStudyDir(),UploadConfig.get_SeriesList(i),"Native","Original__0001")
    tempCollection=os.path.join(tempRootCollection,UploadConfig.get_SeriesList(i),"Native","Original__0001")
    logging.info("tempLocal="+tempLocal)
    logging.info("tempCollection="+tempCollection+" ("+tempRootCollection+")")
   
    ################################################################################ORIGINAL__0001 Collection and below################################################################################      
    if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
      RemoteUploader.addAccessToGroup(os.path.join(tempRootCollection, UploadConfig.get_SeriesList(i)), projectgroupname, "a")
      RemoteUploader.addAccessToGroupRecursive(tempCollection, projectgroupname, "r")
      RemoteUploader.addAccessToGroupRecursive(tempCollection, sitegroupname, "a")
      RemoteUploader.chdir(tempCollection)
    else:
      os.chdir(tempCollection)
      
    # And now for every directory under Original__0001
    logging.info("Adding directories from "+tempLocal+":")
    for direc in os.listdir(tempLocal):
        if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
          if UploadConfig.get_tarDICOM() and direc == 'DICOM.tar.gz':
            # was tarred up -- skip
            continue
          else:
            RemoteUploader.chdir(direc) 
        else:
          curDirec = os.path.join(tempLocal, direc)
          if os.path.isdir(curDirec):
            os.chdir(curDirec)
          else:
            continue
        remote_directory = ""
        if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
          remote_directory=RemoteUploader.pwd()
          logging.info("current Remote directory is "+remote_directory)
        else:
          remote_directory = os.getcwd()
          logging.info("current directory is "+remote_directory)
        # Store the collection for imaging data
        if UploadConfig.get_SeriesType(i) == "extra" and curSegmentID != None:
          # This collection contains extra stuff 
          rawData.append((rawTableID, remote_directory, SiteInfo.get_ScannerID(), isBad, curSegmentID, UploadConfig.m_VisitID.get_int_value(), UploadConfig.get_dbProjectID(), UploadConfig.m_SubjectID.get_value()))
        else:
          if SiteInfo.get_dbInstance() != None:
            if UploadConfig.get_SeriesList(i) != UploadConfig.get_SeriesSourceData(i) and UploadConfig.get_SeriesType(i) != "structural-anon":
  
              SplitData = UploadConfig.get_SeriesSourceData(i).split(":")
              for j in range(len(UploadConfig.get_SeriesSourceID(i))):
                if UploadConfig.get_Skip(j):
                  continue
                sSegmentID = UploadConfig.get_SeriesSourceID(i)[j]
                query = "select count(*) from nc_expsegment where nc_experiment_uniqueid = " + UploadConfig.m_ProjectID.get_value()
                query += " and subjectid = '"
                query += UploadConfig.m_SubjectID.get_value()
                query += "' and componentId = " + UploadConfig.m_VisitID.get_value()
                query += " and studyid = " + UploadConfig.m_StudyID.get_value() + " and name = '" + SplitData[j]
                if sSegmentID == None:
                  query += "' and name =" + UploadConfig.get_SeriesList(i)
                else:
                  query += "' and segmentid =" + sSegmentID
                Database.execute(query)
                results = Database.fetchall()
                if len(results) != 1:
                  logging.warn("\n ########################    Warning:    ############################")
                  logging.warn("The source series, <" + sSegmentID + ", " + SplitData[j] + ">,  is not in HID,")
                  logging.warn("so the data collection " + remote_directory + " is not linked to that series in HID")
                  logging.warn("If you want stored it to HID, please modify the xml file and reupload")
                  logging.warn("###################################################################\n")
                  continue
                else:
                  rawData.append((derivedTableID, remote_directory, SiteInfo.get_ScannerID(), isBad, sSegmentID, UploadConfig.m_VisitID.get_int_value(), UploadConfig.get_dbProjectID(), UploadConfig.m_SubjectID.get_value()))
            elif curSegmentID != None:
              rawData.append((rawTableID, remote_directory, SiteInfo.get_ScannerID(), isBad, curSegmentID, UploadConfig.m_VisitID.get_int_value(), UploadConfig.get_dbProjectID(), UploadConfig.m_SubjectID.get_value()))

        if UploadConfig.get_SeriesType(i) == "functional" or UploadConfig.get_SeriesType(i) == "functional-with-behavioral" or UploadConfig.get_SeriesType(i) == "behavioral":
          # Only get behavioral files
          myglob=None
          if SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_EPRIME:
            myglob = "*.edat"
          elif SiteInfo.get_BehavioralFormat() == Constants.m_DATASPEC_CIGAL:
            myglob = "bhv_*/*"
          if myglob != None:
            names = []
            if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
              names=RemoteUploader.ls(myglob)
            else:
              names=glob.glob(myglob)
            if SiteInfo.get_dbInstance() != None and curSegmentID != None:
              for remoteName in names:
                if remoteName == '': continue
                remoteName = os.path.join(remote_directory, remoteName)
                logging.info(' Adding "' + remoteName + '" to nc_rawData\n')
                rawData.append((rawTableID, remoteName, SiteInfo.get_BehavioralEquipmentID(), isBad, curSegmentID, UploadConfig.m_VisitID.get_int_value(), UploadConfig.get_dbProjectID(), UploadConfig.m_SubjectID.get_value()))
        if UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place":
          RemoteUploader.chdir(tempCollection)  
        else:
          os.chdir(tempCollection)

  sys.stdout.flush()

  # do RLS mapping
  RLSexception = None
  if RemoteUploader != None and UploadConfig.get_FileSystem() != "Local" and UploadConfig.get_FileSystem() != "Local-In-Place" and SiteInfo.get_dbInstance() != None and len(rawData) + len(derivedData) + len(researchData) > 0:
    query = "select value from nc_conf_params where name='globus.rls.server.url.primary'"
    Database.execute(query)
    rlsurl = Database.fetchone()[0]
    try:
      RLS = _RLS(rlsurl, SiteInfo, OPT_debug)
      for lname in map(lambda x: x[1], rawData) + map(lambda x: x[1], researchData):
        # check to see if this logical name exists in RLS
        if len(RLS.query(lname)) > 0:
          # if it does, nuke all entries for this logical name -- this will be
          # the new canonical entry
          RLS.deleteall(lname);
        # add the mapping to RLS
        tpath = RemoteUploader.getScheme() + "://" + RemoteUploader.getHost() + lname
        RLS.create(lname, tpath)
    except Exception, e:
      logging.error('Adding elements to RLS failed with the following exception:' + str(e) + '\n' + ''.join(traceback.format_stack()))
      logging.error('Will continue... (but see message at end for instructions to re-populate RLS)')
      RLSexception = e
  
  if SiteInfo.get_dbInstance() != None:
    query="select count(*) from nc_researchdata"
    Database.execute(query)
    dataCount1=Database.fetchone()[0]
    
    if len(researchData) != 0:
      query = None
      if SiteInfo.get_dbType() == 'postgres':
         query = "insert into nc_researchdata " + \
              "select nextval('uid_seq'), :1, get_database_user_id('" + SiteInfo.get_User() + "'), now(), " + \
              "get_database_user_id('" + SiteInfo.get_User() + "'), 'na', :2, true, " + \
              "'UNKNOWN', '9999', :3"
      else:
         query = "insert into nc_researchData " + \
              "select uid_seq.nextval, :1, get_database_user_id('" + SiteInfo.get_User() + "'), sysdate, " + \
              "get_database_user_id('" + SiteInfo.get_User() + "'), 'na', :2, 1, " + \
              "'UNKNOWN', '9999', :3 from dual"
      Database.executemany(query, researchData)
    
    query="select count(*) from nc_researchdata"
    Database.execute(query)
    dataCount2=Database.fetchone()[0]
    
    query="select count(*) from nc_rawdata"
    Database.execute(query)
    rdataCount1=Database.fetchone()[0]
    
    if len(rawData) != 0:
      #IBO
      query = None
      if SiteInfo.get_dbType() == 'postgres':
         query = "insert into nc_rawdata " + \
              "select nextval('uid_seq'), :1, owner, now(), owner, 'nc_rawdata', segmentid, " + \
              "componentid, nc_experiment_uniqueid, subjectid, :2,  true, " + \
              "protocolversion, protocolid, :3, 'UNKNOWN', '9999', :4  " + \
              "from nc_expsegment where segmentid = :5 and componentid = :6 " + \
              "and nc_experiment_uniqueid = :7 and subjectid = :8"
      else:
         query = "insert into nc_rawdata " + \
              "select uid_seq.nextval, :1, OWNER, sysdate, OWNER, 'nc_rawdata', SEGMENTID, " + \
              "COMPONENTID, NC_EXPERIMENT_UNIQUEID, SUBJECTID, :2,  1, " + \
              "PROTOCOLVERSION, PROTOCOLID, :3, 'UNKNOWN', '9999', :4  " + \
              "from nc_expSegment where SEGMENTID = :5 and COMPONENTID = :6 " + \
              "and NC_EXPERIMENT_UNIQUEID = :7 and SUBJECTID = :8"
      Database.executemany(query, rawData)
  
    query="select count(*) from nc_rawdata"
    Database.execute(query)
    rdataCount2=Database.fetchone()[0]
    
    query="select count(*) from nc_deriveddata"
    Database.execute(query)
    ddataCount1=Database.fetchone()[0]
    
    if len(derivedData) != 0:
      # IBO
      query = None
      if SiteInfo.get_dbType() == 'postgres':
         query = "insert into nc_derivedData " + \
              "select nextval('uid_seq'), :1, owner, now(), owner, 'nc_deriveddata', :2, 0, " + \
              "'UNKNOWN', '9999', segmentid, componentid, nc_experiment_uniqueid, " + \
              "subjectid, :3 from nc_expsegment where segmentid = :4 " + \
              "and componentid = :5 and nc_experiment_uniqueid = :6 and subjectid = :7"
      else:
         query = "insert into nc_derivedData " + \
              "select uid_seq.nextval, :1, OWNER, sysdate, OWNER, 'nc_deriveddata', :2, 0, " + \
              "'UNKNOWN', '9999', SEGMENTID, COMPONENTID, NC_EXPERIMENT_UNIQUEID, " + \
              "SUBJECTID, :3 from nc_expSegment where SEGMENTID = :4 " + \
              "and COMPONENTID = :5 and NC_EXPERIMENT_UNIQUEID = :6 and SUBJECTID = :7"
      Database.executemany(query, derivedData)
    
    query="select count(*) from nc_deriveddata"
    Database.execute(query)
    ddataCount2=Database.fetchone()[0]
    
    logging.info("HID has been updated:")
    logging.info("Number of records in nc_researchData has changed to " + str(dataCount2) + " from " + str(dataCount1))
    logging.info("Number of records in    nc_rawData   has changed to " + str(rdataCount2) + " from " + str(rdataCount1))
    logging.info("Number of records in  nc_derivedData has changed to " + str(ddataCount2) + " from " + str(ddataCount1))

  if RLSexception != None:
    logging.warning('Warning: Adding objects to RLS failed.  You may be able to re-do the RLS portion by running the following command: UpdateRLS %s %s %s' % (UploadConfig.get_RemoteProjectName(), UploadConfig.m_SubjectID.get_value(), UploadConfig.m_VisitID.get_int_value()))
  
  Clean(UploadConfig, 0)
  return 0
  
  
################################################################################
# Rollback a previous upload attempt from the database
################################################################################
def Rollback(Path, UploadConfig, RemotePaths, Database, SiteInfo, RemoteUploader):
  global OPT_update
  global OPT_debug
  
  # Remote study directory
  vc = os.path.join(RemotePaths.get_RemotePath(), UploadConfig.m_SubjectID.get_value(), UploadConfig.m_VisitName.get_value() + "__" + UploadConfig.get_SiteID() + "__" + UploadConfig.m_VisitID.get_value(), UploadConfig.m_StudyName.get_value() + "__" + UploadConfig.m_StudyID.get_value())

  if OPT_update:
    numseries = len(UploadConfig.get_SeriesDirLocal())
    for seriesind in range(numseries):
      if UploadConfig.get_Skip(seriesind):
        continue
      sd = os.path.join(vc,UploadConfig.get_SeriesList(seriesind))
      sp = sd + '%'
      dbdelete("nc_deriveddata", sp, Database)
      dbdelete("nc_rawdata", sp, Database)
      dbdelete("nc_researchdata", sp, Database)
  else:
    # Study directory pattern
    vp = vc + "%"
    dbdelete("nc_deriveddata", vp, Database)
    dbdelete("nc_rawdata", vp, Database)
    dbdelete("nc_researchdata", vp, Database)
  
  logging.info("################################################################################")
  logging.info("#               end of db cleanup" )
  logging.info("################################################################################")
  
  ################################################################################
  #                  Cleanup local dir
  ################################################################################
  
  logging.info("################################################################################")
  logging.info("#               end of local cleanup" )
  logging.info("################################################################################")
  
  ################################################################################
  #                  Cleanup remote
  ################################################################################
  
  if UploadConfig.get_FileSystem() == "Local-Remote" or UploadConfig.get_FileSystem() == "Remote":
    if OPT_update:
      numseries = len(UploadConfig.get_SeriesDirLocal())
      for seriesind in range(numseries):
        if UploadConfig.get_Skip(seriesind):
          continue
        sd = os.path.join(vc,UploadConfig.get_SeriesList(seriesind))
        logging.info("Starting to remove "+sd)
        try:
          RemoteUploader.rmdirtree(sd)
        except Exception, e:
          logging.info("Removing '" + sd + "' failed (probably because it doesn't exist, which is OK)\n")
    else:
      logging.info("Starting to remove "+vc)
      try:
        RemoteUploader.rmdirtree(vc)
      except Exception, e:
        logging.info("Removing '" + vc + "' failed (probably because it doesn't exist, which is OK)\n")
  
    # remove RLS mappings
    query = "select value from nc_conf_params where name='globus.rls.server.url.primary'"
    Database.execute(query)
    rlsurl = Database.fetchone()[0]
    RLS = _RLS(rlsurl, SiteInfo, OPT_debug)
    if OPT_update:
      numseries = len(UploadConfig.get_SeriesDirLocal())
      for seriesind in range(numseries):
        if UploadConfig.get_Skip(seriesind):
          continue
        sd = os.path.join(vc,UploadConfig.get_SeriesList(seriesind))
        logging.info("Removing " + sd + " from RLS")
        entries = RLS.wildquery(sd + "%")
        for entry in entries:
          [lname, ppath] = entry.split('\t')
          RLS.deleteall(lname)
    else:
      logging.info("Removing " + vc + " from RLS")
      entries = RLS.wildquery(vc + "%")
      for entry in entries:
        [lname, ppath] = entry.split('\t')
        RLS.deleteall(lname)

  logging.info("################################################################################")
  logging.info("#               end of remote cleanup" )
  logging.info("################################################################################")
  return 0


################################################################################
# Do the non-human phantom processing and upload
################################################################################
def NonHumanTransfer(SiteInfo, RemotePaths, NativeDir, KSpaceDir):
  logging.info("NativeDir =" + NativeDir)
  BXHInfo = _BXHInfo()
  Species = "Human"
  RootDir = os.getcwd()

  RemoteUploader = RemotePaths.get_RemoteUploader(SiteInfo, OPT_debug);

  if SiteInfo.get_PhantomType() == None:
    logging.warn("Please Enter the Phantom Type (1=Geometric, 2=Stability): ")
    str = sys.stdin.readline()[:-1]
    if str == "1":
      str = "geometric"
    elif str == "2":
      str = "stability"
    SiteInfo.set_PhantomType(str)
  
  ImageFormats = []
  if SiteInfo.get_PhantomType() == "geometric":
    PhantomType = "MRI-Geometric"
    ImageFormats = [SiteInfo.get_GeometricNativeFormat()]
  elif SiteInfo.get_PhantomType() == "stability":
    PhantomType = "MRI-Stability"
    ImageFormats = [SiteInfo.get_StabilityNativeFormat()]
  else:
    logging.warn("Error, unknown phantom type.")
    sys.exit(1)

  NIFTIDir = os.path.join(RootDir, "NiftiFormat")
  TempNativeDir = os.path.join(RootDir, "Native")
  try:
    shutil.rmtree(TempNativeDir)
  except OSError:
    pass
  os.makedirs(TempNativeDir)

  for strippedFile in os.listdir(NativeDir):
    fullFile = os.path.join(NativeDir, strippedFile)
    try:
      os.symlink(fullFile, os.path.join(TempNativeDir, strippedFile))
    except OSError:
      logging.error("ERROR: Unable to create symlink" + str([fullFile, os.path.join(TempNativeDir, strippedFile)]))
      sys.exit(1)
      
    # just in case we linked in files/dirs we're going to create anyway
    try:
      os.unlink(os.path.join(TempNativeDir, "ImageWrapper.xml"))
    except OSError:
      pass

  if len(ImageFormats) == 0 or ImageFormats[0] == _Constants.m_DATASPEC_AUTODETECT:
    [ImageFormats,sdd] = AutoDetectImageDirectory(TempNativeDir)

  if _Constants.m_DATASPEC_DICOM in ImageFormats:
    try:
      os.unlink("Native.tar.gz")
    except OSError:
      pass

  CreateBXHFiles(ImageFormats, TempNativeDir, TempNativeDir, BXHInfo, SiteInfo, NIFTIDir, True)
  
  ###############################################
  # BIRN QA Analysis
  ##############################################
  
  logging.info('##############################################')
  logging.info("# Performing BIRN-QA Analysis")
  logging.info('##############################################')
  if SiteInfo.get_QAAnalysis() == None:
    logging.info("Perform BIRN-QA Analysis? (y/n):")
    SiteInfo.set_QAAnalysis(sys.stdin.readline()[:-1])
  
  if SiteInfo.get_QAAnalysis():
    logging.info("Performing BIRN-QA analysis....")
    (ret, output) = run_cmd("GloverQAWrapper.tcsh " + ("%04d" % SiteInfo.get_AcquisitionSiteID()) + " " + SiteInfo.get_AcquisitionSite() + " " + os.path.join(RootDir, "NiftiFormat", "ImageWrapper.xml"))
    if (ret & 0xff):
      logging.error("ERROR: GloverQAWrapper.tcsh exited with signal %d!  Output of script:\n%s\n" % (ret & 127, output))
      return 1
    elif (ret >> 8) != 0:
      logging.error("ERROR: GloverQAWrapper.tcsh returned with exit code %d!  Output of script:\n%s\n" % (ret >> 8, output))
      return 1
  else:
    logging.info("No BIRN-QA analysis performed....")
      
  ################################################
  #Remote Collection Creation and Uploading
  ################################################
  
  logging.info('##############################################')
  logging.info('#Preparing Remote Collection Hierarchy....      #')
  logging.info('##############################################'    )
  
  Institution = SiteInfo.get_Institution()
  InstitutionID = SiteInfo.get_InstitutionID()
  AcquisitionSite = SiteInfo.get_AcquisitionSite()
  AcquisitionSiteID = SiteInfo.get_AcquisitionSiteID()
  RemotePath = ""
  if RemotePaths.get_RemoteBaseURL():
    RemotePath = os.path.join(RemotePaths.get_RemoteBaseURL()[2], AcquisitionSite)
  elif RemotePaths.get_ProjectPath():
    RemotePath = os.path.join("/home/Projects", RemotePaths.get_ProjectPath(), "Data", PhantomType, AcquisitionSite)
  else:
    RemotePath = os.path.join("/home/BIRN", Species, "Phantom", PhantomType, AcquisitionSite)
  remotestudydir = os.path.join(RemotePath, BXHInfo.get_StudyDate_StudyTime())
  remoteseriesdir = os.path.join(remotestudydir, BXHInfo.get_SeriesNumber())
  projectgroupname = RemotePaths.get_RemoteProjectName()
  sitegroupname = RemotePaths.get_LocalRemoteProjectName(SiteInfo)
  try:
    RemoteUploader.rmdirtree(remoteseriesdir)
  except StandardError, e:
    logging.info("Removing '" + remoteseriesdir + "' failed (if it didn't exist, then this is OK)\n")
  try:
    RemoteUploader.mkdir(remotestudydir)
  except StandardError, e:
    logging.info("Creating directory '" + remotestudydir + "' failed (if it already existed, then this is OK)")
  RemoteUploader.addAccessToGroup(RemotePath, projectgroupname, "r")
  RemoteUploader.addAccessToGroup(RemotePath, sitegroupname, "a")
  if RemotePaths.get_ProjectPath():
    if SiteInfo.get_PhantomType() == "stability":
      RemoteUploader.addAccessToGroup(RemotePaths.get_RemoteProjectName() + " groups " + os.path.join("/home/Projects/", RemotePaths.get_ProjectPath(), "Data") + PhantomType, projectgroupname, "r")
  RemoteUploader.addAccessToGroup(remotestudydir, projectgroupname, "r")
  RemoteUploader.addAccessToGroup(remotestudydir, sitegroupname, "a")
  RemoteUploader.addDirMeta(remotestudydir, "StudyDate", BXHInfo.get_StudyDate())
  RemoteUploader.addDirMeta(remotestudydir, "StudyTime", BXHInfo.get_StudyTime())
  RemoteUploader.addDirMeta(remotestudydir, "Modality", BXHInfo.get_Modality())
  RemoteUploader.addDirMeta(remotestudydir, "Manufacturer", BXHInfo.get_Manufacturer())
  RemoteUploader.addDirMeta(remotestudydir, "Model", BXHInfo.get_Model())
  RemoteUploader.addDirMeta(remotestudydir, "MagneticFieldStrength", BXHInfo.get_MagneticFieldStrength())

  remoteseriesdir = os.path.join(remotestudydir, BXHInfo.get_SeriesNumber())

  RemoteUploader.mkdir(remoteseriesdir)
  RemoteUploader.addAccessToGroup(remotestudydir, projectgroupname, "r")
  RemoteUploader.addAccessToGroup(remotestudydir, sitegroupname, "a")

  # set upload status to "in progress" for top-level directory
  # make sure this is the first metadata item so we can update it
  # later with index 0
  RemoteUploader.addDirMeta(remoteseriesdir, "UploadStatus", "InProgress")

  RemoteUploader.addDirMeta(remoteseriesdir, "SeriesNumber", BXHInfo.get_SeriesNumber())
  RemoteUploader.addDirMeta(remoteseriesdir, "ScanningSequence", BXHInfo.get_ScanningSequence())
  RemoteUploader.addDirMeta(remoteseriesdir, "SequenceVariant", BXHInfo.get_SequenceVariant())
  RemoteUploader.addDirMeta(remoteseriesdir, "AcqPlane", BXHInfo.get_AcqPlane())
  RemoteUploader.addDirMeta(remoteseriesdir, "Matrix_X", BXHInfo.get_Matrix_X())
  RemoteUploader.addDirMeta(remoteseriesdir, "Matrix_Y", BXHInfo.get_Matrix_Y())
  RemoteUploader.addDirMeta(remoteseriesdir, "Matrix_Z", BXHInfo.get_Matrix_Z())
  RemoteUploader.addDirMeta(remoteseriesdir, "PixelSpacing_X", BXHInfo.get_PixelSpacing_X())
  RemoteUploader.addDirMeta(remoteseriesdir, "PixelSpacing_Y", BXHInfo.get_PixelSpacing_Y())
  RemoteUploader.addDirMeta(remoteseriesdir, "PixelSpacing_Z", BXHInfo.get_PixelSpacing_Z())
  RemoteUploader.addDirMeta(remoteseriesdir, "FlipAngle", BXHInfo.get_FlipAngle())
  
  if SiteInfo.get_QAAnalysis():
    logging.info('####################################################')
    logging.info('# Uploading BIRN-QA data directory...    #' )
    logging.info('####################################################')
    remotederiveddir = os.path.join(remoteseriesdir, "Derived_Data")
    RemoteUploader.put(os.path.join(RootDir, "NiftiFormat", "Derived_Data"), remoteseriesdir)
    RemoteUploader.addAccessToGroupRecursive(remotederiveddir, projectgroupname, "r")
    RemoteUploader.updateDirMeta(remotederiveddir, "UploadStatus", "Complete")
    try:
      shutil.rmtree(os.path.join(RootDir, "NiftiFormat", "Derived_Data"))
    except OSError:
      pass
    
  logging.info('####################################################')
  logging.info('# Uploading Native data directory...    #' )
  logging.info('####################################################')
  if ImageFormats[0] == _Constants.m_DATASPEC_DICOM:
    # before uploading, tar up DICOMs (can be thousands of files)
    cwd = os.getcwd()
    os.chdir(RootDir)
    logging.info("tarring up in " + RootDir)
    [retval, rdata] = run_cmd("tar chf Native.tar Native")
    if retval != 0:
      logging.error("ERROR tar-ing up DICOMs!")
      return 1
    [retval, rdata] = run_cmd("gzip Native.tar")
    if retval != 0:
      logging.error("ERROR running gzip!")
      return 1
    RemoteUploader.put("Native.tar.gz", os.path.join(remoteseriesdir, "Native.tar.gz"))
    os.unlink("Native.tar.gz")
    os.chdir(cwd)
  else:
    remotenativedir = os.path.join(remoteseriesdir, "Native")
    RemoteUploader.put(TempNativeDir, remoteseriesdir)
    RemoteUploader.updateDirMeta(remotenativedir, "UploadStatus", "Complete")

  if KSpaceDir:
    logging.info('####################################################')
    logging.info('# Uploading KSPACE data directory...    #' )
    logging.info('####################################################')
    remotekspacedir = os.path.join(remoteseriesdir, "Kspace")
    RemoteUploader.put(KSPACEDIR, remotekspacedir)
    RemoteUploader.addAccessToGroupRecursive(remotekspacedir, projectgroupname, "r")
    RemoteUploader.updateDirMeta(remotekspacedir, "UploadStatus", "Complete")

  logging.info('####################################################')
  logging.info('# Uploading NIFTI data directory...  #' )
  logging.info('####################################################')
  remoteniftidir = os.path.join(remoteseriesdir, "NiftiFormat")
  RemoteUploader.put(NIFTIDir, remoteseriesdir)
  RemoteUploader.addAccessToGroupRecursive(remoteniftidir, projectgroupname, "r")
  RemoteUploader.addDirMeta(remoteniftidir, "UploadStatus", "Complete")

  # set upload status to "complete" for top-level directory
  RemoteUploader.updateDirMeta(remoteseriesdir, "UploadStatus", "Complete")

  RemoteUploader.addAccessToGroupRecursive(remotestudydir, sitegroupname, "a")
  logging.info("Cleaning up....")
  shutil.rmtree(TempNativeDir)
  shutil.rmtree(NIFTIDir)
  logging.info("All Done!")

  logging.info("Deleting RemoteUploader...")
  del RemoteUploader
  logging.info(" ...done")

  return 0


################################################################################
# Read some basic config options for our load.
################################################################################
def LoadProtocolFile(path, SiteInfo, RemotePaths):
  # Load the xml file
  reader = Sax2.Reader()
  doc = reader.fromUri(path)
  
  try:
    RemotePaths.set_RemoteBaseURLstr(XPathEval('/Protocol/RemoteBaseURL', doc)[0].encode('ascii'))
  except IndexError:
    pass
  try:
    SiteInfo.set_QAAnalysis(XPathEval('/Protocol/QAAnalysis', doc)[0].encode('ascii'))
  except IndexError:
    pass
  try:
    SiteInfo.set_PhantomType(XPathEval('/Protocol/PhantomType', doc)[0].encode('ascii'))
  except IndexError:
    pass
  try:
    RemotePaths.set_ProjectPath(XPathEval('/Protocol/Project', doc)[0].encode('ascii'))
  except IndexError:
    pass
  try:
    SiteInfo.set_PhantomID(XPathEval('/Protocol/PhantomID', doc)[0].encode('ascii'))
  except IndexError:
    pass


################################################################################
# Spawn the upload of nonhuman phantoms.
################################################################################
def do_NonHumanUpload(args):
  global LogFile
  global tmpmemloghandler
  
  RemotePaths = _RemotePaths()
  SiteInfo = _SiteInfo()
  StagingPaths = _StagingPaths()
  Database = _Database()
  Database.set_SiteInfo(SiteInfo)

  ScannerDir=os.getcwd()
  # Make sure there are scanners in there
  if which("Upload.py") != None:
    ScannerDir=os.path.dirname(which("Upload.py"))
  
  SeriesDir=os.getcwd()

  # Parse commandline arguments
  if len(args) < 2:
    logging.error("Nonhuman uploads require 1 or 2 additional arguments, <NativeDir> [<KSpaceDir>]")
    return 1

  try:
    logging.info(args)
    i = 0
    while i < len(args):
      if args[i] == "-scanner" or args[i] == "--scanner":
        SiteInfo.set_ScannerConfigFile(args[i + 1])
        logging.info("Scanner found: " + SiteInfo.get_ScannerConfigFile())
        args = args[:i] + args[i + 2:]
      elif args[i] == "-remoteBaseURL" or args[i] == "--remoteBaseURL":
        RemotePaths.set_RemoteBaseURLstr(args[i + 1])
        logging.info("Upload path found: " + urlparse.urlunparse(RemotePaths.get_RemoteBaseURL()))
        args = args[:i] + args[i + 2:]
      elif args[i] == "-phantomType" or args[i] == "--phantomType":
        SiteInfo.set_PhantomType(args[i + 1])
        logging.info("Phantom type found: " + SiteInfo.get_PhantomType())
        args = args[:i] + args[i + 2:]
      elif args[i] == "-qaAnalysis" or args[i] == "--qaAnalysis":
        SiteInfo.set_QAAnalysis(args[i + 1])
        logging.info("QA Analysis flag found: " + str(SiteInfo.get_QAAnalysis()))
        args = args[:i] + args[i + 2:]
      elif args[i] == "-project" or args[i] == "--project":
        RemotePaths.set_ProjectPath(args[i + 1])
        args = args[:i] + args[i + 2:]
      elif args[i] == "-phantomID" or args[i] == "--phantomID":
        SiteInfo.set_PhantomID(args[i + 1])
        args = args[:i] + args[i + 2:]
      else:
        i += 1
  except:
    Usage()
    return 1
    
  # Check to make sure configuration files are found in the directory and files exist.
  ScannerList=filter(lambda x: re.match("^Configure_.*$", x)!=None, os.listdir(ScannerDir))
  if len(ScannerList) == 0:
    logging.warn("Please enter the path to the UploadScript installation")
    logging.warn("directory (to avoid this add the directory to your PATH env variable):")
    ScannerDir=sys.stdin.readline()[:-1]
    if os.path.isdir(ScannerDir)!=True:
      logging.error("Sorry, "+ScannerDir+" does not exist.  Exiting.")
      sys.exit(1)
  
  if SiteInfo.get_ScannerConfigFile() == None:
    # Renew the list in case the dir changed.
    ScannerList=filter(lambda x: re.match("^Configure_.*$", x)!=None, os.listdir(ScannerDir))
    
    # If more than one entry, ask the user.  If just one, choose it.
    ConfigFile=""
    ScannerFiles = []	# All of the Configure* files created by the user
    if len(ScannerList) == 1:
      ConfigFile=os.path.join(dirname(ScannerDir),ScannerList[0])
    elif len(ScannerList) > 1:
      for i in range(len(ScannerList)):
        ScannerFiles.append(re.match("^.*?Configure_(.*)$", ScannerList[i]).group(1))
    
      ScannerNum = select_scanner(ScannerFiles) 
      if ScannerNum == None:
        sys.exit(1)
      while ScannerNum <= 0 or ScannerNum > len(ScannerList):
        logging.warn("Invalid selection (%d).  Please choose an entry between %d and %d." % (ScannerNum, 1, len(ScannerList)))
        ScannerNum = select_scanner(ScannerFiles)

      if ScannerNum == None:
        sys.exit(1)
      
      ConfigFile = ScannerList[ScannerNum - 1]

    SiteInfo.set_ScannerConfigFile(ConfigFile)
    
  ImportConfig(StagingPaths, SiteInfo)
  
  LogFile = os.path.join(StagingPaths.get_TempDIR(), ("_tempUpload.LOG.%d.%d" % (os.getpid(), time.mktime(time.localtime()))))
  filehandler = logging.FileHandler(LogFile, 'w')
  tmpmemloghandler.setTarget(filehandler)
  tmpmemloghandler.close()
  rootlogger = logging.getLogger('')
  rootlogger.removeHandler(tmpmemloghandler)
  rootlogger.addHandler(filehandler)

  LoadProtocolFile(args[1], SiteInfo, RemotePaths)
  NativeDir = args[2]
  KSpaceDir = None
  if len(args) == 4:
    KSpaceDir = args[3]

  NonHumanTransfer(SiteInfo, RemotePaths, NativeDir, KSpaceDir)


################################################################################
# Spawn the upload of human phantoms.
################################################################################
def do_HumanUpload(args):
  global OPT_debug
  global OPT_update
  global OPT_nolinks
  global LogFile
  global tmpmemloghandler
  OPT_update = False
  OPT_nolinks = False

  UploadConfig = _UploadConfig()
  StagingPaths = _StagingPaths()
  RemotePaths = _RemotePaths()
  FIPSPaths = _FIPSPaths()
  SiteInfo = _SiteInfo()
  BXHInfo = _BXHInfo()
  DataPaths = _DataPaths()
  RunConfig = _RunConfig()
  Database = _Database()
  Constants = _Constants()

  behaviorroots = []
  
  # Parse command line arguments
  try:
    i = 0
    while i < len(args):
      if args[i] == "-scanner" or args[i] == "--scanner":
        SiteInfo.set_ScannerConfigFile(args[i + 1])
        logging.info("Scanner found: " + SiteInfo.get_ScannerConfigFile())
        args = args[:i] + args[i+2:]
      elif args[i] == "-remoteBaseURL" or args[i] == "--remoteBaseURL":
        RemotePaths.set_RemoteBaseURLstr(args[i + 1])
        logging.info("Upload path found: " + urlparse.urlunparse(RemotePaths.get_RemoteBaseURL()))
        args = args[:i] + args[i + 2:]
      elif args[i] == "-noTarDICOM" or args[i] == "--noTarDICOM":
        UploadConfig.set_tarDICOM(0)
        logging.info("Will not tar up DICOM directories.")
      elif args[i] == "-update" or args[i] == "--update":
        OPT_update = True
        args = args[:i] + args[i+1:]
      elif args[i] == "-nolinks" or args[i] == "--nolinks":
        OPT_nolinks = True
        args = args[:i] + args[i+1:]
      elif args[i] == "-behaviordir" or args[i] == "--behaviordir":
        behaviorroot = args[i + 1]
        if not os.path.exists(behaviorroot):
          redirect = ResolvePseudoLink(behaviorroot)
          if redirect != None:
            logging.info("Resolving pseudo-link " + behaviorroot + " to " + redirect)
            behaviorroot = redirect
        behaviorroots.append(behaviorroot)
        args = args[:i] + args[i+2:]
      else:
        i += 1
  except:
    Usage()
    return 1

  if len(args) != 4 and len(args) != 5:
    logging.error("You should have 3 or 4 arguments, <Root_ImageDir> <XML Config File> <TargetDir> [<Local_Remote_Home_BIRN>]")
    Usage()
    return 1

  Database.set_SiteInfo(SiteInfo)
  DataPaths.set_DataRoot(os.path.realpath(args[1]))
  SeriesConfigFile=os.path.realpath(args[2])
  DataPaths.set_BehaviorRoots(behaviorroots)

  ScannerDir=os.getcwd()
  # Make sure there are scanners in there
  if which("Upload.py") != None:
    ScannerDir=dirname(which("Upload.py"))
  
  SeriesDir=os.getcwd()
  
  # Check to make sure configuration files are found in the directory and files exist.
  ScannerList=filter(lambda x: re.match("^Configure_.*$", x)!=None, os.listdir(ScannerDir))
  if len(ScannerList) == 0:
    logging.warn("Please enter the path to the UploadScript installation")
    logging.warn("directory (to avoid this add the directory to your PATH env variable):")
    ScannerDir=sys.stdin.readline()[:-1]
    if os.path.isdir(ScannerDir)!=True:
      logging.error("Sorry, "+ScannerDir+" does not exist.  Exiting.")
      sys.exit(1)
  
  # Renew the list in case the dir changed.
  ScannerList=filter(lambda x: re.match("^Configure_.*$", x)!=None, os.listdir(ScannerDir))
  
  # If more than one entry, ask the user.  If just one, choose it.
  ConfigFile=""
  ScannerFiles = []	# All of the Configure* files created by the user
  if SiteInfo.get_ScannerConfigFile() == None:
    if len(ScannerList) == 1:
      ConfigFile=os.path.join(dirname(ScannerDir), ScannerList[0])
    elif len(ScannerList) > 1:
      for i in range(len(ScannerList)):
        ScannerFiles.append(re.match("^.*?Configure_(.*)$", ScannerList[i]).group(1))
    
      ScannerNum=select_scanner(ScannerFiles)
      if ScannerNum == None:
        sys.exit(1)
      while ScannerNum <= 0 or ScannerNum > len(ScannerList):
        logging.warn("Invalid selection (%d).  Please choose an entry between %d and %d." % (ScannerNum, 1, len(ScannerList)))
        ScannerNum=select_scanner(ScannerFiles)
      if ScannerNum == None:
        sys.exit(1)
      ConfigFile=ScannerList[ScannerNum - 1]
    else:
      logging.error("No scanner files found in " + SiteInfo.get_ScannerDir())
      logging.error("Please run <Install.sh> and install your scanner configuration files or add the path to your installation to your PATH env variable!")
      sys.exit(1)
    SiteInfo.set_ScannerConfigFile(ConfigFile)

  ImportConfig(StagingPaths, SiteInfo)
  assert SiteInfo.get_dbType() != None, "dbType was None!"

  LogFile = os.path.join(StagingPaths.get_TempDIR(), ("_tempUpload.LOG.%d.%d" % (os.getpid(), time.mktime(time.localtime()))))
  filehandler = logging.FileHandler(LogFile, 'w')
  tmpmemloghandler.setTarget(filehandler)
  tmpmemloghandler.close()
  rootlogger = logging.getLogger('')
  rootlogger.removeHandler(tmpmemloghandler)
  rootlogger.addHandler(filehandler)

  UploadConfig.set_SiteID(Constants.m_SiteIDs[SiteInfo.get_AcquisitionSiteID()])
  
  # Make sure that the SeriesConfigFile is valid.
  if os.path.isfile(SeriesConfigFile) != True:
   logging.error("ERROR: cannot find XML config file: "+SeriesConfigFile)
   sys.exit(1)
  
  SeriesConfigFile = os.path.realpath(SeriesConfigFile)
  logging.info("XML Config File is "+SeriesConfigFile)
  
  # Load the xml file
  reader = Sax2.Reader()
  SeriesConfigDoc = reader.fromUri(SeriesConfigFile)
  ParseXMLFileSystem(SeriesConfigDoc, UploadConfig)
  del SeriesConfigDoc

  if UploadConfig.get_FileSystem() == "Local-In-Place":
    UploadConfig.set_TargetDir(os.path.realpath(args[1]))
    RemotePaths.set_LocalRemoteHome(os.path.realpath(args[1]))
  else:
    UploadConfig.set_TargetDir(os.path.realpath(args[3]))
    if len(args) == 4:
      RemotePaths.set_LocalRemoteHome(os.path.realpath(args[3]))
    else:
      RemotePaths.set_LocalRemoteHome(os.path.realpath(args[4]))
  
  #Optionally load the fips config, then start building up the temp directory for upload
  if os.path.isdir(args[1]):
    if os.path.isfile("~/.fipsrc"):
      ImportFIPSConfig("~/.fipsrc", FIPSPaths, RemotePaths)
      logging.info('~/.fipsrc Found! Sourcing for Local_Remote_Home_BIRN')
  
    LocalHumanTransfer(ConfigFile, SeriesConfigFile, SiteInfo, Database, UploadConfig, DataPaths, StagingPaths, RemotePaths, Constants, BXHInfo, RunConfig)
  else:
    logging.info("Invalid Root_ImageDir: "+args[1]+"  Directory does not exist!!")
    sys.exit(1)
  
################################################################################
# BEGIN SCRIPT
################################################################################

# globals
OPT_debug = False
OPT_profile = False
LogFile = None
# make standard output unbuffered
sys.stdout = os.fdopen(sys.stdout.fileno(), 'w', 0)

# create logging facility, and attach stdout and stderr
class MinMaxLogFilter(logging.Filter):
  def __init__(self, minlevel=None, maxlevel=None):
    logging.Filter.__init__(self)
    self.minlevel = minlevel
    self.maxlevel = maxlevel
  def filter(self, record):
    if self.minlevel <= record.levelno <= self.maxlevel:
      return 1
stdoutfilter = MinMaxLogFilter(logging.INFO, logging.INFO)
stderrfilter = MinMaxLogFilter(logging.WARNING, logging.CRITICAL)
stdouthandler = logging.StreamHandler(sys.stdout)
stderrhandler = logging.StreamHandler(sys.stderr)
stdouthandler.addFilter(stdoutfilter)
stderrhandler.addFilter(stderrfilter)
rootlogger = logging.getLogger('')
rootlogger.addHandler(stdouthandler)
rootlogger.addHandler(stderrhandler)
# this will store log entries until we can find a file to write them
tmpmemloghandler = logging.handlers.MemoryHandler(1)
rootlogger.addHandler(tmpmemloghandler)
rootlogger.setLevel(logging.DEBUG)

startTime = datetime.datetime.now()
logging.info("Log started: " + str(startTime))
version = ''
releasefile = 'SVNVERSION'
if 'ScriptInstallDir' in os.environ:
  releasefile = os.path.join(os.environ['ScriptInstallDir'], releasefile)
if not os.path.exists(releasefile):
  releasefile = 'RELEASE_VERSION'
  if 'ScriptInstallDir' in os.environ:
    releasefile = os.path.join(os.environ['ScriptInstallDir'], releasefile)
if os.path.exists(releasefile):  
  f = file(releasefile)
  while True:
    line = f.readline()
    if len(line) == 0:
      break;
    version = version + line
  logging.info("SVN version: " + version)

cmdlinestr = ''
for arg in sys.argv:
  newarg = re.sub('([|&;()<> \t])', '\\\\1', arg)
  cmdlinestr = cmdlinestr + " " + newarg
logging.info("Command line:" + cmdlinestr)

# Check commandline args
NonHuman = False
if len(sys.argv) == 1:
  Usage()
  sys.exit(1)

args = sys.argv[:]
try:
  i = 1
  while i < len(args):
    if args[i] == "-nonhuman" or args[i] == "--nonhuman":
      NonHuman = True
      args = args[:i] + args[i+1:]
    elif args[i] == "-debug" or args[i] == "--debug":
      OPT_debug = True
      args = args[:i] + args[i+1:]
    elif args[i] == "-profile" or args[i] == "--profile":
      OPT_profile = True
      args = args[:i] + args[i+1:]
    else:
      i += 1
except:
  Usage()
  sys.exit(1)

if OPT_debug:
  stdouthandler.removeFilter(stdoutfilter)
  stdoutfilter = MinMaxLogFilter(logging.DEBUG, logging.INFO)
  stdouthandler.addFilter(stdoutfilter)

if OPT_profile:
  import guppy.heapy.RM
  from guppy import hpy
  hp=hpy()

syse = None
# use a separate external try/finally statement because try/except/finally
# is reported not to work before version 2.5
try:
  try:
    if NonHuman:
      sys.exit(do_NonHumanUpload(args))
    else:
      sys.exit(do_HumanUpload(args))
  except SystemExit, e:
    syse = e
  except Exception, e:
    logging.critical(''.join(traceback.format_exception(sys.exc_type, sys.exc_value, sys.exc_traceback)))
    syse = SystemExit(1)
    sys.exit(1)
finally:
  endTime = datetime.datetime.now()
  logging.info("Log ended: " + str(endTime))
  logging.info("Total execution time: " + str(endTime - startTime))
  if LogFile != None:
    logging.info("Finished.  See output log here: " + LogFile)
  # wait for all other threads to exit
  logging.info("Enumerating all threads...")
  for t in threading.enumerate(): 
    if (t != threading.currentThread() and t.isAlive() and not t.isDaemon()):
      logging.info(" existing thread " + str(t.getName()))
      #t.join() 
  logging.info(" ...done.")
  if syse != None:
    sys.exit(syse.code)
