/* @author Syam Gadde
 * @version $Id$
 */

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.Iterator;
import java.util.Stack;
import java.util.Vector;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.globus.ftp.FileInfo;
import org.globus.ftp.GridFTPClient;
import org.globus.ftp.GridFTPSession;
import org.globus.ftp.MlsxEntry;
import org.globus.ftp.exception.ServerException;
import org.globus.myproxy.MyProxy;

import org.gridforum.jgss.ExtendedGSSCredential;

import org.ietf.jgss.GSSCredential;

import org.birncommunity.gridftp.tar.TarToPipeRunnable;
import org.birncommunity.gridftp.tar.TwoPartyTarClientFactoryI;
import org.birncommunity.gridftp.tar.TwoPartyTarClientFactoryImpl;
import org.birncommunity.gridftp.tar.TwoPartyTarTransfer;
import org.birncommunity.gridftp.tar.UntarFromPipeRunnable;
import org.birncommunity.gridftp.tar.impl.compress.CompressTar;
import org.birncommunity.gridftp.tar.impl.compress.CompressUntar;
import org.birncommunity.sample.proxy.LocalCredentialHelper;

public class GridFTPCommand
{
    protected boolean debug = false;
    protected int debugIndent = -1;
    protected GridFTPClient ftpclient = null;
    protected boolean skipnextreset = false;
    protected String myproxyhost = null;
    protected int myproxyport = -1;
    protected String gridftphost = null;
    protected int gridftpport = -1;
    protected String myproxyuser = null;
    protected String myproxypwd = null;
    protected GSSCredential credential = null;
    protected String curdir = null;

    protected class GridFTPCommandException
	extends Exception
    {
	static final long serialVersionUID = 0;
	public GridFTPCommandException(String msg, Throwable cause)
	{
	    super(msg, cause);
	}
    }
    public GridFTPCommand()
    {
    }

    public void debugOn()
    {
        this.debug = true;
    }
    public void debugOff()
    {
        this.debug = false;
    }
    public void debugEnter()
    {
        this.debugIndent += 1;
    }
    public void debugExit()
    {
        this.debugIndent -= 1;
    }
    public void debugMsg(String msg)
    {
        if (!this.debug) {
            return;
        }
        System.err.print("DEBUG: ");
        int i;
        for (i = 0; i < this.debugIndent; i++) {
            System.err.print("| ");
        }
        System.err.println(msg);
    }
    public void connectParams(String myproxyhost, int myproxyport, String gridftphost, int gridftpport, String myproxyuser, String myproxypwd)
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::connectParams('" + myproxyhost + "', " + myproxyport + ", '" + gridftphost + "', " + gridftpport + ", '" + myproxyuser + "', <password not printed>)");
	this.myproxyhost = myproxyhost;
	this.myproxyport = myproxyport;
	this.gridftphost = gridftphost;
	this.gridftpport = gridftpport;
	this.myproxyuser = myproxyuser;
	this.myproxypwd = myproxypwd;
        this.debugExit();
    }
    protected void getCredential()
	throws GridFTPCommandException
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::getCredential()");
	if (this.myproxyuser == null || this.myproxypwd == null) {
	    try {
		this.credential = new LocalCredentialHelper().getDefaultCredential();
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error getting local credential (maybe you need to renew your credential, or otherwise, set the MYPROXY_USER and MYPROXY_PWD environment variables)", e);
	    }
	} else {
	    MyProxy myproxy = new MyProxy(this.myproxyhost, this.myproxyport);
	    try {
		this.credential = myproxy.get(this.myproxyuser, this.myproxypwd, 0);
	    } catch (Exception e) {
	        try {
		    this.credential = new LocalCredentialHelper().getDefaultCredential();
	        } catch (Exception e2) {
		    this.debugExit();
		    throw new GridFTPCommandException("Error connecting or authenticating to myproxy", e);
	        }
	    }
	}
        this.debugExit();
    }
    protected void gridFTPConnect()
	throws GridFTPCommandException
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::gridFTPConnect()");
	try {
	    ftpclient = new GridFTPClient(this.gridftphost, this.gridftpport);
	    try {
		ftpclient.authenticate(this.credential);
	    } catch (ServerException e) {
		this.debugMsg("| authentication failed.");
		this.debugMsg("| attempting to get new credential...");
		this.getCredential();
		this.debugMsg("| ...and reauthenticating...");
		ftpclient.authenticate(this.credential);
	    }
            ftpclient.setType(GridFTPSession.TYPE_IMAGE);
	    ftpclient.setMode(GridFTPSession.MODE_STREAM);
	} catch (Exception e) {
            this.debugExit();
	    throw new GridFTPCommandException("Error connecting or authenticating to gridftp", e);
	}
	if (this.curdir != null) {
	    this.cmd_cd(this.curdir);
	}
        this.debugExit();
    }
    public void connect()
	throws GridFTPCommandException
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::connect()");
	this.getCredential();
	this.gridFTPConnect();
        this.debugExit();
    }
    public void disconnect()
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::disconnect()");
	if (this.ftpclient != null) {
	    try {
		this.ftpclient.close();
		this.ftpclient = null;
	    } catch (Exception e) {
		this.debugMsg("| Error disconnecting: " + e.toString());
	    }
	}
        this.debugExit();
    }
    public void checkConnection()
	throws GridFTPCommandException
    {
	this.debugEnter();
	this.debugMsg("GridFTPCommand::checkConnection()");
	if (this.ftpclient == null) {
	    this.gridFTPConnect();
	} else {
	    try {
		this.ftpclient.quote("NOOP");
	    } catch (Exception e) {
		this.debugMsg("| connection failed: " + e.toString());
		this.debugMsg("| attempting to reconnect...");
		this.disconnect();
		this.gridFTPConnect();
	    }
	}
	this.debugExit();
    }
    protected void skipnextreset()
    {
	this.skipnextreset = true;
    }
    public void reset_passive()
	throws GridFTPCommandException
    {
        this.debugEnter();
        this.debugMsg("GridFTPCommand::reset_passive()");
	if (this.skipnextreset) {
	    this.debugMsg("`- skipping this reset");
	    this.skipnextreset = false;
	    this.debugExit();
	    return;
	}
	try {
	    ftpclient.setPassive();
	} catch (Exception e) {
	    throw new GridFTPCommandException("Error calling GridFTPClient::setLocalPassive()", e);
	}
	try {
	    ftpclient.setLocalActive();
	} catch (Exception e) {
	    throw new GridFTPCommandException("Error calling GridFTPClient::setActive()", e);
	}
        this.debugExit();
    }
    protected void reportSuccess(String message)
    {
        this.debugEnter();
	System.out.println("SUCCESS");
	if (message != null) {
	    System.out.println("START MESSAGE");
	    System.out.print(message);
	    if (message.length() > 0 &&
		message.charAt(message.length()-1) != '\n') {
		System.out.print("\n");
	    }
	    System.out.println("END MESSAGE");
	}
        this.debugExit();
    }
    protected void reportFailure(String message)
    {
        this.debugEnter();
	System.out.println("FAILURE");
	if (message != null) {
	    System.out.println("START MESSAGE");
	    System.out.print(message);
	    if (message.charAt(message.length()-1) != '\n') {
		System.out.print("\n");
	    }
	    System.out.println("END MESSAGE");
	}
        this.debugExit();
    }
    protected void reportFailure(Exception e)
    {
        this.debugEnter();
	System.out.println("FAILURE");
	System.out.println("START MESSAGE");
	e.printStackTrace(System.out);
	System.out.println("END MESSAGE");
        this.debugExit();
    }
    protected MlsxEntry doMlst(String path)
	throws GridFTPCommandException
    {
	MlsxEntry info = null;
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::doMlst('" + path + "')");
	    //this.reset_passive();
	    this.debugMsg("|GridFTPClient::mlst('" + path + "')");
	    try {
		info = ftpclient.mlst(path);
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error listing file '" + path + "'", e);
	    }
	} finally {
	    this.debugExit();
	}
	return info;
    }
    protected Vector doMlsd()
	throws GridFTPCommandException
    {
	Vector filelist = null;
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::doMlsd()");
	    this.reset_passive();
	    this.debugMsg("|GridFTPClient::mlsd(null)");
	    try {
		filelist = ftpclient.mlsd(null);
	    } catch (Exception e) {
		//this.skipnextreset(); // didn't use data channel?
		this.debugExit();
		throw new GridFTPCommandException("Error listing directory", e);
	    }
	} finally {
	    this.debugExit();
	}
	return filelist;
    }
    protected Vector doMlsd(String path)
	throws GridFTPCommandException
    {
	Vector filelist = null;
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::doMlsd('" + path + "')");
	    this.reset_passive();
	    this.debugMsg("|GridFTPClient::mlsd('" + path + "')");
	    try {
		if (path.equals("")) {
		    filelist = ftpclient.mlsd(null);
		} else {
		    filelist = ftpclient.mlsd(path);
		}
	    } catch (Exception e) {
		//this.skipnextreset(); // didn't use data channel?
		this.debugExit();
		throw new GridFTPCommandException("Error listing directory '" + path + "'", e);
	    }
	} finally {
	    this.debugExit();
	}
	return filelist;
    }
    public void run(String [] cmdandargs)
    {
        this.debugEnter();
	if (this.debug) {
	    String msg = new String("");
	    msg = msg + "GridFTPCommand::run([";
	    int i;
	    for (i = 0; i < cmdandargs.length; i++) {
		msg = msg + "'" + cmdandargs[i] + "'";
		if (i < cmdandargs.length - 1) {
		    msg = msg + ", ";
		}
	    }
	    msg = msg + "])";
	    this.debugMsg(msg);
	}
	try {
	    this.checkConnection();
	    String successStr = null;
	    String command = cmdandargs[0];
	    if (command.equals("ls")) {
		if (cmdandargs.length == 1) {
		    successStr = this.cmd_ls(null);
		} else if (cmdandargs.length == 2) {
		    if (cmdandargs[1].equals("-l")) {
			successStr = this.cmd_ls(null, true);
		    } else {
			successStr = this.cmd_ls(cmdandargs[1]);
		    }
		} else if (cmdandargs.length == 3) {
		    if (cmdandargs[1].equals("-l")) {
			successStr = this.cmd_ls(cmdandargs[2], true);
		    } else if (cmdandargs[2].equals("-l")) {
			successStr = this.cmd_ls(cmdandargs[1], true);
		    } else {
			this.reportFailure("Too many arguments to 'ls'!");
			this.debugExit();
			return;
		    }
		} else {
		    this.reportFailure("Too many arguments to 'ls'!");
		    this.debugExit();
		    return;
		}
	    } else if (command.equals("pwd")) {
		if (cmdandargs.length != 1) {
		    this.reportFailure("Too many arguments to 'pwd'");
		    this.debugExit();
		    return;
		}
		successStr = this.cmd_pwd();
	    } else if (command.equals("cd")) {
		if (cmdandargs.length < 2) {
		    this.reportFailure("Not enough arguments to 'cd'!");
		    this.debugExit();
		    return;
		}
		this.cmd_cd(cmdandargs[1]);
	    } else if (command.equals("mkdir")) {
		if (cmdandargs.length < 2) {
		    this.reportFailure("Not enough arguments to 'mkdir'!");
		    this.debugExit();
		    return;
		}
		this.cmd_mkdir(cmdandargs[1]);
	    } else if (command.equals("rm")) {
		if (cmdandargs.length == 3) {
		    if (cmdandargs[1].equals("-r")) {
			this.cmd_rmdir_recursive(cmdandargs[2]);
		    } else {
			this.reportFailure("rm requires one argument!");
			this.debugExit();
			return;
		    }
		} else if (cmdandargs.length == 2) {
		    this.cmd_rm(cmdandargs[1]);
		} else {
		    this.reportFailure("rm (or rm -r) requires one argument!");
		    this.debugExit();
		    return;
		}
	    } else if (command.equals("rmdir")) {
		if (cmdandargs.length != 2) {
		    this.reportFailure("rmdir requires one argument!");
		    this.debugExit();
		    return;
		}
		this.cmd_rmdir(cmdandargs[1]);
	    } else if (command.equals("put")) {
		if (cmdandargs.length != 2 && cmdandargs.length != 3) {
		    this.reportFailure("put requires one or two arguments!");
		    this.debugExit();
		    return;
		} else if (cmdandargs.length == 2) {
		    this.cmd_put(cmdandargs[1], ".");
		} else {
		    this.cmd_put(cmdandargs[1], cmdandargs[2]);
		}
	    } else if (command.equals("get")) {
		if (cmdandargs.length != 2 && cmdandargs.length != 3) {
		    this.reportFailure("get requires one or two arguments!");
		    this.debugExit();
		    return;
		} else if (cmdandargs.length == 2) {
		    this.cmd_get(cmdandargs[1], ".");
		} else {
		    this.cmd_get(cmdandargs[1], cmdandargs[2]);
		}
	    } else {
		this.reportFailure("Unsupported command '" + command + "'");
		this.debugExit();
		return;
	    }
	    this.reportSuccess(successStr);
	} catch (GridFTPCommandException e) {
	    this.reportFailure(e);
	}
	this.debugExit();
    }

    protected void cmd_cd(String path)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_cd('" + path + "')");
	    this.debugMsg("|GridFTPClient::changeDir('" + path + "')");
	    try {
		ftpclient.changeDir(path);
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error running changeDir command", e);
	    }
	    this.curdir = this.cmd_pwd();
	} finally {
	    this.debugExit();
	}
    }
    protected String cmd_pwd()
	throws GridFTPCommandException
    {
	String retval = null;
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::pwd()");
	    //this.reset_passive();
	    this.debugMsg("|GridFTPClient::getCurrentDir()");
	    try {
		retval = ftpclient.getCurrentDir();
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error running getCurrentDir command", e);
	    }
	} finally {
	    this.debugExit();
	}
	this.curdir = retval;
	return retval;
    }
    protected String cmd_ls(String path)
	throws GridFTPCommandException
    {
	return this.cmd_ls(path, false);
    }
    protected String cmd_ls(String path, boolean dolong)
	throws GridFTPCommandException
    {
	StringBuffer output = new StringBuffer();
        this.debugEnter();
	try {
	    if (path == null) {
		this.debugMsg("GridFTPCommand::cmd_ls(null, dolong=" + dolong + ")");
	    } else {
		this.debugMsg("GridFTPCommand::cmd_ls('" + path + "', dolong=" + dolong + ")");
	    }
	    if (path == null) {
		path = new String("");
	    }

	    Vector filelist = new Vector();

	    boolean showfullpath = false;
	    Stack dirstack = new Stack();
	    Stack pathcompstack = new Stack();
	    Stack wildcardedstack = new Stack();
	    dirstack.push(new String(""));
	    pathcompstack.push(path);
	    wildcardedstack.push(Boolean.FALSE);
	    while (!dirstack.empty()) {
		String dir = (String)dirstack.pop();
		String pathcomp = (String)pathcompstack.pop();
		Boolean wildcarded = (Boolean)wildcardedstack.pop();
		this.debugMsg(" dir='" + dir + "' pathcomp='" + pathcomp + "'");
		// Pre-conditions:
		// - dir is a string containing zero or more components of a path.
		//   These components represent a directory on the remote server.
		// - pathcomp contains the rest of the components of the remote path
		//   and may contain wildcards.
		// - dir does not contain any wildcard characters
		// - If dir is non-empty, it must contain a trailing slash, so that
		//   dir + pathcomp is a valid path (with optional wildcards)
		// If pathcomp does not contain any wildcards, this iteration will
		// list the directory represented by dir + pathcomp and add the
		// entries to filelist.  If pathcomp does contain wildcards, then
		// this iteration will find the files/directories matching up to the
		// first wildcard component of the combined path dir + pathcomp.
		int wildcardindex = -1;
		if ((wildcardindex = pathcomp.indexOf("*")) == -1) {
		    // no more wildcards -- do listing
		    boolean dodir = false;
		    String fullpath = dir + pathcomp;
		    MlsxEntry info = null;
		    if (fullpath.equals("")) {
			dodir = true;
		    } else {
			if (wildcarded) {
			    try {
				info = this.doMlst(fullpath);
			    } catch (Exception e) {
				// skip, no error because this path had a wildcard
				continue;
			    }
			} else {
			    info = this.doMlst(fullpath);
			}
			String infotype = info.get(MlsxEntry.TYPE);
			if (infotype.equals(MlsxEntry.TYPE_DIR)) {
			    dodir = true;
			}
		    }
		    if (dodir) {
			Vector mlsxlist = this.doMlsd(fullpath);
			Iterator iter = mlsxlist.iterator();
			while (iter.hasNext()) {
			    MlsxEntry tmpinfo = ((MlsxEntry)iter.next());
			    String filename = tmpinfo.getFileName();
			    if (filename == null) {
				filename = new String("???");
			    }
			    if (filename.equals(".") || filename.equals("..")) {
				continue;
			    }
			    String outstr = new String("");
			    if (dolong) {
				outstr += tmpinfo.get(MlsxEntry.TYPE) + "\t" + tmpinfo.get(MlsxEntry.PERM) + "\t" + tmpinfo.get(MlsxEntry.SIZE) + "\t" + tmpinfo.get(MlsxEntry.MODIFY) + "\t";
			    }
			    if (showfullpath) {
				outstr += fullpath;
				if (!fullpath.endsWith("/")) {
				    outstr += '/';
				}
				outstr += filename;
			    } else {
				outstr += filename;
			    }
			    filelist.add(outstr);
			}
		    } else {
			if (dolong) {
			    filelist.add(info.get(MlsxEntry.TYPE) + "\t" + info.get(MlsxEntry.PERM) + "\t" + info.get(MlsxEntry.SIZE) + "\t" + info.get(MlsxEntry.MODIFY) + "\t" + fullpath);
			} else {
			    filelist.add(fullpath);
			}
		    }
		} else {
		    // There is a wildcard in pathcomp.
		    wildcarded = Boolean.TRUE;
		    showfullpath = true;

		    // Move any initial non-wildcard path components from pathcomp
		    // to dir
		    if (wildcardindex != 0) {
			// wildcard does not start pathcomp -- search for
			// preceding slash
			int slashindex = pathcomp.lastIndexOf("/", wildcardindex - 1);
			dir = dir + pathcomp.substring(0, slashindex + 1);
			pathcomp = pathcomp.substring(slashindex + 1);
		    }
		    // We will match the listing of "dir" to the pattern
		    // in pathcomp (up to the next slash, if it exists, which
		    // implies we should only match directories at this stage).  
		    boolean matchonlydirs = false;
		    String pattern = pathcomp;
		    pathcomp = new String("");
		    int slashindex = pattern.indexOf("/");
		    if (slashindex != -1) {
			pathcomp = pattern.substring(slashindex + 1);
			pattern = pattern.substring(0, slashindex);
		    }
		    pattern = pattern.replaceAll("([\\[\\].^$+(){}])", "\\$1");
		    pattern = pattern.replaceAll("\\*", ".*");
		    this.debugMsg(" adjusted: dir='" + dir + "' pattern='" + pattern + "' pathcomp='" + pathcomp + "'");
		    Vector mlsxlist = null;
		    if (dir.equals("")) {
			mlsxlist = this.doMlsd();
		    } else {
			try {
			    mlsxlist = this.doMlsd(dir);
			} catch (Exception e) {
			    this.debugMsg("--ignoring this exception:");
			    this.debugMsg(e.toString());
			    continue;
			}
		    }
		    // put matching directories in a stack so we can add it to
		    // dirstack in reverse order, to emulate depth-first search
		    Stack tmpdirlist = new Stack();
		    Iterator iter = mlsxlist.iterator();
		    while (iter.hasNext()) {
			MlsxEntry info = (MlsxEntry)iter.next();
			String infotype = info.get(MlsxEntry.TYPE);
			String filename = info.getFileName();
			if (filename.equals(".") || filename.equals("..")) {
			    continue;
			}
			if (filename.matches(pattern)) {
			    if (infotype.equals(MlsxEntry.TYPE_DIR)) {
				tmpdirlist.push(filename);
			    } else if (!matchonlydirs) {
				String outstr = new String("");
				if (dolong) {
				    outstr += info.get(MlsxEntry.TYPE) + "\t" + info.get(MlsxEntry.PERM) + "\t" + info.get(MlsxEntry.SIZE) + "\t" + info.get(MlsxEntry.MODIFY) + "\t";
				}
				outstr += dir + filename;
				filelist.add(outstr);
			    }
			}
		    }
		    while (!tmpdirlist.empty()) {
			dirstack.push(dir + (String)tmpdirlist.pop() + "/");
			pathcompstack.push(pathcomp);
			wildcardedstack.push(wildcarded);
		    }
		}
	    }

	    Iterator iter = filelist.iterator();
	    while (iter.hasNext()) {
		String filename = (String)iter.next();
		output.append(filename);
		output.append("\n");
	    }
	} finally {
	    this.debugExit();
	}
	return output.toString();
    }
    protected void cmd_mkdir(String path)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_mkdir('" + path + "')");
	    try {
		ftpclient.makeDir(path);
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error running makeDir command", e);
	    }
	} finally {
	    this.debugExit();
	}
    }

    protected void cmd_rm(String path)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_rm('" + path + "')");
	    try {
		ftpclient.deleteFile(path);
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error running deleteFile command", e);
	    }
	} finally {
	    this.debugExit();
	}
    }

    protected void cmd_rmdir(String path)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_rmdir('" + path + "')");
	    try {
		ftpclient.deleteDir(path);
	    } catch (Exception e) {
		this.debugExit();
		throw new GridFTPCommandException("Error running deleteDir command", e);
	    }
	} finally {
	    this.debugExit();
	}
    }

    protected void cmd_rmdir_recursive(String path)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_rmdir_recursive('" + path + "')");
	    Stack rmdirs = new Stack();
	    Stack stack = new Stack();
	    rmdirs.push(path);
	    stack.push(path);
	    while (!stack.empty()) {
		String curpath = (String)stack.pop();
		Vector filelist = this.doMlsd(curpath);
		Stack newdirs = new Stack();
		int i;
		for (i = 0; i < filelist.size(); i++) {
		    MlsxEntry info = (MlsxEntry)filelist.elementAt(i);
		    String newpath = curpath;
		    newpath = newpath + "/" + info.getFileName();
		    if (info.getFileName().equals(".") || info.getFileName().equals("..")) {
			continue;
		    }
		    String infotype = info.get(MlsxEntry.TYPE);
		    if (infotype.equals(MlsxEntry.TYPE_DIR)) {
			newdirs.push(newpath);
			rmdirs.push(newpath);
		    } else if (infotype.equals(MlsxEntry.TYPE_FILE)) {
			this.debugMsg("}-> GridFTPClient::deleteFile('" + newpath + "')");
			try {
			    ftpclient.deleteFile(newpath);
			} catch (Exception e) {
			    this.debugExit();
			    throw new GridFTPCommandException("Error deleting file '" + newpath + "'", e);
			}
		    }
		}
		// we do it this way so directories are removed in the order they
		// are listed from the server
		while (!newdirs.empty()) {
		    stack.push(newdirs.pop());
		}
	    }
	    while (!rmdirs.empty()) {
		String dir = (String)rmdirs.pop();
		this.debugMsg("}-> GridFTPClient::deleteDir('" + dir + "')");
		try {
		    ftpclient.deleteDir(dir);
		} catch (Exception e) {
		    this.debugExit();
		    throw new GridFTPCommandException("Error running deleteDir command", e);
		}
	    }
	} finally {
	    this.debugExit();
	}
    }

    protected void cmd_put(String localpath, String remotepath)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_put('" + localpath + "', '" + remotepath + "')");
	    File localfile = new File(localpath);
	    if (localfile.isDirectory()) {
		boolean fallthrough = false;
		// See if we can use "tar-stream" approach
		ExecutorService executorService = null;
		try {
		    this.disconnect();
		    TwoPartyTarClientFactoryI clientFactory = new TwoPartyTarClientFactoryImpl();
		    TwoPartyTarTransfer tptt = new TwoPartyTarTransfer(clientFactory, this.gridftphost, this.gridftpport, this.credential);
		    if (tptt.isPopenDriverSupported()) {
			this.debugMsg("| Server claims popen support.  Attempting tar-stream upload...");
			PipedOutputStream pipeOut = new PipedOutputStream();
			PipedInputStream pipeIn = new PipedInputStream(pipeOut);
			executorService = Executors.newSingleThreadExecutor();
			executorService.execute(new TarToPipeRunnable(localpath, pipeOut, new CompressTar()));
			this.debugMsg("}-> TwoPartyTarTransfer::uploadTarFromPipe('" + remotepath + "', pipeIn, pipeOut)");
			tptt.uploadTarFromPipe(remotepath, pipeIn, pipeOut);
		    } else {
			fallthrough = true;
		    }
		} catch (Exception e) {
		    throw new GridFTPCommandException("Error attempting tar stream upload", e);
		} finally {
		    if (executorService != null) {
			executorService.shutdownNow();
		    }
		}
		if (!fallthrough) {
		    return;
		}
		// if "tar-stream" not supported, then fall through
		this.debugMsg("| Server does not support popen.  Attempting normal upload...");
		this.checkConnection();
	    }
	    Stack localstack = new Stack();
	    localstack.push(localfile);
	    Stack remotestack = new Stack();
	    //this.reset_passive();
	    this.debugMsg("}-> GridFTPClient::mlst('" + remotepath + "')");
	    try {
		MlsxEntry info = ftpclient.mlst(remotepath);
		String infotype = info.get(MlsxEntry.TYPE);
		if (infotype.equals(MlsxEntry.TYPE_DIR)) {
		    // remote path is an existing directory, so add last component
		    // of localpath to the remotepath
		    remotestack.push(remotepath + "/" + localfile.getName());
		} else {
		    remotestack.push(remotepath);
		}
	    } catch (Exception e) {
		remotestack.push(remotepath);
	    }
	    while (!localstack.empty()) {
		File curlocalfile = (File)localstack.pop();
		String curremotepath = (String)remotestack.pop();
		if (!curlocalfile.exists()) {
		    throw new GridFTPCommandException("cmd_put: fatal error:", new FileNotFoundException("File '" + curlocalfile.toString() + "' does not exist!"));
		}
		if (curlocalfile.isDirectory()) {
		    //this.reset_passive();
		    this.debugMsg("}-> GridFTPClient::mlst('" + curremotepath + "')");
		    try {
			// try to list it to see if it already exists as a directory
			MlsxEntry info = ftpclient.mlst(curremotepath);
			String infotype = info.get(MlsxEntry.TYPE);
			if (!infotype.equals(MlsxEntry.TYPE_DIR)) {
			    this.cmd_mkdir(curremotepath);
			}
		    } catch (Exception e) {
			// if listing it didn't succeed, then try making the directory
			this.debugMsg("--ignoring this exception:");
			this.debugMsg(e.toString());
			this.cmd_mkdir(curremotepath);
		    }
		    String [] filelist = curlocalfile.list();
		    int i;
		    for (i = 0; i < filelist.length; i++) {
			localstack.push(new File(curlocalfile, filelist[i]));
			remotestack.push(curremotepath + "/" + filelist[i]);
		    }
		} else {
		    this.reset_passive();
		    this.debugMsg("}-> GridFTPClient::put('" + curlocalfile + "' '" + curremotepath + "')");
		    try {
			ftpclient.put(curlocalfile, curremotepath, false);
		    } catch (Exception e) {
			this.debugExit();
			throw new GridFTPCommandException("Error running put command", e);
		    }
		}
	    }
	} finally {
	    this.debugExit();
	}
    }

    protected void cmd_get(String remotepath, String localpath)
	throws GridFTPCommandException
    {
        this.debugEnter();
	try {
	    this.debugMsg("GridFTPCommand::cmd_get('" + remotepath + "', '" + localpath + "')");
	    while (remotepath.length() > 1 && remotepath.endsWith("/")) {
		remotepath = remotepath.substring(0, remotepath.length() - 1);
	    }
	    File localfile = new File(localpath);
	    if (localfile.exists() && localfile.isDirectory()) {
		// local path exists, so add last component of remote pathname
		int lastremoteslash = remotepath.lastIndexOf('/');
		if (lastremoteslash == -1) {
		    localfile = new File(localpath + "/" + remotepath);
		} else {
		    localfile = new File(localpath + "/" + remotepath.substring(lastremoteslash + 1));
		}
	    }
	    {
		// see if it's just a file, and if so, try to download it
		MlsxEntry info = this.doMlst(remotepath);
		String infotype = info.get(MlsxEntry.TYPE);
		if (infotype.equals(MlsxEntry.TYPE_FILE)) {
		    this.reset_passive();
		    try {
			ftpclient.get(remotepath, localfile);
		    } catch (Exception e) {
			this.debugExit();
			throw new GridFTPCommandException("Error getting file '" + remotepath + "' to local file '" + localfile.toString() + "'", e);
		    }
		    this.debugExit();
		    return;
		} else if (infotype.equals(MlsxEntry.TYPE_DIR)) {
		    // otherwise, if remote path is a directory, make the local
		    // directory counterpart
		    localfile.mkdir();
		}
	    }
	    // remote path is a directory, try tar-stream approach first
	    boolean fallthrough = false;
	    // See if we can use "tar-stream" approach
	    ExecutorService executorService = null;
	    try {
		this.disconnect();
		TwoPartyTarClientFactoryI clientFactory = new TwoPartyTarClientFactoryImpl();
		TwoPartyTarTransfer tptt = new TwoPartyTarTransfer(clientFactory, this.gridftphost, this.gridftpport, this.credential);
		if (tptt.isPopenDriverSupported()) {
		    this.debugMsg("| Server claims popen support.  Attempting tar-stream upload...");
		    PipedOutputStream pipeOut = new PipedOutputStream();
		    PipedInputStream pipeIn = new PipedInputStream(pipeOut);
		    executorService = Executors.newSingleThreadExecutor();
		    executorService.execute(new UntarFromPipeRunnable(pipeIn, localpath, new CompressUntar()));
		    String fullremotepath = remotepath;
		    if (remotepath.charAt(0) != '/' && this.curdir != null) {
			fullremotepath = this.curdir + "/" + remotepath;
		    }
		    this.debugMsg("}-> TwoPartyTarTransfer::downloadTarToPipe('" + fullremotepath + "', pipeOut)");
		    tptt.downloadTarToPipe(fullremotepath, pipeOut);
		} else {
		    fallthrough = true;
		}
	    } catch (Exception e) {
		throw new GridFTPCommandException("Error attempting tar stream download", e);
	    } finally {
		if (executorService != null) {
		    executorService.shutdownNow();
		}
	    }
	    if (!fallthrough) {
		return;
	    }
	    // if "tar-stream" not supported, then fall through
	    this.debugMsg("| Server does not support popen.  Attempting normal download...");
	    this.checkConnection();
	    Stack remotestack = new Stack();
	    remotestack.push(remotepath);
	    Stack localstack = new Stack();
	    localstack.push(localfile);
	    while (!remotestack.empty()) {
		String curremotepath = (String)remotestack.pop();
		File curlocalfile = (File)localstack.pop();
		Vector filelist = this.doMlsd(curremotepath);
		int i;
		for (i = 0; i < filelist.size(); i++) {
		    MlsxEntry info = (MlsxEntry)filelist.elementAt(i);
		    if (info.getFileName().equals(".") || info.getFileName().equals("..")) {
			continue;
		    }
		    File newlocalfile = new File(curlocalfile, info.getFileName());
		    String infotype = info.get(MlsxEntry.TYPE);
		    if (infotype.equals(MlsxEntry.TYPE_DIR)) {
			newlocalfile.mkdir();
			remotestack.push(curremotepath + "/" + info.getFileName());
			localstack.push(newlocalfile);
		    } else if (infotype.equals(MlsxEntry.TYPE_FILE)) {
			String newremotepath = curremotepath + "/" + info.getFileName();
			this.debugMsg("}-> GridFTPClient::get('" + newremotepath + "', '" + newlocalfile.toString() + "')");
			this.reset_passive();
			try {
			    ftpclient.get(newremotepath, newlocalfile);
			} catch (Exception e) {
			    this.debugExit();
			    throw new GridFTPCommandException("Error getting file '" + newremotepath + "' to local file '" + newlocalfile.toString() + "'", e);
			}
		    }
		}
	    }
	} finally {
	    this.debugExit();
	}
    }

    public static void main(String [] args)
	throws GridFTPCommandException
    {
	String [] newargs = new String [args.length];
	boolean opt_debug = false;
	boolean opt_noprompt = false;
	boolean ignoreoptions = false;
	int newargslen = 0;
	int i;
	for (i = 0; i < args.length; i++) {
	    String curarg = args[i];
	    if (!ignoreoptions && curarg.startsWith("--")) {
		if (curarg.equals("--")) {
		    ignoreoptions = true;
		    continue;
		}
		if (curarg.equals("--debug")) {
		    opt_debug = true;
		} else if (curarg.equals("--noprompt")) {
		    opt_noprompt = true;
		} else {
		    System.err.println("Unrecognized option: " + curarg);
		    System.exit(1);
		}
	    } else {
		newargs[newargslen++] = args[i];
	    }
	}
	if (newargs.length < 4) {
	    System.err.println("Not enough arguments!");
	    System.err.println("Usage:");
	    System.err.println("  GridFTPCommand [--debug] myproxyhost myproxyport griftphost gridftpport");
	    System.exit(1);
	}
	String username = System.getenv("MYPROXY_USER");
	String password = System.getenv("MYPROXY_PWD");
	GridFTPCommand cmdobj = new GridFTPCommand();
	if (opt_debug) {
	    cmdobj.debugOn();
	}
	cmdobj.connectParams(newargs[0], Integer.parseInt(newargs[1]), newargs[2], Integer.parseInt(newargs[3]), username, password);
	cmdobj.connect();
        //cmdobj.reset_passive();
	BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
	while (true) {
	    String commandline = null;
	    try {
		if (!opt_noprompt) {
		    System.out.print("> ");
		    System.out.flush();
		}
		commandline = br.readLine();
	    } catch (IOException ioe) {
		break;
	    }
	    if (commandline == null) {
		break;
	    }
	    String [] cmdandargs = commandline.split("\\s+");
	    if (cmdandargs.length == 0 || cmdandargs[0].equals("")) {
		continue;
	    }
	    if (cmdandargs[0].equals("exit")) {
		System.exit(0);
	    }
	    cmdobj.run(cmdandargs);
	}
	cmdobj.disconnect();
    }
}

