/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.notes;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.joda.time.DateTime;
import org.nbirn.fbirn.utilities.ChainedSQLException;
import org.nbirn.fbirn.utilities.ExternalProgressUpdater;

/**
 *
 * @author gadde
 */
public class BIRNHIDHierarchyModel implements ExperimentHierarchyModel {

    private static final Logger _logger = Logger.getLogger(BIRNHIDHierarchyModel.class.getName());
    private static final String extendedTupleName = "Note";
    private static final String ontologySource = "fBIRN Notes";
    private static final String TRUNCATESUFFIX = " <truncated -- see 'comments' column>";
    private static final String[] columnNames = new String[]{"id", "namespace", "name", "author", "timestamp", "value", "comment", "paths"};
    private static final String[] columnTypes = new String[]{"varchar", "varchar", "varchar", "varchar", "timestamp", "varchar", "varchar", "varchar"};
    private static final String _siteIDMarker = "__SITEID__=";
    private static final String _expNotesSuffix = "-notes";
    private static final String matchExp = "nc_experiment_uniqueid = (SELECT uniqueid FROM nc_experiment WHERE name = ? )";
    private static final String matchSubject = "subjectid = ?";
    private static final String matchVisit = "componentid = ?";
    private static final String matchStudy = "studyid = ?";
    private static final String matchSeries = "name = ?";
    private static final String whereExp = "WHERE name = ?";
    private static final String whereSubject = "WHERE " + matchSubject;
    private static final String whereVisit = "WHERE " + matchExp + " AND " + matchSubject + " AND " + matchVisit;
    private static final String whereStudy = "WHERE experimentid = (SELECT uniqueid FROM nc_experiment WHERE name = ? ) AND " + matchSubject + " AND " + matchVisit + " AND " + matchStudy;
    private static final String whereSeries = "WHERE " + matchExp + " AND " + matchSubject + " AND " + matchVisit + " AND " + matchStudy + " AND " + matchSeries;
    private static final String whereSeriesNoStudy = "WHERE " + matchExp + " AND " + matchSubject + " AND " + matchVisit + " AND " + matchSeries;
    private Connection _cachedConn = null;
    private Map<BaseExtendedTupleObj, Integer> _cachedExtendedTupleIDs = new HashMap<BaseExtendedTupleObj, Integer>();
    private final DecimalFormat _df4 = new DecimalFormat("0000");
    Map<String, ExtendedTupleObj> _tuplecache = new HashMap<String, ExtendedTupleObj>();
    Map<String, Object> _idcache = new HashMap<String, Object>();

    public URIWithType itemToPath(ExperimentHierarchyItem item) {
        URIWithType retval = null;
        try {
            if (item.expBaseURI != null) {
                retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI().toString() + "/"), item.expBaseURI.toString() + "/");
            } else if (item.expName != null && item.expID != null) {
                retval = new URIWithType(URIWithType.TYPE_DIR, "/home/Projects/" + item.expName + "__" + _df4.format(item.expID.longValue()) + "/");
            }
            if (item.expBaseURI == null) {
                retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), "Data/");
            }
            if (item.subjID == null || item.subjID.isEmpty()) {
                return retval;
            }
            retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), item.subjID + "/");
            if (item.visitName == null && item.visitID == null) {
                return retval;
            }
            retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), ((item.visitName == null) ? "" : item.visitName) + "__" + ((item.visitSiteID == null) ? "" : (_df4.format(item.visitSiteID.longValue()) + "__")) + ((item.visitID == null) ? "" : _df4.format(item.visitID.longValue())) + "/");
            if (item.studyName != null || item.studyID != null) {
                retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), new URI((item.studyName == null) ? "" : item.studyName) + "__" + ((item.studyID == null) ? "" : _df4.format(item.studyID.longValue())) + "/");
            }
            if (item.seriesName == null && item.seriesID == null) {
                return retval;
            }
            if (item.studyName == null && item.studyID == null) {
                // add empty study component
                retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), "(null)/");
            }
            retval = new URIWithType(URIWithType.TYPE_DIR, (retval == null ? null : retval.getURI()), new URI(((item.seriesName == null) ? ("*ID=" + item.seriesID + "*") : item.seriesName) + "/"));
        } catch (URISyntaxException e) {
            retval = null;
            // XXX don't pass up exception for now
        }
        return retval;
    }

    public ExperimentHierarchyItem pathToItem(URIWithType uriwt) throws NoteException {
        // path structure:
        //  /home/Projects/PROJECTNAME__PROJECTID/Data/SUBJECTID/VISITNAME__VISITID/STUDYNAME__STUDYID/SEGMENTNAME[/optional...]
        String scheme = uriwt.getURI().getScheme();
        if (scheme != null && !scheme.equals("file")) {
            throw new NoteException("Scheme of URI '" + uriwt.toString() + "' must be null or 'file:'");
        }
        File path = new File(uriwt.getURI().getPath());
        // path can stop at any component starting with the project
        if (!path.isAbsolute()) {
            throw new NoteException("Path '" + path.getPath() + "' is not absolute!");
        }
        LinkedList<String> comps = new LinkedList<String>();
        File tmppath = path;
        while (tmppath != null) {
            comps.addFirst(tmppath.getName());
            tmppath = tmppath.getParentFile();
        }
        if (comps.size() < 3 || !"".equals(comps.get(0)) || !"home".equals(comps.get(1)) || !"Projects".equals(comps.get(2))) {
            throw new NoteException("Path '" + path.getPath() + "' does not start with /home/Projects !");
        }
        StringBuilder baseURIStr = new StringBuilder();
        baseURIStr.append(comps.getFirst());
        baseURIStr.append("/");
        comps.removeFirst(); // <root>
        baseURIStr.append(comps.getFirst());
        baseURIStr.append("/");
        comps.removeFirst(); // home
        baseURIStr.append(comps.getFirst());
        baseURIStr.append("/");
        comps.removeFirst(); // Projects
        if (comps.size() == 0) {
            throw new NoteException("Path '" + path.getPath() + "' is missing an experiment component!");
        }
        ExperimentHierarchyItem item = new ExperimentHierarchyItem();
        int sep = -1;
        String comp = null;
        baseURIStr.append(comps.getFirst());
        baseURIStr.append("/");
        comp = comps.removeFirst();
        if ((sep = comp.indexOf("__")) == -1) {
            item.expName = comp;
        } else {
            item.expName = comp.substring(0, sep);
            try {
                item.expID = Integer.parseInt(comp.substring(sep + 2));
            } catch (NumberFormatException e) {
                throw new NoteException("Experiment ID '" + comp.substring(sep + 2) + "' does not seem to be numeric!");
            }
        }
        if (comps.size() == 0) {
            return item;
        }
        baseURIStr.append(comps.getFirst());
        baseURIStr.append("/");
        if (!"Data".equals(comps.removeFirst())) {
            throw new NoteException("Missing 'Data' component after experiment component in path '" + path.getPath() + "'");
        }
        try {
            item.expBaseURI = new URI(baseURIStr.toString());
        } catch (URISyntaxException e) {
            throw new NoteException("Extracted baseURI is not valid.", e);
        }
        if (comps.size() == 0) {
            return item;
        }
        item.subjID = comps.removeFirst();
        if (comps.size() == 0) {
            return item;
        }
        comp = comps.removeFirst();
        if ((sep = comp.indexOf("__")) == -1) {
            item.visitName = comp;
        } else {
            item.visitName = comp.substring(0, sep);
            try {
                int sep2;
                if ((sep2 = comp.indexOf("__", sep + 2)) == -1) {
                    item.visitID = Integer.parseInt(comp.substring(sep + 2));
                } else {
                    item.visitSiteID = Integer.parseInt(comp.substring(sep + 2, sep2));
                    item.visitID = Integer.parseInt(comp.substring(sep2 + 2));
                }
            } catch (NumberFormatException e) {
                throw new NoteException("Visit ID '" + comp.substring(sep + 2) + "' does not seem to be numeric!");
            }
        }
        if (comps.size() == 0) {
            return item;
        }
        comp = comps.removeFirst();
        if ((sep = comp.indexOf("__")) == -1) {
            if (!comp.isEmpty() && !comp.equals("(null)")) {
                item.studyName = comp;
            }
        } else {
            item.studyName = comp.substring(0, sep);
            try {
                item.studyID = Integer.parseInt(comp.substring(sep + 2));
            } catch (NumberFormatException e) {
                throw new NoteException("Study ID '" + comp.substring(sep + 2) + "' does not seem to be numeric!");
            }
        }
        if (comps.size() == 0) {
            return item;
        }
        comp = comps.removeFirst();
        if (comp.startsWith("*ID=")) {
            item.seriesID = Integer.parseInt(comp.substring(4, comp.length() - 1));
        } else {
            item.seriesName = comp;
        }
        return item;
    }

    private class Tuple {

        Integer tableID;
        Integer uniqueID;

        public Tuple(Integer tableID_, Integer uniqueID_) {
            tableID = tableID_;
            uniqueID = uniqueID_;
        }
    }

    private void cacheHierarchyTuples(Connection conn) throws NoteException {
        try {
            Statement stmt = null;
            ResultSet rs = null;

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT tableid, uniqueid, name FROM nc_experiment");
            while (rs.next()) {
                Integer tableid = rs.getInt(1);
                Integer uniqueid = rs.getInt(2);
                String name = rs.getString(3);
                final String key = "HIERTUPLEEXP-" + name;
                _logger.log(Level.FINER, "Adding key {0} => value (tableid={1}, uniqueid={2}) to ID cache", new Object[]{key, tableid, uniqueid});
                _idcache.put(key, new Tuple(tableid, uniqueid));
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT tableid, uniqueid, subjectid FROM nc_humansubject");
            while (rs.next()) {
                Integer tableid = rs.getInt(1);
                Integer uniqueid = rs.getInt(2);
                String subjectid = rs.getString(3);
                final String key = "HIERTUPLESUBJ-" + subjectid;
                _logger.log(Level.FINER, "Adding key {0} => value (tableid={1}, uniqueid={2}) to ID cache", new Object[]{key, tableid, uniqueid});
                _idcache.put(key, new Tuple(tableid, uniqueid));
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT tableid, uniqueid, nc_experiment_uniqueid, subjectid, componentid FROM nc_expcomponent");
            while (rs.next()) {
                Integer tableid = rs.getInt(1);
                Integer uniqueid = rs.getInt(2);
                Integer nc_experiment_uniqueid = rs.getInt(3);
                String subjectid = rs.getString(4);
                Integer componentid = rs.getInt(5);
                final String key = "HIERTUPLECOMP-" + nc_experiment_uniqueid + "-" + subjectid + "-" + componentid;
                _logger.log(Level.FINER, "Adding key {0} => value (tableid={1}, uniqueid={2}) to ID cache", new Object[]{key, tableid, uniqueid});
                _idcache.put(key, new Tuple(tableid, uniqueid));
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT tableid, uniqueid, experimentid, subjectid, componentid, studyid FROM nc_expstudy");
            while (rs.next()) {
                Integer tableid = rs.getInt(1);
                Integer uniqueid = rs.getInt(2);
                Integer nc_experiment_uniqueid = rs.getInt(3);
                String subjectid = rs.getString(4);
                Integer componentid = rs.getInt(5);
                Integer studyid = rs.getInt(6);
                final String key = "HIERTUPLESTUDY-" + nc_experiment_uniqueid + "-" + subjectid + "-" + componentid + "-" + studyid;
                _logger.log(Level.FINER, "Adding key {0} => value (tableid={1}, uniqueid={2}) to ID cache", new Object[]{key, tableid, uniqueid});
                _idcache.put(key, new Tuple(tableid, uniqueid));
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT tableid, uniqueid, nc_experiment_uniqueid, subjectid, componentid, studyid, name FROM nc_expsegment");
            while (rs.next()) {
                Integer tableid = rs.getInt(1);
                Integer uniqueid = rs.getInt(2);
                Integer nc_experiment_uniqueid = rs.getInt(3);
                String subjectid = rs.getString(4);
                Integer componentid = rs.getInt(5);
                Integer studyid = rs.getInt(6);
                String segmentname = rs.getString(7);
                final String key = "HIERTUPLESEG-" + nc_experiment_uniqueid + "-" + subjectid + "-" + componentid + "-" + studyid + "-" + segmentname;
                _logger.log(Level.FINER, "Adding key {0} => value (tableid={1}, uniqueid={2}) to ID cache", new Object[]{key, tableid, uniqueid});
                _idcache.put(key, new Tuple(tableid, uniqueid));
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }
        } catch (SQLException e) {
            throw new NoteException("Error getting hierarchy tuples for cache", e);
        }
    }

    private void cacheStoredTupleByNoteID(Connection conn) throws NoteException {
        try {
            Statement stmt = null;
            ResultSet rs = null;

            stmt = conn.createStatement();
            rs = stmt.executeQuery("SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='id'");
            while (rs.next()) {
                Integer storedtupleid = rs.getInt(1);
                String noteid = rs.getString(2);
                final String key = "STOREDTUPLEBYNOTEID-" + noteid;
                _logger.log(Level.FINER, "Adding key {0} => value (storedtupleid={1}) to ID cache", new Object[]{key, storedtupleid});
                _idcache.put(key, storedtupleid);
            }
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }
        } catch (SQLException e) {
            throw new NoteException("Error getting hierarchy tuples for cache", e);
        }
    }

    private class ExperimentIDs {

        Integer experimentUniqueID; // database row ID
        Integer expID; // numeric human-interpretable ID
        String expName; // human-readable name
    }

    private ExperimentIDs matchExperimentIDs(Connection conn, String expMatchStr) throws SQLException {
        ExperimentIDs retval = null;
        String key = "EXPMATCH-" + expMatchStr;
        if (_idcache.containsKey(key)) {
            return (ExperimentIDs) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        stmt = conn.prepareStatement("SELECT name, uniqueid FROM nc_experiment WHERE name LIKE ?");
        stmt.setString(1, expMatchStr);
        try {
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = new ExperimentIDs();
                String fullExpName = rs.getString(1);

                retval.experimentUniqueID = rs.getInt(2);
                int sep = fullExpName.indexOf("__");
                if (sep != -1) {
                    retval.expID = Integer.parseInt(fullExpName.substring(sep + 2));
                    retval.expName = fullExpName.substring(0, sep);
                    _idcache.put(key, retval);
                }
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private String getSubjectID(Connection conn, String name) throws SQLException {
        String retval = null;
        String key = "SUBJNAME-" + name;
        if (_idcache.containsKey(key)) {
            return (String) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT subjectid FROM nc_humansubject WHERE name = ?");
            stmt.setString(1, name);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getString(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private Integer getVisitID(Connection conn, Integer experimentUniqueID, String subjID, String name) throws SQLException {
        Integer retval = null;
        String key = "VISITNAME-" + experimentUniqueID + "-" + subjID + "-" + name;
        if (_idcache.containsKey(key)) {
            return (Integer) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT componentid FROM nc_expcomponent WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND name = ?");
            stmt.setString(3, name);
            stmt.setInt(1, experimentUniqueID);
            stmt.setString(2, subjID);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getInt(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private String getVisitName(Connection conn, Integer experimentUniqueID, String subjID, Integer ID) throws SQLException {
        String retval = null;
        String key = "VISITID-" + experimentUniqueID + "-" + subjID + "-" + ID;
        if (_idcache.containsKey(key)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", key);
            return (String) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT name FROM nc_expcomponent WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ?");
            stmt.setInt(1, experimentUniqueID);
            stmt.setString(2, subjID);
            stmt.setInt(3, ID);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getString(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private Integer getStudyID(Connection conn, Integer experimentUniqueID, String subjID, Integer visitID, String name) throws SQLException {
        Integer retval = null;
        String key = "STUDYNAME-" + experimentUniqueID + "-" + subjID + "-" + visitID + "-" + name;
        if (_idcache.containsKey(key)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", key);
            return (Integer) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT studyid FROM nc_expstudy WHERE experimentid = ? AND subjectid = ? AND componentid = ? AND name = ?");
            stmt.setInt(1, experimentUniqueID);
            stmt.setString(2, subjID);
            stmt.setInt(3, visitID);
            stmt.setString(4, name);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getInt(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private String getStudyName(Connection conn, Integer experimentUniqueID, String subjID, Integer visitID, Integer ID) throws SQLException {
        String retval = null;
        String key = "STUDYID-" + experimentUniqueID + "-" + subjID + "-" + visitID + "-" + ID;
        if (_idcache.containsKey(key)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", key);
            return (String) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT name FROM nc_expstudy WHERE experimentid = ? AND subjectid = ? AND componentid = ? AND studyid = ?");
            stmt = conn.prepareStatement("SELECT name FROM nc_expcomponent WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ?");
            stmt.setInt(1, experimentUniqueID);
            stmt.setString(2, subjID);
            stmt.setInt(3, visitID);
            stmt.setInt(4, ID);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getString(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private Integer getSeriesID(Connection conn, Integer experimentUniqueID, String subjID, Integer visitID, Integer studyID, String name) throws SQLException {
        Integer retval = null;
        String key = "SERIESNAME-" + experimentUniqueID + "-" + subjID + "-" + visitID + "-" + (studyID == null ? "" : studyID) + "-" + name;
        if (_idcache.containsKey(key)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", key);
            return (Integer) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            if (studyID != null) {
                stmt = conn.prepareStatement("SELECT segmentid FROM nc_expsegment WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ? AND studyid = ? AND name = ?");
                stmt.setInt(1, experimentUniqueID);
                stmt.setString(2, subjID);
                stmt.setInt(3, visitID);
                stmt.setInt(4, studyID);
                stmt.setString(5, name);
            } else {
                stmt = conn.prepareStatement("SELECT segmentid FROM nc_expsegment WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ? AND name = ?");
                stmt.setInt(1, experimentUniqueID);
                stmt.setString(2, subjID);
                stmt.setInt(3, visitID);
                stmt.setString(4, name);
            }
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getInt(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    private String getSeriesName(Connection conn, Integer experimentUniqueID, String subjID, Integer visitID, Integer studyID, Integer ID) throws SQLException {
        String retval = null;
        String key = "STUDYID-" + experimentUniqueID + "-" + subjID + "-" + visitID + (studyID == null ? "" : studyID) + "-" + ID;
        if (_idcache.containsKey(key)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", key);
            return (String) _idcache.get(key);
        }
        PreparedStatement stmt = null;
        try {
            if (studyID != null) {
                stmt = conn.prepareStatement("SELECT name FROM nc_expsegment WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ? AND studyid = ? AND segmentid = ?");
                stmt.setInt(1, experimentUniqueID);
                stmt.setString(2, subjID);
                stmt.setInt(3, visitID);
                stmt.setInt(4, studyID);
                stmt.setInt(5, ID);
            } else {
                stmt = conn.prepareStatement("SELECT name FROM nc_expsegment WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ? AND segmentid = ?");
                stmt.setInt(1, experimentUniqueID);
                stmt.setString(2, subjID);
                stmt.setInt(3, visitID);
                stmt.setInt(4, ID);
            }
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                retval = rs.getString(1);
                _idcache.put(key, retval);
            }
        } finally {
            stmt.close();
        }
        return retval;
    }

    /**
     * Fill in only item fields that are necessary to insert a item/note into the database, based on data in the database, if enough info is specified in the other fields.  In this case, the relevant fields are those that go into the path created by {@code itemToPath()}.
     * 
     * @param item
     * @param conn
     * @throws NoteException 
     */
    public void fillInItemIDsAux(ExperimentHierarchyItem item, Connection conn, boolean getSeriesID) throws NoteException {
        Integer experimentUniqueID = null;
        try {
            if ((item.expName != null) || (item.expID != null)) {
                // either expName or expID is defined
                String expMatchStr = null;
                if (item.expID == null) {
                    expMatchStr = item.expName + "__%";
                } else if (item.expName == null) {
                    expMatchStr = "%__" + _df4.format((long) item.expID);
                } else {
                    expMatchStr = item.expName + "__" + _df4.format((long) item.expID);
                }
                ExperimentIDs expIDs = matchExperimentIDs(conn, expMatchStr);
                if (expIDs != null) {
                    experimentUniqueID = expIDs.experimentUniqueID;
                    if (item.expID == null) {
                        item.expID = expIDs.expID;
                    } else if (item.expName == null) {
                        item.expName = expIDs.expName;
                    }
                }
            }
            if (item.subjID == null && item.subjName != null) {
                item.subjID = getSubjectID(conn, item.subjName);
            }
            if (experimentUniqueID != null && item.subjID != null && ((item.visitID == null) != (item.visitName == null))) {
                if (item.visitID == null) {
                    item.visitID = getVisitID(conn, experimentUniqueID, item.subjID, item.visitName);
                } else {
                    item.visitName = getVisitName(conn, experimentUniqueID, item.subjID, item.visitID);
                }
            }
            if (experimentUniqueID != null && item.expID != null && item.subjID != null && item.visitID != null && ((item.studyID == null) != (item.studyName == null))) {
                if (item.studyID == null) {
                    item.studyID = getStudyID(conn, experimentUniqueID, item.subjID, item.visitID, item.studyName);
                } else {
                    item.studyName = getStudyName(conn, experimentUniqueID, item.subjID, item.visitID, item.studyID);
                }
            }
            if (experimentUniqueID != null && item.expID != null && item.subjID != null && item.visitID != null && (((item.seriesID != null) && (item.seriesName == null)) || (getSeriesID && item.seriesID == null && item.seriesName != null))) {
                if (item.seriesID == null) {
                    item.seriesID = getSeriesID(conn, experimentUniqueID, item.subjID, item.visitID, item.studyID, item.seriesName);
                } else {
                    item.seriesName = getSeriesName(conn, experimentUniqueID, item.subjID, item.visitID, item.studyID, item.seriesID);
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error querying database", new ChainedSQLException(e));
        }
    }

    /**
     * Fill in only item fields that are necessary to insert a item/note into the database, based on data in the database, if enough info is specified in the other fields.  In this case, the relevant fields are those that go into the path created by {@code itemToPath()}.
     * 
     * @param item
     * @param conn
     * @throws NoteException 
     */
    public void fillInItemInsertIDs(ExperimentHierarchyItem item, Connection conn) throws NoteException {
        fillInItemIDsAux(item, conn, false);
    }

    /**
     * Fill in all available item fields that are unset based on data in the database, if enough info is specified in the other fields.
     * 
     * @param item
     * @param conn
     * @throws NoteException 
     */
    public void fillInItemIDs(ExperimentHierarchyItem item, Connection conn) throws NoteException {
        fillInItemIDsAux(item, conn, true);
    }

    private int getNextSeq(Connection conn) throws SQLException, NoteException {
        ResultSet rs = conn.createStatement().executeQuery("select nextval('uid_seq')");
        if (!rs.next()) {
            throw new NoteException("Error getting sequence number from database");
        }
        return rs.getInt(1);
    }

    private class BaseExtendedTupleObj {

        final String tupleClass;
        final String tupleSubclass;

        public BaseExtendedTupleObj(String tupleClass_, String tupleSubclass_) {
            tupleClass = tupleClass_;
            tupleSubclass = tupleSubclass_;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final BaseExtendedTupleObj other = (BaseExtendedTupleObj) obj;
            if (!this.tupleClass.equals(other.tupleClass)) {
                return false;
            }
            if (!this.tupleSubclass.equals(other.tupleSubclass)) {
                return false;
            }
            return true;
        }

        @Override
        public int hashCode() {
            int hash = 7;
            hash = 89 * hash + (this.tupleClass != null ? this.tupleClass.hashCode() : 0);
            hash = 89 * hash + (this.tupleSubclass != null ? this.tupleSubclass.hashCode() : 0);
            return hash;
        }
    }

    private class ExtendedTupleObj extends BaseExtendedTupleObj {

        final int tableID;
        final int tupleID;

        public ExtendedTupleObj(int tableID_, int tupleID_, String tupleClass_, String tupleSubclass_) {
            super(tupleClass_, tupleSubclass_);
            tableID = tableID_;
            tupleID = tupleID_;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final ExtendedTupleObj other = (ExtendedTupleObj) obj;
            if (this.tableID != other.tableID) {
                return false;
            }
            if (this.tupleID != other.tupleID) {
                return false;
            }
            return super.equals(obj);
        }

        @Override
        public int hashCode() {
            int hash = 5;
            hash = 11 * hash + this.tableID;
            hash = 11 * hash + this.tupleID;
            hash = 11 * hash + super.hashCode();
            return hash;
        }
    }

    private class StoredTupleObj extends ExtendedTupleObj {

        final Integer siteID;
        Integer storedTupleID = null;

        public StoredTupleObj(int tableID_, int tupleID_, String tupleClass_, String tupleSubclass_, Integer siteID_) {
            super(tableID_, tupleID_, tupleClass_, tupleSubclass_);
            siteID = siteID_;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (!super.equals(obj)) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final StoredTupleObj other = (StoredTupleObj) obj;
            if (this.siteID != other.siteID && (this.siteID == null || !this.siteID.equals(other.siteID))) {
                return false;
            }
            return super.equals(obj);
        }

        @Override
        public int hashCode() {
            int hash = super.hashCode();
            hash = 83 * hash + (this.siteID != null ? this.siteID.hashCode() : 0);
            hash = 83 * hash + super.hashCode();
            return hash;
        }
    }

    private Integer getExtendedTupleID(BaseExtendedTupleObj tuple, Connection conn, boolean create) throws NoteException {
        Integer extendedTupleID = null;
        if (tuple.getClass() != BaseExtendedTupleObj.class) {
            // make sure not to confuse HashMap and equals()
            tuple = new BaseExtendedTupleObj(tuple.tupleClass, tuple.tupleSubclass);
        }
        if (_cachedExtendedTupleIDs.containsKey(tuple)) {
            return _cachedExtendedTupleIDs.get(tuple);
        }
        NoteDBUser dbuser = getOrCreateNoteDBUser(conn);
        try {
            PreparedStatement stmt = null;
            boolean foundOntologySource = false;
            final String sourcekey = "ONTSOURCE-" + ontologySource;
            if (_idcache.containsKey(sourcekey)) {
                _logger.log(Level.FINER, "Found key {0} in ID cache", sourcekey);
                foundOntologySource = true;
            } else {
                stmt = conn.prepareStatement("SELECT ontologysource FROM nc_ontologysource WHERE ontologysource='" + ontologySource + "'");
                try {
                    ResultSet rs = stmt.executeQuery();
                    if (rs.next()) {
                        foundOntologySource = true;
                        _idcache.put(sourcekey, true);
                    }
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }
            if (!foundOntologySource) {
                if (!create) {
                    return null;
                }
                // need to create
                try {
                    stmt = conn.prepareStatement("INSERT INTO nc_ontologysource(ontologysource, uniqueid, tableid, owner, modtime, moduser, sourceuri, description) SELECT '" + ontologySource + "', ?, tableid, ?, now(), ?, null, null FROM nc_tableid WHERE tablename='NC_ONTOLOGYSOURCE'");
                    stmt.setInt(1, getNextSeq(conn));
                    stmt.setInt(2, dbuser.userID);
                    stmt.setInt(3, dbuser.userID);
                    stmt.executeUpdate();
                    _idcache.put(sourcekey, true);
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }

            for (String ontologyConcept : columnNames) {
                boolean foundConcept = false;
                final String conceptkey = "ONTCONCEPT-" + ontologySource + '-' + ontologyConcept;
                if (_idcache.containsKey(conceptkey)) {
                    _logger.log(Level.FINER, "Found key {0} in ID cache", conceptkey);
                    foundConcept = true;
                } else {
                    stmt = conn.prepareStatement("SELECT ontologysource FROM nc_ontologyconcept WHERE ontologysource='" + ontologySource + "' AND concept='" + ontologyConcept + "'");
                    try {
                        ResultSet rs = stmt.executeQuery();
                        if (rs.next()) {
                            foundConcept = true;
                            _idcache.put(conceptkey, true);
                        }
                    } finally {
                        try {
                            stmt.close();
                        } catch (SQLException e) {
                            // ignore
                        }
                    }
                }
                if (!foundConcept) {
                    if (!create) {
                        return null;
                    }
                    // need to create
                    try {
                        stmt = conn.prepareStatement("INSERT INTO nc_ontologyconcept(ontologysource, conceptid, concept, uniqueid, tableid, owner, modtime, moduser, ontologypath) SELECT '" + ontologySource + "', ?, ?, ?, tableid, ?, now(), ?, null FROM nc_tableid WHERE tablename='NC_ONTOLOGYSOURCE'");
                        stmt.setInt(1, getNextSeq(conn));
                        stmt.setString(2, ontologyConcept);
                        stmt.setInt(3, getNextSeq(conn));
                        stmt.setInt(4, dbuser.userID);
                        stmt.setInt(5, dbuser.userID);
                        stmt.executeUpdate();
                        _idcache.put(conceptkey, true);
                    } finally {
                        try {
                            stmt.close();
                        } catch (SQLException e) {
                            // ignore
                        }
                    }
                }
            }

            boolean foundclass = false;
            final String classkey = "CLASS-" + tuple.tupleClass;
            if (_idcache.containsKey(classkey)) {
                _logger.log(Level.FINER, "Found key {0} in ID cache", classkey);
                foundclass = true;
            } else {
                stmt = conn.prepareStatement("SELECT uniqueid FROM nc_tupleclass WHERE tupleclass=?");
                stmt.setString(1, tuple.tupleClass);
                try {
                    ResultSet rs = stmt.executeQuery();
                    if (rs.next()) {
                        foundclass = true;
                        _idcache.put(classkey, true);
                    }
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }
            if (!foundclass) {
                if (!create) {
                    return null;
                }
                // need to create
                try {
                    stmt = conn.prepareStatement("INSERT INTO nc_tupleclass(tupleclass, uniqueid, tableid, owner, modtime, moduser) SELECT ?, ?, tableid, ?, now(), ? FROM nc_tableid WHERE tablename='NC_ONTOLOGYSOURCE'");
                    stmt.setString(1, tuple.tupleClass);
                    stmt.setInt(2, getNextSeq(conn));
                    stmt.setInt(3, dbuser.userID);
                    stmt.setInt(4, dbuser.userID);
                    stmt.executeUpdate();
                    _idcache.put(classkey, true);
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }

            boolean foundsubclass = false;
            final String subclasskey = "SUBCLASS-" + tuple.tupleClass + '-' + tuple.tupleSubclass;
            if (_idcache.containsKey(subclasskey)) {
                _logger.log(Level.FINER, "Found key {0} in ID cache", subclasskey);
                foundsubclass = true;
            } else {
                stmt = conn.prepareStatement("SELECT uniqueid FROM nc_tuplesubclass WHERE tupleclass=? AND tuplesubclass=?");
                stmt.setString(1, tuple.tupleClass);
                stmt.setString(2, tuple.tupleSubclass);
                try {
                    ResultSet rs = stmt.executeQuery();
                    if (rs.next()) {
                        foundsubclass = true;
                        _idcache.put(subclasskey, true);
                    }
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }
            if (!foundsubclass) {
                if (!create) {
                    return null;
                }
                // need to create
                try {
                    stmt = conn.prepareStatement("INSERT INTO nc_tuplesubclass(tuplesubclass, tupleclass, uniqueid, tableid, owner, modtime, moduser) SELECT ?, ?, ?, tableid, ?, now(), ? FROM nc_tableid WHERE tablename='NC_ONTOLOGYSOURCE'");
                    stmt.setString(1, tuple.tupleSubclass);
                    stmt.setString(2, tuple.tupleClass);
                    stmt.setInt(3, getNextSeq(conn));
                    stmt.setInt(4, dbuser.userID);
                    stmt.setInt(5, dbuser.userID);
                    stmt.executeUpdate();
                    _idcache.put(subclasskey, true);
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
            }

            // try to find extended tuple
            stmt = conn.prepareStatement("SELECT uniqueid FROM nc_extendedtuple WHERE name='Note' AND tupleclass=? AND tuplesubclass=?");
            stmt.setString(1, tuple.tupleClass);
            stmt.setString(2, tuple.tupleSubclass);
            try {
                ResultSet rs = stmt.executeQuery();
                if (rs.next()) {
                    extendedTupleID = rs.getInt(1);
                    _cachedExtendedTupleIDs.put(tuple, extendedTupleID);
                }
            } finally {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    // ignore
                }
            }
            if (extendedTupleID == null) {
                if (!create) {
                    return extendedTupleID;
                }
                // need to create
                extendedTupleID = getNextSeq(conn);
                try {
                    stmt = conn.prepareStatement("INSERT INTO nc_extendedtuple(uniqueid, tableid, owner, modtime, moduser, name, tuplesubclass, tupleclass) SELECT ?, tableid, ?, now(), ?, '" + extendedTupleName + "', ?, ? FROM nc_tableid WHERE tablename='NC_EXTENDEDTUPLE'");
                    stmt.setInt(1, extendedTupleID);
                    stmt.setInt(2, dbuser.userID);
                    stmt.setInt(3, dbuser.userID);
                    stmt.setString(4, tuple.tupleClass);
                    stmt.setString(5, tuple.tupleClass);
                    stmt.executeUpdate();
                } finally {
                    try {
                        stmt.close();
                    } catch (SQLException e) {
                        // ignore
                    }
                }
                _cachedExtendedTupleIDs.put(tuple, extendedTupleID);

                // create columns
                int numColumns = columnNames.length;
                for (int i = 0; i < numColumns; i++) {
                    String columnName = columnNames[i];
                    String columnType = columnTypes[i];
                    try {
                        stmt = conn.prepareStatement("INSERT INTO nc_tuplecolumns(extendedtupleid, columnname, columntype, tableid, uniqueid, owner, modtime, moduser, defaultvalue, nullable, columnontology, columnconcept, measurementsystem, measurementunit) SELECT ?, ?, ?, tid.tableid, nextval('uid_seq'), ?, now(), ?, null, true, ?, oc.conceptid, null, null FROM nc_tableid tid, nc_ontologyconcept oc WHERE tid.tablename='NC_TUPLECOLUMNS' AND oc.ontologysource=? AND oc.concept=?");
                        stmt.setInt(1, extendedTupleID);
                        stmt.setString(2, columnName);
                        stmt.setString(3, columnType);
                        stmt.setInt(4, dbuser.userID);
                        stmt.setInt(5, dbuser.userID);
                        stmt.setString(6, ontologySource);
                        stmt.setString(7, ontologySource);
                        stmt.setString(8, columnName);
                        stmt.executeUpdate();
                    } finally {
                        try {
                            stmt.close();
                        } catch (SQLException e) {
                            // ignore
                        }
                    }
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error putting note to extended tuple", new ChainedSQLException(e));
        }
        return extendedTupleID;
    }

    private PreparedStatement getStoredTuplePreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("INSERT INTO nc_storedtuple(uniqueid, basetableid, basetupleid, extendedtupleid, tableid, owner, modtime, moduser) SELECT ?, ?, ?, ?, tableid, ?, now(), ? FROM nc_tableid WHERE tablename='NC_STOREDTUPLE'");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getUpdateStoredTuplePreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("UPDATE nc_storedtuple SET basetableid=?, basetupleid=?, extendedtupleid=?, tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_STOREDTUPLE'), owner=?, modtime=now(), moduser=? WHERE uniqueid=?");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getTupleIntegerPreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("INSERT INTO nc_tupleinteger(tableid, uniqueid, owner, modtime, moduser, extendedtupleid, columnname, columntype, storedtupleid, textvalue, comments, datavalue) SELECT tableid, nextval('uid_seq'), ?, now(), ?, ?, ?, 'integer', ?, ?, ?, ? FROM nc_tableid WHERE tablename='NC_TUPLEINTEGER'");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getTupleTimestampPreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("INSERT INTO nc_tupletimestamp(tableid, uniqueid, owner, modtime, moduser, extendedtupleid, columnname, columntype, storedtupleid, textvalue, comments, datavalue) SELECT tableid, nextval('uid_seq'), ?, now(), ?, ?, ?, 'timestamp', ?, ?, ?, ? FROM nc_tableid WHERE tablename='NC_TUPLETIMESTAMP'");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getUpdateTupleTimestampPreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("UPDATE nc_tupletimestamp SET tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_TUPLETIMESTAMP'), owner=?, modtime=now(), moduser=?, extendedtupleid=?, columntype='timestamp', textvalue=?, comments=?, datavalue=? WHERE storedtupleid=? AND columnname=?");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getTupleVarcharPreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("INSERT INTO nc_tuplevarchar(tableid, uniqueid, owner, modtime, moduser, extendedtupleid, columnname, columntype, storedtupleid, textvalue, comments, datavalue) SELECT tableid, nextval('uid_seq'), ?, now(), ?, ?, ?, 'varchar', ?, ?, ?, ? FROM nc_tableid WHERE tablename='NC_TUPLEVARCHAR'");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private PreparedStatement getUpdateTupleVarcharPreparedStatement(Connection conn) throws NoteException {
        try {
            return conn.prepareStatement("UPDATE nc_tuplevarchar SET tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_TUPLEVARCHAR'), owner=?, modtime=now(), moduser=?, extendedtupleid=?, columntype='timestamp', textvalue=?, comments=?, datavalue=? WHERE storedtupleid=? AND columnname=?");
        } catch (SQLException e) {
            throw new NoteException("Error preparing statement for note insert", new ChainedSQLException(e));
        }
    }

    private Integer getStoredTupleFromNoteID(String noteID, Connection conn) throws NoteException {
        String cacheKey = "STOREDTUPLEBYNOTEID-" + noteID;
        if (_idcache.containsKey(cacheKey)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", cacheKey);
            return (Integer) _idcache.get(cacheKey);
        }
        PreparedStatement stmt = null;
        try {
            stmt = conn.prepareStatement("SELECT storedtupleid FROM nc_tuplevarchar WHERE columnname='id' AND datavalue=?");
            stmt.setString(1, noteID);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                return rs.getInt(1);
            } else {
                return null;
            }
        } catch (SQLException e) {
            throw new NoteException("Error getting stored tuple ID from note ID");
        } finally {
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    //ignore
                }
            }
        }
    }

    private void putStoredTuple(StoredTupleObj tuple, Integer extendedTupleID, Integer userID, Connection conn, PreparedStatement tupleStmt) throws NoteException {
        // INSERT INTO nc_storedtuple(uniqueid, basetableid, basetupleid, extendedtupleid, tableid, owner, modtime, moduser) SELECT ?, ?, ?, ?, tableid, ?, now(), ? FROM nc_tableid WHERE tablename='NC_STOREDTUPLE'
        try {
            tuple.storedTupleID = this.getNextSeq(conn);
            tupleStmt.setInt(1, tuple.storedTupleID);
            tupleStmt.setInt(2, tuple.tableID);
            tupleStmt.setInt(3, tuple.tupleID);
            tupleStmt.setInt(4, extendedTupleID);
            tupleStmt.setInt(5, userID);
            tupleStmt.setInt(6, userID);
            tupleStmt.executeUpdate();
        } catch (SQLException e) {
            throw new NoteException("Error preparing batch note insert", new ChainedSQLException(e));
        }
    }

    private void updateStoredTuple(StoredTupleObj tuple, Integer extendedTupleID, Integer userID, Connection conn, PreparedStatement tupleStmt) throws NoteException {
        // UPDATE nc_storedtuple SET basetableid=?, basetupleid=?, extendedtupleid=?, tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_STOREDTUPLE'), owner=?, modtime=now(), moduser=? WHERE uniqueid=?
        try {
            tupleStmt.setInt(1, tuple.tableID);
            tupleStmt.setInt(2, tuple.tupleID);
            tupleStmt.setInt(3, extendedTupleID);
            tupleStmt.setInt(4, userID);
            tupleStmt.setInt(5, userID);
            tupleStmt.setInt(6, tuple.storedTupleID);
            tupleStmt.executeUpdate();
        } catch (SQLException e) {
            throw new NoteException("Error preparing batch note insert", new ChainedSQLException(e));
        }
    }

    private void putNoteToStatementBatch(StoredTupleObj tuple, Note note, Integer extendedTupleID, Integer userID, PreparedStatement timestampStmt, PreparedStatement varcharStmt) throws NoteException {
        // varcharStmt: INSERT INTO nc_tuplevarchar(tableid, uniqueid, owner, modtime, moduser, extendedtupleid, columnname, columntype, storedtupleid, textvalue, comments, datavalue) SELECT tableid, nextval('uid_seq'), ?, now(), ?, ?, ?, 'varchar', ?, ?, ?, ? FROM nc_tableid WHERE tablename='NC_TUPLEVARCHAR'
        // timestampStmt: INSERT INTO nc_tupletimestamp(tableid, uniqueid, owner, modtime, moduser, extendedtupleid, columnname, columntype, storedtupleid, textvalue, comments, datavalue) SELECT tableid, nextval('uid_seq'), ?, now(), ?, ?, ?, 'timestamp', ?, ?, ?, ? FROM nc_tableid WHERE tablename='NC_TUPLETIMESTAMP'
        String noteID = note.getID();
        try {
            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "id");
            varcharStmt.setInt(5, tuple.storedTupleID);
            varcharStmt.setString(6, noteID);
            varcharStmt.setString(7, null);
            varcharStmt.setString(8, noteID);
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "namespace");
            varcharStmt.setInt(5, tuple.storedTupleID);
            varcharStmt.setString(6, note.getNamespace());
            varcharStmt.setString(7, null);
            varcharStmt.setString(8, note.getNamespace());
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "name");
            varcharStmt.setInt(5, tuple.storedTupleID);
            varcharStmt.setString(6, note.getName());
            varcharStmt.setString(7, null);
            varcharStmt.setString(8, note.getName());
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "author");
            varcharStmt.setInt(5, tuple.storedTupleID);
            varcharStmt.setString(6, note.getAuthor());
            varcharStmt.setString(7, null);
            varcharStmt.setString(8, note.getAuthor());
            varcharStmt.addBatch();

            timestampStmt.setInt(1, userID);
            timestampStmt.setInt(2, userID);
            timestampStmt.setInt(3, extendedTupleID);
            timestampStmt.setString(4, "timestamp");
            timestampStmt.setInt(5, tuple.storedTupleID);
            timestampStmt.setString(6, note.getTimeStamp().toString());
            timestampStmt.setString(7, null);
            timestampStmt.setTimestamp(8, new Timestamp(note.getTimeStamp().getMillis()));
            timestampStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "value");
            varcharStmt.setInt(5, tuple.storedTupleID);
            varcharStmt.setString(6, note.getValue());
            varcharStmt.setString(7, null);
            varcharStmt.setString(8, note.getValue());
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "comment");
            varcharStmt.setInt(5, tuple.storedTupleID);
            {
                String comment = note.getComment();
                String textvalue = comment;
                if (comment.length() > 255) {
                    textvalue = comment.substring(0, 255 - TRUNCATESUFFIX.length()) + TRUNCATESUFFIX;
                }
                varcharStmt.setString(6, textvalue);
                varcharStmt.setString(7, comment); // note we put the comment field in the comments column not the datavalue column, because datavalue only allows 255 characters!
            }
            varcharStmt.setString(8, null);
            varcharStmt.addBatch();

            StringBuilder sb = new StringBuilder();
            boolean first = true;
            for (URIWithType path : note.getPaths()) {
                sb.append(path.toString());
                if (!first) {
                    sb.append("\n");
                }
                first = false;
            }
            String pathsstr = sb.toString();
            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, "paths");
            varcharStmt.setInt(5, tuple.storedTupleID);
            {
                String textvalue = pathsstr;
                if (pathsstr.length() > 255) {
                    textvalue = pathsstr.substring(0, 255 - TRUNCATESUFFIX.length()) + TRUNCATESUFFIX;
                }
                varcharStmt.setString(6, textvalue);
                varcharStmt.setString(7, pathsstr); // note we put the paths field in the comments column not the datavalue column, because datavalue only allows 255 characters!
            }
            varcharStmt.setString(8, pathsstr);
            varcharStmt.addBatch();
        } catch (SQLException e) {
            throw new NoteException("Error preparing batch note insert", new ChainedSQLException(e));
        }
    }

    private void updateNoteToStatementBatch(StoredTupleObj tuple, Note note, Integer extendedTupleID, Integer userID, PreparedStatement timestampStmt, PreparedStatement varcharStmt) throws NoteException {
        // update varcharStmt: UPDATE nc_tuplevarchar SET tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_TUPLEVARCHAR'), owner=?, modtime=now(), moduser=?, extendedtupleid=?, columntype='timestamp', textvalue=?, comments=?, datavalue=? WHERE storedtupleid=? AND columnname=?
        // update timestampStmt: UPDATE nc_tupletimestamp SET tableid=(SELECT tableid FROM nc_tableid WHERE tablename='NC_TUPLETIMESTAMP'), owner=?, modtime=now(), moduser=?, extendedtupleid=?, columntype='timestamp', textvalue=?, comments=?, datavalue=? WHERE storedtupleid=? AND columnname=?
        String noteID = note.getID();
        try {
            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, noteID);
            varcharStmt.setString(5, null);
            varcharStmt.setString(6, noteID);
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "id");
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, note.getNamespace());
            varcharStmt.setString(5, null);
            varcharStmt.setString(6, note.getNamespace());
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "namespace");
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, note.getName());
            varcharStmt.setString(5, null);
            varcharStmt.setString(6, note.getName());
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "name");
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, note.getAuthor());
            varcharStmt.setString(5, null);
            varcharStmt.setString(6, note.getAuthor());
            varcharStmt.setInt(8, tuple.storedTupleID);
            varcharStmt.setString(7, "author");
            varcharStmt.addBatch();

            timestampStmt.setInt(1, userID);
            timestampStmt.setInt(2, userID);
            timestampStmt.setInt(3, extendedTupleID);
            timestampStmt.setString(4, note.getTimeStamp().toString());
            timestampStmt.setString(5, null);
            timestampStmt.setTimestamp(6, new Timestamp(note.getTimeStamp().getMillis()));
            timestampStmt.setInt(7, tuple.storedTupleID);
            timestampStmt.setString(8, "timestamp");
            timestampStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            varcharStmt.setString(4, note.getValue());
            varcharStmt.setString(5, null);
            varcharStmt.setString(6, note.getValue());
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "value");
            varcharStmt.addBatch();

            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            {
                String comment = note.getComment();
                String textvalue = comment;
                if (comment.length() > 255) {
                    textvalue = comment.substring(0, 255 - TRUNCATESUFFIX.length()) + TRUNCATESUFFIX;
                }
                varcharStmt.setString(4, textvalue);
                varcharStmt.setString(5, comment); // note we put the comment field in the comments column not the datavalue column, because datavalue only allows 255 characters!
            }
            varcharStmt.setString(6, null);
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "comment");
            varcharStmt.addBatch();

            StringBuilder sb = new StringBuilder();
            boolean first = true;
            for (URIWithType path : note.getPaths()) {
                sb.append(path.toString());
                if (!first) {
                    sb.append("\n");
                }
                first = false;
            }
            String pathsstr = sb.toString();
            varcharStmt.setInt(1, userID);
            varcharStmt.setInt(2, userID);
            varcharStmt.setInt(3, extendedTupleID);
            {
                String textvalue = pathsstr;
                if (pathsstr.length() > 255) {
                    textvalue = pathsstr.substring(0, 255 - TRUNCATESUFFIX.length()) + TRUNCATESUFFIX;
                }
                varcharStmt.setString(4, textvalue);
                varcharStmt.setString(5, pathsstr); // note we put the paths field in the comments column not the datavalue column, because datavalue only allows 255 characters!
            }
            varcharStmt.setString(6, pathsstr);
            varcharStmt.setInt(7, tuple.storedTupleID);
            varcharStmt.setString(8, "paths");
            varcharStmt.addBatch();
        } catch (SQLException e) {
            throw new NoteException("Error preparing batch note insert", new ChainedSQLException(e));
        }
    }

    private class NoteDBUser {

        Integer userID = null;
        Integer personID = null;

        NoteDBUser(Integer userID_, Integer personID_) {
            userID = userID_;
            personID = personID_;
        }
    }

    private NoteDBUser getOrCreateNoteDBUser(Connection conn) throws NoteException {
        // find the "NoteCollector" person
        boolean createdUser = false;
        Integer userID = null;
        Integer personID = null;
        String userKey = "NoteDBUser";
        String personKey = "NoteDBPerson";
        if (_idcache.containsKey(userKey)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", userKey);
            userID = (Integer) _idcache.get(userKey);
        }
        if (_idcache.containsKey(personKey)) {
            _logger.log(Level.FINER, "Found key {0} in ID cache", personKey);
            personID = (Integer) _idcache.get(personKey);
        }
        try {
            if (userID == null) {
                PreparedStatement stmt = conn.prepareStatement("SELECT uniqueid FROM nc_databaseuser WHERE name=?");
                stmt.setString(1, this.getClass().getCanonicalName());
                try {
                    stmt.execute();
                    ResultSet rs = stmt.getResultSet();
                    if (rs.next()) {
                        userID = rs.getInt(1);
                    } else {
                        stmt.close();
                        userID = getNextSeq(conn);
                        // grab an existing personid for now, will replace later
                        stmt = conn.prepareStatement("INSERT INTO nc_databaseuser(uniqueid, name, tableid, owner, modtime, moduser, userclass, userstatus, isgroup, personid) select ?, ?, t.tableID, ?, now(), ?, (SELECT uniqueid FROM nc_userclass WHERE userclass = 'data processing'), (SELECT uniqueid FROM nc_userstatus WHERE userstatus = 'active'), false, p.uniqueid from nc_tableID t, (SELECT uniqueid FROM nc_person LIMIT 1) p where tableName = 'NC_PERSON'");
                        stmt.setInt(1, userID);
                        stmt.setString(2, this.getClass().getCanonicalName());
                        stmt.setInt(3, userID);
                        stmt.setInt(4, userID);
                        stmt.executeUpdate();
                        createdUser = true;
                    }
                    _idcache.put(userKey, userID);
                } finally {
                    stmt.close();
                }
            }

            if (personID == null) {
                PreparedStatement stmt = conn.prepareStatement("SELECT uniqueid FROM nc_person WHERE first_name=? AND last_name=''");
                stmt.setString(1, this.getClass().getCanonicalName());
                try {
                    stmt.execute();
                    ResultSet rs = stmt.getResultSet();
                    if (rs.next()) {
                        personID = rs.getInt(1);
                    } else {
                        stmt.close();
                        personID = getNextSeq(conn);
                        stmt = conn.prepareStatement("INSERT INTO nc_person(uniqueid, tableid, owner, modtime, moduser, first_name, last_name, email) select ?, tableID, ?, now(), ?, ?, '', '' from nc_tableID where tableName = 'NC_PERSON'");
                        stmt.setInt(1, personID);
                        stmt.setInt(2, userID);
                        stmt.setInt(3, userID);
                        stmt.setString(4, this.getClass().getCanonicalName());
                        stmt.executeUpdate();
                    }
                    _idcache.put(personKey, personID);
                } finally {
                    stmt.close();
                }
            }

            if (createdUser) {
                PreparedStatement stmt = conn.prepareStatement("UPDATE nc_databaseuser SET personid = ? WHERE uniqueid = ?");
                stmt.setInt(1, personID);
                stmt.setInt(2, userID);
                try {
                    stmt.executeUpdate();
                } finally {
                    stmt.close();
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error getting/creating Note user", new ChainedSQLException(e));
        }
        return new NoteDBUser(userID, personID);
    }

    private void createItem(ExperimentHierarchyItem item, Connection conn) throws NoteException {
        if (item.expName == null || item.expID == null) {
            throw new NoteException("HID.putItemDesc: expName and expID cannot be null!");
        }
        try {
            PreparedStatement stmt = null;
            String fullExpName = item.expName + "__" + _df4.format((long) item.expID);
            String table = "nc_experiment";
            String desc = item.expDesc;
            String where = whereExp;
            Integer uniqueID = null;
            if (item.subjID != null) {
                desc = item.subjDesc;
                table = "nc_humansubject";
                where = whereSubject;
                if (item.visitID != null) {
                    desc = item.visitDesc;
                    table = "nc_expcomponent";
                    where = whereVisit;
                    if (item.studyID != null) {
                        desc = item.studyDesc;
                        table = "nc_expstudy";
                        where = whereStudy;
                    }
                    if (item.seriesName != null) {
                        desc = item.seriesDesc;
                        table = "nc_expsegment";
                        if (item.studyID == null) {
                            where = whereSeriesNoStudy;
                        } else {
                            where = whereSeries;
                        }
                    }
                }
            }
            stmt = conn.prepareStatement("SELECT uniqueid FROM " + table + " " + where);
            if (where == whereSubject) {
                stmt.setString(1, item.subjID);
            } else {
                stmt.setString(1, fullExpName);
                if (where != whereExp) {
                    stmt.setString(2, item.subjID);
                    stmt.setInt(3, item.visitID);
                    if (where == whereStudy) {
                        stmt.setInt(4, item.studyID);
                    } else if (where == whereSeries) {
                        stmt.setInt(4, item.studyID);
                        stmt.setString(5, item.seriesName);
                    } else if (where == whereSeriesNoStudy) {
                        stmt.setString(4, item.seriesName);
                    }
                }
            }
            boolean itemExists = false;
            try {
                stmt.execute();
                ResultSet rs = stmt.getResultSet();
                itemExists = rs.next();
                if (itemExists) {
                    uniqueID = rs.getInt(1);
                }
            } finally {
                stmt.close();
            }
            if (!itemExists) {
                // no existing item
                // create the item
                NoteDBUser dbuser = getOrCreateNoteDBUser(conn);
                Integer userID = dbuser.userID;
                Integer personID = dbuser.personID;

                // start with experiment level
                Integer experimentUniqueID = null;
                stmt = conn.prepareStatement("SELECT uniqueid FROM nc_experiment " + whereExp);
                stmt.setString(1, fullExpName);
                try {
                    stmt.execute();
                    ResultSet rs = stmt.getResultSet();
                    itemExists = rs.next();
                    if (itemExists) {
                        experimentUniqueID = rs.getInt(1);
                    }
                } finally {
                    stmt.close();
                }
                if (!itemExists) {
                    ExperimentHierarchyItem expItem = new ExperimentHierarchyItem();
                    expItem.expID = item.expID;
                    expItem.expName = item.expName;
                    URIWithType baseURI = itemToPath(expItem);
                    experimentUniqueID = getNextSeq(conn);
                    stmt = conn.prepareStatement("INSERT INTO nc_experiment(uniqueid, tableid, owner, modtime, moduser, name, description, contactperson, baseuri, isregressiondata) select ?, tableid, ?, now(), ?, ?, '', ?, ?, false from nc_tableid where tablename = 'NC_EXPERIMENT'");
                    stmt.setInt(1, experimentUniqueID);
                    stmt.setInt(2, userID);
                    stmt.setInt(3, userID);
                    stmt.setString(4, fullExpName);
                    stmt.setInt(5, personID);
                    stmt.setString(6, baseURI.getURI().toString());
                    uniqueID = experimentUniqueID;
                    try {
                        stmt.executeUpdate();
                    } finally {
                        stmt.close();
                    }
                }

                if (item.subjID != null) {
                    // subject level
                    stmt = conn.prepareStatement("SELECT uniqueid FROM nc_humanSubject " + whereSubject);
                    stmt.setString(1, item.subjID);
                    try {
                        stmt.execute();
                        ResultSet rs = stmt.getResultSet();
                        itemExists = rs.next();
                    } finally {
                        stmt.close();
                    }
                    if (!itemExists) {
                        uniqueID = getNextSeq(conn);
                        stmt = conn.prepareStatement("INSERT INTO nc_humanSubject(subjectid, uniqueid, tableid, owner, modtime,moduser, extensionname,nc_animalspecies_uniqueid) select ?, ?, tableid, ?, now(), ?, 'humanSubject', (select max(uniqueid) from nc_animalspecies where name = 'researchSubject') from nc_tableid where tablename = 'NC_HUMANSUBJECT'");
                        stmt.setString(1, item.subjID);
                        stmt.setInt(2, uniqueID);
                        stmt.setInt(3, userID);
                        stmt.setInt(4, userID);
                        try {
                            stmt.executeUpdate();
                        } finally {
                            stmt.close();
                        }
                    }

                    if (item.visitID != null) {
                        // visit level
                        Integer visitTypeID = null;
                        stmt = conn.prepareStatement("SELECT uniqueid FROM nc_visittype WHERE visittype=?");
                        stmt.setString(1, this.getClass().getCanonicalName());
                        try {
                            stmt.execute();
                            ResultSet rs = stmt.getResultSet();
                            if (rs.next()) {
                                visitTypeID = rs.getInt(1);
                            } else {
                                stmt.close();
                                visitTypeID = getNextSeq(conn);
                                stmt = conn.prepareStatement("INSERT INTO nc_visittype(visittype, uniqueid, tableid, owner, modtime, moduser, description) select ?, ?, tableID, ?, now(), ?, 'dummy visit type inserted for notes' from nc_tableID where tableName = 'NC_VISITTYPE'");
                                stmt.setString(1, this.getClass().getCanonicalName());
                                stmt.setInt(2, visitTypeID);
                                stmt.setInt(3, userID);
                                stmt.setInt(4, userID);
                                stmt.executeUpdate();
                            }
                        } finally {
                            stmt.close();
                        }

                        stmt = conn.prepareStatement("SELECT uniqueid FROM nc_expcomponent " + whereVisit);
                        stmt.setString(1, fullExpName);
                        stmt.setString(2, item.subjID);
                        stmt.setInt(3, item.visitID);
                        try {
                            stmt.execute();
                            ResultSet rs = stmt.getResultSet();
                            itemExists = rs.next();
                        } finally {
                            stmt.close();
                        }
                        if (!itemExists) {
                            uniqueID = getNextSeq(conn);
                            stmt = conn.prepareStatement("INSERT INTO nc_expcomponent(componentid, nc_experiment_uniqueid, subjectid, uniqueid, tableid, owner, modtime, moduser, time_stamp, description, visittype, name, timeinterval, istimeinterval) select ?, ?, ?, ?, tableid, ?, now(), ?, ?, ?, ?, ?, null, false from nc_tableid where tablename = 'NC_EXPCOMPONENT'");
                            stmt.setInt(1, item.visitID);
                            stmt.setInt(2, experimentUniqueID);
                            stmt.setString(3, item.subjID);
                            stmt.setInt(4, uniqueID);
                            stmt.setInt(5, userID);
                            stmt.setInt(6, userID);
                            stmt.setTimestamp(7, new Timestamp((item.visitModTime == null) ? System.currentTimeMillis() : item.visitModTime.getMillis()));
                            stmt.setString(8, item.visitDesc);
                            stmt.setString(9, this.getClass().getCanonicalName());
                            stmt.setString(10, item.visitName);
                            try {
                                stmt.executeUpdate();
                            } finally {
                                stmt.close();
                            }
                        }

                        if (item.studyID != null) {
                            // study level
                            stmt = conn.prepareStatement("SELECT uniqueid FROM nc_expstudy " + whereStudy);
                            stmt.setString(1, fullExpName);
                            stmt.setString(2, item.subjID);
                            stmt.setInt(3, item.visitID);
                            stmt.setInt(4, item.studyID);
                            try {
                                stmt.execute();
                                ResultSet rs = stmt.getResultSet();
                                itemExists = rs.next();
                            } finally {
                                stmt.close();
                            }
                            if (!itemExists) {
                                uniqueID = getNextSeq(conn);
                                stmt = conn.prepareStatement("INSERT INTO nc_expstudy(studyid, componentid, experimentid, subjectid, uniqueid, tableid, owner, modtime, moduser, time_stamp, description, name, istimeinterval) select ?, ?, ?, ?, ?, tableid, ?, now(), ?, ?, ?, ?, false from nc_tableid WHERE tablename = 'NC_EXPSTUDY'");
                                stmt.setInt(1, item.studyID);
                                stmt.setInt(2, item.visitID);
                                stmt.setInt(3, experimentUniqueID);
                                stmt.setString(4, item.subjID);
                                stmt.setInt(5, uniqueID);
                                stmt.setInt(6, userID);
                                stmt.setInt(7, userID);
                                stmt.setTimestamp(8, new Timestamp((item.studyModTime == null) ? System.currentTimeMillis() : item.studyModTime.getMillis()));
                                stmt.setString(9, item.studyDesc);
                                stmt.setString(10, item.studyName);
                                try {
                                    stmt.executeUpdate();
                                } finally {
                                    stmt.close();
                                }
                            }
                        }

                        if (item.seriesName != null) {
                            // series level
                            String protocolID = this.getClass().getCanonicalName();
                            Integer protocolVersion = null;
                            stmt = conn.prepareStatement("SELECT protocolid, protocolversion FROM nc_protocol WHERE protocolid=?");
                            stmt.setString(1, protocolID);
                            try {
                                stmt.execute();
                                ResultSet rs = stmt.getResultSet();
                                if (rs.next()) {
                                    protocolID = rs.getString(1);
                                    protocolVersion = rs.getInt(2);
                                } else {
                                    stmt.close();
                                    protocolVersion = 1;
                                    stmt = conn.prepareStatement("INSERT INTO nc_protocol(protocolversion, protocolid, uniqueid, tableid, owner, modtime, moduser, name, description) select 1, ?, nextval('uid_seq'), tableID, ?, now(), ?, ?, '' from nc_tableID where tableName = 'NC_PROTOCOL'");
                                    stmt.setString(1, protocolID);
                                    stmt.setInt(2, userID);
                                    stmt.setInt(3, userID);
                                    stmt.setString(4, this.getClass().getCanonicalName());
                                    stmt.executeUpdate();
                                }
                            } finally {
                                stmt.close();
                            }

                            stmt = conn.prepareStatement("SELECT uniqueid FROM nc_expsegment " + ((item.studyID == null) ? whereSeriesNoStudy : whereSeries));
                            stmt.setString(1, fullExpName);
                            stmt.setString(2, item.subjID);
                            stmt.setInt(3, item.visitID);
                            if (item.studyID == null) {
                                stmt.setString(4, item.seriesName);
                            } else {
                                stmt.setInt(4, item.studyID);
                                stmt.setString(5, item.seriesName);
                            }
                            try {
                                stmt.execute();
                                ResultSet rs = stmt.getResultSet();
                                itemExists = rs.next();
                            } finally {
                                stmt.close();
                            }
                            if (!itemExists) {
                                Integer seriesID = null;
                                // don't add studyid because segmentids must be unique to component not study
                                stmt = conn.prepareStatement("SELECT max(segmentid) from nc_expsegment WHERE nc_experiment_uniqueid = ? AND subjectid = ? AND componentid = ?");
                                stmt.setInt(1, experimentUniqueID);
                                stmt.setString(2, item.subjID);
                                stmt.setInt(3, item.visitID);
                                try {
                                    stmt.execute();
                                    ResultSet rs = stmt.getResultSet();
                                    if (rs.next()) {
                                        seriesID = rs.getInt(1) + 1;
                                    } else {
                                        seriesID = 1;
                                    }
                                } finally {
                                    stmt.close();
                                }
                                uniqueID = getNextSeq(conn);
                                stmt = conn.prepareStatement("INSERT INTO nc_expsegment(segmentid, componentid, nc_experiment_uniqueid, subjectid, uniqueid, tableid, owner, modtime, moduser, time_stamp, description, protocolversion, protocolid, studyid, name, istimeinterval, isbad) select ?, ?, ?, ?, ?, tableid, ?, now(), ?, ?, ?, ?, ?, ?, ?, false, false from nc_tableid WHERE tablename = 'NC_EXPSEGMENT'");
                                stmt.setInt(1, seriesID);
                                stmt.setInt(2, item.visitID);
                                stmt.setInt(3, experimentUniqueID);
                                stmt.setString(4, item.subjID);
                                stmt.setInt(5, uniqueID);
                                stmt.setInt(6, userID);
                                stmt.setInt(7, userID);
                                stmt.setTimestamp(8, new Timestamp((item.seriesModTime == null) ? System.currentTimeMillis() : item.studyModTime.getMillis()));
                                stmt.setString(9, item.seriesDesc);
                                stmt.setInt(10, protocolVersion);
                                stmt.setString(11, protocolID);
                                if (item.studyID == null) {
                                    stmt.setNull(12, java.sql.Types.BIGINT);
                                } else {
                                    stmt.setInt(12, item.studyID);
                                }
                                stmt.setString(13, item.seriesName);
                                try {
                                    stmt.executeUpdate();
                                } catch (SQLException e) {
                                    throw new NoteException("Error updating database", new ChainedSQLException(e));
                                } finally {
                                    stmt.close();
                                }
                            }
                        }
                    }
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error updating database", new ChainedSQLException(e));
        }
    }

    private ExtendedTupleObj getExperimentTupleByName(Connection conn, String expFullName) throws SQLException {
        ExtendedTupleObj retval = null;
        String expKey = "EXP" + expFullName;
        if (_tuplecache.containsKey(expKey)) {
            return _tuplecache.get(expKey);
        }
        PreparedStatement stmt = conn.prepareStatement("SELECT tableid, uniqueid FROM nc_experiment WHERE name=?");
        stmt.setString(1, expFullName);
        try {
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                Integer expTableID = rs.getInt(1);
                Integer expUniqueID = rs.getInt(2);
                retval = new ExtendedTupleObj(expTableID, expUniqueID, "nc_experiment", "nc_experiment");
                _tuplecache.put(expKey, retval);
            }
        } finally {
            try {
                stmt.close();
            } catch (SQLException e) {
                // ignore
            }
        }
        return retval;
    }

    private StoredTupleObj itemToTuple(ExperimentHierarchyItem item, Connection conn) throws NoteException {
        try {
            String baseTableName = null;
            List<Object> params = new ArrayList<Object>();
            List<String> whereList = new ArrayList<String>();
            String idcacheKey = null;
            String experimentColumn = "nc_experiment_uniqueid";
            if (item.subjID != null && !(item.expID != null && item.expName != null && item.visitID != null)) {
                // subject level item (not enough other info to uniquely identify visit)
                baseTableName = "nc_humansubject";
                whereList.add("subjectid=?");
                params.add(item.subjID);
                idcacheKey = "HIERTUPLESUBJ";
            } else if (item.expID != null && item.expName != null) {
                baseTableName = "nc_experiment";
                String fullExpName = item.expName + "__" + _df4.format((long) item.expID);
                Integer expUniqueID = null;
                Integer expTableID = null;
                ExtendedTupleObj expTuple = getExperimentTupleByName(conn, fullExpName);
                if (expTuple == null) {
                    return null;
                }
                expUniqueID = expTuple.tupleID;
                expTableID = expTuple.tableID;
                if (expUniqueID == null) {
                    // nothing of use
                    return null;
                }
                if (item.subjID == null || item.visitID == null) {
                    // won't go lower than experiment level, so return now
                    return new StoredTupleObj(expTableID, expUniqueID, "nc_experiment", "nc_experiment", item.visitSiteID);
                }
                baseTableName = "nc_expcomponent";
                whereList.add("subjectid=?");
                params.add(item.subjID);
                whereList.add("componentid=?");
                params.add(item.visitID);
                idcacheKey = "HIERTUPLECOMP";
                if (item.studyID != null) {
                    baseTableName = "nc_expstudy";
                    whereList.add("studyid=?");
                    params.add(item.studyID);
                    if (item.seriesName == null) {
                        experimentColumn = "experimentid";
                        idcacheKey = "HIERTUPLESTUDY";
                    }
                }
                if (item.seriesName != null) {
                    baseTableName = "nc_expsegment";
                    whereList.add("name=?");
                    params.add(item.seriesName);
                    idcacheKey = "HIERTUPLESEG";
                }
                whereList.add(0, experimentColumn + "=?");
                params.add(0, expUniqueID);
            } else {
                return null;
            }
            int numParams = params.size();
            for (int i = 0; i < numParams; i++) {
                idcacheKey += "-" + params.get(i).toString();
            }
            if (_idcache.containsKey(idcacheKey)) {
                _logger.log(Level.FINER, "Found key {0} in ID cache", idcacheKey);
                Tuple tuple = (Tuple) _idcache.get(idcacheKey);
                return new StoredTupleObj(tuple.tableID, tuple.uniqueID, baseTableName, baseTableName, item.visitSiteID);
            }
            StringBuilder sb = new StringBuilder();
            sb.append("SELECT tableid, uniqueid FROM ").append(baseTableName).append(" WHERE ");
            for (int i = 0; i < numParams; i++) {
                if (i > 0) {
                    sb.append(" AND ");
                }
                sb.append(whereList.get(i));
            }
            PreparedStatement stmt = conn.prepareStatement(sb.toString());
            try {
                for (int i = 0; i < numParams; i++) {
                    stmt.setObject(i + 1, params.get(i));
                }
                ResultSet rs = stmt.executeQuery();
                if (rs.next()) {
                    return new StoredTupleObj(rs.getInt(1), rs.getInt(2), baseTableName, baseTableName, item.visitSiteID);
                }
            } catch (SQLException e) {
                throw new NoteException("Error preparing or executing query", new ChainedSQLException(e));
            } finally {
                try {
                    stmt.close();
                } catch (SQLException e) {
                    // so what
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error converting item to tuple", e);
        }
        return null;
    }

    public void putNotes(List<? extends Note> noteList, String dbID, Connection conn, boolean createItems, ExternalProgressUpdater updater) throws NoteException {
        updater.setProgressStart("Writing to HID...");
        Map<ExperimentHierarchyItem, StoredTupleObj> tuplemap = new HashMap<ExperimentHierarchyItem, StoredTupleObj>();
        Map<StoredTupleObj, List<Note>> notemap = new HashMap<StoredTupleObj, List<Note>>();
        if (conn != _cachedConn) {
            _tuplecache.clear();
            _idcache.clear();
            _cachedExtendedTupleIDs.clear();
            _cachedConn = conn;
            cacheHierarchyTuples(conn);
            cacheStoredTupleByNoteID(conn);
        }
        int numItemsToWrite = 0;
        int numNotes = noteList.size();
        for (int i = 0; i < numNotes; i++) {
            updater.setProgress(i * 0.5 / numNotes, null);
            Note note = noteList.get(i);
            HashMap<StoredTupleObj, List<URIWithType>> itemmap = new HashMap<StoredTupleObj, List<URIWithType>>();
            for (URIWithType path : note.getPaths()) {
                ExperimentHierarchyItem item = null;
                try {
                    item = pathToItem(path);
                } catch (NoteException e) {
                    // XXX maybe we'll find a way to pass path errors back
                    continue;
                }
                fillInItemInsertIDs(item, conn);
                StoredTupleObj tuple = null;
                if (!tuplemap.containsKey(item)) {
                    tuple = itemToTuple(item, conn);
                    if (tuple == null) {
                        if (createItems) {
                            createItem(item, conn);
                            tuple = itemToTuple(item, conn);
                        }
                        if (tuple == null) {
                            continue;
                        }
                    }
                    tuplemap.put(item, tuple);
                } else {
                    tuple = tuplemap.get(item);
                }
                List<URIWithType> newPathList = null;
                if (itemmap.containsKey(tuple)) {
                    newPathList = itemmap.get(tuple);
                } else {
                    newPathList = new ArrayList<URIWithType>();
                    itemmap.put(tuple, newPathList);
                }
                newPathList.add(path);
            }
            for (StoredTupleObj tuple : itemmap.keySet()) {
                Note newNote = note.cloneWithoutFiles();
                List<URIWithType> pathList = itemmap.get(tuple);
                for (URIWithType uriwt : pathList) {
                    newNote.addPath(uriwt);
                }
                List<Note> newNoteList = null;
                if (notemap.containsKey(tuple)) {
                    newNoteList = notemap.get(tuple);
                } else {
                    newNoteList = new ArrayList<Note>();
                    notemap.put(tuple, newNoteList);
                    numItemsToWrite++;
                }
                newNoteList.add(newNote);
            }
        }
        NoteDBUser dbuser = getOrCreateNoteDBUser(conn);
        int numItemsWritten = 0;
        PreparedStatement tupleStmt = getStoredTuplePreparedStatement(conn);
        PreparedStatement updateTupleStmt = getUpdateStoredTuplePreparedStatement(conn);
        PreparedStatement timestampStmt = getTupleTimestampPreparedStatement(conn);
        PreparedStatement updateTimestampStmt = getUpdateTupleTimestampPreparedStatement(conn);
        PreparedStatement varcharStmt = getTupleVarcharPreparedStatement(conn);
        PreparedStatement updateVarcharStmt = getUpdateTupleVarcharPreparedStatement(conn);
        try {
            for (Map.Entry<StoredTupleObj, List<Note>> entry : notemap.entrySet()) {
                updater.setProgress(0.5 + (numItemsWritten / (6 * numItemsToWrite)), null);
                StoredTupleObj tuple = entry.getKey();
                Integer extendedTupleID = getExtendedTupleID(tuple, conn, true);
                for (Note note : entry.getValue()) {
                    StoredTupleObj newTuple = new StoredTupleObj(tuple.tableID, tuple.tupleID, tuple.tupleClass, tuple.tupleSubclass, tuple.siteID);
                    newTuple.storedTupleID = getStoredTupleFromNoteID(note.getID(), conn);
                    if (newTuple.storedTupleID == null) {
                        // sets newTuple.storedTupleID
                        putStoredTuple(newTuple, extendedTupleID, dbuser.userID, conn, tupleStmt);
                        putNoteToStatementBatch(newTuple, note, extendedTupleID, dbuser.userID, timestampStmt, varcharStmt);
                    } else {
                        updateStoredTuple(newTuple, extendedTupleID, dbuser.userID, conn, updateTupleStmt);
                        updateNoteToStatementBatch(newTuple, note, extendedTupleID, dbuser.userID, updateTimestampStmt, updateVarcharStmt);
                    }
                }
                numItemsWritten++;
            }
            try {
                timestampStmt.executeBatch();
                updater.setProgress(5.0 / 6.0, null);
                varcharStmt.executeBatch();
            } catch (SQLException e) {
                throw new NoteException("Error batch writing notes to database", new ChainedSQLException(e));
            }
        } finally {
            try {
                tupleStmt.close();
                timestampStmt.close();
                varcharStmt.close();
            } catch (SQLException e) {
                // ignore
            }
        }
        try {
            conn.close();
        } catch (SQLException e) {
            // oh well
        }
        updater.setProgress(1.0, null);
        updater.setProgressFinish("Complete.");
    }

    public List<Note> getNotes(String dbID, Connection conn, ExternalProgressUpdater updater) throws NoteException {
        updater.setProgressStart("Starting read...");
        Map<String, Note> noteidmap = new HashMap<String, Note>();
        String crosstab = null;
        try {
            {
                Statement stmt = conn.createStatement();
                ResultSet rs = stmt.executeQuery("select proname from pg_proc where proname = 'crosstab'");
                if (!rs.next() || rs.getString(1) == null) {
                    /* We do a full outer join (and then filter out NULLs) below
                     * to avoid a query optimization bug in PostgresQL */
                    crosstab = ""
                            + "(SELECT"
                            + "  vcid.storedtupleid AS storedtupleid,"
                            + "  vcid.datavalue AS id,"
                            + "  vcns.datavalue AS ns,"
                            + "  vcname.datavalue AS name,"
                            + "  vcauthor.datavalue AS author,"
                            + "  vctimestamp.datavalue AS timestamp,"
                            + "  vcvalue.datavalue AS value,"
                            + "  vccomment.comments AS comment,"
                            + "  vcpaths.datavalue AS paths"
                            + "  FROM"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='id') vcid,"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='namespace') vcns,"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='name') vcname,"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='author') vcauthor,"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='value') vcvalue,"
                            + "   (SELECT storedtupleid, comments FROM nc_tuplevarchar WHERE columnname='comment') vccomment,"
                            + "   (SELECT storedtupleid, datavalue FROM nc_tuplevarchar WHERE columnname='paths') vcpaths"
                            + "   FULL OUTER JOIN"
                            + "   nc_tupletimestamp vctimestamp ON vcpaths.storedtupleid = vctimestamp.storedtupleid"
                            + "  WHERE"
                            + "   vcid.storedtupleid = vcns.storedtupleid AND"
                            + "   vcid.storedtupleid = vcname.storedtupleid AND"
                            + "   vcid.storedtupleid = vcauthor.storedtupleid AND"
                            + "   vcid.storedtupleid = vcvalue.storedtupleid AND"
                            + "   vcid.storedtupleid = vccomment.storedtupleid AND"
                            + "   vcid.storedtupleid = vcpaths.storedtupleid AND"
                            + "   vctimestamp.columnname='timestamp' AND"
                            + "   vctimestamp.columnname IS NOT NULL AND"
                            + "   vcid.storedtupleid IS NOT NULL)"
                            + " AS report(storedtupleid, id, name, namespace, author, timestamp, value, comment, paths)";
                } else {
                    crosstab = ""
                            + "crosstab('SELECT storedtupleid, columnname, datavalue FROM nc_tuplevarchar WHERE columnname != ''comment'' AND columnname != ''paths'' UNION SELECT storedtupleid, columnname, comments FROM nc_tuplevarchar WHERE columnname = ''comment'' OR columnname = ''paths'' UNION SELECT storedtupleid, columnname, to_char(extract(''epoch'' FROM datavalue), ''999999999999999999999D999999'') FROM nc_tupletimestamp',"
                            + "         'SELECT DISTINCT columnname FROM nc_tuplecolumns WHERE columnname IN(''id'', ''namespace'', ''name'', ''author'', ''timestamp'', ''value'', ''comment'', ''paths'') ORDER BY columnname ASC')"
                            + " AS report(storedtupleid bigint, author varchar, comment varchar, id varchar, name varchar, namespace varchar, paths varchar, timestamp varchar, value varchar)";
                }
            }
            Statement stmt = conn.createStatement();
            ResultSet rs = stmt.executeQuery(""
                    + "SELECT"
                    + "  st.uniqueid, st.basetableid, st.basetupleid, hier.*, report.id, report.namespace, report.name, report.author, report.timestamp, report.value, report.comment, report.paths"
                    + " FROM"
                    + "  nc_extendedtuple et"
                    + "  JOIN nc_storedtuple st"
                    + "   ON et.name='Note' AND st.extendedtupleid=et.uniqueid"
                    + "  JOIN " + crosstab
                    + "   ON report.storedtupleid=st.uniqueid"
                    + "  LEFT JOIN ("
                    + "   SELECT ex.tableid, ex.uniqueid, ex.baseuri, ex.name, NULL::varchar, NULL::varchar, NULL::bigint, NULL::varchar, NULL::bigint, NULL::varchar, NULL::bigint, NULL::varchar"
                    + "    FROM"
                    + "     nc_experiment ex"
                    + "   UNION ALL"
                    + "   SELECT subj.tableid, subj.uniqueid, NULL::varchar, NULL::varchar, subj.subjectid, subj.name, NULL::bigint, NULL::varchar, NULL::bigint, NULL::varchar, NULL::bigint, NULL::varchar"
                    + "    FROM"
                    + "     nc_humansubject subj"
                    + "   UNION ALL"
                    + "   SELECT comp.tableid, comp.uniqueid, ex.baseuri, ex.name, subj.subjectid, subj.name, comp.componentid, comp.name, NULL::bigint, NULL::varchar, NULL::bigint, NULL::varchar"
                    + "    FROM"
                    + "     nc_expcomponent comp"
                    + "     JOIN nc_experiment ex"
                    + "      ON ex.uniqueid=comp.nc_experiment_uniqueid"
                    + "     JOIN nc_humansubject subj"
                    + "      ON subj.subjectid=comp.subjectid"
                    + "   UNION ALL"
                    + "   SELECT study.tableid, study.uniqueid, ex.baseuri, ex.name, subj.subjectid, subj.name, comp.componentid, comp.name, study.studyid, study.name, NULL::bigint, NULL::varchar"
                    + "    FROM"
                    + "     nc_expstudy study"
                    + "     JOIN nc_experiment ex"
                    + "      ON ex.uniqueid=study.experimentid"
                    + "     JOIN nc_humansubject subj"
                    + "      ON subj.subjectid=study.subjectid"
                    + "     JOIN nc_expcomponent comp"
                    + "      ON comp.nc_experiment_uniqueid=study.experimentid AND comp.subjectid=study.subjectid AND comp.componentid=study.componentid"
                    + "   UNION ALL"
                    + "   SELECT segment.tableid, segment.uniqueid, ex.baseuri, ex.name, subj.subjectid, subj.name, comp.componentid, comp.name, study.studyid, study.name, segment.segmentid, segment.name"
                    + "    FROM"
                    + "     nc_expsegment segment"
                    + "     JOIN nc_experiment ex"
                    + "      ON ex.uniqueid=segment.nc_experiment_uniqueid"
                    + "     JOIN nc_humansubject subj"
                    + "      ON subj.subjectid=segment.subjectid"
                    + "     JOIN nc_expcomponent comp"
                    + "      ON comp.nc_experiment_uniqueid=segment.nc_experiment_uniqueid AND comp.subjectid=segment.subjectid AND comp.componentid=segment.componentid"
                    + "     "
                    + "     LEFT JOIN nc_expstudy study"
                    + "      ON (study.experimentid=segment.nc_experiment_uniqueid AND study.subjectid=segment.subjectid AND study.componentid=segment.componentid)"
                    + "  ) hier"
                    + "   ON hier.tableid=st.basetableid AND hier.uniqueid=st.basetupleid");
            int numRows = -1;
            if (rs.getType() != ResultSet.TYPE_FORWARD_ONLY) {
                rs.last();
                numRows = rs.getRow();
                rs.beforeFirst();
            }
            while (rs.next()) {
                if (numRows != -1) {
                    updater.setProgress(rs.getRow() * 1.0 / numRows, null);
                }
                // st.uniqueid, st.basetableid, st.basetupleid, hier.tableid, hier.uniqueid, ex.baseuri, ex.name, subj.subjectid, subj.name, comp.componentid, comp.name, study.studyid, study.name, segment.segmentid, segment.name, report.id, report.namespace, report.name, report.author, report.timestamp, report.value, report.comment, report.paths
                Integer storedtupleid = rs.getInt(1);
                Integer basetableid = rs.getInt(2);
                Integer basetupleid = rs.getInt(3);
                String expBaseURI = rs.getString(6);
                String expName = rs.getString(7);
                String subjectID = rs.getString(8);
                String subjectName = rs.getString(9);
                Integer visitID = rs.getInt(10);
                String visitName = rs.getString(11);
                Integer studyID = rs.getInt(12);
                String studyName = rs.getString(13);
                Integer seriesID = rs.getInt(14);
                String seriesName = rs.getString(15);
                String noteID = rs.getString(16);
                String ns = rs.getString(17);
                String name = rs.getString(18);
                String author = rs.getString(19);
                String timeStampStr = rs.getString(20);
                String value = rs.getString(21);
                String comment = rs.getString(22);
                String pathsstr = rs.getString(23);
                DateTime timeStamp = new DateTime((long) (Double.parseDouble(timeStampStr) * 1000));

                if (pathsstr == null) {
                    // every note should really have at least one path, but if not, we fashion a note from what we can
                    Integer expID = null;
                    int sepind = expName.indexOf("__");
                    if (sepind != -1) {
                        expID = Integer.valueOf(expName.substring(sepind + 2));
                        expName = expName.substring(0, sepind);
                    }
                    // URI expBaseURI_, Integer expID_, String expName_, DateTime expModTime_, String expDesc_, String subjID_, String subjName_, DateTime subjModTime_, String subjDesc_, Integer visitSiteID_, Integer visitID_, String visitName_, DateTime visitModTime_, String visitDesc_, Integer studyID_, String studyName_, DateTime studyModTime_, String studyDesc_, Integer seriesID_, String seriesName_, DateTime seriesModTime_, String seriesDesc_
                    try {
                        ExperimentHierarchyItem item = new ExperimentHierarchyItem(dbID + "#" + storedtupleid, new URI(expBaseURI), expID, expName, null, null, subjectID, subjectName, null, null, null, visitID, visitName, null, null, studyID, studyName, null, null, seriesID, seriesName, null, null);
                        URIWithType uriwt = itemToPath(item);
                        pathsstr = uriwt.toString();
                    } catch (URISyntaxException e) {
                        throw new NoteException("Error parsing URI: '" + expBaseURI + "'", e);
                    }
                }
                List<URIWithType> paths = new ArrayList<URIWithType>();
                while (true) {
                    int strlen = pathsstr.length();
                    if (strlen == 0) {
                        break;
                    }
                    if (strlen <= 5) {
                        throw new NoteException("Error parsing path string: '" + pathsstr + "'");
                    }
                    int type = URIWithType.TYPE_UNKNOWN;
                    String prefix = pathsstr.substring(0, 5);
                    int pathstart = -1;
                    if (prefix.equals("dir: ")) {
                        type = URIWithType.TYPE_DIR;
                        pathstart = 5;
                    } else if (prefix.equals("file:") && strlen > 5 && pathsstr.charAt(5) == ' ') {
                        type = URIWithType.TYPE_FILE;
                        pathstart = 6;
                    } else {
                        throw new NoteException("Error parsing path string: '" + pathsstr + "'");
                    }
                    int pathend = pathsstr.indexOf('\n', pathstart);
                    int newentryind = pathend + 1;
                    if (pathend == -1) {
                        pathend = strlen;
                        newentryind = strlen;
                    }
                    try {
                        paths.add(new URIWithType(type, pathsstr.substring(pathstart, pathend)));
                    } catch (URISyntaxException e) {
                        throw new NoteException("Error parsing URI: '" + pathsstr.substring(pathstart, pathend) + "'", e);
                    }
                    pathsstr = pathsstr.substring(newentryind);
                }
                if (noteidmap.containsKey(noteID)) {
                    // got a note with the same noteID, so we assume the contents are the same.  Just add the paths
                    Note note = noteidmap.get(noteID);
                    note.getPaths().addAll(paths);
                } else {
                    // String namespace, String name, String value, String comment, String author, DateTime timeStamp, ArrayList<URIWithType> files
                    noteidmap.put(noteID, new Note(noteID, ns, name, value, comment, author, timeStamp, paths));
                }
            }
        } catch (SQLException e) {
            throw new NoteException("Error reading notes from database", e);
        }
        List<Note> retval = new ArrayList<Note>(noteidmap.values());
        updater.setProgressStart("Finished read.");
        return retval;
    }
}
