/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.notes;

import java.io.IOException;
import java.io.Reader;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.joda.time.format.ISODateTimeFormat;
import org.joda.time.DateTime;

/**
 * This class represents a generic annotation.
 * @author gadde
 */
public class Note implements Cloneable, Comparable<Note> {

    private final static SimpleDateFormat ISO8601FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
    private final static char FIELDSEP = '|';
    private final static String FIELDSEPSTR = "" + FIELDSEP;
    private final static byte[] FIELDSEPBYTES = new byte[]{(byte) FIELDSEP};
    private final static String NOTEENDSTR = FIELDSEPSTR + FIELDSEPSTR;
    public final static String NULLVALUE = ""; // we use this instead of null to avoid "if null" tests everywhere
    private String _ID = null;
    private String _namespace = null;
    private String _name = null;
    private String _value = null;
    private String _comment = null;
    private String _author = null;
    private DateTime _timeStamp = null;
    private List<URIWithType> _files = null;

    public Note() {
        this(null, null, null, null, null, null, null, (ArrayList<URIWithType>) null);
    }

    public Note(String ID, String namespace, String name, String value, String comment, String author, DateTime timeStamp, URIWithType file) {
        this(ID, namespace, name, value, comment, author, timeStamp, (ArrayList<URIWithType>) null);
        if (file != null) {
            _files.add(file);
        }
    }

    public Note(String ID, String namespace, String name, String value, String comment, String author, DateTime timeStamp, List<URIWithType> files) {
        if (ID == null) {
            String hostname = null;
            try {
                hostname = InetAddress.getLocalHost().getHostName();
            } catch (UnknownHostException ex) {
                hostname = "(unknown host)";
            }
            _ID = "note://" + hostname + "/" + ManagementFactory.getRuntimeMXBean().getName() + "#" + System.currentTimeMillis();
        } else {
            _ID = ID;
        }
        if (namespace == null) {
            _namespace = NULLVALUE;
        } else {
            _namespace = namespace;
        }
        if (name == null) {
            _name = NULLVALUE;
        } else {
            _name = name;
        }
        if (value == null) {
            _value = NULLVALUE;
        } else {
            _value = value;
        }
        if (comment == null) {
            _comment = NULLVALUE;
        } else {
            _comment = comment;
        }
        if (author == null) {
            _author = NULLVALUE;
        } else {
            _author = author;
        }
        if (timeStamp == null) {
            _timeStamp = new DateTime();
        } else {
            _timeStamp = timeStamp;
        }
        if (files != null) {
            _files = new ArrayList<URIWithType>(files);
        } else {
            _files = new ArrayList<URIWithType>();
        }
    }

    public boolean equalsWithoutFiles(Note note) {
        if (note == null) {
            return false;
        }
        if (!_ID.equals(note._ID)) {
            return false;
        }
        if (_namespace == NULLVALUE) {
            if (note._namespace != NULLVALUE) {
                return false;
            }
        } else if (!_namespace.equals(note._namespace)) {
            return false;
        }
        if (_name == NULLVALUE) {
            if (note._name != NULLVALUE) {
                return false;
            }
        } else if (!_name.equals(note._name)) {
            return false;
        }
        if (_value == NULLVALUE) {
            if (note._value != NULLVALUE) {
                return false;
            }
        } else if (!_value.equals(note._value)) {
            return false;
        }
        if (_comment == NULLVALUE) {
            if (note._comment != NULLVALUE) {
                return false;
            }
        } else if (!_comment.equals(note._comment)) {
            return false;
        }
        if (_author == NULLVALUE) {
            if (note._author != NULLVALUE) {
                return false;
            }
        } else if (!_author.equals(note._author)) {
            return false;
        }
        if (!_timeStamp.equals(note._timeStamp)) {
            return false;
        }
        return true;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (obj.getClass() != getClass()) {
            return false;
        }
        Note note = (Note) obj;
        if (!equalsWithoutFiles(note)) {
            return false;
        }
        if (!_files.equals(note._files)) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 19 * hash + (this._ID != null ? this._ID.hashCode() : 0);
        hash = 19 * hash + (this._namespace != null ? this._namespace.hashCode() : 0);
        hash = 19 * hash + (this._name != null ? this._name.hashCode() : 0);
        hash = 19 * hash + (this._value != null ? this._value.hashCode() : 0);
        hash = 19 * hash + (this._comment != null ? this._comment.hashCode() : 0);
        hash = 19 * hash + (this._author != null ? this._author.hashCode() : 0);
        hash = 19 * hash + (this._timeStamp != null ? this._timeStamp.hashCode() : 0);
        hash = 19 * hash + (this._files != null ? this._files.hashCode() : 0);
        return hash;
    }

    @Override
    public Note clone() {
        return new Note(_ID, _namespace, _name, _value, _comment, _author, _timeStamp, _files);
    }

    public Note cloneWithoutFiles() {
        return new Note(_ID, _namespace, _name, _value, _comment, _author, _timeStamp, (ArrayList<URIWithType>) null);
    }

    public String getID() {
        return _ID;
    }

    public void setID(String ID) {
        _ID = ID;
    }

    /**
     * @return namespace for this {@link Note}
     */
    public String getNamespace() {
        return _namespace;
    }

    /**
     * @param namespace namespace for this {@link Note}
     */
    public void setNamespace(String namespace) {
        _namespace = namespace;
    }

    /**
     * @return name for this {@link Note}
     */
    public String getName() {
        return _name;
    }

    /**
     * @param name name for this {@link Note}
     */
    public void setName(String name) {
        _name = name;
    }

    /**
     * @return value for this {@link Note}
     */
    public String getValue() {
        return _value;
    }

    /**
     * @param value value for this {@link Note}
     */
    public void setValue(String _value) {
        this._value = _value;
    }

    /**
     * @return comment for this {@link Note}
     */
    public String getComment() {
        return _comment;
    }

    /**
     * @param comment comment for this {@link Note}
     */
    public void setComment(String comment) {
        _comment = comment;
    }

    /**
     * @return author for this {@link Note}
     */
    public String getAuthor() {
        return _author;
    }

    /**
     * @param author author for this {@link Note}
     */
    public void setAuthor(String author) {
        _author = author;
    }

    /**
     * @return time stamp for this {@link Note}
     */
    public DateTime getTimeStamp() {
        return _timeStamp;
    }

    /**
     * @param timeStamp time stamp for this {@link Note}
     */
    public void setTimeStamp(DateTime timeStamp) {
        _timeStamp = timeStamp;
    }

    /**
     * @return paths for this {@link Note}
     */
    public List<URIWithType> getPaths() {
        return _files;
    }

    /**
     * @param paths paths for this {@link Note}
     */
    public void setPaths(List<URIWithType> paths) {
        _files = paths;
    }

    public void addPath(URIWithType path) {
        _files.add(path);
    }

    /**
     * Same as {@code encodeField(field, null)}
     */
    public static String encodeField(String field) {
        return encodeField(field, null);
    }

    /**
     * Encode a Note field so it can be stored in a Notes file.
     * @param field a {@link String} representing the field to be encoded (may need to be further escaped to meet other requirements for embedding in particular formats)
     * @param specialChars a list of byte values that must be escaped (in addition to the bytes corresponding to non-ASCII-printable characters and '%' that are already escaped)
     * @return a {@link String} representing a UTF-8 encoded byte stream, but with each byte represented as one or more characters -- any byte that is ASCII-printable (i.e. 0x21 <= byte <= 0x7e, except '%' or any char specified in specialChars) maintains the same value in a single char, but any other byte (as well as a byte corresponding to ASCII '%') is represented by a sequence of three characters ('%' A B) where A and B are the most- and least-significant digit of the byte represented as hexadecimal.
     */
    public static String encodeField(String field, byte[] specialChars) {
        if (field == NULLVALUE) {
            return "";
        } else if (field.length() == 0) {
            return "%c0%80"; // special case for (non-null) empty string
        }
        final Charset cs = Charset.forName("UTF-8");
        byte[] sortedSpecialChars = null;
        if (specialChars != null) {
            sortedSpecialChars = Arrays.copyOf(specialChars, specialChars.length);
            Arrays.sort(sortedSpecialChars);
        }
        ByteBuffer bb = cs.encode(field);
        StringBuilder sb = new StringBuilder();
        int bblen = bb.limit();
        for (int i = 0; i < bblen; i++) {
            byte b = bb.get(i);
            if (b < 0x20 || b > 0x7e || b == (byte) '%' || (sortedSpecialChars != null && Arrays.binarySearch(sortedSpecialChars, b) >= 0)) {
                sb.append('%');
                sb.append(Character.forDigit((((int) b) & 0xFF) / 16, 16));
                sb.append(Character.forDigit((((int) b) & 0xFF) % 16, 16));
            } else {
                sb.append((char) b);
            }
        }
        return sb.toString();
    }

    /**
     * Decode a Note field from a Notes file.
     * @param field a {@link String} as encoded using {@link encodeValue}, where every character must be an ASCII-printable character.
     * @return a {@link String} represented by the encoded string, calculated by copying each (single-byte) character to a byte stream but converting any ('%' HEXDIGIT HEXDIGIT) sequence of characters to a 8-bit byte represented by the hexadecimal digits, then taking the resulting byte stream and decoding it as a UTF-8-encoded stream.
     */
    public static String decodeField(String field) {
        final Charset cs = Charset.forName("UTF-8");
        char[] valuechars = field.toCharArray();
        int valuelen = valuechars.length;
        int curbyte = 0;
        byte[] valuebytes = new byte[valuelen];
        for (int i = 0; i < valuelen; i++) {
            char c = valuechars[i];
            if (c < 0x0020 || c > 0x007e) {
                throw new IllegalArgumentException("Bad character in (encoded) note value");
            }
            if (c == '%') {
                if (i + 2 >= valuelen) {
                    throw new IllegalArgumentException("Note value ended in the middle of an encoded character");
                }
                int ic2 = Character.digit(valuechars[i + 1], 16);
                int ic3 = Character.digit(valuechars[i + 2], 16);
                if (ic2 == -1 || ic3 == -1) {
                    throw new IllegalArgumentException("Badly encoded character in note value");
                }
                valuebytes[curbyte] = (byte) ((ic2 * 16) + ic3);
                i += 2;
                curbyte++;
            } else {
                valuebytes[curbyte] = (byte) c;
                curbyte++;
            }
        }
        if (valuelen == 0) {
            return null;
        } else if (valuelen == 2 && valuebytes[0] == 0xC0 && valuebytes[1] == 0x80) {
            // special case for (non-null) empty string
            return "";
        }
        return cs.decode(ByteBuffer.wrap(valuebytes, 0, curbyte)).toString();
    }

    /**
     * Populates the current object with info from a string-exported {@link Note}.
     * @param input {@link String} from which {@link Note} info should be extracted.  Characters corresponding to the extracted note will be deleted (i.e. consumed) from the input StringBuilder.
     * @return a {@link String} representing the unused portion of the input.  If a valid note was not found, then return <code>null</code>;
     */
    public static Note consumeFromString(StringBuilder input, URI baseURI) {
        int inputlen = input.length();
        int curindex = 0;
        if (input.length() < 6 || !"NOTE: ".equals(input.substring(0, 6))) {
            return null;
        }
        curindex = 6;
        
        int delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String ID = decodeField(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
        }
        if (delimind == -1) {
            return null;
        }
        DateTime timeStamp = ISODateTimeFormat.dateTimeParser().parseDateTime(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String namespace = decodeField(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String name = decodeField(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String author = decodeField(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String value = decodeField(input.substring(curindex, delimind));
        curindex = delimind + 1;
        
        delimind = curindex;
        while ((delimind = input.indexOf(FIELDSEPSTR, delimind)) != -1) {
            if (delimind == curindex || input.charAt(delimind - 1) != '\\') {
                break;
            }
            delimind++;
        }
        if (delimind == -1) {
            return null;
        }
        String comment = decodeField(input.substring(curindex, delimind));
        boolean dopaths = false;
        curindex = delimind + 1;
        if (curindex >= inputlen) {
            return null;
        }
        if (input.charAt(curindex) == '\n') {
            dopaths = true;
            curindex += 1;
        } else if (curindex + 2 <= inputlen && "\r\n".equals(input.substring(curindex, curindex + 2))) {
            dopaths = true;
            curindex += 2;
        } else if (input.charAt(curindex) == FIELDSEP) {
            curindex += 1;
        }
        ArrayList<URIWithType> paths = null;
        if (dopaths) {
            paths = new ArrayList<URIWithType>();
            while (true) {
                if (curindex + 2 <= inputlen && NOTEENDSTR.equals(input.substring(curindex, curindex + 2))) {
                    curindex += 2;
                    break;
                }
                if (curindex + 2 > inputlen || input.charAt(curindex + 1) != ' ') {
                    return null;
                }
                int typechar = input.charAt(curindex);
                int type = URIWithType.TYPE_UNKNOWN;
                if (typechar == 'F') {
                    type = URIWithType.TYPE_FILE;
                } else {
                    type = URIWithType.TYPE_DIR;
                }
                curindex += 2;
                int nlind = input.indexOf("\n", curindex);
                if (nlind == -1) {
                    return null;
                }
                int pathend = nlind;
                if (input.charAt(nlind - 1) == '\r') {
                    pathend = nlind - 1;
                }
                String path = input.substring(curindex, pathend);
                try {
                    URIWithType uri = new URIWithType(type, (path.length() > 0 && path.charAt(0) != '/') ? baseURI : null, new URI(path));
                    paths.add(uri);
                } catch (URISyntaxException e) {
                    // XXX skip it for now
                }
                curindex = nlind + 1;
            }
        }
        while (curindex < inputlen) {
            if (input.charAt(curindex) == '\r') {
                curindex++;
            }
            if (curindex < inputlen && input.charAt(curindex) == '\n') {
                curindex++;
            } else {
                break;
            }
        }
        input.delete(0, curindex);
        if ((paths == null || paths.isEmpty()) && baseURI != null) {
            if (paths == null) {
                paths = new ArrayList<URIWithType>();
            }
            paths.add(new URIWithType(URIWithType.TYPE_DIR, baseURI));
        }
        return new Note(ID, namespace, name, value, comment, author, timeStamp, paths);
    }

    private static class MyReaderBuffer {

        private static final int READ_CHUNK_SIZE = 32 * 1024;
        private static final char[] _buf = new char[READ_CHUNK_SIZE];
        private Reader _reader;
        private StringBuffer _sb;

        public MyReaderBuffer(Reader reader) {
            _reader = reader;
            _sb = new StringBuffer();
        }

        private boolean readNextChunk() throws IOException {
            int numRead = _reader.read(_buf);
            if (numRead == -1) {
                return false;
            }
            _sb.append(_buf, 0, numRead);
            return true;
        }

        public boolean isAvailable(int numChars) throws IOException {
            while (numChars >= _sb.length()) {
                if (!readNextChunk()) {
                    return false;
                }
            }
            return true;
        }

        public String substring(int start, int end) {
            return _sb.substring(start, end);
        }

        public int indexOf(String str, int fromIndex) throws IOException {
            int strLength = str.length();
            int ind;
            while ((ind = _sb.indexOf(str, fromIndex)) == -1) {
                int oldBufLength = _sb.length();
                if (!readNextChunk()) {
                    break;
                }
                fromIndex = oldBufLength - (strLength - 1);
            }
            return ind;
        }

        public char charAt(int index) {
            return _sb.charAt(index);
        }

        public void delete(int start, int end) {
            _sb.delete(start, end);
        }
    }

    public static class NoteIterator implements Iterator<Note> {

        private final URI _baseURI;
        private MyReaderBuffer _input;
        private Note _nextNote;

        public NoteIterator(Reader reader, URI baseURI) {
            _baseURI = baseURI;
            _input = new MyReaderBuffer(reader);
            _nextNote = null;
            findNextNote();
        }

        private void findNextNote() {
            _nextNote = null;
            try {
                if (!_input.isAvailable(6)) {
                    return;
                }
                int curindex = 0;
                if (!"NOTE: ".equals(_input.substring(0, 6))) {
                    return;
                }
                curindex = 6;
                int delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String ID = decodeField(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                }
                if (delimind == -1) {
                    return;
                }
                DateTime timeStamp = ISODateTimeFormat.dateTimeParser().parseDateTime(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String namespace = decodeField(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String name = decodeField(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String author = decodeField(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String value = decodeField(_input.substring(curindex, delimind));
                curindex = delimind + 1;
                delimind = curindex;
                while ((delimind = _input.indexOf(FIELDSEPSTR, delimind)) != -1) {
                    if (delimind == curindex || _input.charAt(delimind - 1) != '\\') {
                        break;
                    }
                    delimind++;
                }
                if (delimind == -1) {
                    return;
                }
                String comment = decodeField(_input.substring(curindex, delimind));
                boolean dopaths = false;
                curindex = delimind + 1;
                if (!_input.isAvailable(curindex)) {
                    return;
                }
                if (_input.charAt(curindex) == '\n') {
                    dopaths = true;
                    curindex += 1;
                } else if (_input.isAvailable(curindex + 2) && "\r\n".equals(_input.substring(curindex, curindex + 2))) {
                    dopaths = true;
                    curindex += 2;
                } else if (_input.charAt(curindex) == FIELDSEP) {
                    curindex += 1;
                }
                ArrayList<URIWithType> paths = null;
                if (dopaths) {
                    paths = new ArrayList<URIWithType>();
                    while (true) {
                        if (_input.isAvailable(curindex + 2) && NOTEENDSTR.equals(_input.substring(curindex, curindex + 2))) {
                            curindex += 2;
                            break;
                        }
                        if (!_input.isAvailable(curindex + 2) || _input.charAt(curindex + 1) != ' ') {
                            return;
                        }
                        int typechar = _input.charAt(curindex);
                        int type = URIWithType.TYPE_UNKNOWN;
                        if (typechar == 'F') {
                            type = URIWithType.TYPE_FILE;
                        } else {
                            type = URIWithType.TYPE_DIR;
                        }
                        curindex += 2;
                        int nlind = _input.indexOf("\n", curindex);
                        if (nlind == -1) {
                            return;
                        }
                        int pathend = nlind;
                        if (_input.charAt(nlind - 1) == '\r') {
                            pathend = nlind - 1;
                        }
                        String path = _input.substring(curindex, pathend);
                        try {
                            URIWithType uri = new URIWithType(type, (path.length() > 0 && path.charAt(0) != '/') ? _baseURI : null, new URI(path));
                            paths.add(uri);
                        } catch (URISyntaxException e) {
                            // XXX skip it for now
                        }
                        curindex = nlind + 1;
                    }
                }
                while (_input.isAvailable(curindex)) {
                    if (_input.charAt(curindex) == '\r') {
                        curindex++;
                    }
                    if (_input.isAvailable(curindex) && _input.charAt(curindex) == '\n') {
                        curindex++;
                    } else {
                        break;
                    }
                }
                _input.delete(0, curindex);
                if ((paths == null || paths.isEmpty()) && _baseURI != null) {
                    if (paths == null) {
                        paths = new ArrayList<URIWithType>();
                    }
                    paths.add(new URIWithType(URIWithType.TYPE_DIR, _baseURI));
                }
                _nextNote = new Note(ID, namespace, name, value, comment, author, timeStamp, paths);
            } catch (IOException ex) {
                return;
            }

        }

        public boolean hasNext() {
            return _nextNote != null;
        }

        public Note next() {
            Note retval = _nextNote;
            findNextNote();
            return retval;
        }

        public void remove() {
            throw new UnsupportedOperationException("Remove not supported.");
        }
    }

    /**
     * Returns a {@link Note} with info from the first string-exported {@link Note} in the given {@link Reader}.
     * @param input {@link String} from which {@link Note} info should be extracted.  Characters corresponding to the extracted note will be deleted (i.e. consumed) from the input StringBuilder.
     * @return a {@link String} representing the unused portion of the input.  If a valid note was not found, then return <code>null</code>;
     */
    public static NoteIterator getIteratorFromReader(Reader reader, URI baseURI) {
        return new NoteIterator(reader, baseURI);
    }

    public String exportToString() {
        String esc_ID = encodeField(_ID, FIELDSEPBYTES);
        String esc_timeStamp = ISODateTimeFormat.dateTime().print(_timeStamp);
        String esc_ns = encodeField(_namespace, FIELDSEPBYTES);
        String esc_name = encodeField(_name, FIELDSEPBYTES);
        String esc_author = encodeField(_author, FIELDSEPBYTES);
        String esc_value = encodeField(_value, FIELDSEPBYTES);
        String esc_comment = encodeField(_comment, FIELDSEPBYTES);
        String notestr = "NOTE: " + esc_ID + FIELDSEPSTR + esc_timeStamp + FIELDSEPSTR + esc_ns + FIELDSEPSTR + esc_name + FIELDSEPSTR + esc_author + FIELDSEPSTR + esc_value + FIELDSEPSTR + esc_comment;
        if (_files.size() > 0) {
            notestr += FIELDSEPSTR + "\n";
            for (int i = 0; i < _files.size(); i++) {
                URIWithType file = _files.get(i);
                String typechar = "?";
                int type = file.getType();
                if (type == URIWithType.TYPE_FILE) {
                    typechar = "F";
                } else if (type == URIWithType.TYPE_DIR) {
                    typechar = "D";
                }
                notestr += typechar + " " + _files.get(i).getURI().toString() + "\n";
            }
        }
        notestr += NOTEENDSTR + "\n";
        return notestr;
    }

    public int compareTo(Note other) {
        int retval = 0;
        ArrayList<URIWithType> thisfilessorted = new ArrayList<URIWithType>(this._files);
        ArrayList<URIWithType> otherfilessorted = new ArrayList<URIWithType>(other._files);
        Collections.<URIWithType>sort(thisfilessorted);
        Collections.<URIWithType>sort(otherfilessorted);
        int thisnumfiles = thisfilessorted.size();
        int othernumfiles = otherfilessorted.size();
        for (int i = 0; i < thisnumfiles && i < othernumfiles; i++) {
            retval = thisfilessorted.get(i).compareTo(otherfilessorted.get(i));
            if (retval != 0) {
                return retval;
            }
        }
        retval = (thisnumfiles - othernumfiles);
        if (retval != 0) {
            retval = retval / Math.abs(retval);
            return retval;
        }
        retval = this._namespace.compareTo(other._namespace);
        if (retval != 0) {
            return retval;
        }
        retval = this._name.compareTo(other._name);
        if (retval != 0) {
            return retval;
        }
        retval = this._name.compareTo(other._name);
        if (retval != 0) {
            return retval;
        }
        retval = this._author.compareTo(other._author);
        if (retval != 0) {
            return retval;
        }
        retval = this._timeStamp.compareTo(other._timeStamp);
        if (retval != 0) {
            return retval;
        }
        retval = this._value.compareTo(other._value);
        if (retval != 0) {
            return retval;
        }
        retval = (this._files.size() - other._files.size());
        if (retval != 0) {
            retval = retval / Math.abs(retval);
            return retval;
        }
        retval = this._comment.compareTo(other._comment);
        return retval;
    }
}
