/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.notes;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class maps {@link Note} namespaces or (namespace, name) pairs to one or more (optionally user-specific) privileges, which can be to allow many notes with that identifier, or only one, or none.  This is intended to allow other classes to control user write access to notes.
 * @author gadde
 */
public class NotePrivileges {

    private static final Logger _logger = Logger.getLogger(HID.class.getName());
    public static final int PRIV_UNKNOWN = 0;
    public static final int PRIV_ALLOW_MANY = 1;
    public static final int PRIV_ALLOW_ONE = 2;
    public static final int PRIV_REJECT = 3;
    private ArrayList<Entry> _entries = null;

    public static class Entry {

	public String namespace = null;
	public String name = null;
	public ArrayList<UserPrivilege> privs = null;

	public Entry(String namespace_, String name_) {
	    namespace = namespace_;
	    name = name_;
	    privs = new ArrayList<UserPrivilege>();
	}

	@Override
	public boolean equals(Object obj) {
	    if (obj == null) {
		return false;
	    }
	    if (getClass() != obj.getClass()) {
		return false;
	    }
	    final Entry other = (Entry) obj;
	    if ((this.namespace == null) ? (other.namespace != null) : !this.namespace.equals(other.namespace)) {
		return false;
	    }
	    if ((this.name == null) ? (other.name != null) : !this.name.equals(other.name)) {
		return false;
	    }
	    if (this.privs != other.privs && (this.privs == null || !this.privs.equals(other.privs))) {
		return false;
	    }
	    return true;
	}

	@Override
	public int hashCode() {
	    int hash = 5;
	    hash = 17 * hash + (this.namespace != null ? this.namespace.hashCode() : 0);
	    hash = 17 * hash + (this.name != null ? this.name.hashCode() : 0);
	    hash = 17 * hash + (this.privs != null ? this.privs.hashCode() : 0);
	    return hash;
	}
    }

    public static class UserPrivilege {

	public String user = null;
	public int priv = PRIV_UNKNOWN;

	public UserPrivilege(String user_, int priv_) {
	    if (priv_ != PRIV_UNKNOWN && priv_ != PRIV_ALLOW_MANY && priv_ != PRIV_ALLOW_ONE && priv_ != PRIV_REJECT) {
		throw new IllegalArgumentException("NotePrivileges.UserPrivilege.priv value of " + (new Integer(priv_)).toString() + " is not understood");
	    }
	    user = user_;
	    priv = priv_;
	}

	@Override
	public boolean equals(Object obj) {
	    if (obj == null) {
		return false;
	    }
	    if (getClass() != obj.getClass()) {
		return false;
	    }
	    final UserPrivilege other = (UserPrivilege) obj;
	    if ((this.user == null) ? (other.user != null) : !this.user.equals(other.user)) {
		return false;
	    }
	    if (this.priv != other.priv) {
		return false;
	    }
	    return true;
	}

	@Override
	public int hashCode() {
	    int hash = 7;
	    hash = 11 * hash + (this.user != null ? this.user.hashCode() : 0);
	    hash = 11 * hash + this.priv;
	    return hash;
	}
    }

    public NotePrivileges() {
	_entries = new ArrayList<Entry>();
    }

    @Override
    public boolean equals(Object obj) {
	if (obj == null) {
	    return false;
	}
	if (getClass() != obj.getClass()) {
	    return false;
	}
	final NotePrivileges other = (NotePrivileges) obj;
	if (this._entries != other._entries && (this._entries == null || !this._entries.equals(other._entries))) {
	    return false;
	}
	return true;
    }

    @Override
    public int hashCode() {
	int hash = 7;
	hash = 61 * hash + (this._entries != null ? this._entries.hashCode() : 0);
	return hash;
    }

    public void addPrivilege(String ns, String name, String user, int priv) {
	Entry newentry = null;
	Entry entry = null;
	if (!_entries.isEmpty()) {
	    Entry lastentry = _entries.get(_entries.size() - 1);
	    // we first comparing strings using == to cover the case where they might be the same object or both null
	    if ((ns == lastentry.namespace || (ns != null && ns.equals(lastentry.namespace)))
		    && (name == lastentry.name || (name != null && name.equals(lastentry.name)))) {
		entry = lastentry;
	    }
	}
	if (entry == null) {
	    newentry = new Entry(ns, name);
	    entry = newentry;
	}
	entry.privs.add(new UserPrivilege(user, priv));
	if (newentry != null) {
	    _entries.add(newentry);
	}
    }

    /**
     * Get the first privilege matching the given parameters.
     * @param ns
     * @param name
     * @param user
     * @return {@code PRIV_UNKNOWN}, {@code PRIV_ALLOW_MANY}, {@code PRIV_ALLOW_ONE}, or {@code PRIV_REJECT}
     */
    public int getPrivilege(String ns, String name, String user) {
	Iterator<Entry> entryiter = _entries.iterator();
	while (entryiter.hasNext()) {
	    Entry entry = entryiter.next();
	    if ((entry.namespace == null || entry.namespace.equals(ns)) && (entry.name == null || entry.name.equals(name))) {
		Iterator<UserPrivilege> priviter = entry.privs.iterator();
		while (priviter.hasNext()) {
		    UserPrivilege up = priviter.next();
		    if (up.user == null || up.user.equals(user)) {
			return up.priv;
		    }
		}
	    }
	}
	return PRIV_REJECT;
    }

    public Iterator<Entry> iterator() {
        return _entries.iterator();
    }

    public static NotePrivileges read(File file) throws NoteException {
	FileReader r = null;
	try {
	    r = new FileReader(file);
	    return read(r);
	} catch (FileNotFoundException e) {
	    throw new NoteException("Error opening file '" + file.getPath() + "' for reading", e);
	} catch (NoteException e) {
	    throw new NoteException("Error reading file '" + file.getPath() + "'", e);
	} finally {
	    if (r != null) {
		try {
		    r.close();
		} catch (IOException e) {
		    // oh well
		}
	    }
	}

    }

    public static NotePrivileges read(Reader r) throws NoteException {
	NotePrivileges retval = new NotePrivileges();
	try {
	    BufferedReader br = new BufferedReader(r);
	    String line = null;
	    ArrayList<UserPrivilege> lastprivs = null;
	    String lastns = null;
	    String lastname = null;
	    while ((line = br.readLine()) != null) {
		if (line.startsWith("#")) {
		    continue;
		}
		String[] fields = line.split("\t", 4);
		if (fields.length != 4) {
		    _logger.log(Level.SEVERE, "NotePrivileges.readFromFile: following line does not have four fields: {0}", line);
		    continue;
		}
		String ns = fields[0];
		String name = fields[1];
		String user = fields[2];
		String privstr = fields[3];
		int priv = PRIV_UNKNOWN;
		if ("allowMany".equals(privstr)) {
		    priv = PRIV_ALLOW_MANY;
		} else if ("allowOne".equals(privstr)) {
		    priv = PRIV_ALLOW_ONE;
		} else if ("reject".equals(privstr)) {
		    priv = PRIV_REJECT;
		}
		// wildcards
		if ("*".equals(ns)) {
		    ns = null;
		} else {
		    ns = Note.decodeField(ns);
		}
		if ("*".equals(name)) {
		    name = null;
		} else {
		    name = Note.decodeField(name);
		}
		if ("*".equals(user)) {
		    user = null;
		} else {
		    user = Note.decodeField(user);
		}

		UserPrivilege up = new UserPrivilege(user, priv);
		if (lastprivs == null
			|| (ns != lastns && (ns == null || !ns.equals(lastns)))
			|| (name != lastname && (name == null || !name.equals(lastname)))) {
		    Entry entry = new Entry(ns, name);
		    retval._entries.add(entry);
		    lastprivs = entry.privs;
		}
		lastprivs.add(up);
		lastns = ns;
		lastname = name;
	    }
	} catch (IOException e) {
	    throw new NoteException("Error reading privileges", e);
	}
	return retval;
    }

    public void write(File file) throws NoteException {
	FileWriter w = null;
	try {
	    w = new FileWriter(file);
	    write(w);
	} catch (IOException e) {
	    throw new NoteException("Error opening file '" + file.getPath() + "' for writing", e);
	} catch (NoteException e) {
	    throw new NoteException("Error reading file '" + file.getPath() + "'", e);
	} finally {
	    if (w != null) {
		try {
		    w.close();
		} catch (IOException e) {
		    // oh well
		}
	    }
	}
    }

    public void write(Writer w) throws NoteException {
	NotePrivileges retval = new NotePrivileges();
	try {
	    BufferedWriter bw = new BufferedWriter(w);
	    bw.write("# NotePrivileges file written by NotePrivileges.write()");
	    bw.newLine();
	    Iterator<Entry> entryiter = _entries.iterator();
	    while (entryiter.hasNext()) {
		Entry entry = entryiter.next();
		Iterator<UserPrivilege> priviter = entry.privs.iterator();
		while (priviter.hasNext()) {
		    UserPrivilege up = priviter.next();
		    String ns = entry.namespace;
		    String name = entry.name;
		    String user = up.user;
		    int priv = up.priv;
		    String privstr = "unknown";
		    if (priv == PRIV_ALLOW_MANY) {
			privstr = "allowMany";
		    } else if (priv == PRIV_ALLOW_ONE) {
			privstr = "allowOne";
		    } else if (priv == PRIV_REJECT) {
			privstr = "reject";
		    }
		    byte[] specialChars = {(byte) '*'};
		    if (ns == null) {
			ns = "*";
		    } else {
			ns = Note.encodeField(ns, specialChars);
		    }
		    if (name == null) {
			name = "*";
		    } else {
			name = Note.encodeField(name, specialChars);
		    }
		    if (user == null) {
			user = "*";
		    } else {
			user = Note.encodeField(user, specialChars);
		    }
		    bw.write(ns + "\t" + name + "\t" + user + "\t" + privstr);
		    bw.newLine();
		}
	    }
	    bw.close();
	} catch (IOException e) {
	    throw new NoteException("Error reading privileges", e);
	}
    }
}
