/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.utilities;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.text.AbstractDocument;
import javax.swing.text.BadLocationException;
import javax.swing.text.Position;
import javax.swing.text.Segment;
import javax.swing.undo.UndoableEdit;

/**
 * NOTE: we model an extra "break" character at the end as suggested by
 * http://download.oracle.com/javase/1.4.2/docs/api/javax/swing/text/AbstractDocument.html
 *
 * @author gadde
 */
public class AppendOnlyFileContent implements AbstractDocument.Content {

    File _file;
    long _strLength;
    long _byteLength;
    RandomAccessFile _raf;
    List<MyPosition> _positions;
    List<MarkPoint> _markPoints;
    final ReentrantReadWriteLock _lock;
    final WriteLock _writeLock;
    final ReadLock _readLock;

    /**
     * The AbstractDocument.Content interface deals with character strings
     * but we encode the strings into a UTF-8 byte stream for writing to the
     * file.  This helper class stores information for each chunk of data
     * written to the file, including string length and encoded byte length,
     * to help map between document character indices and file byte indices.
     */
    class MarkPoint implements Comparable<MarkPoint> {

        long bytePos;
        long strPos;
        long byteLength;
        long strLength;

        public MarkPoint(long bytePos_, long strPos_, long byteLength_, long strLength_) {
            bytePos = bytePos_;
            strPos = strPos_;
            byteLength = byteLength_;
            strLength = strLength_;
        }

        public int compareTo(MarkPoint o) {
            if (strPos < o.strPos) {
                return -1;
            }
            if (strPos > o.strPos) {
                return 1;
            }
            return 0;
        }
    }

    /**
     * This is our internal representation of a character position in the
     * document.  We maintain a list of these and update them if the file
     * contents are updated.  We maintain a reference count so they can be
     * deleted when not used.
     */
    class MyPosition implements Comparable<MyPosition> {

        int curPos;
        int refCount;

        MyPosition(int curPos_) {
            curPos = curPos_;
            refCount = 1;
        }

        void incRef() {
            refCount++;
        }

        void decRef() {
            refCount--;
        }

        public int compareTo(MyPosition o) {
            if (curPos < o.curPos) {
                return -1;
            } else if (curPos > o.curPos) {
                return 1;
            }
            return 0;
        }
    }

    /**
     * This is a wrapper around our internal MyPosition, and is returned to
     * those requesting a {@link Position}.
     */
    class YourPosition implements Position {

        MyPosition myPos;

        YourPosition(MyPosition myPos_) {
            myPos = myPos_;
            myPos.incRef();
        }

        @Override
        protected void finalize() throws Throwable {
            myPos.decRef();
        }

        public int getOffset() {
            return myPos.curPos;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (obj.getClass() != getClass()) {
                return false;
            }
            return ((YourPosition) obj).myPos.curPos == myPos.curPos;
        }
    }

    public AppendOnlyFileContent(File file) {
        _positions = new ArrayList<MyPosition>();
        _file = file;
        _lock = new ReentrantReadWriteLock();
        _writeLock = _lock.writeLock();
        _readLock = _lock.readLock();
        setupFile();
        // set up thread to clean up unreferenced positions
        Executors.newSingleThreadExecutor().submit(new Runnable() {

            public void run() {
                try {
                    while (true) {
                        _writeLock.lock();
                        try {
                            int numpos = _positions.size();
                            int lastunreferenced = -1;
                            List<MyPosition> newPositions = new ArrayList<MyPosition>();
                            for (int i = 0; i < numpos; i++) {
                                MyPosition mp = _positions.get(i);
                                if (mp.refCount == 0) {
                                    if (lastunreferenced + 1 != i) {
                                        newPositions.addAll(_positions.subList(lastunreferenced + 1, i));
                                    }
                                    lastunreferenced = i;
                                }
                            }
                            if (lastunreferenced != -1) {
                                // made some changes
                                _positions = newPositions;
                            }
                        } finally {
                            _writeLock.unlock();
                        }
                        Thread.sleep(10000);
                    }
                } catch (InterruptedException ex) {
                    Logger.getLogger(AppendOnlyFileContent.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        });
    }

    public Position createPosition(int offset) throws BadLocationException {
        if (offset > _strLength + 1) {
            throw new BadLocationException("Attempt to access beyond end of file", offset);
        }
        // binary search
        MyPosition myPos = new MyPosition(offset);
        int ind = Collections.binarySearch(_positions, myPos);
        if (ind >= 0) {
            return new YourPosition(_positions.get(ind));
        }
        // was not in list, but ind = (-1 * insertionPoint) - 1
        // where insertionPoint is where we should insert the element
        // to maintain a sorted list
        _positions.add(-1 * (ind + 1), myPos);
        return new YourPosition(myPos);
    }

    public int length() {
        return (int) _strLength + 1;
    }

    public void setText(String t) throws IOException, BadLocationException {
        _writeLock.lock();
        try {
            _raf.close();
            _file.delete();
            setupFile();
            insertString(0, t);
        } finally {
            _writeLock.unlock();
        }
    }

    private void setupFile() {
        _writeLock.lock();
        try {
            _raf = new RandomAccessFile(_file, "rw");
            _markPoints = new ArrayList<MarkPoint>();
            _raf.seek(0);
            _strLength = 0;
            _byteLength = 0;
            String line;
            while ((line = _raf.readLine()) != null) {
                byte[] bytes = line.getBytes("UTF-8");
                _markPoints.add(new MarkPoint(_byteLength, _strLength, bytes.length, line.length()));
                _byteLength += bytes.length;
                _strLength += line.length();
            }
        } catch (FileNotFoundException ex) {
            Logger.getLogger(AppendOnlyFileContent.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(AppendOnlyFileContent.class.getName()).log(Level.SEVERE, null, ex);
        } finally {
            _writeLock.unlock();
        }
    }

    public UndoableEdit insertString(int where, String str) throws BadLocationException {
        _writeLock.lock();
        try {
            if (where != _strLength) {
                throw new BadLocationException("Inserts only allowed at end of document (curLength=" + _strLength + ")", where);
            }
            try {
                _raf.seek(_raf.length());
                byte[] bs = str.getBytes("UTF-8");
                _markPoints.add(new MarkPoint(_byteLength, _strLength, bs.length, str.length()));
                _raf.write(bs);
                _byteLength += bs.length;
                _strLength += str.length();
            } catch (IOException ex) {
                Logger.getLogger(AppendOnlyFileContent.class.getName()).log(Level.SEVERE, null, ex);
            }
            int posind = Collections.binarySearch(_positions, new MyPosition(where));
            int n = _positions.size();
            if (posind < 0) {
                posind = -1 * (posind + 1);
            }
            for (int i = posind; i < n; i++) {
                MyPosition pos = _positions.get(i);
                if (pos.curPos >= where && pos.curPos != 0) { // since array is sorted, this check should be unnecessary
                    pos.curPos += str.length();
                }
            }
        } finally {
            _writeLock.unlock();
        }
        return null;
    }

    public UndoableEdit remove(int where, int nitems) throws BadLocationException {
        if (where < 0) {
            throw new BadLocationException("Negative position!", where);
        }
        _writeLock.lock();
        try {
            if (where + nitems != _strLength && where + nitems != _strLength + 1) {
                throw new BadLocationException("Can only remove items from end of document", where);
            }
            try {
                _raf.setLength(where);
            } catch (IOException ex) {
                Logger.getLogger(AppendOnlyFileContent.class.getName()).log(Level.SEVERE, null, ex);
            }
            int posind = Collections.binarySearch(_positions, new MyPosition(where));
            int n = _positions.size();
            if (posind < 0) {
                posind = -1 * (posind + 1);
            }
            for (int i = posind; i < n; i++) {
                MyPosition pos = _positions.get(i);
                if (pos.curPos >= where && pos.curPos != 0) { // since array is sorted, this check should be unnecessary
                    pos.curPos = where + 1;
                }
            }
            _strLength = where;
            MarkPoint comparemp = new MarkPoint(0, where, 0, nitems);
            int mpind = Collections.binarySearch(_markPoints, comparemp);
            if (mpind < 0) {
                // didn't find an exact match for this position so adjust
                // index to "insertion point", meaning the index of the first
                // element greater than the position ("where") we were
                // searching for, or _markPoints.size() if "where" is greater
                // than all elements (or presumably if the list is empty)
                mpind = -1 * (mpind + 1);
                // Unless mpind is 0 (list is empty), the MarkPoint containing
                // the position "where" is the previous element.
                if (mpind != 0) {
                    mpind--;
                    MarkPoint lastmp = _markPoints.get(mpind);
                    if (lastmp.strPos <= where && where < lastmp.strPos + lastmp.strLength) {
                        Segment s = new Segment();
                        this.getChars((int) lastmp.strPos, (int) (where - lastmp.strPos), s);
                        CharsetEncoder encoder = Charset.forName("UTF-8").newEncoder();
                        CharBuffer cbuf = CharBuffer.wrap(s);
                        try {
                            ByteBuffer bbuf = encoder.encode(cbuf);
                            lastmp.byteLength = bbuf.remaining();
                        } catch (CharacterCodingException e) {
                            throw new BadLocationException("Error encoding character buffer: '" + cbuf + "'", where);
                        }
                        lastmp.strLength = where - lastmp.strPos;
                        mpind++;
                    }
                }
            }
            // at this point, mpind (and all subsequent elements) refers to
            // an invalid element whose data has been deleted, so get rid of
            // them.
            int nummps = _markPoints.size();
            while (mpind < nummps) {
                _markPoints.remove(mpind);
                nummps--;
            }
        } finally {
            _writeLock.unlock();
        }
        return null;
    }

    public String getString(int where, int len) throws BadLocationException {
        Segment txt = new Segment();
        getChars(where, len, txt);
        return txt.toString();
    }

    public void getChars(int where, int len, Segment txt) throws BadLocationException {
        if (where < 0) {
            throw new BadLocationException("Negative position!", where);
        }
        _readLock.lock();
        try {
            if (where + len > _strLength + 1) {
                throw new BadLocationException("Reading beyond end of file!", where);
            }
            MarkPoint comparemp = new MarkPoint(0, where, 0, len);
            int mpind = Collections.binarySearch(_markPoints, comparemp);
            if (mpind < 0) {
                mpind = -1 * (mpind + 1);
            }
            if (mpind > 0) {
                mpind--;
            }
            int nummps = _markPoints.size();
            while (mpind < nummps) {
                MarkPoint tmpmp = _markPoints.get(mpind);
                if (where >= tmpmp.strPos && where < tmpmp.strPos + tmpmp.strLength) {
                    break;
                }
                mpind++;
            }
            char[] chars = new char[len];
            CharBuffer cbuf = CharBuffer.wrap(chars, 0, len);
            int numread = 0;
            CharsetDecoder decoder = Charset.forName("UTF-8").newDecoder();
            try {
                while (mpind < nummps && numread < len) {
                    MarkPoint mp = _markPoints.get(mpind);
                    _raf.seek(mp.bytePos);
                    byte[] bytes = new byte[(int) mp.byteLength];
                    _raf.read(bytes);
                    ByteBuffer bbuf = ByteBuffer.wrap(bytes);
                    if (mp.strPos < where) {
                        decoder.decode(bbuf, CharBuffer.wrap(new char[(int) (where - mp.strPos)]), false);
                    }
                    decoder.decode(bbuf, cbuf, true);
                    numread = cbuf.position();
                    mpind++;
                }
            } catch (IOException e) {
                throw new BadLocationException("Error reading from file", (int) _markPoints.get(mpind).bytePos);
            }
            if (cbuf.position() < len && where + cbuf.position() == _strLength) {
                cbuf.append('\n'); // extra character demanded by AbstractDocument
            }
            if (cbuf.position() < len) {
                throw new BadLocationException("Could not find enough data?  Stopped reading", where + cbuf.position());
            }
            txt.array = chars;
            txt.offset = 0;
            txt.count = len;
        } finally {
            _readLock.unlock();
        }
    }
}
