/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.utilities.download;

import org.ietf.jgss.GSSException;
import org.nbirn.fbirn.utilities.CanceledException;
import org.nbirn.fbirn.utilities.CredentialManager;
import org.nbirn.fbirn.utilities.CredentialManagerException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URI;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.Vector;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;

import org.globus.ftp.FTPClient;
import org.globus.ftp.GridFTPClient;
import org.globus.ftp.GridFTPSession;
import org.globus.ftp.MlsxEntry;

import org.ietf.jgss.GSSCredential;

import org.birncommunity.gridftp.tar.TwoPartyTarClientFactoryI;
import org.birncommunity.gridftp.tar.TwoPartyTarClientFactoryImpl;
import org.birncommunity.gridftp.tar.TwoPartyTarTransfer;
import org.birncommunity.gridftp.tar.UntarFromPipeRunnable;
import org.birncommunity.gridftp.tar.impl.compress.CompressUntar;
import org.birncommunity.gridftp.tar.impl.compress.UntarDirectoryEntryHandlerImplementation;
import org.birncommunity.gridftp.tar.impl.compress.UntarFileEntryHandlerImplementation;
import org.globus.ftp.DataSink;
import org.globus.ftp.DataSinkStream;
import org.globus.ftp.exception.ServerException;
import org.globus.myproxy.MyProxyException;

/**
 *
 * @author gadde
 */
public abstract class WorkerDownload extends WorkerProgressReporter {

    private static final Logger _logger = Logger.getLogger(WorkerDownload.class.getName());
    private static final int DEFAULT_MINCHUNKS = 100;
    boolean _dryrun = false; // don't actually download anything

    protected WorkerDownload(boolean dryrun) {
        _dryrun = dryrun;
    }

    Exception downloadURLDefault(URI url, File dest) {
        try {
            FileOutputStream fos = new FileOutputStream(dest);
            Exception e = downloadURLDefault(url, fos);
            fos.close();
            return e;
        } catch (Exception e) {
            return e;
        }
    }

    Exception downloadURLDefault(URI url, OutputStream os) {
        try {
            InputStream is = url.toURL().openStream();
            byte buf[] = new byte[1024];
            int bytesread = 0;
            while ((bytesread = is.read(buf)) != -1) {
                os.write(buf, 0, bytesread);
            }
        } catch (Exception e) {
            return e;
        }
        return null;
    }

    public void resetPassive(FTPClient ftpclient) throws DownloadException {
        try {
            ftpclient.setPassive();
        } catch (Exception e) {
            throw new DownloadException("Error calling GridFTPClient::setLocalPassive()", e);
        }
        try {
            ftpclient.setLocalActive();
        } catch (Exception e) {
            throw new DownloadException("Error calling GridFTPClient::setActive()", e);
        }
    }

    private void tryConnect(GridFTPClient ftpclient, GSSCredential cred) throws ServerException, IOException {
        if (cred == null) {
            throw new NullPointerException("tryConnect() received cred==null!");
        }
        if (ftpclient == null) {
            throw new NullPointerException("tryConnect() received ftpclient==null!");

        }
        ftpclient.authenticate(cred);
    }

    protected GridFTPClient gridFTPConnect(CredentialManager credman, String gridftphost, int gridftpport) throws CanceledException, DownloadException {
        GridFTPClient ftpclient = null;
        try {
            ftpclient = new GridFTPClient(gridftphost, gridftpport);
        } catch (Exception e) {
            throw new DownloadException("Error getting new GridFTP client", e);
        }
        synchronized (credman) {
            if (isCancelled() == true || credman.wasCanceled()) {
                throw new CanceledException("Operation canceled by user.");
            }
            Exception retval = null;
            GSSCredential cred = credman.getCredential();
            try {
                if (cred != null && cred.getRemainingLifetime() > 0) {
                    try {
                        tryConnect(ftpclient, cred);
                        // success!
                        return ftpclient;
                    } catch (Exception e) {
                        try {
                            ftpclient.close();
                            ftpclient = null;
                        } catch (Exception unused) {
                            // do nothing
                        }
                        throw new DownloadException("Error  connecting to GridFTP server " + gridftphost + ":" + String.valueOf(gridftpport), e);
                    }
                }
            } catch (GSSException e) {
                throw new DownloadException("Error getting credential", e);
            }
            String msg = "Please enter MyProxy and GridFTP connection parameters";
            while (true) {
                cred = credman.getLocalCredential();
                if (cred != null) {
                    if (ftpclient == null) {
                        try {
                            ftpclient = new GridFTPClient(gridftphost, gridftpport);
                        } catch (Exception e) {
                            throw new DownloadException("Error getting new GridFTP client", e);
                        }
                    }

                    try {
                        tryConnect(ftpclient, cred);
                        // success!
                        break;
                    } catch (Exception e) {
                        try {
                            ftpclient.close();
                            ftpclient = null;
                        } catch (Exception unused) {
                            // do nothing
                        }
                    }
                }
                try {
                    credman.getNewCredential(msg);
                    cred = credman.getCredential();
                    if (cred == null) {
                        if (credman.canGetUserInput()) {
                            msg = "Error in connection parameters.  Try again:";
                            continue;
                        } else {
                            throw new DownloadException("Can't get valid credential (try getting one by running myproxy-logon)");
                        }
                    }
                } catch (CredentialManagerException e) {
                    throw new DownloadException("Error getting new credential", e);
                } catch (MyProxyException e) {
                    if (credman.canGetUserInput()) {
                        msg = e.toString();
                        continue;
                    } else {
                        throw new DownloadException("Can't get valid credential (" + e.toString() + ").  Alternative: get one by running myproxy-logon.");
                    }
                }
                if (ftpclient == null) {
                    try {
                        ftpclient = new GridFTPClient(gridftphost, gridftpport);
                    } catch (Exception e) {
                        throw new DownloadException("Error getting new GridFTP client", e);
                    }
                }
                try {
                    tryConnect(ftpclient, cred);
                    // success!
                    break;
                } catch (Exception e) {
                    try {
                        ftpclient.close();
                    } catch (Exception unused) {
                        // do nothing
                    }
                    throw new DownloadException("Error  connecting to GridFTP server " + gridftphost + ":" + String.valueOf(gridftpport), e);
                }
            }
        }
        try {
            ftpclient.setType(GridFTPSession.TYPE_IMAGE);
            ftpclient.setMode(GridFTPSession.MODE_STREAM);
        } catch (Exception e) {
            throw new DownloadException("Error setting type or mode on FTP client (host " + gridftphost + ", port " + String.valueOf(gridftpport) + ")", e);
        }
        return ftpclient;
    }

    protected GridFTPClient checkConnection(CredentialManager credman, GridFTPClient ftpclient, String gridftphost, int gridftpport) throws CanceledException, DownloadException {
        if (ftpclient == null) {
            return gridFTPConnect(credman, gridftphost, gridftpport);
        } else {
            try {
                ftpclient.quote("NOOP");
                return ftpclient;
            } catch (Exception e) {
                try {
                    ftpclient.close();
                } catch (Exception e2) {
                    // do nothing
                }
                return this.gridFTPConnect(credman, gridftphost, gridftpport);
            }
        }
    }

    Exception downloadURLGSIFTP(URI url, File destFile, CredentialManager credman, boolean updateOnly, TimeZone remoteTimeZone, int timeStampCheckDepth) throws CanceledException, DownloadException {
        return downloadURLGSIFTP(url, destFile, null, credman, DEFAULT_MINCHUNKS, timeStampCheckDepth, updateOnly, remoteTimeZone, 0, 100);
    }

    Exception downloadURLGSIFTP(URI url, OutputStream os, CredentialManager credman, boolean updateOnly, TimeZone remoteTimeZone, int timeStampCheckDepth) throws CanceledException, DownloadException {
        return downloadURLGSIFTP(url, null, os, credman, DEFAULT_MINCHUNKS, timeStampCheckDepth, updateOnly, remoteTimeZone, 0, 100);
    }

    Exception downloadURLGSIFTP(URI url, File destFile, OutputStream os, CredentialManager credman, int minChunks, int minDepth, boolean updateOnly, TimeZone remoteTimeZone, double progressStart, double progressEnd) throws CanceledException, DownloadException {
        if (isCancelled()) {
            throw new CanceledException("Operation canceled by user.");
        }
        String gridftphost = url.getHost();
        int gridftpport = (url.getPort() == -1 ? 2811 : url.getPort());
        String remotepath = url.getPath();
        GridFTPClient ftpclient = gridFTPConnect(credman, gridftphost, gridftpport);
        double progress = progressStart;
        while (remotepath.length() > 1 && remotepath.endsWith("/")) {
            remotepath = remotepath.substring(0, remotepath.length() - 1);
        }
        ftpclient = this.checkConnection(credman, ftpclient, gridftphost, gridftpport);
        if (destFile != null && destFile.exists() && destFile.isDirectory()) {
            // local path exists, so add last component of remote pathname
            int lastremoteslash = remotepath.lastIndexOf('/');
            if (lastremoteslash == -1) {
                destFile = new File(destFile, remotepath);
            } else {
                destFile = new File(destFile, remotepath.substring(lastremoteslash + 1));
            }
        }
        downloadGSIFTPAux(ftpclient, gridftphost, gridftpport, remotepath, destFile, (os == null) ? null : new DataSinkStream(os), null, credman, minChunks, minDepth, updateOnly, remoteTimeZone, progressStart, progressEnd);
        if (ftpclient != null) {
            try {
                ftpclient.close();
            } catch (Exception e) {
                // do nothing
            }
        }
        return null;
    }

    private long getTimestamp(MlsxEntry info, TimeZone remoteTimeZone) {
        String modifystr = info.get(MlsxEntry.MODIFY);
        if (modifystr.length() != 14) {
            return -1;
        }
        int year = Integer.parseInt(modifystr.substring(0, 4));
        int mon = Integer.parseInt(modifystr.substring(4, 6)) - 1; // zero-based
        int mday = Integer.parseInt(modifystr.substring(6, 8));
        int hour = Integer.parseInt(modifystr.substring(8, 10));
        int min = Integer.parseInt(modifystr.substring(10, 12));
        int sec = Integer.parseInt(modifystr.substring(12, 14));
        Calendar cal = null;
        if (false && remoteTimeZone != null) { // Apparently gridFTP timestamps are already in GMT???
            cal = GregorianCalendar.getInstance(remoteTimeZone);
        } else {
            cal = GregorianCalendar.getInstance(TimeZone.getTimeZone("GMT"));
        }
        cal.set(year, mon, mday, hour, min, sec);
        cal.set(Calendar.MILLISECOND, 0); // many file systems won't support less than second granularity
        return cal.getTimeInMillis();
    }

    /**
     *
     * @param ftpclient
     * @param gridftphost
     * @param gridftpport
     * @param remotepath
     * @param localfile If not null, the file or directory to which to write the output of remotepath (and localsink is ignored).
     * @param localsink If not null (and localfile is null), this is the sink to which the contents of remotepath will be written. remotepath must refer to a plain file.
     * @param info
     * @param credman
     * @param minChunks
     * @param minDepth
     * @param updateOnly
     * @param remoteTimeZone
     * @param progressStart
     * @param progressEnd
     * @throws CanceledException
     * @throws DownloadException
     */
    void downloadGSIFTPAux(GridFTPClient ftpclient, String gridftphost, int gridftpport, String remotepath, File localfile, DataSink localsink, MlsxEntry info, CredentialManager credman, int minChunks, int minDepth, boolean updateOnly, TimeZone remoteTimeZone, double progressStart, double progressEnd) throws CanceledException, DownloadException {
        if (isCancelled()) {
            throw new CanceledException("Operation canceled by user.");
        }
        _logger.log(Level.FINE, "downloadGSIFTPAux(gridftphost=''{0}'', gridftpport={1}, remotepath=''{2}'', localfile=''{3}'', localsink=''{4}'', minChunks={5}, minDepth={6}, updateOnly={7}, remoteTimeZone=''{8}, progressStart={9}, progressEnd={10})",
                new Object[]{
                    gridftphost,
                    gridftpport,
                    remotepath,
                    (localfile == null) ? "<null>" : localfile.getAbsolutePath(),
                    (localsink == null) ? "<null>" : localsink.toString(),
                    minChunks,
                    minDepth,
                    updateOnly,
                    (remoteTimeZone == null) ? "<null>" : remoteTimeZone.getDisplayName(),
                    progressStart,
                    progressEnd});
        DateFormat dateformat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

        // this associates files with the timestamps that need to be set before exiting this method
        class TimeStampQueueEntry {

            File _file;
            long _timestamp;

            TimeStampQueueEntry(File file, long timestamp) {
                _file = file;
                _timestamp = timestamp;
            }
        }
        final ArrayList<TimeStampQueueEntry> timestampqueue = new ArrayList<TimeStampQueueEntry>();

        ftpclient = this.checkConnection(credman, ftpclient, gridftphost, gridftpport);
        if (info == null) {
            try {
                info = ftpclient.mlst(remotepath);
            } catch (Exception e) {
                throw new DownloadException("Error getting file listing for remotepath " + remotepath, e);
            }
        }
        String infotype = info.get(MlsxEntry.TYPE);
        // see if we need to delete this file/dir
        if (localfile != null && localfile.getName().startsWith("__DELETE__")) {
            String delname = localfile.getName().substring(10);
            File delfile = new File(localfile.getParent(), delname);
            if (delfile.exists() && ((MlsxEntry.TYPE_FILE.equals(infotype) && delfile.isFile()) || (MlsxEntry.TYPE_DIR.equals(infotype) && delfile.isDirectory()))) {
                publish(new ProgressUpdate((_dryrun ? "(not) " : "") + "Deleting " + delfile.getAbsolutePath(), null, 0, 1));
                if (!_dryrun) {
                    File deldir = new File(localfile.getParent(), "__DELETED__");
                    deldir.mkdir();
                    delfile.renameTo(new File(deldir, delname));
                }
            }
            return;
        }
        long remotelastmod = getTimestamp(info, remoteTimeZone);
        boolean localoutofdate = true;
        if (updateOnly && localfile != null) {
            String outofdatestr = "out of date";
            if (remotelastmod != -1 && localfile.exists() && remotelastmod <= localfile.lastModified()) {
                outofdatestr = "up-to-date";
                localoutofdate = false;
            }
            _logger.log(Level.FINE, "Local file {0} seems {1} (local {2}, remote {3}) {4}",
                    new Object[]{
                        localfile.getAbsolutePath(),
                        outofdatestr,
                        dateformat.format(new Date(localfile.lastModified())),
                        dateformat.format(new Date(remotelastmod)),
                        localfile.getAbsolutePath()});
        }
        // see if it's just a file, and if so, try to download it
        if (MlsxEntry.TYPE_FILE.equals(infotype)) {
            try {
                setProgress((int) progressStart);
                if (localoutofdate) {
                    publish(new ProgressUpdate((_dryrun ? "(not) " : "") + "Downloading '" + remotepath + "'", null, 0, 1));
                    resetPassive(ftpclient);
                    if (!_dryrun) {
                        boolean getsucceeded = false;
                        try {
                            if (localfile != null) {
                                ftpclient.get(remotepath, localfile);
                            } else {
                                ftpclient.get(remotepath, localsink, null);
                            }
                            getsucceeded = true;
                        } finally {
                            if (localfile != null && remotelastmod != -1 && localfile.exists()) {
                                            // failed entries get a backdated timestamp
                                long modtime = getsucceeded ? remotelastmod : remotelastmod - 1000;
                                _logger.log(Level.FINE, "Setting timestamp: {0} {1}", new Object[]{localfile.getPath(), dateformat.format(new Date(modtime))});
                                localfile.setLastModified(modtime);
                            }
                            if (localsink != null) {
                                localsink.close();
                            }
                        }
                    }
                } else {
                    _logger.log(Level.FINE, "Skipped up-to-date ''{0}''", remotepath);
                    publish(new ProgressUpdate("Skipped up-to-date '" + remotepath + "'", 0));
                }
                setProgress((int) progressEnd);
            } catch (Exception e) {
                if (localfile != null) {
                    throw new DownloadException("Error getting file '" + remotepath + "' to local file '" + localfile.toString() + "'", e);
                } else {
                    throw new DownloadException("Error getting file '" + remotepath + "'", e);
                }
            }
            return;
        } else if (infotype.equals(MlsxEntry.TYPE_DIR)) {
            // otherwise, if remote path is a directory, make the local
            // directory counterpart
            if (localfile == null) {
                throw new DownloadException("localfile must be specified when downloading from a directory!");
            }
            if (!localfile.exists()) {
                publish(new ProgressUpdate((_dryrun ? "(not) " : "") + "Making directory '" + localfile.getAbsolutePath() + "'", 0));
                if (!_dryrun) {
                    localfile.mkdir();
                    if (remotelastmod != -1) {
                        _logger.log(Level.FINE, "Setting temporary timestamp: {0} {1}", new Object[]{localfile.getPath(), dateformat.format(new Date(remotelastmod - 1000))});

                        localfile.setLastModified(remotelastmod - 1000); // subtract one second so that it is still out-of-date until we are really done
                        timestampqueue.add(new TimeStampQueueEntry(localfile, remotelastmod));
                    }
                }
            }
        }
        // remote path is a directory, and contents of remote directory
        // need to go into a directory named by localfile.
        // try tar-stream approach first
        boolean fallthrough = true;
        if (minChunks <= 1 && minDepth <= 0) {
            if (localoutofdate) {
                // See if we can use "tar-stream" approach
                ExecutorService executorService = null;
                TwoPartyTarClientFactoryI clientFactory = new TwoPartyTarClientFactoryImpl();
                GSSCredential cred = null;
                synchronized (credman) {
                    cred = credman.getCredential();
                    if (cred == null) {
                        cred = credman.getLocalCredential();
                        if (cred == null) {
                            try {
                                credman.getNewCredential("Please enter MyProxy and GridFTP connection parameters");
                            } catch (CredentialManagerException e) {
                                throw new DownloadException("Error getting new credential", e);
                            } catch (MyProxyException e) {
                                throw new DownloadException("Error getting new credential", e);
                            }
                        }
                    }
                }
                TwoPartyTarTransfer tptt = new TwoPartyTarTransfer(clientFactory, gridftphost, gridftpport, cred);
                boolean dopopen = false;
                try {
                    dopopen = tptt.isPopenDriverSupported();
                } catch (Exception e) {
                    throw new DownloadException("Error checking if popen is supported", e);
                }
                if (dopopen) {
                    publish(new ProgressUpdate((_dryrun ? "(not) " : "") + "Downloading (with tar-stream) '" + remotepath + "'", null, 0, 1));
                    if (!_dryrun) {
                        PipedOutputStream pipeOut = new PipedOutputStream();
                        PipedInputStream pipeIn = null;
                        try {
                            pipeIn = new PipedInputStream(pipeOut);
                        } catch (IOException e) {
                            throw new DownloadException("Error creating pipe", e);
                        }
                        final int localfilelen = localfile.getAbsolutePath().length();
                        executorService = Executors.newSingleThreadExecutor();
                        Future<?> untarfuture = null;
                        try {
                            // extend default handlers to set last modified times
                            UntarDirectoryEntryHandlerImplementation dirhandler = new UntarDirectoryEntryHandlerImplementation() {

                                @Override
                                public void handleUntarDirectoryEntry(TarArchiveEntry tae, File destDir) throws IOException {
                                    if (tae.getName().startsWith("__DELETE__")) {
                                        String delname = tae.getName().substring(10);
                                        File newfile = new File(destDir, delname);
                                        if (newfile.exists() && newfile.isDirectory()) {
                                            publish(new ProgressUpdate(null, "...deleting " + newfile.getAbsolutePath().substring(localfilelen), 0, -1));
                                            File deldir = new File(destDir, "__DELETED__");
                                            deldir.mkdir();
                                            newfile.renameTo(new File(deldir, delname));
                                        }
                                    } else {
                                        File newfile = new File(destDir, tae.getName());
                                        publish(new ProgressUpdate(null, "..." + newfile.getAbsolutePath().substring(localfilelen), 0, -1));
                                        boolean succeeded = false;
                                        try {
                                            super.handleUntarDirectoryEntry(tae, destDir);
                                            succeeded = true;
                                        } finally {
                                            // failed entries get a backdated timestamp
                                            long modtime = tae.getModTime().getTime();
                                            newfile.setLastModified(succeeded ? modtime : modtime - 1000);
                                        }
                                        // set the timestamp later too, because additions of files within the directory will change its timestamp anyway
                                        timestampqueue.add(new TimeStampQueueEntry(newfile, tae.getModTime().getTime()));
                                    }
                                }
                            };
                            UntarFileEntryHandlerImplementation filehandler = new UntarFileEntryHandlerImplementation() {

                                @Override
                                public void handleUntarFileEntry(TarArchiveInputStream tais, File destDir, TarArchiveEntry tae) throws IOException {
                                    if (tae.getName().startsWith("__DELETE__")) {
                                        String delname = tae.getName().substring(10);
                                        File newfile = new File(destDir, delname);
                                        if (newfile.exists() && newfile.isDirectory()) {
                                            publish(new ProgressUpdate(null, "...deleting " + newfile.getAbsolutePath().substring(localfilelen), 0, -1));
                                            File deldir = new File(destDir, "__DELETED__");
                                            deldir.mkdir();
                                            newfile.renameTo(new File(deldir, delname));
                                        }
                                    } else {
                                        File newfile = new File(destDir, tae.getName());
                                        publish(new ProgressUpdate(null, "..." + newfile.getAbsolutePath().substring(localfilelen), 0, -1));
                                        boolean succeeded = false;
                                        try {
                                            super.handleUntarFileEntry(tais, destDir, tae);
                                            succeeded = true;
                                        } finally {
                                            // failed entries get a backdated timestamp
                                            long modtime = tae.getModTime().getTime();
                                            newfile.setLastModified(succeeded ? modtime : modtime - 1000);
                                        }
                                    }
                                }
                            };
                            untarfuture = executorService.submit(new UntarFromPipeRunnable(pipeIn, localfile.getParentFile().getAbsolutePath(), new CompressUntar(dirhandler, filehandler)));
                        } catch (Exception e) {
                            throw new DownloadException("Error submitting untar task for execution", e);
                        }
                        String fullremotepath = remotepath;
                        setProgress((int) progressStart);
                        try {
                            tptt.downloadTarToPipe(fullremotepath, pipeOut);
                        } catch (Exception e) {
                            throw new DownloadException("Error downloading tar stream", e);
                        }
                        // now collect any error from the untar task
                        try {
                            untarfuture.get();
                        } catch (InterruptedException e) {
                            throw new CanceledException("Operation canceled", e);
                        } catch (Exception e) {
                            throw new DownloadException("Error running untar task", e);
                        } finally {
                            // make sure to set timestamps on directories even (or especially) on error
                            for (int i = 0; i < timestampqueue.size(); i++) {
                                TimeStampQueueEntry e = timestampqueue.get(i);
                                _logger.log(Level.FINE, "Setting timestamp: {0} {1}", new Object[]{e._file.getPath(), dateformat.format(new Date(e._timestamp))});
                                e._file.setLastModified(e._timestamp);
                            }
                        }
                        if (remotelastmod != -1) {
                            _logger.log(Level.FINE, "Setting timestamp: {0} {1}", new Object[]{localfile.getPath(), dateformat.format(new Date(remotelastmod))});
                            localfile.setLastModified(remotelastmod);
                        }
                    }
                    setProgress((int) progressEnd);
                    fallthrough = false;
                }
            } else {
                _logger.log(Level.FINE, "Skipped up-to-date ''{0}''", remotepath);
                setProgress((int) progressEnd);
                publish(new ProgressUpdate("Skipped up-to-date '" + remotepath + "'", 0));
                fallthrough = false;
            }
        }
        if (!fallthrough) {
            return;
        } // if "tar-stream" not supported, or we need to "chunk" the download, then fall through
        class StackEntry {

            String _remotepath = null;
            File _localfile = null;
            int _mindepth = 0;
            MlsxEntry _info = null;

            StackEntry(String remotepath, File localfile, int mindepth, MlsxEntry info) {
                _remotepath = remotepath;
                _localfile = localfile;
                _mindepth = mindepth;
                _info = info;
            }
        }
        setProgress((int) progressStart);
        ArrayDeque<StackEntry> stack = new ArrayDeque<StackEntry>();
        stack.push(new StackEntry(remotepath, localfile, minDepth, info));
        while (!stack.isEmpty()) {
            if (isCancelled()) {
                throw new CanceledException("Operation canceled by user.");
            }
            StackEntry sentry = null;
            minChunks--;
            if (stack.size() < minChunks || minDepth > 0) {
                // while we are still calibrating our progress bar, we use getFirst()
                // to do a breadth-first traversal -- this makes it easier to provide
                // a more accurate progress bar
                sentry = stack.removeFirst();
            } else {
                // if we've already reached our target granularity for progress
                // bar, then switch to depth-first traversal to get data in chunks
                sentry = stack.removeLast();
            }
            String curremotepath = sentry._remotepath;
            File curlocalfile = sentry._localfile;
            int curmindepth = sentry._mindepth;
            MlsxEntry curinfo = sentry._info;
            // pre-condition: if curremotepath is a file, then curlocalfile is
            // the local path to the file where it should be copied.   If
            // curremotepath is a directory, then curlocalfile is the local
            // path to the directory (which must exist) to which the contents
            // of the remote directory should be copied.  In the directory case,
            // the last component of curremotepath and curlocalfile will always
            // be the same.
            _logger.log(Level.FINE, " curremotepath=''{0}'' curlocalfile=''{1}'' stack.size()={2} minChunks={3} minDepth={4}",
                    new Object[]{
                        curremotepath,
                        curlocalfile,
                        stack.size(),
                        minChunks,
                        curmindepth});
            // if we call ourselves recursively to copy a single file, we have to make sure to specify target as parent directory (so we don't download to extraneous subdirectories)
            if (stack.size() >= minChunks && curmindepth <= 0) {
                // we've reached our target granularity/depth; break off into a chunk that could be downloaded in one tar stream
                double progressChunk = ((progressEnd - progressStart) / (stack.size() + 1));
                downloadGSIFTPAux(ftpclient, gridftphost, gridftpport, curremotepath, curlocalfile, null, curinfo, credman, 1, 0, updateOnly, remoteTimeZone, progressStart, progressStart + progressChunk);
                progressStart += progressChunk;
                continue;
            }
            this.checkConnection(credman, ftpclient, gridftphost, gridftpport);
            this.resetPassive(ftpclient);
            Vector filelist = null;
            try {
                filelist = ftpclient.mlsd(curremotepath);
            } catch (Exception e) {
                throw new DownloadException("Error getting listing for directory " + curremotepath, e);
            }
            int numsubfiles = 0;
            int numsubdirs = 0;
            for (int i = 0; i < filelist.size(); i++) {
                MlsxEntry linfo = (MlsxEntry) filelist.elementAt(i);
                if (linfo.getFileName().equals(".") || linfo.getFileName().equals("..")) {
                    filelist.remove(i);
                    i--;
                    continue;
                }
                String linfotype = linfo.get(MlsxEntry.TYPE);
                if (linfotype.equals(MlsxEntry.TYPE_FILE)) {
                    numsubfiles++;
                } else if (linfotype.equals(MlsxEntry.TYPE_DIR)) {
                    numsubdirs++;
                }
            }
            if (numsubdirs == 0 && curmindepth <= 0) {
                // we were still calibrating our progress bar, but this
                // directory has no subdirectories to do any smaller chunking;
                // why not just attempt to download it as a chunk?
                double progressChunk = ((progressEnd - progressStart) / (stack.size() + 1));
                downloadGSIFTPAux(ftpclient, gridftphost, gridftpport, curremotepath, curlocalfile, null, info, credman, 1, 0, updateOnly, remoteTimeZone, progressStart, progressStart + progressChunk);
                progressStart += progressChunk;
                continue;
            }
            for (int i = 0; i < filelist.size(); i++) {
                if (isCancelled()) {
                    throw new CanceledException("Operation canceled by user.");
                }
                MlsxEntry subinfo = (MlsxEntry) filelist.elementAt(i);
                _logger.log(Level.FINE, "  file {0}", subinfo.getFileName());
                File newlocalfile = new File(curlocalfile, subinfo.getFileName());
                String newremotepath = curremotepath + "/" + subinfo.getFileName();
                String subinfotype = subinfo.get(MlsxEntry.TYPE);
                if (subinfotype.equals(MlsxEntry.TYPE_DIR)) {
                    if (!newlocalfile.exists()) {
                        publish(new ProgressUpdate((_dryrun ? "(not) " : "") + "Making directory '" + localfile.getAbsolutePath() + "'", 0));
                        if (!_dryrun) {
                            newlocalfile.mkdir();
                            long sublastmod = getTimestamp(subinfo, remoteTimeZone);
                            if (sublastmod != -1) {
                                _logger.log(Level.FINE, "Setting temporary timestamp: {0} {1}", new Object[]{newlocalfile.getPath(), dateformat.format(new Date(sublastmod - 1000))});
                                newlocalfile.setLastModified(sublastmod - 1000);
                                timestampqueue.add(new TimeStampQueueEntry(newlocalfile, sublastmod));
                            }
                        }
                        stack.addLast(new StackEntry(newremotepath, newlocalfile, 0, subinfo)); // don't worry about mindepth since it is a new directory
                    } else {
                        stack.addLast(new StackEntry(newremotepath, newlocalfile, curmindepth - 1, subinfo));
                    }
                } else if (subinfotype.equals(MlsxEntry.TYPE_FILE)) {
                    double progressChunk = ((progressEnd - progressStart) / (stack.size() + 1));
                    downloadGSIFTPAux(ftpclient, gridftphost, gridftpport, newremotepath, newlocalfile, null, subinfo, credman, 1, 0, updateOnly, remoteTimeZone, progressStart, progressStart + progressChunk);
                    progressStart += progressChunk;
                }
            }
        }
        for (int i = 0; i < timestampqueue.size(); i++) {
            TimeStampQueueEntry e = timestampqueue.get(i);
            _logger.log(Level.FINE, "Setting timestamp: {0} {1}", new Object[]{e._file.getPath(), dateformat.format(new Date(e._timestamp))});
            e._file.setLastModified(e._timestamp);
        }
        setProgress((int) progressEnd);
    }
}
