#!/bin/bash 

###############################################################################################################
#	This script will install the fBIRN Human upload scripts.  It will collect some
#	basic site specific metadata and create configuration files for each local scanner you
#	will use to upload phantom data.
#
#	The script will check for binary dependencies based on what type of <Native> format data
#	you have.  You need not have the same data format for each of your scanners.  
#
###############################################################################################################

Usage () {
	echo 
	echo "$0 Release ver. " `cat RELEASE_VERSION`
	echo "Usage: $0"
	echo 
        echo "This shell script will set up all site specific information for data importing into the BIRN data hierarchy appropriate for FIPS analysis"
        echo "This installation will allow you to specify the types and formats for MRI scanners at your site, installation directories, and output directories used to store images once imported"
        echo
	echo "See the Readme.html file for help."
	echo; }


echo
echo "###################################################"
echo "#Local Human Data Script Installation Utility    #"
echo "###################################################"
echo

############################################
#Init some vars
############################################

DICOMPathSet=0
PFilePathSet=0
FFilePathSet=0
IowaSignaPathSet=0
SignaPathSet=0
BXHAnalyzePathSet=0
Use_cx_Oracle=0
EXTRA_LIBDIRS=

if [ x"$ORACLE_HOME" = x ]; then
	echo "WARNING: Cannot find ORACLE_HOME.  This means that cx_Oracle will not work,"
	echo "and any database connectivity will have to fall back to the (slow) Java method"
	Use_cx_Oracle=0
else
	EXTRA_LIBDIRS="$ORACLE_HOME/lib/"
fi

export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:$EXTRA_LIBDIRS

############################################
#Read existing ConfigureSite for defaults
############################################
if [ -f ConfigureSite.cache -o -f ConfigureSite -o ConfigureSite.new ] ; then
	if [ -f ConfigureSite.cache ] ; then
		source ConfigureSite.cache
	elif [ -f ConfigureSite ] ; then
		source ConfigureSite
	elif [ -f ConfigureSite.new ] ; then
		source ConfigureSite.new
	fi
	[ -n "${ScriptInstallDir+x}" ] && INSTALLDIR=${ScriptInstallDir}
	[ -n "${InstitutionID+x}" ] && InstitutionID=${InstitutionID}
	[ -n "${AcquisitionSiteID+x}" ] && AcquisitionSiteID=${AcquisitionSiteID}
	[ -n "${TempDIR+x}" ] && TempDIR=${TempDIR}
	[ -n "${DBHost+x}" ] && host=${DBHost}
	[ -n "${DBPort+x}" ] && port=${DBPort}
	[ -n "${DBInstance+x}" ] && instance=${DBInstance}
	[ -n "${DBUser+x}" ] && username=${DBUser}
	[ -n "${DBPass+x}" ] && password=${DBPass}
	[ -n "${DBType+x}" ] && dbtype=${DBType}
	[ -n "${GridFTPHost+x}" ] && gridftphost=${GridFTPHost}
	[ -n "${GridFTPPort+x}" ] && gridftpport=${GridFTPPort}
	[ -n "${MyProxyHost+x}" ] && myproxyhost=${MyProxyHost}
	[ -n "${MyProxyPort+x}" ] && myproxyport=${MyProxyPort}
fi

# This function asks for a response (with prompt) and stores it in the
# variable named by varname.  If response is blank, don't change the variable.
function read_variable_with_default {
	varname=$1
	prompt=$2
	echo "${prompt}"
	if [ -n "${!varname+x}" ] ; then
		# variable named by $varname is set
		echo -n "[press return for '${!varname}'] "
	fi
	read response
	if [ x"${response}" != x ] ; then
        	# got a response -- set the variable
		eval "${varname}=\${response}"
	fi
}

touch ConfigureSite.cache
trap 'if [ -f ConfigureSite.cache.tmp ] ; then cat ConfigureSite.cache.tmp >> ConfigureSite.cache ; rm ConfigureSite.cache.tmp ; fi ; exit 1' INT
rm -f ConfigureSite.cache.tmp
touch ConfigureSite.cache.tmp

############################################
#Set up Installation Directories...
############################################
read_variable_with_default INSTALLDIR "Please Enter Directory to Install Scripts:"
if [ -d $INSTALLDIR ]
then
	curinode=`ls -i -d . | awk '{print $1}'`
	installinode=`ls -i -d ${INSTALLDIR} | awk '{print $1}'`
	if [ $curinode -eq $installinode ] ; then
	    echo "ERROR: you cannot install into the current directory!"
	    exit 1
	fi
        read -p "$INSTALLDIR exists.  Would you like to remove it? [y/n] " REMOVE_IT
        if [ "$REMOVE_IT" = "y" ]
        then
            echo "Removing ${INSTALLDIR}..."
            rm -rf $INSTALLDIR
        fi
fi	

echo "Creating Directory ${INSTALLDIR}..."
mkdir $INSTALLDIR
echo "ScriptInstallDir=\"${INSTALLDIR}\"" >> ConfigureSite.cache.tmp


############################################
# Collect Additional Site-Specific Metadata
############################################
Institution[3]="duke-unc"
Institution[5]="harvard-bwh"
Institution[6]="harvard-mgh"
Institution[7]="ucla-loni"
Institution[8]="ucsd-fmri"
Institution[9]="uci-bic"
Institution[10]="unm-hsc"
Institution[11]="stanford-lucas"
Institution[12]="ui-mhcrc"
Institution[13]="umn-cmrr"
Institution[14]="jhu-cis"
Institution[18]="yale"
Institution[20]="ucsf-cind"
Institution[25]="nih-cit"
Institution[9999]="unknown"
siteIDList=(3 5 6 7 8 9 10 11 12 13 14 18 20 25 9999)
siteIDs=""
for i in "${siteIDList[@]}" ; do
  siteIDs="${siteIDs}:${i}"
done
siteIDs="${siteIDs}:"

echo 
echo "Please Enter Uploader's InstitutionID.  This may be used to "
echo "set group permissions on the remote site."
for i in "${siteIDList[@]}" ; do
	echo "		${i}:	${Institution[$i]}"
done
while true
do
	read_variable_with_default InstitutionID 'Please Enter a Valid InstitutionID:'
	if [ x != x`echo $siteIDs | grep ":${InstitutionID}:"` ] ; then
	    break
	fi
	unset InstitutionID
done
echo "Institution=\"${Institution[$InstitutionID]}\"" >> ConfigureSite.cache.tmp
echo "InstitutionID=${InstitutionID}" >> ConfigureSite.cache.tmp

echo 
echo "Please Enter The Acquisition Site ID.  This will be used to"
echo "label/name directories on the remote site (is overridden by"
echo "<acquisitionSiteID> element in upload XML config files for"
echo "human data).  In most cases, this will be the same as the"
echo "Uploader's InstitutionID (you specified ${InstitutionID})."
echo "You may wish to specify a different ID if you are uploading data"
echo "on behalf of another site, but you want to retain ownership"
echo "of the data."
for i in "${siteIDList[@]}" ; do
	echo "		${i}:	${Institution[$i]}"
done
while true
do
	read_variable_with_default AcquisitionSiteID 'Please Enter a Valid AcquisitionSiteID'
	if [ x != x`echo $siteIDs | grep ":${AcquisitionSiteID}:"` ] ; then
	    break
	fi
	unset AcquisitionSiteID
done
echo "AcquisitionSite=\"${Institution[$AcquisitionSiteID]}\"" >> ConfigureSite.cache.tmp
echo "AcquisitionSiteID=${AcquisitionSiteID}" >> ConfigureSite.cache.tmp

echo
read_variable_with_default TempDIR "Please enter a temporary staging directory for your uploads - storage required ~3 * input data size (will be cleaned after upload):"
if [ -d $TempDIR ]
then
	read -p "$TempDIR exists.  Would you like to remove it? [y/n] " REMOVE_IT
        if [ "$REMOVE_IT" = "y" ]
        then
            echo "Removing ${TempDIR}..."
            rm -rf $TempDIR
            mkdir $TempDIR
        else
            echo "WARNING: Some information in $TempDIR may be overwritten during uploads!"
            
        fi
else
	echo "Making $TempDIR directory...."
	mkdir $TempDIR
fi
echo "TempDIR=\"${TempDIR}\"" >> ConfigureSite.cache.tmp

#############################################################################
#Sets up GridFTP information
#############################################################################
gridftpvalues=
echo
if [ -n "${gridftphost+x}" -o -n "${gridftpport+x}" -o -n "${myproxyhost+x}" -o -n "${myproxyport}" ] ; then
	configGridFTP=y
fi
read_variable_with_default configGridFTP "Would you like to configure GridFTP support during uploads [y/n]"

if [ "$configGridFTP" = "y" ]
then 
	read_variable_with_default gridftphost "Please input the GridFTP server hostname:"
	read_variable_with_default gridftpport "Please input the GridFTP server port (usually 2811):"
	read_variable_with_default myproxyhost "Please input the MyProxy server hostname (e.g. certs.nbirn.org):"
        read_variable_with_default myproxyport "Please input the MyProxy server port (usually 7512):"
	echo "GridFTPHost=\"$gridftphost\"
GridFTPPort=\"$gridftpport\"
MyProxyHost=\"$myproxyhost\"
MyProxyPort=\"$myproxyport\"" >> ConfigureSite.cache.tmp
fi

#############################################################################
#Sets up Database information
#############################################################################
echo
if [ -n "${username+x}" -o -n "${password+x}" -o -n "${instance+x}" -o -n "${dbtype+x}" -o -n "${host+x}" -o -n "${port+x}" ] ; then
	configDB=y
fi
read_variable_with_default configDB "Would you like to configure Human Imaging Database (HID) support during uploads [y/n]"

if [ "$configDB" = "y" ]
then 
	read_variable_with_default username "Please input the user name to connect to the database (e.g. postgres):"
	echo "DBUser=\"$username\"" >> ConfigureSite.cache.tmp
	echo "User=\"$username\"" >> ConfigureSite.cache.tmp
	read_variable_with_default password "Please input the password:"
	echo "DBPass=\"$password\"" >> ConfigureSite.cache.tmp
	read_variable_with_default instance "Please input the database instance/SID (for example, hidprd):"
	echo "DBInstance=\"$instance\"" >> ConfigureSite.cache.tmp
        read_variable_with_default dbtype "Please input the database type (postgres/oracle):"
	echo "DBType=\"$dbtype\"" >> ConfigureSite.cache.tmp
	read_variable_with_default host "Please input the database host (required for postgres databases):"
	echo "DBHost=\"$host\"" >> ConfigureSite.cache.tmp
        if [ x"$host" != x ]; then
		read_variable_with_default port "Please input the database port (usually 1521 for Oracle, 5432 for postgres):"
		echo "DBPort=$port" >> ConfigureSite.cache.tmp
	fi
	if [ "$dbtype" = "postgres" ] ; then
		echo "HostString=\"${host}:${port}/${instance}\"" >> ConfigureSite.cache.tmp
	elif [ "$dbtype" = "oracle" ] ; then
		echo "HostString=\"${host}:${port}:${instance}\"" >> ConfigureSite.cache.tmp
	fi

	#############################################################################
	#inform the user to update database first before running upload.
	#############################################################################
	echo "##########################################################"
	echo "#Testing to see if the database is accessible"
	echo "##########################################################"
	
	cwd=`pwd`
	if [ $Use_cx_Oracle -eq 1 ]; then
		result=`./dbtest.py $cwd 1 $username $password $instance $host $port`

		if [ x"$result" != x1 ];
		then
			echo "WARNING: cx_Oracle connection test failed.  Reverging to (slower) Java method."
			Use_cx_Oracle=0
		else
			echo "Connection test successful!"
		fi
	fi
	
	if [ $Use_cx_Oracle -eq 0 ]; then
		echo "./dbtest.py $cwd 0 $username $password $instance $dbtype $host $port"
		result=`./dbtest.py $cwd 0 $username $password $instance $dbtype $host $port`

		if [ x$result != x1 ];
		then
			echo "ERROR: Java test failed.  Please update your database before running upload."
			exit 1
		else
			echo "Connection test successful!"
		fi
	fi
fi

################################################################################
##Create Configuration.sh script which will assign ENV variables based on system
################################################################################

echo
echo "Creating Configuration Files...."

echo "#!/bin/bash" > Upload.sh
echo "#" >> Upload.sh
echo "# A simple wrapper around Upload.py" >> Upload.sh
echo "#" >> Upload.sh
echo "" >> Upload.sh
echo 'export PATH=$PATH:'"${INSTALLDIR}" >> Upload.sh
echo export PATH="${INSTALLDIR}/bin:"'$PATH' >> Upload.sh
echo "export LD_LIBRARY_PATH="'$LD_LIBRARY_PATH'":$ORACLE_LIBDIR" >> Upload.sh
echo "env ScriptInstallDir=\"${INSTALLDIR}\" \"${INSTALLDIR}/Upload.py\""' $*' >> Upload.sh

#IBO
echo "UseCXOracle=$Use_cx_Oracle" >> ConfigureSite.cache.tmp

if [ "$configDB" != "y" ] ; then
	echo
	read -p "How many scanners would you like to configure support for?: " NumScanners
	for ((i=1; i <= NumScanners; i+=1)); 
	do
		read -p "Please Enter How You Will Refer to Scanner #${i}: " LocalScannerName[$i]
		#LocalScannerName[$i]=`echo ${LocalScannerName[$i]} | tr ' ' '_'`
		read -p "Please Enter the Scanner #${i} Manufacturer: " Manufacturer[$i]
		
		###################################################
		#Collect format specifier of the geometric phantom data
		###################################################
		echo
		echo "Please Specify your Native Format for geometric phantoms: "
		echo "          Auto detect    =         0"
		echo "                DICOM    =         1"
		echo "             GE PFile    =         2"
		echo "  Stanford F/MAG File    =         3"
		echo "              UCI RAW    =         4"
		echo "      Iowa GE5.x/Sign    =         5"
		echo "          Analyze 7.5    =         6"
	 	echo "                 MINC    =         7"
		echo "  Normal GE Signa 5.x    =         8"
		read -p "Format Specifier: " GeometricImageFormat[$i]
	
		if ( [ ${GeometricImageFormat[$i]} -lt 0 ] || [ ${GeometricImageFormat[$i]} -gt 8 ] )
		then
			echo "Error, Please Enter a Valid ${GeometricImageFormat[$i]}"
			echo 'Installation Aborted!'
			exit
		fi
	
		###################################################
		#Collect format specifier of the stability phantom data
		###################################################
		echo
		echo "Please Specify your Native Format for stability phantoms: "
		echo "          Auto detect    =         0"
		echo "                DICOM    =         1"
		echo "             GE PFile    =         2"
		echo "  Stanford F/MAG File    =         3"
		echo "              UCI RAW    =         4"
		echo "      Iowa GE5.x/Sign    =         5"
		echo "          Analyze 7.5    =         6"
	 	echo "                 MINC    =         7"
		echo "  Normal GE Signa 5.x    =         8"
		read -p "Format Specifier: " StabilityImageFormat[$i]
	
		if ( [ ${StabilityImageFormat[$i]} -lt 0 ] || [ ${StabilityImageFormat[$i]} -gt 8 ] )
		then
			echo "Error, Please Enter a Valid ${StabilityImageFormat[$i]}"
			echo 'Installation Aborted!'
			exit
		fi
	
		###################################################
		#Collect format specifier of the EPI Images
		###################################################
		echo
		echo "Please Specify your Native Format for EPI Data: "
		echo "          Auto detect    =         0"
		echo "                DICOM    =         1"
		echo "             GE PFile    =         2"
		echo "  Stanford F/MAG File    =         3"
		echo "              UCI RAW    =         4"
		echo "      Iowa GE5.x/Sign    =         5"
		echo "          Analyze 7.5    =         6"
	 	echo "                 MINC    =         7"
		echo "  Normal GE Signa 5.x    =         8"
		read -p "Format Specifier: " GImageFormat[$i]
	
		if ( [ ${GImageFormat[$i]} -lt 0 ] || [ ${GImageFormat[$i]} -gt 8 ] )
		then
			echo "Error, Please Enter a Valid ${GImageFormat[$i]}"
			echo 'Installation Aborted!'
			exit
		fi
	
		####################################################
		#Collect format specifier of the Structural Images
		####################################################
		echo
		echo "Please Specify your Native Format for Structural Data: "
		echo "          Auto detect    =         0"
		echo "                DICOM    =         1"
		echo "             GE PFile    =         2"
		echo "  Stanford F/MAG File    =         3"
		echo "              UCI RAW    =         4"
		echo "     Iowa GE5.x/Signa    =         5"
		echo "          Analyze 7.5    =         6"
		echo "                 MINC    =         7"
		echo "  Normal GE Signa 5.x    =         8"
				
		read -p "Format Specifier: " SImageFormat[$i]
	
		if ( [ ${SImageFormat[$i]} -lt 0 ] || [ ${SImageFormat[$i]} -gt 8 ] )
		then
			echo "Error, Please Enter a Valid ${SImageFormat[$i]}"
			echo 'Installation Aborted!'
			exit
		fi
		
		#################################################################
		#Collect format specifier of the Anonymized Structural Images
		#################################################################
		echo
		echo "Please Specify your Native Format for Anonymized Structural Data: "
		echo "          Auto detect    =         0"
		echo "                DICOM    =         1"
		echo "             GE PFile    =         2"
		echo "  Stanford F/MAG File    =         3"
		echo "              UCI RAW    =         4"
		echo "     Iowa GE5.x/Signa    =         5"
		echo "          Analyze 7.5    =         6"
		echo "                 MINC    =         7"
		echo "  Normal GE Signa 5.x    =         8"
				
		read -p "Format Specifier: " SAnonImageFormat[$i]
	
		if ( [ ${SAnonImageFormat[$i]} -lt 0 ] || [ ${SAnonImageFormat[$i]} -gt 8 ] )
		then
			echo "Error, Please Enter a Valid ${SAnonImageFormat[$i]}"
			echo 'Installation Aborted!'
			exit
		fi
		
		echo
		read -p "Will you be uploading image segmentations [y/n]?: " segment[$i]
		if [ `echo ${segment[$i]}` = "y" ] 
		then 
				
			#################################################################
			#Collect format specifier of the Structural Image Segmentations
			#################################################################
			echo
			echo "Please Specify your Native Format for Structural Data Segmentations: "
			echo "          Auto detect    =         0"
			echo "                DICOM    =         1"
			echo "             GE PFile    =         2"
			echo "  Stanford F/MAG File    =         3"
			echo "              UCI RAW    =         4"
			echo "     Iowa GE5.x/Signa    =         5"
			echo "          Analyze 7.5    =         6"
			echo "                 MINC    =         7"
			echo "  Normal GE Signa 5.x    =         8"
			read -p "Format Specifier: " SSegmentImageFormat[$i]
		
			if ( [ ${SSegmentImageFormat[$i]} -lt 0 ] || [ ${SSegmentImageFormat[$i]} -gt 8 ] )
			then
				echo "Error, Please Enter a Valid ${SSegmentImageFormat[$i]}"
				echo 'Installation Aborted!'
				exit 1
			fi
		fi
		
		echo
		read -p "Will you be uploading behavioral data [y/n]?: "  behavior[$i]
	 	if [ `echo ${behavior[$i]}` = "y" ] 
		then 
			echo
			read -p "Please enter the name of the behavioral data collection equipment: " BehavioralEquip[$i]
		
		
			#################################################################
			#Collect format specifier of the Behavioral data
			#################################################################
			echo
			echo "Please Specify your Behavioral data format: "
			echo "		EPRIME   =         9"
			echo "		CIGAL    =        10"
			echo "		OTHER    =	 9999"
			read -p "Format Specifier: " BehavioralFormat[$i]
			if ( [ ${BehavioralFormat[$i]} -ne 9 ] && [ ${BehavioralFormat[$i]} -ne 10 ] && [ ${BehavioralFormat[$i]} -ne 9999 ]  )
			then
				echo "Error, Please Enter a Valid ${BehavioralFormat[$i]}"
				echo 'Installation Aborted!'
				exit
			fi
		fi
	done

	for ((i=1; i <= NumScanners; i+=1)); 
	do
		
		LocalScannerName[$i]=`echo ${LocalScannerName[$i]}|sed 's/ /_/g'`
		
		echo "Installing scanner configuration file: ${LocalScannerName[$i]}"
		
		cat ConfigureSite.cache.tmp > Configure_${LocalScannerName[$i]}
	
		echo "exporting format info"
		echo "NativeFormat=${GImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		echo "GeometricNativeFormat=${GeometricImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		echo "StabilityNativeFormat=${StabilityImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		echo "StructFormat=${SImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		echo "StructAnonFormat=${SAnonImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		if [ ${segment[$i]} = "y" ]
		then	
			echo "SegmentFormat=${SSegmentImageFormat[$i]}" >> Configure_${LocalScannerName[$i]}
		fi
		
		echo "exporting scanner info"
		
		Manufacturer[$i]=`echo ${Manufacturer[$i]} | tr '[a-z]' '[A-Z]'`
		echo "ScannerManufacturer=\"${Manufacturer[$i]}\"" >> Configure_${LocalScannerName[$i]}
				
		echo "exporting behavior info"
		if [ ${behavior[$i]} = "y" ]
		then	
			echo "BehavioralFormat=${BehavioralFormat[$i]}" >> Configure_${LocalScannerName[$i]}
			echo "BehavioralEquipment=\"${BehavioralEquip[$i]}\"" >> Configure_${LocalScannerName[$i]}
		fi

	done


fi

	
############################################
#Remove Previous Configuration File
############################################
echo "Removing previous configuration files..."
if [ "$configDB" == "y" ] ; then
    rm -f Configure_*
fi
rm -f ConfigureSite ConfigureSite.cache

mv ConfigureSite.cache.tmp ConfigureSite.cache
cp ConfigureSite.cache ${INSTALLDIR}/ConfigureSite

./ReInstall.sh "${INSTALLDIR}"

rm Upload.sh

echo "Installation successful!"

if [ "$configDB" = "y" ] ; then
	echo "Now do the following to extract scanner info from your database:"
	echo "  cd ${INSTALLDIR}"
	echo "  ./ScannerInfo.sh"
fi

exit 0

