/* @author Syam Gadde
 * @version $Id$
 */

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;

import org.globus.myproxy.MyProxy;
import org.globus.replica.rls.CatalogQuery;
import org.globus.replica.rls.LocalReplicaCatalog;
import org.globus.replica.rls.Mapping;
import org.globus.replica.rls.MappingResult;
import org.globus.replica.rls.RLSConnection;
import org.globus.replica.rls.RLSException;
import org.globus.replica.rls.RLSStatusCode;
import org.globus.replica.rls.Results;
import org.globus.replica.rls.SimpleCatalogQuery;
import org.globus.replica.rls.SimpleQuery;
import org.globus.replica.rls.impl.SimpleRLSConnectionSource;
import org.globus.util.GlobusURL;
import org.ietf.jgss.GSSCredential;

import org.birncommunity.sample.proxy.LocalCredentialHelper;

public class RLSCommand {
    protected boolean debug = false;
    protected int debugIndent = -1;
    protected String debugFunctionName = null;
    protected RLSConnection rlsconn = null;
    public final static int FILE = 1;
    public final static int DIR = 2;
    public final static int UNKNOWN = 3;

    protected class EmbeddedException
	extends Exception
    {
	static final long serialVersionUID = 0;
	public EmbeddedException(String msg, Throwable cause)
	{
	    super(msg, cause);
	}
	public EmbeddedException(String msg)
	{
	    super(msg);
	}
    }

    public RLSCommand()
    {
    }

    public void debugOn()
    {
        this.debug = true;
    }
    public void debugOff()
    {
        this.debug = false;
    }
    public void debugEnter(String functionName)
    {
	this.debugFunctionName = functionName;
        this.debugIndent += 1;
    }
    public void debugExit()
    {
        this.debugIndent -= 1;
    }
    public void debugMsg(String msg)
    {
        if (!this.debug) {
            return;
        }
        int i;
        for (i = 0; i < this.debugIndent; i++) {
            System.err.print("| ");
        }
        System.err.print("DEBUG: ");
	if (this.debugFunctionName != null) {
	    System.err.print("RLSCommand::" + this.debugFunctionName + ": ");
	}
        System.err.println(msg);
    }
    public void debugMsgArgs(String msg)
    {
        if (!this.debug) {
            return;
        }
        int i;
        for (i = 0; i < this.debugIndent; i++) {
            System.err.print("| ");
        }
        System.err.print("DEBUG: ");
	if (this.debugFunctionName != null) {
	    System.err.print("RLSCommand::" + this.debugFunctionName);
	}
	System.err.print("(");
        System.err.print(msg);
	System.err.println(")");
    }

    public void connect(String myproxyhost, int myproxyport, String myproxyuser, String myproxypwd, String rlsURL)
	throws EmbeddedException
    {
        this.debugEnter("connect");
        this.debugMsgArgs("'" + myproxyhost + "', " + myproxyport + ", '" + myproxyuser + "', '" + myproxypwd + "'");
	GSSCredential credential = null;
	if (myproxyuser == null || myproxypwd == null) {
	    try {
		credential = new LocalCredentialHelper().getDefaultCredential();
	    } catch (Exception e) {
		this.debugExit();
		throw new EmbeddedException("Error getting local credential (maybe you need to renew your credential, or otherwise, set the MYPROXY_USER and MYPROXY_PWD environment variables)", e);
	    }
	} else {
	    MyProxy proxy = new MyProxy(myproxyhost, 7512);
	    try {
		credential = proxy.get(myproxyuser, myproxypwd, 0);
	    } catch (Exception e) {
		this.debugExit();
		throw new EmbeddedException("Error connecting or authenticating to myproxy", e);
	    }
	}
	SimpleRLSConnectionSource source = new SimpleRLSConnectionSource();
	GlobusURL url = null;
	try {
	    url = new GlobusURL(rlsURL);
	} catch (Exception e) {
	    this.debugExit();
	    throw new EmbeddedException("Error creating GlobusURL", e);
	}
	try {
	    this.rlsconn = source.connect(url, credential);
	} catch (Exception e) {
	    this.debugExit();
	    throw new EmbeddedException("Error connecting to RLS at " + rlsURL, e);
	}
        this.debugExit();
    }

    public void disconnect()
	throws EmbeddedException
    {
        this.debugEnter("disconnect");
	if (this.rlsconn != null) {
	    try {
		this.rlsconn.close();
	    } catch (RLSException e) {
		this.debugExit();
		throw new EmbeddedException("Error disconnecting", e);
	    }
	}
	this.debugExit();
    }

    protected void reportSuccess(String message)
    {
        this.debugEnter("reportSucess");
	System.out.println("SUCCESS");
	if (message != null) {
	    System.out.println("START MESSAGE");
	    System.out.print(message);
	    if (message.length() > 0 &&
		message.charAt(message.length()-1) != '\n') {
		System.out.print("\n");
	    }
	    System.out.println("END MESSAGE");
	}
        this.debugExit();
    }
    protected void reportFailure(String message)
    {
        this.debugEnter("reportFailure");
	System.out.println("FAILURE");
	if (message != null) {
	    System.out.println("START MESSAGE");
	    System.out.print(message);
	    if (message.charAt(message.length()-1) != '\n') {
		System.out.print("\n");
	    }
	    System.out.println("END MESSAGE");
	}
        this.debugExit();
    }
    protected void reportFailure(Exception e)
    {
        this.debugEnter("reportFailure");
	System.out.println("FAILURE");
	System.out.println("START MESSAGE");
	e.printStackTrace(System.out);
	System.out.println("END MESSAGE");
        this.debugExit();
    }

    static String usageTxt = "Usage: RLSCommand [--debug] myproxyhost myproxyport rlsURL\n";

    public static void usage() {
	System.err.println(usageTxt);
	System.exit(1);
    }

    public void run(String [] cmdandargs)
    {
        this.debugEnter("run");
        if (this.debug) {
            String msg = new String("");
            msg = msg + "with args([";
            int i;
            for (i = 0; i < cmdandargs.length; i++) {
                msg = msg + "'" + cmdandargs[i] + "'";
                if (i < cmdandargs.length - 1) {
                    msg = msg + ", ";
                }
            }
            msg = msg + "])";
            this.debugMsg(msg);
        }
	String successStr = null;
	try {
	    String command = cmdandargs[0];
	    if (command.equals("help")) {
		successStr = "Commands:\n debug [on|off]\n create <logicalName> <target>\n add <logicalName> <target>\n query <logicalName>\n wildquery <logicalNamePattern>\n delete <logicalName> <target>\n deleteall <logicalName>\n";
	    } else if (command.equals("debug")) {
		if (cmdandargs.length != 2) {
		    this.reportFailure("'debug' command requires one argument ('on' or 'off')!");
		    this.debugExit();
		    return;
		}
		if (cmdandargs[1].equals("on")) {
		    this.debugOn();
		    successStr = "Debugging turned on";
		} else if (cmdandargs[1].equals("off")) {
		    this.debugOff();
		    successStr = "Debugging turned off";
		} else {
		    this.reportFailure("'debug' command requires one argument ('on' or 'off')!");
		    this.debugExit();
		    return;
		}
	    } else if (command.equals("create")) {
		if (cmdandargs.length == 3) {
		    this.cmd_create(cmdandargs[1], cmdandargs[2]);
		} else {
		    this.reportFailure("Wrong number of arguments to 'create'!");
		    this.debugExit();
		    return;
		}
	    } else if (command.equals("add")) {
		if (cmdandargs.length == 3) {
		    this.cmd_add(cmdandargs[1], cmdandargs[2]);
		} else {
		    this.reportFailure("Wrong number of arguments to 'add'!");
		    this.debugExit();
		    return;
		}
	    } else if (command.equals("query")) {
		if (cmdandargs.length < 2) {
		    this.reportFailure("Not enough arguments to 'query'!");
		    this.debugExit();
		    return;
		}
		try {
		    successStr = this.cmd_query(cmdandargs[1]);
		} catch (Exception e) {
		    this.reportFailure(e.toString());
		}
	    } else if (command.equals("wildquery")) {
		if (cmdandargs.length != 2) {
		    this.reportFailure("Wrong number of arguments to 'wildquery'!");
		    this.debugExit();
		    return;
		}
		try {
		    successStr = this.cmd_wildquery(cmdandargs[1]);
		} catch (Exception e) {
		    this.reportFailure(e.toString());
		}
	    } else if (command.equals("delete")) {
		if (cmdandargs.length != 3) {
		    this.reportFailure("Wrong number of arguments to 'delete'!");
		    this.debugExit();
		    return;
		}
		this.cmd_delete(cmdandargs[1], cmdandargs[2]);
	    } else if (command.equals("deleteall")) {
		if (cmdandargs.length != 2) {
		    this.reportFailure("Wrong number of arguments to 'deleteall'!");
		    this.debugExit();
		    return;
		}
		this.cmd_deleteall(cmdandargs[1]);
	    } else {
		this.reportFailure("Unsupported command '" + command + "'");
		this.debugExit();
		return;
	    }
	    this.reportSuccess(successStr);
	} catch (EmbeddedException e) {
	    this.reportFailure(e);
	}
        this.debugExit();
    }

    protected void cmd_create(String logicalName, String target)
	throws EmbeddedException
    {
        this.debugEnter("cmd_create");
        this.debugMsgArgs("'" + logicalName + "', " + target + "'");
	List mappings = new LinkedList();
	mappings.add(new Mapping(logicalName, target));
	LocalReplicaCatalog catalog = this.rlsconn.catalog();
	try {
	    List errors = catalog.createMappings(mappings);
	    if (errors.size() > 0) {
		String errorMsg = new String();
		Iterator eiter = errors.iterator();
		while (eiter.hasNext()) {
		    MappingResult result = (MappingResult)eiter.next();
		    errorMsg.concat(result.toString());
		}
		this.debugExit();
		throw new EmbeddedException("Error adding mapping to RLS:\n" + errorMsg);
	    }
	} catch (RLSException e) {
            this.debugExit();
	    throw new EmbeddedException("Error adding mapping to RLS", e);
	}
	this.debugExit();
    }

    protected void cmd_add(String logicalName, String target)
	throws EmbeddedException
    {
        this.debugEnter("cmd_add");
        this.debugMsgArgs("'" + logicalName + "', " + target + "'");
	List mappings = new LinkedList();
	mappings.add(new Mapping(logicalName, target));
	LocalReplicaCatalog catalog = this.rlsconn.catalog();
	try {
	    List errors = catalog.addMappings(mappings);
	    if (errors.size() > 0) {
		String errorMsg = new String();
		Iterator eiter = errors.iterator();
		while (eiter.hasNext()) {
		    MappingResult result = (MappingResult)eiter.next();
		    errorMsg.concat(result.toString());
		}
		this.debugExit();
		throw new EmbeddedException("Error adding mapping to RLS:\n" + errorMsg);
	    }
	} catch (RLSException e) {
            this.debugExit();
	    throw new EmbeddedException("Error adding mapping to RLS", e);
	}
	this.debugExit();
    }

    protected String doQuery(SimpleCatalogQuery query)
	throws EmbeddedException
    {
        this.debugEnter("doQuery");
	String offsetlimitstr = new String("<null>");
	if (query.getOffsetLimit() != null) {
	    offsetlimitstr = "[offset=" + query.getOffsetLimit().offset + ", reslimit=" + query.getOffsetLimit().reslimit + "]";
	}
        this.debugMsgArgs("Query[offsetlimit=" + offsetlimitstr + ", param='" + query.getParam().toString() + "', type=" + query.getType().toString() + "]");
	List mapresults = new ArrayList();
	try {
	    LocalReplicaCatalog catalog = this.rlsconn.catalog();
	    Results results = catalog.query(query);
	    if (results.getRC() == RLSStatusCode.RLS_SUCCESS) {
		List batch = results.getBatch();
		Iterator i = batch.iterator();
		while (i.hasNext()) {
		    MappingResult result = (MappingResult) i.next();
		    if (result.getRC() != RLSStatusCode.RLS_SUCCESS)
			continue;
		    mapresults.add(result);
		}
	    }
	} catch (RLSException e) {
            this.debugExit();
	    return null;
	}
	StringBuffer output = new StringBuffer();
	Iterator iter = mapresults.iterator();
	while (iter.hasNext()) {
	    MappingResult result = (MappingResult)iter.next();
	    output.append(result.getLogical());
	    output.append("\t");
	    output.append(result.getTarget());
	    output.append("\n");
	}
	this.debugExit();
	return output.toString();
    }

    protected String cmd_query(String logicalName)
	throws EmbeddedException
    {
        this.debugEnter("cmd_query");
        this.debugMsgArgs("'" + logicalName + "'");
	SimpleCatalogQuery catalogQuery = new SimpleCatalogQuery(SimpleCatalogQuery.queryMappingsByLogicalName, logicalName, null);
	String retval = this.doQuery(catalogQuery);
	this.debugExit();
	return retval;
    }

    protected String cmd_wildquery(String logicalName)
	throws EmbeddedException
    {
        this.debugEnter("cmd_wildquery");
        this.debugMsgArgs("'" + logicalName + "'");
	SimpleCatalogQuery catalogQuery = new SimpleCatalogQuery(SimpleCatalogQuery.queryMappingsByLogicalNamePattern, logicalName, null);
	String retval = this.doQuery(catalogQuery);
	this.debugExit();
	return retval;
    }

    protected void deleteMappings(List mappings)
	throws EmbeddedException
    {
        this.debugEnter("deleteMappings");
	String mappingsStr = new String();
	mappingsStr.concat("List: [ ");
	Iterator iter = mappings.iterator();
	while (iter.hasNext()) {
	    Mapping mapping = (Mapping)iter.next();
	    mappingsStr.concat("'Mapping: " + mapping.getLogical() + "' => '" + mapping.getTarget() + "'");
	    if (iter.hasNext()) {
		mappingsStr.concat(", ");
	    }
	}
	mappingsStr.concat(" ]");
	this.debugMsgArgs(mappingsStr);
	LocalReplicaCatalog catalog = this.rlsconn.catalog();
	try {
	    List errors = catalog.deleteMappings(mappings);
	    if (errors.size() > 0) {
		String errorMsg = new String();
		Iterator eiter = errors.iterator();
		while (eiter.hasNext()) {
		    MappingResult result = (MappingResult)eiter.next();
		    errorMsg.concat(result.toString());
		}
		this.debugExit();
		throw new EmbeddedException("Error removing mapping from RLS:\n" + errorMsg);
	    }
	} catch (RLSException e) {
            this.debugExit();
	    throw new EmbeddedException("Error removing mapping from RLS", e);
	}
	this.debugExit();
    }

    protected void cmd_delete(String logicalName, String target)
	throws EmbeddedException
    {
        this.debugEnter("cmd_delete");
        this.debugMsgArgs("'" + logicalName + "'");
	List mappings = new LinkedList();
	mappings.add(new Mapping(logicalName, target));
	this.deleteMappings(mappings);
	this.debugExit();
    }

    protected void cmd_deleteall(String logicalName)
	throws EmbeddedException
    {
        this.debugEnter("cmd_deleteall");
        this.debugMsgArgs("'" + logicalName + "'");
	CatalogQuery catalogQuery = new SimpleCatalogQuery(SimpleCatalogQuery.queryMappingsByLogicalName, logicalName, null);
	List mappings = new LinkedList();
	LocalReplicaCatalog catalog = this.rlsconn.catalog();
	Results results = null;
	try {
	    results = catalog.query(catalogQuery);
	} catch (RLSException e) {
	    /* failed query is OK */
	    this.debugExit();
	    return;
	}
	if (results.getRC() == RLSStatusCode.RLS_SUCCESS) {
	    List batch = results.getBatch();
	    Iterator i = batch.iterator();
	    while (i.hasNext()) {
		MappingResult result = (MappingResult) i.next();
		if (result.getRC() != RLSStatusCode.RLS_SUCCESS)
		    continue;
		mappings.add(result);
	    }
	    this.deleteMappings(mappings);
	    this.debugExit();
	}
	this.debugExit();
    }

    public static void main(String[] args)
	throws EmbeddedException
    {
	String [] newargs = new String [args.length];
	boolean opt_debug = false;
	boolean ignoreoptions = false;
	int newargslen = 0;
	int i;
	for (i = 0; i < args.length; i++) {
	    String curarg = args[i];
	    if (!ignoreoptions && curarg.startsWith("--")) {
		if (curarg.equals("--")) {
		    ignoreoptions = true;
		    continue;
		}
		if (curarg.equals("--debug")) {
		    opt_debug = true;
		} else {
		    System.err.println("Unrecognized option: " + curarg);
		    System.exit(1);
		}
	    } else {
		newargs[newargslen++] = args[i];
	    }
	}
	if (newargs.length < 3) {
	    RLSCommand.usage();
	}

	String username = System.getenv("MYPROXY_USER");
	String password = System.getenv("MYPROXY_PWD");
	RLSCommand cmdobj = new RLSCommand();
	if (opt_debug) {
	    cmdobj.debugOn();
	}
	cmdobj.connect(newargs[0], Integer.parseInt(newargs[1]), username, password, newargs[2]);

	BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
	final String delim_white = "\t\r\n ";
	final String delim_whitequote = "\t\r\n \"'";
	final String delim_dquote = "\"";
	final String delim_squote = "'";
	final int state_notinquotes = 0;
	final int state_insinglequotes = 1;
	final int state_indoublequotes = 2;
	while (true) {
	    String commandline = null;
	    try {
		System.out.print("> ");
		System.out.flush();
		commandline = br.readLine();
	    } catch (IOException ioe) {
		System.exit(0);
	    }
	    if (commandline == null) {
		System.exit(0);
	    }

	    Vector cmdvector = new Vector();

	    String curdelim = delim_whitequote;
	    int curstate = state_notinquotes;
	    StringTokenizer parser =
		new StringTokenizer(commandline, curdelim, true);
	    String arg = "";
	    while (parser.hasMoreTokens()) {
		String token = parser.nextToken(curdelim);
		if (curstate == state_insinglequotes) {
		    if (token.equals(delim_squote)) {
			curdelim = delim_whitequote;
			curstate = state_notinquotes;
		    } else {
			arg += token;
		    }
		} else if (curstate == state_indoublequotes) {
		    if (token.equals(delim_dquote)) {
			curdelim = delim_whitequote;
			curstate = state_notinquotes;
		    } else {
			arg += token;
		    }
		} else if (curstate == state_notinquotes) {
		    if (token.equals(delim_squote)) {
			curdelim = delim_squote;
			curstate = state_insinglequotes;
		    } else if (token.equals(delim_dquote)) {
			curdelim = delim_dquote;
			curstate = state_indoublequotes;
		    } else {
		        char firstchar = token.charAt(0);
			if (firstchar == '\r' || firstchar == '\n' || firstchar == '\t' || firstchar == ' ') {
			    cmdvector.add(arg);
			    arg = "";
			} else {
			    arg += token;
			}
		    }
		}
	    }
	    if (!arg.equals("")) {
		cmdvector.add(arg);
	    }

	    String [] cmdandargs = new String [cmdvector.size()];
	    for (i = 0; i < cmdvector.size(); i++) {
		cmdandargs[i] = (String) cmdvector.elementAt(i);
	    }
	    if (cmdandargs.length == 0 || cmdandargs[0].equals("")) {
		continue;
	    }
	    if (cmdandargs[0].equals("exit")) {
		cmdobj.disconnect();
		System.exit(0);
	    }
	    cmdobj.run(cmdandargs);
	}
    }
}
