/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.notes;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.nbirn.fbirn.utilities.CredentialManager;
import org.nbirn.fbirn.utilities.ExternalProgressUpdater;
import org.nbirn.fbirn.utilities.ProgressListener;

/**
 *
 * @author gadde
 */
public abstract class NoteTarget implements ExternalProgressUpdater {

    ArrayList<ProgressListener> _listeners = null;
    boolean _canceled = false;
    ExperimentHierarchyModel _model = null;

    protected NoteTarget(ExperimentHierarchyModel model) {
        _model = model;
    }

    public ExperimentHierarchyModel getModel() {
        return _model;
    }

    public void setModel(ExperimentHierarchyModel model) {
        _model = model;
    }

    public void cancel() {
        _canceled = true;
    }

    public boolean isCanceled() {
        return _canceled;
    }

    public void addProgressListener(ProgressListener listener) {
        if (_listeners == null) {
            _listeners = new ArrayList<ProgressListener>();
        }
        _listeners.add(listener);
    }

    public void removeProgressListener(ProgressListener listener) {
        if (_listeners == null) {
            return;
        }
        _listeners.remove(listener);
    }

    public void setProgressStart(String msg) {
        if (_listeners == null) {
            return;
        }
        for (ProgressListener listener : _listeners) {
            listener.progressStart(this, msg);
        }
    }

    public void setProgressFinish(String msg) {
        if (_listeners == null) {
            return;
        }
        for (ProgressListener listener : _listeners) {
            listener.progressFinish(this, msg);
        }
    }

    public void setProgress(double progress, String msg) {
        if (_listeners == null) {
            return;
        }
        for (ProgressListener listener : _listeners) {
            listener.progressUpdate(this, progress, msg);
        }
    }

    protected void checkInterrupts() throws NoteException {
        if (Thread.currentThread().isInterrupted()) {
            throw new NoteException("Operation interrupted.");
        }
    }

    /**
     * @return list of all notes provided by this source
     */
    public final void putNoteList(List<? extends Note> notelist) throws NoteException {
        setProgressStart(null);
        setProgress(0, null);
        try {
            putNoteListInternal(notelist);
            setProgress(1, null);
            setProgressFinish(null);
        } catch (NoteException e) {
            setProgressFinish("Exception: " + e.getMessage());
            throw e;
        }
    }

    /**
     * Put all notes in the given list to the target.
     * @throws Exception on error
     */
    abstract public void putNoteListInternal(List<? extends Note> notelist) throws NoteException;

    public class NotePathIndex implements Comparable<NotePathIndex> {

        public int noteInd;
        public int pathInd;
        public String path;

        NotePathIndex(int noteInd_, int pathInd_, String path_) {
            noteInd = noteInd_;
            pathInd = pathInd_;
            path = path_;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final NotePathIndex other = (NotePathIndex) obj;
            if (this.noteInd != other.noteInd) {
                return false;
            }
            if (this.pathInd != other.pathInd && ((this.path == null) ? (other.path != null) : !this.path.equals(other.path))) {
                return false;
            }
            return true;
        }

        @Override
        public int hashCode() {
            int hash = 7;
            hash = 47 * hash + this.noteInd;
            hash = 47 * hash + this.pathInd;
            hash = 47 * hash + (this.path != null ? this.path.hashCode() : 0);
            return hash;
        }

        public int compareTo(NotePathIndex other) {
            if (other == null) {
                throw new NullPointerException("compareTo() called on a null object");
            }
            if (this.noteInd < other.noteInd) {
                return -1;
            }
            if (this.noteInd > other.noteInd) {
                return 1;
            }
            if (this.pathInd < other.pathInd) {
                return -1;
            }
            if (this.pathInd > other.pathInd) {
                return 1;
            }
            return this.path.compareTo(other.path);
        }
    }

    public class PrivilegeError {

        public String errMsg;
        public NotePathIndex[] errInds;
        public int priv;

        PrivilegeError(String errMsg_, NotePathIndex[] errInds_, int priv_) {
            errMsg = errMsg_;
            errInds = errInds_;
            priv = priv_;
        }
    }

    public NotePrivileges getPrivileges() throws NoteException {
        return null;
    }

    /**
     *
     */
    public List<PrivilegeError> checkPrivileges(List<? extends Note> noteList, CredentialManager credMan) throws NoteException {
        List<PrivilegeError> privErrs = new ArrayList<PrivilegeError>();
        NotePrivileges privs = getPrivileges();
        if (privs == null) {
            return privErrs;
        }
        String curUser = null;
        try {
            GSSCredential cred = credMan.getCredential();
            if (cred != null && cred.getName() != null) {
                curUser = cred.getName().toString();
            }
        } catch (GSSException e) {
            throw new NoteException("Error getting user", e);
        }
        Map<String, List<NotePathIndex>> pathmap = new HashMap<String, List<NotePathIndex>>();
        int numNotes = noteList.size();
        for (int noteInd = 0; noteInd < numNotes; noteInd++) {
            Note note = noteList.get(noteInd);
            int numPaths = note.getPaths().size();
            for (int pathInd = 0; pathInd < numPaths; pathInd++) {
                URIWithType uriwt = note.getPaths().get(pathInd);
                ExperimentHierarchyItem item = null;
                try {
                    item = _model.pathToItem(uriwt);
                } catch (NoteException e) {
                    // XXX determine how these exceptions get passed up
                    continue;
                }
                String path = _model.itemToPath(item).getURI().getPath();
                List<NotePathIndex> mappedlist = null;
                if (pathmap.containsKey(path)) {
                    mappedlist = pathmap.get(path);
                } else {
                    mappedlist = new ArrayList<NotePathIndex>();
                    pathmap.put(path, mappedlist);
                }
                NotePathIndex npind = new NotePathIndex(noteInd, pathInd, path);
                if (!mappedlist.contains(npind)) {
                    mappedlist.add(npind);
                }
            }
        }
        int[] privList = new int[noteList.size()];
        for (int noteInd = 0; noteInd < numNotes; noteInd++) {
            Note note = noteList.get(noteInd);
            String ns = note.getNamespace();
            String name = note.getName();
            String noteUser = curUser;
            if (note.getAuthor().matches("_.*_")) {
                noteUser = note.getAuthor();
            }
            int priv = privs.getPrivilege(ns, name, noteUser);
            privList[noteInd] = priv;
            if (priv == NotePrivileges.PRIV_REJECT) {
                privErrs.add(new PrivilegeError("Target does not allow user '" + (noteUser == null ? "<null>" : noteUser) + "' to write a note with namespace='" + ns + "' and name='" + name + "'.", new NotePathIndex[]{new NotePathIndex(noteInd, 0, null)}, priv));
            }
        }
        for (Map.Entry<String, List<NotePathIndex>> entry : pathmap.entrySet()) {
            String path = entry.getKey();
            List<NotePathIndex> npList = entry.getValue();
            // at this point, we are free to modify npList, because it will never be used again
            ListIterator<NotePathIndex> npIter = npList.listIterator();
            while (npIter.hasNext()) {
                int iterind = npIter.nextIndex();
                NotePathIndex npind = npIter.next();
                int priv = privList[npind.noteInd];
                if (priv == NotePrivileges.PRIV_ALLOW_ONE) {
                    Note note = noteList.get(npind.noteInd);
                    String ns = note.getNamespace();
                    String name = note.getName();
                    List<NotePathIndex> matches = new ArrayList<NotePathIndex>();
                    matches.add(npind);
                    Set<Integer> removeSet = new TreeSet<Integer>();
                    ListIterator<NotePathIndex> npIter2 = npList.listIterator(npIter.nextIndex());
                    while (npIter2.hasNext()) {
                        int iterind2 = npIter2.nextIndex();
                        NotePathIndex npind2 = npIter2.next();
                        Note note2 = noteList.get(npind2.noteInd);
                        String ns2 = note2.getNamespace();
                        String name2 = note2.getName();
                        if (((ns == null) ? (ns2 == null) : ns.equals(ns2)) && ((name == null) ? (name2 == null) : name.equals(name2))) {
                            matches.add(npind2);
                            // remove it so we won't try to look for matches again, but remove it from npIter, not npIter2, to avoid ConcurrentModificationException
                            removeSet.add(iterind2);
                        }
                    }
                    if (removeSet.size() > 0) {
                        while (npIter.hasNext()) {
                            int tmpiterind = npIter.nextIndex();
                            npIter.next();
                            if (removeSet.contains(tmpiterind)) {
                                npIter.remove();
                            }
                        }
                        // reset npIter
                        npIter = npList.listIterator(iterind);
                        if (npIter.hasNext()) {
                            npIter.next();
                        }
                    }
                    if (matches.size() > 1) {
                        privErrs.add(new PrivilegeError("Target privileges do not allow user '" + (curUser == null ? "<null>" : curUser) + "' to write multiple notes with namespace '" + ns + "' and name '" + name + "'.", matches.toArray(new NotePathIndex[matches.size()]), priv));
                    }
                }
            }
        }
        return privErrs;
    }
}
