/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.utilities;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentEvent.ElementChange;
import javax.swing.event.DocumentEvent.EventType;
import javax.swing.event.DocumentListener;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.AbstractDocument;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.text.Position;
import javax.swing.text.Segment;
import javax.swing.text.SimpleAttributeSet;

/**
 *
 * @author gadde
 */
public class AppendOnlyDocument implements Document {

    private AbstractDocument.Content _content;
    private List<DocumentListener> _listeners;
    private Element[] _linemap;
    private int _linemaplen;
    private RootElement _root;
    private final static AttributeSet NULLATTRS = new SimpleAttributeSet();
    private final static int EOFINDEX = Integer.MAX_VALUE;
    private final static Element[] EMPTYELEMARRAY = new Element[0];
    private ReentrantReadWriteLock _lock;
    private ReadLock _readLock;
    private WriteLock _writeLock;
    private final EndElement _endElement;
    private final Comparator<Element> ELEMCOMPARATOR = new ElementComparator();

    public AppendOnlyDocument(AbstractDocument.Content content) {
        _content = content;
        _listeners = new ArrayList<DocumentListener>();
        _lock = new ReentrantReadWriteLock();
        _readLock = _lock.readLock();
        _writeLock = _lock.writeLock();
        _root = new RootElement();
        _endElement = new EndElement();
        _linemap = new Element[1];
        _linemap[0] = _endElement;
        _linemaplen = 0;
    }

    public int getLength() {
        return _content.length() - 1;
    }

    public void addDocumentListener(DocumentListener listener) {
        _listeners.add(listener);
    }

    public void removeDocumentListener(DocumentListener listener) {
        _listeners.remove(listener);
    }

    public void addUndoableEditListener(UndoableEditListener listener) {
        return;
    }

    public void removeUndoableEditListener(UndoableEditListener listener) {
        return;
    }

    public Object getProperty(Object key) {
        return null;
    }

    public void putProperty(Object key, Object value) {
        return;
    }

    public void remove(int offs, int len) throws BadLocationException {
        _writeLock.lock();
        try {
            final int cachedoffs = offs;
            final int cachedlen = len;
            LineElement compElem = new LineElement(offs, offs + len);
            int ind = Arrays.binarySearch(_linemap, compElem, ELEMCOMPARATOR);
            if (ind < 0) {
                ind = -1 * (ind + 1);
            }
            final Element[] elems = Arrays.copyOfRange(_linemap, ind, _linemaplen);
            _linemap[ind] = _endElement;
            _linemaplen = ind;
            notifyListeners(new DocumentEvent() {

                public int getOffset() {
                    return cachedoffs;
                }

                public int getLength() {
                    return cachedlen;
                }

                public Document getDocument() {
                    return AppendOnlyDocument.this;
                }

                public EventType getType() {
                    return EventType.REMOVE;
                }

                public ElementChange getChange(Element elem) {
                    if (elem == _root) {
                        return new ElementChange() {

                            public Element getElement() {
                                return _root;
                            }

                            public int getIndex() {
                                return 0;
                            }

                            public Element[] getChildrenRemoved() {
                                return elems;
                            }

                            public Element[] getChildrenAdded() {
                                return EMPTYELEMARRAY;
                            }
                        };
                    }
                    return null;
                }
            });
            _content.remove(offs, len);
        } finally {
            _writeLock.unlock();
        }
    }

    public void insertString(int offset, String str, AttributeSet a) throws BadLocationException {
        _writeLock.lock();
        try {
            if (offset != getLength()) {
                throw new BadLocationException("Inserts only allowed at end of file (curlength=" + getLength() + ")", offset);
            }
            _content.insertString(offset, str);
            final int localOffset = offset;
            final int localLength = str.length();
            final int localChangeIndex = _linemaplen;
            // model insert as one element per line
            int numlines = 0;
            int curStrIndex = 0;
            int nlInd = 0;
            while (curStrIndex < localLength) {
                if (_linemaplen + 1 >= _linemap.length) {
                    _linemap = Arrays.copyOf(_linemap, (int) (_linemap.length + (_linemap.length * 0.5) + 1));
                    Arrays.fill(_linemap, _linemaplen, _linemap.length, _endElement);
                }
                nlInd = str.indexOf('\n', curStrIndex);
                _linemap[_linemaplen] = new LineElement(offset + curStrIndex, (nlInd == -1) ? (offset + str.length()) : (offset + nlInd + 1));
                _linemaplen++;
                numlines++;
                if (nlInd == -1) {
                    // no more newlines
                    break;
                }
                curStrIndex = nlInd + 1;
            }
            final int localNumChanges = numlines;
            notifyListeners(new DocumentEvent() {

                public int getOffset() {
                    return localOffset;
                }

                public int getLength() {
                    return localLength;
                }

                public Document getDocument() {
                    return AppendOnlyDocument.this;
                }

                public EventType getType() {
                    return EventType.INSERT;
                }

                public ElementChange getChange(Element elem) {
                    if (elem == _root) {
                        return new ElementChange() {

                            public Element getElement() {
                                return _root;
                            }

                            public int getIndex() {
                                return localChangeIndex;
                            }

                            public Element[] getChildrenRemoved() {
                                return EMPTYELEMARRAY;
                            }

                            public Element[] getChildrenAdded() {
                                return Arrays.copyOfRange(_linemap, localChangeIndex, localChangeIndex + localNumChanges);
                            }
                        };
                    }
                    return null;
                }
            });
        } finally {
            _writeLock.unlock();
        }
    }

    public String getText(int offset, int length) throws BadLocationException {
        _readLock.lock();
        try {
            return _content.getString(offset, length);
        } finally {
            _readLock.unlock();
        }
    }

    public void getText(int offset, int length, Segment txt) throws BadLocationException {
        _readLock.lock();
        try {
            _content.getChars(offset, length, txt);
        } finally {
            _readLock.unlock();
        }
    }

    public Position getStartPosition() {
        try {
            return _content.createPosition(0);
        } catch (BadLocationException ex) {
            Logger.getLogger(AppendOnlyDocument.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }

    public Position getEndPosition() {
        _readLock.lock();
        try {
            return _content.createPosition(_content.length());
        } catch (BadLocationException ex) {
            Logger.getLogger(AppendOnlyDocument.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        } finally {
            _readLock.unlock();
        }
    }

    private int findEnclosingLineIndex(int offset) {
        _readLock.lock();
        try {
            if (offset >= getLength()) {
                return EOFINDEX; // marker for end of file
            }
            Element compElem = new LineElement(offset, 0);
            int ind = Arrays.binarySearch(_linemap, compElem, ELEMCOMPARATOR);
            if (ind < 0) {
                ind = (-1 * (ind + 1)) - 1;
            } else if (ind >= _linemaplen) {
                ind = _linemaplen - 1;
            }
            return ind;
        } finally {
            _readLock.unlock();
        }
    }

    public Position createPosition(int offs) throws BadLocationException {
        return _content.createPosition(offs);
    }

    public Element[] getRootElements() {
        return new Element[]{_root};
    }

    public Element getDefaultRootElement() {
        return _root;
    }

    public void render(Runnable r) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    private void notifyListeners(DocumentEvent e) {
        for (DocumentListener listener : _listeners) {
            DocumentEvent.EventType type = e.getType();
            if (type == DocumentEvent.EventType.CHANGE) {
                listener.changedUpdate(e);
            } else if (type == DocumentEvent.EventType.INSERT) {
                listener.insertUpdate(e);
            } else if (type == DocumentEvent.EventType.REMOVE) {
                listener.removeUpdate(e);
            }
        }
    }

    private class ElementComparator implements Comparator<Element> {

        public int compare(Element o1, Element o2) {
            if (o1 == _endElement) {
                return -1;
            }
            if (o2 == _endElement) {
                return 1;
            }
            int off1 = o1.getStartOffset();
            int off2 = o2.getStartOffset();
            if (off1 < off2) {
                return -1;
            } else if (off1 > off2) {
                return 1;
            }
            int end1 = o1.getEndOffset();
            int end2 = o2.getEndOffset();
            if (end1 < end2) {
                return -1;
            } else if (end1 > end2) {
                return 1;
            }
            return 0;
        }
    }

    private class RootElement implements Element {

        public RootElement() {
        }

        public Document getDocument() {
            return AppendOnlyDocument.this;
        }

        public Element getParentElement() {
            return null;
        }

        public String getName() {
            return "root";
        }

        public AttributeSet getAttributes() {
            return NULLATTRS;
        }

        public int getStartOffset() {
            return 0;
        }

        public int getEndOffset() {
            return _content.length();
        }

        public int getElementIndex(int offset) {
            return findEnclosingLineIndex(offset);
        }

        public int getElementCount() {
            return _linemaplen + 1;
        }

        public Element getElement(int index) {
            if (index == EOFINDEX) {
                return _linemap[_linemaplen];
            }
            return _linemap[index];
        }

        public boolean isLeaf() {
            return false;
        }
    }

    private class LineElement implements Element {

        int _start;
        int _end;

        public LineElement(int start, int end) {
            _start = start;
            _end = end;
        }

        public Document getDocument() {
            return AppendOnlyDocument.this;
        }

        public Element getParentElement() {
            return _root;
        }

        public String getName() {
            return "line";
        }

        public AttributeSet getAttributes() {
            return NULLATTRS;
        }

        public int getStartOffset() {
            return _start;
        }

        public int getEndOffset() {
            return _end;
        }

        public int getElementIndex(int offset) {
            return -1;
        }

        public int getElementCount() {
            return 0;
        }

        public Element getElement(int index) {
            return null;
        }

        public boolean isLeaf() {
            return true;
        }
    }

    /**
     * This {@link Element} will always track the end of file.
     * We really only need one instance of this per document as
     * this class stores no state of its own (except the implicit
     * reference to the outer document class).
     */
    private class EndElement implements Element {

        public Document getDocument() {
            return AppendOnlyDocument.this;
        }

        public Element getParentElement() {
            return _root;
        }

        public String getName() {
            return "end-of-file";
        }

        public AttributeSet getAttributes() {
            return NULLATTRS;
        }

        public int getStartOffset() {
            return getLength();
        }

        public int getEndOffset() {
            return getLength();
        }

        public int getElementIndex(int offset) {
            return -1;
        }

        public int getElementCount() {
            return 0;
        }

        public Element getElement(int index) {
            return null;
        }

        public boolean isLeaf() {
            return true;
        }
    }
}
