/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.utilities.download;

import org.nbirn.fbirn.utilities.CanceledException;
import org.nbirn.fbirn.utilities.CredentialManager;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.SwingWorker;

/**
 *
 * @author gadde
 */
// This is a WorkerDownloadFromURL that has a progress bar and label
// associated with it.  Progress is set by the super class using the
// normal setProgress(), but whenever that happens, we also send a
// progress update to the parent (adjusted by the parentProgressAmount
// allotted to us).
class WorkerURLUpdater extends WorkerDownloadFromURL implements PropertyChangeListener {

    private static final Logger _logger = Logger.getLogger(WorkerURLUpdater.class.getName());
    private DownloaderCatalogEntry _entry = null;
    private File _dest = null;
    private CredentialManager _credman = null;
    private boolean _updateOnly = false;
    private boolean _dryRun = false;
    private TimeZone _remoteTimeZone = null;
    private int _timeStampCheckDepth = -1;
    private WorkerDownloadFromCatalog _parentWorker;
    private double _parentProgressAmount = -1;
    private double _lastProgress = -1;
    private boolean _selfCancel = false;

    public WorkerURLUpdater(DownloaderCatalogEntry entry, File dest, CredentialManager credman, boolean updateOnly, boolean dryRun, TimeZone remoteTimeZone, int timeStampCheckDepth, WorkerDownloadFromCatalog parentWorker, double parentProgressAmount) {
        super(entry.getURI(), dest, credman, updateOnly, dryRun, remoteTimeZone, timeStampCheckDepth);
        _logger.log(Level.FINE, "WorkerURLUpdater: URI {0} allotted {1}/100.0 of parent progress", new Object[]{entry.getURI().toString(), parentProgressAmount});
        _entry = entry;
        _dest = dest;
        _credman = credman;
        _updateOnly = updateOnly;
        _dryRun = dryRun;
        _remoteTimeZone = remoteTimeZone;
        _timeStampCheckDepth = timeStampCheckDepth;
        _parentWorker = parentWorker;
        _parentProgressAmount = parentProgressAmount;
        _lastProgress = -1;
        if (_entry.getCancelButton() != null) {
            _entry.getCancelButton().setEnabled(false);
        }
        if (_entry.getRefreshButton() != null) {
            _entry.getRefreshButton().setEnabled(false);
        }
    }

    // create a waiting clone of ourselves to run if the refresh button is pressed
    private void prepareForRestart() {
        if (_entry.getRefreshButton() != null) {
            final WorkerURLUpdater worker = new WorkerURLUpdater(_entry, _dest, _credman, _updateOnly, _dryRun, _remoteTimeZone, _timeStampCheckDepth, _parentWorker, _parentProgressAmount);
            synchronized (_parentWorker._idletasks) {
                _parentWorker._idletasks.add(worker);
            }
            worker._lastProgress = _lastProgress; // so parent progress bar might go backwards on restart

            _entry.getRefreshButton().addActionListener(new java.awt.event.ActionListener() {

                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    _entry.getRefreshButton().setEnabled(false);
                    // remove current actions for this button
                    ActionListener[] listeners = _entry.getCancelButton().getActionListeners();
                    for (int i = 0; i < listeners.length; i++) {
                        _entry.getRefreshButton().removeActionListener(listeners[i]);
                    }
                    synchronized (_parentWorker._idletasks) {
                        synchronized (_parentWorker._tasks) {
                            _parentWorker._idletasks.remove(worker);
                            _parentWorker._tasks.add(worker);
                        }
                    }
                    Executors.newCachedThreadPool().execute(worker);
                }
            });
            _entry.getRefreshButton().setEnabled(true);
        }
    }

    void actionCancel(java.awt.event.ActionEvent evt) {
        // cancel button was pressed, disable the cancel button
        _entry.getCancelButton().setEnabled(false);
        // remove current actions for this button
        ActionListener[] listeners = _entry.getCancelButton().getActionListeners();
        for (int i = 0; i < listeners.length; i++) {
            _entry.getCancelButton().removeActionListener(listeners[i]);
        }
        _selfCancel = true;
        cancel(true);
    }

    @Override
    protected void cleanup(boolean error) {
        super.cleanup(error);
        if (_selfCancel) {
            // we'll leave it to the new restart worker to take care of our GUI components
            return;
        }
        if (_entry.getProgressBar() != null) {
            _entry.getProgressBar().setIndeterminate(false);
        }
        if (_entry.getCancelButton() != null) {
            _entry.getCancelButton().setEnabled(false);
        }
        if (_entry.getRefreshButton() != null) {
            _entry.getRefreshButton().setEnabled(false);
        }
    }

    @Override
    protected void process(List<ProgressUpdate> chunks) {
        if (isCancelled()) {
            // don't bother processing these if we are canceled
            return;
        }
        super.process(chunks);
        for (int i = 0; i < chunks.size(); i++) {
            ProgressUpdate update = chunks.get(i);
            if (_entry.getMessageLabel() != null && (update.getMessage() != null || update.getSubMessage() == null)) {
                // we clear message label if message and submessage is null;
                // otherwise, if message is null and submessage is not null,
                // that means we keep current message
                String msg = update.getMessage();
                _entry.getMessageLabel().setText(msg == null ? " " : msg);
            }
            if (_entry.getSubMessageLabel() != null) {
                String msg = update.getSubMessage();
                _entry.getSubMessageLabel().setText(msg == null ? " " : msg);
            }
            if (update.getMessage() != null) {
                _parentWorker.sendUpdate(new ProgressUpdate(((_entry.getName() == null) ? "" : _entry.getName()) + " | " + update.getMessage(), 0));
            }
        }
    }

    @Override
    public Void doInBackground() throws CanceledException, DownloadException {
        if (_entry.getCancelButton() != null) {
            _entry.getCancelButton().addActionListener(new java.awt.event.ActionListener() {

                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    actionCancel(evt);
                }
            });
        }
        if (_entry.getCancelButton() != null) {
            _entry.getCancelButton().setEnabled(true);
        }
        if (_entry.getRefreshButton() != null) {
            _entry.getRefreshButton().setEnabled(false);
        }
        if (_entry.getProgressBar() != null) {
            addPropertyChangeListener(new ProgressBarListener(_entry.getProgressBar()));
            _entry.getProgressBar().setIndeterminate(true);
        }
        addPropertyChangeListener(this);
        publish(new ProgressUpdate("Waiting...", 0));
        return super.doInBackground();
    }

    @Override
    public void done() {
        try {
            get();
            cleanup(false);
        } catch (InterruptedException e) {
        } catch (ExecutionException e) {
            if (e.getCause() != null && e.getCause().getClass() == CanceledException.class) {
                // our task has ended and will likely not be around when the cancellation comes back down from the root, so cleanup now
                cleanup(false);
            } else {
                // other error -- maybe can be restarted
                prepareForRestart();
            }
        } catch (CancellationException e) {
            if (isCancelled()) {
                if (_selfCancel) {
                    // user clicked our cancel button; give user option of restarting
                    prepareForRestart();
                } else {
                    // we were canceled (from above); destroy everything
                    cleanup(true);
                }
            }
        }
    }

    public void propertyChange(PropertyChangeEvent evt) {
        String propName = evt.getPropertyName();
        if ("state".equals(propName) && evt.getNewValue() == SwingWorker.StateValue.DONE) {
            Throwable result = null;
            try {
                get();
            } catch (InterruptedException e) {
                result = new CanceledException("Operation canceled.");
            } catch (ExecutionException e) {
                result = e.getCause();
            } catch (CancellationException e) {
                result = new CanceledException("Operation canceled.");
            }
            if (_entry.getMessageLabel() != null) {
                if (result == null) {
                    _entry.getMessageLabel().setText("Completed.");
                } else {
                    _entry.getMessageLabel().setText("ERR: " + result.getMessage());
                }
            }
            if (_entry.getSubMessageLabel() != null) {
                _entry.getSubMessageLabel().setText(" ");
            }
        } else if (propName.equals("progress")) {
            int val = (Integer) evt.getNewValue();
            _logger.log(Level.FINE, "WorkerURLUpdater: URI {0} reporting {1}/100.0 progress", new Object[]{_entry.getURI().toString(), val});
            if (_lastProgress == -1 || val - _lastProgress != 0) {
                double parentincval = (((_lastProgress == -1) ? val : (val - _lastProgress)) * (_parentProgressAmount / 100.0));
                if (_lastProgress == -1) {
                    _logger.log(Level.FINE, "WorkerURLUpdater:   reporting incremental progress of {0} * ( {1} / 100.0) = {2} to parent", new Object[]{val, _parentProgressAmount, parentincval});
                } else {
                    _logger.log(Level.FINE, "WorkerURLUpdater:   reporting incremental progress of ({0} - {1}) * ( {2} / 100.0) = {3} to parent", new Object[]{val, _lastProgress, _parentProgressAmount, parentincval});
                }
                _parentWorker.sendUpdate(new ProgressUpdate(null, parentincval));
            }
            _lastProgress = val;
        }
    }
}
