from UploadFunctions import *

import logging
import math
import os
import platform
import re
import struct
import xml
import xml.xpath

import quopri

__all__ = ["run"]

def _get_exports_list(module):
  try:
    return list(module.__all__)
  except AttributeError:
    return [n for n in dir(module) if n[0] != '_']

allvrs = [
  'AE', 'AS', 'AT', 'CS', 'DA', 'DS', 'DT', 'FL', 'FD', 'IS', 'LO',
  'LT', 'OB', 'OF', 'OW', 'PN', 'SH', 'SL', 'SQ', 'SS', 'ST', 'TM',
  'UI', 'UL', 'US', 'UT', 'UN'
  ]
           
vrs32 = [ 'OB', 'OW', 'OF', 'SQ', 'UN' ]

def try_explicit_little_endian(bytes):
  if len(bytes) < 12:
    return False
  curpos = 0
  group = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  curpos = curpos + 2
  elem = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  vr = 'UN'
  if group != 0xFFFE:
    vr = bytes[curpos:curpos+2]
    curpos = curpos + 2
  if vr not in allvrs:
    return False
  length = 0
  if group != 0xFFFE and vr in vrs32:
    zeros = bytes[curpos:curpos+2]
    if zeros != '\x00\x00':
      return False
    curpos = curpos + 2
  if group == 0xFFFE or vr in vrs32:
    length = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8) + (ord(bytes[curpos+2]) << 16) + (ord(bytes[curpos+3]) <<24)
    curpos = curpos + 4
  else:
    length = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
    curpos = curpos + 2
  if len(bytes) < curpos + 8:
    return False
  group2 = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  curpos = curpos + 2
  elem2 = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  vr2 = 'UN'
  if group2 != 0xFFFE:
    vr2 = bytes[curpos:curpos+2]
    curpos = curpos + 2
  if vr2 not in allvrs:
    return False
  if group2 > group or (group2 == group and elem2 > elem):
    return True

def try_explicit_big_endian(bytes):
  if len(bytes) < 12:
    return False
  curpos = 0
  group = ord(bytes[curpos+1]) + (ord(bytes[curpos]) << 8)
  curpos = curpos + 2
  elem = ord(bytes[curpos+1]) + (ord(bytes[curpos]) << 8)
  vr = 'UN'
  if group != 0xFFFE:
    vr = bytes[curpos:curpos+2]
    curpos = curpos + 2
  if vr not in allvrs:
    return False
  length = 0
  if group != 0xFFFE and vr in vrs32:
    zeros = bytes[curpos:curpos+2]
    if zeros != '\x00\x00':
      return False
    curpos = curpos + 2
  if group == 0xFFFE or vr in vrs32:
    length = ord(bytes[curpos+3]) + (ord(bytes[curpos+2]) << 8) + (ord(bytes[curpos+1]) << 16) + (ord(bytes[curpos]) <<24)
    curpos = curpos + 4
  else:
    length = ord(bytes[curpos+1]) + (ord(bytes[curpos]) << 8)
    curpos = curpos + 2
  if len(bytes) < curpos + 8:
    return False
  group2 = ord(bytes[curpos+1]) + (ord(bytes[curpos]) << 8)
  curpos = curpos + 2
  elem2 = ord(bytes[curpos+1]) + (ord(bytes[curpos]) << 8)
  vr2 = 'UN'
  if group2 != 0xFFFE:
    vr2 = bytes[curpos:curpos+2]
    curpos = curpos + 2
  if vr2 not in allvrs:
    return False
  if group2 > group or (group2 == group and elem2 > elem):
    return True
  return False

def try_implicit_little_endian(bytes):
  if len(bytes) < 8:
    return False
  group = ord(bytes[0]) + (ord(bytes[1]) << 8)
  elem = ord(bytes[2]) + (ord(bytes[3]) << 8)
  length = ord(bytes[4]) + (ord(bytes[5]) << 8) + (ord(bytes[6]) << 16) + (ord(bytes[7]) << 24)
  curpos = 8 + length
  if len(bytes) < curpos + 8:
    return False
  group2 = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  curpos = curpos + 2
  elem2 = ord(bytes[curpos]) + (ord(bytes[curpos+1]) << 8)
  if group2 > group or (group2 == group and elem2 > elem):
    return True
  return False

def try_implicit_big_endian(bytes):
  if len(bytes) < 8:
    return False
  group = (ord(bytes[0]) << 8) + ord(bytes[1])
  elem = (ord(bytes[2]) << 8) + ord(bytes[3])
  length = (ord(bytes[4]) << 24) + (ord(bytes[5]) << 16) + (ord(bytes[6]) << 8) + ord(bytes[7])
  curpos = 8 + length
  if len(bytes) < curpos + 8:
    return False
  group2 = (ord(bytes[curpos]) << 8) + ord(bytes[curpos+1])
  curpos = curpos + 2
  elem2 = (ord(bytes[curpos]) << 8) + ord(bytes[curpos+1])
  if group2 > group or (group2 == group and elem2 > elem):
    return True
  return False

def readdicomelem(f, unpack16='<H', unpack32='<I', explicit=True):
  #logging.debug("readdicomelem(unpack16='" + unpack16 + "', unpack32='" + unpack32 + ", explicit=" + str(explicit) + ")")
  bytes = f.read(2)
  if len(bytes) == 0:
    # probably EOF
    return (None, None, None, None)
  if len(bytes) < 2:
    raise StandardError("Hit EOF before end of DICOM element!")
  taggroup = struct.unpack(unpack16, bytes)[0]
  #logging.debug("bytes=(hex)" + ':'.join(map(lambda x: "%02x" % ord(x), bytes)))
  #logging.debug("taggroup=0x%x" % taggroup)
  bytes = f.read(2)
  if len(bytes) < 2:
    raise StandardError("Hit EOF before end of DICOM element!")
  tagelem = struct.unpack(unpack16, bytes)[0]
  #logging.debug("tagelem=0x%x" % tagelem)
  if taggroup == 0x0002:
    # still in meta info header -- always little-endian explicit
    unpack16 = '<H'
    unpack32 = '<I'
    explicit = True
  vr = 'UN'
  if taggroup == 0xFFFE:
    #logging.debug("vr=[none for special sequence item tags]")
    pass
  elif explicit:
    vr = f.read(2)
    if len(vr) < 2:
      raise StandardError("Hit EOF before end of DICOM element!")
    #logging.debug("vr=%s" % vr)
  fieldlen = 0
  if not explicit or taggroup == 0xFFFE or vr in vrs32:
    if explicit and taggroup != 0xFFFE:
      bytes = f.read(2) # reserved for future use, discard
    bytes = f.read(4)
    if len(bytes) < 4:
      raise StandardError("Hit EOF before end of DICOM element!")
    #logging.debug("fieldlenbytes=" + quopri.encodestring(bytes))
    #logging.debug("fieldlenLE32=" + str(struct.unpack('<I', bytes)))
    #logging.debug("fieldlenBE32=" + str(struct.unpack('>I', bytes)))
    fieldlen = struct.unpack(unpack32, bytes)[0]
    if bytes == '\xFF\xFF\xFF\xFF':
      fieldlen = 0
  else:
    bytes = f.read(2)
    if len(bytes) < 2:
      raise StandardError("Hit EOF before end of DICOM element!")
    #logging.debug("fieldlenbytes=" + quopri.encodestring(bytes))
    #logging.debug("fieldlenLE16=" + str(struct.unpack('<H', bytes)))
    #logging.debug("fieldlenBE16=" + str(struct.unpack('>H', bytes)))
    fieldlen = struct.unpack(unpack16, bytes)[0]
  #logging.debug("fieldlen=" + str(fieldlen))
  fieldval = f.read(fieldlen)
  if len(fieldval) < fieldlen:
    raise StandardError("Hit EOF before end of DICOM element!")
  if vr == 'UI' and fieldval[-1] == '\0':
    fieldval = fieldval[:-1]
  valstr = fieldval
  if len(valstr) > 64:
    valstr = valstr[0:64] + '...'
  #logging.debug("fieldval[0:64]=" + fieldval[0:64])
  #logging.debug("fieldvalencoded=" + quopri.encodestring(valstr))
  #logging.debug("f.tell=%d" % f.tell())
  return (taggroup, tagelem, vr, fieldval)

# only a few VRs are implemented, as required
def interpretVR(fieldbytes, vr, unpack16, unpack32):
  if vr == 'SS':
    if unpack16 == '<H':
      return struct.unpack('<h', fieldbytes)[0]
    elif unpack16 == '>H':
      return struct.unpack('>h', fieldbytes)[0]
  return fieldbytes

def read_meta_header(f):
  littleendian = True
  explicit = True
  origpos = f.tell()
  while True:
    datasetpos = f.tell()
    (taggroup, tagelem, vr, fieldval) = readdicomelem(f)
    if taggroup == None:
      f.seek(origpos)
      return (None, None, None, None)
    #logging.debug("  tag: group=" + str(taggroup)  + " elem=" + str(tagelem))
    if taggroup == 0x0002 and tagelem == 0x0010:
      # assume VR of UI
      if fieldval == '1.2.840.10008.1.2.1':
        littleendian = True
        explicit = True
      elif fieldval == '1.2.840.10008.1.2.2':
        littleendian = False
        explicit = True
      elif fieldval == '1.2.840.10008.1.2':
        # ugh
        littleendian = True
        explicit = False
      else:
        logging.warn("DICOM file does not have a supported transfer syntax (found " + fieldval + ").")
        f.seek(origpos)
        return (None, None, None, None)
    if taggroup > 0x0002:
      break
  f.seek(datasetpos)
  #logging.debug("datasetpos=%d" % datasetpos)
  if littleendian:
    return ('<H', '<I', explicit, datasetpos)
  else:
    return ('>H', '>I', explicit, datasetpos)

def read_dicom_init(f):
  f.seek(0)
  bytes = f.read(128)
  magic = f.read(4)
  if magic == "DICM":
    return read_meta_header(f)
  # no preamble -- try some other tricks to see if it might actually still
  # be a DICOM file; we require two DICOM tags within the first 128 bytes.
  if try_explicit_little_endian(bytes):
    logging.info("Warning -- no DICOM preamble; but detected possible explicit little-endian transfer syntax")
    f.seek(0)
    return ('<H', '<I', 1, 0)
  if try_explicit_big_endian(bytes):
    logging.info("Warning -- no DICOM preamble; but detected possible explicit big-endian transfer syntax")
    f.seek(0)
    return ('>H', '>I', 1, 0)
  if try_implicit_little_endian(bytes):
    logging.info("Warning -- no DICOM preamble; but detected possible implicit little-endian transfer syntax")
    f.seek(0)
    return ('<H', '<I', 0, 0)
  if try_implicit_big_endian(bytes):
    logging.info("Warning -- no DICOM preamble; but detected possible implicit big-endian transfer syntax")
    f.seek(0)
    return ('>H', '>I', 0, 0)
  return (None, None, None, None)
  
def find_dicom_tag(f, groupnum, elemnum, unpack16, unpack32, explicit):
  while True:
    (taggroup, tagelem, vr, fieldval) = readdicomelem(f, unpack16, unpack32, explicit)
    if taggroup == None:
      return (None, None, None, None)
    #logging.debug("  tag: group=" + str(taggroup)  + " elem=" + str(tagelem))
    if taggroup == groupnum and tagelem == elemnum:
      return (taggroup, tagelem, vr, fieldval)

def replace_series_children(series, doc, nameLocal=None, nameStandard=None, ID=None, seriesTime=None, description=None, type=None, paradigm=None, paradigmVersion=None, number=None, source=None, analysislevelname=None, analysislevellevel=None, skip=False):
  replacelist = []
  if nameLocal != None: replacelist.append((('nameLocal',), nameLocal))
  if nameStandard != None: replacelist.append((('nameStandard',), nameStandard))
  if ID != None: replacelist.append((('ID',), ID))
  if seriesTime != None: replacelist.append((('seriesTime',), seriesTime))
  if description != None: replacelist.append((('description',), description))
  if type != None: replacelist.append((('type',), type))
  if paradigm != None: replacelist.append((('paradigm',), paradigm))
  if paradigmVersion != None: replacelist.append((('paradigmVersion',), paradigmVersion))
  if number != None: replacelist.append((('number',), number))
  if source != None: replacelist.append((('source',), source))
  if analysislevelname != None: replacelist.append((('analysislevel', 'name'), analysislevelname))
  if analysislevellevel != None: replacelist.append((('analysislevel', 'level'), analysislevellevel))
  if skip == True: replacelist.append((('skip',), None))
  for (names, value) in replacelist:
    node = series
    for tag in names:
      nodelist = xml.xpath.Evaluate(tag, node)
      if len(nodelist) == 0:
        nodelist = [ node.appendChild(doc.createElement(tag)) ]
      node = nodelist[0]
    # remove all children of this node
    child = node.firstChild
    while child:
      nextchild = child.nextSibling
      child.parentNode.removeChild(child)
      child = nextchild
    # put new text in the node
    if value != None:
      node.appendChild(doc.createTextNode(value))

def extract_asl_cal(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct):
  # For Siemens sites, separate out the calibration volume from the
  # ASL fair-estimate scan
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    logging.info("Extracting ASL calibration volume won't work for Local-In-Place uploads, skipping...")
    return
  aslfairestlist = xml.xpath.Evaluate("/FIPS/series[nameStandard='asl_fairest']", XMLDoc)
  if len(aslfairestlist) == 0:
    logging.info("Can't find asl_fairest series.  Skipping asl_calibration extraction...")
    return
  aslcallist = xml.xpath.Evaluate("/FIPS/series[nameStandard='asl_calibration']", XMLDoc)
  if len(aslcallist) > 0:
    logging.info("Already found asl_calibration series in XML file, won't extract another one.")
    return
  logging.info("Extracting ASL calibration volume from ASL fair estimate scan")
  aslfairest = aslfairestlist[0]
  # copy asl_fairest node for asl_calibration
  aslcal = aslfairest.cloneNode(True)
  # replace some nodes
  replace_series_children(aslcal, XMLDoc, nameStandard='asl_calibration', paradigm='asl-calibration', type='functional', ID='')
  # image directory paths
  newfairpath = StagingPaths.get_TempPath('NEW_asl_fairest')
  newcalpath = StagingPaths.get_TempPath('NEW_asl_calibration')
  oldfairlocal = XPathEval('nameLocal', aslfairest)[0]
  oldfairpath = DataPaths.get_LocalPath(oldfairlocal)
  skip = False
  if len(outputTimestampStruct) > 0:
    # check timestamps
    inputMTime = MaxMTime([(oldfairpath, True, True)])
    outofdate = False
    for seriesname in ['asl_calibration', 'asl_fairest']:
      if seriesname in outputTimestampStruct.keys():
        if inputMTime > outputTimestampStruct[seriesname]['timeStamp']:
          outofdate = True
    if not outofdate:
      skip = True
    if skip:
      logging.info("Previously uploaded asl_fairest data already up-to-date.  Will not re-upload 'asl_fairest'.")
  if not skip:
    # create new image data directories for fairest and calibration image
    os.makedirs(newfairpath)
    os.makedirs(newcalpath)
    # copy files to new image data directories (though split DICOM files
    # between them, based on InstanceNumber)
    sortedfilelist = os.listdir(oldfairpath)
    sortedfilelist.sort()
    for file in sortedfilelist:
      fullpath = os.path.join(oldfairpath, file)
      if os.path.isdir(fullpath):
        shutil.copytree(fullpath, os.path.join(newfairpath, file))
        shutil.copytree(fullpath, os.path.join(newcalpath, file))
        continue
      try:
        f = open(fullpath,"r")
      except IOError:
        continue
      (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
      if unpack16 != None:
        logging.debug(" Looking for InstanceNumber in " + fullpath)
        skipfile = False
        # get InstanceNumber
        (taggroup, tagelem, vr, instancenumber) = find_dicom_tag(f, 0x0020, 0x0013, unpack16, unpack32, explicit)
        if instancenumber == None:
          logging.warn("  Don't know what to do with DICOM file " + fullpath + " -- can't find InstanceNumber!  Skipping...")
          f.close()
          continue
        # note that we assume VR is IS
        if vr != 'UN' and vr != 'IS':
          logging.warn("  InstanceNumber VR is explicitly specified and is not IS!  Skipping...")
          f.close()
          continue
        instancenumber = int(instancenumber)
        # first instance goes to cal volume, everything else to fairest
        if instancenumber == 1:
          logging.debug(" Copying " + fullpath + " to NEW_asl_calibration")
          shutil.copy2(fullpath, newcalpath)
        else:
          logging.debug(" Copying " + fullpath + " to NEW_asl_fairest")
          shutil.copy2(fullpath, newfairpath)
      else:
        # not DICOM, just copy
        shutil.copy2(fullpath, newfairpath)
        # maybe we shouldn't duplicate files in calibration dir
        #shutil.copy2(fullpath, newcalpath)
      f.close()
  # add asl_calibration node to XML document (add as last entry, because
  # series ID will be blank, and we don't want any auto-generated series ID
  # to conflict with earlier explicitly-specified series IDs)
  aslfairest.parentNode.appendChild(aslcal)
  # add new paths to XML document
  # note, the following paths paths are absolute, so they will override
  # the normal source data path
  replace_series_children(aslcal, XMLDoc, nameLocal=newcalpath, skip=skip)
  replace_series_children(aslfairest, XMLDoc, nameLocal=newfairpath, skip=skip)
  if not skip:
    logging.info("Finished extracting ASL calibration volume.")

def extract_B0(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct, ProtocolErrList):
  # For GE sites, separate out the magnitude, real, and imaginary data
  # from the B0 scan
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    logging.info("Extracting B0 data won't work for Local-In-Place uploads, skipping...")
    return
  B0list = xml.xpath.Evaluate("/FIPS/series[nameStandard='B0']", XMLDoc)
  if len(B0list) == 0:
    logging.info("Can't find series with standard name 'B0'.  Skipping B0 extraction...")
    return
  B0maglist = xml.xpath.Evaluate("/FIPS/series[nameStandard='B0_mag1']", XMLDoc)
  B0reallist = xml.xpath.Evaluate("/FIPS/series[nameStandard='B0_real1']", XMLDoc)
  B0imaglist = xml.xpath.Evaluate("/FIPS/series[nameStandard='B0_imag1']", XMLDoc)
  if len(B0maglist) + len(B0reallist) + len(B0imaglist) > 0:
    errMsg = "Found B0_mag1 or B0_real1 or B0_imag1 series in XML file!  GE sites don't do that!"
    ProtocolErrList[0] += 1
    ProtocolErrList[1].append((errMsg, None))
    return
  B0 = B0list[0]
  # new paths
  B0magpath = StagingPaths.get_TempPath('NEW_B0_mag1')
  B0realpath = StagingPaths.get_TempPath('NEW_B0_real1')
  B0imagpath = StagingPaths.get_TempPath('NEW_B0_imag1')
  B0local = XPathEval('nameLocal', B0)[0]
  B0path = DataPaths.get_LocalPath(B0local)
  skip = False
  if len(outputTimestampStruct) > 0:
    # check timestamps
    inputMTime = MaxMTime([(B0path, True, True)])
    outofdate = False
    for seriesname in [ 'B0_mag1', 'B0_real1', 'B0_imag1' ]:
      if seriesname in outputTimestampStruct.keys():
        if inputMTime > outputTimestampStruct[seriesname]['timeStamp']:
          outofdate = True
    if not outofdate:
      skip = True
    if skip:
      logging.info("Previously uploaded B0 data already up-to-date.  Will not re-upload 'B0'.")
  logging.info("Extracting magnitude, real, and imaginary data from B0 scan")
  # copy B0 node for sub-images
  B0mag = B0.cloneNode(True)
  B0real = B0.cloneNode(True)
  B0imag = B0.cloneNode(True)
  # replace some nodes
  replace_series_children(B0mag, XMLDoc, nameStandard='B0_mag1', paradigm='B0_mag', type='structural', ID='', skip=skip)
  replace_series_children(B0real, XMLDoc, nameStandard='B0_real1', paradigm='B0_real', type='structural', ID='', skip=skip)
  replace_series_children(B0imag, XMLDoc, nameStandard='B0_imag1', paradigm='B0_imag', type='structural', ID='', skip=skip)
  if not skip:
    # create new image data directories for sub-images
    os.makedirs(B0magpath)
    os.makedirs(B0realpath)
    os.makedirs(B0imagpath)
    # copy files to new image data directories (though split DICOM files
    # between them, based on value of 0043,002f GERawDataType)
    sortedfilelist = os.listdir(B0path)
    sortedfilelist.sort()
    for file in sortedfilelist:
      fullpath = os.path.join(B0path, file)
      if os.path.isdir(fullpath):
        shutil.copytree(fullpath, os.path.join(B0magpath,file))
        shutil.copytree(fullpath, os.path.join(B0realpath, file))
        shutil.copytree(fullpath, os.path.join(B0imagpath, file))
        continue
      try:
        f = open(fullpath,"r")
      except IOError:
        continue
      (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
      if unpack16 != None:
        logging.debug(" Looking for GERawDataType in " + fullpath)
        skipfile = False
        # get GERawDataType
        (taggroup, tagelem, vr, gerawdatatypebytes) = find_dicom_tag(f, 0x0043, 0x102f, unpack16, unpack32, explicit)
        if gerawdatatypebytes == None:
          logging.warn("  Don't know what to do with DICOM file " + fullpath + " -- can't find GERawDataType!  Skipping...")
          f.close()
          continue
        # note that we assume VR is SS
        if vr != 'UN' and vr != 'SS':
          logging.warn("  InstanceNumber VR is explicitly specified and is not SS!  Skipping...")
          f.close()
          continue
        gerawdatatype = interpretVR(gerawdatatypebytes, 'SS', unpack16, unpack32)
        # gerawdatatype == 0 goes to mag, 2 -> real, 3 -> imag
        if gerawdatatype == 0:
          logging.debug(" Copying " + fullpath + " to NEW_B0_mag1")
          shutil.copy2(fullpath, B0magpath)
        elif gerawdatatype == 2:
          logging.debug(" Copying " + fullpath + " to NEW_B0_real1")
          shutil.copy2(fullpath, B0realpath)
        elif gerawdatatype == 3:
          logging.debug(" Copying " + fullpath + " to NEW_B0_imag1")
          shutil.copy2(fullpath, B0imagpath)
      else:
        # not DICOM, just copy
        shutil.copy2(fullpath, B0magpath)
        shutil.copy2(fullpath, B0realpath)
        shutil.copy2(fullpath, B0imagpath)
      f.close()
  # add sub-image nodes to XML document (add as last entry, because
  # series ID will be blank, and we don't want any auto-generated series ID
  # to conflict with earlier explicitly-specified series IDs)
  B0.parentNode.appendChild(B0mag)
  B0.parentNode.appendChild(B0real)
  B0.parentNode.appendChild(B0imag)
  # remove B0 node
  B0.parentNode.removeChild(B0)
  # add new paths to XML document
  # note, the following paths paths are absolute, so they will override
  # the normal source data path
  replace_series_children(B0mag, XMLDoc, nameLocal=B0magpath)
  replace_series_children(B0real, XMLDoc, nameLocal=B0realpath)
  replace_series_children(B0imag, XMLDoc, nameLocal=B0imagpath)
  if not skip:
    logging.info("Finished extracting B0 data.")

def deface_t1(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct):
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    logging.info("Automatic T1 defacing won't work for Local-In-Place uploads, skipping...")
    return
  t1list = xml.xpath.Evaluate("/FIPS/series[nameStandard='t1']", XMLDoc)
  if len(t1list) == 0:
    logging.info("Can't find t1 series.  Skipping defacing...")
    return
  t1defacelist = xml.xpath.Evaluate("/FIPS/series[nameStandard='t1_deface']", XMLDoc)
  if len(t1defacelist) > 0:
    logging.info("Already found t1_deface series in XML file, won't deface again.")
    return
  logging.info("Defacing T1 scan")
  t1 = t1list[0]
  # new/old paths
  t1defacepath = StagingPaths.get_TempPath('NEW_t1_deface')
  t1local = XPathEval('nameLocal', t1)[0]
  t1path = DataPaths.get_LocalPath(t1local)
  skip = False
  if len(outputTimestampStruct) > 0:
    # check timestamps
    inputMTime = MaxMTime([(t1path, True, True)])
    outofdate = False
    for seriesname in [ 't1', 't1_deface' ]:
      if seriesname in outputTimestampStruct.keys():
        if inputMTime > outputTimestampStruct[seriesname]['timeStamp']:
          outofdate = True
    if not outofdate:
      skip = True
    if skip:
      logging.info("Previously uploaded defaced t1 data already up-to-date.  Will not re-upload defaced 't1'.")
  if not skip:
    # create new image data directory for t1_deface image
    os.makedirs(t1defacepath)
    # find an original DICOM file
    dicomfile = None
    sortedfilelist = os.listdir(t1path)
    sortedfilelist.sort()
    for file in sortedfilelist:
      fullpath = os.path.join(t1path, file)
      if os.path.isdir(fullpath):
        continue
      try:
        f = open(fullpath,"r")
      except IOError:
        continue
      (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
      if unpack16 == None:
        f.close()
        continue
      dicomfile = fullpath
      # get SeriesTime
      (taggroup, tagelem, vr, seriestime) = find_dicom_tag(f, 0x0008, 0x0031, unpack16, unpack32, explicit)
      if seriestime == None:
        logging.warn("  " + fullpath + " doesn't have SeriesTime field!")
        f.close()
        break
      # note that we assume VR is TM
      if vr != 'UN' and vr != 'TM':
        logging.warn("  " + fullpath + " has an explicit SeriesTime VR that is not TM!")
        f.close()
        break
      f.close()
      f2 = open(os.path.join(t1defacepath, 'tmpmetadata.xml'), 'w')
      f2.write('<?xml version="1.0"?><serieslevel xmlns="http://nbirn.net/Resources/Users/Applications/xcede/"><acqProtocol><acqParam name="scantime" type="varchar">%s:%s:%s</acqParam></acqProtocol><provenance /></serieslevel>' % (seriestime[0:2], seriestime[2:4], seriestime[4:6]))
      f2.close()
      break
    if dicomfile == None:
      logging.error("ERROR: Can't find a DICOM file in " + t1path + "!  Skipping...")
      return
    # deface the data
    defbindir = os.path.join(os.environ['ScriptInstallDir'], 'deface_binaries')
    os.putenv('FREESURFER_HOME', defbindir)
    if platform.system() != 'Linux':
      logging.error("ERROR: defacing binaries only available for Linux systems, skipping...")
      return
    archbindir =  os.path.join(os.environ['ScriptInstallDir'], 'deface_binaries', platform.architecture()[0])
    if not os.path.isdir(archbindir):
      logging.error("ERROR: directory " + archbindir + " doesn't exist!  Skipping...")
      return
    run_cmd(os.path.join(archbindir, "mri_deface") + " " + dicomfile + " " + os.path.join(defbindir, 'talairach_mixed.gca') + " " + os.path.join(defbindir, 'face.gca') + " " + os.path.join(t1defacepath, 'mask_volume.mgh'))
    run_cmd(os.path.join(archbindir, "mri_mask") + " " + dicomfile + " " + os.path.join(t1defacepath, 'mask_volume.mgh') + " " + os.path.join(t1defacepath, 't1-deface.nii'))
  # copy t1 for t1_deface
  t1deface = t1.cloneNode(True)
  # replace some nodes
  replace_series_children(t1deface, XMLDoc, nameStandard='t1_deface', paradigm='t1_deface', source='t1', type='structural-anon', skip=skip)
  # add t1_deface node to XML document
  if t1.nextSibling:
    t1.parentNode.insertBefore(t1deface, t1.nextSibling)
  else:
    t1.parentNode.appendChild(t1deface)
  # add new paths to XML document
  # note, the following paths paths are absolute, so they will override
  # the normal source data path
  replace_series_children(t1deface, XMLDoc, nameLocal=t1defacepath)
  if not skip:
    logging.info("Finished defacing T1 scan")

def add_cigal_files(XMLDoc, UploadConfig, StagingPaths, DataPaths):
  ecpath = None
  if UploadConfig.get_FileSystem() == "Local-In-Place":
    logging.info("Automatically adding an cigal_files directory won't work for Local-In-Place uploads, skipping...")
    return
  eclist = xml.xpath.Evaluate("/FIPS/series[nameStandard='cigal_files' or nameStandard='extra_cigal']", XMLDoc)
  if len(eclist) != 0:
    logging.info("Found cigal_files or extra_cigal series.  Don't need to create it...")
    eclocal = XPathEval('nameLocal', eclist[0])[0]
    ecpath = DataPaths.get_LocalPath(eclocal)
    replace_series_children(eclist[0], XMLDoc, nameStandard='cigal_files', type='extra', paradigm='cigal_data', paradigmVersion='1', number='1', analysislevelname='none', analysislevellevel='none')
  else:
    fipslist = xml.xpath.Evaluate("/FIPS", XMLDoc)
    if len(fipslist) == 0:
      logging.info("Can't find FIPS root element in XML document.  Skipping...")
      return
    logging.info("Creating an empty cigal_files directory.")
    # create new image data directory for cigal_files
    ecpath = StagingPaths.get_TempPath('NEW_cigal_files')
    os.makedirs(ecpath)
    # create XML node
    ec = XMLDoc.createElement('series')
    # replace some nodes
    replace_series_children(ec, XMLDoc, nameLocal=ecpath, nameStandard='cigal_files', ID='', seriesTime='00:00:00', description='This series contains full copies of CIGAL directories', type='extra', paradigm='cigal_data', paradigmVersion='1', number='1', analysislevelname='none', analysislevellevel='none')
    # add cigal_files node to XML document
    fipslist[0].appendChild(ec)
    logging.info("Finished creating cigal_files directory.")
  if len(eclist) != 0:
    logging.info("Adding subdirectories of cigal_files as behavioral roots...")
    subdirs = []
    for file in os.listdir(ecpath):
      fullpath = os.path.join(ecpath, file)
      if os.path.isdir(fullpath):
        subdirs.append(fullpath)
    if len(subdirs) == 0:
      logging.info("Found no subdirectories of cigal_files...")
    for subdir in subdirs:
        logging.info(" adding " + subdir + " as a behavioral root dir...")
        DataPaths.get_BehaviorRoots().append(subdir)

def check_cigal_roots(UploadConfig, DataPaths, ProtocolErrList):
  numErrs = 0
  errMsgs = []
  cigalmatch = "^s(\d{12})_(\d+)_(\d+)$"
  for behavroot in DataPaths.get_BehaviorRoots():
    logging.info("Checking whether behavioral root dir " + behavroot + " has the correct naming scheme.")
    while behavroot[-1] == os.sep:
      behavroot = behavroot[0:-1]
    (roothead, roottail) = os.path.split(behavroot)
    logging.debug(" - does '" + roottail + "' match the regexp <<< " + cigalmatch + " >>> ?")
    rootmatchobj = re.search(cigalmatch, roottail)
    if rootmatchobj:
      logging.debug("  - yes")
      if rootmatchobj.group(1) != UploadConfig.m_SubjectID.get_value():
        errMsg = "Warning: the specified behavioral root directory (" + behavroot + ") has a subject ID (" + rootmatchobj.group(1) + ") that doesn't match the subject ID specified in the upload XML file (" + UploadConfig.m_SubjectID.get_value() + ")."
        errMsgs.append(errMsg)
        numErrs += 1
    else:
      logging.debug("  - no")
      errMsg = "Warning: The specified behavioral root directory (" + behavroot + ") does not have a name of the form sXXXXXXXXXXXX_Y_Z where XXX... is the 12-digit BIRN ID and Y and Z are one or more digits."
      errMsgs.append(errMsg)
      numErrs += 1
  if len(DataPaths.get_BehaviorRoots()) and numErrs == 0:
    logging.info("Name of behavioral directories check out!")
  ProtocolErrList[0] += numErrs
  ProtocolErrList[1].extend(map(lambda x: (x, None), errMsgs))

def verify_B0(XMLDoc, DataPaths, ProtocolErrList):
  # For Siemens sites, make sure mag and phase images are labeled correctly
  for checkentry in [ ('B0_mag1', 'ORIGINAL\\PRIMARY\\M\\ND'), ('B0_phase1', 'ORIGINAL\\PRIMARY\\P\\ND') ]:
    (namestandard, expectedimagetype) = checkentry
    nodelist = xml.xpath.Evaluate("/FIPS/series[nameStandard='" + namestandard + "']", XMLDoc)
    if len(nodelist):
      logging.info("Checking " + namestandard + " series to make sure it has image type " + expectedimagetype)
      node = nodelist[0]
      localname = XPathEval('nameLocal', node)[0]
      localpath = DataPaths.get_LocalPath(localname)
      sortedfilelist = os.listdir(localpath)
      sortedfilelist.sort()
      for file in sortedfilelist:
        fullpath = os.path.join(localpath, file)
        try:
          f = open(fullpath,"r")
        except IOError:
          continue
        (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
        if unpack16 == None:
          logging.debug("  " + fullpath + " not DICOM file, skipping...")
          f.close()
          continue
        # get ImageType
        (taggroup, tagelem, vr, imagetype) = find_dicom_tag(f, 0x0008, 0x0008, unpack16, unpack32, explicit)
        if imagetype == None:
          logging.warn("  " + fullpath + " doesn't have ImageType field!  Skipping...")
          f.close()
          continue
        # note that we assume VR is CS
        if vr != 'UN' and vr != 'CS':
          logging.warn("  " + fullpath + " has an explicit ImageType VR that is not CS!  Skipping...")
          f.close()
          continue
        # in CS VR, leading and trailing space are insignificant
        imagetype = string.lstrip(imagetype, " ")
        imagetype = string.rstrip(imagetype, " ")
        if imagetype != expectedimagetype:
          errMsg = "ImageType (" + imagetype + ") of file " + fullpath + " is incorrect (expected to be " + expectedimagetype + ").\n"
          ProtocolErrList[0] += 1
          ProtocolErrList[1].append((errMsg, None))
        f.close()

def verify_phase_encode_dir(XMLDoc, DataPaths, ProtocolErrList):
  # For Siemens sites, make sure phase encode direction is P>>A
  for namestandard in [ 'ot1', 'ot2', 'ot3', 'ot4', 'ot5', 'ot6', 'ot7', 'restFA10', 'restFA77', 'audodd1', 'audodd2', 'B0_mag1', 'B0_phase1', 't2' ]:
    nodelist = xml.xpath.Evaluate("/FIPS/series[nameStandard='" + namestandard + "']", XMLDoc)
    if len(nodelist):
      logging.info("Checking " + namestandard + " series to make sure it was acquired with P>>A phase encode direction")
      node = nodelist[0]
      localname = XPathEval('nameLocal', node)[0]
      localpath = DataPaths.get_LocalPath(localname)
      sortedfilelist = os.listdir(localpath)
      sortedfilelist.sort()
      for file in sortedfilelist:
        foundascconv = 0
        fullpath = os.path.join(localpath, file)
        try:
          f = open(fullpath,"r")
        except IOError:
          continue
        (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
        if unpack16 == None:
          logging.debug("  " + fullpath + " not DICOM file, skipping...")
          f.close()
          continue
        # get ImageType
        (taggroup, tagelem, vr, value) = find_dicom_tag(f, 0x0029, 0x1020, unpack16, unpack32, explicit)
        if value == None:
          logging.warn("  " + fullpath + " doesn't have field 0x0029,0x1020!  Skipping...")
          f.close()
          continue
        # note that we assume VR is OB
        if vr != 'UN' and vr != 'OB':
          logging.warn("  " + fullpath + " has an explicit ImageType VR that is not OB!  Skipping...")
          f.close()
          continue
        # find ASCCONV BEGIN and ASCCONV END (with no intervening BEGIN)
        startind = None
        findret = 0
        while findret != -1:
          findret = value.find("### ASCCONV BEGIN ###\n", findret + 1);
          if findret != -1:
            nextret = value.find("### ASCCONV BEGIN ###\n", findret + 1);
            findend = value.find('### ASCCONV END ###');
            if nextret != -1 and nextret < findend:
              continue
            startind = findret
            break
        if startind == None:
          logging.warn("  Siemens DICOM file " + fullpath + " does not have an embedded ASCII header!  Skipping...")
          f.close()
          continue
        foundascconv = 1
        dipr = None
        lines = value[startind:].split("\n")
        for line in lines:
          if line.startswith('sSliceArray.asSlice[0].dInPlaneRot'):
            (dummy, dipr) = line.split("= ")
            break
          if line.startswith('### ASCCONV END ###'):
            break
        if dipr == None:
          dipr = 0
        else:
          dipr = float(dipr)
        # dInPlaneRot == 0 means A>>P, dInPlaneRot == PI means P>>A
        if dipr < (math.pi/2):
          errMsg = namestandard + " seems to have been acquired A>>P instead of expected P>>A phase encode direction (dInPlaneRot= " + str(dipr) + ").\n"
          ProtocolErrList[0] += 1
          ProtocolErrList[1].append((errMsg, None))
        f.close()
        # found one file with ASCCONV header, should be enough
        break
      if not foundascconv:
        errMsg = "Did not find an embedded ASCII header for any Siemens DICOM file in series " + namestandard + " so could not verify P>>A phase encode direction."
        ProtocolErrList[0] += 1
        ProtocolErrList[1].append((errMsg, None))

def check_fat_sat(XMLDoc, DataPaths, ProtocolErrList):
  # For GE sites, make sure fat sat is turned on
  for paradigm in [ 'working_memory', 'auditory_oddball', 'restFA10', 'restFA77' ]:
    nodelist = xml.xpath.Evaluate("/FIPS/series[paradigm='" + paradigm + "']", XMLDoc)
    for node in nodelist:
      namestandard = XPathEval('nameStandard', node)
      if not namestandard:
        continue
      namestandard = namestandard[0]
      logging.info("Checking " + namestandard + " series to make sure fat sat is turned on")
      localname = XPathEval('nameLocal', node)[0]
      localpath = DataPaths.get_LocalPath(localname)
      sortedfilelist = os.listdir(localpath)
      sortedfilelist.sort()
      for file in sortedfilelist:
        fullpath = os.path.join(localpath, file)
        try:
          f = open(fullpath,"r")
        except IOError:
          continue
        (unpack16, unpack32, explicit, datasetpos) = read_dicom_init(f)
        if unpack16 == None:
          logging.debug("  " + fullpath + " not DICOM file, skipping...")
          f.close()
          continue
        # get SAT field
        (taggroup, tagelem, vr, satbytes) = find_dicom_tag(f, 0x0019, 0x10a4, unpack16, unpack32, explicit)
        if satbytes == None:
          logging.warn("  " + fullpath + " doesn't have (GE Private) SAT field (0019,10a4)!  Skipping...")
          f.close()
          continue
        # note that we assume VR is SS
        if vr != 'UN' and vr != 'SS':
          logging.warn("  " + fullpath + " has an explicit VR for field 0019,10a4 that is not SS!  Skipping...")
          f.close()
          continue
        satval = interpretVR(satbytes, 'SS', unpack16, unpack32)
        if satval != 1:
          errMsg = "SAT field value (" + str(satval) + ") of file " + fullpath + " is incorrect (expected to be 1)\n"
          ProtocolErrList[0] += 1
          ProtocolErrList[1].append((errMsg, None))
        f.close()
        # just stop at one DICOM file per series
        break

def check_asl_dummy(XMLDoc, DataPaths, ProtocolErrList):
  # For GE sites, verify ASL calibration scan has the right number of dummy TRs
  for paradigm in [ 'asl-calibration' ]:
    nodelist = xml.xpath.Evaluate("/FIPS/series[paradigm='" + paradigm + "']", XMLDoc)
    for node in nodelist:
      namestandard = XPathEval('nameStandard', node)
      if not namestandard:
        continue
      namestandard = namestandard[0]
      logging.info("Checking " + namestandard + " series to make sure ndda=8")
      localname = XPathEval('nameLocal', node)[0]
      localpath = DataPaths.get_LocalPath(localname)
      sortedfilelist = os.listdir(localpath)
      sortedfilelist = filter(lambda x: x.endswith('.HEAD'), sortedfilelist)
      sortedfilelist.sort()
      for file in sortedfilelist:
        fullpath = os.path.join(localpath, file)
        try:
          f = open(fullpath,"r")
        except IOError:
          continue
        contents = f.read()
        rematch = re.search('Additional scan parameters for PPbrik:\\\\ndda (\d+)', contents)
        if rematch == None:
          errMsg = "Could not find an ndda value in file " + fullpath + "!\n"
          ProtocolErrList[0] += 1
          ProtocolErrList[1].append((errMsg, None))
          continue
        if rematch.group(1) != '8':
          errMsg = "ndda value (" + rematch.group(0) + ") in file " + fullpath + " should be 8!\n"
          ProtocolErrList[0] += 1
          ProtocolErrList[1].append((errMsg, None))
          continue

def run(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct, ProtocolErrList):
  logging.info("=== HOOK: fBIRNPhaseIII/BeforeSeries ===")
  add_cigal_files(XMLDoc, UploadConfig, StagingPaths, DataPaths)
  check_cigal_roots(UploadConfig, DataPaths, ProtocolErrList)
  if UploadConfig.get_ScannerManufacturer() == 'GE':
    check_fat_sat(XMLDoc, DataPaths, ProtocolErrList)
    check_asl_dummy(XMLDoc, DataPaths, ProtocolErrList)
    extract_B0(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct, ProtocolErrList)
  if UploadConfig.get_ScannerManufacturer() == 'Siemens':
    extract_asl_cal(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct)
    verify_B0(XMLDoc, DataPaths, ProtocolErrList)
    verify_phase_encode_dir(XMLDoc, DataPaths, ProtocolErrList)
  deface_t1(XMLDoc, SiteInfo, UploadConfig, DataPaths, StagingPaths, outputTimestampStruct)

