/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.nbirn.fbirn.utilities.download;

import java.awt.BorderLayout;
import org.nbirn.fbirn.utilities.CanceledException;
import org.nbirn.fbirn.utilities.CredentialManager;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import org.nbirn.fbirn.utilities.ExceptionCollection;
import org.nbirn.fbirn.utilities.ExceptionDialog;

/**
 *
 * @author gadde
 */
public class WorkerDownloadFromCatalog extends WorkerDownload implements PropertyChangeListener {

    private static final Logger _logger = Logger.getLogger(WorkerDownloadFromCatalog.class.getName());
    private static final Font NORMALFONT = new java.awt.Font("Dialog", 0, 12);
    private DownloaderCatalog _catalog = null;
    private File _dest = null;
    private JPanel _topprogpanel = null;
    private TitledBorder _border = null;
    private CredentialManager _credman = null;
    private boolean _updateOnly = false;
    private WorkerProgressReporter _parentworker = null;
    private double _parentProgressAmount = -1;
    private double _lastprogress = -1;
    private double _curprogress = 0;
    private int _logPriority = 0;
    final ArrayList<WorkerProgressReporter> _tasks; // in-progress
    final ArrayList<WorkerProgressReporter> _idletasks; // waiting for a restart signal

    public WorkerDownloadFromCatalog(DownloaderCatalog catalog, File dest, TitledBorder border, CredentialManager credman, boolean updateOnly, boolean dryRun, WorkerProgressReporter parentworker, double parentProgressAmount) {
        super(dryRun);
        _catalog = catalog;
        _dest = dest;
        _border = border;
        _credman = credman;
        _updateOnly = updateOnly;
        _curprogress = 0;
        _parentworker = parentworker;
        _parentProgressAmount = parentProgressAmount;
        _tasks = new ArrayList<WorkerProgressReporter>();
        _idletasks = new ArrayList<WorkerProgressReporter>();
    }

    public void setLoggingPriority(int priority) {
        _logPriority = priority;
    }

    private class AddEntryProgressBar implements Runnable {

        JPanel _parentpanel = null;
        String _title = null;
        JProgressBar _progbar = null;
        JLabel _messageLabel = null;
        JLabel _submessageLabel = null;
        JButton _cancelButton = null;
        JButton _refreshButton = null;
        int _rownum;

        AddEntryProgressBar(JPanel parentpanel, String title, int rownum) {
            _parentpanel = parentpanel;
            _title = title;
            _rownum = rownum;
        }

        public void run() {
            GridBagConstraints constraints = new GridBagConstraints();

            JPanel panel = new JPanel(new GridBagLayout());
            constraints.gridx = 0;
            constraints.gridy = _rownum;
            constraints.weightx = 1.0;
            constraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
            panel.setBorder(new EmptyBorder((_rownum == 0) ? 0 : 8, 0, 0, 0));
            _parentpanel.add(panel, constraints);

            constraints = new GridBagConstraints();
            JLabel label = new JLabel(_title == null ? "" : _title);
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            panel.add(label, constraints);
            _progbar = new JProgressBar();
            _progbar.setBorder(new EmptyBorder(0, 10, 0, 0));
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1.0;
            panel.add(_progbar, constraints);
            _messageLabel = new JLabel(" ");
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 2;
            constraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1.0;
            _messageLabel.setFont(new Font("Dialog", 0, 9));
            panel.add(_messageLabel, constraints);
            _submessageLabel = new JLabel(" ");
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.gridwidth = 1;
            constraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1.0;
            _submessageLabel.setFont(new Font("Dialog", 0, 9));
            panel.add(_submessageLabel, constraints);
            _progbar.setStringPainted(true);
            _progbar.setIndeterminate(true);

            constraints = new GridBagConstraints();
            JPanel buttonpanel = new JPanel(new GridBagLayout());
            constraints.gridx = 1;
            constraints.gridy = _rownum;
            buttonpanel.setBorder(new EmptyBorder((_rownum == 0) ? 0 : 8, 0, 0, 0));
            _parentpanel.add(buttonpanel, constraints);

            constraints = new GridBagConstraints();
            _cancelButton = new JButton();
            _cancelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/nbirn/fbirndownloader/resources/cancel.png")));
            _cancelButton.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            constraints.gridx = 0;
            constraints.gridy = 0;
            buttonpanel.add(_cancelButton, constraints);
            _refreshButton = new JButton();
            _refreshButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/nbirn/fbirndownloader/resources/refresh.png")));
            _refreshButton.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            constraints.gridx = 0;
            constraints.gridy = 1;
            buttonpanel.add(_refreshButton, constraints);
        }
    }

    // end main event thread helper routines
    ///////////////////////////////////////////////////////////////////
    @Override
    protected void process(List<ProgressUpdate> chunks) {
        for (int i = 0; i < chunks.size(); i++) {
            ProgressUpdate update = chunks.get(i);
            double incprog = update.getIncrementalProgress();
            _logger.log(Level.FINE, "WorkerDownloadFromCatalog: catalog ''{0}'' processing incremental progress {1} (current progress {2}) message=''{3}", new Object[]{_catalog.getName(), incprog, _curprogress, update.getMessage()});
            _curprogress += incprog;
            if (update.getMessage() != null || update.getSubMessage() != null) {
                String msg = ((_catalog == null) ? "" : _catalog.getName()) + " | " + update.getMessage() + (update.getSubMessage() == null ? "" : (" [" + update.getSubMessage() + "]"));
                if (_parentworker != null) {
                    _parentworker.sendUpdate(new ProgressUpdate(msg, null, 0, update.getPriority()));
                } else if (_parentworker == null) {
                    if (_catalog.getParentPanel() == null && update.getPriority() > _logPriority) {
                        _logger.info(msg);
                    } else {
                        _logger.log(Level.FINE, msg);
                    }
                }
            }
        }
        if (_curprogress > 100) {
            _curprogress = 100;
        }
        setProgress((int) Math.floor(_curprogress + 0.5));
    }

    public void propertyChange(PropertyChangeEvent evt) {
        String propName = evt.getPropertyName();
        if ("state".equals(propName) && evt.getNewValue() == SwingWorker.StateValue.DONE) {
            Throwable result = null;
            try {
                get();
            } catch (InterruptedException e) {
                result = new CanceledException("Operation canceled.");
            } catch (ExecutionException e) {
                result = e.getCause();
            } catch (CancellationException e) {
                result = new CanceledException("Operation canceled.");
            }
            if (_border != null && result != null) {
                String title = _border.getTitle();
                int errind = title.indexOf(" (ERR: ");
                if (errind != -1) {
                    title = title.substring(0, errind);
                }
                title = title + " (ERR: " + result.getMessage() + ")";
                _border.setTitle(title);
            }
        } else if (propName.equals("progress")) {
            int val = (Integer) evt.getNewValue();
            _logger.log(Level.FINE, "WorkerDownloadFromCatalog: catalog ''{0}'' got progress {1} (last progress {2})", new Object[]{_catalog.getName(), val, _lastprogress});
            if (_lastprogress == -1 || val - _lastprogress != 0) {
                double parentincval = (((_lastprogress == -1) ? val : (val - _lastprogress)) * (_parentProgressAmount / 100.0));
                if (_lastprogress == -1) {
                    _logger.log(Level.FINE, "WorkerDownloadFromCatalog:   reporting incremental progress of {0} * ( {1} / 100.0) = {2} to parent", new Object[]{val, _parentProgressAmount, parentincval});
                } else {
                    _logger.log(Level.FINE, "WorkerDownloadFromCatalog:   reporting incremental progress of ({0} - {1}) * ( {2} / 100.0) = {3} to parent", new Object[]{val, _lastprogress, _parentProgressAmount, parentincval});
                }
                _parentworker.sendUpdate(new ProgressUpdate(null, parentincval));
            }
            _lastprogress = val;
        }
    }

    void cancelTasks() {
        synchronized (_tasks) {
            _logger.log(Level.FINE, "Canceling {0} subtasks for catalog ''{1}''.", new Object[]{_tasks.size(), _catalog.getName()});
            for (int i = 0; i < _tasks.size(); i++) {
                _tasks.get(i).cancel(true);
            }
        }
    }

    private class HiddenPanelLabel extends JLabel {

        public HiddenPanelLabel(String text) {
            super(text);
            super.setFont(NORMALFONT);
        }
    }

    // main download scheduler
    void doDownload(DownloaderCatalog catalog) throws InvocationTargetException, CanceledException, ExceptionCollection {
        int numentries = catalog.getNumEntries();
        int numcatalogs = catalog.getNumCatalogs();
        int downloadtype = catalog.getDownloadType();
        TimeZone remoteTimeZone = catalog.getTimeZone();
        int timeStampCheckDepth = catalog.getTimeStampCheckDepth();

        final double subfrac = 0.00; // how much of a subcatalog's or subentry's progress amount we reserve for ourselves
        if (numentries + numcatalogs <= 0) {
            // nothing else to do here
            setProgress(100);
            return;
        }
        ExceptionCollection result = new ExceptionCollection();
        try {
            JPanel hiddenPanel = (catalog.getSubPanels() == null) ? null : catalog.getSubPanels().getHiddenPanel();
            JPanel basePanel = (catalog.getSubPanels() == null) ? catalog.getParentPanel() : catalog.getSubPanels().getSubPanel();
            if (basePanel != null) {
                // create panels and progress bars for this catalog level
                //  first for entries
                for (int i = 0; i < numentries; i++) {
                    DownloaderCatalogEntry entry = catalog.getEntry(i);
                    AddEntryProgressBar invokee = new AddEntryProgressBar(basePanel, entry.getName(), i);
                    SwingUtilities.invokeAndWait(invokee);
                    if (downloadtype != DownloaderCatalog.SOURCE_PARALLEL) {
                        invokee._progbar.setEnabled(false);
                    }
                    entry.setProgressBar(invokee._progbar);
                    entry.setMessageLabel(invokee._messageLabel);
                    entry.setSubMessageLabel(invokee._submessageLabel);
                    entry.setCancelButton(invokee._cancelButton);
                    entry.setRefreshButton(invokee._refreshButton);
                }
                //  now for sub-catalogs
                for (int i = 0; i < numcatalogs; i++) {
                    DownloaderCatalog subcat = catalog.getCatalog(i);
                    subcat.setSubPanels(new DownloaderCatalogPanels(basePanel, subcat.getName()));
                }
                ((JFrame) basePanel.getTopLevelAncestor()).pack();
            }
            if (hiddenPanel != null) {
                hiddenPanel.removeAll();
                hiddenPanel.add(new HiddenPanelLabel("In progress..."));
            }
            // call the actual downloads
            double progressChunk = 100.0 / (numentries + numcatalogs);
            //  first the entries
            for (int i = 0; i < numentries; i++) {
                DownloaderCatalogEntry entry = catalog.getEntry(i);
                TimeZone tz = entry.getTimeZone();
                if (remoteTimeZone != null && tz == null) {
                    tz = remoteTimeZone;
                }
                int tscd = entry.getTimeStampCheckDepth();
                if (tscd == -1 && timeStampCheckDepth != -1) {
                    tscd = timeStampCheckDepth;
                }
                WorkerURLUpdater worker = new WorkerURLUpdater(entry, _dest, _credman, _updateOnly, _dryrun, tz, tscd, this, (1 - subfrac) * progressChunk);
                synchronized (_tasks) {
                    _tasks.add(worker);
                }
                Executors.newCachedThreadPool().execute(worker);
                if (catalog.getDownloadType() != DownloaderCatalog.SOURCE_PARALLEL) {
                    _logger.log(Level.FINE, "Doing serial download of entry {0}", entry.getURI().toString());
                    try {
                        worker.get(); //block until completed.
                        publish(new ProgressUpdate(null, subfrac * progressChunk));
                    } catch (ExecutionException e) {
                        Throwable ret = e.getCause();
                        if (ret != null && ret.getClass() == CanceledException.class) {
                            if (hiddenPanel != null) {
                                hiddenPanel.removeAll();
                                hiddenPanel.add(new HiddenPanelLabel("Canceled."));
                            }
                            throw (CanceledException) ret;
                        }
                        // this error will be collected again later.
                        // We continue anyway because the user always has the
                        // option of canceling the process explicitly.
                    }
                } else {
                    _logger.log(Level.FINE, "Doing parallel download of entry {0}", entry.getURI().toString());
                }
            }
            for (int i = 0; i < numcatalogs; i++) {
                DownloaderCatalog subcat = catalog.getCatalog(i);
                if (remoteTimeZone != null && subcat.getTimeZone() == null) {
                    subcat.setTimeZone(remoteTimeZone);
                }
                if (timeStampCheckDepth != -1 && subcat.getTimeStampCheckDepth() == -1) {
                    subcat.setTimeStampCheckDepth(timeStampCheckDepth);
                }
                // we reserve a fraction of the progress chunk allotted to this catalog for us to report after the catalog completes downloading
                WorkerDownloadFromCatalog worker = new WorkerDownloadFromCatalog(subcat, _dest, subcat.getBorder(), _credman, _updateOnly, _dryrun, this, (1 - subfrac) * progressChunk);
                worker.addPropertyChangeListener(worker);
                synchronized (_tasks) {
                    _tasks.add(worker);
                }
                Executors.newCachedThreadPool().execute(worker);
                if (catalog.getDownloadType() != DownloaderCatalog.SOURCE_PARALLEL) {
                    _logger.log(Level.FINE, "Doing serial download of catalog {0}", subcat.getName());
                    try {
                        worker.get(); //block until completed.
                        publish(new ProgressUpdate(null, subfrac * progressChunk));
                    } catch (ExecutionException e) {
                        Throwable ret = e.getCause();
                        if (ret != null && ret.getClass() == CanceledException.class) {
                            if (hiddenPanel != null) {
                                hiddenPanel.removeAll();
                                hiddenPanel.add(new HiddenPanelLabel("Canceled."));
                            }
                            throw (CanceledException) ret;
                        }
                        // this error will be collected later.
                        // We continue anyway because the user always has the
                        // option of canceling the process explicitly.
                    }
                } else {
                    _logger.log(Level.FINE, "Doing parallel download of catalog {0}", subcat.getName());
                }
            }
            // wait for background tasks (if any) to complete, and collect errors, if any
            int numerrs = 0;
            _logger.log(Level.FINE, "Catalog ''{0}'' waiting for {1} subtasks to complete.", new Object[]{_catalog.getName(), _tasks.size()});
            while (true) {
                if (isCancelled()) {
                    if (hiddenPanel != null) {
                        hiddenPanel.removeAll();
                        hiddenPanel.add(new HiddenPanelLabel("Canceled."));
                    }
                    throw new InterruptedException("Catalog '" + _catalog.getName() + "' was canceled.  Throwing interrupt...");
                }
                synchronized (_tasks) {
                    for (int i = 0; i < _tasks.size(); i++) {
                        WorkerProgressReporter subworker = _tasks.get(i);
                        if (!subworker.isDone()) {
                            continue;
                        }
                        _logger.log(Level.FINE, "Catalog ''{0}'' removing completed task {1} ({2} tasks remaining)", new Object[]{_catalog.getName(), i, _tasks.size() - 1});
                        _tasks.remove(i);
                        i--;
                        Throwable subresult = null;
                        try {
                            subworker.get();
                            if (catalog.getDownloadType() == DownloaderCatalog.SOURCE_PARALLEL) {
                                // this was a parallel download, so we never updated progress
                                publish(new ProgressUpdate(null, subfrac * progressChunk));
                            }
                        } catch (CancellationException e) {
                            // either we requested the task cancellation or the user did.
                            // in either case, we don't have anything to pass up
                            subresult = e;
                        } catch (ExecutionException e) {
                            subresult = e.getCause();
                            if (subresult != null && subresult.getClass() == CanceledException.class) {
                                if (hiddenPanel != null) {
                                    hiddenPanel.removeAll();
                                    hiddenPanel.add(new HiddenPanelLabel("Canceled."));
                                }
                                throw (CanceledException) subresult;
                            }
                        }
                        _logger.log(Level.FINE, "Task {0} return value is {1}", new Object[]{i + 1, subresult});
                        if (subresult != null) {
                            if (subresult.getClass() == DownloadException.class && subresult.getCause() != null && subresult.getCause().getClass() == ExceptionCollection.class) {
                                ArrayList<Throwable> errlist = ((ExceptionCollection) subresult.getCause()).getList();
                                for (int errind = 0; errind < errlist.size(); errind++) {
                                    result.add(errlist.get(errind));
                                }
                            } else {
                                result.add(subresult);
                            }
                            numerrs++;
                            if (hiddenPanel != null) {
                                hiddenPanel.removeAll();
                                hiddenPanel.add(new HiddenPanelLabel("In progress... (" + numerrs + " errors)"));
                            }
                        }
                    }
                    if (_tasks.isEmpty()) {
                        break;
                    }

                }
                try {
                    Thread.sleep(250);
                } catch (InterruptedException e) {
                    // no problem
                }
            }
            if (hiddenPanel != null) {
                hiddenPanel.removeAll();
                hiddenPanel.add(new HiddenPanelLabel("Completed " + (numerrs == 0 ? "without" : ("with " + numerrs)) + " errors."));
                catalog.getSubPanels().hide();
            }
        } catch (InterruptedException e) {
            _logger.log(Level.FINE, "Catalog ''{0}'' got interrupted.", _catalog.getName());
            cancelTasks();
            throw new CanceledException("Operation canceled.", e);
        } catch (CanceledException e) {
            if (_parentworker == null) {
                // send down cancellation
                _logger.log(Level.FINE, "Catalog ''{0}'' got a CanceledException.", _catalog.getName());
                cancelTasks();
            } else {
                // pass it up and the root downloader will send down cancellation requests
                _logger.log(Level.FINE, "Catalog ''{0}'' got a CanceledException.  Passing it up to parent.", _catalog.getName());
            }
            throw e;
        } finally {
            while (true) {
                synchronized (_tasks) {
                    for (int i = 0; i < _tasks.size(); i++) {
                        WorkerProgressReporter subworker = _tasks.get(i);
                        if (!subworker.isDone()) {
                            continue;
                        }
                        _logger.log(Level.FINE, "(Dying) Catalog ''{0}'' removing completed task {1} out of {2}", new Object[]{_catalog.getName(), i, _tasks.size()});
                        subworker.cleanup(false);
                        _tasks.remove(i);
                        i--;
                    }
                    if (_tasks.isEmpty()) {
                        break;
                    }
                }
                try {
                    Thread.sleep(250);
                } catch (InterruptedException e) {
                    // we don't care
                }
            }
            _logger.log(Level.FINE, "All tasks for catalog ''{0}'' completed.", _catalog.getName());
            cleanup(false);
        }
        if (!result.getList().isEmpty()) {
            _logger.log(Level.FINE, "doDownload() returning {0}", result);
            throw result;
        }
        _logger.log(Level.FINE, "doDownload() returning null");
        return;
    }

    @Override
    protected Void doInBackground() throws CanceledException, DownloadException {
        /*
        if (_parentworker == null && _catalog.getParentPanel() != null) {
        try {
        DownloaderCatalogPanels invokee = new DownloaderCatalogPanels(_catalog.getParentPanel(), _catalog.getName(), false);
        _catalog.setSubPanels(invokee);
        } catch (InterruptedException e) {
        throw new CanceledException("Operation canceled.", e);
        } catch (InvocationTargetException e) {
        throw new DownloadException("Error adding panel", e);
        }
        }
         */
        setProgress(0);
        try {
            doDownload(_catalog);
        } catch (CanceledException e) {
            throw e;
        } catch (ExceptionCollection e) {
            throw new DownloadException("Error in download", e);
        } catch (Exception e) {
            throw new DownloadException("Error in download", e);
        }
        return null;
    }

    @Override
    protected void cleanup(boolean error) {
        synchronized (_tasks) {
            if (_tasks != null) {
                for (int i = 0; i < _tasks.size(); i++) {
                    _tasks.get(i).cleanup(error);
                }
            }
        }
        synchronized (_idletasks) {
            if (_idletasks != null) {
                for (int i = 0; i < _idletasks.size(); i++) {
                    _idletasks.get(i).cleanup(error);
                }
            }
        }
    }

    @Override
    public void done() {
        if (_parentworker != null) {
            // this is a subsidiary process and our root ancestor will take care of collecting/displaying errors
            return;
        }
        try {
            get();
        } catch (Exception e) {
            _logger.log(Level.FINE, "Download resulted in Exception ''{0}'': {1}", new Object[]{e.getClass().getCanonicalName(), e.getMessage()});
            if (_catalog.getParentPanel() != null) {
                if (e.getClass() == ExecutionException.class && e.getCause() != null && e.getCause().getClass() == DownloadException.class && e.getCause().getCause() != null && e.getCause().getCause().getClass() == ExceptionCollection.class) {
                    ExceptionCollection ec = (ExceptionCollection) e.getCause().getCause();
                    ArrayList<Throwable> errlist = ec.getList();
                    JTabbedPane tp = new JTabbedPane();
                    for (int i = 0; i < errlist.size(); i++) {
                        _logger.log(Level.SEVERE, ExceptionDialog.getStackTraceAsString(errlist.get(i)));
                        JScrollPane pane = ExceptionDialog.getScrollPane(errlist.get(i));
                        tp.addTab("Error " + i, pane);
                    }
                    JOptionPane.showMessageDialog(_catalog.getParentPanel(), tp);
                } else if (e.getClass() == CancellationException.class || (e.getClass() == ExecutionException.class && e.getCause() != null && e.getCause().getClass() == CanceledException.class)) {
                    // do nothing
                } else {
                    ExceptionDialog.show(_catalog.getParentPanel(), new DownloadException("Error getting result of download", e));
                }
            }
        }
        cleanup(false);
        return;
    }
}
