package cbfbirn

import groovy.xml.MarkupBuilder
import java.io.File;
import java.text.SimpleDateFormat

class UCLAUploadInputBuilder {
	def local2SubjectIDMap = [:]

	static class SubjectInfo {
		def localId
		def subjectId
		def gender
		def birthDate
		def age
		def condition
		def location
		def scanner = 'Siemens TIM Trio 3T UCLA'
		def viList = []


		@Override
		public String toString() {
			String s = "SubjectInfo::[localId:$localId subjectId:$subjectId gender:$gender " +
					"birthDate:$birthDate condition:$condition location:$location\n" +
					"scanner:$scanner\n"
			viList.each {
				s += "\t" + it.toString() + "\n"
			}
			s += "]"
			return s;
		}
	}

	static class VisitInfo {
		def label
		def csfLoc
		def cbfLoc
		def anatLoc
		def visitDirLoc
		def visitDate
		@Override
		public String toString() {
			return "VisitInfo::[label:$label csfLoc:$csfLoc cbfLoc:$cbfLoc " +
			"anatLoc:$anatLoc visitDirLoc:$visitDirLoc visitDate:$visitDate]"
		}
	}

	def loadCSV(File inCSVFile) {
		def siList = []
		def lines = []
		inCSVFile.text.eachLine { line, lineNo ->
			if (lineNo >= 1 ) lines << line
		}

		for (line in lines) {
			def toks = line.split(/,/);
			if (toks.length != 3)  continue;
			def localId = stripQuotes(toks[0])
			def age = toks[2].toDouble()
			def gender  = toks[1] == 'M' ? 'male' : 'female';
			SubjectInfo si = new SubjectInfo(localId: localId, gender: gender, age:age);
			siList << si;
		}
		return siList
	}

	def loadLocalId2SubjectIDMap(File cvsLookupFile) {
		cvsLookupFile.text.eachLine { line ->
			def toks = line.split(/,/)
			assert toks.size() == 2
			local2SubjectIDMap[toks[0]] = toks[1]
		}
	}

	def prepRawDataDirs(File inRoot, File outRoot, siList, boolean skipCopy = false) {
		assert local2SubjectIDMap.size() > 0
		def map = [:]
		def anatMap = [:]


		def anatDir = "$inRoot/MPRAGE"
		new File(anatDir).eachFile { f ->
			if (f.name.endsWith(".hdr")) {
				def toks = f.name.split(/_/)
				anatMap[toks[0]] = f.absolutePath
			}
		}
		siList.each { si ->
			si.subjectId = local2SubjectIDMap[si.localId]
			String headerFile = anatMap[si.localId]
			String visitDate = getVisitDate(headerFile)
			si.birthDate = getBirthDate(visitDate, si.age)
			String anatLoc = headerFile
			String csf1Loc = getMatchingHeaderFile("$inRoot/T1_1650/Mean_Bold", si.localId)
			String csf2Loc = getMatchingHeaderFile("$inRoot/T1_Custom/Mean_Bold", si.localId)
			String cbf1Loc = getMatchingHeaderFile("$inRoot/T1_1650/Mean_CBF", si.localId)
			String cbf2Loc = getMatchingHeaderFile("$inRoot/T1_Custom/Mean_CBF", si.localId)
			si.viList << new VisitInfo(anatLoc:anatLoc, csfLoc: csf1Loc, cbfLoc: cbf1Loc, label:'T1_1650', visitDate: visitDate)
			si.viList << new VisitInfo(anatLoc:anatLoc, csfLoc: csf2Loc, cbfLoc: cbf2Loc, label:'T1_Custom', visitDate: visitDate)
		}

		siList.each { si ->
			for(VisitInfo vi : si.viList) {
				String vdName = si.localId + "_" + vi.label;
				File destSubjectPath = new File(outRoot, vdName);
				destSubjectPath.mkdirs();
				vi.visitDirLoc = destSubjectPath
				if (!skipCopy) {
					copyAnalyzeSeries(destSubjectPath, vi.anatLoc);
					copyAnalyzeSeries(destSubjectPath, vi.csfLoc);
					copyAnalyzeSeries(destSubjectPath, vi.cbfLoc);
				}
			}
		}
	}

	static copyAnalyzeSeries(File outVisitDir, String headerFile) {
		File destFile = new File( outVisitDir, new File(headerFile).name)
		copyFile(new File(headerFile), destFile)
		File srcImgFile = new File( headerFile.replaceFirst(/.hdr$/, ".img"));
		File destImgFile = new File( outVisitDir, srcImgFile.name)
		copyFile(srcImgFile, destImgFile)
	}

	static String getMatchingHeaderFile(String rootDir, String localId) {
		String headerFile = null
		new File(rootDir).eachFile { f ->
			if (f.name.endsWith(".hdr")) {
				def toks = f.name.split(/_/)
				if (toks[0].equals(localId)) {
					headerFile = f.absolutePath
				}
			}
		}
		return headerFile
	}


	static String getVisitDate(String anatHdrFile) {
		String t = "strings $anatHdrFile".execute().text
		def m = t =~ /(\d+\-\w{3,3}\-\d{4,4})\s+\d+:\d+:\d+\.\d+$/
		def vd = m[0][1]
		SimpleDateFormat sd = new SimpleDateFormat('dd-MMM-yyyy')
		SimpleDateFormat osd = new SimpleDateFormat('MM/dd/yyyy')
		return osd.format( sd.parse(vd) )
	}

	static String getBirthDate(String visitDate, age) {
		SimpleDateFormat osd = new SimpleDateFormat('MM/dd/yyyy')
		def ageIn = Math.floor(age).toInteger()
		def rem = age - Math.floor(age)
		def days = rem * 365
		Calendar cal = Calendar.getInstance()
		cal.setTimeInMillis(osd.parse(visitDate).time)
		cal.add(Calendar.YEAR, -ageIn)
		cal.add(Calendar.DAY_OF_YEAR, -(days.toInteger()))
		println osd.format(cal.getTime())
		return osd.format(cal.getTime())
	}

	static void copyDir(File src, File dest) {
		dest.mkdir();
		def srcFiles = []
		src.eachFile { if (it.isFile()) srcFiles << it }
		srcFiles.each { srcFile ->
			copyFile(srcFile, new File(dest, srcFile.name ));
		}
	}

	static void copyFile(File src, File dest) {
		new AntBuilder().copy(file:"$src.canonicalPath",tofile:"$dest.canonicalPath",overwrite:'true')
	}

	def static writeFile(File outFile, String content) {
		outFile.withWriter { out ->
			out.write(content); out.append("\n")
		}
		println "wrote file:$outFile"
	}

	static String stripQuotes(String csvVal) {
		if (csvVal.indexOf('"') == -1) return csvVal
		def m = csvVal =~ /"([^"]+)"/
		return m[0][1]
	}

	def writeXml(siList, File outFile) {
		assert local2SubjectIDMap.size() > 0
		def writer = new StringWriter()
		def xml = new MarkupBuilder(writer)
		xml.cbfImport() {
			for(si in siList) {
				def subjectID = local2SubjectIDMap[si.localId]
				def location = 'NA';
				for(vi in si.viList) {
					ds(subjectID:subjectID,  diagnosis:'control', expName:'UCLA_PED_ASL',
					visitDate:vi.visitDate, birthDate:si.birthDate, gender: si.gender,
					scannerInfo: si.scanner, location: vi.visitDirLoc)
				}
			}
		}
		println writer.toString()
		if (outFile != null) {
			writeFile(outFile, writer.toString())
		}
	}

	static void upload() {
		File imageDataRoot = new File('/data/burak/ucla_in')
		File rawDataUploadRoot = new File('/data/burak/ucla_out')
		UCLAUploadInputBuilder builder = new UCLAUploadInputBuilder()
		def siList = builder.loadCSV(new File('/home/bozyurt/dev/java/clinical/scripts/ucla_peds_project/PEDS_CBF.csv'));

		siList.each { println it }
		
		println "-" * 80
		builder.loadLocalId2SubjectIDMap(new File('/home/bozyurt/dev/java/clinical/scripts/ucla_peds_project/birnid_map.csv'))

		builder.prepRawDataDirs(imageDataRoot, rawDataUploadRoot, siList, true)
		siList.each { println it }

		builder.writeXml(siList, new File('/home/bozyurt/dev/java/clinical/scripts/ucla_peds_project/ucla_data_import.xml'))
	}

	static void main(args) {
		upload()
	}
}
